/* GameStateAbstractAction.java
 * Created on Mar 11, 2004
 *
 * Part of JTicTacToe
 * (c) beltorak 2004
 * Released under the GPL
 */
package jtttGameState.Actions;

import java.awt.event.ActionEvent;

import javax.swing.AbstractAction;
import javax.swing.Icon;

import jtttGameState.GameState;
import jtttGameState.GameStateChangeEvent;
import jtttGameState.GameStateListener;
import jtttGameState.GameStateListenerList;

/**
 * This abstract class is the super class of all instantiated GameStateJButtons and 
 * GameStateJMenuItems (which are JButtons and JMenuItems that need to receive
 * GameStateChangeEvents).  The concrete classes that inherit
 * this abstract need only implement gameStateChanged(GameStateChangeEvent gsce)
 * and actionPerformed(ActionEvent ae). 
 */
public abstract class GameStateAbstractAction
    extends AbstractAction implements GameStateListener
{
	public String toString() {
		return "GameStateAbstractAction";
	}

	/** GameStateChangeEvent variable */
	GameStateChangeEvent gsce;
	
	/**
	 * Every instance of concrete subclasses of this class maintain
	 * thier own GameStateListenerList.
	 */
	GameStateListenerList gameStateListenerList;
	
	/**
	 * The GameState singleton reference
	 */
	protected GameState gameState;
	
	/**
	 * Default constructor.  Sets Nothing.
	 */
	public GameStateAbstractAction() {
		this(null, null, null);
	}

    /**
     * Alternate constructor that sets three things in one go.
     * @param name The label to display on the object
     * @param icon The icon to display with the name
     * @param tooltip The text that pops up when the mouse hovers over the object
     */
    GameStateAbstractAction(String name, Icon icon, String tooltip) {
		super();
    	putValue(NAME, name);
    	putValue(SHORT_DESCRIPTION, tooltip);
		putValue(SMALL_ICON, icon);
		
		gameStateListenerList = GameStateListenerList.getGameStateListenerList();
		gameState = GameState.getGameState();
		gameStateListenerList.addGameStateListener(this);
    }

	/**
	 * Alternate constructor to set the label and tooltip in one go.
	 * @param name The name displayed on the object
	 * @param tooltip The text that pops up when the mouse "hovers" over the object.
	 */
	public GameStateAbstractAction(String name, String tooltip) {
		this(name, null, tooltip);
	}

    /**
     * A Constructor that specifies only a name
     * @param name The object label.
     */
    public GameStateAbstractAction(String name) {
		this(name, null, null);
    }

    /**
     * Constructor that specifes the name and icon in one go.
     * @param name The label to put on the object
     * @param icon The icon to put next to it.
     */
    public GameStateAbstractAction(String name, Icon icon) {
		this(name, icon, null);
    }

	/**
	 * Convenience wrapper method.
	 * @return true if a game is in progress, false if a game is not in progress.
	 */
	public boolean isGameInProgress() {
		return gameState.isGameInProgress();
	}

	/**
	 * Convenience wrapper method.
	 * @return true if a game is in progress and paused, false if a game is in progress
	 * but not paused.
	 */
	public boolean isGamePaused() {
		return gameState.isGamePaused();
	}

	/**
	 * Convenience wrapper method.  Fires a GameStateChangeEvent if the game state is modified.
	 * @param source The source of the change event
	 * @param b Set to true upon starting a new game, false when the game has ended.
	 */
	public void setGameInProgress(Object source, boolean b) {
		if (gameState.isGameInProgress() != b) {
			gameState.setGameInProgress(b);
			fireGameStateChangeEvent(source);
		}
	}

	/**
	 * Convenience wrapper method.
	 * Fires a GameStateChangeEvent if the GameState is modified. 
	 * @param source The source of the change event
	 * @param b Set to true upon pausing a game in progress, false upon
	 * "Resume"ing a game in progress.  
	 * @throws UnsupportedOperationException if an attempt is made to set this state while gameInProgress == false
	 */
	public void setGamePaused(Object source, boolean b) {
		if (gameState.isGamePaused() != b) {
			gameState.setGamePaused(b);
			fireGameStateChangeEvent(source);
		}
	}
    
    /**
     * Fires a GameStateChangeEvent to every GameStateObject in the list.
     * @param source The source of the GameStateChangeEvent.
     */
    public static void fireGameStateChangeEvent(Object source) {
		GameState gs = GameState.getGameState();
		GameStateListenerList gsll = GameStateListenerList.getGameStateListenerList();

		GameStateListener[] listener = gsll.getGameStateListeners();
		for (int index = listener.length - 1; index >= 0; --index)
			listener[index].gameStateChanged(new GameStateChangeEvent(source, gs));
    }

    /**
     * The implementing Action only changes it's properties (ie, setEnabled(boolean b)) in this method.
     */
    public abstract void gameStateChanged(GameStateChangeEvent gsce);

	/** The implementing Action only changes the game state in this method */
    public abstract void actionPerformed(ActionEvent e);
}

/* end GameStateAbstractAction.java */