/*
 * JgSettings.java, part of JGlyphs
 * 
 * (c) 2004 Trevor "beltorak" Torrez
 * Released under the GPL, either version 2, or at your preference,
 * any later version.
 * 
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package jgSettings;

import java.io.File;

import jgDataLoader.CatagoryListenerList;
import jgGlyphs.GlyphSet;
import jgMain.JgMain;
import jgMain.JgMeta;


/**
 * JgSettings maintains the settings for the program, such as catagory and
 * selected glyphs.
 * TODO: allow multiple catagory selections and include dirs.
 */
public class JgSettings {

	/** Returns the string representation of this class */
	public String toString() {
		return "JgSettings";
	}

	/** An array of Strings representing Characters or sequences for the game */
	protected GlyphSet glyphs;
	/**
	 * Set to true if glyphs were specified on the command line.
	 * This includes a specification of "use all glyphs"
	 */
	protected  boolean glyphsWereSet = false;
	
	/** A string representing the catagory to start with */
	protected String catagory;
	/** Set to true if the catagory was specified on the command line */
	protected boolean catagoryWasSet = false;
	
	/** An extra directory to search for catagory folders in */
	protected File includeDir;
	  
	/** Sets the defaults */
	protected void init() {
		catagory = null;
		glyphs = new GlyphSet();;
	}

	/** Constructor that sets default setings */
	public JgSettings() {
		init();
	}
	
	/** Constructor that sets settings according to the passed command line */
	public JgSettings(String[] args) {
		init();
		// Parse command line
		for ( int k = 0; k < args.length; k++ ) {
			// Check for help
			if ( args[k].equals("--help") || args[k].equals("-h") || args[k].equals("-?") ) {
				JgMeta.printHelp();
				System.exit(0);
			}
			// Check for version
			else if ( args[k].equals("--version") || args[k].equals("-v") ) {
				JgMeta.printVersion();
				System.exit(0);
			}
			// Discern catagory
			else if ( args[k].equals("--catagory") || args[k].equals("-g") ) {
				k++;
				if ( k >= args.length ) {
					System.out.println("Expected catagory");
					System.exit(1);
				}
				catagory = args[k];
				catagoryWasSet = true;
			}
			// Use all catagories
			else if ( args[k].equals("--all-catagories") || args[k].equals("-a") ) {
				catagory = null;
				catagoryWasSet = true;
			}
			// Include a directory for catagory folders
			else if ( args[k].equals("-i") || args[k].equals("--include") ) {
				k++;
				if ( k >= args.length ) {
					System.out.println("Expected include directory");
					System.exit(1);
				}
				includeDir = new File(args[k]).getAbsoluteFile();
				if ( ! includeDir.isDirectory() ) {
					System.out.println(includeDir + " is not a valid directory");
					System.exit(1);
				}
			}
			// Set the letters: this option uses individual Characters for each glyph
			else if ( args[k].equals("--letters" ) || args[k].equals("-l") ) {
				k++;
				if ( k >= args.length ) {
					System.out.println("Expected Characters after \"-l\"");
					System.exit(1);
				}
				glyphs.addGlyphChars(args[k]);
				glyphsWereSet = true;
			}
			// Sets the string type glyphs: this is a colin delimited list of strings
			else if ( args[k].equals("--sequence") ) {
				k++;
				if ( ! args[k].equals(":") ) {
					System.out.println("Expected a colin delimited list; ie \": ch str au :\"");
					System.exit(1);
				}
				k++;
				if ( k >= args.length ) {
					System.out.println("Exptected a string");
					System.exit(1);
				}
				while ( ! args[k].equals(":") ) {
					glyphs.addGlyph(args[k]);
					k++;
					if ( k >= args.length ) {
						System.out.println("Missing end colin");
						System.exit(1);
					}
				}
				glyphsWereSet = true;
			}
			// Carp on invalid argument
			else {
				System.out.println("Invalid argument \"" + args[k] + "\"");
				JgMeta.printHelp();
				System.exit(1);
			}
		}

	}

	/**
	 * Retrieves the catagory name.  A value of null means 'all available catagories'.
	 * @return the catagory name, or null if all available catagories are to be used.
	 */
	public String getCatagory() {
		return catagory;
	}

	/**
	 * Retrieves the GlyphSet instance
	 * @return the GlyphSet instance
	 */
	public GlyphSet getGlyphSet() {
		return glyphs;
	}

	/**
	 * Retireves the user-specified include dir.
	 * @return the user-specified include dir.
	 */
	public File getIncludeDir() {
		return includeDir;
	}

	/**
	 * Sets the catagory to the supplied string, or null to indicate
	 * 'all available catagories'.  If the supplied string is different
	 * from the original, fires a CatagoryChangeEvent.
	 * NOTE: This code smells.
	 * @param cat the catagory to set to.  Use null for
	 * 'all available catagories'.
	 * @param src the Object that has requested the catagory change.
	 */
	public void setCatagory(String cat, Object src) {
		if ( catagory == null ) {
			if ( cat == null ) {
				return;
			}
			if ( JgMain.getDataLoader().isCatagoryValid(cat) ) {
				catagory = new String(cat);
				CatagoryListenerList.fireCatagoryChangeEvent(src);
				return;
			} else {
				System.out.println("Invalid catagory name " + cat);
				System.exit(1);
			}
		}

		if ( cat == null ) {
			catagory = null;
			CatagoryListenerList.fireCatagoryChangeEvent(src);
			return;
		}
		
		if ( ! catagory.equals(cat) ) {
			if ( JgMain.getDataLoader().isCatagoryValid(cat) ) {
				catagory = new String(cat);
				CatagoryListenerList.fireCatagoryChangeEvent(src);
				return;
			} else {
				System.out.println("Invalid catagory name " + cat);
				System.exit(1);
			}
		}
	}
	
	/**
	 * Sets the glyphs used for play.
	 * @param gs the GlyphSet to use
	 */
	public void setGlyphSet(GlyphSet gs) {
		glyphs = new GlyphSet(gs);
	}
	
	/**
	 * Checks to see if the UI should be drawn to select options.
	 * @return true if the UI needs to be drawn; false otherwise.
	 */
	public boolean shouldDrawUI() {
		if ( glyphsWereSet && catagoryWasSet ) {
			return false;
		} else {
			return true;
		}
	}

}


 /* end JgSettings.java */