/*
 * GlyphSet.java, part of JGlyphs
 * 
 * (c) 2004 Trevor "beltorak" Torrez
 * Released under the GPL, either version 2, or at your preference,
 * any later version.
 * 
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package jgGlyphs;

import java.util.Arrays;
import java.util.Vector;

/**
 * A GlyphSet is a collection of Characters and string fragments
 */
public class GlyphSet {

	/** Returns the string representation of this class */
	public String toString() {
		return "GlyphSet " + glyphs.toString();
	}

	/**
	 * The vector of glyphs.
	 * Each glyph is a string, lowercase if applicable.
	 * This includes both string and Character glyphs.
	 */
	protected Vector glyphs;
	
	/** Constructor. */
	public GlyphSet() {
		glyphs = new Vector(26,10);
	}
	
	/** Constructor that effectively "clones" the passed GlyphSet */
	public GlyphSet(GlyphSet gs) {
		glyphs = new Vector(26, 10);
	 	addGlyphs(gs.toArray());
	}

	/**
	 * Adds new glyphs to the set.  Each Character in the string is a
	 * separate glyph to add.
	 * @param g the glyphs to add.
	 */
	public void addGlyphChars(String g) {
		for ( int k = 0; k < g.length(); k++ ) {
			addGlyph(g.substring(k,k+1));
		}
	}
	
	/**
	 * Adds a glyph to the list.  The entire (lowercased) string is added as one glyph.
	 * @param g the glyph to add
	 */
	public void addGlyph(String g) {
		g = g.toLowerCase();
		if ( ! glyphs.contains(g) ) {
			glyphs.add(g);
		}
	}
	
	/**
	 * Adds each String in the array via the addGlyph method
	 * @param ga the array of Strings to add
	 */
	public void addGlyphs(String[] ga) {
		for ( int k = 0; k < ga.length; k++ ) {
			addGlyph(ga[k]);
		}
	}
	
	/**
	 * Adds each glyph in the passed GlyphSet to this GlyphSet
	 * @param gs the GlyphSet to add
	 */
	public void addGlyphs(GlyphSet gs) {
		addGlyphs(gs.toArray());
	}
	
	/** Removes a glyph from the list */
	public void removeGlyph(String g) {
		glyphs.remove(g.toLowerCase());	
	}
	
	/** Returns the number of glyphs in the set */
	public int size() {
		return glyphs.size();
	}
	
	/** Returns true if the glyph set contains the specified glyph */
	public boolean contains(String g) {
		return glyphs.contains(g);
	}
	
	/** Returns all the glyphs in order */
	public String[] toArray() {
		String[] sa = new String[glyphs.size()];
		sa = (String[]) glyphs.toArray(sa);
		Arrays.sort(sa);
		return sa;
	}
	
	/**
	 * Prunes this glyph set to contain only elements in the passed glyph set
	 * @return itself with the offending entries removed.
	 */
	public GlyphSet keepOnly(GlyphSet gs) {
		for ( int k = 0; k < glyphs.size(); k++ ) {
			if ( ! gs.contains((String) glyphs.get(k)) ) {
				glyphs.remove(k);
				k = 0;
			}
		}
		return this;
	}
	
	/**
	 * Removes all glyphs in the passed set from this GlyphSet
	 * @param gs the glyphs to remove, as a GlyphSet
	 * @return itself without the glyphs passed in gs
	 */
	public GlyphSet removeGlyphs(GlyphSet gs) {
		String[] ga = gs.toArray();
		for ( int k = 0; k < ga.length; k++ ) {
			removeGlyph(ga[k]);
		}
		return this;
	}
	
}


/* end GlyphSet.java */