/*
 * GlyphImageWindow.java, part of JGlyphs
 * 
 * (c) 2004 Trevor "beltorak" Torrez
 * Released under the GPL, either version 2, or at your preference,
 * any later version.
 * 
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package jgGlyphs;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.Box;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.border.EtchedBorder;

import jgGame.JgGameClassic;
import jgMain.JgDebug;

/**
 * A GlyphImageWindow is a top level window showing the image and the glyph.
 */
public class GlyphImageWindow extends JFrame
	implements ActionListener
{

	/** Returns the string representation of this class */
	public String toString() {
		if ( glyph != null ) {
			return "GlyphImageWindow '" + glyph + "'";
		} else {
			return "GlyphImageWindow";
		}
	}

	/** The instance for this window */
	protected GlyphImageWindow giWindow;

	/** The glyph for this window. This also serves to identify the window. */
	protected String glyph = null;

	/** The image button, which acts as the image pane */
	protected JButton imageButton;
	
	/** The GlyphImageList this window was created with */
	protected GlyphImage[] images;
	
	/** The currently displaying image */
	protected int imdex;

	/**
	 * The only constructor.  Accepts an array of GlyphImages to display with
	 * this window.
	 * @param gia an array of GlyphImages to select among different images for the glyph
	 */
	public GlyphImageWindow(GlyphImage[] gia) {
		super();
		giWindow = this;
		setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);

		if ( gia.length < 1 ) {
			new JgDebug().crash("passed an empty GlyphImageList()");
		}
		images = gia;
		imdex = 0;

		glyph = gia[0].getGlyph();
		setTitle("The Glyph: " + glyph);
		
		addChildren();
		pack();
		setVisible(true);
	}
	
	/** Adds all the children */
	protected void addChildren() {
		Box m = Box.createVerticalBox();
		m.add(createImageButton(images[imdex]));
		Box c = createNameSelection();
		if ( c != null ) {
			m.add(c);
		}
		getContentPane().add(m);
	}

	/** Creates the image button and its contianer; returns the container*/
	protected Box createImageButton(GlyphImage glyphImage) {
		Box c = Box.createVerticalBox();
		imageButton = new JButton(new ImageIcon(glyphImage.getLocation().getPath()));
		imageButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				JgGameClassic.getGame().removeWindow(giWindow.getName());
				giWindow.dispose();
			}
		});
		c.add(imageButton);
		c.add(new JLabel(glyphImage.getMarkupLabel()));
		return c;
	}

	/**
	 * Creates the image selection part (iff there is more than one image) and its
	 * contianer.  Returns the container.
	 */
	protected Box createNameSelection() {
		Box box = null;
		if ( images.length > 1 ) {
			box = Box.createHorizontalBox();
			box.setBorder(new EtchedBorder());
			JButton ns;
			for ( int k = 0; k < images.length; k++ ) {
				if ( ! images[k].getGlyph().equals(glyph) ) {
					new JgDebug().crash(toString() + " used inconsistant glyphs: " + glyph + " and " + images[k].getGlyph());
				}
				ns = new JButton(images[k].getName());
				ns.setActionCommand(Integer.toString(k));
				ns.addActionListener(this);
				box.add(ns);
			}
		}
		return box;
	}
	
	/**
	 * Retrieves the window name, which is the glyph that this window pertains to
	 */
	public String getName() {
		return glyph;
	}

	/**
	 * Invoked when the picture is changed; removes all children and repaints them.
	 */
	public void actionPerformed(ActionEvent e) {
		imdex = Integer.parseInt(e.getActionCommand());
		getContentPane().removeAll();
		addChildren();
		pack();
		setVisible(true);
	}

}


/* end GlyphImageWindow.java */