/*
 * GlyphImageList.java, part of JGlyphs
 * 
 * (c) 2004 Trevor "beltorak" Torrez
 * Released under the GPL, either version 2, or at your preference,
 * any later version.
 * 
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package jgGlyphs;

import java.io.File;
import java.util.Vector;

/**
 * This is a simple class wrapper around a vector of glyph images.
 */
public class GlyphImageList {

	/** Returns the string representation of this class */
	public String toString() {
		return "GlyphImageList " + giList.toString();
	}

	/** The internal list of glyph images */
	protected Vector giList;

	/**
	 * Constructs the list.
	 */
	public GlyphImageList() {
		giList = new Vector(26,10);
	}

	/**
	 * Retrieves the vector position of a GlyphImage specified by name
	 * @param name the name of the GlyphImage
	 * @return the vector position, or -1 if not found.
	 */
	protected int getGlyphImageIndex(String name) {
		for ( int k = 0; k < giList.size(); k++ ) {
			if ( name.equals(get(k).getName()) ) {
				return k;
			}
		}
		return -1;
	}
	
	/**
	 * Adds a glyph image.  If a glyph image with the same name (as returned by the
	 * getName method of GlyphImage) and glyph (as returned by the getGlyph method) 
	 * already exists, it is replaced.
	 * @param file the File to add as a GlyphImage to add
	 */
	public void addGlyphImage(File file) {
		int k = getGlyphImageIndex(GlyphImage.getName(file));
		// If no stored GlyphImage has the same name, add this one 
		if ( k == -1 ) {
			giList.add(new GlyphImage(file));
		}
		// Otherwise, since the name is the same, if the glyph is different, add it
		else {
			 if ( ! get(k).getGlyph().equals(GlyphImage.getGlyph(file)) ) {
				giList.add(new GlyphImage(file));
			} else {
				giList.remove(k);
				giList.add(new GlyphImage(file));
			}
		}
	}
	
	/**
	 * Removes a glyph image from the list, specified by name
	 * @param name the name of the glyph image to remove.
	 */
	public void removeGlyphImage(String name) {
		int k = getGlyphImageIndex(name);
		if ( k > -1 ) {
			giList.remove(k);
		}
	}
	
	/**
	 * Retrieves a GlyphSet applicable to this list of glyph images.
	 * @return a GlyphSet applicable to this list of glyph images.
	 */
	public GlyphSet getGlyphs() {
		GlyphSet gs = new GlyphSet();
		for ( int k = 0; k < giList.size(); k++ ) {
			gs.addGlyph( ((GlyphImage) giList.get(k)).getGlyph() );
		}
		return gs;
	}

	/**
	 * Retrieves an array of GlyphImages for a specified glyph.
	 */
	public GlyphImage[] getGlyphImagesForGlyph(String glyph) {
		Vector giv = new Vector(giList.size());
		for ( int k = 0; k < giList.size(); k++ ) {
			if ( ((GlyphImage) giList.get(k)).getGlyph().equals(glyph) ) {
				giv.add(giList.get(k));
			}
		}
		return (GlyphImage[]) giv.toArray(new GlyphImage[giv.size()]);
	}
	
	/**
	 * Returns the size of this list
	 * @return the size of this list
	 */
	public int getSize() {
		return giList.size();
	}
	
	/**
	 * Retrieves the kth image in the list
	 * @return the kth GlyphImage in the list
	 */
	public GlyphImage get(int k) {
		return (GlyphImage) giList.get(k);
	}
}


/* end GlyphImageList.java */
