/*
 * GlyphImage.java, part of JGlyphs
 * 
 * (c) 2004 Trevor "beltorak" Torrez
 * Released under the GPL, either version 2, or at your preference,
 * any later version.
 * 
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package jgGlyphs;

import java.io.File;

/**
 * This class breaks down a filename into its basic parts.
 * The glyph is the prefix: either the string up to the first drop ("_") if one
 * exists in the filename, or the first Character otherwise.
 * The name is the normalized name: if the glyph is a multi-Character prefix,
 * this (and the following drop) is removed; the extension is also removed.
 * The loc (location) is the full pathspec.
 */
public class GlyphImage {

	/** Returns the string representation of this class */
	public String toString() {
		return "GlyphImage { glyph: " + getGlyph() + " name: " + getName() + " File: " + loc.getAbsolutePath() + " }";
	}

	/** This is the full pathspec to the image */
	protected File loc;
	
	/**
	 * Constructor, specifies the File location
	 */
	public GlyphImage(File file) {
		loc = file.getAbsoluteFile();
	}
	
	/**
	 * This is a convenience wrapper for the static class method.
	 * Returns the glyph that this image pertains to, lowercased if applicable.
	 * This is either the starting Character (as a String) if the filename does not contain any drops,
	 * or the starting String up to but not including the drop.
	 * @return a String representing the glyph that this image pertains to.
	 */
	public String getGlyph() {
		return getGlyph(loc);
	}
	
	/**
	 * Returns the glyph that this image pertains to, lowercased if applicable.
	 * This is either the starting Character (as a String) if the filename does not contain any drops,
	 * or the starting String up to but not including the drop.
	 * @return a String representing the glyph that this image pertains to.
	 */
	public static String getGlyph(File file) {
		String glyph = file.getName().toLowerCase();
		if ( glyph.indexOf("_") > -1 ) {
			return glyph.substring(0, glyph.indexOf("_"));
		}
		return glyph.substring(0,1);
	}

	/**
	 * This is a convenience wrapper for the static class method.
	 * Returns the normalized name.  This is the name of the file, lowercased if applicable,
	 * with the multi-Character prefix removed (if applicable) and the extension removed.
	 * @return the normallized name as a String
	 */
	public String getName() {
		return getName(loc);
	}
	
	/**
	 * Returns the normalized name.  This is the name of the file, lowercased if applicable,
	 * with the multi-Character prefix removed (if applicable) and the extension removed.
	 * @return the normallized name as a String
	 */
	public static String getName(File file) {
		String name = file.getName().toLowerCase();
		if ( name.indexOf("_") > -1 ) {
			return name.substring(name.indexOf("_")+1, name.lastIndexOf("."));
		}
		return name.substring(0, name.lastIndexOf("."));
	}

	/**
	 * Returns the location of this image.
	 * @return the location of this image as a File.
	 */
	public File getLocation() {
		return loc;
	}

	/**
	 * This is a convenience wrapper for the static class method.
	 * Compares the names of two GlyphImages to see tifthey are equal.
	 * @param gi a GlyphImage to compare names with
	 * @return true if both names are equal, false otherwise.
	 */
	public boolean equalsNameOf(GlyphImage gi) {
		return equalsNameOf(this, gi);
	}

	/**
	 * Compares the names of two GlyphImages to see if they are equal.
	 * @param orange a GlyphImage
	 * @param apple another GlyphImage
	 * @return true if both names are equal, false otherwise.
	 */
	public static boolean equalsNameOf(GlyphImage orange, GlyphImage apple) {
		if ( orange.getName().equals(apple.getName()) ) {
			return true;
		}
		return false;
	}

	/**
	 * Gets the name of the image, marking up the name using html.
	 * @return the html marked up name
	 */
	public static String getMarkupName(GlyphImage gi) {
		return
			"<html><font size=\"7\"><font color=\"red\">" +
			gi.getGlyph() + "</font>, " +
			gi.getName().replaceAll(gi.getGlyph(), "<font color=\"red\">" + gi.getGlyph() + "</font>") +
			"</font></html>";
	}

	/**
	 * This is a convenience wrapper for the static class method.
	 * Gets the name of the image, marking up the name using html.
	 * @return the html marked up name
	 */
	public String getMarkupLabel() {
		return getMarkupName(this);
	}
}


/* end GlyphImage.java */