/*
 * GlyphFilter.java, part of JGlyphs
 * 
 * (c) 2004 Trevor "beltorak" Torrez
 * Released under the GPL, either version 2, or at your preference,
 * any later version.
 * 
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package jgGlyphs;

import java.io.File;
import java.io.FileFilter;

/**
 * This is a FileFilter implemenation that filters files based on its extension and starting Characters.
 * Supported extensions are "jpeg" or "jpg", "gif", and "png".
 * 
 * To filter on a set of Characters, use the constructor GlyphFilter(String); each Character in the
 * passed string denotes acceptable prefixes.
 * 
 * To filter on a starting String, you must wrap the String in a GlyphSet or a String[] and use the
 * appropriate constructor.
 */
public class GlyphFilter implements FileFilter {

	/** Returns the string representation of this class. */
	public String toString() {
		if ( glyphSet == null ) {
			return "GlyphFilter with no glyph restrictions";
		} else {
			return "GlyphFilter { " + glyphSet.toString() + " }";
		}
	}

	/** This is the GlyphSet to filter on, if any */
	protected GlyphSet glyphSet;
	
	/**
	 * Constructor which will accept any starting glyphs.
	 */
	public GlyphFilter() {
		glyphSet = null;
	}

	/**
	 * Constructs a GlyphFilter using the Characters in the passed String as
	 * the list of valid starting Characters.
	 * @param gc the list of valid starting Characters, all together as one String.
	 */
	public GlyphFilter(String gc) {
			glyphSet = new GlyphSet();
			glyphSet.addGlyphChars(gc);
	}
	
	/**
	 * Constructs a GlyphFilter using the supplied array of Strings as the list of glyphs
	 * to filter on.  Each string is taken to be one glyph.
	 * @param ga the String array to filter on.
	 */
	public GlyphFilter(String[] ga) {
		glyphSet = new GlyphSet();
		glyphSet.addGlyphs(ga);
	}
	
	/**
	 * Constructor that takes a GlyphSet.  If the GlyphSet is empty, this is
	 * effectively the same as GlyphFilter().
	 * @param gs the GlyphSet to filter on.
	 */
	public GlyphFilter(GlyphSet gs) {
		if ( gs.size() == 0 ) {
			glyphSet = null;
		} else {
			glyphSet = new GlyphSet(gs);
		}
	}

	/**
	 * Accepts a file if it has a supported extension, and optionally if it starts
	 * with one of the glyphs specified by the GlyphSet this object was created with.
	 */
	public boolean accept(File file) {
		
		if ( ! file.isFile() ) {
			return false;
		}
		if ( ! extensionOK(file) ) {
			return false;
		}
		
		if ( glyphSet == null || prefixOK(file) ) {
			return true;
		}
		
		return false;
	}
	
	/**
	 * Checks that the file has a supported extension.
	 * Supported extensions are ":jpeg", "jpg", "gif", and "png".
	 */
	protected boolean extensionOK(File file) {
		String name = file.getName().toLowerCase();
		if (
			name.endsWith(".jpeg") ||
			name.endsWith(".jpg") ||
			name.endsWith(".gif") ||
			name.endsWith(".png")
		) {
			return true;
		}
		return false;
	}

	/**
	 * Checks that the file starts with one of the glyphs in the internal GlyphSet.
	 */
	protected boolean prefixOK(File file) {
		String[] pre = glyphSet.toArray();
		for ( int k = 0; k < pre.length; k++ ) {
			if ( GlyphImage.getGlyph(file).equals(pre[k]) ) {
				return true;
			}
		}
		return false;
	}

}


/* end GlyphFilter.java */