/*
 * JgGameClassic.java, part of JGlyphs
 * 
 * (c) 2004 Trevor "beltorak" Torrez
 * Released under the GPL, either version 2, or at your preference,
 * any later version.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package jgGame;

import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.Vector;

import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JFrame;

import jgGlyphs.GlyphImageList;
import jgGlyphs.GlyphImageWindow;
import jgGlyphs.GlyphSet;
import jgMain.JgMain;

/**
 * This is the class for the actual game
 * TODO: most of this needs to be super-classed to JgGame, which this
 * should then extend.
 */
public class JgGameClassic extends JFrame
	implements ActionListener
{

	/** Returns the string representation of this class */
	public String toString() {
		return "JgGameClassic";
	}

	/** The instance of this class */
	protected static JgGameClassic game;

	/** Represents the set of available glyphs */
	protected GlyphSet glyphs;

	/** The set extant windows */
	protected Vector windows;

	/** The available glyphs */
	protected GlyphImageList images;

	/** Constructor.  Creates the JFrame, initializes internal data. */
	protected JgGameClassic() {
		super("Java Glyphs");
		setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		glyphs = JgMain.getSettings().getGlyphSet();
		windows = new Vector(glyphs.size());
		images = new GlyphImageList();
	}
	
	/** This is how the game is started. */
	public static void begin() {
		game = new JgGameClassic();
		game.setData();
		game.addChildren();

		game.pack();
		game.setVisible(true);
	}

	/** 
	 * Returns the singleton instance of the game
	 * @return the singelton instance of the game
	 */
	public static JgGameClassic getGame() {
		if ( game == null ) {
			game = new JgGameClassic();
		}
		return game;
	}

	/** Sets internal data */
	public void setData() {
		glyphs = JgMain.getDataLoader().getAvailableGlyphs(JgMain.getSettings().getGlyphSet());
		images = JgMain.getDataLoader().getAvailableGlyphImages(glyphs);
	}

	/** Adds all children */
	public void addChildren() {
		Box mbox = Box.createVerticalBox();
		mbox.add(createGlyphButtons());
		mbox.add(Box.createVerticalGlue());
		mbox.add(createExitButton());
		getContentPane().add(mbox);
	}

	/**
	 * Creates the main point of the game, which is the buttons for the glyphs.
	 * Each button is placed in a row of ten, and the Box that surrounds the whole
	 * shebang is returned.
	 * @return a Container containing all the glyph buttons.
	 */
	protected Box createGlyphButtons() {
		Box box = Box.createVerticalBox();

		Box sBox = Box.createHorizontalBox();
		String[] ga = glyphs.toArray();
		for ( int k = 0; k < ga.length; k++ ) {
			if ( k % 10 == 0 ) {
				sBox.add(Box.createGlue());
				sBox = Box.createHorizontalBox();
				box.add(sBox);
			}
			JButton b = new JButton(ga[k]);
			// Set the size of the glyphs inside the buttons
			b.setFont(b.getFont().deriveFont((float) 30));
			b.setBackground(getRandomColor());
			b.addActionListener(this);
			sBox.add(b);
		}
		sBox.add(Box.createGlue());

		return box;
	}

	/**
	 * Used to get a psuedo-random color between the possibilities
	 * of red, blue, and green.
	 * @return a Color for red, blue, or green depending on chance (mostly).
	 */
	protected Color getRandomColor() {
		Color c;
		double r = Math.random();
		if ( r > .66 ) {
			c = new Color(127,127,255);
		} else if ( r > .33 ) {
			c = new Color(127,255,127);
		} else {
			c = new Color(255,127,127);
		}
		return c;
	}

	/**
	 * Creates the convenient means for quitting the game.
	 * @return a Container containing the exit button. 
	 */
	protected Box createExitButton() {
		Box box = Box.createHorizontalBox();
		JButton b = new JButton("Exit");
		b.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				System.exit(0);
			}
		});
		box.add(b);
		return box;
	}
	
	/**
	 * Locates a window in the internal Vector given the name (glyph) of the window.
	 * @param name the glyph pertaining to the window
	 * @return the GlyphImageWindow if one is currently extant, otherwise null.
	 */
	protected GlyphImageWindow findWindow(String name) {
		for ( int k = 0; k < windows.size(); k++ ) {
			if ( ((GlyphImageWindow) windows.get(k)).getName().equals(name) ) {
				return (GlyphImageWindow) windows.get(k);
			}
		}
		return null;
	}
	
	/** Removes a window by name (glyph) from the internal Vector. */
	public void removeWindow(String name) {
		for ( int k = 0; k < windows.size(); k++ ) {
			if ( ((GlyphImageWindow) windows.get(k)).getName().equals(name) ) {
				windows.remove(k);
			}
		}
	}
	
	/**
	 * The action performed when a glyph buton is pressed.
	 * Either finds the current window (if one is extant) and brings it to the front,
	 * or creates a new one (and adds it to the internal Vector of window lists).
	 * When creating the new window, it adds a callback to remove the window from
	 * the Vector when the window is closed.  This is kinda clunky since the window
	 * must remove (via the remove method above) itself it uses the dispose method.
	 */
	public void actionPerformed(ActionEvent e) {
		GlyphImageWindow window = findWindow(e.getActionCommand());
		if ( window == null ) {
			window = new GlyphImageWindow(images.getGlyphImagesForGlyph( e.getActionCommand() ));
			window.addWindowListener(new WindowAdapter() {
				public void windowClosing(WindowEvent e) {
					windows.remove(e.getComponent());
				}
			});
			windows.add(window);
		} else {
			window.toFront();
		}
	}

}


/* end JgGameClassic.java */