/*
 * JgDataLoader.java, part of JGlyphs
 * 
 * (c) 2004 Trevor "beltorak" Torrez
 * Released under the GPL, either version 2, or at your preference,
 * any later version.
 * 
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package jgDataLoader;

import java.io.File;
import java.io.FileFilter;
import java.util.regex.Pattern;

import jgGlyphs.GlyphFilter;
import jgGlyphs.GlyphImage;
import jgGlyphs.GlyphImageList;
import jgGlyphs.GlyphSet;
import jgMain.JgMain;

/**
 * The data loader is resposible for maintaing a list of catagory folder locations,
 * validating catagories, and loading media in the catagories.
 * TODO: If wrapped in a jar file, use a default catagory (directory) in the jar file.
 * TODO: Find out where Apple places user specific data, for both MAC OS X and
 * previous OS versions....
 */
public class DataLoader {
	/** Returns the string representation of this class */
	public String toString() {
		return "DataLoader";
	}

	/** Constructor.  Initializes the catagory locations. */
	public DataLoader() {
		catagoryList = new CatagoryList();
		globalDir = new File(System.getProperty("user.dir") + File.separator + "jglyphs.dat");
		addCatagoriesFromDir(globalDir);
		
		// Adds the user specific catagories (in the user's home dir)
		if ( File.separator.equals("\\") ) {
			// The Windows 9x variants have no true concept of separate users, so this
			// will be skipped.
			Pattern username = Pattern.compile(System.getProperty("user.name"), Pattern.CASE_INSENSITIVE);
			if ( username.matcher(System.getProperty("user.home")).matches() ) {
				userDir = new File(System.getProperty("user.home") + "\\appdata\\jglyphs");
				addCatagoriesFromDir(userDir);
			}
		} else if ( File.separator.equals("/") ) {
			userDir = new File(System.getProperty("user.home") + File.separator + ".jglyphs");
			addCatagoriesFromDir(userDir);
		} else if ( File.separator.equals(":") ) {
			// I'm not sure about macintosh fs details.  The system wide directory and the user-specified
			// directories will have to suffice.
		}
		includeDir = JgMain.getSettings().getIncludeDir(); 
		if (  includeDir != null ) {
			addCatagoriesFromDir(includeDir);
		}
	}

	/** Private instance of the catagory list */
	protected CatagoryList catagoryList;

	/** Global location of catagory folders */
	protected File globalDir;
	
	/** User specific location of catagory folders (in the user's home dir) */
	protected File userDir;

	/** User specified directory. */
	protected File includeDir;

	/** Checks to see if a catagory is valid */
	public boolean isCatagoryValid(String cat) {
		return catagoryList.contains(cat);
	}

	/**
	 * Adds catagory names to the list of catagories, using the directory entries
	 * in 'dir'.
	 * @param dir the directory to look in
	 */
	public void addCatagoriesFromDir(File dir) {
		if ( ! dir.isDirectory() ) {
			return;
		}
		File[] cat = dir.listFiles(new FileFilter() {
			public boolean accept(File pathname) {
				return pathname.isDirectory();
			}
		});
		for ( int k = 0; k < cat.length; k++ ) {
			catagoryList.add(cat[k].getName());
		}
	}
	
	/**
	 * Locates a catagory by name.  The search order is the 
	 * user specified directory (if applicable), the user specific
	 * directory (if applicable), and finally the global directory.
	 * The search is case sensative if the underlying file system is case sensative.
	 * @param name the name of the catagory.
	 * @return the File location of this catagory, or null if not found.
	 */
	public File locateCatagory(String name) {
		File loc;
		
		if ( includeDir != null ) {
			loc = new File(includeDir, name).getAbsoluteFile();
			if ( loc.isDirectory() ) {
				return loc;
			}
		}
		
		if ( userDir != null ) {
			loc = new File(userDir, name).getAbsoluteFile();
			if ( loc.isDirectory() ) {
				return loc;
			}
		}
		
		loc = new File(globalDir, name).getAbsoluteFile();
		if ( loc.isDirectory() ) {
			return loc;
		}
		
		System.out.println("Could not find catagory " + name);
		return null;
	}
	
	/**
	 * Retrieves the names of all the available catagories.
	 * @return a String array of catagory names.
	 */
	public String[] getCatagoryNames() {
		return catagoryList.getCatagories();
	}

	/**
	 * Retrieves a new GlyphSet indicating all available glyphs for the selected catagory,
	 * retstricted to the passed GlyphSet (if not empty).
	 * @param restrict the GlyphSet to restrict the retreival to.
	 * @return a new GlyphSet indicating all available glyphs for the selected catagory
	 */
	public GlyphSet getAvailableGlyphs(GlyphSet restrict) {

		GlyphSet gs = new GlyphSet();
		String[] catagories;
		if ( JgMain.getSettings().getCatagory() == null ) {
			catagories = getCatagoryNames();
		} else {
			catagories = new String[1];
			catagories[0] = JgMain.getSettings().getCatagory();
		}
		
		GlyphFilter gf = new GlyphFilter(restrict);
		for ( int k = 0; k < catagories.length; k++ ) {
			File catDir = locateCatagory(catagories[k]);
			File[] flist = catDir.listFiles(gf);
			for ( int j = 0; j < flist.length; j++ ) {
				String glyph = GlyphImage.getGlyph( flist[j] );
				gs.addGlyph( glyph );
			}
		}
		
		return gs;
	}
	
	/**
	 * Retrieves all available glyphs with no restrictions.
	 * @return a new GlyphSet indicating all available glyphs for the selected catagory
	 */
	public GlyphSet getAvailableGlyphs() {
		return getAvailableGlyphs(new GlyphSet());
	}

	/**
	 * Retrieves all available glyph images restricted to the passed GlyphSet
	 * @param restrict the GlyphSet to filter on
	 * @return a GlyphImageList containing all available glyphs images
	 */
	public GlyphImageList getAvailableGlyphImages(GlyphSet restrict) {
		GlyphImageList gil = new GlyphImageList();
		String[] cats;
		if ( JgMain.getSettings().getCatagory() == null ) {
			cats = getCatagoryNames();
		} else {
			cats = new String[1];
			cats[0] = JgMain.getSettings().getCatagory();
		}
		
		for ( int k = 0; k < cats.length; k++ ) {
			File[] gis = locateCatagory(cats[k]).listFiles(new GlyphFilter(restrict));
			for ( int j = 0; j < gis.length; j++ ) {
				gil.addGlyphImage(gis[j]);
			}
		}
		return gil;
	}
	
	/**
	 * Retrieves all available glyph images with no glyph restrictions
	 * @return a GlyphImageList containing all available glyph images
	 */
	public GlyphImageList getAvailableGlyphImages() {
		return getAvailableGlyphImages(new GlyphSet());
	}

}


/* end JgDataLoader.java */