;**************** READ HARD DRIVE SYSTEM CALL *****************
hdread:				;Reads/writes sectors from/to hard disk: cl=#sectors, ax=cylinder #, bl=sector #
				;ch = set bit7=write, bit4 = drive: 0=master, 1=slave <-????, low 4 bits head #
				;bh: bit7: 0=HardDiskController0, 1=HDC1
				;    bit4: 0=master, 1=slave
				;es:edi -> buffer, OR IF writing set: ds:esi as source...
	push fs
	push dword shared_data_sel
	pop fs			;This is just popping shared_data_sel, not the original fs

	test bh, 128
	jz hdread_hdc0port
		mov word [fs:temp_data5], 0170h
		jmp hdread_gotport
hdread_hdc0port:	mov word [fs:temp_data5], 01f0h
hdread_gotport:

	call hdread_wait4ready
	pushad
	pushfd		;Save the interrupt enable flag and the direction flag
	push cx
	push cx		;These are parameters that need to be saved
	push ax
	push bx

	cld			;Clear the direction pointer so the rep commands aren't messed up
	sti			;Allow interrupts so that 

	mov dx, 0a1h	;irq masking register: mask irq so no interrupt generated
	in al, dx
	test bh, 128
	jz hdread_dohdc0
		or al, 080h
		jmp hdread_outmask
hdread_dohdc0:
	or al, 040h	;01000000xb = mask irq14
hdread_outmask:	out dx, al

	;If there was a precompensation cylinder, it would go to port baseport + 1 (1f1h for hdc0)
	;but for now keep it simple. (precomp cylinder would be divided by 4)

	mov dx, word [fs:temp_data5]		;Set #sectors
	add dx, 2
	mov al, cl		;cl was passed as a parameter and cx hasn't been altered yet.
	out dx, al
	
	inc dl
	pop bx
	mov al, bl		;sector #
	out dx, al

	inc dl
	pop ax
	out dx, al		;cylinder low

	inc dl
	mov al, ah
	out dx, al		;cylinder high

	mov al, ch
	and al, 0bfh	;10111111xb = clear bit 6 = choose CHS
	or al, 0a0h	;=10100000xb = set bits 7 & 5 = mandatory to be set, no significance...
	inc dl		;bitmap = 1, CHS (LBA = 1), 1, Drive #, Head #
	out dx, al

	inc dl
	pop cx
	mov bh, ch		;DON'T USE BX UNTIL AFTER DATA IS READ/WRITTEN...!!!!!!!!!!!!!!!!!!!
	test ch, 080h
	jnz hdread_writecommand
	mov al, 020h
	or byte [fs:hd0_command], 080h	;set bit7 to indicate a read command is taking place
	jmp hdread_outputcommand
hdread_writecommand:    mov al, 030h    ;write sectors w/ retry command
	or byte [fs:hd0_command], 040h
hdread_outputcommand:   out dx, al
       
hdread_doread:
	call hdread_unmaskirq
	pop cx			;Get # sectors to set # loops
	and ecx, 000000ffh	;only count cl, not ecx b/c only up to 256 at a time
hdread_readingloop:
	push ecx		;Save the loop number

	mov dx, word [fs:temp_data5]
	add dx, 7
	test bh, 080h
	jz hdread_readpause	;If bit 7 isn't set it is a read command so do a pause for read
hdread_writepause:
	mov ecx, 10
	call sys_delay_csel:0
	test al, 1
	jz hdread_dataready	;If no error and delay time is up, continue with write
	jmp hdread_err2		;Hard drive returned an error
hdread_readpause:
	in al, dx
	test al, 08h    ;if 0 then data not ready(=10001000xb = busy&data req)
	jnz hdread_dataready
	test al, 1
	jz hdread_readpause	;If no error yet, continue cycle
	jmp hdread_err2		;Hard drive returned an error

hdread_dataready:
	mov dx, word [fs:temp_data5]	;Data Port
	mov ecx, 256		;256 = # words in a sector

	test bh, 080h		;= 10000000xb : if bit7 set then write, this was originaly in ch, but moved to save it.
	jz hdread_read		;If not set, read
	rep outsw		;Else write
	jmp hdread_wroteasector
hdread_read: rep insw		;Read a sector to es:edi
hdread_wroteasector:

	pop ecx			;Get loop number back
	loop hdread_readingloop

	mov byte [fs:hd0_command], 0	;Now that loop is over, make sure hd is registered as not busy.

	add dx, 7
	in al, dx
	and al, 1		;Check for an error
	jnz hdread_err3		;Some other error occurred

hdread_done:
	popfd
	popad
	pop fs
	clc
	retf
	;*********************** Called procedures ************************

hdread_wait4ready:
	pushad
	mov dx, word [fs:temp_data5]
	add dx, 7
	mov ecx, 5000		;Try no more than 5000x. (5000 x (1/100ms) = 50ms maximum wait)
hdread_waitloop:
	in al, dx
	test al, 080h
	jz hdread_ready	;If bit6 was set (drive ready) and controller isn't executing a command, it's ready
hdread_notyet:	
	push ecx
	mov ecx, 10
	call sys_delay_csel:0	;delay for 1/100ms
	pop ecx
	loop hdread_waitloop

hdread_err1:		;HARD DRIVE TOO BUSY ERROR: pushad hasn't occurred, it is just popping it from hdread_wait4read
	popad		;remove the data pushed from the beginning of hdread_wait4ready subroutine
	pop eax		;pop called eip for call hdread_wait4ready
	pop fs		;Pop the first thing on the stack (before hdread_wait4ready was called)

	stc		;store carry flag as a return error
	mov ax, 1	;ERROR: hard drive too busy
	retf         ;First pushad & pushfd don't have to be popped because hdread_wait4ready was called before the push...

hdread_ready:
	popad
      ret

	;********* more error handlers **********
hdread_err2:		;Hard drive not ready to transfer data
	call hdread_unmaskirq
	pop ecx		;ecx still hasn't been popped from the transfer loop
	popfd
	popad		;pop this from the beginning of hdread
	pop fs
	stc
	mov ax, 2
        retf
hdread_err3:		;More detailed error occurred with ata
	call hdread_unmaskirq
	pop ecx
	popfd
	popad
	pop fs
	stc
	mov ax, 3
        retf
	;*********** Unmask IRQ **************
hdread_unmaskirq:
	push eax
	in al, 0a1h
	cmp word [fs:temp_data5], 01f0h
	jz hdread_unmaskhdc0
		and al, 07fh
		jmp hdread_dounmask
hdread_unmaskhdc0:
	and al, 0bfh	;and with 10111111xb to clear bit 6
hdread_dounmask:	out 0a1h, al
	pop eax
	ret
end_hdread:
;******************** END OF HARD DISK READING SYSTEM CALL ********************

;******************** GET HARD DISK INFORMATION SYSTEM CALL *******************
hdinfo:				;Get hard disk info. and return a selector to the 0200h byte buffer via es
				;In: al: bit7: 0=HardDiskController0, 1=HDC1
				;	 bit4: 0=master, 1=slave
	pushad
	pushfd			;Save interrupt enable and direction flags

	cld				;Clear direction flag so rep commands aren't messed up
	;REMEMBER: Don't push es because it uses es to return a pointer

	mov bx, sys_hdinfo_dsel	;This sets up a pointer to the data area for the insw
	mov es, bx
	xor edi, edi
	mov bl, al	;Save low byte for later reference

	test al, 128
	jz hdinfo_hdc0
		mov word [es:current_hdc_base], 0170h
		jmp hdinfo_doit
hdinfo_hdc0:	mov word [es:current_hdc_base], 01f0h
hdinfo_doit:
	mov ecx, 32000	;max timeout
	mov ax, 080h	;mask with 10000000xb and wait until result is 0xb (not busy)
	mov dx, word [es:current_hdc_base]
	add dx, 7
	call hdinfo_wait4ready

	;This below is only for selecting drive, head doesn't matter for getting hd info
	dec dx		;drive/head register
	push ebx	;Save bl because it holds the original al
	and bl, 010h	;Only need bit4 = master/slave
	mov al, 0a0h	;=10100000xb=1,not LBA,1,drive0 - might change from the OR below,head0000
	or al, bl
	pop ebx
	out dx, al

	mov ecx, 32000
	mov ax, 040c0h	;1000000011000000xb=AND status reg w/ 11000000xb and wait until
	inc dx		;the result of this AND = 10000000xb (drive ready, not busy)
	call hdinfo_wait4ready

	;MASK irq14/15 (depends on fdc0 or fdc1) = Hard drive interrupts
	mov dx, 0a1h	;mask register
	in al, dx	;get current mask
	test bl, 128
	jz hdinfo_maskhdc0
		or al, 080h
		jmp hdinfo_domask
hdinfo_maskhdc0:	or al, 040h	;mask bit6 or 7 (int14: bits are arranged int0-int7 in bits 0-7 of port 21h)
hdinfo_domask:				;int8-int15 in bits 0-7 of port 0a1h
	out dx, al

	mov dx, word [es:current_hdc_base]
	add dx, 7
	mov al, 0ech
	cli		;We disable interrupts for transfer, but the old int status bit in eflags will be
			;restored at the end.
	out dx, al

	mov ax, 0808h	;mask with 00001000xb until it is equal. (Data request)
	mov ecx, 32000	;timeout. 
	call hdinfo_wait4ready	;Port is still set at 01f7h so it doesn't need to be changed.

	mov ecx, 0100h		;# of bytes in the info. block to be transferred
	mov dx, word [es:current_hdc_base]	;data port
	rep insw		;read all data to es:edi (which were set up at the beginning)

	add dl, 7
	mov ax, 080h		;mask with busy and wait until not busy
	mov ecx, 32000
	call hdinfo_wait4ready

	in al, dx	;read status
	test al, 1
	jnz hdinfo_err2

	mov dx, 0a1h
	in al, dx
	test bl, 128
	jz hdinfo_unmaskhdc0
		and al, 07fh
		jmp hdinfo_dounmask
hdinfo_unmaskhdc0:	and al, 0bfh	;10111111xb = unmask irq14
hdinfo_dounmask:
	out dx, al

	popfd
	popad
	clc
	retf
	;************** Calls *****************
hdinfo_wait4ready:	;Reads value from port in dx, ANDs it with the value passed to
	push eax	;the procedure in al and compares it to the value in ah. When the
	push ebx	;compare yields zero, it returns. This timeouts after a while.
			;Uses ecx as max # cycles
	mov bl, al
hdinfo_wait4ready_loop:
	in al, dx
	and al, bl
	cmp al, ah
	jz hdinfo_wait4ready_done
	loop hdinfo_wait4ready_loop

;ERROR1 if it made it through this loop. Error1 is also available for reporting in other parts of the code
;To allow this, the pops from hdinfo_wait4ready were place before the label

	pop ebx	;pop params from this procedure
	pop eax
	pop eax	;pop the eip near call to hdinfo_wait4ready from stack
hdinfo_err1:
	popfd
	popad	;pop from the beginning of hdinfo
	stc	;set carry to indicate failure
	mov ax, 1	;Error1 = HD took to long / too busy
	retf
hdinfo_wait4ready_done:
	pop ebx
	pop eax
	clc
	ret

hdinfo_err2:
	popfd
	popad
	stc
	mov ax, 2
	retf
end_hdinfo:
	;************* hdinfo DATA space ***************
hdinfo_data:			;For more info on this data, view hdinfo.txt
	temp_genconfig	equ	0
			dw	0
	temp_numcylinders	equ	$ - hdinfo_data
			dw	0
	temp_reserved	dw	0
	temp_numheads	equ	$ - hdinfo_data
			dw	0
	temp_oldstuff	dd	0
	temp_numsectors	equ	$ - hdinfo_data
			dw	0
        therest TIMES (0200h - 0eh) db 0        ;total length of buffer = 0200h
                        ;length of temp_genconfig to temp_numsectors = 0eh, so difference
                        ;is needed to be allocated
	current_hdc_base	equ	$ - hdinfo_data
			dw	0
end_hdinfo_data:

;********************* READ/WRITE TO FLOPPY SYSTEM CALL ***************************
fd_rw:		;in:    al bits: bit7 set if Multi-track, MFM encoding, skip deleted data address mark, bits4->0 are:
		;		read: al will probably be 0e6h=11100110xb
		;		write: al will probably be 0e5h=11100101xb
		;	ah bits: 00000, bit2=head #, bits1-0 = floppy drive #
		;	cl = # sectors (!!!cannot be > 63 b/c 63=7e00h / 512(original_offset / 512)), ch=cylinder#, bl=sector#
		;	out: transfer & !!!!!not yet: ecx = number of sectors actually transferred.
	push eax
	push edx
	push ebx
	push edi
	push fs
	pushfd		;Save flags just in case interrupts weren't enabled (we're going to enable them)

	push eax
	push ecx
fd_rw_ecx_location:
	push eax
	push eax	;NOTE: relying on ecx and ebx not to be altered - watch not to interfere
fd_rw_ecx_offset	equ	$ - fd_rw_ecx_location

	mov ax, shared_data_sel
	mov fs, ax

	sti
	in al, 021h
	and al, 0bfh	;clear bit6 = irq6 = floppy irq mask
	out 021h, al	;This makes sure the floppy irq isn't masked

	;***** See if number of sectors to be transferred is too many.
	cmp cl, 64
	jb fd_rw_not_toomanysectors	;If it isn't, continue.
	mov cl, 63				;Otherwise, allow max number
	mov byte [ss:esp+fd_rw_ecx_offset], cl	;put new # sectors into the pushed value of ecx
fd_rw_not_toomanysectors:

	;***** Figure out which drive is being called on and turn on the corresponding drive's motor
	pop eax
	and ah, 03h		;Only need last to bits (drive select)
	jz fd_rw_fd0on
	cmp ah, 1
	je fd_rw_fd1on
	cmp ah, 2
	je fd_rw_fd2on	;if passes here it must be drive3

fd_rw_fd3on:
	mov al, 08fh	;Set bit3 of high nibble al to indicate fd0 motor on; fxb = 1111=dma,fd enable, select drive 11 = 3
	jmp fd_rw_fd_on
fd_rw_fd0on:
	mov al, 01ch	;Set bit0 of high nibble al to indicate fd0 motor on; cxb = 1100 =dma,fd enable, select drive 00=0
	jmp fd_rw_fd_on
fd_rw_fd1on:
	mov al, 02dh	;Set bit1 of high nibble al to indicate fd1 motor on; dxb = 1101 =dma,fd enable, select drive 01=1
	jmp fd_rw_fd_on
fd_rw_fd2on:
	mov al, 04eh	;Set bit2 of high nibble al to indicate fd2 motor on; exb = 1110 =dma,fd enable, select drive 10=2

	;***** Have drive, set motors accordingly
fd_rw_fd_on:
	or al, 4	;Need to set bit2 so that drive is enabled
	or al, byte [fs:fd_motors]	;Set bits in al of any other fd motors on
	call fd_rw_motorset		;Set motor running status
	and al, 0f0h			;Get the motor status byte alone w/out dma, fd enable, etc.
	or byte [fs:fd_motors], al	;set the bit in fd_motors to indicate which motors are on

	;***** If there has been a disk change, send recalibrate command and then continue, if not jump over the recal code
	mov edx, 03f7h
	in al, dx
	test al, 080h
	jz fd_rw_notnewdisk

	or byte [fs:fd_command], 32	;set seeking busy bit so irq will have to reset it
	mov al, 07h			;before fd_rw_wait4int lets this code continue
	call fd_rw_sendbyte	;recalibrate command
	mov al, 0
	call fd_rw_sendbyte	;a zero that goes with the recalibrate command

	call fd_rw_wait4int	;wait until fd irq clears the bit
	;***** Seek to cylinder
fd_rw_notnewdisk:
	xor eax, eax
	mov al, ch	;get cylinder# into al
	call fd_rw_seek
	;***** Set Data Rate
	mov edx, 03f7h	;configuration control register
	mov al, 0
	out dx, al	;This sets data rate to 500K/s

	;***** Prepare to set up a DMA channel: convert #sectors into #bytes
		;Don't worry about ecx here, it was pushed at beginning and will be popped after dma channel is set up.
	and ecx, 0ffh			;get only cl (= # sectors)
	shl ecx, 9				;equivalent to multiplying by 512
	;***** Now, find out if it is a read to floppy or write to floppy operation
	pop eax
	and al, 01fh	;=00011111xb: read command will be ???00110xb, write command will be 00101xb
	cmp al, 06h
	jz fd_rw_dmaread
	cmp al, 05h
	jz fd_rw_dmawrite
	jmp fd_rw_err1
	;!!!!! NOTE: A floppy read operation is matched with a DMA write operation (read from floppy & dma writes to mem)
fd_rw_dmaread:				;ecx is already set up
	mov eax, 0602h			;ah:00=demand,0=address increment,0=auto initializing,01=write to memory,10=channel2
						;al:channel 2 is floppy channel
	mov edi, 0				;offset 0: transfer right to beginning of memory
	call sys_dma_xfer_csel:0
	or byte [fs:fd_command], 128	;indicate read operation in progress
	jmp fd_rw_continue
fd_rw_dmawrite:
	mov eax, 0a02h			;ah=00 0 0 10 10xb:demand,address inc,auto init,read from mem,channel2; al=ch2
	mov edi, 0				;offset 0: transfer starting from first sector in memory
	call sys_dma_xfer_csel:0
	or byte [fs:fd_command], 64
fd_rw_continue:

	pop ecx		;Get original ecx value back (If cl (#sectors) was > 64 it will be 63 now (max allowed))
	pop eax		;Get command

	call fd_rw_sendbyte	;Send it

	mov al, ah	;Put head, floppy drive select into al
	call fd_rw_sendbyte

	mov al, ch	;Get cylinder# into al
	call fd_rw_sendbyte

	mov al, ah	;Put head back into al, but this time only head...
	shr al, 2	;put head at beginning of al b/c only head (not drive select also)
	call fd_rw_sendbyte

	mov al, bl			;Sector#
	call fd_rw_sendbyte

	mov al, 2
	call fd_rw_sendbyte	;2 means 2x100h = 200h = 512 bytes per sector

	mov al, 36			;18/cylinder, 36/track
	call fd_rw_sendbyte	;sectors per track

	mov al, 01bh		;the "gap3 while reading/writing a 1.44MB floppy according to GAZ
	call fd_rw_sendbyte

	mov al, 0ffh
	call fd_rw_sendbyte	;This is the Data Length - only used if bytes/sector was set to 0

	call fd_rw_wait4int	;WAIT until interrupt says this is done...

	call fd_rw_getbyte	;Get the 1st result byte of 7
	push eax			;Save this to analyze later
	call fd_rw_getbyte	;Get 2nd result byte of 7 bytes total
	push eax

	mov ecx, 5		;Get the last 5 result bytes of 7 total (view hardware.txt in helppc21.zip for more info.)
fd_rw_gettingresults:
	call fd_rw_getbyte
	loop fd_rw_gettingresults

	pop eax			;Get status byte 1 back
	test al, 2			;If bit1 is set controller got a write protect error
	jnz fd_rw_err3		;Disk is write protected
	test al, 128		;If bit7 is set, the sector# it tried to read from was > #sectors/track
	jnz fd_rw_err4

	pop eax			;Get status byte 0 back
	and al, 0c0h		;only need bits 7-6
	jz fd_rw_success		;If these are 00 then there was no error
	cmp al, 128			;If only bit7 set, it was an invalid command
	jz fd_rw_err1_after_pops
	jmp fd_rw_err2

fd_rw_success:
	mov ecx, 15000		;give it 15ms just in case
	call sys_delay_csel:0

	mov al, 0ch		;Turn off motors
	call fd_rw_motorset
	mov byte [fs:fd_motors], 0

	popfd
	pop fs
	pop edi
	pop ebx
	pop edx
	pop eax
	retf
	;*********** Error 1 = invalid command
fd_rw_err1:
	pop ecx	;These two hadn't been popped at the first check for err1
	pop eax
fd_rw_err1_after_pops:
	popfd
	pop fs
	pop edi
	pop ebx
	pop edx
	pop eax
      stc
      mov ax, 1
      retf
	;*********** Error 2 = operation terminated abnormally
fd_rw_err2:
	popfd
	pop fs
	pop edi
	pop ebx
	pop edx
	pop eax
	stc
	mov ax, 2
	retf
	;*********** Error 3 = write operation couldn't finish b/c write protected
fd_rw_err3:
	pop eax
	popfd
	pop fs
	pop edi
	pop ebx
	pop edx
	pop eax
	stc
	mov ax, 3
	retf
	;*********** Error 4 = sector# it tried to read from was > #sectors/track
fd_rw_err4:
	pop eax
	popfd
	pop fs
	pop edi
	pop ebx
	pop edx
	pop eax
	stc
	mov ax, 4
	retf
	;**************
fd_rw_sendbyte:		;in: al = byte to send
	push edx	;out: sends byte to port 03f5h = diskette command register
	push eax
fd_rw_sendbyte_loop:
	mov dx, 03f4h
	in al, dx	;get bit7=if set, data ready; bit6=if set, data from FPC to CPU (bad)
	and al, 0c0h	;test bits 7 & 6
	cmp al, 080h
	jz fd_rw_sendit
	mov dx, 080h
	in al, dx
	jmp fd_rw_sendbyte_loop
fd_rw_sendit:
	pop eax
	mov dx, 03f5h
	out dx, al
	pop edx
	ret
	;**************
fd_rw_getbyte:		;in: nothing
	push edx	;out: byte from data register (3f5h) in al
fd_rw_getbyte_loop:
	mov dx, 03f4h
	in al, dx
	and al, 0d0h
	cmp al, 0d0h
	jz fd_rw_getit
	mov dx, 080h
	in al, dx
	jmp fd_rw_getbyte_loop
fd_rw_getit:
	mov dx, 03f5h
	in al, dx
	pop edx
	ret
	;**************
fd_rw_wait4int:			;Wait until the last floppy operation has completed
	push eax
	push ecx
        push edx
        mov edx, 01f4h
fd_rw_wait4int2:
	mov ecx, 0ffffh
fd_rw_wait4int3:
	cmp byte [fs:fd_command], 0
	jz fd_rw_done_waiting
	loop fd_rw_wait4int3
        in al, dx
	test al, 01fh
	jnz fd_rw_wait4int2
fd_rw_done_waiting:
        pop edx
	pop ecx
	pop eax
	ret
	;**************
fd_rw_motorset:		;in: al set as explained below
	push ecx
	push edx
	mov edx, 03f2h	;Digital output register bits7-4=motor on for fd drives D-A respectively
			;bit3=DMA enable, bit2=FD controller enable, bits1-0=current drive selected
        out dx, al
	mov ecx, 500000
	call sys_delay_csel:0	;delay for 500ms for motor to warm up
	pop edx
	pop ecx
	ret
	;**************
fd_rw_seek:		;in: motor must be on, ch = cylinder%
	push ecx
	or byte [fs:fd_command], 32

	mov al, 0fh	;seek command
	call fd_rw_sendbyte
	mov al, 0		;head 0 no matter what b/c it can read from both sides of disk anyway without seeking
	call fd_rw_sendbyte
	mov al, ch		;Get cylinder# into al
	call fd_rw_sendbyte

	call fd_rw_wait4int

	mov ecx, 15000
	call sys_delay_csel:0	;delay for 15ms to let head settle
	pop ecx
	ret
end_fd_rw:

;***************************** Convert LBA to sys_hdread_csel CHS format for a hard disk *****************************
LBA_CHS:		;in:	eax = LBA #
			;	ch = set bit7=write, bit4 = drive: 0=master, 1=slave <-????, low 4 bits head #
			;	bh: 	bit7: 0=HardDiskController0, 1=HDC1
			;    		bit4: 0=master, 1=slave
			;out:	CHS in sys_hdread_csel format (ax, ch, bx are set up)
			;	cl = max # sectors that can be read/written from this location
			;	NOTE: cl may exceed 63 sectors, which is the max. that hdread can read from/write to
			;		at a time.
	push fs
	push esi
	push edx
	push word shared_data_sel
	pop fs
	push ebx	;These params are access before the end of the code...
	push ecx
	push eax

	test bh, 80h
	jnz LBA_CHS_hdc1
	test bh, 10h
	jnz LBA_CHS_hdc0_slave

		;must be hdc0 master
	mov esi, hd0_numheads
	jmp LBA_CHS_gothdinfo
LBA_CHS_hdc0_slave:
	mov esi, hd1_numheads
	jmp LBA_CHS_gothdinfo
LBA_CHS_hdc1:
	test bh, 10h
	jnz LBA_CHS_hdc1_slave

		;must be hdc1 master
	mov esi, hd2_numheads
	jmp LBA_CHS_gothdinfo
LBA_CHS_hdc1_slave:
	mov esi, hd3_numheads
LBA_CHS_gothdinfo:
	mov bx, word [fs:esi]		;#heads
	mov ax, word [fs:esi + 4]	;#sectors
	and eax, 0ffffh
	and ebx, 0ffffh
	mul ebx			;Get #sectors/cylinder
	mov ebx, eax		;Put this into ebx
	pop eax			;Pop eax = first param (first available on stack)
	div ebx			;Get cylinder# into eax and remaining sectors after that cylinder into edx (remainder)
	mov ecx, eax		;Save cylinder# in ecx because ecx is on the stack and we'll get it back in the end, but
				;for now we need a place to save the cylinder#
	mov eax, edx		;Get remaining sector offset after that cylinder
	mov bx, word [fs:esi + 4]	;#sectors/head
	and ebx, 0ffffh
	div ebx			;Get head# into eax and sector number into edx

	cmp dl, 0	;If sector # is 0, then it must have been the last sector on the last head or cylinder
	jnz LBA_CHS_setup
		cmp eax, 0	;If head isn't zero then we only have to change the head
		jnz LBA_CHS_onlychghead	;If it is zero, we have to change the cylinder also
			dec ecx
			mov ax, word [fs:esi]	;Get #heads into ax, so the dec will bring it back 1
LBA_CHS_onlychghead:
	dec eax		;decrement the head #
	mov dx, word [fs:esi + 4]	;Get the #sectors into dl b/c it was 0 b/c it was on the last sector
LBA_CHS_setup:
	mov ebx, ecx		;Save cylinder#
	pop ecx			;Get ch param
	and ch, 0f0h		;Only keep high 4 bits of it
	and al, 0fh		;Get head number into low 4bits of al
	or ch, al		;Set the corresponding head bits
	mov eax, ebx		;Get cylinder# back into ax
	pop ebx			;Get last param
	mov bl, dl		;Put sector# into bl and save bh as it is
	mov dx, word [fs:esi + 4]
	sub dl, bl		;Get # sectors/head and subtract the starting sector to get the max. # sectors that can
				;be written to/read from disk when bl is the first sector to read/write from/to.
	inc dl			;If 63 sectors/head and starting at 1, we can read/write 63 not 63 - 1, so we have to add 1
	mov cl, dl		;put this into cl and return

	pop edx
	pop esi
	pop fs
	retf
end_LBA_CHS:
