#!/usr/bin/guile -s
!#

;;; EXPRESIONES

;; una expresion puede ser
;;   (1) una constante
;;   (2) una variable
;;   (3) una aplicacion compuesta por n expresiones
;;       ordenadas. para una expresion e, para todo
;;       i entero, 0 <= i < n, esta definida la expresion
;;       e[i] (la i-esima componente de e)

;; predicados sobre expresiones

(define (variable? expresion)
	(and (symbol? expresion)
			 (char-upper-case? (string-ref (symbol->string expresion) 0))))

(define (aplicacion? expresion)
 (pair? expresion))

(define (constante? expresion)
	(not (or (variable? expresion) (pair? expresion))))

;;; ENTORNOS

;; un entorno es una lista de pares (variable . expresion).
;; una variable esta ligada en un entorno e cuando es
;; la primera componente de alguno de los pares en e.
;; el valor de la variable v en un entorno e es la segunda
;; componente del par p tal que:
;;   p esta en e
;;   v es la primera componente de p
;;   p es el primer par en e tal que v es su primera componente

;; si la variable esta ligada en el entorno, devuelve el 
;; valor de la variable en el entorno.
;; si no esta ligada devuelve #f
(define (valor entorno variable)
 (let ((it (assq variable entorno)))
	(and it (cdr it))))

;;; REGLAMENTOS

;; un reglamento es una lista de definiciones
;; una definicion es un par de expresiones (e1 . e2) que satisface:
;;   (1) en e2 solo aparecen variables que aparecen en e1
;; e1 es la expresion que se define (definiendum)
;; e2 es la expresion que define (definiens)

;;; UNIFICACION

;; unifica dos expresiones
;; e1 y e2 unifican si:
;;  (1) ambas son constantes iguales
;;  (2) e1 es una variable
;;  (3) e1 y e2 son aplicaciones de tamao n
;;      y para todo i entero, 0 <= i < n
;;      e1[i] y e2[i] unifican
;; si e1 y e2 no unifican, devuelve #f
;; si e1 y e2 unifican, devuelve un entorno
;;   donde las variables en e1 estan ligadas
;;   a las subexpresiones de e2 que se encuentran
;;   en las posiciones correspondientes
;; si una variable aparece dos veces en
;;   e1, las expresiones unifican si y solo si
;;   las subexpresiones de e2 correspondientes
;;   son iguales entre si (igualdad de observadores,
;;   no identidad)
(define (unificar e1 e2)
	(let ((resultado '()))
		(and
			(let aux ((e1 e1) (e2 e2))
				(cond
					((constante? e1) (eq? e1 e2))
					((variable? e1)
					 (let ((estaba? (assoc e1 resultado)))
						(cond
						 ((not estaba?)
							(set! resultado (acons e1 e2 resultado)))
						 (else (equal? (cdr estaba?) e2)))))
					((aplicacion? e1)
					 (and (aplicacion? e2)
								(let unificar-aplicaciones ((e1 e1) (e2 e2))
									(cond
										((null? e1) (null? e2))
										((null? e2) #f)
										(else (and (aux (car e1) (car e2))
															 (unificar-aplicaciones (cdr e1) (cdr e2))))))))))
			resultado)))

;; resuelve la expresion dada en el entorno, i.e.
;; reemplaza todas las variables en la expresion
;; por el valor de dichas variables en el entorno.
;; PRECONDICION: todas las variables en la expresion
;; estan ligadas en el entorno
(define (resolver expresion entorno)
 (let aux ((expr expresion))
	 (cond
		 ((constante? expr) expr)
		 ((variable? expr) (valor entorno expr))
		 ((aplicacion? expr) (map aux expr)))))

;; devuelve un par: la primera definicion de
;; un reglamento cuyo definiendum unifica con la
;; expresion dada, y el entorno que resulta de la
;; unificacion.
;; de no ser posible, devuelve #f.
(define (primera-definicion-que-unifica reglamento expresion)
 (if (null? reglamento)
	 #f
	(let ((unificacion (unificar (caar reglamento) expresion)))
		(if unificacion
		 (cons (car reglamento) unificacion)
		 (primera-definicion-que-unifica (cdr reglamento) expresion)))))

;; devuelve la expresion despues de un paso de expansion, i.e.
;; si en el reglamento dado hay alguna definicion 
;; cuyo definiendum unifica con la expresion, devuelve
;; el resultado de resolver el definiens en el entorno
;; resultante de la unificacion.
;; en caso contrario, la expresion es inexpansible en 
;; el reglamento y devuelve #f.
(define (expandir1 reglamento expresion)
 (let ((definicion/entorno
				(primera-definicion-que-unifica reglamento expresion)))
	(and definicion/entorno
			 (resolver (cdar definicion/entorno) (cdr definicion/entorno)))))

;; da pasos sobre la expresion en el reglamento
;; hasta que ya no se pueden dar pasos
(define (maximizar reglamento expresion paso)
 (let ((it (paso reglamento expresion)))
	(if it
	 (maximizar reglamento it paso)
	 expresion)))

;; expande maximalmente la expresion en el reglamento
(define (expandir reglamento expresion)
 (maximizar reglamento expresion expandir1))

;; un paso de reduccion es la expansion de una
;; expresion hasta hacerla inexpansible.
;; primero intenta expandir la expresion
;; en caso de ser imposible, lo intenta con
;; sus subexpresiones.
;; si todas las expresiones estan expandidas
;; al maximo, la expresion es irreducible y
;; devuelve #f.
(define (reducir1 reglamento expresion)
 (let ((nueva (expandir reglamento expresion)))
	(if (equal? expresion nueva)
	  ; si la expansion es igual a la expresion
		(cond
		 ; si es una aplicacion
		 ((aplicacion? expresion)
			(let aux ((expr expresion) (acc '()))
			 (if (null? expr)
				 #f
				(let ((r (reducir1 reglamento (car expr))))
				 (if r
					(append
						(reverse acc)
						(list r)
						(cdr expr))
					(aux (cdr expr) (cons (car expr) acc)))))))
		 ; si no es una aplicacion, la expresion es irreducible
		 (else #f))
	  ; si la expansion no es igual a la expresion
		; devuelve la expansion
		nueva)))

;; reduce maximalmente la expresion en el reglamento
(define (reducir reglamento expresion)
 (maximizar reglamento expresion reducir1))

;; lee un reglamento de un archivo en la forma:
;;  [definiendum definiens]*
(define (leer-reglamento nombre-archivo)

 (define (leer-pares f)
	 (let ((r1 (read f)))
		 (and (not (eof-object? r1))
					(let ((r2 (read f)))
						(and (not (eof-object? r2))
								 (cons r1 r2))))))

 (let ((reglamento '()))
	 (call-with-input-file nombre-archivo
		 (lambda (f)
			(let agregar-pares ((pares (leer-pares f)))
				(if pares
				 (begin
					 (set! reglamento (cons pares reglamento))
					 (agregar-pares (leer-pares f)))))))
	 (reverse! reglamento)))

;; principal -- si se pasa un argumento,
;; lee un reglamento del archivo y corre
;; un REPL
(define (main argumentos)
 (if (null? argumentos)
	(format #t "Uso: hume.scm <archivo>\n")
	(let repl ((reglamento (leer-reglamento (car argumentos))))
	 (format #t "hume> ")
	 (let ((r (read)))
		(if (eof-object? r)
		 (exit)
		 (format #t "~S\n" (reducir reglamento r))))
	 (repl reglamento))))

(main (cdr (program-arguments)))

;(define reglas (leer-reglamento "berk"))
