/*
* SocketTunnel - Socket Proxies to create services proxies that can 
* be controlled by application test suites.
*
* Copyright (C) 2004  Abhilash Koneri
* 
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
package net.sf.jshell.socketUtil;

import java.net.*;
import java.io.*;

/** 
 * <code>SocketTunnel<code> creates a proxy to the socket that hosts a service.
 * Instances of this class, bind to a port on the local machines and stream the
 * data to the real <code>InetAddress</code> where the service is being hosted.
 * The proxy does not exist beyond the first connection by the client. 
 * <p>
 * Instances of this class can be controlled by the client to restart after 
 * an interval of time. They can be used to test application behavior on 
 * service outages.
 * </p>
 *
 * <u>Usage</u><br>
 *
 * <pre>
 * 
 *  public class MyAppTest
 *  {
 *
 *   public void test1()
 *   {
 *
 *      .
 *      .
 *      .
 *      SocketTunnel st = new SocketTunnel("server.acme.com",22, 8000);
 *      
 *      // MyApp myApp = new MyApp("localhost",8000);
 *      // myApp.doLotsOfWork();
 *      .
 *      .
 *      st.restart(40);
 *
 *      // myApp.doLotsOfWork();
 *    }
 *   }
 *
 *
 * </pre>
 * 
 * @author  Abhilash Koneri 
 * @version 0.2
 */
public class SocketTunnel implements Runnable
{
    
    /** 
     * ServerSocket to receive client connections.
     */
    ServerSocket serverSocket;
    
    /** 
     * Client socket to the real service. 
     */
    Socket socketToServer;
    
    /** 
     * Socket from the client software. 
     */
    Socket socketFromClient;

    /** 
     * Hostname of the server hosting the "real" service. 
     */
    String realHost;
    
    /** 
     * Port number on the host where the "real" service exists. 
     */
    int realPort;
    
    /** 
     * Port number used to bind the server socket. 
     */
    int localPort;
    
    /** 
     * Time (in seconds) before the service proxy is restarted. 
     */
    int sleepTime;
    
    /** 
     * Data tunnel to  the real service.
     */
    DataTunnel dt1;
    
    /** 
     * Data tunnel to the client. 
     */
    DataTunnel dt2;

    /** 
     * Constructor : creates instances of this class with the name of the host where
     * the real service exists and its port parameters. 
     * 
     * @param realHost Name of the host where the real services is hosted.
     * @param realPort Port on which the real service is hosted.
     * @param localPort Port used to proxy the service.
     */
    protected SocketTunnel(String realHost, int realPort, int localPort)
    {
        this.realHost = realHost;
        this.realPort = realPort;
        this.localPort = localPort;
        this.sleepTime = -1;
    }

    /** 
     * The main routine of the tread, binds to the proxy port on the localhost. On receipt
     * of a client connection, it starts the data channels between the proxy and
     * the real service and between the proxy and the client. The server socket is stopped
     * after the first connection.
     *
     */
    public void run()
    {
        try
        {
            dt1 = new DataTunnel("ToReal:");
            dt2 = new DataTunnel("ToClient:");

            if(sleepTime > 0)
            {
                Thread.sleep(sleepTime*1000);
                sleepTime = 0;
            }

            serverSocket = new ServerSocket(localPort);
            InputStream forwardHostInputStream = null;
            OutputStream forwardHostOutputStream = null;
            socketToServer = new Socket(realHost, realPort);
            
            Log.log("Binding to proxy port "+localPort);
            socketFromClient = serverSocket.accept();
            Log.log("Received socket "+socketFromClient);

            InputStream inputStream1 = socketFromClient.getInputStream();
            forwardHostOutputStream = socketToServer.getOutputStream();
            dt1.pumpData(inputStream1, forwardHostOutputStream);

            forwardHostInputStream = socketToServer.getInputStream();
            OutputStream outputStream2 = socketFromClient.getOutputStream();
            dt2.pumpData(forwardHostInputStream, outputStream2);
            serverSocket.close();
        }
        catch(Exception e)
        {
            Log.log("Main tunnel crashed!");
            Log.log(e);
        }

    }

    /** 
     * Starts the Socket tunnel on the proxy port on a child daemon thread. 
     */
    public void start()
    {
        Thread th = new Thread(this);
        th.setDaemon(true);
        th.start();
    }

    /** 
     * Kills the running channels between the proxy, the real server and
     * the client and schedules the proxy to restart after the number of
     * seconds.  
     * 
     * @param secondsToRestart  Number of seconds after which the proxy
     *                          is available
     * @throws IOException  is thrown if the service cannot be started.
     */
    public void restart(int secondsToRestart)  throws IOException
    {
        if(this.serverSocket != null)
        {
            this.sleepTime = secondsToRestart;
            kill();
            Thread th = new Thread(this);
            th.setDaemon(true);
            th.start();
        }
    }

    /** 
     * Kills the running channels between the proxy, the real server
     * and the client. 
     * 
     * @throws IOException If the channels cannot be killed.
     */
    public void kill() throws IOException
    {
        if(this.serverSocket != null)
        {
            if(socketFromClient != null)
                socketFromClient.close();
            socketToServer.close();
            serverSocket.close();
            dt1.stopTunnel();
            dt2.stopTunnel();
        }
    }


    public static void main(String[] args) throws Throwable
    {
        if(args.length < 3)
        {
            Log.log("Usage: java SocketTunnel <forwarding host> <port on forwarding host> <local port>");
            System.exit(1);
        }
        SocketTunnel st = new SocketTunnel(args[0], 
                Integer.parseInt(args[1].trim()), 
                Integer.parseInt(args[2].trim()));
        st.start();

        while(true)
        {
            Thread.sleep(20000);
            Log.log("restarting service in 5 seconds");
            st.restart(5);
        }
    }
}

/** 
 * <code>DataTunnel</code> is used to pipe the streams between the client
 * and the real service. 
 * 
 * @author Abhilash Koneri (Abhilash.Koneri@gmail.com)
 * @version 0.2
 */
class DataTunnel extends Thread
{
    
    /** 
     * Input stream.
     */
    InputStream inputStream;
    
    /** 
     * Output stream. 
     */
    OutputStream outputStream;
    
    /** 
     * Name of the channel. 
     */
    String tunnelName;
    
    /** 
     * Flag to stop the treads running the channel. 
     */
    boolean stop;

    /** 
     * Constructor used to create channels with a specified name. 
     * 
     * @param tunnelName Name of the channel.
     */
    DataTunnel(String tunnelName)
    {
        this.tunnelName = tunnelName;
    }

    /** 
     * Moves data from the input stream to the output stream in a seperate
     * daemon thread.
     * 
     * @param inputStream Input stream
     * @param outputStream Output stream
     */
    void pumpData(InputStream inputStream,
                     OutputStream outputStream)
    {
        this.inputStream = inputStream;
        this.outputStream = outputStream;
        setDaemon(true);
        start();
    }

    /** 
     * Stops the data streaming. Input and output stream are closed. 
     */
    void stopTunnel()
    {
        this.stop = true;
        try
        {
            if(inputStream != null)
                inputStream.close();
            if(outputStream != null)
                outputStream.close();
        }
        catch(IOException ioE)
        {
            Log.log(ioE);
        }
    }

    /** 
     * Data from the input stream is piped to the output stream. <i>Buffer size of 4K is used
     * </i> 
     */
    public void run()
    {
        int bytesRead;
        byte[] data = new byte[4096];

        while(!stop)
        {
            try
            {
                while((bytesRead = inputStream.read(data)) != -1)
                {
                    Log.log(tunnelName+"::Sending "+bytesRead);
                    outputStream.write(data, 0, bytesRead);
                }
                outputStream.flush();
            }
            catch(IOException ioE)
            {
                Log.log(tunnelName+"errored!");
                Log.log(ioE);
            }
            finally
            {
                try
                {
                    if(outputStream != null)
                        outputStream.close();
                    if(inputStream != null)
                        inputStream.close();
                    outputStream = null;
                    inputStream = null;
                }
                catch(IOException ioE)
                {
                }
            }
        }
        Log.log("Thread completed!");
    }
}

/** 
 * Logging used by the socket tunnel. Contains static methods to log 
 * messages into the console in debug mode. 
 * 
 * @author Abhilash Koneri (Abhilash.Koneri@gmail.com)
 * @version 0.2
 */
class Log 
{
    
    /** 
     * Debug flag. 
     */
    public static boolean debug = true;

    /** 
     * Logs message to the console in debug mode. 
     * 
     * @param s 
     */
    static void log(String s)
    {
        if(debug)
            System.out.println(s);
    }

    /** 
     * Logs the stack trace to the console in debug mode. 
     * 
     * @param e 
     */
    static void log(Throwable e)
    {
        if(debug)
            e.printStackTrace(System.out);
    }
}
