/*
 *                          The AVRIL System
 *
 *  AVRIL stands for A Virtual Reality Interface Library.  It's a library
 *  of routines for doing fast, polygon-based rendering on a variety of
 *  platforms.  It also provides support for device i/o, and will provide
 *  simulation,  user interface, interaction detection and more.
 *
 *  It's designed to be easy to use, portable, and fast.
 *
 */

/* Copyright 1994 by Bernie Roehl */

/*
   You may use this code for your own non-commercial projects without
   paying any fees or royalties.  "Non-commercial", in this context,
   means that the software you write is given away for free to anyone
   who wants it.
   
   Commercial use, including shareware, requires a licensing
   fee and a specific written agreement with the author.

   All programs created using this software (both commercial and
   non-commercial) must acknowledge the use of the AVRIL library,
   both in the documentation and in a banner screen at the start or
   end of the program.

   For more information, contact Bernie Roehl (broehl@uwaterloo.ca).

*/

/* This is version 2.0 -- January, 1995 */

/* Three special data types are used in AVRIL:

        vrl_Scalar  -- a measure of virtual distance
        vrl_Factor  -- a multiplication factor, usually in the range -1 to +1
        vrl_Angle   -- measured in 65536th's of a degree

   In the floating-point version of the code, they're all floats.  In
   the fixed-point version, they're all stored in a 32-bit word; they
   use differing numbers of bits for the integer and fractional parts of
   their values.
 */

#define VRL_VERSION 2

#define VRL_PC_COMPATABLE 1
#define VRL_USE_FIXED_POINT 1

#include <stdio.h>      /* FILE */
#ifdef __GNUC__
#include <djgppstd.h>   /* memcpy() */
#else
#include <string.h>     /* memcpy() */
#endif

typedef short int vrl_Boolean;    /* zero or non-zero */
typedef short int vrl_16bit;      /* a 16 bit integer */
typedef long int vrl_32bit;       /* a 32 bit integer */
typedef unsigned short int vrl_unsigned16bit;   /* a 16-bit unsigned integer */
typedef unsigned long int vrl_unsigned32bit;    /* a 32-bit unsigned integer */

#define abs32(v) labs(v)          /* long absolute value */

#ifdef VRL_USE_FIXED_POINT
typedef vrl_32bit vrl_Scalar;  /* 32.0 */
typedef vrl_32bit vrl_Factor;  /* 3.29 */
#define VRL_UNITY 536870912L   /* 2**29 */
typedef vrl_32bit vrl_Angle;   /* 16.16 */
#define VRL_ANGLECONVERSION 65536  /* 2**16 */
vrl_Factor vrl_ScalarDivide(vrl_Scalar a, vrl_Scalar b);
vrl_Scalar vrl_ScalarMultDiv(vrl_Scalar a, vrl_Scalar b, vrl_Scalar c);
vrl_Scalar vrl_FactorMultiply(vrl_Factor a, vrl_Scalar b);
#define vrl_ScalarRound(a) (a)
#define vrl_ScalarAbs(a) abs32(a)

#else  /* floating-point */
typedef float vrl_Scalar;
typedef float vrl_Factor;
#define VRL_UNITY 1.0
typedef float vrl_Angle;
#define VRL_ANGLECONVERSION 1
#define vrl_ScalarDivide(a, b) (((float) (a)) / ((float) (b)))
#define vrl_ScalarMultDiv(a, b, c) ((((float) (a)) * ((float) (b))) / ((float) (c)))
#define vrl_FactorMultiply(a, b) (((float) (a)) * ((float) (b)))
vrl_Scalar vrl_ScalarRound(vrl_Scalar a);
#define vrl_ScalarAbs(a) fabs((float) (a))
#endif

/* Additional types */

typedef vrl_unsigned32bit vrl_Time;   /* a time in milliseconds */
typedef vrl_unsigned32bit vrl_Color;  /* a four-byte value, red low-order */

typedef vrl_16bit vrl_ScreenPos;      /* a screen-space pixel location */
typedef vrl_32bit vrl_ScreenCoord;    /* a screen-space coordinate */
#define VRL_SCREEN_FRACT_BITS 8       /* screen coords are 24.8 */

/*
   Conversion routines.  You should always use these, to ensure your code
   will work in both the floating-point and fixed-point implementations.
 */

#define float2scalar(f) ((vrl_Scalar) (f))
#define scalar2float(s) ((float) (s))
#define float2factor(f) ((vrl_Factor) ((f) * VRL_UNITY))
#define factor2float(a) (((float) (a)) / VRL_UNITY)
#define float2angle(f) ((vrl_Angle) ((f) * VRL_ANGLECONVERSION))
#define angle2float(a) (((float) (a)) / VRL_ANGLECONVERSION)

/* Math functions */

void vrl_MathInit(void);     /* initializes math routines */
vrl_Factor vrl_Sine(vrl_Angle angle);
vrl_Factor vrl_Cosine(vrl_Angle angle);

/* Vector functions */

typedef vrl_Scalar vrl_Vector[3];

#define X 0
#define Y 1
#define Z 2
#define XROT 3
#define YROT 4
#define ZROT 5

void vrl_VectorCreate(vrl_Vector result, vrl_Scalar x, vrl_Scalar y, vrl_Scalar z);
void vrl_VectorAdd(vrl_Vector result, vrl_Vector v1, vrl_Vector v2);
void vrl_VectorSub(vrl_Vector result, vrl_Vector v1, vrl_Vector v2);
void vrl_VectorNegate(vrl_Vector v);
vrl_Factor vrl_VectorDotproduct(vrl_Vector v1, vrl_Vector v2);
vrl_Scalar vrl_VectorCrossproduct(vrl_Vector result, vrl_Vector v1, vrl_Vector v2);
vrl_Scalar vrl_VectorMagnitude(vrl_Vector v);
void vrl_VectorNormalize(vrl_Vector v);
vrl_Scalar vrl_VectorDistance(vrl_Vector v1, vrl_Vector v2);
void vrl_VectorScale(vrl_Vector v, vrl_Scalar newmag);
void vrl_VectorRescale(vrl_Vector v, vrl_Scalar newmag);

#define vrl_VectorCopy(destination, source) memcpy((destination), (source), sizeof(vrl_Vector))
#define vrl_VectorPrint(out, str, v) fprintf(out, "%s%f,%f,%f", (str), scalar2float((v)[X]), scalar2float((v)[Y]), scalar2float((v)[Z]))
#define vrl_VectorEqual(v1, v2) ((v1)[X] == (v2)[X] && (v1)[Y] == (v2)[Y] && (v1)[Z] == (v2)[Z])
#define vrl_VectorZero(v) ((v)[X] = (v)[Y] = (v)[Z] = 0)

extern vrl_Vector vrl_VectorNULL;  /* the [0,0,0] vector */

/* note that normalized vectors actually have vrl_Factors as elements */

/* Matrix functions */

typedef vrl_Scalar vrl_Matrix[4][3];

/* The rotational elements of a matrix are actually vrl_Factors, not
   vrl_Scalars; only the translational elements are vrl_Scalars. */

void vrl_MatrixIdentity(vrl_Matrix m);
void vrl_MatrixMultiply(vrl_Matrix result, vrl_Matrix m1, vrl_Matrix m2);
void vrl_MatrixInverse(vrl_Matrix result, vrl_Matrix m);
void vrl_MatrixRotX(vrl_Matrix m, vrl_Angle angle, vrl_Boolean leftside);
void vrl_MatrixRotY(vrl_Matrix m, vrl_Angle angle, vrl_Boolean leftside);
void vrl_MatrixRotZ(vrl_Matrix m, vrl_Angle angle, vrl_Boolean leftside);
void vrl_MatrixRotVector(vrl_Matrix m, vrl_Angle angle, vrl_Vector vector, vrl_Boolean leftside);
void vrl_MatrixResetRotations(vrl_Matrix m);
void vrl_MatrixGetBasis(vrl_Vector v, vrl_Matrix m, int axis);
void vrl_MatrixSetBasis(vrl_Matrix m, vrl_Vector v, int axis);
void vrl_MatrixTranslate(vrl_Matrix result, vrl_Vector v, vrl_Boolean leftside);
void vrl_MatrixGetRotations(vrl_Matrix m, vrl_Angle *rx, vrl_Angle *ry, vrl_Angle *rz);

#define vrl_MatrixCopy(result, m) memcpy((result), (m), sizeof(vrl_Matrix))
#define vrl_MatrixGetTranslation(v, m) vrl_VectorCopy((v), (m)[3])
#define vrl_MatrixSetTranslation(m, v) vrl_VectorCopy((m)[3], (v))

/* Transformation functions */

void vrl_Transform(vrl_Vector result, vrl_Matrix m, vrl_Vector v);
vrl_Scalar vrl_TransformX(vrl_Matrix m, vrl_Vector v);
vrl_Scalar vrl_TransformY(vrl_Matrix m, vrl_Vector v);
vrl_Scalar vrl_TransformZ(vrl_Matrix m, vrl_Vector v);

/* Data types */

typedef struct _vrl_world vrl_World;
typedef struct _vrl_object vrl_Object;
typedef struct _vrl_shape vrl_Shape;
typedef struct _vrl_rep vrl_Rep;
typedef struct _vrl_facet vrl_Facet;
typedef struct _vrl_surface vrl_Surface;
typedef struct _vrl_light vrl_Light;
typedef struct _vrl_camera vrl_Camera;
typedef struct _vrl_device vrl_Device;
typedef struct _vrl_device_channel vrl_DeviceChannel;
typedef struct _vrl_serial_port vrl_SerialPort;
typedef struct _vrl_surface_map vrl_Surfacemap;
typedef struct _vrl_hue vrl_Hue;
typedef struct _vrl_palette vrl_Palette;
typedef struct _vrl_stereo_conf vrl_StereoConfiguration;

typedef enum
		{
		VRL_COORD_LOCAL = 0, VRL_COORD_PARENT,
		VRL_COORD_WORLD, VRL_COORD_OBJREL
		} vrl_CoordFrame;

struct _vrl_hue
	{
	unsigned char start;       /* starting index into palette */
	unsigned char maxshade;    /* maximum number of shades */
	};

struct _vrl_palette
	{
	unsigned char data[256][3];
	vrl_Hue huemap[256];
	int changed : 1;	
	};

void vrl_PaletteInit(vrl_Palette *pal);
int vrl_PaletteRead(FILE *in, vrl_Palette *pal);
vrl_Color vrl_PaletteGetEntry(vrl_Palette *pal, int n);
void vrl_PaletteSetEntry(vrl_Palette *pal, int n, vrl_Color color);

#define vrl_PaletteHasChanged(pal) ((pal)->changed)
#define vrl_PaletteSetChanged(pal, flag) ((pal)->changed = (flag))
#define vrl_PaletteGetHuemap(pal) ((pal)->huemap)
	
/* Worlds */

struct _vrl_world
	{
	vrl_Object *objects;      /* tree of objects */
	vrl_Light *lights;        /* linked list of lights */
	vrl_Camera *cameras;      /* linked list of cameras */
	vrl_Camera *camera;       /* current camera */
	vrl_Camera *left_camera;  /* left-eye camera */
	vrl_Camera *right_camera; /* right-eye camera */
	vrl_StereoConfiguration *stereo;  /* stereoscopic viewing information */
	vrl_Factor ambient;       /* ambient light level */
	vrl_Scalar scale;         /* millimeters per unit of virtual space */
	vrl_Scalar movestep;      /* default movement step size */
	vrl_Angle rotstep;        /* default rotation step size */
	vrl_Boolean movement_mode : 1;    /* non-zero if we can fly by looking up */
	vrl_Boolean use_stereo : 1;       /* non-zero if we want stereoscopic viewing */
	vrl_Boolean screenclear : 1;      /* set if we should clear the screen */
	vrl_Boolean horizon : 1;          /* set if we should draw a horizon   */
	vrl_Color horizoncolors[10];      /* entry 0 is ground, entry n is sky */
	int nhorizoncolors;       /* number of colors used in horizoncolors[] */
	vrl_Palette palette;      /* the palette, if one is used */
	};

vrl_World *vrl_WorldInit(vrl_World *world);
void vrl_WorldAddLight(vrl_Light *light);
void vrl_WorldRemoveLight(vrl_Light *light);
vrl_Light *vrl_WorldFindLight(char *name);
void vrl_WorldAddCamera(vrl_Camera *camera);
void vrl_WorldRemoveCamera(vrl_Camera *camera);
vrl_Camera *vrl_WorldFindCamera(char *name);
void vrl_WorldAddObject(vrl_Object *obj);
void vrl_WorldRemoveObject(vrl_Object *obj);
vrl_Object *vrl_WorldFindObject(char *name);
int vrl_WorldCountObjects(void);
int vrl_WorldCountFacets(void);
int vrl_WorldCountLights(void);
int vrl_WorldCountCameras(void);
void vrl_WorldGetBounds(vrl_Vector v1, vrl_Vector v2);
void vrl_WorldGetCenter(vrl_Vector v);
vrl_Scalar vrl_WorldGetSize(void);

#define vrl_WorldCreate() vrl_WorldInit(vrl_malloc(sizeof(vrl_World)))
#define vrl_WorldSetScreenClear(n) (vrl_current_world->screenclear = (n) ? 1 : 0)
#define vrl_WorldGetScreenClear() (vrl_current_world->screenclear)
#define vrl_WorldToggleScreenClear() (vrl_current_world->screenclear = !vrl_current_world->screenclear)
#define vrl_WorldSetStereo(n) (vrl_current_world->use_stereo = (n) ? 1 : 0)
#define vrl_WorldGetStereo() (vrl_current_world->use_stereo)
#define vrl_WorldToggleStereo() (vrl_current_world->use_stereo = !vrl_current_world->use_stereo)
#define vrl_WorldSetStereoConfiguration(conf) (vrl_current_world->stereo = (conf))
#define vrl_WorldGetStereoConfiguration() (vrl_current_world->stereo)
#define vrl_WorldSetHorizon(n) (vrl_current_world->horizon = (n) ? 1 : 0)
#define vrl_WorldGetHorizon() (vrl_current_world->horizon)
#define vrl_WorldToggleHorizon() (vrl_current_world->horizon = !vrl_current_world->horizon)
#define vrl_WorldSetMovementMode(n) (vrl_current_world->movement_mode = (n) ? 1 : 0)
#define vrl_WorldGetMovementMode() (vrl_current_world->movement_mode)
#define vrl_WorldToggleMovementMode() (vrl_current_world->movement_mode = !vrl_current_world->movement_mode)
#define vrl_WorldSetMovestep(distance) (vrl_current_world->movestep = (distance))
#define vrl_WorldGetMovestep() (vrl_current_world->movestep)
#define vrl_WorldSetTurnstep(angle) (vrl_current_world->rotstep = (angle))
#define vrl_WorldGetTurnstep() (vrl_current_world->rotstep)
#define vrl_WorldSetScale(s) (vrl_current_world->scale = (s))
#define vrl_WorldGetScale() (vrl_current_world->scale)
#define vrl_WorldSetAmbient(amb) (vrl_current_world->ambient = (amb))
#define vrl_WorldGetAmbient() (vrl_current_world->ambient)
#define vrl_WorldSetGroundColor(color) (vrl_current_world->horizoncolors[0] = (color))
#define vrl_WorldGetGroundColor() (vrl_current_world->horizoncolors[0])
#define vrl_WorldSetSkyColor(color) (vrl_current_world->horizoncolors[vrl_current_world->nhorizoncolors-1] = (color))
#define vrl_WorldGetSkyColor() (vrl_current_world->horizoncolors[vrl_current_world->nhorizoncolors-1])
#define vrl_WorldGetPalette() (&vrl_current_world->palette)
#define vrl_WorldGetLights() (vrl_current_world->lights)
#define vrl_WorldGetCameras() (vrl_current_world->cameras)
#define vrl_WorldGetObjectTree() (vrl_current_world->objects)
#define vrl_WorldSetCamera(cam) (vrl_current_world->camera = (cam))
#define vrl_WorldGetCamera() (vrl_current_world->camera)
#define vrl_WorldSetLeftCamera(cam) (vrl_current_world->left_camera = (cam))
#define vrl_WorldGetLeftCamera() (vrl_current_world->left_camera)
#define vrl_WorldSetRightCamera(cam) (vrl_current_world->right_camera = (cam))
#define vrl_WorldGetRightCamera() (vrl_current_world->right_camera)
#define vrl_WorldUpdate() vrl_ObjectUpdate(vrl_WorldGetObjectTree())

extern vrl_World *vrl_current_world;   /* the currently active world */

#define vrl_WorldSetCurrent(world) (vrl_current_world = (world))
#define vrl_WorldGetCurrent() (vrl_current_world)

/* Objects */

typedef int (*vrl_ObjectFunction)(vrl_Object *obj);

struct _vrl_object
	{
	vrl_Shape *shape;           /* geometry information */
	vrl_Surfacemap *surfmap;    /* array of pointers to surface descriptors */
	vrl_Matrix localmat;        /* transformation matrix relative to our parent */
	vrl_Matrix globalmat;       /* transformation matrix relative to the world */
	vrl_Object *parent;         /* pointer to our parent in the hierarchy */
	vrl_Object *children;       /* pointer to our children */
	vrl_Object *siblings;       /* pointers to our siblings */
	vrl_Vector minbound, maxbound;  /* bounding box (world coords) */
	vrl_Boolean fixed : 1;              /* set if object is immobile */
	vrl_Boolean moved : 1;              /* set when our local matrix has changed */
	vrl_Boolean rotate_box : 1;         /* set if bounding box should rotate */
	vrl_Boolean highlight : 1;          /* set if object is highlighted */
	vrl_Boolean invisible : 1;          /* set if object is invisible */
	unsigned char layer;        /* the layer we're on (0 for all, 1-255) */
	vrl_Object *contents;       /* points to objects contained by this one (not used) */
	vrl_Rep *forced_rep;        /* if not NULL, forces a rep to be used */
	char *name;                 /* name of the object (may be NULL) */
	void *applic_data;          /* pointer to application-specific data */
	vrl_ObjectFunction function;   /* object function */
	vrl_Object *next;           /* points to next object on a list */
	};

vrl_Object *vrl_ObjectInit(vrl_Object *obj);
vrl_Object *vrl_ObjectCreate(vrl_Shape *shape);
vrl_Object *vrl_ObjectCopy(vrl_Object *obj);
void vrl_ObjectDestroy(vrl_Object *object);
void vrl_ObjectMove(vrl_Object *obj, vrl_Scalar x, vrl_Scalar y, vrl_Scalar z);
void vrl_ObjectRelMove(vrl_Object *obj, vrl_Scalar x, vrl_Scalar y, vrl_Scalar z);
void vrl_ObjectRotVector(vrl_Object *obj, vrl_Angle angle, vrl_Vector vector);
void vrl_ObjectRotReset(vrl_Object *obj);
void vrl_ObjectRotate(vrl_Object *obj, vrl_Angle angle, int axis, vrl_CoordFrame frame, vrl_Object *relative_to);
void vrl_ObjectTranslate(vrl_Object *obj, vrl_Vector v, vrl_CoordFrame frame, vrl_Object *relative_to);
void vrl_ObjectLookAt(vrl_Object *obj, vrl_Vector forward, vrl_Vector up);
vrl_Object *vrl_ObjectAttach(vrl_Object *obj, vrl_Object *newparent);
vrl_Object *vrl_ObjectDetach(vrl_Object *obj);
vrl_Object *vrl_ObjectUpdate(vrl_Object *object);
void vrl_ObjectTraverse(vrl_Object *object, int (*function)(vrl_Object *obj));
void vrl_ObjectMakeFixed(vrl_Object *object);
void vrl_ObjectMakeMovable(vrl_Object *object);
vrl_Object *vrl_ObjectFindRoot(vrl_Object *obj);
vrl_Scalar vrl_ObjectComputeDistance(vrl_Object *obj1, vrl_Object *obj2);

#define vrl_ObjectRotX(obj, angle) vrl_ObjectRotate((obj), (angle), X, VRL_COORD_PARENT, NULL)
#define vrl_ObjectRotY(obj, angle) vrl_ObjectRotate((obj), (angle), Y, VRL_COORD_PARENT, NULL)
#define vrl_ObjectRotZ(obj, angle) vrl_ObjectRotate((obj), (angle), Z, VRL_COORD_PARENT, NULL)
#define vrl_ObjectVectorMove(obj, v) vrl_ObjectMove((obj), (v)[X], (v)[Y], (v)[Z])
#define vrl_ObjectVectorRelMove(obj, v) vrl_ObjectTranslate((obj), (v), VRL_COORD_PARENT, NULL)
#define vrl_ObjectSetLayer(object, lay) ((obj)->layer = (lay))
#define vrl_ObjectGetLayer(object) ((object)->layer)
#define vrl_ObjectSetShape(object, shp) ((object)->shape = (shp))
#define vrl_ObjectGetShape(object) ((object)->shape)
#define vrl_ObjectSetSurfacemap(object, map) ((object)->surfmap = (map))
#define vrl_ObjectGetSurfacemap(object) ((object)->surfmap)
#define vrl_ObjectSetHighlight(object, high) ((object)->highlight = (high))
#define vrl_ObjectGetHighlight(object) ((object)->highlight)
#define vrl_ObjectToggleHighlight(object) ((object)->highlight = !(object)->highlight)
#define vrl_ObjectSetVisibility(object, vis) ((object)->invisible = ((vis) ? 1 : 0))
#define vrl_ObjectGetVisibility(object) ((object)->invisible)
#define vrl_ObjectToggleVisibility(object) ((object)->invisible = !(object)->invisible)
#define vrl_ObjectIsFixed(object) ((object)->fixed)
#define vrl_ObjectGetMinbounds(object, v) vrl_VectorCopy((v), (object)->minbound)
#define vrl_ObjectGetMaxbounds(object, v) vrl_VectorCopy((v), (object)->maxbound)
#define vrl_ObjectSetRep(object, r) ((object)->forced_rep = (r))
#define vrl_ObjectGetRep(object) ((object)->forced_rep)
#define vrl_ObjectGetWorldX(object) ((object)->globalmat[3][X])
#define vrl_ObjectGetWorldY(object) ((object)->globalmat[3][Y])
#define vrl_ObjectGetWorldZ(object) ((object)->globalmat[3][Z])
#define vrl_ObjectGetWorldLocation(object, v) vrl_VectorCopy((v), (object)->globalmat[3])
#define vrl_ObjectGetRelativeX(object) ((object)->localmat[3][X])
#define vrl_ObjectGetRelativeY(object) ((object)->localmat[3][Y])
#define vrl_ObjectGetRelativeZ(object) ((object)->localmat[3][Z])
#define vrl_ObjectGetRelativeLocation(object, v) vrl_VectorCopy((v), (object)->localmat[3])
#define vrl_ObjectGetWorldRotations(object, rx, ry, rz) vrl_MatrixGetRotations((object)->globalmat, (rx), (ry), (rz))
#define vrl_ObjectGetRelativeRotations(object, rx, ry, rz) vrl_MatrixGetRotations((object)->localmat, (rx), (ry), (rz))
#define vrl_ObjectSetName(obj, str) ((obj)->name = (str))
#define vrl_ObjectGetName(obj) ((obj)->name)
#define vrl_ObjectSetApplicationData(obj, data) ((obj)->applic_data = (data))
#define vrl_ObjectGetApplicationData(obj) ((obj)->applic_data)
#define vrl_ObjectSetFunction(obj, fn) ((obj)->function = (fn))
#define vrl_ObjectGetFunction(obj) ((obj)->function)
#define vrl_ObjectGetForwardVector(object, v) vrl_MatrixGetBasis((v), (object)->globalmat, Z)
#define vrl_ObjectGetRightVector(object, v) vrl_MatrixGetBasis((v), (object)->globalmat, X)
#define vrl_ObjectGetUpVector(object, v) vrl_MatrixGetBasis((v), (object)->globalmat, Y)
#define vrl_ObjectGetParent(object) ((object)->parent)

/* Shapes */

struct _vrl_shape
	{
	vrl_Vector center;  /* center of bounding sphere */
	vrl_Scalar radius;  /* radius of bounding sphere */
	vrl_Vector minbound, maxbound;  /* bounding box */
	vrl_Surfacemap *default_surfacemap;  /* default surface map for this shape */
	vrl_Rep *replist;  /* linked list of representations */
	char *name;        /* name of this shape */
	vrl_Shape *next;   /* shapes are kept in a linked list */
	};

vrl_Shape *vrl_ShapeInit(vrl_Shape *shape);
void vrl_ShapeComputeBounds(vrl_Shape *shape);
void vrl_ShapeRescale(vrl_Shape *shape, float sx, float sy, float sz);
void vrl_ShapeOffset(vrl_Shape *shape, vrl_Scalar tx, vrl_Scalar ty, vrl_Scalar tz);
void vrl_ShapeUpdate(vrl_Shape *shape);
void vrl_ShapeTransform(vrl_Matrix mat, vrl_Shape *shape);
vrl_Rep *vrl_ShapeGetRep(vrl_Shape *shape, vrl_Scalar size);
void vrl_ShapeAddRep(vrl_Shape *shape, vrl_Rep *rep, vrl_Scalar size);
void vrl_ShapeTraverseReps(vrl_Shape *shape, int (*function)(vrl_Rep *rep));       /* NEEDED? */
int vrl_ShapeCountReps(vrl_Shape *shape);
vrl_Shape *vrl_ShapeGetList(void);
vrl_Shape *vrl_ShapeFind(char *name);
void vrl_ShapeComputeVertexNormals(vrl_Shape *shape);

#define vrl_ShapeCreate() vrl_ShapeInit(vrl_malloc(sizeof(vrl_Shape)))
#define vrl_ShapeGetRadius(shape) ((shape)->radius)
#define vrl_ShapeGetCenter(shape, v) vrl_VectorCopy((v), (shape)->center)
#define vrl_ShapeGetMinbounds(shape, v) vrl_VectorCopy((v), (shape)->minbound)
#define vrl_ShapeGetMaxbounds(shape, v) vrl_VectorCopy((v), (shape)->maxbound)
#define vrl_ShapeGetSurfacemap(shape) ((shape)->default_surfacemap)
#define vrl_ShapeSetSurfacemap(shape, map) ((shape)->default_surfacemap = (map))
#define vrl_ShapeGetName(shape) ((shape)->name)
#define vrl_ShapeSetName(shape, str) ((shape)->name = (str))
#define vrl_ShapeGetNext(shape) ((shape)->next)
#define vrl_ShapeGetFirstRep(shape) ((shape)->replist)

/* Representations */

typedef struct { int v1, v2; } vrl_Edge;

typedef enum
		{
		VRL_SORT_NONE = 0, VRL_SORT_FARTHEST, VRL_SORT_NEAREST,
		VRL_SORT_AVERAGE, VRL_SORT_OTHER
		} vrl_SortingType;

struct _vrl_rep
	{
	vrl_Scalar size;    /* size (in pixels) at which to use this rep */
	vrl_Rep *next;      /* next less-detailed rep */
	vrl_SortingType sorttype;  /* type of poly sorting to do on this rep */
	int nvertices;      /* number of vertices (and normals if present) */
	vrl_Vector *vertices;   /* array of vertices */
	vrl_Vector *normals;    /* array of vertex normals; can be NULL */
	vrl_Facet *facets;  /* facets are kept in a linked list */
	vrl_Edge *edges;
	};

vrl_Rep *vrl_RepInit(vrl_Rep *rep, int nvertices, vrl_Boolean has_normals);
void vrl_RepAddFacet(vrl_Rep *rep, vrl_Facet *facet);
void vrl_RepTraverseVertices(vrl_Rep *rep, int (*function)(vrl_Vector *vertex, vrl_Vector *normal));
void vrl_RepTraverseFacets(vrl_Rep *rep, int (*function)(vrl_Facet *facet));
vrl_Facet *vrl_RepGetFacet(vrl_Rep *rep, int n);
vrl_Facet *vrl_RepFindFacet(vrl_Rep *rep, unsigned int id);
int vrl_RepCountFacets(vrl_Rep *rep);
void vrl_RepComputeVertexNormals(vrl_Rep *rep);
void vrl_RepBuildEdges(vrl_Rep *rep);

#define vrl_RepCreate(nv, norm) vrl_RepInit(vrl_malloc(sizeof(vrl_Rep)), (nv), (norm))
#define vrl_RepSetSorting(rep, type) ((rep)->sorttype = (type))
#define vrl_RepGetSorting(rep) ((rep)->sorttype)
#define vrl_RepGetSize(rep) ((rep)->size)
#define vrl_RepCountVertices(rep) ((rep)->nvertices)
#define vrl_RepGetVertex(rep, n, v) vrl_VectorCopy((v), (rep)->vertices[n])
#define vrl_RepSetVertex(rep, n, v) vrl_VectorCopy((rep)->vertices[n], (v))
#define vrl_RepHasNormals(rep) ((rep)->normals)
#define vrl_RepGetNormal(rep, n, v) vrl_VectorCopy((v), (rep)->normals[n])
#define vrl_RepGetNext(rep) ((rep)->next)

/* Facets */

struct _vrl_facet
	{
	int surface;            /* index into object's surface array */
	vrl_Vector normal;      /* perpendicular to facet, left-hand rule */
	unsigned int id;        /* identifier for this facet */
	vrl_Boolean highlight : 1;      /* facet is highlighted */
	vrl_Boolean interior  : 1;      /* facet is on the interior of an object (not used) */
	vrl_Facet *details;     /* points to linked list of detail facets (not used) */
	vrl_Facet *nearside, *farside;  /* only farside is used */
	int npoints;  /* number of points in the facet */
	int *points;  /* indices into array of vertices of the facet points */
	int *edges;   /* indices into array edges */
	};

vrl_Facet *vrl_FacetInit(vrl_Facet *facet, int npts);
void vrl_FacetTraverse(vrl_Facet *facet, int (*function)(vrl_Facet *f));
void vrl_FacetComputeNormal(vrl_Facet *facet, vrl_Vector *vertices);

#define vrl_FacetCreate(npts) vrl_FacetInit(vrl_malloc(sizeof(vrl_Facet)), (npts))
#define vrl_FacetSetSurfnum(facet, n) ((facet)->surface = (n))
#define vrl_FacetGetSurfnum(facet) ((facet)->surface)
#define vrl_FacetCountPoints(facet) ((facet)->npoints)
#define vrl_FacetGetPoint(facet, n) ((facet)->points[n])
#define vrl_FacetGetVertex(rep, facet, n, v) vrl_VectorCopy((v), (rep)->vertices[(facet)->points[n]])
#define vrl_FacetSetHighlight(facet, high) ((facet)->highlight = ((high) ? 1 : 0))
#define vrl_FacetGetHighlight(facet) ((facet)->highlight)
#define vrl_FacetToggleHighlight(facet) ((facet)->highlight = !(facet)->highlight)
#define vrl_FacetSetInterior(facet, inter) ((facet)->interior = ((inter) ? 1 : 0))
#define vrl_FacetGetInterior(facet) ((facet)->interior)
#define vrl_FacetToggleInterior(facet) ((facet)->interior = !(facet)->interior)
#define vrl_FacetSetId(facet, n) ((facet)->id = (n))
#define vrl_FacetGetId(facet) ((facet)->id)
#define vrl_FacetGetNormal(facet, v) vrl_VectorCopy((v), (facet)->normal)
#define vrl_FacetSetPoint(facet, n, m) ((facet)->points[n] = (m))
#define vrl_FacetGetPoint(facet, n) ((facet)->points[n])

/* Textures */

typedef void vrl_Texture;  /* for now... */

/* Surfaces */

typedef enum
		{
		VRL_SURF_SIMPLE = 0, VRL_SURF_FLAT, VRL_SURF_METAL,
		VRL_SURF_GLASS, VRL_SURF_GOURAUD, VRL_SURF_SPECULAR
		} vrl_SurfaceType;

typedef unsigned char vrl_Exponent;

struct _vrl_surface
	{
	vrl_SurfaceType type;
	unsigned char hue;
	unsigned char brightness;
	vrl_Exponent exp;
	vrl_Surface *next;
	vrl_Texture *texture;
	};

vrl_Surface *vrl_SurfaceFromDesc(vrl_unsigned16bit desc, vrl_Surface *surf);
vrl_unsigned16bit vrl_SurfaceToDesc(vrl_Surface *surf);
vrl_Surface *vrl_SurfaceInit(vrl_Surface *surf);
vrl_Surface *vrl_SurfaceGetList(void);
vrl_Surface *vrl_SurfaceCreate(unsigned char hue);

#define vrl_SurfaceSetType(surf, t) ((surf)->type = (t))
#define vrl_SurfaceGetType(surf) ((surf)->type)
#define vrl_SurfaceSetHue(surf, h) ((surf)->hue = (h))
#define vrl_SurfaceGetHue(surf) ((surf)->hue)
#define vrl_SurfaceSetBrightness(surf, b) ((surf)->brightness = (b))
#define vrl_SurfaceGetBrightness(surf) ((surf)->brightness)
#define vrl_SurfaceSetExponent(surf, e) ((surf)->exp = (e))
#define vrl_SurfaceGetExponent(surf) ((surf)->exp)
#define vrl_SurfaceSetTexture(surf, t) ((surf)->texture = (t))
#define vrl_SurfaceGetTexture(surf) ((surf)->texture)
#define vrl_SurfaceGetNext(surf) ((surf)->next)

struct _vrl_surface_map
	{
	int nentries;
	vrl_Surface **entries;
	vrl_Surfacemap *next;
	};

vrl_Surfacemap *vrl_SurfacemapCreate(int n);
vrl_Surfacemap *vrl_SurfacemapGetList(void);

#define vrl_SurfacemapCountEntries(map) ((map)->nentries)
#define vrl_SurfacemapSetSurface(map, surfnum, surf) ((map)->entries[surfnum] = (surf))
#define vrl_SurfacemapGetSurface(map, surfnum) ((map)->entries[surfnum])
#define vrl_SurfacemapGetNext(map) ((map)->next)

/* Lights */

typedef enum
	{
	VRL_LIGHT_AMBIENT = 0, VRL_LIGHT_DIRECTIONAL, VRL_LIGHT_POINTSOURCE
	} vrl_LightingType;

struct _vrl_light
	{
	vrl_LightingType type;  /* type of light source */
	vrl_Boolean on : 1;     /* set if the light is on */
	vrl_Factor intensity;   /* how bright the light is */
	vrl_Object *object;     /* the object this light is associated with, if any */
	vrl_Light *next;        /* lights are kept in a linked list */
	char *name;             /* name of this light */
	void *applic_data;      /* pointer to application-specific data */
	int (*function)(vrl_Light *light);  /* light function */
	};

vrl_Light *vrl_LightInit(vrl_Light *light);
vrl_Light *vrl_LightCreate(void);
void vrl_LightDestroy(vrl_Light *light);

#define vrl_LightSetType(light, ltype) ((light)->type = (ltype))
#define vrl_LightGetType(light) ((light)->type)
#define vrl_LightOn(light) ((light)->on = 1)
#define vrl_LightOff(light) ((light)->on = 0)
#define vrl_LightToggle(light) ((light)->on = !((light)->on))
#define vrl_LightIsOn(light) ((light)->on)
#define vrl_LightSetIntensity(light, inten) ((light)->intensity = (inten))
#define vrl_LightGetIntensity(light) ((light)->intensity)
#define vrl_LightAssociate(light, obj) ((light)->object = (obj))
#define vrl_LightDisAssociate(light) ((light)->object = NULL)
#define vrl_LightGetObject(light) ((light)->object)
#define vrl_LightSetName(light, str) ((light)->name = (str))
#define vrl_LightGetName(light) ((light)->name)
#define vrl_LightGetNext(light) ((light)->next)
#define vrl_LightSetApplicationData(light, data) ((light)->applic_data = (data))
#define vrl_LightGetApplicationData(light) ((light)->applic_data)

#define vrl_LightMove(light, x, y, z) vrl_ObjectMove((light)->object, x, y, z)
#define vrl_LightRelMove(light, x, y, z) vrl_ObjectRelMove((light)->object, (x), (y), (z))
#define vrl_LightVectorMove(light, v) vrl_ObjectVectorMove((light)->object, (v))
#define vrl_LightVectorRelMove(light, v) vrl_ObjectVectorRelMove((light)->object, (v))
#define vrl_LightRotX(light, angle) vrl_ObjectRotX((light)->object, (angle))
#define vrl_LightRotY(light, angle) vrl_ObjectRotY((light)->object, (angle))
#define vrl_LightRotZ(light, angle) vrl_ObjectRotZ((light)->object, (angle))
#define vrl_LightRotVector(light, angle, vector) vrl_ObjectRotVector((light)->object, (angle), (vector))
#define vrl_LightRotReset(light) vrl_ObjectRotReset((light)->object)
#define vrl_LightRotate(light, angle, axis, frame, relative_to) vrl_ObjectRotate((light)->obj, (angle), (axis), (frame), (relative_to))
#define vrl_LightTranslate(light, v, axis, frame, relative_to) vrl_ObjectTranslate((light)->obj, (v), (frame), (relative_to))
#define vrl_LightLookAt(light, forward, up) vrl_ObjectLookAt((light)->obj, (forward), (up))
#define vrl_LightAttach(light, newparent) vrl_ObjectAttach((light)->object, (newparent))
#define vrl_LightDetach(light) vrl_ObjectDetach((light)->object)

#define vrl_LightGetWorldLocation(light, v) vrl_ObjectGetWorldLocation((light)->object, (v))
#define vrl_LightGetWorldRotations(light, rx, ry, rz) vrl_ObjectGetWorldRotations((light)->object, (rx), (ry), (rz))
#define vrl_LightGetRelativeLocation(light, v) vrl_ObjectGetRelativeLocation((light)->object, (v))
#define vrl_LightGetRelativeRotations(light, rx, ry, rz) vrl_ObjectGetRelativeRotations((light)->object, (rx), (ry), (rz))

#define vrl_LightGetWorldX(light) ((light)->object->globalmat[3][X])
#define vrl_LightGetWorldY(light) ((light)->object->globalmat[3][Y])
#define vrl_LightGetWorldZ(light) ((light)->object->globalmat[3][Z])

#define vrl_LightGetRelativeX(light) ((light)->object->localmat[3][X])
#define vrl_LightGetRelativeY(light) ((light)->object->localmat[3][Y])
#define vrl_LightGetRelativeZ(light) ((light)->object->localmat[3][Z])

/* Cameras */

struct _vrl_camera
	{
	vrl_Scalar hither;   /* distance to near clipping plane */
	vrl_Scalar yon;      /* distance to far culling plane */
	float zoom;          /* zoom factor (1/tan(FOV/2)) */
	float aspect;        /* aspect ratio */
	vrl_Boolean ortho : 1;    /* set if we want orthographic projection (not used) */
	vrl_Scalar orthodist;     /* apparent "distance" for orthographic projection (not used) */
	vrl_Object *object;       /* the object this camera is attached to */
	unsigned char need_updating;  /* set when zoom or aspect is changed */
	/* these next four are only used internally, for object culling */
	vrl_Factor aright, cright, btop, ctop;
	vrl_Camera *next;       /* cameras are kept in a linked list */
	char *name;             /* name of this camera */
	void *applic_data;      /* pointer to application-specific data */
	};

vrl_Camera *vrl_CameraInit(vrl_Camera *camera);
vrl_Camera *vrl_CameraCreate(void);
void vrl_CameraDestroy(vrl_Camera *camera);

#define vrl_CameraSetZoom(camera, zf) (((camera)->zoom = (zf)), (camera)->need_updating = 1)
#define vrl_CameraGetZoom(camera) ((camera)->zoom)
#define vrl_CameraSetAspect(camera, asp) (((camera)->aspect = (asp)), (camera)->need_updating = 1)
#define vrl_CameraGetAspect(camera) ((camera)->aspect)
#define vrl_CameraSetHither(camera, h) ((camera)->hither = (h))
#define vrl_CameraGetHither(camera) ((camera)->hither)
#define vrl_CameraSetYon(camera, y) ((camera)->yon = (y))
#define vrl_CameraGetYon(camera) ((camera)->yon)
#define vrl_CameraAssociate(camera, obj) ((camera)->object = (obj))
#define vrl_CameraGetObject(camera) ((camera)->object)
#define vrl_CameraSetName(camera, str) ((camera)->name = (str))
#define vrl_CameraGetName(camera) ((camera)->name)
#define vrl_CameraGetNext(camera) ((camera)->next)
#define vrl_CameraSetApplicationData(cam, data) ((cam)->applic_data = (data))
#define vrl_CameraGetApplicationData(cam) ((cam)->applic_data)

#define vrl_CameraMove(camera, x, y, z) vrl_ObjectMove((camera)->object, x, y, z)
#define vrl_CameraRelMove(camera, x, y, z) vrl_ObjectRelMove((camera)->object, (x), (y), (z))
#define vrl_CameraVectorMove(camera, v) vrl_ObjectVectorMove((camera)->object, (v))
#define vrl_CameraVectorRelMove(camera, v) vrl_ObjectVectorRelMove((camera)->object, (v))
#define vrl_CameraRotX(camera, angle) vrl_ObjectRotX((camera)->object, (angle))
#define vrl_CameraRotY(camera, angle) vrl_ObjectRotY((camera)->object, (angle))
#define vrl_CameraRotZ(camera, angle) vrl_ObjectRotZ((camera)->object, (angle))
#define vrl_CameraRotVector(camera, angle, vector) vrl_ObjectRotVector((camera)->object, (angle), (vector))
#define vrl_CameraRotReset(camera) vrl_ObjectRotReset((camera)->object)
#define vrl_CameraRotate(camera, angle, axis, frame, relative_to) vrl_ObjectRotate((camera)->object, (angle), (axis), (frame), (relative_to))
#define vrl_CameraTranslate(camera, v, axis, frame, relative_to) vrl_ObjectTranslate((camera)->object, (v), (frame), (relative_to))
#define vrl_CameraLookAt(camera, forward, up) vrl_ObjectLookAt((camera)->object, (forward), (up))
#define vrl_CameraAttach(camera, newparent) vrl_ObjectAttach((camera)->object, (newparent))
#define vrl_CameraDetach(camera) vrl_ObjectDetach((camera)->object)

#define vrl_CameraGetWorldX(camera) ((camera)->object->globalmat[3][X])
#define vrl_CameraGetWorldY(camera) ((camera)->object->globalmat[3][Y])
#define vrl_CameraGetWorldZ(camera) ((camera)->object->globalmat[3][Z])

#define vrl_CameraGetRelativeX(camera) ((camera)->object->localmat[3][X])
#define vrl_CameraGetRelativeY(camera) ((camera)->object->localmat[3][Y])
#define vrl_CameraGetRelativeZ(camera) ((camera)->object->localmat[3][Z])

#define vrl_CameraGetWorldLocation(camera, v) vrl_ObjectGetWorldLocation((camera)->object, (v))
#define vrl_CameraGetWorldRotations(camera, rx, ry, rz) vrl_ObjectGetWorldRotations((camera)->object, (rx), (ry), (rz))
#define vrl_CameraGetRelativeLocation(camera, v) vrl_ObjectGetRelativeLocation((camera)->object, (v))
#define vrl_CameraGetRelativeRotations(camera, rx, ry, rz) vrl_ObjectGetRelativeRotations((camera)->object, (rx), (ry), (rz))

#define vrl_CameraGetForwardVector(camera, v) vrl_ObjectGetForwardVector((camera)->object, (v))
#define vrl_CameraGetRightVector(camera, v) vrl_ObjectGetRightVector((camera)->object, (v))
#define vrl_CameraGetUpVector(camera, v) vrl_ObjectGetUpVector((camera)->object, (v))

/* Memory allocation functions */

void *vrl_malloc(unsigned int nbytes);
void *vrl_calloc(unsigned int nitems, unsigned int item_size);
void vrl_free(void *ptr);

/* Rendering functions */

typedef struct
	{
	vrl_Boolean memory : 1;       /* set if the renderer ran out of memory */
	vrl_Boolean objects : 1;      /* set if there were too many objects  */
	vrl_Boolean facets : 1;       /* set if there were too many facets   */
	} vrl_RenderStatus;

vrl_Boolean vrl_RenderInit(int maxvert, int maxf, int maxobjs, int maxlights, unsigned int mempoolsize);
void vrl_RenderQuit(void);
void vrl_RenderBegin(vrl_Camera *camera, vrl_Light *lights);
void vrl_RenderSetAmbient(vrl_Factor amb);
void vrl_RenderHorizon(void);
vrl_RenderStatus *vrl_RenderObjlist(vrl_Object *objects);
void vrl_RenderSetHorizontalShift(int n);
void vrl_RenderSetDrawMode(int mode);
int vrl_RenderGetDrawMode(void);
void vrl_RenderSetWireframeColor(vrl_Color color);
vrl_Color vrl_RenderGetWireframeColor(void);
void vrl_RenderSetHighlightColor(vrl_Color color);
vrl_Color vrl_RenderGetHighlightColor(void);

void vrl_RenderMonitorInit(int x, int y);  /* monitor the specified point */
vrl_Boolean vrl_RenderMonitorRead(vrl_Object **obj, vrl_Facet **facet, int *vertnum);  /* returns zero if none */

void vrl_TransformVertexToScreen(vrl_ScreenCoord *x, vrl_ScreenCoord *y, vrl_Object *obj, vrl_Vector vertex);

/* Layer support */

void vrl_LayerOn(int n);
void vrl_LayerOff(int n);
void vrl_LayerToggle(int n);
vrl_Boolean vrl_LayerIsOn(int n);
void vrl_LayerAllOn(void);
void vrl_LayerAllOff(void);

/* File i/o routines */

void vrl_FileSetLoadpath(char *path);
char *vrl_FileFixupFilename(char *fname);
vrl_Shape *vrl_ReadPLG(FILE *in);
int vrl_WritePLG(vrl_Shape *shape, FILE *out);
void vrl_SetReadPLGscale(float x, float y, float z);
void vrl_SetReadPLGoffset(float x, float y, float z);
int vrl_ReadWLD(FILE *in);
void vrl_ReadWLDfeature(int argc, char *argv[], char *rawtext);
vrl_Object *vrl_ReadFIG(FILE *in, vrl_Object *parent, char *rootname);
void vrl_SetReadFIGpartArray(vrl_Object **ptr, int maxparts);
void vrl_SetReadFIGscale(float x, float y, float z);
vrl_Object *vrl_ReadObjectPLG(FILE *in);

vrl_Object *vrl_ObjectLoadPLGfile(char *filename);
vrl_Object *vrl_ObjectLoadFIGfile(char *filename);
int vrl_LoadWLDfile(char *filename);
int vrl_ObjectSavePLGfile(vrl_Object *object, char *filename);

/* Raster support */

typedef struct
	{
	vrl_ScreenPos width, height, depth;
	vrl_ScreenPos left, top, right, bottom;
	vrl_ScreenPos rowbytes;
#ifdef __BORLANDC__
	unsigned char far *data;
#else
	unsigned char *data;
#endif
	} vrl_Raster;

vrl_Raster *vrl_RasterCreate(vrl_ScreenPos width, vrl_ScreenPos height, vrl_unsigned16bit depth);
void vrl_RasterDestroy(vrl_Raster *raster);
void vrl_RasterSetWindow(vrl_Raster *raster, vrl_ScreenPos left, vrl_ScreenPos top, vrl_ScreenPos right, vrl_ScreenPos bottom);
void vrl_RasterGetWindow(vrl_Raster *raster, vrl_ScreenPos *left, vrl_ScreenPos *top, vrl_ScreenPos *right, vrl_ScreenPos *bottom);

#define vrl_RasterGetHeight(r) ((r)->height)
#define vrl_RasterGetWidth(r) ((r)->width)
#define vrl_RasterGetDepth(r) ((r)->depth)
#define vrl_RasterGetRowbytes(r) ((r)->rowbytes)
#define vrl_RasterSetRowbytes(r, n) ((r)->rowbytes = (n))
#define vrl_RasterReadScanline(r, n, buff) memcpy((buff), &(r)->data[(n)*(r)->rowbytes], (r)->rowbytes)
#define vrl_RasterWriteScanline(r, n, buff) memcpy(&(r)->data[(n)*(r)->rowbytes], (buff), (r)->rowbytes)
#define vrl_RasterGetData(r) ((r)->data)

/* Display routines */

typedef struct _vrl_outvertex vrl_OutputVertex;

struct _vrl_outvertex
	{
	vrl_ScreenCoord x,y,z;            /* X, Y screen coordinates and Z-depth */
	vrl_16bit red, green, blue;       /* components of the color */
	vrl_OutputVertex *next, *prev;    /* doubly-linked circular list */
	/* don't rely on anything below here staying the same */
	vrl_unsigned16bit u, v;           /* texture map coordinates */
	vrl_Factor intensity;             /* intensity of light at this vertex */
	vrl_unsigned16bit outcode;        /* used for XY clipping */
	/* future versions of AVRIL may have additional info per vertex */
	};

typedef struct _vrl_outfacet vrl_OutputFacet;

struct _vrl_outfacet
	{
	vrl_OutputVertex *points;     /* linked list of vertices for this facet */
	vrl_Surface *surface;         /* surface properties */
	vrl_Color color;              /* color of this facet (flat shading only) */
	/* don't rely on anything below here staying the same */
	vrl_Factor intensity;         /* amount of light falling on this facet */
	int xclip;                    /* this facet needs X clipping */
	int yclip;                    /* this facet needs Y clipping */
	int highlight : 1;            /* this facet is highlighted */
	vrl_Facet *original;          /* points back at the facet we came from */
	void *outobj;                 /* points back at the outobject we belong to */
	vrl_ScreenCoord minbound[3], maxbound[3];  /* bounding box for smarter sorting */
	vrl_OutputFacet *details;     /* linked list of detail facets */
	vrl_OutputFacet *next;        /* only used to link detail facets */
	/* future versions of AVRIL may have additional info per facet */
	};

typedef enum
		{
		VRL_DISPLAY_GET_VERSION = 0, VRL_DISPLAY_GET_DESCRIPTION,
		VRL_DISPLAY_INIT, VRL_DISPLAY_QUIT,
		VRL_DISPLAY_CLEAR, VRL_DISPLAY_POINT, VRL_DISPLAY_LINE,
		VRL_DISPLAY_CLOSED_LINE, VRL_DISPLAY_POLY, VRL_DISPLAY_BOX,
		VRL_DISPLAY_TEXT, VRL_DISPLAY_TEXT_POSITION,
		VRL_DISPLAY_GET_TEXTWIDTH, VRL_DISPLAY_GET_TEXTHEIGHT,
		VRL_DISPLAY_GET_SORTORDER, VRL_DISPLAY_CAN_GOURAUD,
		VRL_DISPLAY_UPDATE_PALETTE, VRL_DISPLAY_CAN_XY_CLIP,
		VRL_DISPLAY_BEGIN_FRAME, VRL_DISPLAY_END_FRAME,
		VRL_DISPLAY_SET_RASTER, VRL_DISPLAY_GET_RASTER,
		VRL_DISPLAY_SET_Z_BUFFER, VRL_DISPLAY_GET_Z_BUFFER,
		VRL_DISPLAY_USE_Z_BUFFER, VRL_DISPLAY_CLEAR_Z_BUFFER,
		VRL_DISPLAY_SET_SHADING,
		} vrl_DisplayCommand;

typedef vrl_32bit vrl_DisplayDriverFunction(vrl_DisplayCommand cmd, vrl_32bit lparm, void *pparm1);
extern vrl_DisplayDriverFunction *vrl_current_display_driver;
#define vrl_DisplaySetDriver(driver) (vrl_current_display_driver = (driver))

int vrl_DisplayInit(vrl_Raster *raster);
void vrl_DisplayQuit(void);
void vrl_DisplayClear(vrl_Color color);
void vrl_DisplayPoint(vrl_ScreenPos x, vrl_ScreenPos y, vrl_Color color);
void vrl_DisplayLine(vrl_ScreenPos x1, vrl_ScreenPos y1, vrl_ScreenPos x2, vrl_ScreenPos y2, vrl_Color color);
void vrl_DisplayBox(vrl_ScreenPos x1, vrl_ScreenPos y1, vrl_ScreenPos x2, vrl_ScreenPos y2, vrl_Color color);
void vrl_DisplayText(vrl_ScreenPos x, vrl_ScreenPos y, vrl_Color color, char *message);

#define vrl_DisplayGetTextWidth(string) (*vrl_current_display_driver)(VRL_DISPLAY_GET_TEXTWIDTH, 0, string)
#define vrl_DisplayGetTextHeight(string) (*vrl_current_display_driver)(VRL_DISPLAY_GET_TEXTHEIGHT, 0, string)
#define vrl_DisplayGetSortOrder() (*vrl_current_display_driver)(VRL_DISPLAY_GET_SORTORDER, 0, NULL)
#define vrl_DisplayCanGouraud() (*vrl_current_display_driver)(VRL_DISPLAY_CAN_GOURAUD, 0, NULL)
#define vrl_DisplayCanXYclip() (*vrl_current_display_driver)(VRL_DISPLAY_CAN_XY_CLIP, 0, NULL)
#define vrl_DisplayGetVersion() (*vrl_current_display_driver)(VRL_DISPLAY_GET_VERSION, 0, NULL)
char *vrl_DisplayGetDescription(void);
#define vrl_DisplayBeginFrame() (*vrl_current_display_driver)(VRL_DISPLAY_BEGIN_FRAME, 0, NULL)
#define vrl_DisplayEndFrame() (*vrl_current_display_driver)(VRL_DISPLAY_END_FRAME, 0, NULL)
#define vrl_DisplaySetRaster(r) (*vrl_current_display_driver)(VRL_DISPLAY_SET_RASTER, 0, (r))
vrl_Raster *vrl_DisplayGetRaster(void);
#define vrl_DisplaySetZbuffer(r) (*vrl_current_display_driver)(VRL_DISPLAY_SET_Z_BUFFER, 0, (r))
#define vrl_DisplayGetZbuffer() (*vrl_current_display_driver)(VRL_DISPLAY_GET_Z_BUFFER, 0, NULL)
#define vrl_DisplayClearZbuffer(depth) (*vrl_current_display_driver)(VRL_DISPLAY_CLEAR_Z_BUFFER, depth, NULL)
#define vrl_DisplayUseZbuffer(flag) (*vrl_current_display_driver)(VRL_DISPLAY_USE_Z_BUFFER, flag, NULL)
#define vrl_DisplayUpdatePalette(palette) (*vrl_current_display_driver)(VRL_DISPLAY_UPDATE_PALETTE, 0, palette)
#define vrl_DisplaySetShading(value) (*vrl_current_display_driver)(VRL_DISPLAY_SET_SHADING, (value), NULL)

vrl_Color vrl_DisplayComputeColor(vrl_Surface *surf, vrl_Factor intensity, vrl_Factor ambient, vrl_Scalar depth);
void vrl_DisplayComputeVertexColor(vrl_OutputVertex *v, vrl_Surface *surf, vrl_Factor intensity, vrl_Factor ambient, vrl_Scalar depth);

void vrl_DisplaySetWindow(vrl_ScreenPos x1, vrl_ScreenPos y1, vrl_ScreenPos x2, vrl_ScreenPos y2);
void vrl_DisplayGetWindow(vrl_ScreenPos *x1, vrl_ScreenPos *y1, vrl_ScreenPos *x2, vrl_ScreenPos *y2);

#define vrl_DisplayGetWidth() vrl_RasterGetWidth(vrl_DisplayGetRaster())
#define vrl_DisplayGetHeight() vrl_RasterGetHeight(vrl_DisplayGetRaster())
#define vrl_DisplayGetDepth() vrl_RasterGetDepth(vrl_DisplayGetRaster())
#define vrl_DisplayUpdate() vrl_VideoBlit(vrl_DisplayGetRaster())

/* Video support */

typedef enum
		{
		VRL_VIDEO_GET_VERSION = 0, VRL_VIDEO_GET_DESCRIPTION,
		VRL_VIDEO_SETUP, VRL_VIDEO_SHUTDOWN, VRL_VIDEO_GET_MODE,
		VRL_VIDEO_SET_DRAW_PAGE, VRL_VIDEO_SET_VIEW_PAGE, VRL_VIDEO_GET_NPAGES,
		VRL_VIDEO_HAS_PALETTE, VRL_VIDEO_SET_PALETTE,
		VRL_VIDEO_SET_NTSC, VRL_VIDEO_CHECK_RETRACE,
		VRL_VIDEO_GET_RASTER, VRL_VIDEO_BLIT,
		VRL_VIDEO_CURSOR_HIDE, VRL_VIDEO_CURSOR_SHOW, VRL_VIDEO_CURSOR_RESET,
		VRL_VIDEO_CURSOR_MOVE, VRL_VIDEO_CURSOR_SET_APPEARANCE
		} vrl_VideoCommand;

typedef vrl_32bit vrl_VideoDriverFunction(vrl_VideoCommand cmd, vrl_32bit lparm, void *pparm1);
extern vrl_VideoDriverFunction *vrl_current_video_driver;
#define vrl_VideoSetDriver(driver) (vrl_current_video_driver = (driver))

#define vrl_VideoSetup(mode) (*vrl_current_video_driver)(VRL_VIDEO_SETUP, (mode), NULL)
void vrl_VideoShutdown(void);
#define vrl_VideoGetMode() (*vrl_current_video_driver)(VRL_VIDEO_GET_MODE, 0, NULL)
void vrl_VideoSetDrawPage(int page);
int vrl_VideoGetDrawPage(void);
void vrl_VideoSetViewPage(int page);
int vrl_VideoGetViewPage(void);
int vrl_VideoGetNpages(void);
#define vrl_VideoHasPalette() (*vrl_current_video_driver)(VRL_VIDEO_HAS_PALETTE, 0, NULL)
void vrl_VideoSetPalette(int start, int end, vrl_Palette *palette);  /* updates the palette */
void vrl_VideoGetPalette(int start, int end, vrl_Palette *palette);
#define vrl_VideoSetNTSC(flag) (*vrl_current_video_driver)(VRL_VIDEO_SET_NTSC, flag, NULL)
#define vrl_VideoCheckRetrace() (*vrl_current_video_driver)(VRL_VIDEO_CHECK_RETRACE, 0, NULL)
vrl_Raster *vrl_VideoGetRaster(void);
#define vrl_VideoBlit(buffer)  (*vrl_current_video_driver)(VRL_VIDEO_BLIT, 0, (buffer))
#define vrl_VideoGetVersion() (*vrl_current_video_driver)(VRL_VIDEO_GET_VERSION, 0, NULL)
char *vrl_VideoGetDescription(void);
#define vrl_VideoCursorHide() (*vrl_current_video_driver)(VRL_VIDEO_CURSOR_HIDE, 0, NULL)
#define vrl_VideoCursorShow() (*vrl_current_video_driver)(VRL_VIDEO_CURSOR_SHOW, 0, NULL)
#define vrl_VideoCursorReset() (*vrl_current_video_driver)(VRL_VIDEO_CURSOR_RESET, 0, NULL)
#define vrl_VideoCursorMove(x, y) (*vrl_current_video_driver)(VRL_VIDEO_CURSOR_MOVE, (((long) (x)) << 16) | (((long) (y)) & 0xFFFF), NULL)
#define vrl_VideoCursorSetAppearance(app) (*vrl_current_video_driver)(VRL_VIDEO_CURSOR_SET_APPEARANCE, app)

/* Stereoscopic viewing support */

typedef enum
	{
	VRL_STEREOEYE_NEITHER = 0,
	VRL_STEREOEYE_LEFT,
	VRL_STEREOEYE_RIGHT,
	VRL_STEREOEYE_BOTH
	} VRL_STEREO_EYE;

typedef enum
	{
	VRL_STEREOTYPE_NONE = 0, VRL_STEREOTYPE_SEQUENTIAL,
	VRL_STEREOTYPE_ANAGLYPH_SEQUENTIAL, VRL_STEREOTYPE_ANAGLYPH_WIRE_ALTERNATE,
	VRL_STEREOTYPE_ENIGMA, VRL_STEREOTYPE_FRESNEL,
	VRL_STEREOTYPE_CYBERSCOPE, VRL_STEREOTYPE_CRYSTALEYES,
	VRL_STEREOTYPE_CHROMADEPTH, VRL_STEREOTYPE_SIRDS,
	VRL_STEREOTYPE_TWOCARDS, VRL_STEREOTYPE_ANAGLYPH_SOLID_ALTERNATE
	} VRL_STEREO_TYPE;

void vrl_DisplayStereoSetType(VRL_STEREO_TYPE stype);
VRL_STEREO_TYPE vrl_DisplayStereoGetType(void);

void vrl_DisplayStereoSetDrawEye(VRL_STEREO_EYE eye);
VRL_STEREO_EYE vrl_DisplayStereoGetDrawEye(void);

void vrl_DisplayStereoSetViewEye(VRL_STEREO_EYE eye);
VRL_STEREO_EYE vrl_DisplayStereoGetViewEye(void);

void vrl_DisplayStereoSetLeftWindow(vrl_ScreenPos x1, vrl_ScreenPos y1, vrl_ScreenPos x2, vrl_ScreenPos y2);
void vrl_DisplayStereoSetRightWindow(vrl_ScreenPos x1, vrl_ScreenPos y1, vrl_ScreenPos x2, vrl_ScreenPos y2);

void vrl_DisplayStereoSetCardFunction(void (*function)(VRL_STEREO_EYE));

struct _vrl_stereo_conf      /* stores details about the stereo configuration */
	{
	VRL_STEREO_TYPE type;                  /* type of stereo viewing system */ 
	float eyespacing;                      /* eye spacing in millimeters */
	float convergence;                     /* convergence distance in millimeters */
	vrl_Angle leftrot, rightrot;           /* amount by which each eye is rotated around Y */
	int left_eye_shift, right_eye_shift;   /* extra offset due to lateral screen displacement */
	int center_offset;                     /* offset to center of image for each eye */
	int two_eyes;                          /* two eyes used */
	vrl_Scalar chromanear, chromafar;      /* bounds for Chromadepth */
	};

int vrl_StereoSetup(void);   /* creates a pair of cameras */
int vrl_StereoConfigure(vrl_StereoConfiguration *conf);
vrl_StereoConfiguration *vrl_StereoInitConfiguration(vrl_StereoConfiguration *conf);

#define vrl_StereoCreateConfiguration() vrl_StereoInitConfiguration(vrl_malloc(sizeof(vrl_StereoConfiguration)))

#define vrl_StereoSetType(conf, st_type) ((conf)->type = (st_type))
#define vrl_StereoGetType(conf) ((conf)->type)

#define vrl_StereoGetNeyes(conf) ((conf)->two_eyes)

#define vrl_StereoSetLeftEyeShift(conf, shift) ((conf)->left_eye_shift = (shift))
#define vrl_StereoGetLeftEyeShift(conf) ((conf)->left_eye_shift)
#define vrl_StereoSetRightEyeShift(conf, shift) ((conf)->right_eye_shift = (shift))
#define vrl_StereoGetRightEyeShift(conf) ((conf)->right_eye_shift)

#define vrl_StereoSetLeftEyeRotation(conf, rot) ((conf)->leftrot = (rot))
#define vrl_StereoGetLeftEyeRotation(conf) ((conf)->leftrot)
#define vrl_StereoSetRightEyeRotation(conf, rot) ((conf)->rightrot = (rot))
#define vrl_StereoGetRightEyeRotation(conf) ((conf)->rightrot)

#define vrl_StereoSetEyespacing(conf, spacing) ((conf)->eyespacing = (spacing))
#define vrl_StereoGetEyespacing(conf) ((conf)->eyespacing)

#define vrl_StereoSetConvergence(conf, conv) ((conf)->convergence = (conv))
#define vrl_StereoGetConvergence(conf) ((conf)->convergence)

#define vrl_StereoGetTotalLeftShift(conf) (-((conf)->center_offset + (conf)->left_eye_shift))
#define vrl_StereoGetTotalRightShift(conf) ((conf)->center_offset + (conf)->right_eye_shift)

#define vrl_StereoSetChromaNear(conf, val) ((conf)->chromanear = (val))
#define vrl_StereoGetChromaNear(conf) ((conf)->chromanear)

#define vrl_StereoSetChromaFar(conf, val) ((conf)->chromafar = (val))
#define vrl_StereoGetChromaFar(conf) ((conf)->chromafar)

/* Retrace handling */

void vrl_SetRetraceHandler(void (*function)(void));

/* XY clipping */

void vrl_DisplaySetXYclip(vrl_ScreenPos left, vrl_ScreenPos right, vrl_ScreenPos top, vrl_ScreenPos bottom);

vrl_OutputVertex *vrl_DisplayXYclipPoint(vrl_OutputVertex *vertices, unsigned int flags);
vrl_OutputVertex *vrl_DisplayXYclipLine(vrl_OutputVertex *vertices, unsigned int flags);
vrl_OutputVertex *vrl_DisplayXYclipPoly(vrl_OutputVertex *vertices, unsigned int flags);

#define VRL_DISPLAY_XYCLIP_CLIP_X     0x01
#define VRL_DISPLAY_XYCLIP_CLIP_Y     0x02
#define VRL_DISPLAY_XYCLIP_Z          0x04
#define VRL_DISPLAY_XYCLIP_INTENSITY  0x08

/* Timer routines */

vrl_Boolean vrl_TimerInit(void);
void vrl_TimerQuit(void);
vrl_Time vrl_TimerRead(void);
void vrl_TimerDelay(vrl_Time time);
#define vrl_TimerGetTickRate() (1000)

/* Mouse routines */

vrl_Boolean vrl_MouseInit(void);
void vrl_MouseQuit(void);
vrl_Boolean vrl_MouseReset(void);
vrl_Boolean vrl_MouseRead(int *x, int *y, unsigned int *buttons);
void vrl_MouseSetUsage(int u);
int vrl_MouseGetUsage(void);
void vrl_MouseSetPointer(void *u);
void *vrl_MouseGetPointer(void);

/* Keyboard routines */

vrl_Boolean vrl_KeyboardCheck(void);
unsigned int vrl_KeyboardRead(void);

/* System routines */

vrl_Boolean vrl_SystemStartup(void);
void vrl_SystemRun(void);
vrl_RenderStatus *vrl_SystemRender(vrl_Object *list);
vrl_Time vrl_SystemGetRenderTime(void);
vrl_Time vrl_SystemGetFrameRate(void);
void vrl_SystemCommandLine(int argc, char *argv[]);

void vrl_SystemRequestRefresh(void);
vrl_Boolean vrl_SystemQueryRefresh(void);

void vrl_SystemStartRunning(void);
void vrl_SystemStopRunning(void);
vrl_Boolean vrl_SystemIsRunning(void);

/* Routines defined by application */

void vrl_ApplicationDrawUnder(void);
void vrl_ApplicationDrawOver(vrl_RenderStatus *stat);
void vrl_ApplicationInit(void);
void vrl_ApplicationKey(unsigned int c);
void vrl_ApplicationMouseUp(int x, int y, unsigned int buttons);

/* User interface routines */

void vrl_UserInterfaceBox(int w, int h, int *x, int *y);
void vrl_UserInterfacePopText(char *text[]);
void vrl_UserInterfacePopMsg(char *msg);
int vrl_UserInterfacePopMenu(char *text[]);
int vrl_UserInterfaceMenuDispatch(char *text[], int (**funcs)(void));
unsigned vrl_UserInterfacePopPrompt(char *prompt, char *buff, int n);
int vrl_UserInterfaceDismiss(void);

void vrl_UserInterfaceDrawCompass(vrl_Camera *camera, int compass_x, int compass_y, int compass_armlen);
void vrl_UserInterfaceDropText(int x, int y, vrl_Color color, char *text);

/* Pseudo-tasking routines */

vrl_Boolean vrl_TaskCreate(void (*function)(void), void *data, vrl_Time period);
void vrl_TaskRun(void);
void *vrl_TaskGetData(void);
vrl_Time vrl_TaskGetElapsed(void);
vrl_Time vrl_TaskGetTimeNow(void);

/* Routines for creating primitive shapes */

vrl_Shape *vrl_PrimitiveBox(vrl_Scalar width, vrl_Scalar height, vrl_Scalar depth, vrl_Surfacemap *map);
vrl_Shape *vrl_PrimitiveCone(vrl_Scalar radius, vrl_Scalar height, int nsides, vrl_Surfacemap *map);
vrl_Shape *vrl_PrimitiveCylinder(vrl_Scalar bottom_radius, vrl_Scalar top_radius, vrl_Scalar height, int nsides, vrl_Surfacemap *map);
vrl_Shape *vrl_PrimitivePrism(vrl_Scalar width, vrl_Scalar height, vrl_Scalar depth, vrl_Surfacemap *map);
vrl_Shape *vrl_PrimitiveSphere(vrl_Scalar radius, int vsides, int hsides, vrl_Surfacemap *map);

/* Device support */

typedef enum
		{
		VRL_DEVICE_INIT = 0, VRL_DEVICE_QUIT, VRL_DEVICE_POLL,
		VRL_DEVICE_RESET, VRL_DEVICE_SET_RANGE
		} vrl_DeviceCommand;

typedef int vrl_DeviceDriverFunction(vrl_DeviceCommand cmd, vrl_Device *device);
typedef int vrl_DeviceOutputFunction(vrl_Device *device, int parm1, vrl_Scalar parm2);

typedef enum
	{
	VRL_MOTION_NONE = 0, VRL_MOTION_RELATIVE, VRL_MOTION_ABSOLUTE
	} vrl_DeviceMotionMode;

typedef int vrl_Buttonmap[][2];

/* In the following struct, fields marked with a (^) can be set by the
   device function; not all of them need to be, since reasonable defaults
   are already set for each.  Fields marked with a (^^) must be set for
   every device. */

struct _vrl_device
	{
	char *nickname;             /* name used by application for this device */
	int version;                /* device struct version number (^) */
	char *desc;                 /* user-readable device description (^^) */
	vrl_DeviceDriverFunction *fn;   /* the driver function itself */
	vrl_SerialPort *port;       /* pointer to serial port */
	vrl_DeviceMotionMode rotation_mode;     /* rotation mode for this device (^) */
	vrl_DeviceMotionMode translation_mode;  /* translation mode for this device (^) */
	int mode;                   /* mode of operation (^) */
	vrl_Time period;            /* milliseconds between reads (^) */
	vrl_Time lastread;          /* timer value when we last read */
	int nbuttons;               /* number of buttons the device has (^) */
	vrl_unsigned32bit buttons;      /* current button status */
	vrl_unsigned32bit bchanged;     /* which buttons have changed state */
	vrl_Buttonmap *buttonmap;   /* button mapping table for 2D devices */
	int noutput_channels;       /* number of output channels (^) */
	vrl_DeviceOutputFunction *outfunc;  /* function to call to generate output (^) */
	int nchannels;              /* number of input channels the device has (^^) */
	vrl_DeviceChannel *channels;   /* pointer to array of channels (^^) */
	void *localdata;            /* data used only by the driver (^) */
	vrl_Device *next;           /* devices are kept in a linked list */
	};

vrl_Device *vrl_DeviceOpen(vrl_DeviceDriverFunction fn, vrl_SerialPort *port);
void vrl_DeviceClose(vrl_Device *device);
int vrl_DevicePoll(vrl_Device *device);
int vrl_DeviceReset(vrl_Device *device);
int vrl_DeviceSetRange(vrl_Device *device);
int vrl_DevicePollAll(void);
void vrl_DeviceCloseAll(void);
void vrl_DeviceOutput(vrl_Device *device, int channel, vrl_Scalar value);
vrl_Device *vrl_DeviceGetFirst(void);
vrl_Device *vrl_DeviceFind(char *nickname);

#define vrl_DeviceGetNickname(device) ((device)->nickname)
#define vrl_DeviceSetNickname(device, name) ((device)->nickname = (name))
#define vrl_DeviceGetNext(dev) ((dev)->next)
#define vrl_DeviceGetDesc(device) ((device)->desc)
#define vrl_DeviceSetPeriod(device, per) ((device)->period = (per))
#define vrl_DeviceGetPeriod(device) ((device)->period)
#define vrl_DeviceGetMode(device) ((device)->mode)
#define vrl_DeviceSetMode(device, m) ((device)->mode = (m))
#define vrl_DeviceGetPort(device) ((device)->port)
#define vrl_DeviceGetNchannels(device) ((device)->nchannels)
#define vrl_DeviceGetNButtons(device) ((device)->nbuttons)
#define vrl_DeviceGetButtons(device) ((device)->buttons)
#define vrl_DeviceGetChangedButtons(device) ((device)->bchanged)
#define vrl_DeviceGetNOutputChannels(device) ((device)->noutput_channels)
#define vrl_DeviceGetRotationMode(device) ((device)->rotation_mode)
#define vrl_DeviceGetTranslationMode(device) ((device)->translation_mode)

#define vrl_DeviceGetButtonmap(device) ((device)->buttonmap)
#define vrl_DeviceSetButtonmap(device, map) ((device)->buttonmap = (map))

struct _vrl_device_channel
	{
	vrl_32bit centerpoint;        /* value of center point in raw device coords */
	vrl_32bit deadzone;           /* minimum acceptable value in device coords */
	vrl_32bit range;              /* maximum absolute value relative to zero */
	vrl_Scalar scale;             /* maximum returned value */
	vrl_Boolean accumulate : 1;   /* if set, accumulate values */
	vrl_Boolean changed : 1;      /* set if rawvalue has changed */
	vrl_32bit rawvalue;           /* current value in raw device coordinates */
	vrl_32bit oldvalue;           /* previous value in raw device coordinates */
	vrl_Scalar value;             /* current value, clipped, centered and deadzoned and accumulated */
	};

#define vrl_DeviceGetCenter(device, channel) ((device)->channels[channel].centerpoint)
#define vrl_DeviceGetDeadzone(device, channel) ((device)->channels[channel].deadzone)
#define vrl_DeviceSetDeadzone(device, channel, value) ((device)->channels[channel].deadzone = (value))
#define vrl_DeviceGetScale(device, channel) ((device)->channels[channel].scale)
#define vrl_DeviceSetScale(device, channel, value) ((device)->channels[channel].scale = (value))
#define vrl_DeviceGetAccumulate(device, channel) ((device)->channels[channel].accumulate)
#define vrl_DeviceSetAccumulate(device, channel, value) ((device)->channels[channel].accumulate = (value))
#define vrl_DeviceGetChanged(device, channel) ((device)->channels[channel].changed)
#define vrl_DeviceGetRawValue(device, channel) ((device)->channels[channel].rawvalue)
#define vrl_DeviceGetValue(device, channel) ((device)->channels[channel].value)

/* Serial input/output support */

typedef enum
		{
		VRL_PARITY_EVEN = 0, VRL_PARITY_ODD, VRL_PARITY_NONE
		} vrl_ParityType;

struct _vrl_serial_port {
	unsigned int address;  /* hardware address */
	int irq;               /* IRQ level (*not* interrupt number!) */
	int buffsize;          /* size of the buffer */
	char *buffer;          /* pointer to the buffer */
	int in, out;           /* offsets into buffer */
	vrl_SerialPort *next;  /* serial ports are kept in a linked list */
	};

vrl_SerialPort *vrl_SerialOpen(unsigned int address, int irq, unsigned int buffsize);
void vrl_SerialClose(vrl_SerialPort *port);
void vrl_SerialCloseAll(void);
void vrl_SerialSetParameters(vrl_SerialPort *port, unsigned int baud, vrl_ParityType parity, int databits, int stopbits);
vrl_Boolean vrl_SerialCheck(vrl_SerialPort *port);
unsigned int vrl_SerialGetc(vrl_SerialPort *port);
void vrl_SerialPutc(unsigned int c, vrl_SerialPort *port);
void vrl_SerialPutString(unsigned char *s, vrl_SerialPort *p);
void vrl_SerialFlush(vrl_SerialPort *p);
void vrl_SerialSetDTR(vrl_SerialPort *port, vrl_Boolean value);
void vrl_SerialSetRTS(vrl_SerialPort *port, vrl_Boolean value);
void vrl_SerialFifo(vrl_SerialPort *p, int n);

typedef struct _device_packet_buffer
	{
	int buffsize;           /* size of a packet */
	int ind;                /* current index into buffer */
	unsigned char *buffer;  /* pointer to the buffer itself */
	} vrl_DevicePacketBuffer;

vrl_DevicePacketBuffer *vrl_DeviceCreatePacketBuffer(int buffsize);
void vrl_DeviceDestroyPacketBuffer(vrl_DevicePacketBuffer *buff);
vrl_Boolean vrl_DeviceGetPacket(vrl_SerialPort *port, vrl_DevicePacketBuffer *buff);
#define vrl_DevicePacketGetBuffer(buff) (((vrl_DevicePacketBuffer *)buff)->buffer)

/* Configuration file support */

void vrl_ConfigSetCompassDisplay(vrl_Boolean flag);
vrl_Boolean vrl_ConfigGetCompassDisplay(void);
void vrl_ConfigToggleCompassDisplay(void);
void vrl_ConfigSetPositionDisplay(vrl_Boolean flag);
vrl_Boolean vrl_ConfigGetPositionDisplay(void);
void vrl_ConfigTogglePositionDisplay(void);
void vrl_ConfigSetFramerateDisplay(vrl_Boolean flag);
vrl_Boolean vrl_ConfigGetFramerateDisplay(void);
void vrl_ConfigToggleFramerateDisplay(void);

int vrl_ReadCFG(FILE *in);
int vrl_ReadCFGfile(char *filename);
int vrl_ReadCFGProcessLine(char *buff);
void vrl_ConfigStartup(char *filename);

/* PCX file support */

vrl_Boolean vrl_ReadPCX(FILE *in);
vrl_Boolean vrl_WritePCX(FILE *out);

/*
   These #defines are provided for the sake of backward compatability to
   release 1.1 of AVRIL; they should no longer be used.
 */

/* Cursors are now part of the display module, not the mouse module */
#define vrl_MouseCursorHide() vrl_DisplayCursorHide()
#define vrl_MouseCursorShow() vrl_DisplayCursorShow()

/* Highlighting and wireframing are now part of the rendering module, not
   the display module */
#define vrl_DisplaySetDrawMode(mode) vrl_RenderSetDrawMode(mode)
#define vrl_DisplayGetDrawMode() vrl_RenderGetDrawMode()
#define vrl_DisplaySetWireframeColor(color) vrl_RenderSetWireframeColor(color)
#define vrl_DisplayGetWireframeColor() vrl_RenderGetWireframeColor()
#define vrl_DisplaySetHighlightColor(color) vrl_RenderSetHighlightColor(color)
#define vrl_DisplayGetHighlightColor() vrl_RenderGetHighlightColor()

/* The old vrl_ObjectGetRotations() function is now replaced by
   vrl_ObjectGetWorldRotations(), for the sake of consistency; same for
   the routines which get an object's location. */

#define vrl_ObjectGetRotations(object, rx, ry, rz) vrl_ObjectGetWorldRotations((object), (rx), (ry), (rz))
#define vrl_ObjectGetX(object) vrl_ObjectGetWorldX(object)
#define vrl_ObjectGetY(object) vrl_ObjectGetWorldY(object)
#define vrl_ObjectGetZ(object) vrl_ObjectGetWorldZ(object)
#define vrl_ObjectGetLocation(object, v) vrl_ObjectGetWorldLocation((object), (v))

/* Same for cameras */

#define vrl_CameraGetRotations(camera, rx, ry, rz) vrl_CameraGetWorldRotations((camera), (rx), (ry), (rz))
#define vrl_CameraGetX(camera) vrl_CameraGetWorldX(camera)
#define vrl_CameraGetY(camera) vrl_CameraGetWorldY(camera)
#define vrl_CameraGetZ(camera) vrl_CameraGetWorldZ(camera)
#define vrl_CameraGetLocation(camera, v) vrl_CameraGetWorldLocation((camera), (v))

/* Same for lights */

#define vrl_LightGetRotations(light, rx, ry, rz) vrl_LightGetWorldRotations((light), (rx), (ry), (rz))
#define vrl_LightGetX(light) vrl_LightGetWorldX(light)
#define vrl_LightGetY(light) vrl_LightGetWorldY(light)
#define vrl_LightGetZ(light) vrl_LightGetWorldZ(light)
#define vrl_LightGetLocation(light, v) vrl_LightGetWorldLocation((light), (v))

/* Not currently documented in the manual, mostly because they're in a state
   of flux:

    vrl_ShapeInit()   
    vrl_ShapeCreate()
    vrl_ShapeComputeBounds()
    vrl_ShapeGetRadius()
    vrl_ShapeGetCenter()
    vrl_ShapeGetMinbounds()
    vrl_ShapeGetMaxbounds()
    vrl_RepInit()
    vrl_RepCreate()
    vrl_RepAddFacet()
    vrl_RepGetFacet()
    vrl_RepHasNormals()
    vrl_RepGetNormal()
    vrl_FacetInit()
    vrl_FacetCerate()
    vrl_FacetTraverse())
    vrl_FacetComputeNormal()
    vrl_FacetSetInterior()
    vrl_FacetGetInterior()
    vrl_FacetToggleInterior()
    vrl_FacetGetNormal()
    vrl_FacetSetPoint()
    vrl_FacetGetPoint()
    vrl_WritePLG()
    vrl_ObjectSavePLGfile()
    vrl_SetRetraceHandler()
    vrl_DisplayGetSortOrder()
    vrl_DisplayComputeColor()
    vrl_DisplayComputeVertexColor()
    vrl_VideoSetNTSC()
    vrl_Texture typedef
    vrl_SurfaceSetTexture()
    vrl_SurfaceGetTexture()
    everything related to the XY clipping
    all the configuration file routines

 */

/* End of avril.h */
