#version 120
//hybrid system

#define MIN_EPSILON 3e-7
varying vec3 dir; //from vertex shader
uniform ivec2 size;//size of image in pixels
uniform vec3 eye;
uniform float time, maxDepth, udef[32];//timing and user defined params
const int steps=64,shadowsteps=32;
float pixelScale=1.0/max(float(size.x),float(size.y));
const float aocc=0.75,aoe=0.75,spec=1.0,specExp=32.0,shadows=0.2;//light params
vec3 cBGround=vec3(0.0,0.0,0.0),cLight=vec3(1.0,1.0,1.0),diffuse,ambient;
vec3 posLight=eye+vec3(0.0,1.0,0.0),normal,light,halfLV;
//uniform sampler2D tex; //access with texture2D(tex,pt)
float minDist=16.0;
bool bColoring=false;

void Scale(inout vec4 z, float scale){
	z=vec4(scale*z.xyz,z.w*abs(scale));
}
void ScaleOffset(inout vec4 z, float scale, vec3 offset) { 
	 z = z*scale - vec4(offset,0.0)*(scale-1.0);
}
void Rotate2D(inout vec2 z, float angle){//usage: Rotate2D(z.xy,0.785);
	float csat = cos(angle),ssat = sin(angle);
	z*=mat2(csat,ssat,-ssat,csat);
}
void Rotate3D(inout vec4 z, vec3 rot){//vec3(pitch,yaw,roll)
	float cx=cos(rot.x),cy=cos(rot.y),cz=cos(rot.z),sx=sin(rot.x),sy=sin(rot.y),sz=sin(rot.z);
	z*=mat3(cz*cy+sz*sx*sy,sz*cx,-cz*sy+sz*sx*cy,-sz*cy+cz*sx*sy,cz*cx,sz*sy+cz*sx*cy,cx*sy,-sx,cx*cy);
}
void SphereFold(inout vec4 z, float maxRadius){//>1.0
	z*=clamp(1.0/dot(z.xyz,z.xyz),1.0,maxRadius);
}
void CylinderFold(inout vec3 z, float maxRadius){//usage CylinderFold(z.xzw);
	z*=clamp(1.0/dot(z.xy,z.xy),1.0,maxRadius);
}
void CrossFold(inout vec4 z, float maxRadius){
	z*=clamp(1.0/max(dot(z.xy,z.xy),dot(z.xz,z.xz)),1.0,maxRadius);
}
void SquareFold(inout vec4 z, float maxRadius){
	z*=clamp(1.0/max(z.x*z.x,max(z.y*z.y,z.z*z.z)),1.0,maxRadius);
}
void SingleFold(inout vec4 z, float maxRadius){
	z*=clamp(1.0/z.x*z.x,1.0,maxRadius);
}
void KaliFold(inout vec4 z, float scale){
	z*=scale/dot(z.xyz,z.xyz);
}
void BoxFold(inout vec4 z){
	z.xyz=clamp(z.xyz,-1.0,1.0)*2.0-z.xyz;
}
void MembraneFold(inout vec2 z){//usage: MembraneFold(z.xz);
	z.xy=clamp(z.xy, -1.0, 1.0) *2.0-z.xy;
}
void AbsFold(inout vec4 z){
	z.xyz=abs(z.xyz+1.0)-1.0;
}
void GenericFold(inout vec4 z, vec3 normal){
	z.xyz-=2.0*min(0.0,dot(z.xyz,normal))*normal;
}
void FunkyReflect(inout vec4 z){
	z.xyz=abs(z.zyx);
	if(z.x<z.y)z.xy=z.yx;
}
void MengerReflect(inout vec4 z){
	z = abs(z);
	if (z.x<z.y)z.xy = z.yx;
	if (z.x<z.z)z.xz = z.zx;
	if (z.y<z.z)z.yz = z.zy;
}
void Menger(inout vec4 z, float scale, vec3 offset) {//3.0,vec3(1.0)
	MengerReflect(z);
	ScaleOffset(z,scale,offset);
	if(z.z<-0.5*offset.z*(scale-1.0))z.z+=offset.z*(scale-1.0);
}
void OctoReflect(inout vec4 z){
	if (z.x+z.y<0.0) z.xy = -z.yx;
	if (z.x+z.z<0.0) z.xz = -z.zx;
	if (z.x-z.y<0.0) z.xy = z.yx;
	if (z.x-z.z<0.0) z.xz = z.zx;
}
void Octo(inout vec4 z, float scale, vec3 offset) {//2.0,vec3(1.0,0.0,0.0)
	OctoReflect(z);
	ScaleOffset(z,scale,offset);
}
void BulbFold(inout vec4 z, float r, float power) {
	float zi = atan( z.y,z.x ) * power;
	float zo = asin( z.z/r ) * power;//+time
	float zr = pow( r, power-1.0 );
	z = zr*vec4(r*vec3(cos(zo)*cos(zi), cos(zo)*sin(zi), sin(zo)),z.w*power);
}
void BulbFold2(inout vec4 z, float r, float power) {
	float zi = atan( z.y,z.x ) * power;
	float zo = asin( z.z/r ) * power;//+time
	float zr = pow( r, power-1.0 );
	z = zr*vec4(r*vec3(sin(zo)*cos(zi), sin(zo)*sin(zi), cos(zo)),z.w*power);
}
void PolyBulbFold(inout vec4 z, float r, float c1, float p1, float c2, float p2) {
	float zi = atan( z.y,z.x );
	float zo = asin( z.z/r );
	float zr1 = pow( r, p1-1.0 ),zr2=pow(r,p2-1.0);
	z = zr1*c1*vec4(r*vec3(cos(zo*p1)*cos(zi*p1), cos(zo*p1)*sin(zi*p1), sin(zo*p1)),z.w*p1) +
		zr2*c2*vec4(r*vec3(cos(zo*p2)*cos(zi*p2), cos(zo*p2)*sin(zi*p2), sin(zo*p2)),z.w*p2);
}

float BulbShape(in vec4 z, float radius){return 0.5*log(radius)*radius/z.w;}
float SphereShape(in vec4 z, float radius){return (length(z.xyz)-radius)/z.w;}
float BoxShape(in vec4 z, float radius){return (max(abs(z.x),max(abs(z.y),abs(z.z)))-radius)/z.w;}
float CylinderShape(in vec4 z, float radius, float height){return (max(abs(z.y)-0.5*height,length(z.xz))-radius)/z.w;}
float TorusShape(in vec4 z, float radius1, float radius2){return (length(vec2(length(z.xy)-radius1,z.z))-radius2)/z.w;}
float PrismShape(in vec4 z, float width, float len){return max(abs(z.z)-len,max(-z.y,(max(abs(z.x+z.y),abs(z.x-z.y))-width)))/z.w;}
float OctoShape(in vec4 z, float radius){return (max(abs(z.x+z.y+z.z),max(abs(-z.x-z.y+z.z),max(abs(-z.x+z.y-z.z),abs(z.x-z.y-z.z))))-radius)/z.w;}
float PKBasicShape(in vec4 z, float thickness){return max(length(z.xy)-1.0,abs(length(z.xy)*z.z-thickness)/sqrt(dot(z.xyz,z.xyz)+abs(thickness)))/z.w;}

#define Union min
#define Intersection max 
#define Blend mix
float Difference(float d1, float d2){return max(d1,-d2);}
float BlobbyUnion(float a, float b, float mx){a+=mx;b+=mx;return (a+b-sqrt(abs(a*a+b*b-2.0*(1.0-mx)*a*b)))/(2.0-mx)-mx;}
float BlobbyIntersection(float a, float b, float mx){a+=mx;b+=mx;return (0.25+0.5-0.5*mx)/(2.0-mx)*(a+b+sqrt(a*a+b*b-(2.0-2.0*mx)*a*b))-mx;}

float DE2(in vec3 z0){//this is a sample second estimate for blends etc
	vec4 z=vec4(z0,1.0),p0=z;
	float r=length(z0.xyz);
	for(int n=0;n<4 && r < 10.0;n++){BulbFold2(z,r,4.0);z+=p0;r=length(z.xyz);}
	return BulbShape(z,r);
}

float DE(in vec3 z0){
	vec4 z = vec4(z0,1.0),p0=z;
	float r=length(z0.xyz);
	minDist = min(minDist,dot(z.xyz,z.xyz));
	for (int n = 0; n < 7 && r < 10.0; n++) {
		BulbFold(z,r,4.0);
		z+=p0;
		r=length(z.xyz);
		minDist = min(minDist,dot(z.xyz,z.xyz));		
	} 
	if(bColoring){
		diffuse=vec3(0.7,0.3,0.1)+z0*0.125;
		ambient=diffuse*0.25;
  	}
	return BulbShape(z,r);
}

float fakeAO(vec3 ray, vec3 norm, float eps) {//from rrrola
	float ao_eps=eps*100.0,ao=1.0,w=0.1/ao_eps,dist=2.0*ao_eps;
	for (int i=0; i<5; i++) {
		float D = DE(ray + norm*dist);
		ao -= (dist-D) * w; w *= 0.5; dist = dist*2.0 - ao_eps;
	}
  	return clamp(ao, 0.85, 1.0);
}

void main() {
	vec3 ray = eye, rayDir = normalize(dir);
	vec4 color=vec4(cBGround,1.0);
	int iSteps;
	float rayLen=0.0,dist=maxDepth,eps=MIN_EPSILON;
	for(iSteps=0;iSteps<steps && dist>=eps && rayLen<=maxDepth;iSteps++){
		rayLen+=dist=DE(ray+rayLen*rayDir);
		eps=max(MIN_EPSILON,pixelScale*rayLen);
	}
	ray+=rayLen*rayDir;
	bColoring=true;
	if(dist<eps){//we hit the fractal
		float d=DE(ray);
		bColoring=false;
		float cheapAO=(1.0 - max(1.0 - minDist, 0.0) * aocc)*(1.0 - (float(iSteps) / float(steps))*aoe);//(1.0 - max(1.0 - minDist, 0.0) * aocc)*
		vec2 ve=vec2(eps,0.0);
		normal=normalize(vec3(-DE(ray-ve.xyy)+DE(ray+ve.xyy),-DE(ray-ve.yxy)+DE(ray+ve.yxy),-DE(ray-ve.yyx)+DE(ray+ve.yyx)));
		//cheapAO*=fakeAO(ray,normal,eps);
		light=normalize(posLight+rayDir);
		halfLV = normalize(light - rayDir);
 		diffuse=(dot(normal,light)*0.5+0.75)*diffuse+ambient;
		/* 
		//FAKE SOFT SHADOWS
		vec3 light_direction=ray-posLight;
		float srl=0.0,mxlen=length(light_direction)-eps*4.0;
		light_direction=normalize(light_direction);
		ray=posLight;dist=minDist=maxDepth;//reset marching orders
		int iSSteps;
		for (iSSteps= 0;iSSteps<shadowsteps && eps<dist && mxlen>srl;iSSteps++) {
			srl+=dist = DE(ray);
			ray += dist * light_direction;// March ray forward 
		}	
		if(dist<eps && srl<mxlen) diffuse *= 1.0 - shadows;
		else {
			diffuse+=spec*pow(max(dot(normal,halfLV),0.0),specExp)*cLight;
			diffuse*=1.0 - shadows*float(iSSteps) / float(shadowsteps)*aoe;
		}
		//END OF SHADOWS
		*/
		diffuse+=spec*pow(clamp(dot(normal,halfLV),0.0,1.0),specExp)*cLight; //comment for shadows
		color.rgb=diffuse*cheapAO;
	}else if(iSteps==steps)color.rgb*=0.0;
	color.rgb=mix(color.rgb,cBGround,clamp(log(3.0*rayLen/maxDepth),0.0,1.0));//FOG
	color.rgb=mix(color.rgb,vec3(0.0,0.05,0.1),float(iSteps)/float(steps));//GLOW
	gl_FragColor = clamp(color, 0.0, 1.0);
	gl_FragDepth = min(rayLen/maxDepth,0.99);
}


