/*  Copyright Daryl Gray 2001 email daryl.g@visto.com
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>

#include "gtksortheading.h"


static GtkHBoxClass
*gtk_sort_heading_parent_class;
static void
gtk_sort_heading_class_init				(GtkSortHeadingClass	*Klass);
static void
gtk_sort_heading_init					(GtkSortHeading		*sh);
static void
gtk_sort_heading_set_group				(GtkSortHeading		*sh,
							GSList			*group);
static void
gtk_sort_heading_set_others_to_off			(GtkSortHeading		*sh);
static void
gtk_sort_heading_destroy				(GtkObject		*sh);
static gboolean
gtk_sort_heading_eventbox_event 			(GtkWidget		*widget,
							GdkEvent		*event,
							GtkSortHeading		*sh);



enum {
	MODE_CHANGED,
	LAST_SIGNAL
};

static gint
gtk_sort_heading_signals[LAST_SIGNAL] = { 0 };

struct _GtkSortHeadingPrivate {
	GtkCList *clist;
	gint clist_col;
	gint sort_col;
	gint ident;
	GtkWidget *eventbox;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *frame;
	GtkWidget *arrow;
	GtkSortHeadingMode *mode;
};

GtkType
gtk_sort_heading_get_type 	  			(void)
{
	static GtkType type = 0;

	if (!type) {
		GtkTypeInfo info = {
			"GtkSortHeading",
			sizeof (GtkSortHeading),
			sizeof (GtkSortHeadingClass),
			(GtkClassInitFunc) gtk_sort_heading_class_init,
			(GtkObjectInitFunc) gtk_sort_heading_init,
			NULL, /* reserved 1 */
			NULL, /* reserved 2 */
			(GtkClassInitFunc) NULL
		};

		type = gtk_type_unique (gtk_hbox_get_type (), &info);
	}

	return type;
}

void
gtk_sort_heading_set_group				(GtkSortHeading 	*sh,
							GSList			*group)
{
	gint gslist_len;
	
	gslist_len = g_slist_length (group);
	++gslist_len;
	sh->priv->ident = gslist_len;
	sh->group = g_slist_append (group, sh);
	
}

GSList
*gtk_sort_heading_group					(GtkSortHeading		*sh)
{
	return sh->group;
}

void
gtk_sort_heading_set_others_to_off			(GtkSortHeading		*sh)
{
	gint gslist_len, i;
	GSList *node;
	
	node = sh->group;
	gslist_len = g_slist_length (node);
	for (i = 0; i < gslist_len; i++) {
		if (GTK_SORT_HEADING (node->data)->priv->ident != sh->priv->ident) {
			gtk_sort_heading_set_mode (GTK_SORT_HEADING (node->data), GTK_SORT_HEADING_OFF);
		}
		node = g_slist_next (node);
	}
}

GtkWidget
*gtk_sort_heading_new					(GSList			*group,
							GtkCList		*parent_clist,
							gint			column,
							const gchar		*label)
{
	GtkWidget *sh;
	
	sh = gtk_type_new (gtk_sort_heading_get_type ());
	GTK_SORT_HEADING (sh)->priv->clist = parent_clist;
	GTK_SORT_HEADING (sh)->priv->clist_col = column;
	
	gtk_sort_heading_set_group (GTK_SORT_HEADING (sh), group);
	if (label)
		gtk_sort_heading_set_label (GTK_SORT_HEADING (sh), label);
	
	gtk_sort_heading_set_sort_column (GTK_SORT_HEADING (sh), column);
	gtk_sort_heading_set_mode (GTK_SORT_HEADING (sh), GTK_SORT_HEADING_ASCENDING);
	return GTK_WIDGET (sh);
}

void
gtk_sort_heading_set_new_group				(GtkSortHeading		*sh,
							GSList			*ngroup)
{
	g_return_if_fail (sh != NULL);
	g_return_if_fail (GTK_IS_SORT_HEADING (sh));
	
	sh->group = ngroup;
}

void
gtk_sort_heading_destroy				(GtkObject		*sh)
{
	GtkSortHeading *gtsh;
	gint gslist_len,i;
	GSList *node;
	
	g_return_if_fail (sh != NULL);
	g_return_if_fail (GTK_IS_SORT_HEADING (sh));
	
	gtsh = GTK_SORT_HEADING (sh);
	gtk_object_destroy (sh);
	gtsh->group = g_slist_remove (gtsh->group, (gpointer *) gtsh);
	gslist_len = g_slist_length (gtsh->group);
	node = gtsh->group;
	for (i = 0; i < gslist_len; i++){
		if (GTK_SORT_HEADING (node->data)->priv->ident != gtsh->priv->ident) {
			gtk_sort_heading_set_new_group (GTK_SORT_HEADING (node->data), gtsh->group);
		}
		node = g_slist_next (node);
	}
	g_free (gtsh->priv->mode);
	g_free (gtsh->priv);
	if (gslist_len == 0){
		g_slist_free (gtsh->group);
	}
		
	

}

void
gtk_sort_heading_set_label				(GtkSortHeading		*sh,
							const gchar		*label)
{
	g_return_if_fail (sh != NULL);
	g_return_if_fail (GTK_IS_OBJECT (GTK_OBJECT (sh)));
	g_return_if_fail (GTK_IS_SORT_HEADING (GTK_OBJECT (sh)));
	
	gtk_label_set_text (GTK_LABEL (sh->priv->label), label);
}

void
gtk_sort_heading_set_label_justify			(GtkSortHeading		*sh,
							GtkJustification	jtype)
{
	g_return_if_fail (sh != NULL);
	g_return_if_fail (GTK_IS_OBJECT (GTK_OBJECT (sh)));
	g_return_if_fail (GTK_IS_SORT_HEADING (GTK_OBJECT (sh)));
	
	gtk_label_set_justify (GTK_LABEL (sh->priv->label), jtype);
	if (jtype == GTK_JUSTIFY_RIGHT){
		gtk_misc_set_alignment (GTK_MISC (sh->priv->label), 1, 0.5);
	}else if (jtype == GTK_JUSTIFY_CENTER){
		gtk_misc_set_alignment (GTK_MISC (sh->priv->label), 0.5, 0.5);
	}else{
		gtk_misc_set_alignment (GTK_MISC (sh->priv->label), 0, 0.5);
	}
}

GtkSortHeadingMode
gtk_sort_heading_get_mode				(GtkSortHeading		*sh)
{
	g_return_val_if_fail (sh != NULL, GTK_SORT_HEADING_UNKNOWN);
	g_return_val_if_fail (GTK_IS_OBJECT (GTK_OBJECT (sh)), GTK_SORT_HEADING_UNKNOWN);
	g_return_val_if_fail (GTK_IS_SORT_HEADING (GTK_OBJECT (sh)),  GTK_SORT_HEADING_UNKNOWN);
	
	return (*sh->priv->mode);
}

gboolean
gtk_sort_heading_eventbox_event				(GtkWidget		*widget,
							GdkEvent		*event,
							GtkSortHeading		*sh)
{
	if (event->type == GDK_BUTTON_RELEASE) {
		if (event->button.button == 1) {
			if ((*sh->priv->mode == GTK_SORT_HEADING_DESCENDING) || (*sh->priv->mode == GTK_SORT_HEADING_OFF)) {
				gtk_sort_heading_set_mode (sh, GTK_SORT_HEADING_ASCENDING);
			}else if (*sh->priv->mode == GTK_SORT_HEADING_ASCENDING) {
				gtk_sort_heading_set_mode (sh, GTK_SORT_HEADING_DESCENDING);
			}
			return TRUE;
		}
	}
	return FALSE;
}

void
gtk_sort_heading_set_mode				(GtkSortHeading		*gcsh,
							GtkSortHeadingMode	mode)
{
	g_return_if_fail (gcsh != NULL);
	g_return_if_fail (GTK_IS_OBJECT (GTK_OBJECT (gcsh)));
	g_return_if_fail (GTK_IS_SORT_HEADING (GTK_OBJECT (gcsh)));
	
	if (mode != *gcsh->priv->mode) {
		*gcsh->priv->mode = mode;
		gtk_clist_freeze (GTK_CLIST (gcsh->priv->clist));
		if (gcsh->priv->arrow) {
			gtk_widget_destroy (gcsh->priv->arrow);
			gcsh->priv->arrow = NULL;
		}
		if (mode == GTK_SORT_HEADING_ASCENDING) {
			gcsh->priv->arrow = gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_IN);
			gtk_clist_set_sort_column (GTK_CLIST (gcsh->priv->clist), gcsh->priv->sort_col);
			gtk_clist_set_sort_type (GTK_CLIST (gcsh->priv->clist), GTK_SORT_ASCENDING);
			gtk_clist_sort (GTK_CLIST (gcsh->priv->clist));
			gtk_sort_heading_set_others_to_off (gcsh);
			gtk_signal_emit (GTK_OBJECT (gcsh), gtk_sort_heading_signals[MODE_CHANGED]);
		} else if (mode == GTK_SORT_HEADING_DESCENDING) {
			gcsh->priv->arrow = gtk_arrow_new (GTK_ARROW_UP, GTK_SHADOW_IN);
			gtk_clist_set_sort_column (GTK_CLIST (gcsh->priv->clist), gcsh->priv->sort_col);
			gtk_clist_set_sort_type (GTK_CLIST (gcsh->priv->clist), GTK_SORT_DESCENDING);
			gtk_clist_sort (GTK_CLIST (gcsh->priv->clist));
			gtk_sort_heading_set_others_to_off (gcsh);
			gtk_signal_emit (GTK_OBJECT (gcsh), gtk_sort_heading_signals[MODE_CHANGED]);
		}
		if (gcsh->priv->arrow){
			gtk_container_add (GTK_CONTAINER (gcsh->priv->frame), gcsh->priv->arrow);
			gtk_widget_show(gcsh->priv->arrow);
		}
		gtk_clist_set_auto_sort (GTK_CLIST (gcsh->priv->clist), TRUE);
		gtk_clist_thaw (GTK_CLIST (gcsh->priv->clist));
	}
}

static void
gtk_sort_heading_class_init				(GtkSortHeadingClass	*Klass)
{
	GtkObjectClass *object_class;
	/*GtkWidgetClass *widget_class;*/
	
	object_class = (GtkObjectClass *) Klass;
	gtk_sort_heading_parent_class = gtk_type_class (gtk_hbox_get_type ());
	
	object_class->destroy = gtk_sort_heading_destroy;
	gtk_sort_heading_signals[MODE_CHANGED] = gtk_signal_new (
		"mode_changed",
		GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET (GtkSortHeadingClass, gtk_sort_heading),
		gtk_signal_default_marshaller,
		GTK_TYPE_NONE, 0);

	gtk_object_class_add_signals (object_class, gtk_sort_heading_signals, LAST_SIGNAL);
	Klass->gtk_sort_heading = NULL;
}

static void
gtk_sort_heading_init					(GtkSortHeading		*sh)
{
	sh->priv = g_new0 (GtkSortHeadingPrivate, 1);
	
	/*
	 * Create the widget
	 */
	sh->priv->hbox = gtk_hbox_new (FALSE, 0);
	sh->priv->eventbox = gtk_event_box_new ();
	gtk_container_add (GTK_CONTAINER (sh->priv->eventbox), sh->priv->hbox);
	
	sh->priv->label = gtk_label_new ("");
	gtk_box_pack_start (GTK_BOX (sh->priv->hbox), sh->priv->label, TRUE, TRUE, 0);
	
	sh->priv->frame = gtk_frame_new (NULL);
	gtk_widget_set_usize (sh->priv->frame, 15, 15);
  	gtk_frame_set_shadow_type (GTK_FRAME (sh->priv->frame), GTK_SHADOW_NONE);
  	gtk_box_pack_start (GTK_BOX (sh->priv->hbox), sh->priv->frame, FALSE, FALSE, 0);
  	
	sh->priv->arrow = NULL;
	gtk_box_pack_end (GTK_BOX (sh), sh->priv->eventbox, TRUE, TRUE, 0);

	gtk_widget_show_all (sh->priv->eventbox);
	sh->priv->mode = (GtkSortHeadingMode *) g_malloc (sizeof (GtkSortHeadingMode));
	gtk_signal_connect (GTK_OBJECT (sh->priv->eventbox), "event", GTK_SIGNAL_FUNC (gtk_sort_heading_eventbox_event), sh);

	
}

void
gtk_sort_heading_set_sort_column			(GtkSortHeading		*sh,
							gint			column)
{
	g_return_if_fail (sh != NULL);
	g_return_if_fail (GTK_IS_SORT_HEADING (sh));
	
	sh->priv->sort_col = column;
}
