/*  Copyright Daryl Gray 2000 email daryl.g@visto.com
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <string.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include "fax-viewer.h"

struct _LibGefaxFaxViewerPrivate {
	GtkWidget	*scrolled_window;
	GtkWidget	*viewport;
	GtkWidget	*image;
	GtkWidget	*alignment;
	guint		expose_signal;
	GdkPixbuf	*pixbuf;
	gboolean	loading : 1;
	gint		pixbuf_width;
	gint		pixbuf_height;
	GdkInterpType	interp_type;
	gdouble		scale;
	GList		*PageList;
	gint		total_pages;
	gint		current_page;
};

enum {
	RIGHT_CLICK,
	SCALE_CHANGED,
	PAGE_CHANGED,
	LAST_SIGNAL
};

static void
fax_viewer_refresh_current_page			(LibGefaxFaxViewer		*fv);

static void
fax_viewer_load_current_page			(LibGefaxFaxViewer		*fv);

void
fax_viewer_free_filenames_list			(GList				*FileList);

static gboolean
fax_viewer_expose_event				(GtkWidget			*widget,
						GdkEventExpose			*event,
						LibGefaxFaxViewer		*fv);

static gint
fax_viewer_event				(GtkWidget			*w,
						GdkEvent			*event,
						GtkObject			*fv);

/*==================================Widget Functions=========================================*/
static GtkHBox
*lib_gefax_fax_viewer_parent_class;

static gint
lib_gefax_fax_viewer_signals[LAST_SIGNAL] = { 0 };

static void
lib_gefax_fax_viewer_class_init			(LibGefaxFaxViewerClass		*Klass);

static void
lib_gefax_fax_viewer_init			(LibGefaxFaxViewer		*FaxViewer);

static void
lib_gefax_fax_viewer_destroy			(GtkObject			*FaxViewer);


static void
fax_viewer_refresh_current_page			(LibGefaxFaxViewer		*fv)
{
	gint size_x, size_y;
	GdkPixbuf *dest;
	GtkWidget *img = fv->priv->image;
	
	fv->priv->loading = TRUE;
	size_x = fv->priv->pixbuf_width * fv->priv->scale;
	size_y = fv->priv->pixbuf_height * fv->priv->scale;
	gtk_signal_handler_block(GTK_OBJECT (fv->priv->image), fv->priv->expose_signal);
	
	dest = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, size_x, size_y);
	gdk_pixbuf_scale (fv->priv->pixbuf, dest, 0, 0, size_x, size_y, 0, 0,
			(double) size_x / fv->priv->pixbuf_width,
			(double) size_y / fv->priv->pixbuf_height, GDK_INTERP_NEAREST);
	gtk_drawing_area_size (GTK_DRAWING_AREA (fv->priv->image), size_x, size_y);
	gtk_widget_set_usize (fv->priv->alignment, size_x, size_y);
	while (gtk_events_pending ()) {
		gtk_main_iteration ();
	}
	gdk_pixbuf_render_to_drawable (dest, img->window, img->style->fg_gc[GTK_STATE_NORMAL],
			0, 0, 0, 0, size_x, size_y,
			GDK_RGB_DITHER_NORMAL, 0, 0);
	gdk_pixbuf_unref (dest);
	if (fv->priv->interp_type != GDK_INTERP_NEAREST) {
		gtk_drawing_area_size (GTK_DRAWING_AREA (fv->priv->image), size_x, size_y);
	}

	gtk_signal_handler_unblock(GTK_OBJECT (fv->priv->image), fv->priv->expose_signal);
	fv->priv->loading = FALSE;
}

static void
fax_viewer_load_current_page			(LibGefaxFaxViewer		*fv)
{
	fv->priv->loading = TRUE;
	if (fv->priv->pixbuf) {
		gdk_pixbuf_unref (fv->priv->pixbuf);
	}
	fv->priv->pixbuf = gdk_pixbuf_new_from_file (fv->priv->PageList->data);
	g_return_if_fail (fv->priv->pixbuf != NULL);
	fv->priv->pixbuf_width = gdk_pixbuf_get_width (fv->priv->pixbuf);
	fv->priv->pixbuf_height = gdk_pixbuf_get_height (fv->priv->pixbuf);
	fax_viewer_refresh_current_page (fv);
	fv->priv->loading = FALSE;
	gtk_signal_emit (GTK_OBJECT (fv), lib_gefax_fax_viewer_signals[PAGE_CHANGED], fv->priv->current_page);
}

void
fax_viewer_free_filenames_list			(GList			*FileList)
{
	gint list_len, i;
	GList *List;
	
	g_return_if_fail (FileList != NULL);
	
	List = g_list_first(FileList);
	list_len = g_list_length (List);
	for (i = 0; i < list_len; i++) {
		gchar *data = List->data;
		if (data) {
			g_free (data);
		}
		List = g_list_next (List);
	}
	g_list_free (FileList);
}

static gboolean
fax_viewer_expose_event					(GtkWidget			*widget,
							GdkEventExpose			*event,
							LibGefaxFaxViewer		*fv)
{
	GdkPixbuf *dest;
	if (!fv->priv->pixbuf) {
		return FALSE;
	} else {
		GdkInterpType i_type = fv->priv->interp_type;

		fv->priv->loading = TRUE;
		/*Not sure if we are gaining anything here*/
		if (fv->priv->scale == 1.0) {
			i_type = GDK_INTERP_NEAREST;
		}
		gdk_window_set_back_pixmap (widget->window, NULL, FALSE);
 
		dest = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, event->area.width, event->area.height);
		gdk_pixbuf_scale (fv->priv->pixbuf, dest, 0, 0, event->area.width, event->area.height, -event->area.x, -event->area.y,
			(double) widget->allocation.width / fv->priv->pixbuf_width,
			(double) widget->allocation.height / fv->priv->pixbuf_height, i_type);
		gdk_pixbuf_render_to_drawable (dest, widget->window, widget->style->fg_gc[GTK_STATE_NORMAL],
			0, 0, event->area.x, event->area.y, event->area.width, event->area.height,
			GDK_RGB_DITHER_NORMAL, event->area.x, event->area.y);
		gdk_pixbuf_unref (dest);
		fv->priv->loading = FALSE;
		return TRUE;
	}
}

static gint
fax_viewer_event				(GtkWidget			*w,
						GdkEvent			*event,
						GtkObject			*fv)
{
	if (event->type == GDK_BUTTON_PRESS) {
		if (event->button.button == 3) {
			GdkEventButton *bevent = (GdkEventButton *) event;
			gtk_signal_emit (fv, lib_gefax_fax_viewer_signals[RIGHT_CLICK],
					bevent->button, bevent->time);
			return TRUE;
		}
	}
	return FALSE;
}

/*==================================Widget Functions=========================================*/
static void
lib_gefax_fax_viewer_init			(LibGefaxFaxViewer		*FaxViewer)
{
	LibGefaxFaxViewerPrivate *P;
	GtkHBox *box;
	
	FaxViewer->priv = g_new0 (LibGefaxFaxViewerPrivate, 1);
	P = FaxViewer->priv;
	
	P->pixbuf = NULL;
	P->PageList = NULL;
	P->total_pages = 0;
	P->current_page = 0;
	P->pixbuf_width = 0;
	P->pixbuf_height = 0;

	/*Create the widgets*/
	P->scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	box = GTK_HBOX (FaxViewer);
	gtk_box_pack_start (GTK_BOX (box), P->scrolled_window, TRUE, TRUE, 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (P->scrolled_window),
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	/*to keep small view centered*/
	P->alignment = gtk_alignment_new (0.5, 0.5, 0, 0);
	P->viewport = gtk_viewport_new (NULL, NULL);
	gtk_container_add (GTK_CONTAINER (P->scrolled_window), P->viewport);
	gtk_container_add (GTK_CONTAINER (P->viewport), P->alignment);
	gtk_viewport_set_shadow_type (GTK_VIEWPORT (P->viewport), GTK_SHADOW_IN);
	/*the image*/
	gtk_widget_push_visual (gdk_rgb_get_visual ());
	gtk_widget_push_colormap (gdk_rgb_get_cmap ());
	P->image = gtk_drawing_area_new ();
	gtk_widget_pop_visual ();
	gtk_widget_pop_colormap ();

	gtk_container_add (GTK_CONTAINER (P->alignment), P->image);

        gtk_widget_show_all (P->scrolled_window);
        gtk_widget_ref (P->image);
        gtk_widget_add_events (P->image, GDK_EXPOSURE_MASK);
	P->expose_signal = gtk_signal_connect (GTK_OBJECT (P->image), "expose_event",
			GTK_SIGNAL_FUNC (fax_viewer_expose_event), FaxViewer);
}

static void
lib_gefax_fax_viewer_destroy			(GtkObject			*FaxViewer)

{
	LibGefaxFaxViewer *FView;
	LibGefaxFaxViewerPrivate *P;
	
	g_return_if_fail (FaxViewer != NULL);
	g_return_if_fail (LIBGEFAX_IS_FAX_VIEWER (FaxViewer));
	
	FView = LIBGEFAX_FAX_VIEWER (FaxViewer);
	P = FView->priv;
	
	if (P->pixbuf) {
		gdk_pixbuf_unref (P->pixbuf);
	}
	if (P->PageList) {
		fax_viewer_free_filenames_list (P->PageList);
	}
	gtk_widget_unref (P->image);
	g_free (P);
	
	if (GTK_OBJECT_CLASS (lib_gefax_fax_viewer_parent_class)->destroy) {
		(*GTK_OBJECT_CLASS (lib_gefax_fax_viewer_parent_class)->destroy) (GTK_OBJECT (FaxViewer));
	}
	return;	
}

static void
lib_gefax_fax_viewer_class_init			(LibGefaxFaxViewerClass		*Klass)
{
	GtkObjectClass *object_class;
	
	object_class = (GtkObjectClass *) Klass;
	lib_gefax_fax_viewer_parent_class = gtk_type_class (gtk_hbox_get_type ());
	
	object_class->destroy = lib_gefax_fax_viewer_destroy;
	lib_gefax_fax_viewer_signals[RIGHT_CLICK] = gtk_signal_new (
		"right_click", GTK_RUN_FIRST, object_class->type,
		GTK_SIGNAL_OFFSET (LibGefaxFaxViewerClass, lib_gefax_fax_viewer),
		gtk_marshal_NONE__INT_INT, GTK_TYPE_NONE, 2, GTK_TYPE_INT, GTK_TYPE_INT);
	lib_gefax_fax_viewer_signals[SCALE_CHANGED] = gtk_signal_new (
		"scale_changed", GTK_RUN_FIRST, object_class->type,
		GTK_SIGNAL_OFFSET (LibGefaxFaxViewerClass, lib_gefax_fax_viewer),
		gtk_marshal_NONE__BOOL, GTK_TYPE_NONE, 1, GTK_TYPE_BOOL);
	lib_gefax_fax_viewer_signals[PAGE_CHANGED] = gtk_signal_new (
		"page_changed", GTK_RUN_FIRST, object_class->type,
		GTK_SIGNAL_OFFSET (LibGefaxFaxViewerClass, lib_gefax_fax_viewer),
		gtk_marshal_NONE__INT, GTK_TYPE_NONE, 1, GTK_TYPE_INT);

	gtk_object_class_add_signals (object_class, lib_gefax_fax_viewer_signals, LAST_SIGNAL);
	Klass->lib_gefax_fax_viewer = NULL;
}

/*==================================Global Functions=========================================*/
GtkType
lib_gefax_fax_viewer_get_type 	  		(void)
{
	static GtkType type = 0;

	if (!type) {
		GtkTypeInfo info = {
			"LibGefaxFaxViewer",
			sizeof (LibGefaxFaxViewer),
			sizeof (LibGefaxFaxViewerClass),
			(GtkClassInitFunc) lib_gefax_fax_viewer_class_init,
			(GtkObjectInitFunc) lib_gefax_fax_viewer_init,
			NULL, /* reserved 1 */
			NULL, /* reserved 2 */
			(GtkClassInitFunc) NULL
		};

		type = gtk_type_unique (gtk_hbox_get_type (), &info);
	}

	return type;
}

GtkWidget
*lib_gefax_fax_viewer_new			(void)
{
	GtkWidget *FaxViewer;
	LibGefaxFaxViewer *FaxView;
	LibGefaxFaxViewerPrivate *Priv;
	
	FaxViewer = gtk_type_new (lib_gefax_fax_viewer_get_type ());
	
	FaxView = LIBGEFAX_FAX_VIEWER (FaxViewer);
	
	Priv = FaxView->priv;
	
	Priv->interp_type = GDK_INTERP_TILES;
	Priv->scale = 0.35;
	gtk_signal_connect (GTK_OBJECT (Priv->scrolled_window), "event",
 			GTK_SIGNAL_FUNC (fax_viewer_event), FaxViewer);
 	Priv->loading = FALSE;
	return FaxViewer;
}

GtkWidget
*lib_gefax_fax_viewer_construct				(const gchar		*file_or_directory,
							gdouble			scale,
							GdkInterpType		interp_type)
{
	GtkWidget *FaxViewer;
	LibGefaxFaxViewer *FaxView;
	LibGefaxFaxViewerPrivate *Priv;
	
	g_return_val_if_fail (file_or_directory != NULL, NULL);
	g_return_val_if_fail ((scale < LIBGEFAX_FAX_VIEWER_SCALE_MAX) || (scale > LIBGEFAX_FAX_VIEWER_SCALE_MIN), NULL);

	FaxViewer = gtk_type_new (lib_gefax_fax_viewer_get_type ());
	
	FaxView = LIBGEFAX_FAX_VIEWER (FaxViewer);
	Priv = FaxView->priv;

	Priv->interp_type = interp_type;
	Priv->scale = scale;
	
	if (g_file_test (file_or_directory, G_FILE_TEST_ISFILE)) {
		lib_gefax_fax_viewer_load_file (FaxView, file_or_directory);
	} else if (g_file_test (file_or_directory, G_FILE_TEST_ISDIR)) {
		lib_gefax_fax_viewer_load_directory (FaxView, file_or_directory);
	} else {
		g_warning (_("file_or_directory is not a file or directory"));
	}
	gtk_signal_connect (GTK_OBJECT (Priv->scrolled_window), "event",
 			GTK_SIGNAL_FUNC (fax_viewer_event), FaxViewer);
 	Priv->loading = FALSE;
	return FaxViewer;
}

void
lib_gefax_fax_viewer_set_interp_type			(LibGefaxFaxViewer	*fv,
							GdkInterpType		interp_type)
{
	g_return_if_fail (LIBGEFAX_IS_FAX_VIEWER (fv));
	if (fv->priv->interp_type == interp_type) {
		return;
	} else {
		fv->priv->interp_type = interp_type;
		if (fv->priv->pixbuf) {
			fax_viewer_refresh_current_page (fv);
		}
	}
}

gboolean
lib_gefax_fax_viewer_load_directory			(LibGefaxFaxViewer	*fv,
							const gchar		*directory)
{
	
	gint fcount;
	gint files;
	gchar *fname;
	gchar *ftype;
	struct dirent **filelist;
	
	g_return_val_if_fail (LIBGEFAX_IS_FAX_VIEWER (fv), FALSE);
	if (fv->priv->loading) {
		return FALSE;
	}
	if (fv->priv->PageList) {
		fax_viewer_free_filenames_list (fv->priv->PageList);
		fv->priv->PageList = NULL;
	}
	if (fv->priv->pixbuf) {
		gdk_pixbuf_unref (fv->priv->pixbuf);
		fv->priv->pixbuf = NULL;
	}
	fv->priv->total_pages = 0;
	
	files = scandir (directory, &filelist, 0, alphasort);
	if (files > 2) {
		for (fcount = 1; fcount < files; ++fcount) {
			fname = g_strdup_printf ("%s%s", directory, filelist[fcount]->d_name);
			if (g_file_test (fname, G_FILE_TEST_ISFILE)) {
				ftype = g_strdup (gnome_mime_type_or_default_of_file (fname, "UNKNOWN"));
				if (strcmp (ftype, "image/tiff") == 0) {
					fv->priv->PageList = g_list_append (fv->priv->PageList, fname);
					++fv->priv->total_pages;
				} else {
					g_free (fname);
				}
				g_free (ftype);
			}
		}	
	}
	if (fv->priv->PageList) {
		fv->priv->current_page = 1;
		fax_viewer_load_current_page (fv);
		return TRUE;
	} else {
		/*Clear Image*/
		return FALSE;
	}
}

void
lib_gefax_fax_viewer_load_page_list			(LibGefaxFaxViewer	*fv,
							GList			*PageList)
{
	g_return_if_fail (LIBGEFAX_IS_FAX_VIEWER (fv));
	g_return_if_fail (PageList != NULL);

	if (fv->priv->loading) {
		return;
	}
	fv->priv->loading = TRUE;
	if (fv->priv->PageList) {
		fax_viewer_free_filenames_list (fv->priv->PageList);
	}
	fv->priv->PageList = PageList;
	if (fv->priv->pixbuf) {
		gdk_pixbuf_unref (fv->priv->pixbuf);
		fv->priv->pixbuf = NULL;
	}
	fv->priv->total_pages = g_list_length (PageList);
	fv->priv->current_page = 1;
	fax_viewer_load_current_page (fv);
	fv->priv->loading = FALSE;
	return;	
}

gboolean
lib_gefax_fax_viewer_load_file				(LibGefaxFaxViewer	*fv,
							const gchar		*file_name)
{
	g_return_val_if_fail (g_file_test (file_name, G_FILE_TEST_ISFILE), FALSE);
	g_return_val_if_fail (LIBGEFAX_IS_FAX_VIEWER (fv), FALSE);
	g_return_val_if_fail (strcmp (gnome_mime_type_or_default_of_file (file_name, "UNKNOWN"), "image/tiff") == 0, FALSE);
	
	if (fv->priv->loading) {
		return FALSE;
	}
	if (fv->priv->PageList) {
		fax_viewer_free_filenames_list (fv->priv->PageList);
		fv->priv->PageList = NULL;
	}
	if (fv->priv->pixbuf) {
		gdk_pixbuf_unref (fv->priv->pixbuf);
		fv->priv->pixbuf = NULL;
	}
	fv->priv->PageList = g_list_append (fv->priv->PageList, g_strdup (file_name));
	fv->priv->total_pages = 1;
	fv->priv->current_page = 1;
	fax_viewer_load_current_page (fv);
	return TRUE;
}

gboolean
lib_gefax_fax_viewer_show_png_file			(LibGefaxFaxViewer	*fv,
							const gchar		*file_name)
{
	g_return_val_if_fail (file_name != NULL, FALSE);
	g_return_val_if_fail (g_file_test (file_name, G_FILE_TEST_ISFILE), FALSE);
	g_return_val_if_fail (LIBGEFAX_IS_FAX_VIEWER (fv), FALSE);
	g_return_val_if_fail (strcmp (gnome_mime_type_or_default_of_file (file_name, "UNKNOWN"), "image/png") != 0, FALSE);
	
	if (fv->priv->loading) {
		return FALSE;
	}
	fv->priv->loading = TRUE;
	if (fv->priv->pixbuf) {
		gdk_pixbuf_unref (fv->priv->pixbuf);
		fv->priv->pixbuf = NULL;
	}
	fv->priv->pixbuf = gdk_pixbuf_new_from_file (file_name);
	g_return_val_if_fail (fv->priv->pixbuf != NULL, FALSE);
	fv->priv->pixbuf_width = gdk_pixbuf_get_width (fv->priv->pixbuf);
	fv->priv->pixbuf_height = gdk_pixbuf_get_height (fv->priv->pixbuf);
	fv->priv->scale = 1.0;
	fax_viewer_refresh_current_page (fv);
	fv->priv->loading = FALSE;
	return TRUE;
}

void
lib_gefax_fax_viewer_set_scale				(LibGefaxFaxViewer	*fv,
							gdouble			scale)
{
	g_return_if_fail ((scale < LIBGEFAX_FAX_VIEWER_SCALE_MAX) && (scale > LIBGEFAX_FAX_VIEWER_SCALE_MIN));
	
	if ((fv->priv->loading) && (fv->priv->PageList)) {
		return;
	}
	fv->priv->scale = scale;

	/*Redraw Image*/
	if (fv->priv->PageList) {
		fax_viewer_refresh_current_page (fv);
		gtk_signal_emit (GTK_OBJECT (fv), lib_gefax_fax_viewer_signals[SCALE_CHANGED], fv->priv->scale);
	}
}

gdouble
lib_gefax_fax_viewer_get_scale				(LibGefaxFaxViewer	*fv)
{
	g_return_val_if_fail (LIBGEFAX_IS_FAX_VIEWER (fv), -1.0);

	return (gdouble) fv->priv->scale;
}

gint
lib_gefax_fax_viewer_get_pages				(LibGefaxFaxViewer	*fv)
{
	g_return_val_if_fail (LIBGEFAX_IS_FAX_VIEWER (fv), -1);

	return fv->priv->total_pages;
}

void
lib_gefax_fax_viewer_show_first_page			(GtkWidget		*unused,
							LibGefaxFaxViewer	*fv)
{
	GList *First;

	g_return_if_fail (fv->priv->PageList != NULL);
	g_return_if_fail (LIBGEFAX_IS_FAX_VIEWER (fv));
	
	if (fv->priv->loading) {
		return;
	}
	if (fv->priv->current_page == 1) {
		return;
	}
	
	First = g_list_first (fv->priv->PageList);
	
	if (First != fv->priv->PageList) {
		fv->priv->PageList = First;
		fv->priv->current_page = 1;
		fax_viewer_load_current_page (fv);
	}
}

void
lib_gefax_fax_viewer_show_last_page			(GtkWidget		*unused,
							LibGefaxFaxViewer	*fv)
{
	GList *Last;

	g_return_if_fail (fv->priv->PageList != NULL);
	g_return_if_fail (LIBGEFAX_IS_FAX_VIEWER (fv));

	if (fv->priv->loading) {
		return;
	}
	if (fv->priv->current_page == fv->priv->total_pages) {
		return;
	}
	
	Last = g_list_last (fv->priv->PageList);
	
	if (Last != fv->priv->PageList) {
		fv->priv->PageList = Last;
		fv->priv->current_page = fv->priv->total_pages;
		fax_viewer_load_current_page (fv);
	}
}

void
lib_gefax_fax_viewer_show_next_page			(GtkWidget		*unused,
							LibGefaxFaxViewer	*fv)
{
	GList *Next;

	g_return_if_fail (fv->priv->PageList != NULL);
	g_return_if_fail (LIBGEFAX_IS_FAX_VIEWER (fv));

	if (fv->priv->loading) {
		return;
	}
	if (fv->priv->current_page == fv->priv->total_pages) {
		return;
	}
	
	Next = g_list_next (fv->priv->PageList);
	
	if (Next) {
		fv->priv->PageList = Next;
		++fv->priv->current_page;
		fax_viewer_load_current_page (fv);
	}
}

void
lib_gefax_fax_viewer_show_previous_page			(GtkWidget		*unused,
							LibGefaxFaxViewer	*fv)
{
	GList *Previous;

	g_return_if_fail (fv->priv->PageList != NULL);
	g_return_if_fail (LIBGEFAX_IS_FAX_VIEWER (fv));
	
	if (fv->priv->loading) {
		return;
	}
	if (fv->priv->current_page == 1) {
		return;
	}
	
	Previous = g_list_previous (fv->priv->PageList);
	
	if (Previous) {
		fv->priv->PageList = Previous;
		--fv->priv->current_page;
		fax_viewer_load_current_page (fv);
	}
}

