/*  Copyright Daryl Gray 2001 email daryl.g@visto.com
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "textfile_wrap.h"
#include <stdio.h>


#define END_OF_FILE	-2
#define END_OF_LINE	-1


struct _LibGefaxInternalData{
	FILE *in;
	FILE *out;
	gint c;
	gchar word[LIBGEFAX_WRAPTEXTFILE_MAX_WORDLENGTH + 1];
	gint max_wordlength;
	gint word_len;
	gint line_len;
	gint tab_len;
};

static gboolean print_word				(LibGefaxWrapTextResult *r);
static gboolean print_newline				(LibGefaxWrapTextResult *r);
static gboolean print_spaces_to_file			(LibGefaxWrapTextResult *r, gint space_len);
static gint get_next_word				(LibGefaxInternalData *p);
static LibGefaxWrapTextResult *clean_up_for_return	(LibGefaxWrapTextResult *r);

gboolean print_spaces_to_file				(LibGefaxWrapTextResult *r, gint space_len)
{
	gint i = 0;

	while ( i != space_len){
		if (putc( ' ', r->priv->out) == EOF){
			r->error = LIBGEFAX_WRAPTEXTFILE_ERROR_OUTFILE;
			return TRUE;
		}
		++i;
	}
	return FALSE;
}

gboolean print_word						(LibGefaxWrapTextResult *r)
{
	gint i = 0;

	while ( r->priv->word[i] != '\0'){
		if (putc( r->priv->word[i], r->priv->out) == EOF){
			r->error = LIBGEFAX_WRAPTEXTFILE_ERROR_OUTFILE;
			return TRUE;
		}
		++i;
	}
	++r->words;
	return FALSE;
}

gboolean print_newline						(LibGefaxWrapTextResult *r)
{
	if (putc( '\n', r->priv->out) == EOF){
		r->error = LIBGEFAX_WRAPTEXTFILE_ERROR_OUTFILE;
		return TRUE;
	}
	++r->file_linecount;
	return FALSE;
}

gint get_next_word						(LibGefaxInternalData *p)
{
	int len = 0;
	gint spaces = 0;
	
	p->word_len = 0;
	
	if (p->c == EOF)
		return END_OF_FILE;
	if (p->c == '\n'){
		p->c = getc(p->in);
		return END_OF_LINE;
	}
	
	while ((p->c == ' ') || (p->c == '\t')){
		if (p->c == ' ')
			++spaces;
		if (p->c == '\t')
			spaces += p->tab_len;
		p->c = getc(p->in);
	}
	while ((p->c != ' ') && (p->c != '\n') && (p->c != EOF) && (len < p->max_wordlength)){
		p->word[len] = p->c;
		p->c = getc(p->in);
		++len;
	}
	p->word[len] = '\0';
	p->word_len = len;
	return spaces;
}

LibGefaxWrapTextResult *clean_up_for_return		(LibGefaxWrapTextResult *r)
{
	if (r->priv->in) fclose(r->priv->in);
	if (r->priv->out) fclose(r->priv->out);
	g_free(r->priv);
	if ((r->error != LIBGEFAX_WRAPTEXTFILE_ERROR_NONE) && (r->return_file != NULL)){
		if (g_file_test(r->return_file, G_FILE_TEST_ISFILE)){
			remove(r->return_file);
		}
		g_free (r->return_file);
		r->return_file = NULL;
	}
	return r;
}

LibGefaxWrapTextResult *libGefax_wrap_text_file		(const gchar *infile, gint tab_expand, gint line_length)
{
	LibGefaxWrapTextResult *result;
	gint spaces;
	
	result = (LibGefaxWrapTextResult *) g_malloc0 ( sizeof (LibGefaxWrapTextResult));
	result->priv = (LibGefaxInternalData *) g_malloc ( sizeof (LibGefaxInternalData));
	result->priv->line_len = 0;
	result->priv->max_wordlength = LIBGEFAX_WRAPTEXTFILE_MAX_WORDLENGTH;
	
	/*Set return values to 0 and check paramaters*/
	result->file_linecount = 0;
	result->lines_wrapped = 0;
	result->words = 0;
	
	if (!g_file_test(infile, G_FILE_TEST_ISFILE)){
		result->error = LIBGEFAX_WRAPTEXTFILE_ERROR_NOFILE;
		return (clean_up_for_return(result));
	}
	if (line_length < LIBGEFAX_WRAPTEXTFILE_MIN_LINELENGTH){
		result->error = LIBGEFAX_WRAPTEXTFILE_ERROR_LINELENGTH;
		return (clean_up_for_return(result));
	}
	if (strncmp((gnome_mime_type_of_file(infile)), "text/", 5) != 0){
		result->error = LIBGEFAX_WRAPTEXTFILE_ERROR_MIMETYPE;
		return (clean_up_for_return(result));
	}
	if ((result->priv->in = fopen(infile, "r")) == NULL){
		result->error = LIBGEFAX_WRAPTEXTFILE_ERROR_NOFILEACCESS;
		return (clean_up_for_return(result));
	}
	result->return_file = g_strdup("/tmp/Gefax.XXXXXX");
	result->return_file = mktemp(result->return_file);
	if ((result->priv->out = fopen(result->return_file, "w")) == NULL){
		result->error = LIBGEFAX_WRAPTEXTFILE_ERROR_OUTFILE;
		return (clean_up_for_return(result));
	}
	
	if (tab_expand > LIBGEFAX_WRAPTEXTFILE_MAX_TABEXPAND){
		tab_expand = LIBGEFAX_WRAPTEXTFILE_MAX_TABEXPAND;
	}
	result->priv->tab_len = tab_expand;
	/*Safely go ahead now*/
	result->error = LIBGEFAX_WRAPTEXTFILE_ERROR_NONE;
	
	if (line_length < LIBGEFAX_WRAPTEXTFILE_MAX_WORDLENGTH)
		result->priv->max_wordlength = line_length - 1;
	spaces = 0;
	result->priv->c = getc(result->priv->in);
	while (1){
		spaces = get_next_word(result->priv);
		if (spaces == END_OF_FILE)
			break;
		if (spaces == END_OF_LINE) {
			if (print_newline(result))
				break;
			result->priv->line_len = 0;
			result->priv->word_len = 0;
			continue;
		}
		result->priv->line_len = (spaces + result->priv->word_len + result->priv->line_len);
		if (result->priv->line_len > line_length){
			if (print_newline(result))
				break;
			++result->lines_wrapped;
			if (result->priv->word_len > 0){
				if (print_word(result))
					break;
				result->priv->line_len = result->priv->word_len;
				result->priv->word_len = 0;
			}else {
				result->priv->line_len = 0;
			}
			
		}else{
			if (spaces > 0) print_spaces_to_file(result, spaces);
			if (print_word(result))
				break;
			result->priv->word_len = 0;
		}
		
	}
	return (clean_up_for_return(result));
}

