/*  Copyright Daryl Gray 2000 email daryl.g@visto.com
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#ifndef BACKWARDS_COMPATIBLE
#	define BACKWARDS_COMPATIBLE
#endif

#ifdef BACKWARDS_COMPATIBLE
#	include <dirent.h>
#	include <sys/stat.h>
#endif BACKWARDS_COMPATIBLE

#include <stdio.h>  
#include <glib.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>

#include "history-data.h"

/*
 *Reading
*/

static GSList
*add_xml_fax_pages_to_data			(xmlNodePtr		xmlnode,
						GSList			*PList);

static GList
*add_xml_fax_to_data				(xmlNodePtr		xmlnode,
						GList			*FList,
						gint			key);


/*
 *Saving
*/

static void
write_fax_pages_to_xml				(xmlNodePtr		xmlnode,
						GSList			*Pl);

static void
write_fax_to_xml				(xmlNodePtr		xmlnode,
						Fax_Data		*Fd);

/*
 * GCompareFunc used to sort Fax_Page GSList by index
*/

static gint
history_data_cmp_keys				(gconstpointer		Fp1,
						gconstpointer		Fp2);

/*
 * Free a Fax_Data structure
*/

static void
free_Fax_Data					(Fax_Data		*Fd);

/*
 *Reading
*/

static GSList
*add_xml_fax_pages_to_data			(xmlNodePtr		xmlnode,
						GSList			*PList)
{
	gchar *tmp;
	Fax_Page *Fp = g_malloc0 (sizeof (Fax_Page));
	
	Fp->page_file_name = xmlGetProp (xmlnode, "page_path");
	Fp->page_thumb_file_name = xmlGetProp (xmlnode, "page_thumb_path");
	tmp = xmlGetProp (xmlnode, "page_number");
	Fp->index = atoi (tmp);
	g_free (tmp);
	
	return (g_slist_append (PList, Fp));
}

static GList
*add_xml_fax_to_data				(xmlNodePtr		xmlnode,
						GList			*FList,
						gint			key)
{
	gchar *tmp;
	Fax_Data *Fd = g_malloc0 (sizeof (Fax_Data));
	
	Fd->key = key;
	Fd->directory = xmlGetProp (xmlnode, "path");
	Fd->polled = xmlGetProp (xmlnode, "polled");
	tmp = xmlGetProp (xmlnode, "pages");
	Fd->pages = atoi (tmp);
	g_free (tmp);
	tmp = xmlGetProp (xmlnode, "save_time");
	Fd->save_time = atol (tmp);
	g_free (tmp);
	tmp = xmlGetProp (xmlnode, "print_time");
	Fd->print_time = atol (tmp);
	g_free (tmp);
	
	xmlnode = xmlnode->childs;
	while (xmlnode) {
		/* add the fax page to the list*/
		Fd->Pages = add_xml_fax_pages_to_data (xmlnode, Fd->Pages);
		xmlnode = xmlnode->next;
	}
	
	return (g_list_append (FList, Fd));
}

GList
*lib_gefax_history_data_read_xml_file			(const gchar		*directory_path)
{
	GList *FaxList = NULL;
	gchar *file_path;
	gint key = 1;
	xmlDocPtr doc;
	xmlNodePtr node;

	file_path = g_strdup_printf ("%s%s", directory_path, LIB_GEFAX_HISTORY_DATA_FILE);
	
	/*check to see if a file exists*/
	if (!g_file_test (file_path, G_FILE_TEST_ISFILE)) {
		return NULL;
	}
	
	doc = xmlParseFile (file_path);
	g_free (file_path);

	/* if there is no root element or,
	if it doesn't have a name or,
	if it isn't a GefaxHistory node get out!*/
	if (!doc->root || !doc->root->name || g_strcasecmp (doc->root->name, "GefaxHistory") != 0) {
		xmlFreeDoc (doc);
		return NULL;
	}
	/* find <Fax> nodes and add them to the list, this just loops
	through all the children of the root of the document */
	for (node = doc->root->childs; node != NULL; node = node->next) {

		/* add the fax to the list*/
		FaxList = add_xml_fax_to_data (node, FaxList, key++);
	}
	xmlFreeDoc (doc);

	return (FaxList);
}

/*
 *Saving
*/

static void
write_fax_pages_to_xml				(xmlNodePtr		xmlnode,
						GSList			*Pl)
{
	while (Pl) {
		xmlNodePtr newxml;
		gchar *tmp;
		Fax_Page *Fp = Pl->data;
		
		/* make a new xml node (as a child of xmlnode) with an empty content */
		newxml = xmlNewChild (xmlnode, NULL, "Page", NULL);

		/* set properties on it */
		xmlSetProp (newxml, "page_path", Fp->page_file_name);
		if (Fp->page_thumb_file_name) {
			xmlSetProp (newxml, "page_thumb_path", Fp->page_thumb_file_name);
		}
		tmp = g_strdup_printf ("%d", Fp->index);
		xmlSetProp (newxml,"page_number", tmp);
		g_free (tmp);

		Pl = g_slist_next (Pl);
	}
}

static void
write_fax_to_xml				(xmlNodePtr		xmlnode,
						Fax_Data		*Fd)
{
	xmlNodePtr newxml;
	gchar *tmp;

	/* make a new xml node (as a child of xmlnode) with an empty content */
	newxml = xmlNewChild (xmlnode, NULL, "Fax", NULL);

	/* set properties on it */
	xmlSetProp (newxml, "path", Fd->directory);
	xmlSetProp (newxml, "polled", Fd->polled);
	tmp = g_strdup_printf ("%d", Fd->pages);
	xmlSetProp (newxml,"pages", tmp);
	g_free (tmp);
	tmp = g_strdup_printf ("%ld", Fd->save_time);
	xmlSetProp (newxml,"save_time", tmp);
	g_free (tmp);
	tmp = g_strdup_printf ("%ld", Fd->print_time);
	xmlSetProp (newxml,"print_time", tmp);
	g_free (tmp);

	/*Now add pages*/
	write_fax_pages_to_xml (newxml, Fd->Pages);
	
}

gboolean
lib_gefax_history_data_save_xml_file		(const gchar		*directory_path,
						GList			*FaxList)
{
	xmlDocPtr doc;
	gchar *file_path;
	gboolean result = FALSE;
	
	g_return_val_if_fail (FaxList != NULL, result);
	g_return_val_if_fail (directory_path != NULL, result);
	
	/* create new xml document with version 1.0 */
	doc = xmlNewDoc("1.0");

	/* create a new root node "GefaxHistory" */
	doc->root = xmlNewDocNode (doc, NULL, "GefaxHistory", NULL);
	
	while (FaxList) {
		Fax_Data *Fd = FaxList->data;
		
		write_fax_to_xml (doc->root, Fd);
		FaxList = g_list_next (FaxList);
	}

	/*Now save the doc to disk*/
	file_path = g_strdup_printf ("%s%s", directory_path, LIB_GEFAX_HISTORY_DATA_FILE);
	if (xmlSaveFile (file_path, doc) == -1) {
		result = FALSE;
	} else {
		result = TRUE;
	}
	g_free (file_path);
	xmlFreeDoc (doc);
	return result;
}

/*
 *Free data structure
*/
static void
free_Fax_Data					(Fax_Data		*Fd)
{
	GSList *Pl = Fd->Pages;

	g_return_if_fail (Fd != NULL);

	g_free (Fd->directory);
	g_free (Fd->polled);
		
	/*Free pages*/
	while (Pl) {
		Fax_Page *Fp = Pl->data;
			
		g_free (Fp->page_file_name);
		if (Fp->page_thumb_file_name) {
			g_free (Fp->page_thumb_file_name);
		}
		g_free (Fp);
		Pl = g_slist_next (Pl);
	}
	g_free (Fd);
}

void
lib_gefax_history_data_free_data		(GList			*FaxList)
{
	GList *Fl = FaxList;
	
	while (Fl) {
		Fax_Data *Fd = Fl->data;

		free_Fax_Data (Fd);
		Fl = g_list_next (Fl);
	}
	g_list_free (FaxList);
		
}

Fax_Data
*lib_gefax_history_data_get_fax_data_from_key	(GList			*FaxList,
						gint			key)
{
	GList *Fl = FaxList;
	
	g_return_val_if_fail (Fl != NULL, NULL);
	g_return_val_if_fail (key > 0, NULL);
	
	while (Fl) {
		Fax_Data *Fd = Fl->data;
		
		if (Fd->key == key) {
			return Fd;
		}
		Fl = g_list_next (Fl);
	}
	return NULL;
}


GList
*lib_gefax_history_data_remove_fax_data_from_key(GList			*FaxList,
						gint			key)
{
	GList *Fl = FaxList;
	Fax_Data *Fd;
	
	Fd = lib_gefax_history_data_get_fax_data_from_key (FaxList, key);
	if (Fd) {
		Fl = g_list_remove (FaxList, Fd);
		free_Fax_Data (Fd);
		return Fl;
	} else {
		return FaxList;
	}
}

GList
*lib_gefax_history_data_append_fax_data		(GList			*FaxList,
						Fax_Data		*FaxData)
{
	GList *Fl = FaxList;
	gint highest_key = 0;
	
	g_return_val_if_fail (FaxData != NULL, NULL);
	
	while (Fl) {
		Fax_Data *Fd = Fl->data;
		
		if (Fd->key > highest_key) {
			highest_key = Fd->key;
		}
		Fl = g_list_next (Fl);
	}
	++highest_key;
	FaxData->key = highest_key;
	
	return g_list_append (FaxList, FaxData);
}

static gint
history_data_cmp_keys				(gconstpointer		Fp1,
						gconstpointer		Fp2)
{
	gint index1, index2;

	index1 = ((Fax_Page *) Fp1)->index;
	index2 = ((Fax_Page *) Fp2)->index;
	
	if (index1 == index2) {
		return 0;
	} else if (index1 < index2) {
		return -1;
	} else {
		return 1;
	}
}

GList
*lib_gefax_history_data_get_file_list_from_key	(GList			*FaxList,
						gint			key)
{
	GList *PageList = NULL;
	GSList *Pl = NULL;
	GSList *Nl = NULL;
	Fax_Data *Fd;

	g_return_val_if_fail (FaxList != NULL, NULL);
	
	Fd = lib_gefax_history_data_get_fax_data_from_key (FaxList, key);
	
	if (!Fd) {
		return NULL;
	}
	
	Pl = g_slist_copy (Fd->Pages);
	Nl = g_slist_sort (Pl, history_data_cmp_keys);
	while (Nl) {
		gchar *path;
		Fax_Page *Fp = Nl->data;

		path = g_strdup_printf ("%s%s", Fd->directory, Fp->page_file_name);
		PageList = g_list_append (PageList, path);
		Nl = g_slist_next (Nl);
	}
	g_slist_free (Pl);
	return PageList;
}

void
lib_gefax_history_data_free_file_list		(GList			*FileList)
{
	GList *Fl = FileList;
	
	g_return_if_fail (Fl != NULL);
	
	while (Fl) {
		gchar *file = Fl->data;
		
		g_free (file);
		Fl = g_list_next (Fl);
	}
	g_list_free (Fl);
}

gchar
*lib_gefax_history_data_get_fax_page_thumb	(Fax_Data		*FaxData,
						gint			page)
{
	GSList *Pl;

	g_return_val_if_fail (FaxData != NULL, NULL);
	
	Pl = FaxData->Pages;
	while (Pl) {
		Fax_Page *Fp = Pl->data;
		
		if (Fp->index == page) {			
			if (Fp->page_thumb_file_name) {
				return g_strdup_printf ("%s%s", FaxData->directory, Fp->page_thumb_file_name);
			} else {
				return NULL;
			}
		}
		Pl = g_slist_next (Pl);
	}
	return NULL;
}

/*========================BACKWARS COMPATABILTY To Be REMOVED=======================*/
#ifdef BACKWARDS_COMPATIBLE

static void
history_data_generate_thumbs			(GList			*FileList)
{
	GList *Fl = FileList;
	
	while (Fl) {
		Fax_Data *Fd = Fl->data;
		GSList *Pl = Fd->Pages;
		while (Pl) {
			gchar *command;
			Fax_Page *Fp = Pl->data;

			command = g_strdup_printf ("thumbnail %s%s %s%s", Fd->directory,
						Fp->page_file_name, Fd->directory, Fp->page_thumb_file_name);
			if (gnome_execute_shell (Fd->directory, command) == -1) {
				g_warning ("Unable to generate thumbnails");
			}
			Pl = g_slist_next (Pl);
		}
		Fl = g_list_next (Fl);
	}
}

static GSList
*get_Fax_Pages					(const gchar		*dname)
{
	GSList *PList = NULL;
	
	gint fcount;
	gint files;
	gchar *fname;
	gchar *ftype;
	struct dirent **filelist;
	gint index = 1;
	
	files = scandir (dname, &filelist, 0, alphasort);
	if (files > 2) {
		for (fcount = 1; fcount < files; ++fcount) {
			fname = g_strdup_printf ("%s/%s", dname, filelist[fcount]->d_name);
			if (g_file_test (fname, G_FILE_TEST_ISFILE)) {
				ftype = g_strdup (gnome_mime_type_or_default_of_file (fname, "UNKNOWN"));
				if (strcmp (ftype, "image/tiff") == 0) {
					gchar *tmp;
					Fax_Page *Fp = g_malloc (sizeof (Fax_Page));

					Fp->index = index;
					Fp->page_file_name = g_strdup_printf ("Page%d.tiff", index);
					Fp->page_thumb_file_name = g_strdup_printf ("%s.thumb", Fp->page_file_name);
					tmp = g_strdup_printf ("%s/%s", dname, Fp->page_file_name);
					if (rename (fname, tmp) == -1){
						g_warning("Error moving file");
					}
					g_free (tmp);
					PList = g_slist_append (PList, Fp);
					++index;
				} else {
					g_free (fname);
				}
				g_free (ftype);
			}
		}	
	}
	return PList;
}

static Fax_Data
*get_Fax_Data					(const gchar		*dname,
						const gchar		*fname)
{
	FILE *file;
	gchar buff[100];
	Fax_Data *Fd = NULL;

	if ((file = fopen (fname,"r")) == NULL) {
		return NULL;
	}
	fgets (buff, sizeof (buff), file);
	if (strncmp (buff, "gefax-0.4 ", 9) != 0) {
		return NULL;
	}
	Fd = g_malloc0 (sizeof (Fax_Data));
	Fd->directory = g_strdup_printf ("%s/", dname);
	fgets (buff, sizeof (buff), file);
	Fd->save_time = (time_t) atol (buff);
	fgets (buff, sizeof (buff), file);
	Fd->pages = atoi (buff);
	fgets (buff, sizeof (buff), file);
	Fd->print_time = (time_t) atol (buff);
	fgets (buff, sizeof (buff), file);
	Fd->polled = g_strdup (buff);
	Fd->Pages = get_Fax_Pages (dname);
	fclose (file);
	remove (fname);
	return Fd;
}

GList
*lib_gefax_history_data_read_old_history	(const gchar		*directory_path)
{
	GList *FileList = NULL;
	gint no_files;
	struct dirent **filelist;
	
	if (g_file_test (directory_path, G_FILE_TEST_ISDIR)) {
		no_files = scandir (directory_path, &filelist, 0, alphasort );
		if (no_files > 2) {
			gint fcount;
			
			for (fcount = 0; fcount < no_files; ++fcount) {
				gchar *dname;

				dname = g_strdup_printf ("%s%s", directory_path, filelist[fcount]->d_name);
				if ((g_file_test (dname, G_FILE_TEST_ISDIR)) && (filelist[fcount]->d_name[0] != '.')) {
					gchar *fname;

					fname = g_strdup_printf("%s/gefax.metadata", dname);
					if (g_file_test (fname, G_FILE_TEST_ISFILE)) {
						Fax_Data *Fd;
						
						Fd = get_Fax_Data (dname, fname);
						if (Fd) {
							FileList = lib_gefax_history_data_append_fax_data (FileList, Fd);
						}
					}
					g_free (fname);
				}
				g_free (dname);
			}
		}
	}
	history_data_generate_thumbs (FileList);
	return FileList;
}

#endif BACKWARDS_COMPATIBLE

