/*  Copyright Daryl Gray 2001 email daryl.g@visto.com
 * 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __VCARD_TYPES_H__
#define __VCARD_TYPES_H__

#include "../address-book.h"

#define SEP ';'
#define ENDTAGS ':'

/*
 * PrimaryType is the first "tag" of a vcard file entry.
 * E.g in the record [TEL;HOME:04 687792] TEL is the PrimaryTag as it identifies the
 * record as some sort of Telephone number
*/
typedef enum {
	BEGIN,
	END,
	FN,
	N,
	ADR,
	TEL,
	EMAIL,
	BDAY,
	REV,
	ORG,
	TITLE,
	URL,
	CATEGORIES,
	NOTE,
	KEY,
	TZ,
	GEO,
	LOGO,
	UNKNOWN
} PrimaryType;

/*
 * SubType/s are the "tag/s" after the first "tag" of a vcard file entry.
 * E.g in the record [TEL;HOME;FAX:04 687792] HOME and FAX are the SubTags.
 * They describe the Telephone number attributes
*/

enum SubType {
	DOM			= 1,
	POSTAL			= 2,
	PARCEL			= 3,
	INTL			= 4,
	HOME			= 5,
	WORK			= 6,
	PREF			= 7,
	VOICE			= 8,
	FAX			= 9,
	MSG			= 10,
	CELL			= 11,
	PAGER			= 12,
	BBS			= 13,
	MODEM			= 14,
	CAR			= 15,
	ISDN			= 16,
	VIDEO			= 17,
	AOL			= 18,
	APPLELINK		= 19,
	ATTMAIL			= 20,
	CIS			= 21,
	EWORLD			= 22,
	INTERNET		= 23,
	IBMMail			= 24,
	MCIMAIL			= 25,
	POWERSHARE		= 26,
	PRODIGY			= 27,
	TLX			= 28,
	X400			= 29,
	PGP			= 30,
	X509			= 31,
	GIF			= 32,
	BMP			= 33,
	PMB			= 34,
	TIFF			= 35,
	PS			= 36,
	PDF			= 37,
	JPEG			= 38,
	SUBUNKNOWN		= 39,
	VCARD			= 40
};

/*
 * These need to corrispond to SubType.
 * This is in order to check that a SubType or "attribute" is valid for
 * the "record's" PrimaryType
 * YES some overlap because some SubTypes are valid for two PrimaryType's
*/

#define ADDR_LOWER		1
#define ADDR_UPPER		6
#define TEL_LOWER		5
#define TEL_UPPER		17
#define EMAIL_LOWER		18
#define EMAIL_UPPER		29
#define KEY_LOWER		30
#define KEY_UPPER		31
#define LOGO_LOWER		32
#define LOGO_UPPER		37

/*
 * The EncodeType is used to determine how to read and write the data.
 * This information is also stored in the event of an unknown record so we 
 * know how to write the data back to the file. A record is unknown if:
 *	1). An unknown PrimaryType is encountered
 *	2). An unknown SubType is encountered
 *	3). One or more SubType/s are not valid for a PrimaryType
 * It is not nescessary to keep this information for all records because
 * we know thier encoding
*/

typedef enum {
	ENC_QUOTED_PRINTABLE,
	ENC_BASE64,
	ENC_NONE
} EncodeType;

typedef struct _PTag {
	PrimaryType	type;
	char		*tag;		/*only if type UNKNOWN*/
} PTag;

typedef struct _STag {
	gint 		type;
	char		*tag;		/*only if type SUBUNKNOWN*/
} STag;

/*
 * A type capable of temporarily storing any "record" as the file is read
 * In all cases Data->data is an allocated gchar
 * NOTE: in the case of [ADR:;;228 Edward Street;Brisbane;Q.L.D;4000;Australia]
 * the Data list will be 8 long with the first 2 having NULL DATA->data as the fields
 * are empty. This is needed because the fields are in a specific order. Also, the
 * S_Tags will be NULL as there are no SubTypes in this record.
 *
 * In the case of QUOTED_PRINTABLE & BASE64 Data, Data will only be one record
 * with '\n' used to separate lines
*/

typedef struct _FieldData {
	int		c;		/*last read char*/
	EncodeType	encode;
	PTag 		*P_Tag;
	GSList 		*S_Tags;
	GSList 		*Data;
} FieldData;

#endif /*__VCARD_TYPES_H_*/
