/*  Copyright Daryl Gray 2001 email daryl.g@visto.com
 * 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <dirent.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>

#include "vcard.h"
#include "vcard_util.h"
#include "vcard_save.h"

#define BEGIN_VCARD	"BEGIN:VCARD"
#define END_VCARD	"END:VCARD"
#define NEW_LINE	"\r\n"

static gboolean
lib_gefax_save_Card					(FILE			*f,
							LibGefaxContact		*Card);

static gboolean
lib_gefax_save_Identity					(LibGefaxIdentity	*I,
							FILE			*f);
static gboolean
lib_gefax_save_Business					(LibGefaxBusiness	*B,
							FILE			*f);

static gboolean
lib_gefax_save_TimeZone					(LibGefaxTimeZone	*T,
							FILE			*f);

static gboolean
lib_gefax_save_GeoLocation				(LibGefaxGeo		*G,
							FILE			*f);

static gboolean
lib_gefax_save_PublicKey				(LibGefaxPublicKey	*K,
							FILE			*f);

static gboolean
lib_gefax_save_Logo					(LibGefaxLogo		*L,
							FILE			*f);

static gboolean
lib_gefax_save_Revision					(LibGefaxDate		*D,
							FILE			*f);

static gboolean
lib_gefax_save_BirthDate				(LibGefaxDate		*D,
							FILE			*f);

static gboolean
lib_gefax_save_Anniversary				(LibGefaxDate		*D,
							FILE			*f);

static gboolean
lib_gefax_save_Address					(GSList			*L,
							FILE			*f);

static gboolean
lib_gefax_save_Tel					(GSList			*L,
							FILE			*f);

static gboolean
lib_gefax_save_Email					(GSList			*L,
							FILE			*f);

static gboolean
lib_gefax_save_Categories				(GSList			*L,
							FILE			*f);

static gboolean
lib_gefax_save_Unknown					(GSList			*L,
							FILE			*f);

static gboolean
lib_gefax_save_Note					(gchar			*n,
							FILE			*f);

static gboolean
lib_gefax_write_Quoted_Printable			(FILE			*f,
							gchar			*string);
static gboolean
lib_gefax_write_Base64					(FILE			*f,
							gchar			*string);
static gboolean
lib_gefax_write_Quoted_Printable			(FILE			*f,
							gchar			*string)
{
	gint i = 0;
	gint length = 0;
	
	while (string[i] != '\0') {
		if (length >= 74) {
			if (fprintf (f, "=%s", NEW_LINE) == -1) {
				return TRUE;
			}
			length = 0;
		}
		if (string[i] == '\n') {
			if (fprintf (f, "=0A") == -1) {
				return TRUE;
			}
			length += 3;
			if (length >= 74) {
				if (fprintf (f, "=%s", NEW_LINE) == -1) {
					return TRUE;
				}
			length = 0;
			}
		} else {
			if (fputc (string[i], f) == EOF) {
				return TRUE;
			}
			++length;
		}
		++i;
	}
	if (fprintf (f, "%s", NEW_LINE) == -1) {
		return TRUE;
	}
	return FALSE;
		
}
static gboolean
lib_gefax_write_Base64					(FILE			*f,
							gchar			*string)
{
	gint i;
	
	if (fprintf (f, "    ") == -1) {
		return TRUE;
	}
	for (i = 0; i != strlen (string); i++) {
		if (string[i] == '\n') {
			if (fprintf (f, "%s    ", NEW_LINE) == -1) {
				return TRUE;
			}
		} else {
			if (fputc (string[i], f) == EOF) {
				return TRUE;
			}
		}
	}
	if (fprintf (f, "%s%s", NEW_LINE, NEW_LINE) == -1) {
		return TRUE;
	}
	return FALSE;
}

static gboolean
lib_gefax_save_Identity					(LibGefaxIdentity	*I,
							FILE			*f)
{
	if ((I->family) || (I->prefix) || (I->suffix) || (I->additional) || (I->given)) {
		if (fprintf (f, "N:") == -1) {
			return TRUE;
		}
	}
	if (I->family) {
		if (fprintf (f, "%s",I->family) == -1) {
			return TRUE;
		}
	}
	if ((I->prefix) || (I->suffix) || (I->additional) || (I->given)) {
		if (fprintf (f, ";") == -1) {
			return TRUE;
		}
	}
	if (I->given) {
		if (fprintf (f, "%s",I->given) == -1) {
			return TRUE;
		}
	}
	if ((I->prefix) || (I->suffix) || (I->additional)) {
		if (fprintf (f, ";") == -1) {
			return TRUE;
		}
	}
	if (I->additional) {
		if (fprintf (f, "%s",I->additional) == -1) {
			return TRUE;
		}
	}
	if ((I->prefix) || (I->suffix)) {
		if (fprintf (f, ";") == -1) {
			return TRUE;
		}
	}
	if (I->prefix) {
		if (fprintf (f, "%s",I->prefix) == -1) {
			return TRUE;
		}
	}
	if (I->suffix) {
		if (fprintf (f, ";%s",I->suffix) == -1) {
			return TRUE;
		}
	}
	if ((I->family) || (I->prefix) || (I->suffix) || (I->additional) || (I->given)) {
		if (fprintf (f, "%s", NEW_LINE) == -1) {
			return TRUE;
		}
	}	
	return FALSE;
}
static gboolean
lib_gefax_save_Business					(LibGefaxBusiness	*B,
							FILE			*f)
{
	if (B->org_title) {
		if (fprintf (f, "TITLE:%s%s", B->org_title, NEW_LINE) == -1) {
			return TRUE;
		}
	}
	if (B->org_name) {
		if (fprintf (f, "ORG:%s%s", B->org_name, NEW_LINE) == -1) {
			return TRUE;
		}
	}	
	return FALSE;
}

static gboolean
lib_gefax_save_TimeZone					(LibGefaxTimeZone	*T,
							FILE			*f)
{
	if ((T->hours > 0) || (T->minutes > 0)) {
		/*TZ:Hours:Mins*/
		if (fprintf (f, "TZ:%02d:%02d%s", T->hours , T->minutes, NEW_LINE) == -1) {
			return TRUE;
		}
	}
	return FALSE;
}

static gboolean
lib_gefax_save_GeoLocation				(LibGefaxGeo		*G,
							FILE			*f)
{
	if ((G->longitude > 0) || (G->latitude > 0)) {
		if (fprintf (f, "GEO:%03.02f,%03.02f%s", G->longitude , G->latitude, NEW_LINE) == -1) {
			return TRUE;
		}
	}
	return FALSE;
}

static gboolean
lib_gefax_save_PublicKey				(LibGefaxPublicKey	*K,
							FILE			*f)
{
	if (!K->key) {
		return FALSE;
	}
	if (K->type == LIBGEFAX_PGP) {
		if (fprintf (f, "KEY;PGP:") == -1) {
			return TRUE;
		}
	} else {
		if (fprintf (f, "KEY;X509:") == -1) {
			return TRUE;
		}
	}
	if (fprintf (f, "%s%s", K->key, NEW_LINE) == -1) {
		return TRUE;
	}
	return FALSE;
}

static gboolean
lib_gefax_save_Logo					(LibGefaxLogo		*L,
							FILE			*f)
{
	if (!L->data) {
		return FALSE;
	}	
	if (fprintf (f, "LOGO") == -1) {
		return TRUE;
	}
	if (L->type == LIBGEFAX_LOGO_GIF) {
		if (fprintf (f, ";%s", "GIF") == -1) {
			return TRUE;
		}
	} else if (L->type == LIBGEFAX_LOGO_BMP) {
		if (fprintf (f, ";%s", "BMP") == -1) {
			return TRUE;
		}
	} else if (L->type == LIBGEFAX_LOGO_PMB) {
		if (fprintf (f, ";%s", "PMB") == -1) {
			return TRUE;
		}
	} else if (L->type == LIBGEFAX_LOGO_TIFF) {
		if (fprintf (f, ";%s", "TIFF") == -1) {
			return TRUE;
		}
	} else if (L->type == LIBGEFAX_LOGO_PS) {
		if (fprintf (f, ";%s", "PS") == -1) {
			return TRUE;
		}
	} else if (L->type == LIBGEFAX_LOGO_PDF) {
		if (fprintf (f, ";%s", "PDF") == -1) {
			return TRUE;
		}
	} else if (L->type == LIBGEFAX_LOGO_JPEG) {
		if (fprintf (f, ";%s", "JPEG") == -1) {
			return TRUE;
		}
	} else {
		g_warning ("LOGO type is Unknown - falling back to JPEG");
		if (fprintf (f, ";%s", "JPEG") == -1) {
			return TRUE;
		}
	}
	if (fprintf (f, ";BASE64:") == -1) {
		return TRUE;
	}
	if (lib_gefax_write_Base64 (f, L->data)) {
		return TRUE;
	}
	return FALSE;
}

static gboolean
lib_gefax_save_Revision					(LibGefaxDate		*D,
							FILE			*f)
{
	if ((D->years > 0) || (D->months > 0) ||
		(D->days > 0) || (D->hours > 0) ||
		(D->minutes > 0) || (D->seconds > 0)) {
		/*REV:2001-05-08T04:57:02*/
		if (fprintf (f, "REV:%04d-%02d-%02dT%02d:%02d:%02d%s",
			D->years, D->months, D->days,
			D->hours, D->minutes, D->seconds, NEW_LINE) == -1) {
			return TRUE;
		}
	}
	return FALSE;
}

static gboolean
lib_gefax_save_BirthDate				(LibGefaxDate		*D,
							FILE			*f)
{
	if ((D->years > 0) || (D->months > 0) || (D->days > 0)) {
		/*BDAY:1972-11-15*/
		if (fprintf (f, "BDAY:%04d-%02d-%02d%s",
			D->years, D->months, D->days, NEW_LINE) == -1) {
			return TRUE;
		}
	}
	return FALSE;
}

static gboolean
lib_gefax_save_Anniversary				(LibGefaxDate		*D,
							FILE			*f)
{
	gboolean error = FALSE;
	
	return error;
}

static gboolean
lib_gefax_save_Note					(gchar			*n,
							FILE			*f)
{
	if (fprintf (f, "NOTE;QUOTED-PRINTABLE:") == -1) {
		return TRUE;
	}	
	return (lib_gefax_write_Quoted_Printable (f, n));
}

static gboolean
lib_gefax_save_Address					(GSList			*L,
							FILE			*f)
{
	const gint ADDRESS_TYPES_LENGTH = 6;
	const gchar *ADDRESS_TYPES[] = {
		";HOME", ";WORK", ";POSTAL", ";PARCEL", ";DOM", ";INTL"
	};
	gint list_len, i, ii;
	

	list_len = g_slist_length (L);
	for (i = 0; i < list_len; i++) {
		LibGefaxAddress *A = L->data;
		if (fprintf (f, "ADR") == -1) {
			return TRUE;
		}
		/*Print tags*/
		for (ii = 0; ii < ADDRESS_TYPES_LENGTH; ii++) {
			if (A->type & 1 << ii) {
				if (fprintf (f, "%s", ADDRESS_TYPES[ii]) == -1) {
					return TRUE;
				}
			}
		}

		if (fprintf (f, ":") == -1) {
			return TRUE;
		}
		/*Print address*/
		if (A->post_box) {
			if (fprintf (f, "%s",A->post_box) == -1) {
				return TRUE;
			}
		}
		if ((A->extended) || (A->street) || (A->city) || (A->region) || (A->post_code) || (A->country)) {
			if (fprintf (f, ";") == -1) {
				return TRUE;
			}
		}
		if (A->extended) {
			if (fprintf (f, "%s",A->extended) == -1) {
				return TRUE;
			}
		}
		if ((A->street) || (A->city) || (A->region) || (A->post_code) || (A->country)) {
			if (fprintf (f, ";") == -1) {
				return TRUE;
			}
		}
		if (A->street) {
			if (fprintf (f, "%s",A->street) == -1) {
				return TRUE;
			}
		}
		if ((A->city) || (A->region) || (A->post_code) || (A->country)) {
			if (fprintf (f, ";") == -1) {
				return TRUE;
			}
		}
		if (A->city) {
			if (fprintf (f, "%s",A->city) == -1) {
				return TRUE;
			}
		}
		if ((A->region) || (A->post_code) || (A->country)) {
			if (fprintf (f, ";") == -1) {
				return TRUE;
			}
		}
		if (A->region) {
			if (fprintf (f, "%s",A->region) == -1) {
				return TRUE;
			}
		}
		if ((A->post_code) || (A->country)) {
			if (fprintf (f, ";") == -1) {
				return TRUE;
			}
		}
		if (A->post_code) {
			if (fprintf (f, "%s",A->post_code) == -1) {
				return TRUE;
			}
		}
		if (A->country) {
			if (fprintf (f, ";%s",A->country) == -1) {
				return TRUE;
			}
		}
		if (fprintf (f, "%s", NEW_LINE) == -1) {
			return TRUE;
		}
		L = g_slist_next (L);
	}
	return FALSE;
}


static gboolean
lib_gefax_save_Tel					(GSList			*L,
							FILE			*f)
{
	const gchar *TEL_TYPES[] = {
		";PREF", ";WORK", ";HOME", ";VOICE", ";FAX", ";MSG", ";CELL",
		";PAGER", ";BBS", ";MODEM", ";CAR", ";ISDN", ";VIDEO"
	};
	const gint TEL_TYPES_LENGTH = 13;
	gint list_len, i, ii;
	
	list_len = g_slist_length (L);
	for (i = 0; i < list_len; i++) {
		LibGefaxTelephone *T = L->data;
		if (fprintf (f, "TEL") == -1) {
			return TRUE;
		}

		/*Print tags*/									/*Complete*/
		for (ii = 0; ii < TEL_TYPES_LENGTH; ii++) {
			if (T->type & 1 << ii) {
				if (fprintf (f, "%s", TEL_TYPES[ii]) == -1) {
					return TRUE;
				}
			}
		}

		/*Print Number*/
		if (fprintf (f, ":%s%s", T->number, NEW_LINE) == -1) {
			return TRUE;
		}
		L = g_slist_next (L);
	}
	return FALSE;
	
}


static gboolean
lib_gefax_save_Email					(GSList			*L,
							FILE			*f)
{
	const gchar *EMAIL_TYPES[] = {
		";AOL", ";APPLELINK", ";ATTMAIl", ";CIS", ";EWORLD", ";INTERNET",
		";IBMMail", ";MCIMAIL", ";POWERSHARE", ";PRODIGY", ";TLX", ";X400"
	};
	gint list_len, i, ii;
	
	list_len = g_slist_length (L);
	for (i = 0; i < list_len; i++) {
		LibGefaxEmail *E = L->data;
		if (fprintf (f, "EMAIL") == -1) {
			return TRUE;
		}
		/*Print tags*/
		for (ii = 0; ii < LIBGEFAX_EMAIL_NONE; ii++) {								/*Complete*/
			if (E->type == ii) {
				if (fprintf (f, "%s", EMAIL_TYPES[ii]) == -1) {
					return TRUE;
				}
			}
		}

		/*Print Email*/
		if (fprintf (f, ":%s%s", E->number, NEW_LINE) == -1) {
			return TRUE;
		}
		L = g_slist_next (L);
	}
	return FALSE;
	
}


static gboolean
lib_gefax_save_Categories				(GSList			*L,
							FILE			*f)
{
	gint list_len, i;
	gchar *string = NULL;
	gchar *str_tmp;
	
	if (fprintf (f, "CATEGORIES;QUOTED-PRINTABLE:") == -1) {
		return TRUE;
	}
	list_len = g_slist_length (L);
	for (i = 0; i < list_len; i++) {
		gchar *cat = L->data;
		if (string) {
			str_tmp = g_strdup (string);
			g_free (string);
			string = g_strdup_printf ("%s,%s", str_tmp, cat);
			g_free (str_tmp);
		} else {
			string = g_strdup (cat);
		}
		L = g_slist_next (L);
	}
	if (string) {
		if (lib_gefax_write_Quoted_Printable (f, string)) {
			return TRUE;
		}
	}
	return FALSE;
}


static gboolean
lib_gefax_save_Unknown					(GSList			*L,
							FILE			*f)
{
	gint list_len, i;
	
	list_len = g_slist_length (L);
	for (i = 0; i < list_len; i++) {
		LibGefaxUnknown *Uk = L->data;
		gint len, l_len;
		GSList *Tags = Uk->Tags;
		GSList *Data = Uk->Data;
		l_len = g_slist_length (Tags);
		for (len = 0; len < l_len; len++) {
			gchar *tag = Tags->data;
			if (len > 0) {
				if (fprintf (f, ";") == -1) {
					return TRUE;
				}
			}
			if (fprintf (f, "%s", tag) == -1) {
				return TRUE;
			}
			Tags = g_slist_next (Tags);
		}
		if (Uk->data_type == LIBGEFAX_ENC_QUOTED_PRINTABLE) {
			if (fprintf (f, ";QUOTED-PRINTABLE:") == -1) {
				return TRUE;
			}
			if (lib_gefax_write_Quoted_Printable (f, Data->data)) {
				return TRUE;
			}
		} else if (Uk->data_type == LIBGEFAX_ENC_BASE64) {
			if (fprintf (f, ";BASE64:") == -1) {
				return TRUE;
			}
			if (lib_gefax_write_Base64 (f, Data->data)) {
				return TRUE;
			}
		} else {
			if (fprintf (f, ":") == -1) {
				return TRUE;
			}
			l_len = g_slist_length (Data);
			for (len = 0; len < l_len; len++) {
				gchar *data = Data->data;
				if (len > 0) {
					if (fprintf (f, ";") == -1) {
						return TRUE;
					}
				}
				if (fprintf (f, "%s", data) == -1) {
					return TRUE;
				}
				Data = g_slist_next (Data);
			}
			if (fprintf (f, "%s", NEW_LINE) == -1) {
				return TRUE;
			}	
		}
		L = g_slist_next (L);
	}
	return FALSE;
}

static gboolean
lib_gefax_save_Card					(FILE			*f,
							LibGefaxContact		*Card)
{
	gboolean error = FALSE;
	LibGefaxIdentity 	*I = Card->Identity;
	LibGefaxBusiness 	*B = Card->Business;
	LibGefaxTimeZone	*T = Card->TimeZone;
	LibGefaxGeo		*G = Card->GeoLocation;
	LibGefaxPublicKey	*K = Card->PublicKey;
	LibGefaxLogo		*L = Card->Logo;
	LibGefaxDate		*Rd = Card->Revision;
	LibGefaxDate		*Bd = Card->BirthDate;
	LibGefaxDate		*Ad = Card->Anniversary;
	GSList 			*Al = Card->Address;
	GSList 			*Tl = Card->Tel;
	GSList 			*El = Card->Email;
	GSList 			*Cl = Card->Categories;
	GSList			*Ul = Card->Unknown;
	
	if (fprintf (f, "%s%s", BEGIN_VCARD, NEW_LINE) == -1) {
		return TRUE;
	}
	if (Card->filename) {
		if (fprintf (f, "FN:%s%s", Card->filename, NEW_LINE) == -1) {
			return TRUE;
		}
	}
	if (Card->url) {
		if (fprintf (f, "URL:%s%s", Card->url, NEW_LINE) == -1) {
			return TRUE;
		}
	}
	if (Card->note) {
		error = lib_gefax_save_Note (Card->note, f);
		if (error) {
			return error;
		}
	}
	if (I) {
		error = lib_gefax_save_Identity (I, f);
		if (error) {
			return error;
		}
	}
	if (B) {
		error = lib_gefax_save_Business (B, f);
		if (error) {
			return error;
		}
	}
	if (T) {
		error = lib_gefax_save_TimeZone (T, f);
		if (error) {
			return error;
		}
	}
	if (G) {
		error = lib_gefax_save_GeoLocation (G, f);
		if (error) {
			return error;
		}
	}
	if (K) {
		error = lib_gefax_save_PublicKey (K, f);
		if (error) {
			return error;
		}
	}
	if (L) {
		error = lib_gefax_save_Logo (L, f);
		if (error) {
			return error;
		}
	}
	if (Rd) {
		error = lib_gefax_save_Revision (Rd, f);
		if (error) {
			return error;
		}
	}
	if (Bd) {
		error = lib_gefax_save_BirthDate (Bd, f);
		if (error) {
			return error;
		}
	}
	if (Ad) {
		error = lib_gefax_save_Anniversary (Ad, f);
		if (error) {
			return error;
		}
	}
	if (Al) {
		error = lib_gefax_save_Address (Al, f);
		if (error) {
			return error;
		}
	}
	if (Tl) {
		error = lib_gefax_save_Tel (Tl, f);
		if (error) {
			return error;
		}
	}
	if (El) {
		error = lib_gefax_save_Email (El, f);
		if (error) {
			return error;
		}
	}
	if (Cl) {
		error = lib_gefax_save_Categories (Cl, f);
		if (error) {
			return error;
		}
	}
	if (Ul) {
		error = lib_gefax_save_Unknown (Ul, f);
		if (error) {
			return error;
		}
	}
	if (fprintf (f, "%s%s%s", END_VCARD, NEW_LINE, NEW_LINE) == -1) {
		return TRUE;
	}
		
	return error;
}

gboolean
lib_gefax_save_Vcard_file				(GList			*AddressBook,
							const gchar		*absolute_file_path)
{
	GList *ab = NULL;
	LibGefaxContact *Card;
	FILE *f;
	gint ab_length, i;
	gboolean error = FALSE;
	
	if ((f = fopen (absolute_file_path, "w")) == NULL) {
		error = TRUE;
		return error;
	} else {
		ab = AddressBook;
		ab_length = g_list_length (ab);
		for (i = 0; i < ab_length; i++) {
			Card = LIB_GEFAX_CONTACT (ab);
			error = lib_gefax_save_Card (f, Card);
			if (error) {
				break;
			}
			ab = g_list_next (ab);
		}
	}
	fclose (f);
	if (error) {
		remove (absolute_file_path);
	}
	return error;
		
}
