/*  Copyright Daryl Gray 2001 email daryl.g@visto.com
 * 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <dirent.h>
#include <ctype.h>
#include <stdio.h>
#include "vcard_types.h"
#include "vcard.h"
#include "add_field.h"

typedef struct _AnyTag {
	gchar	tag[20];
	int	lastchar;
	gint	length;
} AnyTag;

static FieldData
*get_next_field_data					(FILE 			*f);

static AnyTag
*get_next_any_tag					(FILE			*f);

static gboolean
get_primary_tag						(FILE 			*f,
							FieldData 		*fd);

static gboolean
get_secondary_tags					(FILE 			*f,
							FieldData 		*fd);
static gchar
*update_data						(gchar			*data,
							gchar			*buff);

static gboolean
get_field_data_list					(FILE 			*f,
							FieldData 		*fd);

static int
look_ahead						(FILE 			*f);

static int
skip_to_newline						(FILE 			*f,
							int			current);

static gboolean
verify_sub_tag						(GSList			*tags,
							gint			tag);

static GList
*abort_read_Vcard_file					(GList			*ab);

static AnyTag
*get_next_any_tag					(FILE			*f);

static int
look_ahead						(FILE 			*f)
{
	int c;
	c = getc (f);
	ungetc (c,f);
	return c;
}

static int
skip_to_newline						(FILE 			*f,
							int			current)
{
	while (current != '\n')
		current = fgetc (f);
	return current;
}

static gboolean
verify_sub_tag						(GSList			*tags,
							gint			tag)
{
	GSList *node;
	gint i;
	gint gslist_len = g_slist_length (tags);
	
	node = tags;
	for (i = 0; i < gslist_len; i++) {
		STag *stag = node->data;
		if (stag->type == tag)
			return TRUE;
		node = g_slist_next (node);
	}
	return FALSE;
}

static FieldData
*free_field_data					(FieldData		*fd)
{
	GSList *node;
	gint i;
	gint gslist_len = g_slist_length (fd->S_Tags);

	if (fd->P_Tag) {
		if (fd->P_Tag->tag)
			g_free (fd->P_Tag->tag);
		g_free (fd->P_Tag);
	}
	if (fd->S_Tags) {
		node = fd->S_Tags;
		for (i = 0; i < gslist_len; i++) {
			if (((STag *) (node->data))->tag)
				g_free (((STag *) (node->data))->tag);
			g_free (node->data);
			node = g_slist_next (node);
		}
		g_slist_free (fd->S_Tags);
	}
	g_free (fd);
	return NULL;
}

static AnyTag
*get_next_any_tag					(FILE			*f)
{
	AnyTag *at;
	
	at = g_malloc0 (sizeof (AnyTag));
	at->length = 0;
	while (1) {
		at->lastchar = fgetc (f);
		if (at->length == 20) {		/*too long ERROR*/
			at->lastchar = EOF;
			break;
		}
		if ((at->lastchar != ENDTAGS)
			&& (at->lastchar != SEP)
			&& (at->lastchar != '\r')
			&& (at->lastchar != '\n')
			&& (at->lastchar != EOF)) {
			at->tag[at->length] = at->lastchar;
			++at->length;
		} else {
			at->tag[at->length] = '\0';
			break;
		}
	}
	return at;
}

/*Return TRUE if UNKNOWN or ERROR*/
static gboolean
get_primary_tag						(FILE 			*f,
							FieldData		*fd)
{
	PTag *P_Tag = g_malloc0 (sizeof (PTag));
	AnyTag *at;
	gboolean unknown = FALSE;
	
	at = get_next_any_tag (f);
	/*check for an error*/
	if ((at->lastchar != SEP) && (at->lastchar != ENDTAGS)) {
		fd->c = EOF;
		g_free (at);
		g_free (P_Tag);
		return TRUE;
	} else {
		fd->c = at->lastchar;
	}
	/*OK - now we have to work out what tag it is.
	Check first 2 chars and the length if length > 2
	don't worry about P_Tag->tag if known*/
	if (at->tag[0] == 'A') {        /*ADR*/
		if ((at->tag[1] == 'D') && (at->length == 3)) {
			P_Tag->type = ADR;
		} else {	/*Unknown*/
			unknown = TRUE;
		}
	} else if (at->tag[0] == 'B') { /*BDAY BEGIN*/
		if ((at->tag[1] == 'D') && (at->length == 4)) {
			P_Tag->type = BDAY;
		} else if ((at->tag[1] == 'E') && (at->length == 5)) {
			P_Tag->type = BEGIN;
		} else {	/*Unknown*/
			unknown = TRUE;
		}
	} else if (at->tag[0] == 'C') { /*CATEGORIES*/
		if ((at->tag[1] == 'A') && (at->length == 10)) {
			P_Tag->type = CATEGORIES;
		} else {	/*Unknown*/
			unknown = TRUE;
		}
	} else if (at->tag[0] == 'E') { /*END EMAIL*/
		if ((at->tag[1] == 'N') && (at->length == 3)) {
			P_Tag->type = END;
		} else if ((at->tag[1] == 'M') && (at->length == 5)) {
			P_Tag->type = EMAIL;
		} else {	/*Unknown*/
			unknown = TRUE;
		}
	} else if (at->tag[0] == 'F') { /*FN*/
		if ((at->tag[1] == 'N') && (at->length == 2)) {
			P_Tag->type = FN;
		} else {	/*Unknown*/
			unknown = TRUE;
		}
	} else if (at->tag[0] == 'G') { /*GEO*/
		if ((at->tag[1] == 'E') && (at->length == 3)) {
			P_Tag->type = GEO;
		} else {	/*Unknown*/
			unknown = TRUE;
		}
	} else if (at->tag[0] == 'K') { /*KEY*/
		if ((at->tag[1] == 'E') && (at->length == 3)) {
			P_Tag->type = KEY;
		} else {	/*Unknown*/
			unknown = TRUE;
		}
	} else if (at->tag[0] == 'L') { /*LOGO*/
		if ((at->tag[1] == 'O') && (at->length == 4)) {
			P_Tag->type = LOGO;
		} else {	/*Unknown*/
			unknown = TRUE;
		}
	} else if (at->tag[0] == 'N') { /*N NOTE*/
		if (at->length == 1) {
			P_Tag->type = N;
		} else if ((at->tag[1] == 'O') && (at->length == 4)) {
			P_Tag->type = NOTE;
		} else {	/*Unknown*/
			unknown = TRUE;
		}
	} else if (at->tag[0] == 'O') { /*ORG*/
		if ((at->tag[1] == 'R') && (at->length == 3)) {
			P_Tag->type = ORG;
		} else {	/*Unknown*/
			unknown = TRUE;
		}
	} else if (at->tag[0] == 'R') { /*REV*/
		if ((at->tag[1] == 'E') && (at->length == 3)) {
			P_Tag->type = REV;
		} else {	/*Unknown*/
			unknown = TRUE;
		}
	} else if (at->tag[0] == 'T') { /*TZ TEL TITLE*/
		if ((at->tag[1] == 'Z') && (at->length == 2)) {
			P_Tag->type = TZ;
		} else if ((at->tag[1] == 'E') && (at->length == 3)) {
			P_Tag->type = TEL;
		} else if ((at->tag[1] == 'I') && (at->length == 5)) {
			P_Tag->type = TITLE;
		} else {	/*Unknown*/
			unknown = TRUE;
		}
	} else if (at->tag[0] == 'U') { /*URL*/
		if ((at->tag[1] == 'R') && (at->length == 3)) {
			P_Tag->type = URL;
		} else {	/*Unknown*/
			unknown = TRUE;
		}
	} else {
		unknown = TRUE;
	}
	if (unknown) {
		P_Tag->type = UNKNOWN;
		P_Tag->tag = g_strdup (at->tag);
		printf ("UNKNOWN TAG = %s\n", P_Tag->tag);
	}
	g_free (at);
	fd->P_Tag = P_Tag;
	return unknown;
}

static gboolean
get_secondary_tags					(FILE			*f,
							FieldData		*fd)
{
	AnyTag *at;
	gboolean unknown = FALSE;
	
	/* List of tags we are searching for
	AOL ATTMAIL APPLELINK
	BBS BMP
	CAR CIS CELL
	DOM
	EWORLD
	FAX
	GIF
	HOME
	INTL ISDN IBMMail INTERNET
	JPEG
	MSG MODEM MCIMAIL
	PS PDF PMB PGP PREF PAGER PARCEL POSTAL PRODIGY POWERSHARE
	TLX TIFF
	VOICE VCARD VIDEO
	WORK
	X400 X509			
	*/

	while (1) {
		STag *S_Tag = g_malloc0 (sizeof (STag));
		
		at = get_next_any_tag (f);
		/*check for an error*/
		if (at->lastchar == EOF) {
			fd->c = EOF;
			g_free (at);
			g_free (S_Tag);
			return TRUE;
		} else {
			fd->c = at->lastchar;
		}
		/* Check for a EncodeType QUOTED-PRINTABLE or BASE64*/
		if (at->tag[0] == 'Q') {
			if ((at->tag[1] == 'U') && (at->length == 16)) {
				fd->encode = ENC_QUOTED_PRINTABLE;
				g_free (at);
				g_free (S_Tag);
				if (fd->c == SEP)
					continue;
				else
					break;
			}
		} else if (at->tag[0] == 'B') {
			if ((at->tag[1] == 'A') && (at->length == 6)) {
				fd->encode = ENC_BASE64;
				g_free (at);
				g_free (S_Tag);
				if (fd->c == SEP)
					continue;
				else
					break;
			}
		}
		/*OK - now we have to work out what tag it is.
		Check first 2 chars and the length if length > 2
		don't worry about S_Tag->tag if known*/
		if (at->tag[0] == 'A') {        /*AOL ATTMAIL APPLELINK*/
			if ((at->tag[1] == 'O') && (at->length == 3)) {
				S_Tag->type = AOL;
			} else if ((at->tag[1] == 'T') && (at->length == 7)) {
				S_Tag->type = ATTMAIL;
			} else if ((at->tag[1] == 'P') && (at->length == 9)) {
				S_Tag->type = APPLELINK;
			} else {	/*Unknown*/
				unknown = TRUE;
			}
		} else if (at->tag[0] == 'B') { /*BBS BMP*/
			if ((at->tag[1] == 'B') && (at->length == 3)) {
				S_Tag->type = BBS;
			} else if ((at->tag[1] == 'M') && (at->length == 3)) {
				S_Tag->type = BMP;
			} else {	/*Unknown*/
				unknown = TRUE;
			}
		} else if (at->tag[0] == 'C') { /*CAR CIS CELL*/
			if ((at->tag[1] == 'A') && (at->length == 3)) {
				S_Tag->type = CAR;
			} else if ((at->tag[1] == 'I') && (at->length == 3)) {
				S_Tag->type = CIS;
			} else if ((at->tag[1] == 'E') && (at->length == 4)) {
				S_Tag->type = CELL;
			} else {	/*Unknown*/
				unknown = TRUE;
			}
		} else if (at->tag[0] == 'D') { /*DOM*/
			if ((at->tag[1] == 'O') && (at->length == 3)) {
				S_Tag->type = DOM;
			} else {	/*Unknown*/
				unknown = TRUE;
			}
		} else if (at->tag[0] == 'E') { /*EWORLD*/
			if ((at->tag[1] == 'W') && (at->length == 6)) {
				S_Tag->type = EWORLD;
			} else {	/*Unknown*/
				unknown = TRUE;
			}
		} else if (at->tag[0] == 'F') { /*FAX*/
			if ((at->tag[1] == 'A') && (at->length == 3)) {
				S_Tag->type = FAX;
			} else {	/*Unknown*/
				unknown = TRUE;
			}
		} else if (at->tag[0] == 'G') { /*GIF*/
			if ((at->tag[1] == 'I') && (at->length == 3)) {
				S_Tag->type = GIF;
			} else {	/*Unknown*/
				unknown = TRUE;
			}
		} else if (at->tag[0] == 'H') { /*HOME*/
			if ((at->tag[1] == 'O') && (at->length == 4)) {
				S_Tag->type = HOME;
			} else {	/*Unknown*/
				unknown = TRUE;
			}
		} else if (at->tag[0] == 'I') { /*INTL ISDN IBMMail INTERNET*/
			if ((at->tag[1] == 'N') && (at->length == 4)) {
				S_Tag->type = INTL;
			} else if ((at->tag[1] == 'S') && (at->length == 4)) {
				S_Tag->type = ISDN;
			} else if ((at->tag[1] == 'B') && (at->length == 7)) {
				S_Tag->type = IBMMail;
			} else if ((at->tag[1] == 'N') && (at->length == 8)) {
				S_Tag->type = INTERNET;
			} else {	/*Unknown*/
				unknown = TRUE;
			}
		} else if (at->tag[0] == 'J') { /*JPEG*/
			if ((at->tag[1] == 'E') && (at->length == 4)) {
				S_Tag->type = JPEG;
			} else {	/*Unknown*/
				unknown = TRUE;
			}
		} else if (at->tag[0] == 'M') { /*MSG MODEM MCIMAIL*/
			if ((at->tag[1] == 'S') && (at->length == 3)) {
				S_Tag->type = MSG;
			} else if ((at->tag[1] == 'O') && (at->length == 5)) {
				S_Tag->type = MODEM;
			} else if ((at->tag[1] == 'C') && (at->length == 7)) {
				S_Tag->type = MCIMAIL;
			} else {	/*Unknown*/
				unknown = TRUE;
			}
		} else if (at->tag[0] == 'P') { /*PS PDF PMB PGP PREF PAGER PARCEL POSTAL PRODIGY POWERSHARE*/
			if ((at->tag[1] == 'S') && (at->length == 2)) {
				S_Tag->type = PS;
			} else if ((at->tag[1] == 'D') && (at->length == 3)) {
				S_Tag->type = PDF;
			} else if ((at->tag[1] == 'M') && (at->length == 3)) {
				S_Tag->type = PMB;
			} else if ((at->tag[1] == 'G') && (at->length == 3)) {
				S_Tag->type = PGP;
			} else if ((at->tag[1] == 'R') && (at->length == 4)) {
				S_Tag->type = PREF;
			} else if ((at->tag[1] == 'A') && (at->length == 5)) {
				S_Tag->type = PAGER;
			} else if ((at->tag[1] == 'A') && (at->length == 6)) {
				S_Tag->type = PARCEL;
			} else if ((at->tag[1] == 'O') && (at->length == 6)) {
				S_Tag->type = POSTAL;
			} else if ((at->tag[1] == 'R') && (at->length == 7)) {
				S_Tag->type = PRODIGY;
			} else if ((at->tag[1] == 'O') && (at->length == 10)) {
				S_Tag->type = POWERSHARE;
			} else {	/*Unknown*/
				unknown = TRUE;
			}
		} else if (at->tag[0] == 'T') { /*TLX TIFF*/
			if ((at->tag[1] == 'L') && (at->length == 3)) {
				S_Tag->type = TLX;
			} else if ((at->tag[1] == 'I') && (at->length == 4)) {
				S_Tag->type = TIFF;
			} else {	/*Unknown*/
				unknown = TRUE;
			}
		} else if (at->tag[0] == 'V') { /*VOICE VCARD VIDEO*/
			if ((at->tag[1] == 'O') && (at->length == 5)) {
				S_Tag->type = VOICE;
			} else if ((at->tag[1] == 'C') && (at->length == 5)) {
				S_Tag->type = VCARD;
			} else if ((at->tag[1] == 'I') && (at->length == 5)) {
				S_Tag->type = VIDEO;
			} else {	/*Unknown*/
				unknown = TRUE;
			}
		} else if (at->tag[0] == 'W') { /*WORK*/
			if ((at->tag[1] == 'O') && (at->length == 4)) {
				S_Tag->type = WORK;
			} else {	/*Unknown*/
				unknown = TRUE;
			}
		} else if (at->tag[0] == 'X') { /*X400 X509*/
			if ((at->tag[1] == '4') && (at->length == 4)) {
				S_Tag->type = X400;
			} else if ((at->tag[1] == '5') && (at->length == 4)) {
				S_Tag->type = X509;
			} else {	/*Unknown*/
				unknown = TRUE;
			}
		} else {
			unknown = TRUE;
		}
		if (unknown) {
			S_Tag->type = SUBUNKNOWN;
			S_Tag->tag = g_strdup (at->tag);
			printf ("UNKNOWN SECONDARY TAG=%s\n", at->tag);
		}
		g_free (at);
		fd->S_Tags = g_slist_append (fd->S_Tags, S_Tag);
		if (fd->c == '\r') {
			while (fd->c != '\n')
				fd->c = fgetc (f);
			break;
		} else if (fd->c == '\n') {
			fd->c = fgetc (f);
			break;
		}
		if (fd->c == ENDTAGS)
			break;
	}
	return unknown;
}

static gchar
*update_data						(gchar			*data,
							gchar			*buff)
{
	gchar *data_tmp;
	
	if (data) {
		data_tmp = g_strdup (data);
		g_free (data);
		data = g_strdup_printf ("%s%s", data_tmp, buff);
		g_free (data_tmp);
	} else {
		data = g_strdup (buff);
	}
	return data;
}

static void
get_quoted_printable_data				(FILE 			*f,
							FieldData 		*fd)
{
	const gint MAX_LINE_LEN = 75;
	gchar buff[MAX_LINE_LEN + 4];
	gchar *data = NULL;
	gint i, ci;
	int la;
	
	ci = 0;
	
	for (i = 0; i < 76; i++) {
		buff[ci] = fgetc (f);
		if (buff[ci] == '=') {
			la = look_ahead (f);
			if ((la == '\r') || (la == '\n')) {	/*more on the next line*/
				buff[ci] = '\0';
				data = update_data (data, buff);
				/*move up to the next line*/
				fd->c = 'n';
				fd->c = skip_to_newline	(f, fd->c);
				i = 0;
				ci = 0;
				continue;
			} else {
				++i;
				++ci;
				buff[ci] = fgetc (f);
				if (buff[ci] == '0') {
					++ci;
					++i;
					buff[ci] = fgetc (f);
					if (buff[ci] == 'A') {	/*replace = with new line*/
						ci -= 2;
						buff[ci] = '\n';
					}
				}
				++ci;
				continue;
			}
		} else if ((buff[ci] == '\r') || (buff[ci] == '\n')) {
			fd->c = buff[ci];
			buff[ci] = '\0';
			data = update_data (data, buff);	
			/*move up to the next line*/
			fd->c = skip_to_newline	(f, fd->c);
			break;
		}
		++ci;
	}
	if (data) {
		fd->Data = g_slist_append (fd->Data, data);
		data = NULL;
	} else {
		printf ("\t\tERROR\n");
	}				
}

/*
 * NOTE This does not yet convert to the base64 format for binary encoding		FIX
 * It only removes white space and removes the '\t' and '\r' 's.
*/
static void
get_base64_data					(FILE 			*f,
						FieldData 		*fd)
{
	const gint MAX_LINE_LEN = 75;
	gchar buff[MAX_LINE_LEN + 4];
	gchar *data = NULL;
	gint ci = 0;
	
	fd->c = fgetc (f);
	while (1) {
		if (ci >= MAX_LINE_LEN) {	/*Unrecoverable ERROR*/
			if (data) g_free(data);
			fd->c = EOF;
			printf("\t\tERROR\n");
			break;
		}
		if ((fd->c == ' ') || (fd->c == '\t') || (fd->c == '\r')) {
			fd->c = fgetc (f);
			continue;
		} else if (fd->c == '\n') {
			fd->c = fgetc (f);
			if (fd->c != ' ') { /*end of data - move to the end of line*/
				fd->c = skip_to_newline	(f, fd->c);
				buff[ci] = '\0';
				data = update_data (data, buff);
				fd->Data = g_slist_append (fd->Data, data);
				break;
			} else {
				buff[ci] = '\n';
				buff[ci + 1] = '\0';
				data = update_data (data, buff);
				ci = 0;
				fd->c = fgetc (f);
				continue;
			}
		} else if (fd->c == EOF) {	/*Unrecoverable ERROR*/
			if (data) g_free (data);
			printf("\t\tERROR\n");
			break;
		} else {
			buff[ci] = fd->c;
		}
		fd->c = fgetc (f);
		++ci;
	}
}

static void
get_delimeted_data_list				(FILE 			*f,
						FieldData 		*fd)
{
	const gint MAX_BUFF_LEN = 255;
	gchar buff[MAX_BUFF_LEN + 1];
	gchar *data = NULL;
	gint i = 0;
	gint ci = 0;
	
	while (1) {
		buff[i] = fgetc (f);
		if (buff[i] == SEP) {
			if (ci == 0) {
				fd->Data = g_slist_append (fd->Data, NULL);
				ci = 0;
				continue;
			} else {
				buff[i] = '\0';
				data = update_data (data, buff);
				fd->Data = g_slist_append (fd->Data, data);
				data = NULL;
				i = 0;
				ci = 0;
				continue;
			}
		} else if ((buff[i] == '\r') || (buff[i] == '\0')) {
			fd->c = buff[i];
			buff[i] = '\0';
			data = update_data (data, buff);
			fd->Data = g_slist_append (fd->Data, data);
			fd->c = skip_to_newline	(f, fd->c);
			break;
		} else if (buff[i] == '\n') {
			fd->c = buff[i];
			buff[i] = '\0';
			data = update_data (data, buff);
			fd->Data = g_slist_append (fd->Data, data);
			break;
		} else if (buff[i] == EOF) {		/*Unrecoverable ERROR*/
			if (data) g_free (data);
			fd->c = EOF;
			printf("\t\tERROR\n");
			break;
		}else if (i == MAX_BUFF_LEN) {
			buff[MAX_BUFF_LEN + 1] = '\0';
			data = update_data (data, buff);
			i = 0;
			++ci;
			continue;
		}
		++ci;
		++i;
	}
}

static gboolean
get_field_data_list					(FILE 			*f,
							FieldData 		*fd)
{
	if (fd->encode == ENC_QUOTED_PRINTABLE) {
		get_quoted_printable_data (f, fd);
	} else if (fd->encode == ENC_BASE64) {
		get_base64_data (f, fd);
	} else {
		get_delimeted_data_list (f, fd);
	}
	if (fd->c == EOF) {
		return FALSE;
	} else {
		return TRUE;
	} 
}

/*get all tags and data fields from a line of f*/

static FieldData
*get_next_field_data					(FILE		*f)
{
	FieldData *fd = g_malloc0 (sizeof (FieldData));
	
	fd->encode = ENC_NONE;
	if (get_primary_tag(f,fd)) {
		if (fd->c == EOF) {
			return (free_field_data (fd));
		}
	}
	if ((fd->P_Tag->type == BEGIN) || (fd->P_Tag->type == END)) {
		fd->c = SEP;
	}
	if ((fd->c != ENDTAGS) && (fd->c != EOF)) {
		if (get_secondary_tags (f,fd)) {
			if (fd->c == EOF) {
				return (free_field_data (fd));
			}
		}
	}
	
	if ((fd->c != EOF)
		&& (fd->P_Tag->type != BEGIN)
		&& (fd->P_Tag->type != END)
		&& (fd->c != '\n')) {
		if (!get_field_data_list (f,fd)) {
			return (free_field_data (fd));
		}
	}
	
	return fd;
}

static GList
*abort_read_Vcard_file					(GList 			*ABook)
{
	lib_gefax_free_address_book (ABook);
	printf ("An ERROR occured while reading address book file\n");
	return NULL;
}

GList
*lib_gefax_read_Vcard_file				(gchar			*absolute_file_path)
{
	FILE *f;
	GList *NewAddressBook = NULL;
	LibGefaxContact *NewCard = NULL;
	FieldData *FData;
	int lh;
	
	if ((f = fopen (absolute_file_path, "r")) == NULL) {
		return NULL;
	}
	lh = look_ahead (f);
	while (lh != EOF) {
		if ((lh == '\r') || (lh == '\n')){
			lh = getc (f);
			lh = look_ahead (f);
			continue;
		}
		FData = get_next_field_data (f);
		if (!FData) {
			break;
		}else if (FData->c == EOF) {
			break;
		}
		if (FData->P_Tag->type == TEL) {
			add_TEL_to_contact (NewCard, FData);
		} else if (FData->P_Tag->type == EMAIL) {
			add_EMAIL_to_contact (NewCard, FData);
		} else if (FData->P_Tag->type == ADR) {
			add_ADR_to_contact (NewCard, FData);
		} else if (FData->P_Tag->type == FN) {
			add_FN_to_contact (NewCard, FData);
		} else if (FData->P_Tag->type == N) {
			add_N_to_contact (NewCard, FData);
		} else if (FData->P_Tag->type == BDAY) {
			add_BDAY_to_contact (NewCard, FData);
		} else if (FData->P_Tag->type == REV) {
			add_REV_to_contact (NewCard, FData);
		} else if (FData->P_Tag->type == ORG) {
			add_ORG_to_contact (NewCard, FData);
		} else if (FData->P_Tag->type == TITLE) {
			add_TITLE_to_contact (NewCard, FData);
		} else if (FData->P_Tag->type == URL) {
			add_URL_to_contact (NewCard, FData);
		} else if (FData->P_Tag->type == CATEGORIES) {
			add_CATEGORIES_to_contact (NewCard, FData);
		} else if (FData->P_Tag->type == NOTE) {
			add_NOTE_to_contact (NewCard, FData);
		} else if (FData->P_Tag->type == KEY) {
			add_KEY_to_contact (NewCard, FData);
		} else if (FData->P_Tag->type == TZ) {
			add_TZ_to_contact (NewCard, FData);
		} else if (FData->P_Tag->type == GEO) {
			add_GEO_to_contact (NewCard, FData);
		} else if (FData->P_Tag->type == LOGO) {
			add_LOGO_to_contact (NewCard, FData);
		} else if ((FData->P_Tag->type == UNKNOWN) || (verify_sub_tag (FData->S_Tags,SUBUNKNOWN))) {
			add_UNKNOWN_to_contact (NewCard, FData);
		} else if (FData->P_Tag->type == BEGIN) {
			if (verify_sub_tag (FData->S_Tags,VCARD)) {
				NewCard = lib_gefax_contact_new ();
				/*add to address book*/
				NewAddressBook = lib_gefax_add_contact (NewAddressBook, NewCard);
			} else {
				/*an Error in file QUIT*/
				FData = free_field_data (FData);
				NewAddressBook = abort_read_Vcard_file (NewAddressBook);
				break;
			}
		} else if (FData->P_Tag->type == END) {
			if (!verify_sub_tag (FData->S_Tags,VCARD)) {
				/*an Error in file QUIT*/
				FData = free_field_data (FData);
				NewAddressBook = abort_read_Vcard_file (NewAddressBook);
				break;
			}
		} else if (!FData) {
			/*an Error in file QUIT*/
			NewAddressBook = abort_read_Vcard_file (NewAddressBook);
			break;
		} else {
			printf ("ERROR Didn't handle Known Primary TAG\n");
		}
		FData = free_field_data (FData);
		lh = look_ahead (f);
	}
	fclose (f);
	return NewAddressBook;
	
}
