/*  Copyright Daryl Gray 2000 email daryl.g@visto.com
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include <time.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include "contact-dialog.h"

#define ADDRESS_TYPES_LENGTH	6
#define PHONE_TYPES_LENGTH	13
#define EMAIL_TYPES_LENGTH	12
#define NO_KEY			-1

struct _ContactDialogPriv {
	gboolean	new;		/*True if new contact*/
	LibGefaxContact	*Contact;	/*Contact to edit or, new contact*/
	GList		*AddressBook;	/*Associated Address book*/
	gint		contact_key;	/*Original contact key if !new */
	gint		address_key;	/*currently selected address key*/
	gint		phone_key;	/*currently selected phone key*/
	gint		email_key;	/*currently selected email key*/
};

struct _I_Tab {
	GtkWidget	*entry_file;
	GtkWidget	*entry_name_family;
	GtkWidget	*entry_name_given;
	GtkWidget	*entry_name_middle;
	GtkWidget	*entry_name_prefix;
	GtkWidget	*entry_name_suffix;
	GtkWidget	*entry_cats;
	GtkWidget	*btn_cats;
	GtkWidget	*entry_org_title;
	GtkWidget	*entry_org_name;
	GtkWidget	*dateedit_birth;
	gboolean	bday_changed;	/*True if Birth Date changed*/
};

struct _A_Tab {
	GtkWidget	*entry_addr_postoffice;
	GtkWidget	*entry_addr_extended;
	GtkWidget	*entry_addr_street;
	GtkWidget	*entry_addr_city;
	GtkWidget	*entry_addr_region;
	GtkWidget	*entry_addr_postcode;
	GtkWidget	*entry_addr_country;
	GtkWidget	*types[ADDRESS_TYPES_LENGTH];
	GtkWidget	*btn_add;
	GtkWidget	*btn_modify;
	GtkWidget	*btn_remove;
	GtkWidget	*clist_addr;
};

struct _P_Tab {
	GtkWidget	*entry_phone_number;
	GtkWidget	*types[PHONE_TYPES_LENGTH];
	GtkWidget	*btn_add;
	GtkWidget	*btn_modify;
	GtkWidget	*btn_remove;
	GtkWidget	*clist_phone;
};

struct _N_Tab {
	GtkWidget	*entry_url;
	GtkWidget	*entry_email;
	GtkWidget	*types[EMAIL_TYPES_LENGTH];
	GtkWidget	*btn_add;
	GtkWidget	*btn_modify;
	GtkWidget	*btn_remove;
	GtkWidget	*clist_email;
};

struct _X_Tab {
	GtkWidget	*rad_pgp;
	GtkWidget	*rad_x509;
	GtkWidget	*text_key;
	GtkWidget	*spin_hrs;
	GtkWidget	*spin_mins;
	GtkWidget	*spin_lat;
	GtkWidget	*spin_lon;
	GtkWidget	*text_comment;
};

enum {
	ADDRESSBOOK_CHANGED,
	DIALOG_HELP,
	LAST_SIGNAL
};

/*==================================Edit Contact==========================================*/

/*
 *Add,modify,remove - address,telephone,email
*/

static gchar
*my_gtk_entry_get_text				(GtkWidget			*entry);
static void
my_gtk_entry_set_text				(GtkWidget			*entry,
						gchar				*text);
static void
my_g_free					(gchar				*data);
static void
contact_dialog_clean_address			(LibGefaxAddress		*A);
static int
contact_dialog_get_address_type			(A_Tab				*ATab);
static void
contact_dialog_clean_telephone			(LibGefaxTelephone		*T);
static int
contact_dialog_get_telephone_type		(P_Tab				*PTab);
static void
contact_dialog_clean_email			(LibGefaxEmail			*E);
static int
contact_dialog_get_email_type			(N_Tab				*NTab);
static void
contact_dialog_add_address			(GtkButton			*button,
						LibGefaxContactDialog		*CDiag);
static void
contact_dialog_modify_address			(GtkButton			*button,
						LibGefaxContactDialog		*CDiag);
static void
contact_dialog_remove_address			(GtkButton			*button,
						LibGefaxContactDialog		*CDiag);
static void
contact_dialog_add_telephone			(GtkButton			*button,
						LibGefaxContactDialog		*CDiag);
static void
contact_dialog_modify_telephone			(GtkButton			*button,
						LibGefaxContactDialog		*CDiag);
static void
contact_dialog_remove_telephone			(GtkButton			*button,
						LibGefaxContactDialog		*CDiag);
static void
contact_dialog_add_email			(GtkButton			*button,
						LibGefaxContactDialog		*CDiag);
static void
contact_dialog_modify_email			(GtkButton			*button,
						LibGefaxContactDialog		*CDiag);
static void
contact_dialog_remove_email			(GtkButton			*button,
						LibGefaxContactDialog		*CDiag);
static void
lib_gefax_contact_update_ITAB_data		(I_Tab				*ITab,
						LibGefaxContact			*C);
static void
lib_gefax_contact_update_NTAB_data		(N_Tab				*NTab,
						LibGefaxContact			*C);
static void
lib_gefax_contact_update_XTAB_data		(X_Tab				*XTab,
						LibGefaxContact			*C);
static void
card_dialog_update_contact			(LibGefaxContactDialog		*CDiag);

/*
 *Display address,phone,email when a clist row is selected
*/

static gint
contact_dialog_get_clist_key			(GtkCList			*clist,
						gint				row);
static void
contact_dialog_display_address			(GtkCList			*clist,
						gint				row,
						gint				column,
						GdkEvent			*event,
						LibGefaxContactDialog		*CDiag);
static void
contact_dialog_display_phone			(GtkCList			*clist,
						gint				row,
						gint				column,
						GdkEvent			*event,
						LibGefaxContactDialog		*CDiag);
static void
contact_dialog_display_email			(GtkCList			*clist,
						gint				row,
						gint				column,
						GdkEvent			*event,
						LibGefaxContactDialog		*CDiag);


/*
 *Fill Dialog with Contact
*/

static void
lib_gefax_contact_dialog_fill_ITAB		(I_Tab				*ITab,
						LibGefaxContact			*C);
static void
lib_gefax_contact_dialog_fill_ATAB		(A_Tab				*ATab,
						LibGefaxContact			*C);
static void
lib_gefax_contact_dialog_fill_PTAB		(P_Tab				*PTab,
						LibGefaxContact			*C);
static void
lib_gefax_contact_dialog_fill_NTAB		(N_Tab				*NTab,
						LibGefaxContact			*C);
static void
lib_gefax_contact_dialog_fill_XTAB		(X_Tab				*XTab,
						LibGefaxContact			*C);

static void
lib_gefax_contact_dialog_fill			(LibGefaxContactDialog		*CDiag);

/*==================================Dialog Signals==========================================*/

static void
on_card_dialog_apply				(GtkWidget			*widget,
						LibGefaxContactDialog		*CDiag);
static void
on_card_dialog_ok				(GtkWidget			*widget,
						LibGefaxContactDialog		*CDiag);

static void
on_card_dialog_help				(GnomePropertyBox		*gnomepropertybox,
						gint				arg1,
						LibGefaxContactDialog		*CDiag);


/*==================================Identity Tab============================================*/
/*
 *Make a string from 5 name fields and put it in the File Name text entry
*/
static void
contact_dialog_take_filename_from_name		(GtkButton			*button,
						I_Tab				*ITab);
static void
contact_dialog_dateedit_birth_changed		(GtkWidget			*date_edit,
						I_Tab				*ITab);
/*
 *Build the Identity Tab + callbacks
*/
static void
contact_dialog_create_I_TAB			(I_Tab				*ITab,
						GtkWidget			*notebook);

/*==================================Address Tab=============================================*/

static void
ATab_address_changed_call			(GtkWidget			*widget,
						LibGefaxContactDialog		*CDiag);
static void
contact_dialog_create_A_TAB			(A_Tab				*ATab,
						GtkWidget			*notebook);

/*==================================Phone Tab===============================================*/
static void
PTab_phone_changed_call				(GtkWidget			*widget,
						LibGefaxContactDialog		*CDiag);
static void
contact_dialog_create_P_TAB			(P_Tab				*PTab,
						GtkWidget			*notebook);

/*==================================Network Tab=============================================*/
static void
NTab_email_changed_call				(GtkWidget			*widget,
						LibGefaxContactDialog		*CDiag);
static void
contact_dialog_create_N_TAB			(N_Tab				*NTab,
						GtkWidget			*notebook);

/*==================================Additional Tab==========================================*/
static void
contact_dialog_create_X_TAB			(X_Tab				*XTab,
						GtkWidget			*notebook);

/*==================================Widget Functions=========================================*/
static gint
lib_gefax_contact_dialog_signals[LAST_SIGNAL] = { 0 };

static GnomePropertyBoxClass
*lib_gefax_contact_dialog_parent_class;

static void
lib_gefax_contact_dialog_class_init		(LibGefaxContactDialogClass	*Klass);

static void
lib_gefax_contact_dialog_init			(LibGefaxContactDialog		*ContactDiag);

static void
lib_gefax_contact_dialog_destroy		(GtkObject			*ContactDiag);

/*==================================Edit Contact==========================================*/

static gchar
*my_gtk_entry_get_text				(GtkWidget			*entry)
{
	gchar *text;
	text = gtk_entry_get_text (GTK_ENTRY (entry));
	if (text[0] != '\0') {
		return (g_strdup (text));
	} else {
		return NULL;
	}
}

static void
my_gtk_entry_set_text				(GtkWidget			*entry,
						gchar				*text)
{
	if (text) {
		gtk_entry_set_text (GTK_ENTRY (entry), text);
	} else {
		gtk_entry_set_text (GTK_ENTRY (entry), "");
	}
	return;
}

static void
my_g_free					(gchar				*data)
{
	if (data) {
		g_free (data);
	}
}

static void
contact_dialog_clean_address			(LibGefaxAddress		*A)
{
	A->type = 0;
	my_g_free (A->post_box);
	my_g_free (A->extended);
	my_g_free (A->street);
	my_g_free (A->city);
	my_g_free (A->region);
	my_g_free (A->post_code);
	my_g_free (A->country);
}
static int
contact_dialog_get_address_type			(A_Tab				*ATab)
{
	int type = 0;
	gint i;
	
	for (i = 0; i < ADDRESS_TYPES_LENGTH; i++) {
		if (GTK_TOGGLE_BUTTON (ATab->types[i])->active) {
	    		type |= 1 << i;
	    	}
	}
	return type;
}

static void
contact_dialog_clean_telephone			(LibGefaxTelephone		*T)
{
	T->type = 0;
	my_g_free (T->number);
}

static int
contact_dialog_get_telephone_type		(P_Tab				*PTab)
{
	int type = 0;
	gint i;
	
	for (i = 0; i < PHONE_TYPES_LENGTH; i++) {
		if (GTK_TOGGLE_BUTTON (PTab->types[i])->active) {
	    		type |= 1 << i;
	    	}
	}
	return type;
}

static void
contact_dialog_clean_email			(LibGefaxEmail			*E)
{
	E->type = 0;
	my_g_free (E->number);
}

static int
contact_dialog_get_email_type			(N_Tab				*NTab)
{
	int type = 0;
	gint i;
	
	for (i = 0; i < EMAIL_TYPES_LENGTH; i++) {
		if (GTK_TOGGLE_BUTTON (NTab->types[i])->active) {
	    		type = i;
	    		break;
	    	}
	}
	return type;
}

static void
contact_dialog_add_address			(GtkButton			*button,
						LibGefaxContactDialog		*CDiag)
{
	A_Tab *Atab = CDiag->ATab;
	LibGefaxAddress *Address = LIB_GEFAX_ADDRESS_NEW ();
	GnomePropertyBox *pb;
	
	pb = GNOME_PROPERTY_BOX (CDiag);
	gnome_property_box_changed (pb);
	
	Address->type = contact_dialog_get_address_type (Atab);
	Address->post_box = my_gtk_entry_get_text (Atab->entry_addr_postoffice);
	Address->extended = my_gtk_entry_get_text (Atab->entry_addr_extended);
	Address->street = my_gtk_entry_get_text (Atab->entry_addr_street);
	Address->city = my_gtk_entry_get_text (Atab->entry_addr_city);
	Address->region = my_gtk_entry_get_text (Atab->entry_addr_region);
	Address->post_code = my_gtk_entry_get_text (Atab->entry_addr_postcode);
	Address->country = my_gtk_entry_get_text (Atab->entry_addr_country);
	
	if (!lib_gefax_contact_add_address (CDiag->Priv->Contact, Address)) {
		gnome_error_dialog (_("Address already exists"));
		contact_dialog_clean_address (Address);
		g_free (Address);
	} else {
		lib_gefax_contact_dialog_fill_ATAB (Atab, CDiag->Priv->Contact);
	}
	CDiag->Priv->address_key = NO_KEY;
}

static void
contact_dialog_modify_address			(GtkButton			*button,
						LibGefaxContactDialog		*CDiag)
{
	LibGefaxAddress *Address;
	A_Tab *Atab = CDiag->ATab;
	GnomePropertyBox *pb;
	
	pb = GNOME_PROPERTY_BOX (CDiag);
	gnome_property_box_changed (pb);
	
	Address = lib_gefax_contact_get_address_by_key (CDiag->Priv->Contact, CDiag->Priv->address_key);
	contact_dialog_clean_address (Address);
	
	Address->type = contact_dialog_get_address_type (Atab);
	Address->post_box = my_gtk_entry_get_text (Atab->entry_addr_postoffice);
	Address->extended = my_gtk_entry_get_text (Atab->entry_addr_extended);
	Address->street = my_gtk_entry_get_text (Atab->entry_addr_street);
	Address->city = my_gtk_entry_get_text (Atab->entry_addr_city);
	Address->region = my_gtk_entry_get_text (Atab->entry_addr_region);
	Address->post_code = my_gtk_entry_get_text (Atab->entry_addr_postcode);
	Address->country = my_gtk_entry_get_text (Atab->entry_addr_country);
	
	lib_gefax_contact_dialog_fill_ATAB (Atab, CDiag->Priv->Contact);
	CDiag->Priv->address_key = NO_KEY;
}

static void
contact_dialog_remove_address			(GtkButton			*button,
						LibGefaxContactDialog		*CDiag)
{
	LibGefaxAddress *Address;
	GnomePropertyBox *pb;
	
	pb = GNOME_PROPERTY_BOX (CDiag);
	gnome_property_box_changed (pb);
	
	Address = lib_gefax_contact_get_address_by_key (CDiag->Priv->Contact, CDiag->Priv->address_key);

	if (lib_gefax_contact_remove_address (CDiag->Priv->Contact, Address)) {
		lib_gefax_contact_dialog_fill_ATAB (CDiag->ATab, CDiag->Priv->Contact);
	} else {
		gnome_error_dialog (_("Address doesn't exist"));
	}
	CDiag->Priv->address_key = NO_KEY;
	
}

static void
contact_dialog_add_telephone			(GtkButton			*button,
						LibGefaxContactDialog		*CDiag)
{
	P_Tab *Ptab = CDiag->PTab;
	LibGefaxTelephone *Tel = LIB_GEFAX_TELEPHONE_NEW ();
	GnomePropertyBox *pb;
	
	pb = GNOME_PROPERTY_BOX (CDiag);
	gnome_property_box_changed (pb);
	
	Tel->type = contact_dialog_get_telephone_type (Ptab);
	Tel->number = my_gtk_entry_get_text (Ptab->entry_phone_number);

	if (!lib_gefax_contact_add_tel (CDiag->Priv->Contact, Tel)) {
		gnome_error_dialog (_("Telephone number already exists"));
		contact_dialog_clean_telephone (Tel);
		g_free (Tel);
	} else {
		lib_gefax_contact_dialog_fill_PTAB (Ptab, CDiag->Priv->Contact);
	}
	CDiag->Priv->phone_key = NO_KEY;
}

static void
contact_dialog_modify_telephone			(GtkButton			*button,
						LibGefaxContactDialog		*CDiag)
{
	P_Tab *Ptab = CDiag->PTab;
	LibGefaxTelephone *Tel;
	GnomePropertyBox *pb;
	
	pb = GNOME_PROPERTY_BOX (CDiag);
	gnome_property_box_changed (pb);
	
	Tel = lib_gefax_contact_get_telephone_by_key (CDiag->Priv->Contact, CDiag->Priv->phone_key);	
	contact_dialog_clean_telephone (Tel);
	Tel->type = contact_dialog_get_telephone_type (Ptab);
	Tel->number = my_gtk_entry_get_text (Ptab->entry_phone_number);

	lib_gefax_contact_dialog_fill_PTAB (Ptab, CDiag->Priv->Contact);
	CDiag->Priv->phone_key = NO_KEY;
}

static void
contact_dialog_remove_telephone			(GtkButton			*button,
						LibGefaxContactDialog		*CDiag)
{
	LibGefaxTelephone *Tel;
	GnomePropertyBox *pb;
	
	pb = GNOME_PROPERTY_BOX (CDiag);
	gnome_property_box_changed (pb);
	
	Tel = lib_gefax_contact_get_telephone_by_key (CDiag->Priv->Contact, CDiag->Priv->phone_key);

	if (lib_gefax_contact_remove_tel (CDiag->Priv->Contact, Tel)) {
		lib_gefax_contact_dialog_fill_PTAB (CDiag->PTab, CDiag->Priv->Contact);
	} else {
		gnome_error_dialog (_("Telephone number doesn't exist"));
	}
	CDiag->Priv->phone_key = NO_KEY;
}

static void
contact_dialog_add_email			(GtkButton			*button,
						LibGefaxContactDialog		*CDiag)
{
	N_Tab *Ntab = CDiag->NTab;
	LibGefaxEmail *Email = LIB_GEFAX_EMAIL_NEW ();
	GnomePropertyBox *pb;
	
	pb = GNOME_PROPERTY_BOX (CDiag);
	gnome_property_box_changed (pb);
	
	Email->type = contact_dialog_get_email_type (Ntab);
	Email->number = my_gtk_entry_get_text (Ntab->entry_email);

	if (!lib_gefax_contact_add_email (CDiag->Priv->Contact, Email)) {
		gnome_error_dialog (_("Email address already exists"));
		contact_dialog_clean_email (Email);
		g_free (Email);
	} else {
		lib_gefax_contact_dialog_fill_NTAB (Ntab, CDiag->Priv->Contact);
	}
	CDiag->Priv->email_key = NO_KEY;
}

static void
contact_dialog_modify_email			(GtkButton			*button,
						LibGefaxContactDialog		*CDiag)
{
	N_Tab *Ntab = CDiag->NTab;
	LibGefaxEmail *Email;
	GnomePropertyBox *pb;
	
	pb = GNOME_PROPERTY_BOX (CDiag);
	gnome_property_box_changed (pb);
	
	Email = lib_gefax_contact_get_email_by_key (CDiag->Priv->Contact, CDiag->Priv->email_key);
	contact_dialog_clean_email (Email);
	Email->type = contact_dialog_get_email_type (Ntab);
	Email->number = my_gtk_entry_get_text (Ntab->entry_email);
	
	lib_gefax_contact_dialog_fill_NTAB (Ntab, CDiag->Priv->Contact);
	CDiag->Priv->email_key = NO_KEY;
}

static void
contact_dialog_remove_email			(GtkButton			*button,
						LibGefaxContactDialog		*CDiag)
{
	LibGefaxEmail *Email;
	GnomePropertyBox *pb;
	
	pb = GNOME_PROPERTY_BOX (CDiag);
	gnome_property_box_changed (pb);
	
	Email = lib_gefax_contact_get_email_by_key (CDiag->Priv->Contact, CDiag->Priv->email_key);

	if (lib_gefax_contact_remove_email (CDiag->Priv->Contact, Email)) {
		lib_gefax_contact_dialog_fill_NTAB (CDiag->NTab, CDiag->Priv->Contact);
	} else {
		gnome_error_dialog (_("Email address doesn't exist"));
	}
	CDiag->Priv->email_key = NO_KEY;
}

static gchar
*update_gchar_from_entry			(gchar				*existing,
						GtkWidget			*entry)
{
	gchar *text;
	if (existing) {
		g_free (existing);
	}
	
	text = gtk_entry_get_text (GTK_ENTRY (entry));
	if (text[0] != '\0'){
		return (g_strdup (text));
	} else {
		return NULL;
	}
}

static void
lib_gefax_contact_update_ITAB_data		(I_Tab				*ITab,
						LibGefaxContact			*C)
{
	LibGefaxIdentity 	*I = C->Identity;
	LibGefaxBusiness 	*B = C->Business;
	LibGefaxDate		*D = C->BirthDate;
	gchar *cats;
	struct tm *tm;
	time_t time;
	
	C->filename	= update_gchar_from_entry (C->filename, ITab->entry_file);
	I->family	= update_gchar_from_entry (I->family, ITab->entry_name_family);
	I->additional	= update_gchar_from_entry (I->additional, ITab->entry_name_middle);
	I->given	= update_gchar_from_entry (I->given, ITab->entry_name_given);
	I->prefix	= update_gchar_from_entry (I->prefix, ITab->entry_name_prefix);
	I->suffix	= update_gchar_from_entry (I->suffix, ITab->entry_name_suffix);
	
	B->org_name	= update_gchar_from_entry (B->org_name, ITab->entry_org_name);
	B->org_title	= update_gchar_from_entry (B->org_title, ITab->entry_org_title);
	
	cats		= update_gchar_from_entry (NULL, ITab->entry_cats);
	lib_gefax_contact_set_categorys (C, cats);
	
	time = gnome_date_edit_get_date (GNOME_DATE_EDIT (ITab->dateedit_birth)); 
	tm = localtime(&time);
	
	if (ITab->bday_changed) {
		D->years	= tm->tm_year + 1900;
		D->months	= tm->tm_mon + 1;
		D->days		= tm->tm_mday;
		ITab->bday_changed = FALSE;
	}
}

static void
lib_gefax_contact_update_NTAB_data		(N_Tab				*NTab,
						LibGefaxContact			*C)
{
	C->url		= update_gchar_from_entry (C->url, NTab->entry_url);
}

static void
lib_gefax_contact_update_XTAB_data		(X_Tab				*XTab,
						LibGefaxContact			*C)
{
	LibGefaxTimeZone	*T = C->TimeZone;
	LibGefaxGeo		*G = C->GeoLocation;
	LibGefaxPublicKey	*K = C->PublicKey;
	
	T->hours	= gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (XTab->spin_hrs));
	T->minutes	= gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (XTab->spin_mins));
	G->latitude	= gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (XTab->spin_lat));
	G->longitude	= gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (XTab->spin_lon));
	
	if (GTK_TOGGLE_BUTTON (XTab->rad_pgp)->active) {
		K->type = LIBGEFAX_PGP;
	} else {
		K->type = LIBGEFAX_X509;
	}
	if (K->key) {
		g_free (K->key);
	}
	K->key = gtk_editable_get_chars (GTK_EDITABLE (XTab->text_key), 0,
			gtk_text_get_length (GTK_TEXT (XTab->text_key)));
	if (K->key[0] == '\0'){
		g_free (K->key);
		K->key = NULL;
	}
	if (C->note) {
		g_free (C->note);
	}
	C->note = gtk_editable_get_chars (GTK_EDITABLE (XTab->text_comment), 0,
			gtk_text_get_length (GTK_TEXT (XTab->text_comment)));
	if (C->note[0] == '\0'){
		g_free (C->note);
		C->note = NULL;
	}
}

static void
card_dialog_update_contact			(LibGefaxContactDialog		*CDiag)
{
	LibGefaxContact *Contact = CDiag->Priv->Contact;
	LibGefaxDate	*R = Contact->Revision;
	struct tm *tm;
	time_t curtime;

	lib_gefax_contact_update_ITAB_data (CDiag->ITab, Contact);
	lib_gefax_contact_update_NTAB_data (CDiag->NTab, Contact);
	lib_gefax_contact_update_XTAB_data (CDiag->XTab, Contact);

	/*Update Revision time*/
	curtime = time (NULL);
	tm = gmtime (&curtime);
	R->years	= tm->tm_year + 1900;
	R->months	= tm->tm_mon + 1;
	R->days		= tm->tm_mday;
	R->seconds	= tm->tm_sec;
	R->minutes	= tm->tm_min;
	R->hours	= tm->tm_hour;
	
}

static gint
contact_dialog_get_clist_key			(GtkCList			*clist,
						gint				row)
{
	gchar *key_char;

	gtk_clist_get_text (GTK_CLIST (clist), row, 1, &key_char);
	return atoi (key_char);
}

static void
contact_dialog_display_address			(GtkCList			*clist,
						gint				row,
						gint				column,
						GdkEvent			*event,
						LibGefaxContactDialog		*CDiag)
{
	A_Tab *Atab = CDiag->ATab;
	LibGefaxAddress *Address;
	gint i;

	CDiag->Priv->address_key = contact_dialog_get_clist_key (GTK_CLIST (Atab->clist_addr), row);
	Address = lib_gefax_contact_get_address_by_key (CDiag->Priv->Contact, CDiag->Priv->address_key);
	
	for (i = 0; i < ADDRESS_TYPES_LENGTH; i++) {
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (Atab->types[i]), Address->type & 1 << i);
	}
	
	my_gtk_entry_set_text (Atab->entry_addr_postoffice, Address->post_box);
	my_gtk_entry_set_text (Atab->entry_addr_extended, Address->extended);
	my_gtk_entry_set_text (Atab->entry_addr_street, Address->street);
	my_gtk_entry_set_text (Atab->entry_addr_city, Address->city);
	my_gtk_entry_set_text (Atab->entry_addr_region, Address->region);
	my_gtk_entry_set_text (Atab->entry_addr_postcode, Address->post_code);
	my_gtk_entry_set_text (Atab->entry_addr_country, Address->country);
	
	gtk_widget_set_sensitive (Atab->btn_add, FALSE);
	gtk_widget_set_sensitive (Atab->btn_modify, FALSE);
	gtk_widget_set_sensitive (Atab->btn_remove, TRUE);
	
	
}

static void
contact_dialog_display_phone			(GtkCList			*clist,
						gint				row,
						gint				column,
						GdkEvent			*event,
						LibGefaxContactDialog		*CDiag)
{
	P_Tab *Ptab = CDiag->PTab;
	LibGefaxTelephone *Tel;
	gint i;

	CDiag->Priv->phone_key = contact_dialog_get_clist_key (GTK_CLIST (Ptab->clist_phone), row);
	Tel = lib_gefax_contact_get_telephone_by_key (CDiag->Priv->Contact, CDiag->Priv->phone_key);
	
	for (i = 0; i < PHONE_TYPES_LENGTH; i++) {
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (Ptab->types[i]), Tel->type & 1 << i);
	}
	
	my_gtk_entry_set_text (Ptab->entry_phone_number, Tel->number);
	
	gtk_widget_set_sensitive (Ptab->btn_add, FALSE);
	gtk_widget_set_sensitive (Ptab->btn_modify, FALSE);
	gtk_widget_set_sensitive (Ptab->btn_remove, TRUE);
}

static void
contact_dialog_display_email			(GtkCList			*clist,
						gint				row,
						gint				column,
						GdkEvent			*event,
						LibGefaxContactDialog		*CDiag)
{
	N_Tab *Ntab = CDiag->NTab;
	LibGefaxEmail *Email;
	gint i;

	CDiag->Priv->email_key = contact_dialog_get_clist_key (GTK_CLIST (Ntab->clist_email), row);
	Email = lib_gefax_contact_get_email_by_key (CDiag->Priv->Contact, CDiag->Priv->email_key);
	
	for (i = 0; i < EMAIL_TYPES_LENGTH; i++) {
		if (Email->type == i) {
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (Ntab->types[i]), TRUE);
			break;
		}
	}
	
	my_gtk_entry_set_text (Ntab->entry_email, Email->number);
	
	gtk_widget_set_sensitive (Ntab->btn_add, FALSE);
	gtk_widget_set_sensitive (Ntab->btn_modify, FALSE);
	gtk_widget_set_sensitive (Ntab->btn_remove, TRUE);
}

static void
lib_gefax_contact_dialog_fill_ITAB		(I_Tab				*ITab,
						LibGefaxContact			*C)
{
	LibGefaxIdentity 	*I = C->Identity;
	LibGefaxBusiness 	*B = C->Business;
	LibGefaxDate		*D = C->BirthDate;
	gchar *cats;
	time_t tmp_time;
	struct tm tm = {0, 0, 0, 0, 0, 0, 0, 0, 0};
	
	my_gtk_entry_set_text (ITab->entry_file, C->filename);
	my_gtk_entry_set_text (ITab->entry_name_family, I->family);
	my_gtk_entry_set_text (ITab->entry_name_given, I->given);
	my_gtk_entry_set_text (ITab->entry_name_middle, I->additional);
	my_gtk_entry_set_text (ITab->entry_name_prefix, I->prefix);
	my_gtk_entry_set_text (ITab->entry_name_suffix, I->suffix);
	cats = lib_gefax_contact_get_categorys_as_string (C);
	my_gtk_entry_set_text (ITab->entry_cats, cats);
	if (cats) {
		g_free (cats);
	}
	my_gtk_entry_set_text (ITab->entry_org_title, B->org_title);
	my_gtk_entry_set_text (ITab->entry_org_name, B->org_name);
	
	if (D->years > 1) {
		tm.tm_year = D->years - 1900;
		tm.tm_mon  = D->months - 1;
		tm.tm_mday = D->days;
		
		tmp_time = mktime (&tm);
		gnome_date_edit_set_time (GNOME_DATE_EDIT (ITab->dateedit_birth), tmp_time);
	}
	
}
static void
lib_gefax_contact_dialog_fill_ATAB		(A_Tab				*ATab,
						LibGefaxContact			*C)
{
	GSList *A = C->Address;
	gint list_len, i;
	
	gtk_widget_set_sensitive (ATab->btn_add, FALSE);
	gtk_widget_set_sensitive (ATab->btn_modify, FALSE);
	gtk_widget_set_sensitive (ATab->btn_remove, FALSE);
	
	gtk_clist_clear (GTK_CLIST (ATab->clist_addr));
	list_len = g_slist_length (A);
	for (i = 0; i < list_len; i++) {
		LibGefaxAddress *Address = A->data;
		gchar *data[2];
		while (1) {
			if (Address->post_box) {
				data[0] = Address->post_box;
				break;
			}
			if (Address->extended) {
				data[0] = Address->extended;
				break;
			}
			if (Address->street) {
				data[0] = Address->street;
				break;
			}
			if (Address->city) {
				data[0] = Address->city;
				break;
			}
			if (Address->region) {
				data[0] = Address->region;
				break;
			}
			if (Address->post_code) {
				data[0] = Address->post_code;
				break;
			}
			if (Address->country) {
				data[0] = Address->country;
				break;
			}
		}
		data[1] = g_strdup_printf ("%d", Address->key);
		gtk_clist_append (GTK_CLIST (ATab->clist_addr), data);
		g_free (data[1]);
		A = g_slist_next (A);
	}
}
static void
lib_gefax_contact_dialog_fill_PTAB		(P_Tab				*PTab,
						LibGefaxContact			*C)
{
	GSList *L = C->Tel;
	gint list_len, i;
	
	gtk_widget_set_sensitive (PTab->btn_add, FALSE);
	gtk_widget_set_sensitive (PTab->btn_modify, FALSE);
	gtk_widget_set_sensitive (PTab->btn_remove, FALSE);
	
	gtk_clist_clear (GTK_CLIST (PTab->clist_phone));
	list_len = g_slist_length (L);
	for (i = 0; i < list_len; i++) {
		LibGefaxTelephone *Telephone = L->data;
		gchar *data[2];
		if (Telephone->number) {
			data[0] = Telephone->number;
		} else {
			L = g_slist_next (L);
			continue;
		}
		data[1] = g_strdup_printf ("%d", Telephone->key);
		gtk_clist_append (GTK_CLIST (PTab->clist_phone), data);
		g_free (data[1]);
		L = g_slist_next (L);
	}
}
static void
lib_gefax_contact_dialog_fill_NTAB		(N_Tab				*NTab,
						LibGefaxContact			*C)
{
	GSList *L = C->Email;
	gint list_len, i;
	
	gtk_widget_set_sensitive (NTab->btn_add, FALSE);
	gtk_widget_set_sensitive (NTab->btn_modify, FALSE);
	gtk_widget_set_sensitive (NTab->btn_remove, FALSE);
	
	gtk_clist_clear (GTK_CLIST (NTab->clist_email));
	list_len = g_slist_length (L);
	for (i = 0; i < list_len; i++) {
		LibGefaxEmail *Email = L->data;
		gchar *data[2];
		if (Email->number) {
			data[0] = Email->number;
		} else {
			L = g_slist_next (L);
			continue;
		}
		data[1] = g_strdup_printf ("%d", Email->key);
		gtk_clist_append (GTK_CLIST (NTab->clist_email), data);
		g_free (data[1]);
		L = g_slist_next (L);
	}
	my_gtk_entry_set_text (NTab->entry_url, C->url);
}

static void
lib_gefax_contact_dialog_fill_XTAB		(X_Tab				*XTab,
						LibGefaxContact			*C)
{
	LibGefaxTimeZone	*T = C->TimeZone;
	LibGefaxGeo		*G = C->GeoLocation;
	LibGefaxPublicKey	*K = C->PublicKey;

	if (K->type == LIBGEFAX_PGP) {
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (XTab->rad_pgp), TRUE);
	} else {
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (XTab->rad_x509), TRUE);
	}
	if (K->key) {
		gtk_text_insert (GTK_TEXT (XTab->text_key), NULL, NULL, NULL, K->key, -1);
	}
	
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (XTab->spin_hrs), T->hours);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (XTab->spin_mins), T->minutes);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (XTab->spin_lat), G->latitude);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (XTab->spin_lon), G->longitude);
	
	if (C->note) {
		gtk_text_insert (GTK_TEXT (XTab->text_comment), NULL, NULL, NULL, C->note, -1);
	}
	
}

static void
lib_gefax_contact_dialog_fill			(LibGefaxContactDialog		*CDiag)
{
	LibGefaxContact *Contact = CDiag->Priv->Contact;
	
	lib_gefax_contact_dialog_fill_ITAB (CDiag->ITab, Contact);
	lib_gefax_contact_dialog_fill_ATAB (CDiag->ATab, Contact);
	lib_gefax_contact_dialog_fill_PTAB (CDiag->PTab, Contact);
	lib_gefax_contact_dialog_fill_NTAB (CDiag->NTab, Contact);
	lib_gefax_contact_dialog_fill_XTAB (CDiag->XTab, Contact);
	
}

/*==================================Dialog Signals==========================================*/

static void
card_dialog_update_address_book			(LibGefaxContactDialog		*CDiag)
{
	card_dialog_update_contact (CDiag);
	if (CDiag->Priv->new) {
		CDiag->Priv->AddressBook = lib_gefax_add_contact (CDiag->Priv->AddressBook,
								CDiag->Priv->Contact);		
	} else {
		
		LibGefaxContact *Existing = lib_gefax_contact_get_by_key (CDiag->Priv->AddressBook,
									CDiag->Priv->contact_key);
		gint index = g_list_index (CDiag->Priv->AddressBook, Existing);
		CDiag->Priv->AddressBook = g_list_remove (CDiag->Priv->AddressBook, Existing);
		CDiag->Priv->Contact->Key = CDiag->Priv->contact_key;
		CDiag->Priv->AddressBook = g_list_insert (CDiag->Priv->AddressBook, CDiag->Priv->Contact, index);

	}
	gtk_signal_emit (GTK_OBJECT (CDiag), lib_gefax_contact_dialog_signals[ADDRESSBOOK_CHANGED],
						CDiag->Priv->AddressBook, CDiag->Priv->contact_key);
}

static void
on_card_dialog_apply				(GtkWidget			*widget,
						LibGefaxContactDialog		*CDiag)
{
	
	card_dialog_update_address_book (CDiag);
	CDiag->Priv->Contact = lib_gefax_contact_clone_by_key (CDiag->Priv->AddressBook, CDiag->Priv->contact_key);
	if (CDiag->Priv->Contact) {
		CDiag->Priv->new = FALSE;
	} else {
		g_warning ("Unable to clone contact");
		gtk_widget_destroy (GTK_WIDGET (CDiag));
	}
	
}

static void
on_card_dialog_ok				(GtkWidget			*widget,
						LibGefaxContactDialog		*CDiag)
{
	if (GTK_WIDGET_SENSITIVE (GNOME_PROPERTY_BOX (CDiag)->apply_button)) {
		card_dialog_update_address_book (CDiag);
		CDiag->Priv->Contact = NULL;
	}
}

static void
on_card_dialog_help				(GnomePropertyBox		*gnomepropertybox,
						gint				arg1,
						LibGefaxContactDialog		*CDiag)
{

	gtk_signal_emit (GTK_OBJECT (CDiag), lib_gefax_contact_dialog_signals[DIALOG_HELP], arg1);
}

/*==================================Identity Tab============================================*/

static void
contact_dialog_take_filename_from_name		(GtkButton			*button,
						I_Tab				*ITab)
{
	const gint DATA_SIZE = 5;
	gchar *data[DATA_SIZE];
	gint i;
	
	data[0] = gtk_entry_get_text (GTK_ENTRY (ITab->entry_name_suffix));
	data[1] = gtk_entry_get_text (GTK_ENTRY (ITab->entry_name_prefix));
	data[2] = gtk_entry_get_text (GTK_ENTRY (ITab->entry_name_given));
	data[3] = gtk_entry_get_text (GTK_ENTRY (ITab->entry_name_middle));
	data[4] = gtk_entry_get_text (GTK_ENTRY (ITab->entry_name_family));
	
	gtk_entry_set_text (GTK_ENTRY (ITab->entry_file), "");
	
	for (i = 0; i < DATA_SIZE; i++) {
		if (strlen (data[i]) > 0) {
			if (i > 0) {
				gtk_entry_append_text (GTK_ENTRY (ITab->entry_file), " ");
			}
			gtk_entry_append_text (GTK_ENTRY (ITab->entry_file), data[i]);
		}
	}
}

static void
contact_dialog_dateedit_birth_changed		(GtkWidget			*date_edit,
						I_Tab				*ITab)
{
	ITab->bday_changed = TRUE;
}

static void
contact_dialog_create_I_TAB			(I_Tab				*ITab,
						GtkWidget			*notebook)
{
	GtkWidget *vbox1;
	GtkWidget *hbox1;
	GtkWidget *btn_take_filename_from_name;
	GtkWidget *lbl_fa;
	GtkWidget *fra_name;
	GtkWidget *table4;
	GtkWidget *label7;
	GtkWidget *label6;
	GtkWidget *hbox5;
	GtkWidget *hbox6;
	GtkWidget *label5;
	GtkWidget *label12;
	GtkWidget *label13;
	GtkWidget *hbox4;
 	GtkWidget *fra_org;
	GtkWidget *table2;
	GtkWidget *label10;
	GtkWidget *label11;
	GtkWidget *lbl_cats;
	GtkWidget *hbox_cats;
	GtkWidget *vbox2;
	GtkWidget *hbox8;
	GtkWidget *label14;
	GtkWidget *lbl_ident;
	
	ITab->bday_changed = FALSE;

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox1);
	gtk_container_add (GTK_CONTAINER (notebook), vbox1);

	hbox1 = gtk_hbox_new (FALSE, 8);
	gtk_widget_show (hbox1);
	gtk_box_pack_start (GTK_BOX (vbox1), hbox1, FALSE, TRUE, 8);
	gtk_widget_set_usize (hbox1, -2, 22);

	lbl_fa = gtk_label_new (_("File As:"));
	gtk_widget_show (lbl_fa);
	gtk_box_pack_start (GTK_BOX (hbox1), lbl_fa, FALSE, TRUE, 0);
	gtk_label_set_justify (GTK_LABEL (lbl_fa), GTK_JUSTIFY_RIGHT);
	gtk_misc_set_alignment (GTK_MISC (lbl_fa), 1, 0.5);
	gtk_misc_set_padding (GTK_MISC (lbl_fa), 10, 0);

	ITab->entry_file = gtk_entry_new ();
	gtk_widget_show (ITab->entry_file);
	gtk_box_pack_start (GTK_BOX (hbox1), ITab->entry_file, TRUE, TRUE, 0);

	btn_take_filename_from_name = gtk_button_new_with_label (_("Take from name"));
	gtk_widget_show (btn_take_filename_from_name);
	gtk_box_pack_start (GTK_BOX (hbox1), btn_take_filename_from_name, FALSE, TRUE, 8);	

	fra_name = gtk_frame_new (_("Name"));
	gtk_widget_show (fra_name);
	gtk_box_pack_start (GTK_BOX (vbox1), fra_name, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (fra_name), 5);

	table4 = gtk_table_new (3, 4, FALSE);
	gtk_widget_show (table4);
	gtk_container_add (GTK_CONTAINER (fra_name), table4);
	gtk_container_set_border_width (GTK_CONTAINER (table4), 5);
	gtk_table_set_row_spacings (GTK_TABLE (table4), 8);
	gtk_table_set_col_spacings (GTK_TABLE (table4), 8);

	ITab->entry_name_family = gtk_entry_new ();
	gtk_widget_show (ITab->entry_name_family);
	gtk_table_attach (GTK_TABLE (table4), ITab->entry_name_family, 1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

	ITab->entry_name_given = gtk_entry_new ();
	gtk_widget_show (ITab->entry_name_given);
	gtk_table_attach (GTK_TABLE (table4), ITab->entry_name_given, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

	ITab->entry_name_middle = gtk_entry_new ();
	gtk_widget_show (ITab->entry_name_middle);
	gtk_table_attach (GTK_TABLE (table4), ITab->entry_name_middle, 3, 4, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

	label7 = gtk_label_new (_("Prefix:"));
	gtk_widget_show (label7);
	gtk_table_attach (GTK_TABLE (table4), label7, 0, 1, 2, 3,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
	gtk_label_set_justify (GTK_LABEL (label7), GTK_JUSTIFY_RIGHT);
	gtk_misc_set_alignment (GTK_MISC (label7), 1, 0.5);

	label6 = gtk_label_new (_("Family:"));
	gtk_widget_show (label6);
	gtk_table_attach (GTK_TABLE (table4), label6, 0, 1, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
	gtk_label_set_justify (GTK_LABEL (label6), GTK_JUSTIFY_RIGHT);
	gtk_misc_set_alignment (GTK_MISC (label6), 1, 0.5);

	hbox5 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox5);
	gtk_table_attach (GTK_TABLE (table4), hbox5, 1, 2, 2, 3,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);

	ITab->entry_name_prefix = gtk_entry_new ();
	gtk_widget_show (ITab->entry_name_prefix);
	gtk_box_pack_start (GTK_BOX (hbox5), ITab->entry_name_prefix, FALSE, FALSE, 0);
	gtk_widget_set_usize (ITab->entry_name_prefix, 60, -2);

	hbox6 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox6);
	gtk_table_attach (GTK_TABLE (table4), hbox6, 3, 4, 2, 3,
                    (GtkAttachOptions) (GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);

	ITab->entry_name_suffix = gtk_entry_new ();
	gtk_widget_show (ITab->entry_name_suffix);
	gtk_box_pack_start (GTK_BOX (hbox6), ITab->entry_name_suffix, FALSE, TRUE, 0);
	gtk_widget_set_usize (ITab->entry_name_suffix, 60, -2);

	label5 = gtk_label_new (_("Given:"));
	gtk_widget_show (label5);
	gtk_table_attach (GTK_TABLE (table4), label5, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
	gtk_label_set_justify (GTK_LABEL (label5), GTK_JUSTIFY_RIGHT);
	gtk_misc_set_alignment (GTK_MISC (label5), 1, 0.5);

	label12 = gtk_label_new (_("Middle:"));
	gtk_widget_show (label12);
	gtk_table_attach (GTK_TABLE (table4), label12, 2, 3, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
	gtk_label_set_justify (GTK_LABEL (label12), GTK_JUSTIFY_RIGHT);
	gtk_misc_set_alignment (GTK_MISC (label12), 1, 0.5);

	label13 = gtk_label_new (_("Suffix:"));
	gtk_widget_show (label13);
	gtk_table_attach (GTK_TABLE (table4), label13, 2, 3, 2, 3,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
	gtk_label_set_justify (GTK_LABEL (label13), GTK_JUSTIFY_RIGHT);
	gtk_misc_set_alignment (GTK_MISC (label13), 1, 0.5);

	hbox_cats = gtk_hbox_new (FALSE, 5);
	gtk_widget_show (hbox_cats);
	gtk_box_pack_start (GTK_BOX (vbox1), hbox_cats, FALSE, TRUE, 5);
	
	lbl_cats = gtk_label_new (_("Categories:"));
	gtk_widget_show (lbl_cats);
	gtk_label_set_justify (GTK_LABEL (lbl_cats), GTK_JUSTIFY_RIGHT);
	gtk_misc_set_alignment (GTK_MISC (lbl_cats), 1, 0.5);
	gtk_misc_set_padding (GTK_MISC (lbl_cats), 10, 0);
	gtk_box_pack_start (GTK_BOX (hbox_cats), lbl_cats, FALSE, TRUE, 0);
	
	ITab->entry_cats = gtk_entry_new ();
	gtk_widget_show (ITab->entry_cats);
	gtk_box_pack_start (GTK_BOX (hbox_cats), ITab->entry_cats, TRUE, TRUE, 0);
	
	ITab->btn_cats = gtk_button_new_with_label (_("Edit Categories"));
	gtk_widget_show (ITab->btn_cats);
	gtk_box_pack_start (GTK_BOX (hbox_cats), ITab->btn_cats, FALSE, TRUE, 8);
		
	hbox4 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox4);
	gtk_box_pack_start (GTK_BOX (vbox1), hbox4, FALSE, TRUE, 0);

	fra_org = gtk_frame_new (_("Organization"));
	gtk_widget_show (fra_org);
	gtk_box_pack_start (GTK_BOX (hbox4), fra_org, TRUE, TRUE, 0);
	gtk_widget_set_usize (fra_org, 242, -2);
	gtk_container_set_border_width (GTK_CONTAINER (fra_org), 5);

	table2 = gtk_table_new (2, 2, FALSE);
	gtk_widget_show (table2);
	gtk_container_add (GTK_CONTAINER (fra_org), table2);
	gtk_container_set_border_width (GTK_CONTAINER (table2), 5);
	gtk_table_set_row_spacings (GTK_TABLE (table2), 8);
	gtk_table_set_col_spacings (GTK_TABLE (table2), 8);

	ITab->entry_org_title = gtk_entry_new ();
	gtk_widget_show (ITab->entry_org_title);
	gtk_table_attach (GTK_TABLE (table2), ITab->entry_org_title, 1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

	ITab->entry_org_name = gtk_entry_new ();
	gtk_widget_show (ITab->entry_org_name);
	gtk_table_attach (GTK_TABLE (table2), ITab->entry_org_name, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

	label10 = gtk_label_new (_("Name:"));
	gtk_widget_show (label10);
	gtk_table_attach (GTK_TABLE (table2), label10, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
	gtk_label_set_justify (GTK_LABEL (label10), GTK_JUSTIFY_RIGHT);
	gtk_misc_set_alignment (GTK_MISC (label10), 1, 0.5);

	label11 = gtk_label_new (_("Title:"));
	gtk_widget_show (label11);
	gtk_table_attach (GTK_TABLE (table2), label11, 0, 1, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
	gtk_label_set_justify (GTK_LABEL (label11), GTK_JUSTIFY_RIGHT);
	gtk_misc_set_alignment (GTK_MISC (label11), 1, 0.5);

	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox2);
	gtk_box_pack_start (GTK_BOX (hbox4), vbox2, TRUE, TRUE, 0);

	hbox8 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox8);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox8, FALSE, FALSE, 10);

	label14 = gtk_label_new (_("Birthdate:"));
	gtk_widget_show (label14);
	gtk_box_pack_start (GTK_BOX (hbox8), label14, FALSE, TRUE, 0);
	gtk_label_set_justify (GTK_LABEL (label14), GTK_JUSTIFY_RIGHT);
	gtk_misc_set_alignment (GTK_MISC (label14), 1, 0.5);

	ITab->dateedit_birth = gnome_date_edit_new ((time_t) 0, FALSE, FALSE);
	gtk_widget_show (ITab->dateedit_birth);
	gtk_box_pack_start (GTK_BOX (hbox8), ITab->dateedit_birth, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (ITab->dateedit_birth), 5);

	lbl_ident = gtk_label_new (_("Identity"));
	gtk_widget_show (lbl_ident);
	
	gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 0), lbl_ident);
	gtk_notebook_set_menu_label_text (GTK_NOTEBOOK (notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 0),_("Identity"));
	
	gtk_signal_connect (GTK_OBJECT (ITab->dateedit_birth), "date_changed",
					GTK_SIGNAL_FUNC (contact_dialog_dateedit_birth_changed), ITab);
	gtk_signal_connect (GTK_OBJECT (btn_take_filename_from_name), "clicked",
				GTK_SIGNAL_FUNC (contact_dialog_take_filename_from_name), ITab);
	
gtk_widget_set_sensitive (ITab->btn_cats, FALSE);
}

/*==================================Address Tab=============================================*/
static void
ATab_address_changed_call			(GtkWidget			*widget,
						LibGefaxContactDialog		*CDiag)
{
	gchar *text;
	gint total_len = 0;
	
	text = gtk_entry_get_text (GTK_ENTRY (CDiag->ATab->entry_addr_postoffice));
	total_len = strlen (text);
	text = gtk_entry_get_text (GTK_ENTRY (CDiag->ATab->entry_addr_extended));
	total_len += strlen (text);
	text = gtk_entry_get_text (GTK_ENTRY (CDiag->ATab->entry_addr_street));
	total_len += strlen (text);
	text = gtk_entry_get_text (GTK_ENTRY (CDiag->ATab->entry_addr_city));
	total_len += strlen (text);
	text = gtk_entry_get_text (GTK_ENTRY (CDiag->ATab->entry_addr_region));
	total_len += strlen (text);
	text = gtk_entry_get_text (GTK_ENTRY (CDiag->ATab->entry_addr_postcode));
	total_len += strlen (text);
	text = gtk_entry_get_text (GTK_ENTRY (CDiag->ATab->entry_addr_country));
	total_len += strlen (text);
	
	if (total_len > 0){
		if (CDiag->Priv->address_key == NO_KEY) {
			gtk_widget_set_sensitive (CDiag->ATab->btn_add, TRUE);
			gtk_widget_set_sensitive (CDiag->ATab->btn_modify, FALSE);
			gtk_widget_set_sensitive (CDiag->ATab->btn_remove, FALSE);
		} else {
			gtk_widget_set_sensitive (CDiag->ATab->btn_add, TRUE);
			gtk_widget_set_sensitive (CDiag->ATab->btn_modify, TRUE);
			gtk_widget_set_sensitive (CDiag->ATab->btn_remove, FALSE);
		}
	} else {
		gtk_widget_set_sensitive (CDiag->ATab->btn_add, FALSE);
		gtk_widget_set_sensitive (CDiag->ATab->btn_modify, FALSE);
		gtk_widget_set_sensitive (CDiag->ATab->btn_remove, FALSE);
	}
}

static void
contact_dialog_create_A_TAB			(A_Tab				*ATab,
						GtkWidget			*notebook)
{
	const gchar *ADDRESS_TYPES[] = {
		N_("Home"), N_("Work"), N_("Postal Box"),
		N_("Parcel"), N_("Domestic"), N_("International")
	};
	gint row, col, i;
	
	GtkWidget *hbox9;
	GtkWidget *fra_address_info;
	GtkWidget *table5;
	GtkWidget *fra_type;
	GtkWidget *table_type;
	GtkWidget *label21;
	GtkWidget *label20;
	GtkWidget *fra_addr_list;
	GtkWidget *label19;
	GtkWidget *label18;
	GtkWidget *label17;
	GtkWidget *label16;
	GtkWidget *label15;
	GtkWidget *hbox10;
	GtkWidget *fra_toolbar;
	GtkWidget *vbox3;
	GtkWidget *lbl_space1, *lbl_space2;
	GtkWidget *toolbar;
	GtkWidget *tmp_toolbar_icon;
	GtkWidget *lbl_addr;
	
	hbox9 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox9);
	gtk_container_add (GTK_CONTAINER (notebook), hbox9);
	gtk_container_set_border_width (GTK_CONTAINER (hbox9), 5);

	fra_address_info = gtk_frame_new (_("Address Information"));
	gtk_widget_show (fra_address_info);
	gtk_container_set_border_width (GTK_CONTAINER (fra_address_info), 5);
	gtk_box_pack_start (GTK_BOX (hbox9), fra_address_info, TRUE, TRUE, 0);

	table5 = gtk_table_new (9, 2, FALSE);
	gtk_widget_show (table5);
	gtk_container_add (GTK_CONTAINER (fra_address_info), table5);
	gtk_container_set_border_width (GTK_CONTAINER (table5), 8);
	gtk_table_set_row_spacings (GTK_TABLE (table5), 1);
	gtk_table_set_col_spacings (GTK_TABLE (table5), 10);

	fra_type = gtk_frame_new (_("Type:"));
	gtk_widget_show (fra_type);
	gtk_table_attach (GTK_TABLE (table5), fra_type, 0, 2, 7, 9,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_SHRINK | GTK_FILL), 0, 0);

	table_type = gtk_table_new (7, 2, FALSE);
	gtk_widget_show (table_type);
	gtk_container_add (GTK_CONTAINER (fra_type), table_type);
	
	row = 0;
	col = 0;
	for (i = 0; i < ADDRESS_TYPES_LENGTH; i++) {
		ATab->types[i] = gtk_check_button_new_with_label (_(ADDRESS_TYPES[i]));
		gtk_widget_show (ATab->types[i]);
		gtk_table_attach (GTK_TABLE (table_type), ATab->types[i], col, col + 1, row, row + 1,
				(GtkAttachOptions) (GTK_FILL),
				(GtkAttachOptions) (0), 0, 0);
		if (row == 2) {
			row = 0;
			col = 1;
		} else {
			++row;
		}
	}	

	label21 = gtk_label_new (_("Country:"));
	gtk_widget_show (label21);
	gtk_table_attach (GTK_TABLE (table5), label21, 0, 1, 6, 7,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
	gtk_label_set_justify (GTK_LABEL (label21), GTK_JUSTIFY_RIGHT);
	gtk_misc_set_alignment (GTK_MISC (label21), 1, 0.5);

	label20 = gtk_label_new (_("Post Code:"));
	gtk_widget_show (label20);
	gtk_table_attach (GTK_TABLE (table5), label20, 0, 1, 5, 6,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
	gtk_label_set_justify (GTK_LABEL (label20), GTK_JUSTIFY_RIGHT);
	gtk_misc_set_alignment (GTK_MISC (label20), 1, 0.5);

	label19 = gtk_label_new (_("Region:"));
	gtk_widget_show (label19);
	gtk_table_attach (GTK_TABLE (table5), label19, 0, 1, 4, 5,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
	gtk_label_set_justify (GTK_LABEL (label19), GTK_JUSTIFY_RIGHT);
	gtk_misc_set_alignment (GTK_MISC (label19), 1, 0.5);

	label18 = gtk_label_new (_("City:"));
	gtk_widget_show (label18);
	gtk_table_attach (GTK_TABLE (table5), label18, 0, 1, 3, 4,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
	gtk_label_set_justify (GTK_LABEL (label18), GTK_JUSTIFY_RIGHT);
	gtk_misc_set_alignment (GTK_MISC (label18), 1, 0.5);

	label17 = gtk_label_new (_("Street:"));
	gtk_widget_show (label17);
	gtk_table_attach (GTK_TABLE (table5), label17, 0, 1, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
	gtk_label_set_justify (GTK_LABEL (label17), GTK_JUSTIFY_RIGHT);
	gtk_misc_set_alignment (GTK_MISC (label17), 1, 0.5);

	label16 = gtk_label_new (_("Extended:"));
	gtk_widget_show (label16);
	gtk_table_attach (GTK_TABLE (table5), label16, 0, 1, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
	gtk_label_set_justify (GTK_LABEL (label16), GTK_JUSTIFY_RIGHT);
	gtk_misc_set_alignment (GTK_MISC (label16), 1, 0.5);

	label15 = gtk_label_new (_("Post Office:"));
	gtk_widget_show (label15);
	gtk_table_attach (GTK_TABLE (table5), label15, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
	gtk_label_set_justify (GTK_LABEL (label15), GTK_JUSTIFY_RIGHT);
	gtk_misc_set_alignment (GTK_MISC (label15), 1, 0.5);

	ATab->entry_addr_country = gtk_entry_new ();
	gtk_widget_show (ATab->entry_addr_country);
	gtk_table_attach (GTK_TABLE (table5), ATab->entry_addr_country, 1, 2, 6, 7,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

	ATab->entry_addr_postoffice = gtk_entry_new ();
	gtk_widget_show (ATab->entry_addr_postoffice);
	gtk_table_attach (GTK_TABLE (table5), ATab->entry_addr_postoffice, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

	ATab->entry_addr_extended = gtk_entry_new ();
	gtk_widget_show (ATab->entry_addr_extended);
	gtk_table_attach (GTK_TABLE (table5), ATab->entry_addr_extended, 1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

	ATab->entry_addr_street = gtk_entry_new ();
	gtk_widget_show (ATab->entry_addr_street);
	gtk_table_attach (GTK_TABLE (table5), ATab->entry_addr_street, 1, 2, 2, 3,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

	ATab->entry_addr_city = gtk_entry_new ();
	gtk_widget_show (ATab->entry_addr_city);
	gtk_table_attach (GTK_TABLE (table5), ATab->entry_addr_city, 1, 2, 3, 4,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

	ATab->entry_addr_region = gtk_entry_new ();
	gtk_widget_show (ATab->entry_addr_region);
	gtk_table_attach (GTK_TABLE (table5), ATab->entry_addr_region, 1, 2, 4, 5,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

	ATab->entry_addr_postcode = gtk_entry_new ();
	gtk_widget_show (ATab->entry_addr_postcode);
	gtk_table_attach (GTK_TABLE (table5), ATab->entry_addr_postcode, 1, 2, 5, 6,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

	hbox10 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox10);
	gtk_box_pack_start (GTK_BOX (hbox9), hbox10, TRUE, TRUE, 0);

	fra_toolbar = gtk_frame_new (NULL);
	gtk_widget_show (fra_toolbar);
	gtk_container_set_border_width (GTK_CONTAINER (fra_toolbar), 5);
	gtk_frame_set_shadow_type (GTK_FRAME (fra_toolbar), GTK_SHADOW_IN);
	gtk_box_pack_start (GTK_BOX (hbox10), fra_toolbar, FALSE, FALSE, 0);
	
	vbox3 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox3);
	gtk_container_add (GTK_CONTAINER (fra_toolbar), vbox3);
	
	lbl_space1 = gtk_label_new ("");
	gtk_widget_show (lbl_space1);
	gtk_box_pack_start (GTK_BOX (vbox3), lbl_space1, FALSE, FALSE, 0);
	toolbar = gtk_toolbar_new (GTK_ORIENTATION_VERTICAL, GTK_TOOLBAR_BOTH);
	gtk_widget_show (toolbar);
	gtk_box_pack_start (GTK_BOX (vbox3), toolbar, TRUE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (toolbar), 10);
	gtk_toolbar_set_space_size (GTK_TOOLBAR (toolbar), 6);
	gtk_toolbar_set_button_relief (GTK_TOOLBAR (toolbar), GTK_RELIEF_NONE);

	tmp_toolbar_icon = gnome_stock_pixmap_widget (notebook, GNOME_STOCK_PIXMAP_ADD);
	ATab->btn_add = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
                                GTK_TOOLBAR_CHILD_BUTTON,
                                NULL,
                                _("Add"),
                                NULL, NULL,
                                tmp_toolbar_icon, NULL, NULL);
	gtk_widget_show (ATab->btn_add);

	gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));

	tmp_toolbar_icon = gnome_stock_pixmap_widget (notebook, GNOME_STOCK_PIXMAP_PREFERENCES);
	ATab->btn_modify = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
                                GTK_TOOLBAR_CHILD_BUTTON,
                                NULL,
                                _("Modify"),
                                NULL, NULL,
                                tmp_toolbar_icon, NULL, NULL);
	gtk_widget_show (ATab->btn_modify);

	gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));

	tmp_toolbar_icon = gnome_stock_pixmap_widget (notebook, GNOME_STOCK_PIXMAP_REMOVE);
	ATab->btn_remove = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
                                GTK_TOOLBAR_CHILD_BUTTON,
                                NULL,
                                _("Remove"),
                                NULL, NULL,
                                tmp_toolbar_icon, NULL, NULL);
	gtk_widget_show (ATab->btn_remove);

	lbl_space2 = gtk_label_new ("");
	gtk_widget_show (lbl_space2);
	gtk_box_pack_start (GTK_BOX (vbox3), lbl_space2, FALSE, FALSE, 0);
	
	fra_addr_list = gtk_frame_new (_("Address List"));
	gtk_widget_show (fra_addr_list);
	gtk_box_pack_start (GTK_BOX (hbox10), fra_addr_list, TRUE, TRUE, 0);
	gtk_widget_set_usize (fra_addr_list, 145, -2);
	gtk_container_set_border_width (GTK_CONTAINER (fra_addr_list), 5);
	gtk_frame_set_label_align (GTK_FRAME (fra_addr_list), 0.5, 0.5);

	ATab->clist_addr = gtk_clist_new (2);
	gtk_widget_show (ATab->clist_addr);
	gtk_container_add (GTK_CONTAINER (fra_addr_list), ATab->clist_addr);
	gtk_clist_set_column_width (GTK_CLIST (ATab->clist_addr), 0, 80);
	gtk_clist_set_column_visibility (GTK_CLIST (ATab->clist_addr), 1, FALSE);
	gtk_clist_column_titles_hide (GTK_CLIST (ATab->clist_addr));
	gtk_clist_set_shadow_type (GTK_CLIST (ATab->clist_addr), GTK_SHADOW_IN);

	lbl_addr = gtk_label_new (_("Address"));
	gtk_widget_show (lbl_addr);
	
	gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 1), lbl_addr);
	gtk_notebook_set_menu_label_text (GTK_NOTEBOOK (notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 1),_("Address"));
	
	gtk_widget_set_sensitive (ATab->btn_add, FALSE);
	gtk_widget_set_sensitive (ATab->btn_modify, FALSE);
	gtk_widget_set_sensitive (ATab->btn_remove, FALSE);
}

/*==================================Phone Tab=============================================*/
static void
PTab_phone_changed_call				(GtkWidget			*widget,
						LibGefaxContactDialog		*CDiag)
{
	gchar *text;

	text = gtk_entry_get_text (GTK_ENTRY (CDiag->PTab->entry_phone_number));
	
	if (strlen (text) > 0){
		if (CDiag->Priv->phone_key == NO_KEY) {
			gtk_widget_set_sensitive (CDiag->PTab->btn_add, TRUE);
			gtk_widget_set_sensitive (CDiag->PTab->btn_modify, FALSE);
			gtk_widget_set_sensitive (CDiag->PTab->btn_remove, FALSE);
		} else {
			gtk_widget_set_sensitive (CDiag->PTab->btn_add, TRUE);
			gtk_widget_set_sensitive (CDiag->PTab->btn_modify, TRUE);
			gtk_widget_set_sensitive (CDiag->PTab->btn_remove, FALSE);
		}
	} else {
		gtk_widget_set_sensitive (CDiag->PTab->btn_add, FALSE);
		gtk_widget_set_sensitive (CDiag->PTab->btn_modify, FALSE);
		gtk_widget_set_sensitive (CDiag->PTab->btn_remove, FALSE);
	}
}

static void
contact_dialog_create_P_TAB			(P_Tab				*PTab,
						GtkWidget			*notebook)
{
	const gchar *PHONE_TYPES[] =	{
		N_("Preferred"), N_("Work"), N_("Home"), N_("Voice"),
		N_("Fax"), N_("Message Recorder"), N_("Cellular"),
		N_("Pager"), N_("Bulletin Board"), N_("Modem"),
		N_("Car"), N_("ISDN"), N_("Video")
	};
	gint row, col, i;
	
	GtkWidget *hbox_phone;
	GtkWidget *fra_phone_info;
	GtkWidget *vbox_phone_info;
	GtkWidget *hbox_phone_num;
	GtkWidget *lbl_phone_num;
	GtkWidget *fra_tel_type;
	GtkWidget *table_phone_type;
	GtkWidget *fra_toolbar;
	GtkWidget *vbox_toolbar;
	GtkWidget *lbl_space1;
	GtkWidget *toolbar_phone;
	GtkWidget *tmp_toolbar_icon;
	GtkWidget *lbl_space2;
	GtkWidget *fra_phone_list;
 	GtkWidget *lbl_phone;
	
	hbox_phone = gtk_hbox_new (FALSE, 0);
  
	gtk_widget_show (hbox_phone);
	gtk_container_set_border_width (GTK_CONTAINER (hbox_phone), 5);
	gtk_container_add (GTK_CONTAINER (notebook), hbox_phone);

	fra_phone_info = gtk_frame_new (_("Phone Information"));
	gtk_widget_show (fra_phone_info);
	gtk_box_pack_start (GTK_BOX (hbox_phone), fra_phone_info, FALSE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (fra_phone_info), 5);

	vbox_phone_info = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox_phone_info);
	gtk_container_add (GTK_CONTAINER (fra_phone_info), vbox_phone_info);

	hbox_phone_num = gtk_hbox_new (FALSE, 5);
	gtk_widget_show (hbox_phone_num);
	gtk_box_pack_start (GTK_BOX (vbox_phone_info), hbox_phone_num, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (hbox_phone_num), 5);

	lbl_phone_num = gtk_label_new (_("Number:"));
	gtk_widget_show (lbl_phone_num);
	gtk_box_pack_start (GTK_BOX (hbox_phone_num), lbl_phone_num, FALSE, FALSE, 0);

	PTab->entry_phone_number = gtk_entry_new ();
	gtk_widget_show (PTab->entry_phone_number);
	gtk_box_pack_start (GTK_BOX (hbox_phone_num), PTab->entry_phone_number, TRUE, TRUE, 0);

	fra_tel_type = gtk_frame_new (_("Type:"));
	gtk_widget_show (fra_tel_type);
	gtk_box_pack_start (GTK_BOX (vbox_phone_info), fra_tel_type, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (fra_tel_type), 5);

	table_phone_type = gtk_table_new (7, 2, FALSE);
	gtk_widget_show (table_phone_type);
	gtk_container_add (GTK_CONTAINER (fra_tel_type), table_phone_type);
	
	row = 0;
	col = 0;
	for (i = 0; i < PHONE_TYPES_LENGTH; i++) {
		PTab->types[i] = gtk_check_button_new_with_label (_(PHONE_TYPES[i]));
		gtk_widget_show (PTab->types[i]);
		gtk_table_attach (GTK_TABLE (table_phone_type), PTab->types[i], col, col + 1, row, row + 1,
				(GtkAttachOptions) (GTK_FILL),
				(GtkAttachOptions) (0), 0, 0);
		if (row == 6) {
			row = 0;
			col = 1;
		} else {
			++row;
		}
	}

	fra_toolbar = gtk_frame_new (NULL);
	gtk_widget_show (fra_toolbar);
	gtk_container_set_border_width (GTK_CONTAINER (fra_toolbar), 5);
	gtk_box_pack_start (GTK_BOX (hbox_phone), fra_toolbar, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type (GTK_FRAME (fra_toolbar), GTK_SHADOW_IN);

	vbox_toolbar = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox_toolbar);
	gtk_container_add (GTK_CONTAINER (fra_toolbar), vbox_toolbar);

	lbl_space1 = gtk_label_new ("");
	gtk_widget_show (lbl_space1);
	gtk_box_pack_start (GTK_BOX (vbox_toolbar), lbl_space1, FALSE, FALSE, 0);

	toolbar_phone = gtk_toolbar_new (GTK_ORIENTATION_VERTICAL, GTK_TOOLBAR_BOTH);
	gtk_widget_show (toolbar_phone);
	gtk_box_pack_start (GTK_BOX (vbox_toolbar), toolbar_phone, TRUE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (toolbar_phone), 10);
	gtk_toolbar_set_space_size (GTK_TOOLBAR (toolbar_phone), 6);
	gtk_toolbar_set_button_relief (GTK_TOOLBAR (toolbar_phone), GTK_RELIEF_NONE);

	tmp_toolbar_icon = gnome_stock_pixmap_widget (notebook, GNOME_STOCK_PIXMAP_ADD);
	PTab->btn_add = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar_phone),
                                GTK_TOOLBAR_CHILD_BUTTON,
                                NULL,
                                _("Add"),
                                NULL, NULL,
                                tmp_toolbar_icon, NULL, NULL);
	gtk_widget_show (PTab->btn_add);

	gtk_toolbar_append_space (GTK_TOOLBAR (toolbar_phone));

	tmp_toolbar_icon = gnome_stock_pixmap_widget (notebook, GNOME_STOCK_PIXMAP_PREFERENCES);
	PTab->btn_modify = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar_phone),
                                GTK_TOOLBAR_CHILD_BUTTON,
                                NULL,
                                _("Modify"),
                                NULL, NULL,
                                tmp_toolbar_icon, NULL, NULL);
	gtk_widget_show (PTab->btn_modify);

	gtk_toolbar_append_space (GTK_TOOLBAR (toolbar_phone));

	tmp_toolbar_icon = gnome_stock_pixmap_widget (notebook, GNOME_STOCK_PIXMAP_REMOVE);
	PTab->btn_remove = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar_phone),
                                GTK_TOOLBAR_CHILD_BUTTON,
                                NULL,
                                _("Remove"),
                                NULL, NULL,
                                tmp_toolbar_icon, NULL, NULL);
	gtk_widget_show (PTab->btn_remove);

	lbl_space2 = gtk_label_new ("");
	gtk_widget_show (lbl_space2);
	gtk_box_pack_start (GTK_BOX (vbox_toolbar), lbl_space2, FALSE, FALSE, 0);

	fra_phone_list = gtk_frame_new (_("Phone List"));
	gtk_widget_show (fra_phone_list);
	gtk_box_pack_start (GTK_BOX (hbox_phone), fra_phone_list, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (fra_phone_list), 5);
	gtk_frame_set_label_align (GTK_FRAME (fra_phone_list), 0.5, 0.5);

	PTab->clist_phone = gtk_clist_new (2);
	gtk_widget_show (PTab->clist_phone);
	gtk_container_add (GTK_CONTAINER (fra_phone_list), PTab->clist_phone);
	gtk_clist_set_column_width (GTK_CLIST (PTab->clist_phone), 0, 80);
	gtk_clist_set_column_visibility (GTK_CLIST (PTab->clist_phone), 1, FALSE);
	gtk_clist_column_titles_hide (GTK_CLIST (PTab->clist_phone));


	lbl_phone = gtk_label_new (_("Phone"));
	gtk_widget_show (lbl_phone);
	
	gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 2), lbl_phone);
	gtk_notebook_set_menu_label_text (GTK_NOTEBOOK (notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 2),_("Phone"));
	
	gtk_widget_set_sensitive (PTab->btn_add, FALSE);
	gtk_widget_set_sensitive (PTab->btn_modify, FALSE);
	gtk_widget_set_sensitive (PTab->btn_remove, FALSE);
}

/*==================================Network Tab=============================================*/
static void
NTab_email_changed_call				(GtkWidget			*widget,
						LibGefaxContactDialog		*CDiag)
{
	gchar *text;

	text = gtk_entry_get_text (GTK_ENTRY (CDiag->NTab->entry_email));
	
	if (strlen (text) > 0){
		if (CDiag->Priv->email_key == NO_KEY) {
			gtk_widget_set_sensitive (CDiag->NTab->btn_add, TRUE);
			gtk_widget_set_sensitive (CDiag->NTab->btn_modify, FALSE);
			gtk_widget_set_sensitive (CDiag->NTab->btn_remove, FALSE);
		} else {
			gtk_widget_set_sensitive (CDiag->NTab->btn_add, TRUE);
			gtk_widget_set_sensitive (CDiag->NTab->btn_modify, TRUE);
			gtk_widget_set_sensitive (CDiag->NTab->btn_remove, FALSE);
		}
	} else {
		gtk_widget_set_sensitive (CDiag->NTab->btn_add, FALSE);
		gtk_widget_set_sensitive (CDiag->NTab->btn_modify, FALSE);
		gtk_widget_set_sensitive (CDiag->NTab->btn_remove, FALSE);
	}
}

static void
contact_dialog_create_N_TAB			(N_Tab				*NTab,
						GtkWidget			*notebook)
{
	const gchar *EMAIL_TYPES[] = {
		N_("America On-Line"), N_("Apple Link"), N_("AT&T"), N_("CIS"),
		N_("e-World"), N_("Internet"), N_("IBM"), N_("MCI"),
		N_("Power Share"), N_("Prodigy"), N_("TLX"), N_("X400")
	};
	gint row, col, i;
	
	GtkWidget *lbl_network;
	GtkWidget *vbox_network;
	GtkWidget *hbox_url;
	GtkWidget *lbl_url;
	GtkWidget *hbox_network;
	GtkWidget *fra_phone_info;
	GtkWidget *vbox_phone_info;
	GtkWidget *hbox_phone_num;
	GtkWidget *lbl_phone_num;
	GtkWidget *fra_tel_type;
	GSList *type_group = NULL;
	GtkWidget *table_phone_type;
	GtkWidget *fra_toolbar;
	GtkWidget *vbox_toolbar;
	GtkWidget *lbl_space1;
	GtkWidget *toolbar_phone;
	GtkWidget *tmp_toolbar_icon;
	GtkWidget *lbl_space2;
	GtkWidget *fra_phone_list;
	
	vbox_network = gtk_vbox_new (FALSE, 0);
  	gtk_widget_show (vbox_network);
  	gtk_container_set_border_width (GTK_CONTAINER (vbox_network), 5);
	gtk_container_add (GTK_CONTAINER (notebook), vbox_network);
	
	hbox_url = gtk_hbox_new (FALSE, 5);
  	gtk_widget_show (hbox_url);
  	gtk_box_pack_start (GTK_BOX (vbox_network), hbox_url, FALSE, FALSE, 5);
  	gtk_container_set_border_width (GTK_CONTAINER (hbox_url), 5);
  	
  	lbl_url = gtk_label_new (_("Web Site:"));
	gtk_widget_show (lbl_url);
	gtk_box_pack_start (GTK_BOX (hbox_url), lbl_url, FALSE, FALSE, 0);
	
	NTab->entry_url = gtk_entry_new ();
	gtk_widget_show (NTab->entry_url);
	gtk_box_pack_start (GTK_BOX (hbox_url), NTab->entry_url, TRUE, TRUE, 0);
	
	hbox_network = gtk_hbox_new (FALSE, 0);
  	gtk_widget_show (hbox_network);
  	gtk_box_pack_start (GTK_BOX (vbox_network), hbox_network, TRUE, TRUE, 0);
	
	fra_phone_info = gtk_frame_new (_("Email Information"));
	gtk_widget_show (fra_phone_info);
	gtk_box_pack_start (GTK_BOX (hbox_network), fra_phone_info, FALSE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (fra_phone_info), 5);

	vbox_phone_info = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox_phone_info);
	gtk_container_add (GTK_CONTAINER (fra_phone_info), vbox_phone_info);

	hbox_phone_num = gtk_hbox_new (FALSE, 5);
	gtk_widget_show (hbox_phone_num);
	gtk_box_pack_start (GTK_BOX (vbox_phone_info), hbox_phone_num, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (hbox_phone_num), 5);

	lbl_phone_num = gtk_label_new (_("Address:"));
	gtk_widget_show (lbl_phone_num);
	gtk_box_pack_start (GTK_BOX (hbox_phone_num), lbl_phone_num, FALSE, FALSE, 0);

	NTab->entry_email = gtk_entry_new ();
	gtk_widget_show (NTab->entry_email);
	gtk_box_pack_start (GTK_BOX (hbox_phone_num), NTab->entry_email, TRUE, TRUE, 0);

	fra_tel_type = gtk_frame_new (_("Type:"));
	gtk_widget_show (fra_tel_type);
	gtk_box_pack_start (GTK_BOX (vbox_phone_info), fra_tel_type, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (fra_tel_type), 5);

	table_phone_type = gtk_table_new (6, 2, FALSE);
	gtk_widget_show (table_phone_type);
	gtk_container_add (GTK_CONTAINER (fra_tel_type), table_phone_type);
	
	row = 0;
	col = 0;
	for (i = 0; i < EMAIL_TYPES_LENGTH; i++) {
		NTab->types[i] = gtk_radio_button_new_with_label (type_group, _(EMAIL_TYPES[i]));
		type_group = gtk_radio_button_group (GTK_RADIO_BUTTON (NTab->types[i]));
		gtk_widget_show (NTab->types[i]);
		gtk_table_attach (GTK_TABLE (table_phone_type), NTab->types[i], col, col + 1, row, row + 1,
				(GtkAttachOptions) (GTK_FILL),
				(GtkAttachOptions) (0), 0, 0);
		if (row == 5) {
			row = 0;
			col = 1;
		} else {
			++row;
		}
	}

	fra_toolbar = gtk_frame_new (NULL);
	gtk_widget_show (fra_toolbar);
	gtk_container_set_border_width (GTK_CONTAINER (fra_toolbar), 5);
	gtk_box_pack_start (GTK_BOX (hbox_network), fra_toolbar, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type (GTK_FRAME (fra_toolbar), GTK_SHADOW_IN);

	vbox_toolbar = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox_toolbar);
	gtk_container_add (GTK_CONTAINER (fra_toolbar), vbox_toolbar);

	lbl_space1 = gtk_label_new ("");
	gtk_widget_show (lbl_space1);
	gtk_box_pack_start (GTK_BOX (vbox_toolbar), lbl_space1, FALSE, FALSE, 0);

	toolbar_phone = gtk_toolbar_new (GTK_ORIENTATION_VERTICAL, GTK_TOOLBAR_BOTH);
	gtk_widget_show (toolbar_phone);
	gtk_box_pack_start (GTK_BOX (vbox_toolbar), toolbar_phone, TRUE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (toolbar_phone), 10);
	gtk_toolbar_set_space_size (GTK_TOOLBAR (toolbar_phone), 6);
	gtk_toolbar_set_button_relief (GTK_TOOLBAR (toolbar_phone), GTK_RELIEF_NONE);

	tmp_toolbar_icon = gnome_stock_pixmap_widget (notebook, GNOME_STOCK_PIXMAP_ADD);
	NTab->btn_add = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar_phone),
                                GTK_TOOLBAR_CHILD_BUTTON,
                                NULL,
                                _("Add"),
                                NULL, NULL,
                                tmp_toolbar_icon, NULL, NULL);
	gtk_widget_show (NTab->btn_add);

	gtk_toolbar_append_space (GTK_TOOLBAR (toolbar_phone));

	tmp_toolbar_icon = gnome_stock_pixmap_widget (notebook, GNOME_STOCK_PIXMAP_PREFERENCES);
	NTab->btn_modify = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar_phone),
                                GTK_TOOLBAR_CHILD_BUTTON,
                                NULL,
                                _("Modify"),
                                NULL, NULL,
                                tmp_toolbar_icon, NULL, NULL);
	gtk_widget_show (NTab->btn_modify);

	gtk_toolbar_append_space (GTK_TOOLBAR (toolbar_phone));

	tmp_toolbar_icon = gnome_stock_pixmap_widget (notebook, GNOME_STOCK_PIXMAP_REMOVE);
	NTab->btn_remove = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar_phone),
                                GTK_TOOLBAR_CHILD_BUTTON,
                                NULL,
                                _("Remove"),
                                NULL, NULL,
                                tmp_toolbar_icon, NULL, NULL);
	gtk_widget_show (NTab->btn_remove);

	lbl_space2 = gtk_label_new ("");
	gtk_widget_show (lbl_space2);
	gtk_box_pack_start (GTK_BOX (vbox_toolbar), lbl_space2, FALSE, FALSE, 0);

	fra_phone_list = gtk_frame_new (_("Email List"));
	gtk_widget_show (fra_phone_list);
	gtk_box_pack_start (GTK_BOX (hbox_network), fra_phone_list, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (fra_phone_list), 5);
	gtk_frame_set_label_align (GTK_FRAME (fra_phone_list), 0.5, 0.5);

	NTab->clist_email = gtk_clist_new (2);
	gtk_widget_show (NTab->clist_email);
	gtk_container_add (GTK_CONTAINER (fra_phone_list), NTab->clist_email);
	gtk_clist_set_column_width (GTK_CLIST (NTab->clist_email), 0, 80);
	gtk_clist_set_column_visibility (GTK_CLIST (NTab->clist_email), 1, FALSE);
	gtk_clist_column_titles_hide (GTK_CLIST (NTab->clist_email));



	lbl_network = gtk_label_new (_("Network"));
	gtk_widget_show (lbl_network);
	
	gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 3), lbl_network);
	gtk_notebook_set_menu_label_text (GTK_NOTEBOOK (notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 3),_("Network"));
	
	gtk_widget_set_sensitive (NTab->btn_add, FALSE);
	gtk_widget_set_sensitive (NTab->btn_modify, FALSE);
	gtk_widget_set_sensitive (NTab->btn_remove, FALSE);
}

/*==================================Additional Tab===========================================*/

static void
contact_dialog_create_X_TAB			(X_Tab				*XTab,
						GtkWidget			*notebook)
{
	GtkWidget *lbl_net;
	GtkWidget *vbox1;
	GtkWidget *fra_pk;
	GtkWidget *vbox2;
	GtkWidget *hbox1;
	GSList *key_group_group = NULL;
	GtkWidget *scrolledwindow1;
	GtkWidget *fra_geo;
	GtkWidget *hbox2;
	GtkWidget *frame3;
	GtkWidget *hbox3;
	GtkObject *spinbutton1_adj;
	GtkWidget *label3;
	GtkObject *spinbutton2_adj;
	GtkWidget *label4;
	GtkWidget *frame4;
	GtkWidget *hbox4;
	GtkObject *spin_lat_adj;
	GtkWidget *label2;
	GtkObject *spin_lon_adj;
	GtkWidget *label1;
	GtkWidget *fra_comment;
	GtkWidget *scrolledwindow2;

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox1);
	gtk_container_add (GTK_CONTAINER (notebook), vbox1);
	gtk_container_set_border_width (GTK_CONTAINER (vbox1), 5);

	fra_pk = gtk_frame_new (_("Public Key"));
	gtk_widget_show (fra_pk);
	gtk_box_pack_start (GTK_BOX (vbox1), fra_pk, TRUE, TRUE, 5);

	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox2);
	gtk_container_add (GTK_CONTAINER (fra_pk), vbox2);

	hbox1 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox1);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox1, TRUE, TRUE, 0);

	XTab->rad_pgp = gtk_radio_button_new_with_label (key_group_group, _("PGP"));
	key_group_group = gtk_radio_button_group (GTK_RADIO_BUTTON (XTab->rad_pgp));
	gtk_widget_show (XTab->rad_pgp);
	gtk_box_pack_start (GTK_BOX (hbox1), XTab->rad_pgp, FALSE, TRUE, 0);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (XTab->rad_pgp), TRUE);

	XTab->rad_x509 = gtk_radio_button_new_with_label (key_group_group, _("X509"));
	key_group_group = gtk_radio_button_group (GTK_RADIO_BUTTON (XTab->rad_x509));
	gtk_widget_show (XTab->rad_x509);
	gtk_box_pack_start (GTK_BOX (hbox1), XTab->rad_x509, FALSE, TRUE, 0);

	scrolledwindow1 = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_show (scrolledwindow1);
	gtk_box_pack_start (GTK_BOX (vbox2), scrolledwindow1, TRUE, TRUE, 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow1), GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);

	XTab->text_key = gtk_text_new (NULL, NULL);
	gtk_widget_show (XTab->text_key);
	gtk_container_add (GTK_CONTAINER (scrolledwindow1), XTab->text_key);
	gtk_text_set_editable (GTK_TEXT (XTab->text_key), TRUE);

	fra_geo = gtk_frame_new (_("Geographical"));
	gtk_widget_show (fra_geo);
	gtk_box_pack_start (GTK_BOX (vbox1), fra_geo, TRUE, TRUE, 5);

	hbox2 = gtk_hbox_new (TRUE, 0);
	gtk_widget_show (hbox2);
	gtk_container_add (GTK_CONTAINER (fra_geo), hbox2);

	frame3 = gtk_frame_new (_("Time Zone"));
	gtk_widget_show (frame3);
	gtk_box_pack_start (GTK_BOX (hbox2), frame3, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (frame3), 3);

	hbox3 = gtk_hbox_new (FALSE, 6);
	gtk_widget_show (hbox3);
	gtk_container_add (GTK_CONTAINER (frame3), hbox3);
	gtk_container_set_border_width (GTK_CONTAINER (hbox3), 5);

	spinbutton1_adj = gtk_adjustment_new (0, -12, 12, 1, 1, 1);
	XTab->spin_hrs = gtk_spin_button_new (GTK_ADJUSTMENT (spinbutton1_adj), 1, 0);
	gtk_widget_show (XTab->spin_hrs);
	gtk_box_pack_start (GTK_BOX (hbox3), XTab->spin_hrs, TRUE, TRUE, 0);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (XTab->spin_hrs), TRUE);
	gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (XTab->spin_hrs), TRUE);

	label3 = gtk_label_new (_("Hrs,"));
	gtk_widget_show (label3);
	gtk_box_pack_start (GTK_BOX (hbox3), label3, FALSE, FALSE, 0);

	spinbutton2_adj = gtk_adjustment_new (0, 0, 59, 1, 1, 1);
	XTab->spin_mins = gtk_spin_button_new (GTK_ADJUSTMENT (spinbutton2_adj), 1, 0);
	gtk_widget_show (XTab->spin_mins);
	gtk_box_pack_start (GTK_BOX (hbox3), XTab->spin_mins, TRUE, TRUE, 0);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (XTab->spin_mins), TRUE);
	gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (XTab->spin_mins), TRUE);

	label4 = gtk_label_new (_("Mins."));
	gtk_widget_show (label4);
	gtk_box_pack_start (GTK_BOX (hbox3), label4, FALSE, FALSE, 0);

	frame4 = gtk_frame_new (_("Geographic Position"));
	gtk_widget_show (frame4);
	gtk_box_pack_start (GTK_BOX (hbox2), frame4, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (frame4), 3);

	hbox4 = gtk_hbox_new (FALSE, 6);
	gtk_widget_show (hbox4);
	gtk_container_add (GTK_CONTAINER (frame4), hbox4);
	gtk_container_set_border_width (GTK_CONTAINER (hbox4), 5);

	spin_lat_adj = gtk_adjustment_new (0, -90, 90, 0.01, 1, 1);
	XTab->spin_lat = gtk_spin_button_new (GTK_ADJUSTMENT (spin_lat_adj), 1, 2);
	gtk_widget_show (XTab->spin_lat);
	gtk_box_pack_start (GTK_BOX (hbox4), XTab->spin_lat, TRUE, TRUE, 0);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (XTab->spin_lat), TRUE);
 	gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (XTab->spin_lat), TRUE);

	label2 = gtk_label_new (_("Lat,"));
	gtk_widget_show (label2);
	gtk_box_pack_start (GTK_BOX (hbox4), label2, FALSE, FALSE, 0);

	spin_lon_adj = gtk_adjustment_new (0, -180, 180, 0.01, 1, 1);
	XTab->spin_lon = gtk_spin_button_new (GTK_ADJUSTMENT (spin_lon_adj), 1, 2);
	gtk_widget_show (XTab->spin_lon);
	gtk_box_pack_start (GTK_BOX (hbox4), XTab->spin_lon, TRUE, TRUE, 0);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (XTab->spin_lon), TRUE);
	gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (XTab->spin_lon), TRUE);

	label1 = gtk_label_new (_("Lon."));
	gtk_widget_show (label1);
	gtk_box_pack_start (GTK_BOX (hbox4), label1, FALSE, FALSE, 0);

	fra_comment = gtk_frame_new (_("Comment"));
	gtk_widget_show (fra_comment);
	gtk_box_pack_start (GTK_BOX (vbox1), fra_comment, TRUE, TRUE, 0);

	scrolledwindow2 = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_show (scrolledwindow2);
	gtk_container_add (GTK_CONTAINER (fra_comment), scrolledwindow2);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow2), GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);

	XTab->text_comment = gtk_text_new (NULL, NULL);
	gtk_widget_show (XTab->text_comment);
	gtk_container_add (GTK_CONTAINER (scrolledwindow2), XTab->text_comment);
	gtk_text_set_editable (GTK_TEXT (XTab->text_comment), TRUE);

	lbl_net = gtk_label_new (_("Additional"));
	gtk_widget_show (lbl_net);
	
	gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 4), lbl_net);
	gtk_notebook_set_menu_label_text (GTK_NOTEBOOK (notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 4),_("Additional"));
	
}

/*==================================Widget Functions=========================================*/

static void
contact_dialog_box_changed			(GtkWidget			*widget,
						GnomePropertyBox		*pb)
{
	gnome_property_box_changed (pb);
}

static void
lib_gefax_contact_dialog_init			(LibGefaxContactDialog		*ContactDiag)
{
	
	gint i;
	GnomePropertyBox *pb;
	
	pb = GNOME_PROPERTY_BOX (ContactDiag);
	ContactDiag->Priv = g_new0 (ContactDialogPriv, 1);
	ContactDiag->ITab = g_new0 (I_Tab, 1);
	ContactDiag->ATab = g_new0 (A_Tab, 1);
	ContactDiag->PTab = g_new0 (P_Tab, 1);
	ContactDiag->NTab = g_new0 (N_Tab, 1);
	ContactDiag->XTab = g_new0 (X_Tab, 1);
	
	gtk_window_set_position (GTK_WINDOW (ContactDiag), GTK_WIN_POS_CENTER);
	gtk_window_set_default_size (GTK_WINDOW (ContactDiag), 520, 420);
	gtk_window_set_policy (GTK_WINDOW (ContactDiag), TRUE, TRUE, FALSE);

	gtk_notebook_set_tab_hborder (GTK_NOTEBOOK (pb->notebook), 10);
	gtk_notebook_popup_enable (GTK_NOTEBOOK (pb->notebook));
	
	contact_dialog_create_I_TAB (ContactDiag->ITab, pb->notebook);
	contact_dialog_create_A_TAB (ContactDiag->ATab, pb->notebook);
	contact_dialog_create_P_TAB (ContactDiag->PTab, pb->notebook);
	contact_dialog_create_N_TAB (ContactDiag->NTab, pb->notebook);
	contact_dialog_create_X_TAB (ContactDiag->XTab, pb->notebook);
	
	ContactDiag->Priv->address_key = NO_KEY;
	ContactDiag->Priv->phone_key = NO_KEY;
	ContactDiag->Priv->email_key = NO_KEY;

	gtk_signal_connect (GTK_OBJECT (ContactDiag->ATab->clist_addr), "select_row",
					GTK_SIGNAL_FUNC (contact_dialog_display_address), ContactDiag);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->PTab->clist_phone), "select_row",
					GTK_SIGNAL_FUNC (contact_dialog_display_phone), ContactDiag);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->NTab->clist_email), "select_row",
					GTK_SIGNAL_FUNC (contact_dialog_display_email), ContactDiag);
	
	gtk_signal_connect (GTK_OBJECT (ContactDiag->ATab->btn_add), "clicked",
					GTK_SIGNAL_FUNC (contact_dialog_add_address), ContactDiag);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->ATab->btn_modify), "clicked",
					GTK_SIGNAL_FUNC (contact_dialog_modify_address), ContactDiag);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->ATab->btn_remove), "clicked",
					GTK_SIGNAL_FUNC (contact_dialog_remove_address), ContactDiag);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->PTab->btn_add), "clicked",
					GTK_SIGNAL_FUNC (contact_dialog_add_telephone), ContactDiag);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->PTab->btn_modify), "clicked",
					GTK_SIGNAL_FUNC (contact_dialog_modify_telephone), ContactDiag);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->PTab->btn_remove), "clicked",
					GTK_SIGNAL_FUNC (contact_dialog_remove_telephone), ContactDiag);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->NTab->btn_add), "clicked",
					GTK_SIGNAL_FUNC (contact_dialog_add_email), ContactDiag);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->NTab->btn_modify), "clicked",
					GTK_SIGNAL_FUNC (contact_dialog_modify_email), ContactDiag);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->NTab->btn_remove), "clicked",
					GTK_SIGNAL_FUNC (contact_dialog_remove_email), ContactDiag);
	

	gtk_signal_connect (GTK_OBJECT (ContactDiag->ATab->entry_addr_postoffice), "changed",
					GTK_SIGNAL_FUNC (ATab_address_changed_call), ContactDiag);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->ATab->entry_addr_extended), "changed",
					GTK_SIGNAL_FUNC (ATab_address_changed_call), ContactDiag);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->ATab->entry_addr_street), "changed",
					GTK_SIGNAL_FUNC (ATab_address_changed_call), ContactDiag);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->ATab->entry_addr_city), "changed",
					GTK_SIGNAL_FUNC (ATab_address_changed_call), ContactDiag);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->ATab->entry_addr_region), "changed",
					GTK_SIGNAL_FUNC (ATab_address_changed_call), ContactDiag);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->ATab->entry_addr_postcode), "changed",
					GTK_SIGNAL_FUNC (ATab_address_changed_call), ContactDiag);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->ATab->entry_addr_country), "changed",
					GTK_SIGNAL_FUNC (ATab_address_changed_call), ContactDiag);
	
	for (i = 0; i < ADDRESS_TYPES_LENGTH; i++) {
		gtk_signal_connect (GTK_OBJECT (ContactDiag->ATab->types[i]), "toggled",
					GTK_SIGNAL_FUNC (ATab_address_changed_call), ContactDiag);
	}
	gtk_signal_connect (GTK_OBJECT (ContactDiag->PTab->entry_phone_number), "changed",
					GTK_SIGNAL_FUNC (PTab_phone_changed_call), ContactDiag);
	
	for (i = 0; i < PHONE_TYPES_LENGTH; i++) {
		gtk_signal_connect (GTK_OBJECT (ContactDiag->PTab->types[i]), "toggled",
					GTK_SIGNAL_FUNC (PTab_phone_changed_call), ContactDiag);
	}
	gtk_signal_connect (GTK_OBJECT (ContactDiag->NTab->entry_email), "changed",
					GTK_SIGNAL_FUNC (NTab_email_changed_call), ContactDiag);
	
	for (i = 0; i < EMAIL_TYPES_LENGTH; i++) {
		gtk_signal_connect (GTK_OBJECT (ContactDiag->NTab->types[i]), "toggled",
					GTK_SIGNAL_FUNC (NTab_email_changed_call), ContactDiag);
	}

	gtk_signal_connect (GTK_OBJECT (GNOME_PROPERTY_BOX (ContactDiag)->apply_button), "clicked",
  					GTK_SIGNAL_FUNC (on_card_dialog_apply), ContactDiag);
  	gtk_signal_connect (GTK_OBJECT (GNOME_PROPERTY_BOX (ContactDiag)->ok_button), "clicked",
  					GTK_SIGNAL_FUNC (on_card_dialog_ok), ContactDiag);
  	gtk_signal_connect (GTK_OBJECT (ContactDiag), "help",
					GTK_SIGNAL_FUNC (on_card_dialog_help), NULL);
	
	return;
}

static void
lib_gefax_contact_dialog_class_init		(LibGefaxContactDialogClass	*Klass)
{
	GtkObjectClass *object_class;
	
	object_class = (GtkObjectClass *) Klass;
	lib_gefax_contact_dialog_parent_class = gtk_type_class (gnome_property_box_get_type ());
	
	object_class->destroy = lib_gefax_contact_dialog_destroy;
	lib_gefax_contact_dialog_signals[ADDRESSBOOK_CHANGED] = gtk_signal_new (
		"addressbook_changed",
		GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET (LibGefaxContactDialogClass, lib_gefax_contact_dialog),
		gtk_marshal_NONE__POINTER_INT,
		GTK_TYPE_NONE, 2, GTK_TYPE_POINTER, GTK_TYPE_INT);
	lib_gefax_contact_dialog_signals[DIALOG_HELP] = gtk_signal_new (
		"dialog_help",
		GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET (LibGefaxContactDialogClass, lib_gefax_contact_dialog),
		gtk_marshal_NONE__INT,
		GTK_TYPE_NONE, 1, GTK_TYPE_INT);

	gtk_object_class_add_signals (object_class, lib_gefax_contact_dialog_signals, LAST_SIGNAL);
	Klass->lib_gefax_contact_dialog = NULL;
}

static void
lib_gefax_contact_dialog_destroy		(GtkObject			*ContactDiag)
{
	LibGefaxContactDialog *CDiag;
	
	g_return_if_fail (ContactDiag != NULL);
	g_return_if_fail (LIBGEFAX_IS_CONTACT_DIALOG (ContactDiag));
	
	CDiag = LIBGEFAX_CONTACT_DIALOG (ContactDiag);
	if (CDiag->Priv->Contact) {
		lib_gefax_free_contact (CDiag->Priv->Contact);
	}
	
	if (GTK_OBJECT_CLASS (lib_gefax_contact_dialog_parent_class)->destroy)
		(*GTK_OBJECT_CLASS (lib_gefax_contact_dialog_parent_class)->destroy) (GTK_OBJECT (ContactDiag));
	g_free (CDiag->Priv);
	g_free (CDiag->ITab);
	g_free (CDiag->ATab);
	g_free (CDiag->PTab);
	g_free (CDiag->NTab);
	g_free (CDiag->XTab);
	return;
}

/*==================================Widget Public=========================================*/

GtkType
lib_gefax_contact_dialog_get_type  		(void)
{
	static GtkType type = 0;

	if (!type) {
		GtkTypeInfo info = {
			"LibGefaxContactDialog",
			sizeof (LibGefaxContactDialog),
			sizeof (LibGefaxContactDialogClass),
			(GtkClassInitFunc) lib_gefax_contact_dialog_class_init,
			(GtkObjectInitFunc) lib_gefax_contact_dialog_init,
			NULL, /* reserved 1 */
			NULL, /* reserved 2 */
			(GtkClassInitFunc) NULL
		};

		type = gtk_type_unique (gnome_property_box_get_type (), &info);
	}

	return type;
}

GtkWidget
*lib_gefax_contact_dialog_new			(GList			*LibGefaxAddressBook,
						gint			key)
{
	GtkWidget *ContactDialog;
	LibGefaxContactDialog *ContactDiag;
	ContactDialogPriv *Priv;
	GnomePropertyBox *pb;
	
	ContactDialog = gtk_type_new (lib_gefax_contact_dialog_get_type ());
	
	ContactDiag = LIBGEFAX_CONTACT_DIALOG (ContactDialog);
	Priv = ContactDiag->Priv;
	Priv->AddressBook = LibGefaxAddressBook;
	if (key == LIBGEFAX_CONTACT_DIALOG_NEW_CONTACT) {
		Priv->new = TRUE;
		Priv->Contact = lib_gefax_contact_new ();
	} else {
		Priv->Contact = lib_gefax_contact_clone_by_key (LibGefaxAddressBook, key);
		if (Priv->Contact) {
			Priv->new = FALSE;
			Priv->contact_key = key;
			lib_gefax_contact_dialog_fill (ContactDiag);
		} else {
			gtk_widget_destroy (ContactDialog);
			return NULL;
		}
	}
	pb = GNOME_PROPERTY_BOX (ContactDiag);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->ITab->entry_file), "changed",
				GTK_SIGNAL_FUNC (contact_dialog_box_changed), pb);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->ITab->entry_name_family), "changed",
				GTK_SIGNAL_FUNC (contact_dialog_box_changed), pb);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->ITab->entry_name_middle), "changed",
				GTK_SIGNAL_FUNC (contact_dialog_box_changed), pb);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->ITab->entry_name_given), "changed",
				GTK_SIGNAL_FUNC (contact_dialog_box_changed), pb);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->ITab->entry_name_prefix), "changed",
				GTK_SIGNAL_FUNC (contact_dialog_box_changed), pb);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->ITab->entry_name_suffix), "changed",
				GTK_SIGNAL_FUNC (contact_dialog_box_changed), pb);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->ITab->entry_org_name), "changed",
				GTK_SIGNAL_FUNC (contact_dialog_box_changed), pb);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->ITab->entry_org_title), "changed",
				GTK_SIGNAL_FUNC (contact_dialog_box_changed), pb);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->ITab->entry_cats), "changed",
				GTK_SIGNAL_FUNC (contact_dialog_box_changed), pb);
	gtk_signal_connect (GTK_OBJECT (GNOME_DATE_EDIT (ContactDiag->ITab->dateedit_birth)->date_entry), "changed",
				GTK_SIGNAL_FUNC (contact_dialog_box_changed), pb);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->NTab->entry_url), "changed",
				GTK_SIGNAL_FUNC (contact_dialog_box_changed), pb);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->XTab->rad_pgp), "toggled",
				GTK_SIGNAL_FUNC (contact_dialog_box_changed), pb);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->XTab->text_key), "changed",
				GTK_SIGNAL_FUNC (contact_dialog_box_changed), pb);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->XTab->text_comment), "changed",
				GTK_SIGNAL_FUNC (contact_dialog_box_changed), pb);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->XTab->spin_hrs), "changed",
				GTK_SIGNAL_FUNC (contact_dialog_box_changed), pb);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->XTab->spin_mins), "changed",
				GTK_SIGNAL_FUNC (contact_dialog_box_changed), pb);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->XTab->spin_lat), "changed",
				GTK_SIGNAL_FUNC (contact_dialog_box_changed), pb);
	gtk_signal_connect (GTK_OBJECT (ContactDiag->XTab->spin_lon), "changed",
				GTK_SIGNAL_FUNC (contact_dialog_box_changed), pb);
	
	return GTK_WIDGET (ContactDialog);
}










