/*  Copyright Daryl Gray 2001 email daryl.g@visto.com
 * 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __ADDRESS_BOOK_H__
#define __ADDRESS_BOOK_H__

#include "addressbook-enums.h"
#include "contact-dialog.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#define LIB_GEFAX_CONTACT(address_book)		(address_book->data)

/* These convieniance macro's will be private when the contact dialog is part of the library?
 * Or, are they a bad idear from the beginning?
*/
#define LIB_GEFAX_CONTACT_NEW()			(g_malloc0 (sizeof (LibGefaxContact)))
#define LIB_GEFAX_ADDRESS_NEW()			(g_malloc0 (sizeof (LibGefaxAddress)))
#define LIB_GEFAX_TELEPHONE_NEW()		(g_malloc0 (sizeof (LibGefaxTelephone)))
#define LIB_GEFAX_EMAIL_NEW()			(g_malloc0 (sizeof (LibGefaxEmail)))
#define LIB_GEFAX_TELEPHONE_NEW()		(g_malloc0 (sizeof (LibGefaxTelephone)))
#define LIB_GEFAX_IDENTITY_NEW()		(g_malloc0 (sizeof (LibGefaxIdentity)))
#define LIB_GEFAX_BUSINESS_NEW()		(g_malloc0 (sizeof (LibGefaxBusiness)))
#define LIB_GEFAX_TIMEZONE_NEW()		(g_malloc0 (sizeof (LibGefaxTimeZone)))
#define LIB_GEFAX_GEOGRAPHIC_NEW()		(g_malloc0 (sizeof (LibGefaxGeo)))
#define LIB_GEFAX_PUBLICKEY_NEW()		(g_malloc0 (sizeof (LibGefaxPublicKey)))
#define LIB_GEFAX_LOGO_NEW()			(g_malloc0 (sizeof (LibGefaxLogo)))
#define LIB_GEFAX_DATE_NEW()			(g_malloc0 (sizeof (LibGefaxDate)))
#define LIB_GEFAX_UNKNOWN_NEW()			(g_malloc0 (sizeof (LibGefaxUnknown)))

typedef struct _LibGefaxIdentity		LibGefaxIdentity;
typedef struct _LibGefaxBusiness		LibGefaxBusiness;
typedef struct _LibGefaxTimeZone		LibGefaxTimeZone;
typedef struct _LibGefaxGeo			LibGefaxGeo;
typedef struct _LibGefaxPublicKey		LibGefaxPublicKey;
typedef struct _LibGefaxLogo			LibGefaxLogo;
typedef struct _LibGefaxDate			LibGefaxDate;
typedef struct _LibGefaxAddress       		LibGefaxAddress;
typedef struct _LibGefaxTelephone		LibGefaxTelephone;
typedef struct _LibGefaxEmail			LibGefaxEmail;
typedef struct _LibGefaxContact			LibGefaxContact;
typedef struct _LibGefaxUnknown			LibGefaxUnknown;

typedef struct _LibGefaxContactPrivate    	LibGefaxContactPrivate;

struct _LibGefaxAddress {
	gint			key;
	int		 	type;			/*LibGefaxAddressType*/
	gchar 			*post_box;
	gchar 			*extended;
	gchar 			*street;
	gchar 			*city;
	gchar 			*region;
	gchar 			*post_code;
	gchar 			*country;
};

struct _LibGefaxTelephone {
	gint			key;
	int		 	type;			/*LibGefaxPhoneType*/
	gchar 			*number;
};

struct _LibGefaxEmail {
	gint			key;
	LibGefaxEmailType 	type;
	gchar 			*number;
};

struct _LibGefaxIdentity {	
	gchar 			*family;
	gchar 			*additional;
	gchar 			*given;
	gchar 			*prefix;
	gchar 			*suffix;
};

struct _LibGefaxBusiness {
	gchar 			*org_name;
	gchar 			*org_title;
};

struct _LibGefaxTimeZone {
	gint 			hours;
	gint 			minutes;
};

struct _LibGefaxGeo {
	gfloat 			latitude;
	gfloat 			longitude;
}; 

struct _LibGefaxPublicKey {
	LibGefaxKeyType 	type;
	gchar 			*key;
};

struct _LibGefaxLogo {
	LibGefaxLogoType 	type;
	gchar 			*data;
};

struct _LibGefaxDate {
	gint 			seconds;
	gint 			minutes;
	gint 			hours;
	gint 			days;
	gint 			months;
	gint 			years;
};

struct _LibGefaxUnknown {
	GSList 			*Tags;			/*Type gchar & in order (eg LABEL,HOME)*/
	GSList 			*Data;			/*Type gchar & in order - may contain NULL fields*/
	LibGefaxEncodeType	data_type;		/*Needed to know how to write Data to file again*/
};

struct _LibGefaxContact {
	gint			Key;			/*Generated when file is read. Identifies the contact & NEVER changes
							 so this may be used to identify contacts in a GTK CList for example*/
	LibGefaxContactPrivate	*Priv;
	LibGefaxIdentity 	*Identity;
	LibGefaxBusiness 	*Business;
	LibGefaxTimeZone	*TimeZone;
	LibGefaxGeo		*GeoLocation;
	LibGefaxPublicKey	*PublicKey;
	LibGefaxDate		*Revision;		/*Last time Modified?*/
	LibGefaxDate		*BirthDate;
	LibGefaxDate		*Anniversary;
	LibGefaxLogo		*Logo;
	gchar 			*filename;		/*Card File Name FN*/
	gchar 			*url;			/*Web Page*/
	gchar 			*note;
	GSList 			*Address;		/*List of LibGefaxAddress*/
	GSList 			*Tel;			/*List of LibGefaxTelephone*/
	GSList 			*Email;			/*List of LibGefaxEmail*/
	GSList 			*Categories;		/*List of Categories (gchar)*/
	GSList			*Unknown;		/*List of LibGefaxUnknown*/
};


/*******************************************************************
 *
 *	Functions to manipulate the entire address book
 *
 *******************************************************************/
 
/*
 * Reading and saving files
 * After calling "lib_gefax_read_file" you must keep this pointer unmodified at all times
 * When you are finished with the address book call "lib_gefax_free_address_book".
 * No other "GList pointers" need to be freed as they only point to a certain position/s (such as after a search)
 * in the first list.
*/
GList
*lib_gefax_read_file					(gchar			*absolute_file_path);

gboolean
lib_gefax_save_file					(GList			*LibGefaxAddressBook,
							gchar			*absolute_file_path,
							gboolean		new_file,
							LibGefaxFileType	format);

/*
 *Free the entire address book
*/
void
lib_gefax_free_address_book				(GList			*LibGefaxAddressBook);

/*
 *Add a new contact to an existing address book - appended
*/

GList
*lib_gefax_add_contact					(GList			*LibGefaxAddressBook,
							LibGefaxContact 	*contact);

/*
 *Remove a contact from an existing address book
*/

GList
*lib_gefax_remove_contact				(GList			*LibGefaxAddressBook,
							LibGefaxContact 	*contact);

/*
 * Return a pointer to a contact corrisponding to "key"
 * Returns NULL if not found
*/

LibGefaxContact
*lib_gefax_contact_get_by_key				(GList			*LibGefaxAddressBook,
							gint			key);

/*
 * Return a newly allocated copy of contact with "key"
 * The returned contact has key set to 0 Used by contact-dialog
*/

LibGefaxContact
*lib_gefax_contact_clone_by_key				(GList			*LibGefaxAddressBook,
							gint			key);

/*
 * Filter address book depending on the existance / non - existance of "filter"
 * DO NOT free the result with lib_gefax_free_address_book - just call g_list_free()!
 * Free the original address book with "lib_gefax_free_address_book"
*/

GList
*lib_gefax_filter_address_book				(GList			*LibGefaxAddressBook,
							LibGefaxFilterType	filter,
							gboolean		excludeing);

/*
 * Search contact for "string" depending on the "search" & "search_type"
 * You should not search ALL if possible as is is much slower
 * Returns TRUE if string is found and FALSE if not found or any incorrect parameters
*/
gboolean
lib_gefax_search_contact				(LibGefaxContact	*Contact,
							const gchar		*string,
							LibGefaxSearch		search,
							LibGefaxSearchType	search_type,
							gboolean		case_sensitive);



/*******************************************************************
 *
 *	Functions to manipulate a single existing contact
 *
 *******************************************************************/

/*
 * Returns a new empty contact with Key set to 0
 * Use "lib_gefax_add_contact" to add the contact to an address book as
 * this will ensure a unique Key
*/

LibGefaxContact
*lib_gefax_contact_new					(void);

void
lib_gefax_free_contact					(LibGefaxContact		*contact);

/*
 * Add an address to the given contact. Type must be specified in the address passed.
 * Returns TRUE on success, FALSE if the address already exists
*/

gboolean
lib_gefax_contact_add_address				(LibGefaxContact	*contact,
							LibGefaxAddress		*address);

/*
 * Remove and free address from contact corrisponding to address
 * If there are duplicates (as gnomecard will allow) the first in the list is removed
 * Returns TRUE on success
*/

gboolean
lib_gefax_contact_remove_address			(LibGefaxContact	*contact,
							LibGefaxAddress		*address);

LibGefaxAddress
*lib_gefax_contact_get_address_by_key			(LibGefaxContact	*contact,
							gint			key);

/*
 *Similar deal to above address
*/						
gboolean
lib_gefax_contact_add_tel				(LibGefaxContact	*contact,
							LibGefaxTelephone	*telephone);

gboolean
lib_gefax_contact_remove_tel				(LibGefaxContact	*contact,
							LibGefaxTelephone	*telephone);

LibGefaxTelephone
*lib_gefax_contact_get_telephone_by_key			(LibGefaxContact	*contact,
							gint			key);

gboolean
lib_gefax_contact_add_email				(LibGefaxContact	*contact,
							LibGefaxEmail		*email);

gboolean
lib_gefax_contact_remove_email				(LibGefaxContact	*contact,
							LibGefaxEmail		*email);
LibGefaxEmail
*lib_gefax_contact_get_email_by_key			(LibGefaxContact	*contact,
							gint			key);
/*Not working yet*/
gboolean
lib_gefax_contact_add_category				(LibGefaxContact	*contact,
							gchar			*category);
/*Not working yet*/
gboolean
lib_gefax_contact_remove_category			(LibGefaxContact	*contact,
							gchar			*category);

/*
 * "category_list" is a commer deliminated string eg "home,work,personal"
 * Existing categories not specified in "category_string" are removed
*/

void
lib_gefax_contact_set_categorys				(LibGefaxContact	*contact,
							gchar			*category_string);

/*
 * Return a commer deliminated string eg "home,work,personal" or NULL if
 * no existing categories in contact - FREE string after Use!
*/

gchar
*lib_gefax_contact_get_categorys_as_string		(LibGefaxContact	*contact);


#ifdef __cplusplus
}
#endif /* __cplusplus */


#endif /* __ADDRESS_BOOK_H__ */


