/*  Copyright Daryl Gray 2001 email daryl.g@visto.com
 * 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <gnome.h>
#include <dirent.h>
#include <ctype.h>
#include <stdio.h>
#include <fnmatch.h>

#include "address-book.h"
#include "vcard/vcard.h"
#include "vcard/vcard_save.h"

#define LIB_GEFAX_PRIVATE_NEW()			(g_malloc0 (sizeof (LibGefaxContactPrivate)))

struct _LibGefaxContactPrivate {
	gboolean	deleted;
};

static gchar
*my_g_strdup						(gchar				*data);

static void
lib_gefax_contact_copy_date				(LibGefaxDate			*Nd,
							LibGefaxDate			*Od);

static GSList
*lib_gefax_contact_copy_Address				(GSList				*A);

static GSList
*lib_gefax_contact_copy_Tel				(GSList				*T);

static GSList
*lib_gefax_contact_copy_Email				(GSList				*E);

static GSList
*lib_gefax_contact_copy_Unknown				(GSList				*U);

static GSList
*lib_gefax_contact_copy_gchar_GSList			(GSList				*Ol);

static void
lib_gefax_free_address					(LibGefaxAddress		*address);

static void
lib_gefax_free_tel					(LibGefaxTelephone		*tel);

static void
lib_gefax_free_email					(LibGefaxEmail			*email);

static void
lib_gefax_free_unknown					(LibGefaxUnknown		*Unknown);

static gboolean
lib_gefax_compare_address				(LibGefaxAddress		*Addr1,
							LibGefaxAddress			*Addr2);

static gboolean
lib_gefax_compare_telephone				(LibGefaxTelephone		*Tel1,
							LibGefaxTelephone		*Tel2);
static gboolean
lib_gefax_compare_email					(LibGefaxEmail			*Email1,
							LibGefaxEmail			*Email2);

static gchar
*my_g_strdup						(gchar				*data)
{
	gchar *new_str = NULL;

	if (data) {
		new_str = g_strdup (data);
	}
	return new_str;
}


static void
lib_gefax_contact_copy_date				(LibGefaxDate		*Nd,
							LibGefaxDate		*Od)
{
	Nd->seconds	= Od->seconds;
	Nd->minutes	= Od->minutes;
	Nd->hours	= Od->hours;
	Nd->days	= Od->days;
	Nd->months	= Od->months;
	Nd->years	= Od->years;
}

static GSList
*lib_gefax_contact_copy_Address				(GSList				*A)
{
	GSList *N = NULL;
	gint list_len, i;
	
	if (!A) {
		return NULL;
	}

	list_len = g_slist_length (A);
	for (i = 0; i < list_len; i++) {
		LibGefaxAddress *OldA = A->data;
		LibGefaxAddress *NewA = LIB_GEFAX_ADDRESS_NEW ();

		NewA->key	= OldA->key;
		NewA->type	= OldA->type;
		NewA->post_box	= my_g_strdup (OldA->post_box);
		NewA->extended	= my_g_strdup (OldA->extended);
		NewA->street	= my_g_strdup (OldA->street);
		NewA->city	= my_g_strdup (OldA->city);
		NewA->region	= my_g_strdup (OldA->region);
		NewA->post_code	= my_g_strdup (OldA->post_code);
		NewA->country	= my_g_strdup (OldA->country);

		N = g_slist_append (N, NewA);
		A = g_slist_next (A);
	}
	return N;
}

static GSList
*lib_gefax_contact_copy_Tel				(GSList				*T)
{
	GSList *N = NULL;
	gint list_len, i;
	
	if (!T) {
		return NULL;
	}
	
	list_len = g_slist_length (T);
	for (i = 0; i < list_len; i++) {
		LibGefaxTelephone *OldT = T->data;
		LibGefaxTelephone *NewT = LIB_GEFAX_TELEPHONE_NEW ();

		NewT->key	= OldT->key;
		NewT->type	= OldT->type;
		NewT->number	= my_g_strdup (OldT->number);

		N = g_slist_append (N, NewT);
		T = g_slist_next (T);
	}
	return N;
}

static GSList
*lib_gefax_contact_copy_Email				(GSList				*E)
{
	GSList *N = NULL;
	gint list_len, i;
	
	if (!E) {
		return NULL;
	}
	
	list_len = g_slist_length (E);
	for (i = 0; i < list_len; i++) {
		LibGefaxEmail *OldE = E->data;
		LibGefaxEmail *NewE = LIB_GEFAX_EMAIL_NEW ();

		NewE->key	= OldE->key;
		NewE->type	= OldE->type;
		NewE->number	= my_g_strdup (OldE->number);

		N = g_slist_append (N, NewE);
		E = g_slist_next (E);
	}
	return N;
}

static GSList
*lib_gefax_contact_copy_Unknown				(GSList				*U)
{
	GSList *N = NULL;
	gint list_len, i;
	
	if (!U) {
		return NULL;
	}
	
	list_len = g_slist_length (U);
	for (i = 0; i < list_len; i++) {
		LibGefaxUnknown *OldU = U->data;
		LibGefaxUnknown *NewU = LIB_GEFAX_UNKNOWN_NEW ();
		
		NewU->Tags	= lib_gefax_contact_copy_gchar_GSList (OldU->Tags);
		NewU->Data	= lib_gefax_contact_copy_gchar_GSList (OldU->Data);
		NewU->data_type	= OldU->data_type;

		N = g_slist_append (N, NewU);
		U = g_slist_next (U);
	}
	return N;
}

static GSList
*lib_gefax_contact_copy_gchar_GSList			(GSList				*Ol)
{
	GSList *N = NULL;
	gint list_len, i;
	
	if (!Ol) {
		return NULL;
	}
	
	list_len = g_slist_length (Ol);
	for (i = 0; i < list_len; i++) {
		gchar *old = Ol->data;
		gchar *new;

		new = my_g_strdup (old);

		N = g_slist_append (N, new);
		Ol = g_slist_next (Ol);
	}
	return N;
}

static gboolean
lib_gefax_compare_address				(LibGefaxAddress		*Addr1,
							LibGefaxAddress			*Addr2)
{
	if (Addr1->type != Addr2->type) {
		return FALSE;
	}
	
	if (((Addr1->post_box) && (!Addr2->post_box)) ||
		((!Addr1->post_box) && (Addr2->post_box))) {
		return FALSE;
	} else if (Addr1->post_box) {
		if (strcmp (Addr1->post_box, Addr2->post_box) != 0) {
			return FALSE;
		}
	}
	if (((Addr1->extended) && (!Addr2->extended)) ||
		((!Addr1->extended) && (Addr2->extended))) {
		return FALSE;
	} else if (Addr1->extended) {
		if (strcmp (Addr1->extended, Addr2->extended) != 0) {
			return FALSE;
		}
	}
	if (((Addr1->street) && (!Addr2->street)) ||
		((!Addr1->street) && (Addr2->street))) {
		return FALSE;
	} else if (Addr1->street) {
		if (strcmp (Addr1->street, Addr2->street) != 0) {
			return FALSE;
		}
	}
	if (((Addr1->city) && (!Addr2->city)) ||
		((!Addr1->city) && (Addr2->city))) {
		return FALSE;
	} else if (Addr1->city) {
		if (strcmp (Addr1->city, Addr2->city) != 0) {
			return FALSE;
		}
	}
	if (((Addr1->region) && (!Addr2->region)) ||
		((!Addr1->region) && (Addr2->region))) {
		return FALSE;
	} else if (Addr1->region) {
		if (strcmp (Addr1->region, Addr2->region) != 0) {
			return FALSE;
		}
	}
	if (((Addr1->post_code) && (!Addr2->post_code)) ||
		((!Addr1->post_code) && (Addr2->post_code))) {
		return FALSE;
	} else if (Addr1->post_code) {
		if (strcmp (Addr1->post_code, Addr2->post_code) != 0) {
			return FALSE;
		}
	}
	if (((Addr1->country) && (!Addr2->country)) ||
		((!Addr1->country) && (Addr2->country))) {
		return FALSE;
	} else if (Addr1->country) {
		if (strcmp (Addr1->country, Addr2->country) != 0) {
			return FALSE;
		}
	}
	return TRUE;
}

static gboolean
lib_gefax_compare_telephone				(LibGefaxTelephone		*Tel1,
							LibGefaxTelephone		*Tel2)
{
	if (Tel1->type != Tel2->type) {
		return FALSE;
	}
	
	if (((Tel1->number) && (!Tel2->number)) ||
		((!Tel1->number) && (Tel2->number))) {
		return FALSE;
	} else if (Tel1->number) {
		if (strcmp (Tel1->number, Tel2->number) != 0) {
			return FALSE;
		}
	}
	return TRUE;
}

static gboolean
lib_gefax_compare_email					(LibGefaxEmail			*Email1,
							LibGefaxEmail			*Email2)
{
		if (Email1->type != Email2->type) {
		return FALSE;
	}
	
	if (((Email1->number) && (!Email2->number)) ||
		((!Email1->number) && (Email2->number))) {
		return FALSE;
	} else if (Email1->number) {
		if (strcmp (Email1->number, Email2->number) != 0) {
			return FALSE;
		}
	}
	return TRUE;
}

static void
lib_gefax_free_address					(LibGefaxAddress		*address)
{
	if (address->post_box)		g_free (address->post_box);
	if (address->extended)		g_free (address->extended);
	if (address->street)		g_free (address->street);
	if (address->city)		g_free (address->city);
	if (address->region)		g_free (address->region);
	if (address->post_code)		g_free (address->post_code);
	if (address->country)		g_free (address->country);
	g_free (address);
}

static void
lib_gefax_free_tel					(LibGefaxTelephone		*tel)
{
	if (tel->number)	g_free (tel->number);
	g_free (tel);
}

static void
lib_gefax_free_email					(LibGefaxEmail			*email)
{
	if (email->number)	g_free (email->number);
	g_free (email);
}

static void
lib_gefax_free_unknown					(LibGefaxUnknown		*Unknown)
{
	GSList *List = NULL;
	gint list_length, i;
	
	List = Unknown->Tags;
	if (List) {
		list_length = g_slist_length (List);
		for (i = 0; i < list_length; i++) {
			if (List->data)
				g_free (List->data);
			List = g_slist_next (List);
		}
		g_slist_free (List);
	}
	List = Unknown->Data;
	if (List) {
		list_length = g_slist_length (List);
		for (i = 0; i < list_length; i++) {
			if (List->data)
				g_free (List->data);
			List = g_slist_next (List);
		}
		g_slist_free (List);
	}
	
}

GList
*lib_gefax_read_file					(gchar			*absolute_file_path)
{
	FILE *cf;
	gchar buff[16];
	
	g_return_val_if_fail (absolute_file_path != NULL, NULL);
	
	/*peek into the file to determine type*/
	if ((cf = fopen (absolute_file_path, "r")) == NULL) {
		printf ("libGefax ERROR %s cannot be opened\n", absolute_file_path);
		return NULL;
	}
	if (fgets (buff, 15, cf) != NULL) {
		if (strncmp (buff, "BEGIN:VCARD", 11) == 0) {
			fclose (cf);
			return (lib_gefax_read_Vcard_file (absolute_file_path));
		}
	}
	return NULL;

}

gboolean
lib_gefax_save_file					(GList			*LibGefaxAddressBook,
							gchar			*absolute_file_path,
							gboolean		new_file,
							LibGefaxFileType	format)
{
	g_return_val_if_fail (LibGefaxAddressBook != NULL, TRUE);
	g_return_val_if_fail (absolute_file_path != NULL, TRUE);
	
	if (new_file) {
		/*Don't bother to check for file - Overwrite it if it exists*/
		if (lib_gefax_save_Vcard_file (LibGefaxAddressBook, absolute_file_path)) {
			/*An Error Occured*/
			return TRUE;
		}
	} else {
		/*Rename (Back up) existing file/s if they exist first*/
		/*...........*/
		if (lib_gefax_save_Vcard_file (LibGefaxAddressBook, absolute_file_path)) {
			/*An Error Occured - Un Backup file/s*/
			return TRUE;
		}
	}
	return FALSE;
}

void
lib_gefax_free_address_book				(GList			*LibGefaxAddressBook)
{
	gint list_length, i;
	GList *list = LibGefaxAddressBook;
	
	list_length = g_list_length (list);
	for (i = 0; i < list_length; i++) {
		LibGefaxContact *c = LIB_GEFAX_CONTACT (list);
		lib_gefax_free_contact (c);
		list = g_list_next (list);
	}
	g_list_free (LibGefaxAddressBook);
		
}

GList
*lib_gefax_add_contact					(GList			*LibGefaxAddressBook,
							LibGefaxContact 	*contact)
{
	GList *Ab = LibGefaxAddressBook;
	gint list_length, i;
	gint key = 0;
	
	g_return_val_if_fail (contact != NULL, Ab);
	
	list_length = g_list_length (Ab);
	for (i = 0; i < list_length; i++) {
		LibGefaxContact *c = LIB_GEFAX_CONTACT (Ab);
		if (c->Key > key) {
			key = c->Key;
		}
		Ab = g_list_next (Ab);
	}
	++key;
	contact->Key = key;
	return g_list_append (LibGefaxAddressBook, contact);
}

GList
*lib_gefax_remove_contact				(GList			*LibGefaxAddressBook,
							LibGefaxContact 	*contact)
{
	GList *Ab = LibGefaxAddressBook;
	
	g_return_val_if_fail (Ab != NULL, NULL);
	g_return_val_if_fail (contact != NULL, Ab);
	
	Ab = g_list_remove (Ab, contact);
	lib_gefax_free_contact (contact);
	return Ab;
}


LibGefaxContact
*lib_gefax_contact_get_by_key				(GList			*LibGefaxAddressBook,
							gint			key)
{
	GList *ABook = LibGefaxAddressBook;
	gint list_len, i;
	
	g_return_val_if_fail (LibGefaxAddressBook != NULL, NULL);
	g_return_val_if_fail (key > 0, NULL);
	
	list_len = g_list_length (ABook);
	for (i = 0; i < list_len; i++) {
		LibGefaxContact *c = LIB_GEFAX_CONTACT (ABook);
		if (c->Key == key) {
			return c;
		}
		ABook = g_list_next (ABook);
	}
	
	return NULL;
}

LibGefaxContact
*lib_gefax_contact_clone_by_key				(GList			*LibGefaxAddressBook,
							gint			key)
{
	LibGefaxContact *Existing, *New;
	
	g_return_val_if_fail (LibGefaxAddressBook != NULL, NULL);
	g_return_val_if_fail (key > 0, NULL);
	
	Existing = lib_gefax_contact_get_by_key (LibGefaxAddressBook, key);
	
	if (!Existing) {
		return NULL;
	}
	New = lib_gefax_contact_new ();
	New->Identity->family		= my_g_strdup (Existing->Identity->family);
	New->Identity->additional	= my_g_strdup (Existing->Identity->additional);
	New->Identity->given		= my_g_strdup (Existing->Identity->given);
	New->Identity->prefix		= my_g_strdup (Existing->Identity->prefix);
	New->Identity->suffix		= my_g_strdup (Existing->Identity->suffix);
	New->Business->org_name		= my_g_strdup (Existing->Business->org_name);
	New->Business->org_title	= my_g_strdup (Existing->Business->org_title);
	New->TimeZone->minutes		= Existing->TimeZone->minutes;
	New->TimeZone->hours		= Existing->TimeZone->hours;
	New->GeoLocation->latitude	= Existing->GeoLocation->latitude;
	New->GeoLocation->longitude	= Existing->GeoLocation->longitude;
	New->PublicKey->type		= Existing->PublicKey->type;
	New->PublicKey->key		= my_g_strdup (Existing->PublicKey->key);
	lib_gefax_contact_copy_date (New->Revision, Existing->Revision);
	lib_gefax_contact_copy_date (New->BirthDate, Existing->BirthDate);
	lib_gefax_contact_copy_date (New->Anniversary, Existing->Anniversary);
	if (Existing->Logo) {
		New->Logo		= LIB_GEFAX_LOGO_NEW ();
		New->Logo->type		= Existing->Logo->type;
		New->Logo->data		= my_g_strdup (Existing->Logo->data);
	}
	New->filename			= my_g_strdup (Existing->filename);
	New->url			= my_g_strdup (Existing->url);
	New->note			= my_g_strdup (Existing->note);
	New->Address			= lib_gefax_contact_copy_Address (Existing->Address);
	New->Tel			= lib_gefax_contact_copy_Tel (Existing->Tel);
	New->Email			= lib_gefax_contact_copy_Email (Existing->Email);
	New->Categories			= lib_gefax_contact_copy_gchar_GSList (Existing->Categories);
	New->Unknown			= lib_gefax_contact_copy_Unknown (Existing->Unknown);
	return New;

}

static gboolean
lib_gefax_telephone_has_type				(GSList			*Tel,
							gint			type)
{
	gint list_len, i;
	GSList *T = Tel;
	
	list_len = g_slist_length (T);
	for (i = 0; i < list_len; i ++) {
		if (((LibGefaxTelephone *) T->data)->type & type) {
			return TRUE;
		}
		T = g_slist_next (T);
	}
	return FALSE;
}

GList
*lib_gefax_filter_address_book				(GList			*LibGefaxAddressBook,
							LibGefaxFilterType	filter,
							gboolean		excludeing)
{
	GList *O = LibGefaxAddressBook;
	GList *A = NULL;
	gint list_len, i;
	
	list_len = g_list_length (O);
	for (i = 0; i < list_len; i ++) {
		LibGefaxContact *C = LIB_GEFAX_CONTACT (O);
		if (filter == LIBGEFAX_FILTER_PHONE) {
			if (C->Tel) {
				if (!excludeing) {
					A = g_list_append (A, C);
				}
			} else {
				if (excludeing) {
					A = g_list_append (A, C);
				}
			}
		} else if (filter == LIBGEFAX_FILTER_FAX) {
			if (C->Tel) {
				if (!excludeing) {
					if (lib_gefax_telephone_has_type (C->Tel, LIBGEFAX_PHONE_FAX)) {
						A = g_list_append (A, C);
					}
				}
			} else {
				if (excludeing) {
					if (lib_gefax_telephone_has_type (C->Tel, LIBGEFAX_PHONE_FAX)) {
						A = g_list_append (A, C);
					}
				}
			}
		} else if (filter == LIBGEFAX_FILTER_ADDRESS) {
			if (C->Address) {
				if (!excludeing) {
					A = g_list_append (A, C);
				}
			} else {
				if (excludeing) {
					A = g_list_append (A, C);
				}
			}
		} else if (filter == LIBGEFAX_FILTER_EMAIL) {
			if (C->Email) {
				if (!excludeing) {
					A = g_list_append (A, C);
				}
			} else {
				if (excludeing) {
					A = g_list_append (A, C);
				}
			}
		} else { /*LIBGEFAX_FILTER_URL*/
			if (C->url) {
				if (!excludeing) {
					A = g_list_append (A, C);
				}
			} else {
				if (excludeing) {
					A = g_list_append (A, C);
				}
			}
		}
		O = g_list_next (O);
	}
	return A;
}

/*Stolen from gnomecard & modified*/
static gboolean
address_book_match_pattern				(const gchar		*pattern,
							gchar			*str,
							gboolean		sens)
{
	gchar *txt;
	gboolean found = FALSE;
	
	if (!str) {
		return FALSE;
	}
	if (sens) {
		txt = str;
	} else {
		g_strup (txt = g_strdup (str));
	}
	if (fnmatch (pattern, txt, 0) == 0) {
		found = TRUE;
	}
	
	if (! sens) {
		g_free(txt);
	}
	return found;
}

static gboolean
address_book_search_contact_identity			(LibGefaxContact	*C,
							const gchar		*pattern,
							gboolean		case_sens)
{
	LibGefaxIdentity *I = C->Identity;
	gboolean found;
	
	found = address_book_match_pattern (pattern,C->filename ,case_sens);
	if (found) {
		return (found);
	}
	found = address_book_match_pattern (pattern,I->family ,case_sens);
	if (found) {
		return (found);
	}
	found = address_book_match_pattern (pattern,I->additional ,case_sens);
	if (found) {
		return (found);
	}
	found = address_book_match_pattern (pattern,I->given ,case_sens);
	if (found) {
		return (found);
	}
	found = address_book_match_pattern (pattern,I->prefix ,case_sens);
	if (found) {
		return (found);
	}
	found = address_book_match_pattern (pattern,I->suffix ,case_sens);

	return found;
}

static gboolean
address_book_search_contact_address			(LibGefaxContact	*C,
							const gchar		*pattern,
							gboolean		case_sens)
{
	gboolean found = FALSE;
	GSList *Al = C->Address;
	gint list_len, i;
	
	if (!Al) {
		return found;
	}
	list_len = g_slist_length (Al);
	for (i = 0; i < list_len; i++) {
		LibGefaxAddress *A = Al->data;
		
		found = address_book_match_pattern (pattern, A->post_box, case_sens);
		if (found) {
			break;
		}
		found = address_book_match_pattern (pattern, A->extended, case_sens);
		if (found) {
			break;
		}
		found = address_book_match_pattern (pattern, A->street, case_sens);
		if (found) {
			break;
		}
		found = address_book_match_pattern (pattern, A->city, case_sens);
		if (found) {
			break;
		}
		found = address_book_match_pattern (pattern, A->region, case_sens);
		if (found) {
			break;
		}
		found = address_book_match_pattern (pattern, A->post_code, case_sens);
		if (found) {
			break;
		}
		found = address_book_match_pattern (pattern, A->country, case_sens);
		if (found) {
			break;
		}		
		Al = g_slist_next (Al);
	}
	return found;
}

static gboolean
address_book_search_contact_cats			(LibGefaxContact	*C,
							const gchar		*pattern,
							gboolean		case_sens)
{
	gboolean found = FALSE;
	GSList *Cl = C->Categories;
	gint list_len, i;
	
	if (!Cl) {
		return found;
	}
	list_len = g_slist_length (Cl);
	for (i = 0; i < list_len; i++) {
		gchar *D = Cl->data;
		
		found = address_book_match_pattern (pattern, D, case_sens);
		if (found) {
			break;
		}
		Cl = g_slist_next (Cl);
	}
	return found;
}

static gboolean
address_book_search_contact_business			(LibGefaxContact	*C,
							const gchar		*pattern,
							gboolean		case_sens)
{
	LibGefaxBusiness *B = C->Business;
	gboolean found;
	
	found = address_book_match_pattern (pattern,B->org_name ,case_sens);
	if (found) {
		return (found);
	}
	found = address_book_match_pattern (pattern,B->org_title ,case_sens);

	return found;
}

static gboolean
address_book_search_contact_telephone			(LibGefaxContact	*C,
							const gchar		*pattern,
							gboolean		case_sens)
{
	gboolean found = FALSE;
	GSList *Tl = C->Tel;
	gint list_len, i;
	
	if (!Tl) {
		return found;
	}
	list_len = g_slist_length (Tl);
	for (i = 0; i < list_len; i++) {
		LibGefaxTelephone *T = Tl->data;
		
		found = address_book_match_pattern (pattern, T->number, case_sens);
		if (found) {
			break;
		}
		Tl = g_slist_next (Tl);
	}
	return found;
}

static gboolean
address_book_search_contact_email			(LibGefaxContact	*C,
							const gchar		*pattern,
							gboolean		case_sens)
{
	gboolean found = FALSE;
	GSList *El = C->Email;
	gint list_len, i;
	
	if (!El) {
		return found;
	}
	list_len = g_slist_length (El);
	for (i = 0; i < list_len; i++) {
		LibGefaxEmail *E = El->data;
		
		found = address_book_match_pattern (pattern, E->number, case_sens);
		if (found) {
			break;
		}
		El = g_slist_next (El);
	}
	return found;
}

static gboolean
address_book_search_contact_all				(LibGefaxContact	*C,
							const gchar		*pattern,
							gboolean		case_sens)
{
	gboolean found;
	
	found = address_book_search_contact_identity (C, pattern, case_sens);
	if (found) {
		return found;
	}
	found = address_book_search_contact_address (C, pattern, case_sens);
	 if (found) {
		return found;
	}
	found = address_book_search_contact_cats (C, pattern, case_sens);
	if (found) {
		return found;
	}
	found = address_book_search_contact_telephone (C, pattern, case_sens);
	if (found) {
		return found;
	}
	found = address_book_search_contact_business (C, pattern, case_sens);
	if (found) {
		return found;
	}
	found = address_book_search_contact_email (C, pattern, case_sens);
	if (found) {
		return found;
	}
	/*Search url and note*/
	found = address_book_match_pattern (pattern, C->url ,case_sens);
	if (found) {
		return (found);
	}
	found = address_book_match_pattern (pattern, C->note ,case_sens);

	
						
	return found;
}

gboolean
lib_gefax_search_contact				(LibGefaxContact	*Contact,
							const gchar		*string,
							LibGefaxSearch		search,
							LibGefaxSearchType	search_type,
							gboolean		case_sensitive)
{
	gchar *pattern;
	gboolean found;
	gchar *txt;
	
	g_return_val_if_fail (Contact != NULL, FALSE);
	g_return_val_if_fail (string != NULL, FALSE);
	
	/*Build the required pattern*/
	if (!case_sensitive) {
		g_strup (txt = g_strdup (string));
	} else {
		txt = g_strdup (string);
	}
	if (search_type == LIBGEFAX_SEARCHTYPE_CONTAINS) {
		pattern = g_strdup_printf ("*%s*", txt);
	} else if (search_type == LIBGEFAX_SEARCHTYPE_END) {
		pattern = g_strdup_printf ("*%s", txt);
	} else if (search_type == LIBGEFAX_SEARCHTYPE_BEGIN) {
		pattern = g_strdup_printf ("%s*", txt);
	} else {/*LIBGEFAX_SEARCHTYPE_IS - do nothing*/
		pattern = txt;
	}	
	
	if (search == LIBGEFAX_SEARCH_IDENTITY) {
		found = address_book_search_contact_identity (Contact, pattern, case_sensitive);
	} else if (search == LIBGEFAX_SEARCH_ADDRESS) {
		found = address_book_search_contact_address (Contact, pattern, case_sensitive);
	} else if (search == LIBGEFAX_SEARCH_CATEGORY) {
		found = address_book_search_contact_cats (Contact, pattern, case_sensitive);
	} else if (search == LIBGEFAX_SEARCH_TELEPHONE) {
		found = address_book_search_contact_telephone (Contact, pattern, case_sensitive);
	} else if (search == LIBGEFAX_SEARCH_BUSINESS) {
		found = address_book_search_contact_business (Contact, pattern, case_sensitive);
	} else if (search == LIBGEFAX_SEARCH_EMAIL) {
		found = address_book_search_contact_email (Contact, pattern, case_sensitive);
	} else {	/*LIBGEFAX_SEARCH_ALL*/
		found = address_book_search_contact_all (Contact, pattern, case_sensitive);
	}
	g_free (txt);
	if (search_type != LIBGEFAX_SEARCHTYPE_IS) {
		g_free (pattern);
	}
	return found;
}

/******************************************************************/

LibGefaxContact
*lib_gefax_contact_new					(void)
{
	LibGefaxContact *c;
	
	/*allocate memory*/
	c =			LIB_GEFAX_CONTACT_NEW ();
	c->Priv =		LIB_GEFAX_PRIVATE_NEW ();
	c->Identity =		LIB_GEFAX_IDENTITY_NEW ();
	c->Business =		LIB_GEFAX_BUSINESS_NEW ();
	c->TimeZone =		LIB_GEFAX_TIMEZONE_NEW ();
	c->GeoLocation =	LIB_GEFAX_GEOGRAPHIC_NEW ();
	c->PublicKey =		LIB_GEFAX_PUBLICKEY_NEW ();
	c->Revision =		LIB_GEFAX_DATE_NEW ();
	c->BirthDate =		LIB_GEFAX_DATE_NEW ();
	c->Anniversary =	LIB_GEFAX_DATE_NEW ();
	
	c->Key = 0;
	c->Priv->deleted = FALSE;
	return c;
	
	
}

void
lib_gefax_free_contact					(LibGefaxContact		*contact)
{
	LibGefaxContactPrivate	*P = contact->Priv;
	LibGefaxIdentity 	*I = contact->Identity;
	LibGefaxBusiness 	*B = contact->Business;
	LibGefaxTimeZone	*T = contact->TimeZone;
	LibGefaxGeo		*G = contact->GeoLocation;
	LibGefaxPublicKey	*K = contact->PublicKey;
	LibGefaxLogo		*L = contact->Logo;
	
	GSList *list = NULL;
	gint list_length, i;

	g_free (P);
	if (I->family) 		g_free (I->family);
	if (I->additional) 	g_free (I->additional);
	if (I->given) 		g_free (I->given);
	if (I->prefix) 		g_free (I->prefix);
	if (I->suffix) 		g_free (I->suffix);
	g_free (I);
	if (B->org_name)	g_free (B->org_name);
	if (B->org_title)	g_free (B->org_title);
	g_free (B);
	g_free (T);
	g_free (G);
	if (K->key)		g_free (K->key);
	g_free (K);
	g_free (contact->Revision);
	g_free (contact->BirthDate);
	g_free (contact->Anniversary);
	if (contact->filename)	g_free (contact->filename);
	if (contact->url)	g_free (contact->url);
	if (contact->note)	g_free (contact->note);
	if (L) {
		if (L->data)	g_free (L->data);
		g_free (L);
	}
	
	list = contact->Address;
	if (list) {
		list_length = g_slist_length (list);
		for (i = 0; i < list_length; i++) {
			LibGefaxAddress *Ad = list->data;
			lib_gefax_free_address (Ad);
				list = g_slist_next (list);
		}
		g_slist_free (list);
	}
	list = contact->Tel;
	if (list) {
		list_length = g_slist_length (list);
		for (i = 0; i < list_length; i++) {
			LibGefaxTelephone *Te = list->data;
			lib_gefax_free_tel (Te);
				list = g_slist_next (list);
		}
		g_slist_free (list);
	}
	list = contact->Email;
	if (list) {
		list_length = g_slist_length (list);
		for (i = 0; i < list_length; i++) {
			LibGefaxEmail *Em = list->data;
			lib_gefax_free_email (Em);
				list = g_slist_next (list);
		}
		g_slist_free (list);
	}
	list = contact->Categories;
	if (list) {
		list_length = g_slist_length (list);
		for (i = 0; i < list_length; i++) {
			g_free (list->data);
			list = g_slist_next (list);
		}
		g_slist_free (list);
	}
	list = contact->Unknown;
	if (list) {
		list_length = g_slist_length (list);
		for (i = 0; i < list_length; i++) {
			lib_gefax_free_unknown (list->data);
			g_free (list->data);
			list = g_slist_next (list);
		}
		g_slist_free (list);
	}
	
	g_free (contact);
}

gboolean
lib_gefax_contact_add_address				(LibGefaxContact		*contact,
							LibGefaxAddress			*address)
{
	GSList *Al = contact->Address;
	gint list_len, i;
	
	g_return_val_if_fail (address != NULL, FALSE);
	
	list_len = g_slist_length (Al);
	for (i = 0; i < list_len; i++) {
		LibGefaxAddress *A = Al->data;
		if (lib_gefax_compare_address (address, A)) {
			return FALSE;
		}
		if (address->key <= A->key) {
			address->key = A->key + 1;
		}
		Al = g_slist_next (Al);
	}
	contact->Address = g_slist_append (contact->Address, address);
	return TRUE;
}

gboolean
lib_gefax_contact_remove_address			(LibGefaxContact		*contact,
							LibGefaxAddress			*address)
{
	GSList *Al = contact->Address;
	gint list_len, i;
	
	g_return_val_if_fail (Al != NULL, FALSE);
	g_return_val_if_fail (address != NULL, FALSE);
		
	list_len = g_slist_length (Al);
	for (i = 0; i < list_len; i++) {
		LibGefaxAddress *A = Al->data;
		if (address->key == A->key) {
			contact->Address = g_slist_remove (contact->Address, address);
			lib_gefax_free_address (address);
			return TRUE;
		}
		Al = g_slist_next (Al);
	}
	return FALSE;
}

LibGefaxAddress
*lib_gefax_contact_get_address_by_key			(LibGefaxContact	*contact,
							gint			key)
{
	GSList *Al = contact->Address;
	gint list_len, i;
	
	g_return_val_if_fail (Al != NULL, NULL);
	g_return_val_if_fail (contact != NULL, NULL);
		
	list_len = g_slist_length (Al);
	for (i = 0; i < list_len; i++) {
		LibGefaxAddress *A = Al->data;
		if (A->key == key) {
			return A;
		}
		Al = g_slist_next (Al);
	}
	return NULL;
}

gboolean
lib_gefax_contact_add_tel				(LibGefaxContact		*contact,
							LibGefaxTelephone		*telephone)
{
	GSList *Pl = contact->Tel;
	gint list_len, i;
	
	g_return_val_if_fail (telephone != NULL, FALSE);
	
	list_len = g_slist_length (Pl);
	for (i = 0; i < list_len; i++) {
		LibGefaxTelephone *T = Pl->data;
		if (lib_gefax_compare_telephone (telephone, T)) {
			return FALSE;
		}
		if (telephone->key <= T->key) {
			telephone->key = T->key + 1;
		}
		Pl = g_slist_next (Pl);
	}
	contact->Tel = g_slist_append (contact->Tel, telephone);
	return TRUE;
}

gboolean
lib_gefax_contact_remove_tel				(LibGefaxContact		*contact,
							LibGefaxTelephone		*telephone)
{
	GSList *Pl = contact->Tel;
	gint list_len, i;
	
	g_return_val_if_fail (Pl != NULL, FALSE);
	g_return_val_if_fail (telephone != NULL, FALSE);
	
	list_len = g_slist_length (Pl);
	for (i = 0; i < list_len; i++) {
		LibGefaxTelephone *T = Pl->data;
		if (telephone->key == T->key) {
			contact->Tel = g_slist_remove (contact->Tel, telephone);
			lib_gefax_free_tel (telephone);
			return TRUE;
		}
		Pl = g_slist_next (Pl);
	}
	return FALSE;
}

LibGefaxTelephone
*lib_gefax_contact_get_telephone_by_key			(LibGefaxContact	*contact,
							gint			key)
{
	GSList *Tl = contact->Tel;
	gint list_len, i;
	
	g_return_val_if_fail (Tl != NULL, NULL);
	g_return_val_if_fail (contact != NULL, NULL);
		
	list_len = g_slist_length (Tl);
	for (i = 0; i < list_len; i++) {
		LibGefaxTelephone *T = Tl->data;
		if (T->key == key) {
			return T;
		}
		Tl = g_slist_next (Tl);
	}
	return NULL;
}

gboolean
lib_gefax_contact_add_email				(LibGefaxContact		*contact,
							LibGefaxEmail			*email)
{
	GSList *El = contact->Email;
	gint list_len, i;
	
	g_return_val_if_fail (email != NULL, FALSE);
	
	list_len = g_slist_length (El);
	for (i = 0; i < list_len; i++) {
		LibGefaxEmail *E = El->data;
		if (lib_gefax_compare_email (email, E)) {
			return FALSE;
		}
		if (email->key <= E->key) {
			email->key = E->key + 1;
		}
		El = g_slist_next (El);
	}
	contact->Email = g_slist_append (contact->Email, email);
	return TRUE;
}

gboolean
lib_gefax_contact_remove_email				(LibGefaxContact		*contact,
							LibGefaxEmail			*email)
{
	GSList *El = contact->Email;
	gint list_len, i;
	
	g_return_val_if_fail (El != NULL, FALSE);
	g_return_val_if_fail (email != NULL, FALSE);
	
	list_len = g_slist_length (El);
	for (i = 0; i < list_len; i++) {
		LibGefaxEmail *E = El->data;
		if (email->key == E->key) {
			contact->Email = g_slist_remove (contact->Email, email);
			lib_gefax_free_email (email);
			return TRUE;
		}
		El = g_slist_next (El);
	}
	return FALSE;
}

LibGefaxEmail
*lib_gefax_contact_get_email_by_key			(LibGefaxContact	*contact,
							gint			key)
{
	GSList *El = contact->Email;
	gint list_len, i;
	
	g_return_val_if_fail (El != NULL, NULL);
	g_return_val_if_fail (contact != NULL, NULL);
		
	list_len = g_slist_length (El);
	for (i = 0; i < list_len; i++) {
		LibGefaxEmail *E = El->data;
		if (E->key == key) {
			return E;
		}
		El = g_slist_next (El);
	}
	return NULL;
}

gboolean
lib_gefax_contact_add_category				(LibGefaxContact		*contact,
							gchar				*category)
{
	return FALSE;
}

gboolean
lib_gefax_contact_remove_category			(LibGefaxContact		*contact,
							gchar				*category)
{
	return FALSE;
}

void
lib_gefax_contact_set_categorys				(LibGefaxContact		*contact,
							gchar				*category_string)
{
	/*Categories are commer(,) delimeted?*/

	gchar *p1,*p2;
	gint list_length, i;
	
	g_return_if_fail (contact != NULL);
	if (category_string == NULL) {
		return;
	}
	if (contact->Categories) {
		list_length = g_slist_length (contact->Categories);
		for (i = 0; i < list_length; i++) {
			g_free (contact->Categories->data);
			contact->Categories = g_slist_next (contact->Categories);
		}
		g_slist_free (contact->Categories);
	}
	contact->Categories = NULL;
	
	p1 = category_string;
	while (*p1 != '\0') {
		p2 = strchr(p1, ',');
		if (!p2) {
			contact->Categories = g_slist_append (contact->Categories, g_strdup (p1));
			break;
		}
		*p2 = '\0';
		++p2;
		contact->Categories = g_slist_append (contact->Categories, g_strdup (p1));
		p1 = p2;
	}
}


gchar
*lib_gefax_contact_get_categorys_as_string		(LibGefaxContact		*contact)
{
	GSList *Cl = contact->Categories;
	gchar *string = NULL;
	
	g_return_val_if_fail (contact != NULL, NULL);
	
	if (Cl) {
		gint list_len, i;
		gchar *str_tmp;
	
		list_len = g_slist_length (Cl);
		for (i = 0; i < list_len; i++) {
			gchar *cat = Cl->data;
			if (string) {
				str_tmp = g_strdup (string);
				g_free (string);
				string = g_strdup_printf ("%s,%s", str_tmp, cat);
				g_free (str_tmp);
			} else {
				string = g_strdup (cat);
			}
			Cl = g_slist_next (Cl);
		}
	}
	return string;
}

