/////////////////////////////////////////////////////////////////////////////
// Name:        cube.cpp
// Purpose:     wxGLCanvas demo program
// Author:      Julian Smart
// Modified by:
// Created:     04/01/98
// RCS-ID:      $Id: cube.cpp,v 1.19 2005/03/27 18:08:08 VZ Exp $
// Copyright:   (c) Julian Smart
// Licence:     wxWindows licence
/////////////////////////////////////////////////////////////////////////////

#if defined(__GNUG__) && !defined(__APPLE__)
#pragma implementation
#pragma interface
#endif

// For compilers that support precompilation, includes "wx.h".
#include "wx/wxprec.h"

#ifdef __BORLANDC__
#pragma hdrstop
#endif

#ifndef WX_PRECOMP
#include "wx/wx.h"
#endif

#if !wxUSE_GLCANVAS
    #error "OpenGL required: set wxUSE_GLCANVAS to 1 and rebuild the library"
#endif

#include "test.h"
//#include "../../sample.xpm"

#ifndef __WXMSW__     // for StopWatch, see remark below
  #if defined(__WXMAC__) && !defined(__DARWIN__)
    #include <utime.h>
    #include <unistd.h>
  #else
    #include <sys/time.h>
    #include <sys/unistd.h>
  #endif
#else
#include <sys/timeb.h>
#endif

#define ID_NEW_WINDOW 10000
#define ID_DEF_ROTATE_LEFT_KEY 10001
#define ID_DEF_ROTATE_RIGHT_KEY 10002

#include <GL/glu.h>
#include <GL/glut.h>
#include "mesh.h"
#include "math3d.h"

char filename[128];


/*----------------------------------------------------------
  Control to get a keycode
  ----------------------------------------------------------*/
class ScanCodeCtrl : public wxTextCtrl
{
public:
    ScanCodeCtrl( wxWindow* parent, wxWindowID id, int code,
        const wxPoint& pos, const wxSize& size );

    void OnChar( wxKeyEvent& WXUNUSED(event) )
    {
        // Do nothing
    }

    void OnKeyDown(wxKeyEvent& event);

private:

    // Any class wishing to process wxWidgets events must use this macro
    DECLARE_EVENT_TABLE()
};

BEGIN_EVENT_TABLE( ScanCodeCtrl, wxTextCtrl )
    EVT_CHAR( ScanCodeCtrl::OnChar )
    EVT_KEY_DOWN( ScanCodeCtrl::OnKeyDown )
END_EVENT_TABLE()

ScanCodeCtrl::ScanCodeCtrl( wxWindow* parent, wxWindowID id, int code,
    const wxPoint& pos, const wxSize& size )
    : wxTextCtrl( parent, id, wxEmptyString, pos, size )
{
    SetValue( wxString::Format(wxT("0x%04x"), code) );
}

void ScanCodeCtrl::OnKeyDown( wxKeyEvent& event )
{
    SetValue( wxString::Format(wxT("0x%04x"), event.GetKeyCode()) );
}

/*------------------------------------------------------------------
 Dialog for defining a keypress
-------------------------------------------------------------------*/

class ScanCodeDialog : public wxDialog
{
public:
    ScanCodeDialog( wxWindow* parent, wxWindowID id, const int code,
        const wxString &descr, const wxString& title );
    int GetValue();

private:

    ScanCodeCtrl       *m_ScanCode;
    wxTextCtrl         *m_Description;
};

ScanCodeDialog::ScanCodeDialog( wxWindow* parent, wxWindowID id,
    const int code, const wxString &descr, const wxString& title )
    : wxDialog( parent, id, title, wxDefaultPosition, wxSize(96*2,76*2) )
{
    new wxStaticText( this, wxID_ANY, _T("Scancode"), wxPoint(4*2,3*2),
        wxSize(31*2,12*2) );
    m_ScanCode = new ScanCodeCtrl( this, wxID_ANY, code, wxPoint(37*2,6*2),
        wxSize(53*2,14*2) );

    new wxStaticText( this, wxID_ANY, _T("Description"), wxPoint(4*2,24*2),
        wxSize(32*2,12*2) );
    m_Description = new wxTextCtrl( this, wxID_ANY, descr, wxPoint(37*2,27*2),
        wxSize(53*2,14*2) );

    new wxButton( this, wxID_OK, _T("Ok"), wxPoint(20*2,50*2), wxSize(20*2,13*2) );
    new wxButton( this, wxID_CANCEL, _T("Cancel"), wxPoint(44*2,50*2),
        wxSize(25*2,13*2) );
}

int ScanCodeDialog::GetValue()
{
    int code;
    wxString buf = m_ScanCode->GetValue();
    wxSscanf( buf.c_str(), _T("%i"), &code );
    return code;
}

/*----------------------------------------------------------------------
  Utility function to get the elapsed time (in msec) since a given point
  in time (in sec)  (because current version of wxGetElapsedTime doesnt
  works right with glibc-2.1 and linux, at least for me)
-----------------------------------------------------------------------*/
unsigned long StopWatch( unsigned long *sec_base )
{
  unsigned long secs,msec;

#if defined(__WXMSW__)
  struct timeb tb;
  ftime( &tb );
  secs = tb.time;
  msec = tb.millitm;
#elif defined(__WXMAC__) && !defined(__DARWIN__)
  wxLongLong tl = wxGetLocalTimeMillis();
  secs = (unsigned long) (tl.GetValue() / 1000);
  msec = (unsigned long) (tl.GetValue() - secs*1000);
#else
  int m_bv1,m_bv2;
  // think every unice has gettimeofday
  struct timeval tv;
  gettimeofday( &tv, (struct timezone *)NULL );
  secs = tv.tv_sec;
  msec = tv.tv_usec/1000;
#endif

  if( *sec_base == 0 )
    *sec_base = secs;

  return( (secs-*sec_base)*1000 + msec );
}

/*----------------------------------------------------------------
  Implementation of Test-GLCanvas
-----------------------------------------------------------------*/

BEGIN_EVENT_TABLE(TestGLCanvas, wxGLCanvas)
    EVT_SIZE(TestGLCanvas::OnSize)
    EVT_PAINT(TestGLCanvas::OnPaint)
    EVT_ERASE_BACKGROUND(TestGLCanvas::OnEraseBackground)
    EVT_KEY_DOWN( TestGLCanvas::OnKeyDown )
    EVT_KEY_UP( TestGLCanvas::OnKeyUp )
    EVT_ENTER_WINDOW( TestGLCanvas::OnEnterWindow )

    // mouse events:
    EVT_MOTION(TestGLCanvas::OnMouseMotion)
    EVT_LEFT_DOWN(TestGLCanvas::OnMouseLeftDown)
    EVT_RIGHT_DOWN(TestGLCanvas::OnMouseRightDown)
    EVT_MIDDLE_DOWN(TestGLCanvas::OnMouseMiddleDown)
    EVT_LEFT_UP(TestGLCanvas::OnMouseLeftUp)
    EVT_RIGHT_UP(TestGLCanvas::OnMouseRightUp)
END_EVENT_TABLE()

unsigned long  TestGLCanvas::m_secbase = 0;
int            TestGLCanvas::m_TimeInitialized = 0;
unsigned long  TestGLCanvas::m_xsynct;
unsigned long  TestGLCanvas::m_gsynct;

#define CHOOSE_VERTEX 0
#define CHOOSE_BOUND 1
#define CHOOSE_HANDLE 2
#define CHOOSE_TRANS 3

TestGLCanvas::TestGLCanvas(wxWindow *parent, wxWindowID id,
    const wxPoint& pos, const wxSize& size, long style, const wxString& name)
    : wxGLCanvas(parent, (wxGLCanvas*) NULL, id, pos, size, style|wxFULL_REPAINT_ON_RESIZE , name )
{
    m_init = false;
    m_gllist = 0;
    m_rleft = WXK_LEFT;
    m_rright = WXK_RIGHT;
    pwin = parent;

    printf("CONS\n");
    m_prevX = m_prevY = 0;
    trackball(m_q, 0.0f, 0.0f, 0.0f, 0.0f);
    trackball(m_qTrans, 0.0f, 0.0f, 0.0f, 0.0f);
    initModel(&m_model, filename);

    m_zDist = 1.5f;
    m_xDist = 0;
    m_yDist = 0;

    m_show_vert = -1;
    m_choose_state = CHOOSE_VERTEX;
    m_bv1=-1;
    m_bv2=-1;

    m_draw_modified = 0;
    m_show_local_frames = 0;
    m_show_mod_frames = 0;
    m_show_mod_set = 0;
    m_polygon_mode = 0;     // wireframe

    /*
    m_choosing_bound = 0;
    m_bound_init = 0;
    for (int i=0; i < 2; i++)
	m_bvinit[i] = 0;
    bound_set.clear();
    m_showPath = 0;
    m_showFrames = 0;
    m_choosing_conn = 0;
    conn_comp_set.clear();
    m_matM = m_matN = 0;
    A = NULL;
    B = NULL;
    //X = NULL;
    qr = NULL;
    m_choosing_handle = 0;
    handle_set.clear();

    Vector3D rv = {1,1,1};
    rotMatrix(PI/2, &rv, &trans_m[0][0]);
    //rotMatrix(0, &rv, &trans_m[0][0]);
    m_show_mod_frames = 0;

    // coefs:
    Ac = NULL;
    Bc = NULL;
    qrc = NULL;
    m_got_coords = 0;
    */
    Refresh(false);
}

TestGLCanvas::TestGLCanvas(wxWindow *parent, const TestGLCanvas *other,
    wxWindowID id, const wxPoint& pos, const wxSize& size, long style,
    const wxString& name )
    : wxGLCanvas(parent, other->GetContext(), id, pos, size, 
            style|wxFULL_REPAINT_ON_RESIZE , name)
{
    m_init = false;
    m_gllist = other->m_gllist; // share display list
    m_rleft = WXK_LEFT;
    m_rright = WXK_RIGHT;

    printf("CONS2: NOTHING IS INITIALIZED!!!\n");
    //initModel(&m_model);
    //m_prevX = m_prevY = 0;
    //trackball(m_q, 0.0f, 0.0f, 0.0f, 0.0f);
}

TestGLCanvas::~TestGLCanvas()
{
    //free(m_model.polbuf);
}

void TestGLCanvas::Render()
{
    wxPaintDC dc(this);

#ifndef __WXMOTIF__
    if (!GetContext()) return;
#endif
    SetCurrent();

    // Init OpenGL once, but after SetCurrent
    if (!m_init)
    {
	InitGL();
	m_init = true;
    }

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective( 45.0, 1.3, 0.01, 10000.0 );
    glMatrixMode(GL_MODELVIEW);

    /* clear color and depth buffers */
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);


    // Transformations
    GLfloat m[4][4];
    glLoadIdentity();
    glTranslatef( m_xDist, m_yDist, -m_zDist );
    build_rotmatrix( m,m_q);
    glMultMatrixf( &m[0][0] );

    // draw model
    if (m_draw_modified)
	m_model.drawModified(handle_set,mod_set);
    else
	m_model.drawOrigFlat();

    if (m_show_mod_set) {
	m_model.drawSet(mod_set);
    }
    
    if (m_show_vert >= 0) {
	m_model.drawVertexFan(m_show_vert);
    }

    if (!handle_set.empty()) {
	glColor3f(0,1,1);
	m_model.drawSet(handle_set);
    }

    // draw vertices defining a path if bound is being chosen
    if (m_choose_state == CHOOSE_BOUND) {
	glColor3f(1,0,0);
	if (m_bv1 >= 0) {
	    m_model.drawVertex(m_bv1);
	}
	if (m_bv2 >= 0) {
	    m_model.drawVertex(m_bv2);
	}
	if (m_bv1 >= 0 && m_bv2 >= 0 && path.size() != 0) {
	    m_model.drawPath(path);
	}
    }

    // draw the bound set itself
    if (!bound_set.empty()) {
	glColor3f(0,0,1);
	m_model.drawSet(bound_set);
    }

    if (m_show_mod_frames) {
	m_model.drawModifiedFrames(bound_set,handle_set,mod_set);
    }
    if (m_show_local_frames) {
	m_model.drawLocalFrames();
    }

    if (m_choose_state == CHOOSE_TRANS) {
        m_model.drawModifiedFrames(handle_set);
    }

    glFlush();
    SwapBuffers();
}

void TestGLCanvas::OnEnterWindow( wxMouseEvent& WXUNUSED(event) )
{
    SetFocus();
}

void TestGLCanvas::OnPaint( wxPaintEvent& WXUNUSED(event) )
{
    Render();
}

void TestGLCanvas::OnSize(wxSizeEvent& event)
{
    // this is also necessary to update the context on some platforms
    wxGLCanvas::OnSize(event);

    // set GL viewport (not called by wxGLCanvas::OnSize on all platforms...)
    int w, h;
    GetClientSize(&w, &h);
#ifndef __WXMOTIF__
    if (GetContext())
#endif
    {
        SetCurrent();
        glViewport(0, 0, (GLint) w, (GLint) h);
    }
}

void TestGLCanvas::OnEraseBackground(wxEraseEvent& WXUNUSED(event))
{
  // Do nothing, to avoid flashing.
}

void TestGLCanvas::InitGL()
{
    SetCurrent();
    initGL();
}

GLfloat TestGLCanvas::CalcRotateSpeed( unsigned long acceltime )
{
  GLfloat t,v;

  t = ((GLfloat)acceltime) / 1000.0f;

  if( t < 0.5f )
    v = t;
  else if( t < 1.0f )
    v = t * (2.0f - t);
  else
    v = 0.75f;

  return(v);
}

GLfloat TestGLCanvas::CalcRotateAngle( unsigned long lasttime,
                                  unsigned long acceltime )
{
    GLfloat t,s1,s2;

    t = ((GLfloat)(acceltime - lasttime)) / 1000.0f;
    s1 = CalcRotateSpeed( lasttime );
    s2 = CalcRotateSpeed( acceltime );

    return( t * (s1 + s2) * 135.0f );
}

void TestGLCanvas::Action( long code, unsigned long lasttime,
                           unsigned long acceltime )
{
    GLfloat angle = CalcRotateAngle( lasttime, acceltime );

    if (code == m_rleft)
        Rotate( angle );
    else if (code == m_rright)
            Rotate( -angle );
}

void TestGLCanvas::OnKeyDown( wxKeyEvent& event )
{
    long evkey = event.GetKeyCode();
    if (evkey == 0) return;

    if (!m_TimeInitialized)
    {
        m_TimeInitialized = 1;
        m_xsynct = event.GetTimestamp();
        m_gsynct = StopWatch(&m_secbase);

        m_Key = evkey;
        m_StartTime = 0;
        m_LastTime = 0;
        m_LastRedraw = 0;
    }

    Refresh(false);

    unsigned long currTime = event.GetTimestamp() - m_xsynct;

    if (evkey != m_Key)
    {
        m_Key = evkey;
        m_LastRedraw = m_StartTime = m_LastTime = currTime;
    }

    if (currTime >= m_LastRedraw)      // Redraw:
    {
        Action( m_Key, m_LastTime-m_StartTime, currTime-m_StartTime );

#if defined(__WXMAC__) && !defined(__DARWIN__)
        m_LastRedraw = currTime;    // StopWatch() doesn't work on Mac...
#else
        m_LastRedraw = StopWatch(&m_secbase) - m_gsynct;
#endif
        m_LastTime = currTime;
    }

    switch (event.GetKeyCode()) {
    case 'b': case 'B':
	if (m_choose_state == CHOOSE_BOUND)
	    m_choose_state = CHOOSE_VERTEX;
	else m_choose_state = CHOOSE_BOUND;
	m_bv1 = m_bv2 = -1;
	path.clear();
	break;
    case 'c': case 'C':
	bound_set.clear();
	m_bv1 = m_bv2 = -1;
	break;
    case 'h': case 'H':
	if (m_choose_state == CHOOSE_HANDLE)
	    m_choose_state = CHOOSE_VERTEX;
	else m_choose_state = CHOOSE_HANDLE;
	m_bv1 = m_bv2 = -1;
	path.clear();
	break;
    default:
	break;
    }
    
    event.Skip();
}

void TestGLCanvas::OnKeyUp( wxKeyEvent& event )
{
    m_Key = 0;
    m_StartTime = 0;
    m_LastTime = 0;
    m_LastRedraw = 0;

    switch (event.GetKeyCode()) {
    case 'l': case 'L':
	m_show_local_frames = !m_show_local_frames;
	break;
    case 's': case 'S':	    // start processing
        if (m_choose_state != CHOOSE_TRANS) {
            m_choose_state = CHOOSE_TRANS;
            trackball(m_qTrans, 0.0f, 0.0f, 0.0f, 0.0f);
            for (set<ushort>::iterator i=handle_set.begin(); 
                    i != handle_set.end(); i++)
                for (int j=0; j < 3; j++)
                    VEC_COPY(m_model.verts[*i].mvframe[j],
                             m_model.verts[*i].vframe[j]);
            Refresh(false);
        } else {
            m_choose_state = CHOOSE_VERTEX;
            m_bv1 = m_bv2 = -1;
            path.clear();
            if (!handle_set.empty()) {
                mod_set.clear();
                set<ushort> tmp_set;
                m_model.getConnComp(*(handle_set.begin()), bound_set, tmp_set);
                // get rid of handles
                set_difference(tmp_set.begin(), tmp_set.end(),
                        handle_set.begin(), handle_set.end(),
                        inserter(mod_set, mod_set.begin()));
                m_model.solve(bound_set, handle_set, mod_set);
            }
        }
	break;
    case 'f': case 'F':
        m_polygon_mode = !m_polygon_mode;
        if (m_polygon_mode == 1)
            glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
        else
            glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
        break;
    case 'p': case 'P':
	m_show_mod_set = !m_show_mod_set;
	break;
    case 'm': case 'M':
	m_show_mod_frames = !m_show_mod_frames;
	break;
    case '1':
	m_draw_modified = !m_draw_modified;
	break;
    default:
	break;
    }

    event.Skip();
    Refresh(false);
}

void TestGLCanvas::OnMouseLeftDown(wxMouseEvent& event)
{
    //wxMessageDialog ms(this, "bla bla");
    //ms.ShowModal();
    event.Skip();
}

void TestGLCanvas::OnMouseLeftUp(wxMouseEvent& event)
{

}

void TestGLCanvas::OnMouseMiddleDown(wxMouseEvent& event)
{
    if (m_choose_state == CHOOSE_BOUND) {
	int v = m_model.chooseVertex(event.GetX(),event.GetY());
	bound_set.erase((ushort)v);
    }
    else if (m_choose_state == CHOOSE_HANDLE) {
	int v = m_model.chooseVertex(event.GetX(),event.GetY());
	handle_set.erase((ushort)v);
    }
    Refresh(false);
}

void TestGLCanvas::OnMouseRightDown(wxMouseEvent& event)
{
    if (m_choose_state == CHOOSE_BOUND) {
	int v = m_model.chooseVertex(event.GetX(),event.GetY());
	// first, path is just one vertex
	m_bv1 = m_bv2 = v;
	path.clear();
	path.push_front(v);
    }
    else if (m_choose_state == CHOOSE_HANDLE) {
	int v = m_model.chooseVertex(event.GetX(),event.GetY());
	// if this vertex is not also in boundary set, add it to handle
	if (v >= 0 && bound_set.find((ushort)v) == bound_set.end())
	    handle_set.insert((ushort)v);
    }
    else if (m_choose_state == CHOOSE_VERTEX) {
	m_show_vert = m_model.chooseVertex(event.GetX(),event.GetY());
    }
    /*
    printf("\nCHOSE VERTEX: %d\n", m_show_vert);
    if (m_show_vert >= 0) {
	ushort v = m_show_vert;
	printf("edge_num = %d\n", m_model.verts[v].adj_num);
	for (int a=0; a < m_model.verts[v].adj_num; a++) {
	    ushort e = m_model.verts[v].edges[a];
	    edge_t *ep = &m_model.edges[e];
	    printf("[%d]: edge=%d, n=%d, (A,B)=(%d,%d), (a,b)=(%d,%d), r=%d\n",
		    a,m_model.verts[v].edges[a],
		    m_model.edgeOtherVert(e,v),
		    ep->A,ep->B,ep->a,ep->b,m_model.edgeOutFaceA(e,v));
	}
    }
    */
    Refresh(false);
}

void TestGLCanvas::OnMouseRightUp(wxMouseEvent& event)
{
    if (m_choose_state == CHOOSE_BOUND) {
	// if released mouse and a valid path is chosen, add it to bound set
	if (m_bv1 >= 0 && m_bv2 >= 0 && path.size() != 0) {
	    for (list<ushort>::iterator i=path.begin(); i != path.end(); i++)
		// if not in handle set, add it to boundary
		if (handle_set.find(*i) == handle_set.end())
		    bound_set.insert(*i);
	}
	m_bv1 = m_bv2 = -1;
	path.clear();
    }
    Refresh(false);
}

void TestGLCanvas::OnMouseMotion(wxMouseEvent& event)
{
    if (m_choose_state == CHOOSE_TRANS 
            &&event.Dragging() && event.RightIsDown()) {
        wxSize sz( GetClientSize() );
        /* drag in progress, simulate trackball */
        float spin_quat[4];
        trackball(spin_quat,
                (2.0*m_prevX -       sz.x) / (float)sz.x,
                (     sz.y - 2.0*m_prevY) / (float)sz.y,
                (     2.0*event.GetX() - sz.x) / (float)sz.x,
                (    sz.y - 2.0*event.GetY()) / (float)sz.y);

        add_quats( spin_quat, m_qTrans, m_qTrans);
        GLfloat tm[4][4];
        build_rotmatrix(tm, m_qTrans);
        for (set<ushort>::iterator i=handle_set.begin();
                i != handle_set.end(); i++)
            for (int j=0; j < 3; j++)
                multMatVec(&tm[0][0],
                    &m_model.verts[*i].vframe[j],
                    &m_model.verts[*i].mvframe[j]);
        Refresh(false);
    } 
    else if ( event.Dragging() && event.LeftIsDown())
    {
        if (event.ShiftDown()) {
            m_zDist += (float)(m_prevY-event.GetY())*0.01f;
        }
	else if (event.ControlDown()) {
	    m_xDist -= (float)(m_prevX-event.GetX())*0.005f;
	    m_yDist += (float)(m_prevY-event.GetY())*0.005f;
	}
        else {
            wxSize sz( GetClientSize() );
            /* drag in progress, simulate trackball */
            float spin_quat[4];
            trackball(spin_quat,
                    (2.0*m_prevX -       sz.x) / (float)sz.x,
                    (     sz.y - 2.0*m_prevY) / (float)sz.y,
                    (     2.0*event.GetX() - sz.x) / (float)sz.x,
                    (    sz.y - 2.0*event.GetY()) / (float)sz.y);

            add_quats( spin_quat, m_q, m_q);
            /* orientation has changed, redraw mesh */
        }
	Refresh(false);
    } 
    else if (event.Dragging() && event.RightIsDown()) {
	// if choosing bound, then find path to define it
	if (m_choose_state == CHOOSE_BOUND) {
	    // if first vertex on path is defined, find second (with path)
	    if (m_bv1 >= 0) {
		int v = m_model.chooseVertex(event.GetX(),event.GetY()); 
		m_bv2 = v;
		if (m_bv2 >= 0) {
		    path.clear();
		    m_model.findPath(m_bv1,m_bv2,path);
		}
	    }
	}
	Refresh(false);
    }

    m_prevX = event.GetX();
    m_prevY = event.GetY();
}

void TestGLCanvas::OnMouseEvent(wxMouseEvent& event)
{
    if (event.ButtonDown(wxMOUSE_BTN_LEFT))
    {
	
    }
}

void TestGLCanvas::Rotate( GLfloat deg )
{
    SetCurrent();

    glMatrixMode(GL_MODELVIEW);
    glRotatef((GLfloat)deg, 0.0f, 0.0f, 1.0f);
    Refresh(false);
}


/* -----------------------------------------------------------------------
  Main Window
-------------------------------------------------------------------------*/

BEGIN_EVENT_TABLE(MyFrame, wxFrame)
    EVT_MENU(wxID_EXIT, MyFrame::OnExit)
    EVT_MENU( ID_NEW_WINDOW, MyFrame::OnNewWindow)
    EVT_MENU( ID_DEF_ROTATE_LEFT_KEY, MyFrame::OnDefRotateLeftKey)
    EVT_MENU( ID_DEF_ROTATE_RIGHT_KEY, MyFrame::OnDefRotateRightKey)
END_EVENT_TABLE()

// My frame constructor
MyFrame::MyFrame(wxWindow *parent, const wxString& title, const wxPoint& pos,
    const wxSize& size, long style)
    : wxFrame(parent, wxID_ANY, title, pos, size, style)
{
    m_canvas = NULL;
    //SetIcon(wxIcon(sample_xpm));
}

// Intercept menu commands
void MyFrame::OnExit( wxCommandEvent& WXUNUSED(event) )
{
    // true is to force the frame to close
    Close(true);
}

MyFrame *MyFrame::Create(MyFrame *parentFrame, bool isCloneWindow)
{
    wxString str = wxT("wxWidgets OpenGL Cube Sample");
    if (isCloneWindow) str += wxT(" - Clone");

    MyFrame *frame = new MyFrame(NULL, str, wxDefaultPosition,
        wxSize(800, 600));

    // Make a menubar
    wxMenu *winMenu = new wxMenu;

    winMenu->Append(wxID_EXIT, _T("&Close"));
    winMenu->Append(ID_NEW_WINDOW, _T("&New") );
    wxMenuBar *menuBar = new wxMenuBar;
    menuBar->Append(winMenu, _T("&Window"));

    winMenu = new wxMenu;
    winMenu->Append(ID_DEF_ROTATE_LEFT_KEY, _T("Rotate &left"));
    winMenu->Append(ID_DEF_ROTATE_RIGHT_KEY, _T("Rotate &right"));
    menuBar->Append(winMenu, _T("&Key"));

    //frame->SetMenuBar(menuBar);

    if (parentFrame)
    {
        frame->m_canvas = new TestGLCanvas( frame, parentFrame->m_canvas,
            wxID_ANY, wxDefaultPosition, wxDefaultSize );
    }
    else
    {
        frame->m_canvas = new TestGLCanvas(frame, wxID_ANY,
            wxDefaultPosition, wxDefaultSize);
    }

    // Show the frame
    frame->Show(true);

    return frame;
}

void MyFrame::OnNewWindow( wxCommandEvent& WXUNUSED(event) )
{
    (void) Create(this, true);
}

void MyFrame::OnDefRotateLeftKey( wxCommandEvent& WXUNUSED(event) )
{
    ScanCodeDialog dial( this, wxID_ANY, m_canvas->m_rleft,
        wxString(_T("Left")), _T("Define key") );

    int result = dial.ShowModal();

    if( result == wxID_OK )
        m_canvas->m_rleft = dial.GetValue();
}

void MyFrame::OnDefRotateRightKey( wxCommandEvent& WXUNUSED(event) )
{
    ScanCodeDialog dial( this, wxID_ANY, m_canvas->m_rright,
        wxString(_T("Right")), _T("Define key") );

    int result = dial.ShowModal();

    if( result == wxID_OK )
        m_canvas->m_rright = dial.GetValue();
}

/*------------------------------------------------------------------
  Application object ( equivalent to main() )
------------------------------------------------------------------ */

IMPLEMENT_APP(MyApp)

bool MyApp::OnInit()
{
    glutInit(&argc, argv);

    if (argc != 2) {
        printf("USAGE: test filename\n");
        return false;
    }

    strcpy(filename, argv[1]);
    printf("FILENAME: %s\n", filename);

    // Create the main frame window
    (void) MyFrame::Create(NULL);

    return true;
}
