
#ifndef MESH_H_
#define MESH_H_

#include <iostream>
#include <set>
#include <iostream>
#include <queue>
#include <list>
#include <stack>
#include <algorithm>
using namespace std;

//#include "/home/mike/private/programming/jama/tnt.h"
//#include "/home/mike/private/programming/jama/jama_qr.h"
#include "jama/tnt.h"
#include "jama/jama_qr.h"
using namespace TNT;
using namespace JAMA;


#define MAX_VERTICES 8000 // Max number of vertices (for each object)
#define MAX_POLYGONS 8000 // Max number of polygons (for each object)

#define INVALID_VERT ((ushort)-1)
#define INVALID_EDGE ((ushort)-1)


// vertex, stores indieces of all edges emanating from it
// so not "model-constant" as original winged-edge data structure
// (memory for 'edges' is allocated from one big array
// for all vertices at once in buildStructure())
typedef struct {
    Vector3D p;	    // position of vertex
    ushort *edges;  // indices of edges bounding this vert
    ushort adj_num; // number of neighbors (size of edges)

    // local frame (original)
    union {
	struct {
	    Vector3D T,B,N; 
	};
	float aframe[9];
	float mframe[3][3];
	Vector3D vframe[3];
    };
    // local frame (modified)
    union {
	struct {
	    Vector3D mT, mB, mN;
	};
	float maframe[9];
	float mmfame[3][3];
	Vector3D mvframe[3];
    };

    // coefficients giving each edge in terms of neighbor frame
    // Ex: for int i, 0 <= i < adj_num, suppose this vertex is v
    // and v.edges[i].a == v, so neighbor of v is n=v.edges[i].b
    // (in reality its Mesh.edges[v.edges[i]].a since v.edges[i] is edge index)
    // then q[i] is a vector giving coefficients of edge n-v in terms of frame at n
    // so: n-v = q[i].x*n.T + q[i].y*n.B + q[i].z*n.N
    // (important that original frame {T,B,N} is used)
    Vector3D *q;	    // each edge in terms of neighbors frame

    ushort parent;	    // for path finding
    float dist;		    // path finding
    unsigned char space;    // for use by some algs
} vertex_t;


// face doesnt store much: 3 indices to its vertices
// as well as normal to this face
// (normals at vertices are averages of their adjacent faces)
typedef struct {
    union {
	struct {
	    ushort a,b,c;   // indices of vertices of this triangle
	};
	ushort ind[3];      // indices as array (for inside loops)
    };
    Vector3D N;             // normal to this face
} face_t;


// edge is the main structure of winged-edge representation
// stores the two vertex indices, two face indices (on each side)
// as well as two edges of each adjacent face
// Also has the coefficients of expressing local frame
// at vertex a in terms of local frame at vertex b (so this
// computation is done once per edge, not once per vertex)
typedef struct {
    ushort a,b;	    // vertices of this edge (a,b)
    ushort A,B;	    // right and left face respectively
    ushort aC, aCC; // other edges of right and left faces
    ushort bC, bCC; // C - clockwise, CC - counter clockwise
    float length;   // length of the original edge

    // coefficients, express frame at a in terms of b:
    // union for easier access in different ways
    union {
        // give each coefficient as a vector:
        // Ex: a.T = cT.x*b.T + cT.y*b.B + cT.z*b.N
	struct {
	    Vector3D cT,cB,cN;  
	};
        // same vectors only access as one linear array
	float acoefs[9];
        // same vectors, only now access as a matrix
        // C stores matrices by row (iterating over columns)
	float mcoefs[3][3]; // vectors are rows
	Vector3D vcoefs[3]; // same vectors in one vector-array
    };
} edge_t;


// The mapcoord type, 2 texture coordinates for each vertex
typedef struct{
    float u,v;
} mapcoord_type;






// Main mesh class, stores model as a variant of static 
// winged-edge data structure
// almost everything is public for easier access
// memory for the whole class is allocated in one whole 'chunk'
// and then redistributed among vertices as needed
class Mesh
{
    // used to hold adjacent lists for each vertex
    ushort adjacent_buf[3*MAX_POLYGONS];
    // used to hold edge coefficients for each vertex
    Vector3D q_buf[3*MAX_POLYGONS];

    // for shortest path comparison object
    class CompareDist
    { 
	Mesh *mesh;
    public: 
	CompareDist() {}
	CompareDist(Mesh *m) : mesh(m) {}
	int operator() (int v1, int v2);
    };

    // maps vertex into position in Xf-vector
    int vind[MAX_VERTICES];

public:
    Mesh();
    
    char name[20];
    vertex_t verts[MAX_VERTICES];
    face_t faces[MAX_POLYGONS];
    edge_t edges[MAX_VERTICES];
    float avg_edge_len;

    // for frames
    Array2D<float> *Af;
    Array1D<float> *Bf;
    Array1D<float> Xf;
    QR<float> *qrf;
    Array2D<float> *Ac;
    Array1D<float> *Bc;
    Array1D<float> Xc;
    QR<float> *qrc;
    

    void scaleModel();
    void buildStructure();
    void merge();
    void calcNormals();
    void calcTB(int v);
    void calcTB();
    void calcLocalCoefs();
    void getConnComp(ushort v, const set<ushort> &bound, set<ushort> &s);
    void solve(const set<ushort> &bound_set, 
		const set<ushort> &handle_set,
		const set<ushort> &mod_set);
    void setAf(const set<ushort> &bound_set, 
		const set<ushort> &handle_set,
		const set<ushort> &mod_set);
    void setAc(const set<ushort> &bound_set, 
		const set<ushort> &handle_set,
		const set<ushort> &mod_set);
    void drawModifiedFrames(const set<ushort> &bound_set, 
	const set<ushort> &handle_set,
	const set<ushort> &mod_set);
    void drawModifiedFrames(const set<ushort> &s);

    ushort vertNeighbor(ushort v, ushort a);
    ushort vertFindNeighbor(ushort v, ushort n);
    ushort vertFindEdge(ushort v, ushort e);

    int edgeInVert(ushort e, ushort v);
    int edgeOutVert(ushort e, ushort v);
    ushort edgeOtherVert(ushort e, ushort v);
    ushort edgeOutFaceA(ushort e, ushort v);
    ushort edgeOutFaceB(ushort e, ushort v);
    
    void drawModified(const set<ushort> &handle_set,
			const set<ushort> &mod_set);


    void drawVertex(int v);
    void drawVertexFan(int v);
    void drawOrigAverage();
    void drawOrigFlat();
    int chooseVertex(int x, int y);
    int findPath(int p1, int p2, list<ushort> &path);
    void drawPath(const list<ushort> &path);
    void drawSet(const set<ushort> &s);
    void drawLocalFrame(int v, float c=1.0f);
    void drawLocalFrames();


    void printLocalFrame(int v);
    void printRelFrame(int v1, int v2);

    mapcoord_type mapcoord[MAX_VERTICES];
    int vert_num;
    int face_num;
    int edge_num;
    //int id_texture;
};


void initModel(Mesh *model, char *filename);
void initGL();
/*
int chooseVertex(obj_type_ptr o, float *q, int x, int y);
void calcNormals(obj_type_ptr o);
int findPath(obj_type_ptr o, int p1, int p2, int *m_vbound, 
	int *m_tris_num, int *m_tris, 
	Vector3D &m_T, Vector3D &m_B, Vector3D &m_N);
void build_conn_str(obj_type_ptr o);
void mergeVertices(obj_type_ptr o);
void calcTB(obj_type_ptr o, int v);
void calcTB(obj_type_ptr o);
void calcLocalCoefs(obj_type_ptr o);
void getConnComp(obj_type_ptr o, int v, 
	const set<int> &bound, set<int> &s);
void showModifiedFrames(obj_type_ptr o, const set<int> &s, int *i2m, 
				    const Array1D<float> &X, float s);
void printLocalFrame(obj_type_ptr o, int v);
void printRelFrame(obj_type_ptr o, int v1, int v2);
*/
void scaleModel(Mesh *o);

#endif
