
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>
#include <stdio.h>
#include <stdlib.h>
#include "trackball.h"
#include <math.h>
#include "math3d.h"
#include "mesh.h"
#include "3dsloader.h"
#include <limits.h>


#define RADIUS_EDGE_FRAC 0.1

Mesh::Mesh() :
    Af(NULL), Bf(NULL), qrf(NULL),
    Ac(NULL), Bc(NULL), qrc(NULL)
{

}


// used to compare the distances between vertices v1 & v2
// while searching for a shortest path on mesh
int Mesh::CompareDist::operator() (int v1, int v2)
{
    return mesh->verts[v1].dist > mesh->verts[v2].dist;
}


void Mesh::printLocalFrame(int v)
{
    printf("Frame for %d:\n", v);
    printf("\t"); printVector3D(verts[v].T);
    printf("\t"); printVector3D(verts[v].B);
    printf("\t"); printVector3D(verts[v].N);
}

void Mesh::printRelFrame(int v1, int v2)
{
    int a=vertFindNeighbor(v1,v2);
    if (a == INVALID_VERT) {
	printf("Vertices %d and %d are not neighbors!\n",v1,v2);
	return;
    }
    vertex_t *vp1 = &verts[v1];
    printVector3D(edges[vp1->edges[a]].cT);
    printVector3D(edges[vp1->edges[a]].cB);
    printVector3D(edges[vp1->edges[a]].cN);
}

// loads the model and calculates needed data structures within it
void initModel(Mesh *model, char *filename)
{
    Load3DS(model, filename);
    //Load3DS(model, "cube2.3ds");
    //Load3DS(model, "At-pt.3ds");
    //Load3DS(model, "myship.3ds");
    //Load3DS(model, "violin.3ds");
    //Load2DS(model, "chesspawn.3ds");
    //Load3DS(model, "d1.3ds");
    //Load3DS(model, "spaceship.3ds");
    //Load3DS(model, "helicopter.3ds");

    printf("sizeof vert = %d\n", sizeof(vertex_t));

    model->merge();
    model->scaleModel();
    model->buildStructure();
    model->calcNormals();
    model->calcTB();
    model->calcLocalCoefs();
}

void initGL()
{
    /* set viewing projection */
    glMatrixMode(GL_PROJECTION);
    gluPerspective( 45.0, 1.3, 1.0, 100.0 );

    /* position viewer */
    glMatrixMode(GL_MODELVIEW);
    glTranslatef(0.0f, 0.0f, -2.0f);

    /* position object */
    glRotatef(30.0f, 1.0f, 0.0f, 0.0f);
    glRotatef(30.0f, 0.0f, 1.0f, 0.0f);

    glEnable(GL_DEPTH_TEST);
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);

    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    //glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);

}



// calculate the local frame at this vertex
// should be called after buildStructure and calcNormals
void Mesh::calcTB(int v)
{
    float m;
    Vector3D x,N;
    vertex_t *vp = &verts[v];
    vertex_t *ep = &verts[vertNeighbor(v,0)];
    VEC_SUB_VEC(ep->p,vp->p,x);
    m = DOT(x,vp->N);
    VEC_MULT_SCALAR(vp->N,m,N);
    VEC_SUB_VEC(x,N,vp->T);
    VEC_NORM(vp->T,m);
    CROSS(vp->N,vp->T,vp->B);
    VEC_NORM(vp->B,m);

    //printf("NORM: %f %f %f\n", DOT(vnormals[v],vnormals[v]),
	    //DOT(vT[v],vT[v]), DOT(vB[v],vB[v]));
}

// calculate local frames for all vertices
// should be called after build_conn_str and calcNormals
void Mesh::calcTB()
{
    for (int i=0; i < vert_num; i++)
	calcTB(i);
}

// for each vertex, express local frames in terms of its neighbors
void Mesh::calcLocalCoefs()
{
    vertex_t *va, *vb;
    edge_t *ep;
    // for each edge, calculate local coefficients
    // of vertex e.a in terms of frame at vertex e.b
    for (ushort e=0; e < edge_num; e++) {
	ep = &edges[e];
	va = &verts[ep->a];
	vb = &verts[ep->b];
	for (int r=0; r < 3; r++)
	    for (int c=0; c < 3; c++)
		ep->mcoefs[r][c] = DOT(va->vframe[r],vb->vframe[c]);
    }
    
    // for each vertex v and its neighbor a, 
    // express edge (v,a) in local frame of a
    // (to optimize, iterate over edges)
    Vector3D e;
    Vector3D *buf = q_buf;
    vertex_t *vp, *ap;
    for (int v=0; v < vert_num; v++) {
	vp = &verts[v];
	vp->q = buf;	    // set buffer aside to hold coefficients
	buf += vp->adj_num;
	// for each neighbor
	for (int a=0; a < vp->adj_num; a++) {
	    ap = &verts[vertNeighbor(v,a)];
	    VEC_SUB_VEC(vp->p,ap->p,e);	// get the edge (v,a)
	    for (int i=0; i < 3; i++)
		vp->q[a].a[i] = DOT(e,vp->vframe[i]);
	}
    }
}

// get all the vertices reachable from v
// NOT containing vertices from set bound
// all inserted into s
// (used to search for the connected component which is going to be modified)
void Mesh::getConnComp(ushort v, const set<ushort> &bound, set<ushort> &s)
{
    printf("connComp: v = %d\n", v);
    printf("bound size = %d\n", bound.size());
    vertex_t *vp;
    stack<int> queue;
    queue.push(v);
    ushort va;
    while (!queue.empty()) {
	v = queue.top();
	queue.pop();
	vp = &verts[v];
	s.insert(v);
	for (int a=0; a < vp->adj_num; a++) {
	    // if not in boundary, insert this neighbor
	    va = vertNeighbor(v,a);
	    if (bound.find(va) == bound.end()
		    && s.find(va) == s.end())
		queue.push(va);
	}
    }
}

// draw modified frame at 
void Mesh::drawModifiedFrames(const set<ushort> &s)
{
    GLboolean blight;
    glGetBooleanv(GL_LIGHTING, &blight);
    glDisable(GL_LIGHTING);
    glLineWidth(3);

    vertex_t *vp;
    float f = 0.5*avg_edge_len;
    float c = 1;
    Vector3D tv;
    for (set<ushort>::const_iterator i=s.begin(); i != s.end(); i++) {
        vp = &verts[*i];
        glPushMatrix();
        glTranslatef(vp->p.x,vp->p.y,vp->p.z);
        glBegin(GL_LINES);
        glColor3f(c,0,0);
        VEC_MULT_SCALAR(vp->mT,f,tv); glVertex3f(0,0,0); glVertex3fv(tv.a);
        glColor3f(0,c,0);
        VEC_MULT_SCALAR(vp->mB,f,tv); glVertex3f(0,0,0); glVertex3fv(tv.a);
        glColor3f(0,0,c);
        VEC_MULT_SCALAR(vp->mN,f,tv); glVertex3f(0,0,0); glVertex3fv(tv.a);
        glEnd();
        glPopMatrix();
    }

    glLineWidth(1);
    if (blight)
	glEnable(GL_LIGHTING);
}

// draws the modified frames for all vertices in these three sets
void Mesh::drawModifiedFrames(const set<ushort> &bound_set, 
	const set<ushort> &handle_set,
	const set<ushort> &mod_set)
{
    if (Xf.dim() == 0)
	return;
    GLboolean blight;
    glGetBooleanv(GL_LIGHTING, &blight);
    glDisable(GL_LIGHTING);
    glLineWidth(3);

    Vector3D tv;
    vertex_t *vp;
    float f = 0.5*avg_edge_len;
    for (set<ushort>::const_iterator i=handle_set.begin();
	    i != handle_set.end(); i++) {
	glPushMatrix();
	vp = &verts[*i];
	glTranslatef(vp->p.x,vp->p.y,vp->p.z);
	glBegin(GL_LINES);
	VEC_MULT_SCALAR(vp->mT,f,tv);
	glColor3f(0.5,0,0); glVertex3f(0,0,0); glVertex3fv(tv.a);
	VEC_MULT_SCALAR(vp->mB,f,tv);
	glColor3f(0,0.5,0); glVertex3f(0,0,0); glVertex3fv(tv.a);
	VEC_MULT_SCALAR(vp->mN,f,tv);
	glColor3f(0,0,0.5); glVertex3f(0,0,0); glVertex3fv(tv.a);
	glEnd();
	glPopMatrix();
    }

    int r = 0;
    for (set<ushort>::const_iterator i=mod_set.begin(); 
	    i!= mod_set.end(); i++,r+=9) {
	glPushMatrix();
	vp = &verts[*i];
	glTranslatef(vp->p.x,vp->p.y,vp->p.z);
	glBegin(GL_LINES);
	tv.x=Xf[r]; tv.y=Xf[r+1]; tv.z=Xf[r+2];
	VEC_SCALE(tv,f);
	glColor3f(0.5,0,0); glVertex3f(0,0,0); glVertex3fv(tv.a);
	tv.x=Xf[r+3]; tv.y=Xf[r+4]; tv.z=Xf[r+5];
	VEC_SCALE(tv,f);
	glColor3f(0,0.5,0); glVertex3f(0,0,0); glVertex3fv(tv.a);
	tv.x=Xf[r+6]; tv.y=Xf[r+7]; tv.z=Xf[r+8];
	VEC_SCALE(tv,f);
	glColor3f(0,0,0.5); glVertex3f(0,0,0); glVertex3fv(tv.a);
	glEnd();
	glPopMatrix();
    }
    glLineWidth(1);
    if (blight)
	glEnable(GL_LIGHTING);
}


// scale so max dimension is one
// and so that center is at the origin
void Mesh::scaleModel()
{
    float lmin[]={1.0E8,1.0E8,1.0E8};
    float lmax[]={-1.0E8,-1.0E8,-1.0E8};
    double mid[] = {0,0,0};
    float dim[3];
    for (int v=0; v < vert_num; v++) {
	for (int i=0; i < 3; i++) {
	    //mid[i] += verts[v].p.a[i];
	    if (verts[v].p.a[i] < lmin[i])
		lmin[i] = verts[v].p.a[i];
	    else if (verts[v].p.a[i] > lmax[i])
		lmax[i] = verts[v].p.a[i];
	}
    }
    for (int i=0; i < 3; i++) {
	dim[i] = lmax[i]-lmin[i];
	//mid[i] /= vert_num;
	mid[i] = (lmax[i]+lmin[i])/2;
	printf("[%d]: (%f,%f) dim=%f mid=%f\n", 
		i,lmin[i],lmax[i],dim[i],mid[i]);
    }
    float f = MAX(dim[0],dim[1]);
    f = MAX(f,dim[2]);
    for (int v=0; v < vert_num; v++)
	for (int i=0; i < 3; i++) {
	    verts[v].p.a[i] -= mid[i];
	    verts[v].p.a[i] /= f;
	}
}

// return vertex which is a neighbor at index v.edges[a]
ushort Mesh::vertNeighbor(ushort v, ushort a)
{
    if (a >= verts[v].adj_num)
	return INVALID_VERT;
    ushort e = verts[v].edges[a];
    if (edges[e].a == v)
	return edges[e].b;
    else return edges[e].a;
}

// return index of n in v.edges if n is a neighbor of v
ushort Mesh::vertFindNeighbor(ushort v, ushort n)
{
    for (ushort a=0; a < verts[v].adj_num; a++)
	if (vertNeighbor(v,a) == n)
	    return a;
    return INVALID_VERT;
}

// return the index (in respect to v.edges) of
// edge e (if it exists)
ushort Mesh::vertFindEdge(ushort v, ushort e)
{
    for (ushort a=0; a < verts[v].adj_num; a++)
	if (verts[v].edges[a] == e)
	    return a;
    return INVALID_VERT;
}

// return true if edge e points into vertex v
int Mesh::edgeInVert(ushort e, ushort v)
{
    return edges[e].b == v;
}

// return true if edge e leaves out of vertex v
int Mesh::edgeOutVert(ushort e, ushort v)
{
    return edges[e].a == v;
}

// build the winged-edge data structure
void Mesh::buildStructure()
{
    // init each vertex
    for (int v=0; v < face_num; v++)
	verts[v].adj_num = 0;
    // calculate how many triangles each vertex is adjacent to
    for (int t=0; t < face_num; t++) {
	face_t *p = &faces[t];
	verts[p->a].adj_num++;
	verts[p->b].adj_num++;
	verts[p->c].adj_num++;
    }
    // allocate space to hold edge indices for each vertex
    //polbuf = (int*)malloc(sizeof(int)*3*face_num);
    ushort *vbuf = adjacent_buf;
    // for each vertex, allocate space for adjacent list of edges
    for (int v=0; v < vert_num; v++) {
	verts[v].edges = vbuf;
	// also initialize all its adjacent edges to INVALID
	//for (int e=0; e < verts[v].adj_num; e++)
	    //vbuf[e] = INVALID_EDGE;
	vbuf+=verts[v].adj_num;
	verts[v].adj_num=0;	// set to 0 for further processing
    }
    // for each vertex, save faces its adjacent to in place of edges
    edge_num = 0;
    ushort eab, ebc, eca;
    ushort iab, ibc, ica;
    for (int t=0; t < face_num; t++) {
	face_t *p = &faces[t];
	vertex_t *va = &verts[p->a];
	vertex_t *vb = &verts[p->b];
	vertex_t *vc = &verts[p->c];
	int initab=0,initbc=0,initca=0;
	// edge (a,b)
	if ((iab = vertFindNeighbor(p->a,p->b)) == INVALID_VERT) {
	    eab = edge_num++;
	    va->edges[va->adj_num++] = eab;
	    vb->edges[vb->adj_num++] = eab;
	    edges[eab].a = p->a;
	    edges[eab].b = p->b;
	} else {
	    eab = va->edges[iab];
	    initab = 1;
	}
	// edge (b,c)
	if ((ibc = vertFindNeighbor(p->b,p->c)) == INVALID_VERT) {
	    ebc = edge_num++;
	    vb->edges[vb->adj_num++] = ebc;
	    vc->edges[vc->adj_num++] = ebc;
	    edges[ebc].a = p->b;
	    edges[ebc].b = p->c;
	} else {
	    ebc = vb->edges[ibc];
	    initbc = 1;
	}
	// edge (c,a)
	if ((ica = vertFindNeighbor(p->c,p->a)) == INVALID_VERT) {
	    eca = edge_num++;
	    vc->edges[vc->adj_num++] = eca;
	    va->edges[va->adj_num++] = eca;
	    edges[eca].a = p->c;
	    edges[eca].b = p->a;
	} else {
	    eca = vc->edges[ica];
	    initca = 1;
	}

	//printf("between %d\n", t);
	//printf("AB: e=%d\n", eab);
	//printf("BC: e=%d\n", ebc);
	//printf("CA: e=%d\n", eca);

	// should work for inconsistent triangle orientations
	// edge (a,b)
	//if (edgeOutVert(eab,p->a)) {
	if (!initab) {
	    edges[eab].A = t;
	    edges[eab].aC = ebc;
	    edges[eab].aCC = eca;
	} else {
	    edges[eab].B = t;
	    edges[eab].bC = ebc;
	    edges[eab].bCC = eca;
	}
	// edge (b,c)
	//if (edgeOutVert(ebc,p->b)) {
	if (!initbc) {
	    edges[ebc].A = t;
	    edges[ebc].aC = eca;
	    edges[ebc].aCC = eab;
	} else {
	    edges[ebc].B = t;
	    edges[ebc].bC = eca;
	    edges[ebc].bCC = eab;
	}
	// edges (c,a)
	//if (edgeOutVert(eca,p->c)) {
	if (!initca) {
	    edges[eca].A = t;
	    edges[eca].aC = eab;
	    edges[eca].aCC = ebc;
	} else {
	    edges[eca].B = t;
	    edges[eca].bC = eab;
	    edges[eca].bCC = ebc;
	}
    }
    // rearrange edges around each vertex in-order
    for (int v=0; v < vert_num; v++) {
	ushort e = verts[v].edges[0], a=0;
	do {
	    verts[v].edges[a] = e;
	    if (edgeInVert(e,v))
		e = edges[e].bCC;
	    else
		e = edges[e].aCC;
	    a++;
	} while (e != verts[v].edges[0]);
    }
    printf("\nHERE? num=%d\n\n", edge_num);
    // compute the length of each edges
    Vector3D tmp;
    avg_edge_len = 0;
    for (ushort e=0; e < edge_num; e++) {
	VEC_SUB_VEC(verts[edges[e].a].p,verts[edges[e].b].p,tmp);
	edges[e].length = VEC_MAG(tmp);
	avg_edge_len += edges[e].length;
    }
    avg_edge_len /= edge_num;

    printf("Building Strucutre Complete:\n");
    printf("edge_num = %d, supposed to be = %d\n",
	    edge_num, vert_num+face_num-2);
}   


// connects all the triangles into one connected mesh 
// (deals with problem of vertices being at the same position
// but having different index)
// this merges vertices that are close enough to each other
// and sets the face indices appropriately
#define MERGE_EPS 0.0001	// largest distance between merged vertices
void Mesh::merge()
{
    //obj_type_ptr res = (obj_type_ptr)malloc(sizeof(obj_type));
    int *a = (int*)malloc(sizeof(int)*vert_num);
    int old_vert_num = vert_num;

    for (int i=0; i < vert_num; i++) a[i] = i;
    int vnum = 0;
    for (int i=0; i < vert_num; i++) {
	vertex_t *vp = &verts[i];
	if (a[i] != i)	// already done
	    continue;
	verts[vnum].p.x = vp->p.x;
	verts[vnum].p.y = vp->p.y;
	verts[vnum].p.z = vp->p.z;
	a[i] = vnum;	// its possible that vnum < i, so move it
	for (int j=i+1; j < vert_num; j++) {
	    if (a[j] == j) {
		float dx = vp->p.x-verts[j].p.x;
		float dy = vp->p.y-verts[j].p.y;
		float dz = vp->p.z-verts[j].p.z;
		if (
			(dx >= -MERGE_EPS && dx <= MERGE_EPS) &&
			(dy >= -MERGE_EPS && dy <= MERGE_EPS) &&
			(dz >= -MERGE_EPS && dz <= MERGE_EPS)) {
		    a[j] = vnum;
		}
	    }
	}
	vnum++;
    }
    for (int p=0; p < face_num; p++) {
	face_t *pp = &faces[p];
	pp->a = a[pp->a];
	pp->b = a[pp->b];
	pp->c = a[pp->c];
    }
    vert_num = vnum;

    edge_num = vert_num+face_num-2;


    // WHY???
    //for (int i=0; i < vert_num; i++)
	//verts[i].index = i;
    free(a);

    printf("Done merging. Reduced from %d to %d vertices.\n", 
            old_vert_num, vert_num);
    printf("face_num = %d, edge_num = %d\n", face_num, edge_num);
}


// show vertex as a small sphere
void Mesh::drawVertex(int v)
{
    GLboolean blight;
    glGetBooleanv(GL_LIGHTING, &blight);
    glDisable(GL_LIGHTING);
    vertex_t *vp = &verts[v];
    glPushMatrix();
    glTranslatef(vp->p.x,vp->p.y,vp->p.z);
    glutSolidSphere(RADIUS_EDGE_FRAC*avg_edge_len,5,5);
    glPopMatrix();
    if (blight)
	glEnable(GL_LIGHTING);
}

// return vertex of edge e not equals to v
ushort Mesh::edgeOtherVert(ushort e, ushort v)
{
    return edges[e].a == v ? edges[e].b : edges[e].a;
}

// display the vertex and its adjacent triangles 
void Mesh::drawVertexFan(int v)
{
    GLboolean blight;
    glGetBooleanv(GL_LIGHTING, &blight);
    GLint pmode[2];
    glGetIntegerv(GL_POLYGON_MODE, pmode);

    glDisable(GL_LIGHTING);
    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    //glDisable(GL_DEPTH_TEST);

    vertex_t *vp = &verts[v];
    glColor3f(1,0,0);
    glPushMatrix();
    glTranslatef(vp->p.x,vp->p.y,vp->p.z);
    glutSolidSphere(RADIUS_EDGE_FRAC*avg_edge_len,5,5);
    glPopMatrix();

    // vertices fan around vp
    
    glLineWidth(3);
    glBegin(GL_TRIANGLES);
    for (int i=0; i < vp->adj_num; i++) {
	ushort e = vp->edges[i];
	ushort p = edgeOutFaceA(e,v);
	glVertex3fv(verts[faces[p].a].p.a);
	glVertex3fv(verts[faces[p].b].p.a);
	glVertex3fv(verts[faces[p].c].p.a);
    }
    glEnd();
    glLineWidth(1);

    //glEnable(GL_DEPTH_TEST);
    if (blight)
	glEnable(GL_LIGHTING);
    glPolygonMode(GL_FRONT, pmode[0]);
    glPolygonMode(GL_BACK, pmode[1]);
}

// pick a vertex using OpenGL's picking mechanism (from red book)
// at mouse position (x,y)
int Mesh::chooseVertex(int x, int y)
{
    int bufsize = 4*vert_num;
    unsigned *buf = (unsigned*)malloc(sizeof(unsigned)*bufsize);
    GLint viewport[4];
    int hits;
    glGetIntegerv (GL_VIEWPORT, viewport);

    glSelectBuffer (bufsize, buf);
    (void) glRenderMode (GL_SELECT);

    glInitNames();
    glPushName((unsigned)-1);

    glMatrixMode (GL_PROJECTION);
    glPushMatrix ();
    glLoadIdentity ();
    gluPickMatrix((GLdouble) x, 
	    (GLdouble) (viewport[3] - y), 20.0, 20.0, viewport);

    gluPerspective( 45.0, 1.3, 1.0, 700.0 );
    glMatrixMode(GL_MODELVIEW);

    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    for (int v=0; v < vert_num; v++) {
	glLoadName((GLuint)v);
	glBegin(GL_POINTS);
	glVertex3fv((GLfloat*)&verts[v]);
	glEnd();
    }
    
    glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    glFlush();

    hits = glRenderMode(GL_RENDER);

    //printf("hits = %d\n", hits);

    unsigned zhit = 0xffffffff;
    int vhit = -1;
    for (int h=0; h < 4*hits; h+=4) {
	if (buf[h] != 1) {
	    //printf("names of stack = %d!\n", buf[h]);
	    break;
	}
	//printf("h=%d: zmin = %d, zmax = %d, name = %d\n",
		//h, buf[h+1],buf[h+2],buf[h+3]);
	if (zhit > buf[h+1]) {
	    vhit = buf[h+3];
	    zhit = buf[h+1];
	}
    }

    free(buf);
    return vhit;
}

// return the A face if edge e leaves out of v
// (so if sitting at vertex v and going
// around its adjacent edges in order, then
// always returns 'right' face)
ushort Mesh::edgeOutFaceA(ushort e, ushort v)
{
    if (edges[e].a == v)
	return edges[e].A;
    return edges[e].B;
}

// this returns the 'left' face of edge e at vertex v
ushort Mesh::edgeOutFaceB(ushort e, ushort v)
{
    if (edges[e].a == v)
	return edges[e].B;
    return edges[e].A;
}

// calculate per-triangle and per-vertex normals
// normals at each vertex is average of its triangle normals
void Mesh::calcNormals()
{
    Vector3D v1, v2;
    vertex_t *a, *b, *c;
    face_t *pp;
    float mag;
    // per triangle normals:
    for (int p=0; p < face_num; p++) {
	pp = &faces[p];
	a = &verts[pp->a];
	b = &verts[pp->b];
	c = &verts[pp->c];
	//v1.x = b->p.x-a->p.x; v1.y = b->p.y-a->p.y; v1.z = b->p.z-a->p.z;
	//v2.x = c->p.x-a->p.x; v2.y = c->p.y-a->p.y; v2.z = c->p.z-a->p.z;
	VEC_SUB_VEC(b->p,a->p,v1);
	VEC_SUB_VEC(c->p,a->p,v2);
	CROSS(v1,v2,pp->N);
	VEC_NORM(pp->N,mag);
    }

    vertex_t *vp;
    // per vertex normals (average):
    for (int v=0; v < vert_num; v++) {
	vp = &verts[v];
	vp->N.x = vp->N.y = vp->N.z = 0;
	for (int e=0; e < vp->adj_num; e++) {
	    ushort p = edgeOutFaceA(vp->edges[e],v);
	    VEC_ADD_VEC(vp->N,faces[p].N,vp->N);
	}
	VEC_SCALE(vp->N,vp->adj_num);
	VEC_NORM(vp->N,mag);
    }
}

// draw original model with per-triangle normals
void Mesh::drawOrigFlat()
{
    glColor3f(1,1,1);
    glLineWidth(1);
    glBegin(GL_TRIANGLES);
    for (int p=0; p < face_num; p++) {
	int p1 = faces[p].a;
	int p2 = faces[p].b;
	int p3 = faces[p].c;
	glNormal3fv(faces[p].N.a);
	glVertex3fv(verts[p1].p.a);
	glVertex3fv(verts[p2].p.a);
	glVertex3fv(verts[p3].p.a);
    }
    glEnd();
}

// draw original model using normals at each vertex
void Mesh::drawOrigAverage()
{
    glColor3f(1,1,1);
    glLineWidth(1);
    glBegin(GL_TRIANGLES);
    for (int p=0; p < face_num; p++) {
	ushort p1 = faces[p].a;
	ushort p2 = faces[p].b;
	ushort p3 = faces[p].c;
	glNormal3fv(verts[p1].N.a);
	glVertex3fv(verts[p1].p.a);
	glNormal3fv(verts[p2].N.a);
	glVertex3fv(verts[p2].p.a);
	glNormal3fv(verts[p3].N.a);
	glVertex3fv(verts[p3].p.a);
    }
    glEnd();
}

#define VERT_SET_DONE(v) {v->space |= 0x1;}
#define VERT_SET_QUEUE(v) {v->space |= 0x2;}
#define VERT_IS_DONE(v) (v->space & 0x1)
#define VERT_IS_QUEUE(v) (v->space & 0x2)
// find a path on object from v1 to v2
// return a list of vertices on the path
int Mesh::findPath(int v1, int v2, list<ushort> &path)
{
    //priority_queue<int,deque<int>,CompareDist> pqueue;
    CompareDist compObject(this);
    priority_queue<int,deque<int>,CompareDist> pqueue(compObject);
    for (int i=0; i < vert_num; i++) {
	verts[i].space = 0;
	verts[i].parent = INVALID_VERT;
	verts[i].dist = HUGE_VALF;
    }
    vertex_t *vp, *ap;
    ushort v = v1, a;
    edge_t *ep;
    verts[v1].dist = 0;
    // find parents for all the vertices
    while (v != v2) {
	vp = &verts[v];
	VERT_SET_DONE(vp);
	for (ushort e=0; e < vp->adj_num; e++) {
	    a = vertNeighbor(v,e);
	    ap = &verts[a];
	    ep = &edges[vp->edges[e]];
	    if (!VERT_IS_DONE(ap)) {
		if (ap->dist > vp->dist + ep->length) {
		    ap->parent = v;
		    ap->dist = vp->dist + ep->length;
		}
		if (!VERT_IS_QUEUE(ap)) {
		    pqueue.push(a);
		    VERT_SET_QUEUE(ap);
		}
	    }
	}
	if (pqueue.size() == 0) {
	    printf("findPath: path from %d to %d not found\n",v1,v2);
	    return 0;
	}
	v = pqueue.top();
	pqueue.pop();
    }
    // extract the path from v1 to v2
    v = v2;
    while (v != v1) {
	path.push_front(v);
	v = verts[v].parent;
	if (v == INVALID_VERT) {
	    printf("findPath: Invalid parent!\n");
	    return 0;
	}
    }
    path.push_front(v);	// push v1
    printf("found path, size = %d\n", path.size());
    return path.size();
}

// draws path as series of edges for each consecutive
// pair of vertices in list path
void Mesh::drawPath(const list<ushort> &path)
{
    GLboolean blight;
    glGetBooleanv(GL_LIGHTING, &blight);
    GLint pmode[2];
    glGetIntegerv(GL_POLYGON_MODE, pmode);

    glDisable(GL_LIGHTING);
    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);

    glLineWidth(3);

    glBegin(GL_LINES);
    ushort v1,v2;
    list<ushort>::const_iterator p=path.begin();
    list<ushort>::const_iterator i=path.begin();
    i++;
    for (; i != path.end(); i++,p++) {
	v1 = *p;
	v2 = *i;
	glVertex3fv(verts[v1].p.a);
	glVertex3fv(verts[v2].p.a);
    }
    glEnd();
    glLineWidth(1);
    if (blight)
	glEnable(GL_LIGHTING);
    glPolygonMode(GL_FRONT, pmode[0]);
    glPolygonMode(GL_BACK, pmode[1]);
}

// draws a set of vertices
void Mesh::drawSet(const set<ushort> &s)
{
    vertex_t *vp;
    float r = 0.005;
    GLboolean blight;
    glGetBooleanv(GL_LIGHTING, &blight);
    glDisable(GL_LIGHTING);

    for (set<ushort>::const_iterator v=s.begin(); v != s.end(); v++) {
	vp = &verts[*v];
	glPushMatrix();
	glTranslatef(vp->p.x,vp->p.y,vp->p.z);
	glutSolidSphere(RADIUS_EDGE_FRAC*avg_edge_len,4,4);
	glPopMatrix();
    }

    if (blight)
	glEnable(GL_LIGHTING);
}

// draws local frame at vertex v
void Mesh::drawLocalFrame(int v, float c)
{
    float s = 0.5*avg_edge_len;

    GLboolean blight;
    glGetBooleanv(GL_LIGHTING, &blight);
    glDisable(GL_LIGHTING);

    Vector3D tv;
    glPushMatrix();
    vertex_t *vp = &verts[v];
    glTranslatef(vp->p.x,vp->p.y,vp->p.z);
    glBegin(GL_LINES);
	VEC_MULT_SCALAR(vp->T,s,tv);
	glColor3f(c,0,0); glVertex3f(0,0,0); glVertex3fv(tv.a);
	VEC_MULT_SCALAR(vp->B,s,tv);
	glColor3f(0,c,0); glVertex3f(0,0,0); glVertex3fv(tv.a);
	VEC_MULT_SCALAR(vp->N,s,tv);
	glColor3f(0,0,c); glVertex3f(0,0,0); glVertex3fv(tv.a);
    glEnd();
    glPopMatrix();
    if (blight)
	glEnable(GL_LIGHTING);
}

// draws a local frame at each vertex
void Mesh::drawLocalFrames()
{
    glLineWidth(3);
    for (int i=0; i < vert_num; i++)
	drawLocalFrame(i);
    glLineWidth(1);
}

// set all coefficients of Af and Bf
// they should already be allocated of appropriate size
void Mesh::setAf(const set<ushort> &bound_set, 
		const set<ushort> &handle_set,
		const set<ushort> &mod_set)
{
    for (ushort v=0; v < vert_num; v++) {
	vind[v] = INT_MIN;
    }

    // erase all Af and Bf
    for (int i=0; i < Af->dim1(); i++)
	for (int j=0; j < Af->dim2(); j++)
	    (*Af)[i][j] = 0;
    for (int i=0; i < Bf->dim(); i++)
	(*Bf)[i] = 0;

    // set vertex-to-X-pos mapping
    int c = 0;
    for (set<ushort>::const_iterator v=mod_set.begin();
	    v != mod_set.end(); v++,c++)
	vind[*v] = c;

    int r=0;
    vertex_t *vp,*ap;
    ushort va;
    int cv, ca;
    edge_t *ep;
    for (set<ushort>::const_iterator v=mod_set.begin();
	    v != mod_set.end(); v++) {
	vp = &verts[*v];
	cv = 9*vind[*v];
	for (ushort a=0; a < vp->adj_num; a++,r+=9) {
	    va = vertNeighbor(*v,a);
	    ap = &verts[va];
	    ep = &edges[vp->edges[a]];
	    // set identity first
	    for (int k=0; k < 9; k++)
		(*Af)[r+k][cv+k] = 1;
	    // if in the boundary set, bring to right side using original frames
	    if (bound_set.find(va) != bound_set.end()) {
		Vector3D frame[3];
		if (edgeOutVert(vp->edges[a],*v)) {
		    for (int i=0; i < 3; i++)
			vecLinComb(&ap->T, &ap->B, &ap->N,
				&ep->vcoefs[i], &frame[i]);
		}
		else {
		    Vector3D coefs[3];
		    // take transpose of coefficients first
		    for (int i=0; i < 3; i++)
			for (int j=0; j < 3; j++)
			    coefs[i].a[j] = ep->mcoefs[j][i];
		    for (int i=0; i < 3; i++)
			vecLinComb(&ap->T, &ap->B, &ap->N,
				&coefs[i], &frame[i]);
		}
		for (int i=0; i < 3; i++)
		    for (int j=0; j < 3; j++)
			(*Bf)[r+3*i+j] = frame[i].a[j];

	    } 
	    // if in the handle set, bring to right side using modified frames
	    else if (handle_set.find(va) != handle_set.end()) {
		Vector3D frame[3];
		if (edgeOutVert(vp->edges[a],*v)) {
		    for (int i=0; i < 3; i++)
			vecLinComb(&ap->mT, &ap->mB, &ap->mN,
				&ep->vcoefs[i], &frame[i]);
		}
		else {
		    Vector3D coefs[3];
		    // take transpose of coefficients first
		    for (int i=0; i < 3; i++)
			for (int j=0; j < 3; j++)
			    coefs[i].a[j] = ep->mcoefs[j][i];
		    for (int i=0; i < 3; i++)
			vecLinComb(&ap->mT, &ap->mB, &ap->mN,
				&coefs[i], &frame[i]);
		}
		for (int i=0; i < 3; i++)
		    for (int j=0; j < 3; j++)
			(*Bf)[r+3*i+j] = frame[i].a[j];
	    } 
	    // if this is a modifiable vertex
	    else {
		ca = 9*vind[va];
		// this edge leaves v
		if (edgeOutVert(vp->edges[a],*v)) {
		    for (int rr=0; rr < 3; rr++)
			for (int cc=0; cc < 3; cc++)
			    for (int k=0; k < 3; k++)
				(*Af)[r+3*rr+k][ca+3*cc+k] = 
				    -ep->mcoefs[rr][cc];
		} 
		// this edge enters v (have to transpose)
		else {
		    for (int rr=0; rr < 3; rr++)
			for (int cc=0; cc < 3; cc++)
			    for (int k=0; k < 3; k++)
				(*Af)[r+3*rr+k][ca+3*cc+k] = 
				    -ep->mcoefs[cc][rr];
		}
	    }
	}
    }
}

void Mesh::setAc(const set<ushort> &bound_set, 
		const set<ushort> &handle_set,
		const set<ushort> &mod_set)
{
    // set all Ac to 0
    for (int i=0; i < Ac->dim1(); i++)
	for (int j=0; j < Ac->dim2(); j++)
	    (*Ac)[i][j] = 0;


    int c = 0;
    for (set<ushort>::const_iterator v=handle_set.begin();
	    v != handle_set.end(); v++,c++) 
	vind[*v] = c-handle_set.size();

    int r = 0, ca, cv;
    vertex_t *vp, *ap;
    ushort va;
    edge_t *ep;
    Vector3D tv;
    // for all vertices in handle_set and mod_set
    for (set<ushort>::const_iterator v=handle_set.begin();
	    v != mod_set.end(); v++) 
    {
	if (v == handle_set.end())
	    v = mod_set.begin();
	vp = &verts[*v];
	cv = 3*(handle_set.size()+vind[*v]);
	for (ushort a=0; a < vp->adj_num; a++,r+=3) {
	    va = vertNeighbor(*v, a);
	    ap = &verts[va];
	    ep = &edges[vp->edges[a]];
	    vecLinComb(&vp->mT,&vp->mB,&vp->mN,&vp->q[a],&tv);
	    // set identity in Ac and right side
	    for (int i=0; i < 3; i++) {
		(*Bc)[r+i] = tv.a[i];
		(*Ac)[r+i][cv+i] = 1;
	    }
	    // if adjacent vertex is on the boundary
	    if (bound_set.find(va) != bound_set.end()) {
		for (int i=0; i < 3; i++)
		    (*Bc)[r+i] += ap->p.a[i];
	    }
	    else {
		ca = 3*(handle_set.size()+vind[va]);
		for (int i=0; i < 3; i++)
		    (*Ac)[r+i][ca+i] = -1;
	    }
	}
    }
}

void Mesh::solve(const set<ushort> &bound_set, 
		const set<ushort> &handle_set,
		const set<ushort> &mod_set)
{
    // transform handles
    /*
    Vector3D rv = {0,1,0};
    float trans_m[4][4];
    rotMatrix(PI/2, &rv, &trans_m[0][0]);
    for (set<ushort>::const_iterator h=handle_set.begin();
	    h != handle_set.end(); h++)
	for (int k=0; k < 3; k++) {
	    multMatVec(&trans_m[0][0],
		    &verts[*h].vframe[k], &verts[*h].mvframe[k]);
	    VEC_NORM(verts[*h].mvframe[k], mag);
	}
    */
    float mag;
    
    // allocate Af and Bf
    if (Af != NULL) delete Af;
    if (Bf != NULL) delete Bf;
    if (qrf != NULL) delete qrf;
    // calculate dimensions
    int m=0,n=0;
    for (set<ushort>::const_iterator i=mod_set.begin();
	    i != mod_set.end(); i++) {
	n += 9;
	m += 9*verts[*i].adj_num;
    }
    printf("solve: m=%d n=%d\n", m, n);
    Af = new Array2D<float>(m,n);
    Bf = new Array1D<float>(m);
    setAf(bound_set,handle_set,mod_set);

    //cout << *Af;
    //printf("\n");
    //cout << *Bf;

    // allocate QR
    qrf = new QR<float>(*Af);
    printf("QR full = %d\n", qrf->isFullRank());
    Xf = qrf->solve(*Bf);
    printf("solve: Xf.dim = %d\n", Xf.dim());

    // can erase Af now, since it is represented in qrf:
    delete Af;
    Af = NULL;

    // normalize the modified frames
    Vector3D tv;
    for (int i=0; i < n; i+=3) {
	tv.x = Xf[i+0];
	tv.y = Xf[i+1];
	tv.z = Xf[i+2];
	VEC_NORM(tv,mag);
	Xf[i+0] = tv.x;
	Xf[i+1] = tv.y;
	Xf[i+2] = tv.z;
    }
    int c = 0;
    for (set<ushort>::const_iterator i=mod_set.begin();
	    i != mod_set.end(); i++,c+=9)
	for (int k=0; k < 9; k++)
	    verts[*i].maframe[k] = Xf[c+k];


    // allocate Ac
    if (Ac != NULL) delete Ac;
    if (Bc != NULL) delete Bc;
    if (qrc != NULL) delete qrc;
    int nc = 3*(handle_set.size()+mod_set.size());
    int mc = m/3;
    for (set<ushort>::const_iterator i=handle_set.begin();
	    i != handle_set.end(); i++)
	mc += 3*verts[*i].adj_num;
    Ac = new Array2D<float>(mc,nc);
    Bc = new Array1D<float>(mc);
    setAc(bound_set,handle_set,mod_set);
    qrc = new QR<float>(*Ac);
    Xc = qrc->solve(*Bc);

    cout << *Ac;
    printf("\n");
    cout << *Bc;
    printf("\n");
    cout << Xc;

    delete Ac;
    Ac = NULL;
}

void Mesh::drawModified(const set<ushort> &handle_set,
			const set<ushort> &mod_set)
{
    glBegin(GL_TRIANGLES);
    for (int p=0; p < face_num; p++) {
	for (int k=0; k < 3; k++) {
	    glNormal3fv(faces[p].N.a);
	    if (vind[faces[p].ind[k]] == INT_MIN)
		glVertex3fv(verts[faces[p].ind[k]].p.a);
	    else {
		int c = 3*(handle_set.size()+vind[faces[p].ind[k]]);
		glVertex3f(Xc[c],Xc[c+1],Xc[c+2]);
	    }
	}
    }
    glEnd();
}
