/**
* @author Carlo Guglielmin
* @version 1.3
*
* Scribe.java
*  Contains all the GUI and functionality
*  Note: This class is messy and very repetitive in spots
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but without any warranty; without even the implied warranty of
* merchantability or fitness for a particular purpose. See the
* GNU General Public License (License.txt) for details.
*
* Copyright (C) 2004 Carlo Guglielmin
*/

import java.io.*;
import javax.swing.*;
import javax.swing.JOptionPane;
import javax.swing.filechooser.*;
import javax.swing.border.*;
import javax.swing.event.*;
import javax.swing.plaf.*;
import javax.swing.plaf.metal.*;
import java.awt.*;
import java.awt.event.*;
import java.sql.Time;
import java.sql.Date;
import java.util.Vector;
import java.util.StringTokenizer;

public class Scribe implements ActionListener, ChangeListener
{
	/* VARIABLES START */
	// Constants
        final double VERSION = 1.3;
	final String TITLE = (" - Scribe v" + VERSION);
        final String NEW_FILE = "Unsaved";
	final String SYMBOL = "> ";
        final Cursor BUTTON_CURSOR = new Cursor(Cursor.DEFAULT_CURSOR);
        final Cursor MENUITEM_CURSOR = new Cursor(Cursor.DEFAULT_CURSOR);
        final Font MENU_FONT = new Font("Arial", Font.BOLD, 13);
        final Font MENUITEM_FONT = new Font("Arial", Font.PLAIN, 14);
	final Runtime core = Runtime.getRuntime();
	// The name of the current file opened
	String fileName = NEW_FILE;
        // The default path for file browsers to start in
        String filePath = "default";
        // The class to execute
        String targetClass = "none";
	// The last search key
	String previousSearch = "";
	// File object used for storage of properties
	File file;
	// Used throughout for dialog answers
	int choice;
	// File browser for creating, saving, and loading
	JFileChooser fileBrowser = new JFileChooser();
	// Color chooser for setting the text area attributes
	JColorChooser colorChooser = new JColorChooser();
	// Store the returned result
	int result;
	// Program start time
	Time programStart = new Time(System.currentTimeMillis());
	// Document open time
	Time documentStart = new Time(System.currentTimeMillis());
	// The default font used for the text area
        Font textAreaFont = new Font("DialogInput", Font.PLAIN, 12);
	// The standard Scribe icon
        ImageIcon scribeIcon = new ImageIcon("Icon.gif", "Standard Scribe Icon 20x20");
	// The main GUI window
	JFrame window = new JFrame(fileName + TITLE);
	// Used for all additional popup dialogs
	JDialog popupDialog = new JDialog(window);
	// Content pane for the popup dialog
	JPanel popupPane = new JPanel(new FlowLayout(FlowLayout.CENTER, 5, 5));
	// The custom content pane for the main GUI window
	JPanel contentPane = new JPanel(new BorderLayout());
	// Main TextArea the user types in
	JTextArea paper = new JTextArea("");
	// A single line to update the user
	JTextField statusField = new JTextField("");
        // The area to note the line number
        JTextField lineField = new JTextField("1", 5);
	// Scroll pane for the TextArea paper
        JToolBar statusBar = new JToolBar(JToolBar.HORIZONTAL);
	JScrollPane roller = new JScrollPane(paper, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
						    JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
	// The menu bar object
	JMenuBar menuBar = new JMenuBar();
	// Various menus to be placed on the menu bar
	JMenu systemMenu = new JMenu    ("   System    ", false);
	JMenu functionMenu = new JMenu  ("  Function   ", false);
	JMenu preferenceMenu = new JMenu(" Preferences ", false);
	JMenu reportMenu = new JMenu    ("   Report    ", false);
        JMenu detailMenu = new JMenu    ("   Detail    ", false);
	// Items for the various menus
	JMenuItem createItem, saveItem, saveasItem, loadItem, hardcopyItem,
                  quitItem, searchItem, clearallItem, gotolineItem, removelineItem,
                  styleItem, canvasItem, directoryItem, countItem, timelapseItem,
                  virtualmachineItem, helpItem, aboutItem, licenseItem, overlayItem;
        JButton commandButton = makeButton(" ", "Execute the current Status Bar text", 0);
        JRadioButtonMenuItem wordwrapItem, autocleanItem;
	// Buttons for the various popup dialogs
        JButton closeButton = makeButton("Close", "Close this dialog", KeyEvent.VK_C);
	JButton searchButton = makeButton("Search", "Find the search key", KeyEvent.VK_S); String allText;
	JButton replaceButton = makeButton("Replace", "Replace the selected text", KeyEvent.VK_R); JTextField searchField;
        JButton selectButton = makeButton("Select", "Use the current font choices", KeyEvent.VK_S); JTextField sizeField; JComboBox fontType; JList fontList;
        JButton backgroundButton = makeButton("Background", "Set the background color", 0);
        JButton foregroundButton = makeButton("Foreground", "Set the foreground color", 0);
        JButton highlightButton = makeButton("Highlight", "Set the text highlight color", 0); int flag = -1; Color userColor = Color.WHITE;
        JButton selectedButton = makeButton("Selected", "Set the selected text color", 0);
        JButton cursorButton = makeButton("Cursor", "Set the cursor color", 0);
        JButton setButton = makeButton("Set", "Use this color for the attribute", KeyEvent.VK_S);
	JButton returnButton = makeButton("Return", "Return to the available list", KeyEvent.VK_R);
	/* VARIABLES END */

	/**
	* Default constructor, which sets the properties of initial components
	*/
	public Scribe()
	{
		// Set up the main GUI window properties
		window.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		// Change the small corner icon on the title bar if the file exists
		if ((scribeIcon.getIconWidth() == 20) && (scribeIcon.getIconHeight() == 20)) // Check for correct 20x22 size
			window.setIconImage(scribeIcon.getImage());
		// Set up the default popup dialog properties
		popupDialog.setResizable(false);
		popupDialog.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
		popupDialog.setContentPane(popupPane);
		// Add listeners to the popup buttons
                commandButton.setBorder(new BevelBorder(BevelBorder.RAISED));
                commandButton.setFont(new Font("DialogInput", Font.PLAIN, 11));
	} // End of default constructor

	/**
	* Method that handles all errors in a reasonable fashion
	*@param the error
	*@return none
	*/
	public void handleError(Exception all)
	{
		status("Warning, error occurred!");
		JOptionPane.showMessageDialog(window, "An error has occurred and Scribe v" + VERSION + " must close.\n" +
				  	      "Please send a description of the problem so it can be fixed.\n",
					      "Error!", JOptionPane.ERROR_MESSAGE);
                if (VERSION < 1.0) // Only print stack trace if still in development phase
			all.printStackTrace();
		System.exit(-1);
	} // End of caughtError method

	/**
	* Method to set the status field with a message and the current time
	*@param String message to set
	*@return none
	*/
	public void status(String message)
	{
                if (autocleanItem.isSelected()) // Auto garbage clean enabled
                {
                        if ((core.totalMemory()/2) > core.freeMemory())
                                core.gc();
                }
		statusField.setText(new Time(System.currentTimeMillis()).toString() + SYMBOL + message);
	} // End of status method

	/**
	* Method for detecting button clicks and other actions
	*@param the action performed
	*@return none
	*/
	public void actionPerformed(ActionEvent e)
	{
		Object source = e.getSource();
                if ((source != selectButton) && (source != searchButton) && (source != replaceButton))
                        popupDialog.setVisible(false);

                core.gc();

		try{

		// Quit button clicked
		if (source == quitItem)
			quitCall();
		// Create button clicked
		else if (source == createItem)
			createCall();
		// Save button clicked
		else if (source == saveItem)
			saveCall();
		// Save as button clicked
		else if (source == saveasItem)
			saveasCall();
		// Load button clicked
		else if (source == loadItem)
			loadCall();
		// Hard Copy button clicked
		else if (source == hardcopyItem)
			hardcopyCall();
		// Search button clicked
		else if (source == searchItem)
			searchCall();
                // Clear All button clicked
                else if (source == clearallItem)
		{
			paper.setText(null);
			status("File " + fileName + " cleared of all characters");
		}
                // Goto Line button clicked
                else if (source == gotolineItem)
                        gotoCall();
                // Remove Line button clicked
                else if (source == removelineItem)
                {
                        try{
                        paper.replaceRange("", paper.getLineStartOffset(Integer.parseInt(lineField.getText())-1),
                                               paper.getLineEndOffset(Integer.parseInt(lineField.getText())-1));
                        status("Line number " + lineField.getText() + " was removed");
                        }catch (Exception all) {
                                handleError(all);
                        }
                }
		// Style button clicked
		else if (source == styleItem)
			styleCall();
		// Canvas button clicked
		else if (source == canvasItem)
			canvasCall();
                // Directory button clicked
                else if (source == directoryItem)
                        directoryCall();
                // Word wrap radio button clicked
		else if (source == wordwrapItem)
			wordwrapCall();
                // Autoclean radio button clicked
                else if (source == autocleanItem)
                        autocleanCall();
		// Count button clicked
		else if (source == countItem)
			countCall();
		// Time lapse button clicked
		else if (source == timelapseItem)
			timelapseCall();
		// Virtual machine clicked
		else if (source == virtualmachineItem)
			virtualmachineCall();
		// Help button clicked
		else if (source == helpItem)
			helpCall();
		// About button clicked
		else if (source == aboutItem)
			aboutCall();
		// License button clicked
		else if (source == licenseItem)
			licenseCall();
                // Overlay button clicked
                else if (source == overlayItem)
                {
                        status("Overlay information dialog created");
                        Overlay theme = new Overlay();
                        JOptionPane.showMessageDialog(window, "Name: " + theme.getName() +
                                                      "\nAuthor: " + theme.getAuthor() +
                                                      "\nDescription: " + theme.getDescription(),
                                                      "Overlay", JOptionPane.INFORMATION_MESSAGE);
                        status("Overlay information dialog closed");
                } // End of overlayItem click
		// Close button clicked (popup dialog)
		else if (source == closeButton)
                {
                        // Deselect any highlighted text
                        paper.setSelectionStart(paper.getCaretPosition());
                        paper.setSelectionEnd(paper.getCaretPosition());
			popupDialog.setVisible(false);
                } // End of closeButton click
		// Search button clicked (popup dialog)
		else if (source == searchButton)
			searchSubCall();
		// Replace button clicked (popup dialog)
		else if (source == replaceButton)
			replaceSubCall();
		// Select button clicked (popup dialog)
		else if (source == selectButton)
			selectSubCall();
		// Set background button clicked (popup dialog)
		else if (source == backgroundButton)
		{
			colorChooser.setColor(paper.getBackground());
			status("Color chooser dialog created for background");
			popupDialog.setVisible(false);
			flag = 0;
			canvasCall();
		}
		// Set foreground button clicked (popup dialog)
		else if (source == foregroundButton)
		{
			colorChooser.setColor(paper.getForeground());
			status("Color chooser dialog created for text");
			popupDialog.setVisible(false);
			flag = 1;
			canvasCall();
		}
		// Set highlight button clicked (popup dialog)
		else if (source == highlightButton)
		{
			colorChooser.setColor(paper.getSelectionColor());
			status("Color chooser dialog created for highlight");
			popupDialog.setVisible(false);
			flag = 2;
			canvasCall();
		}
		// Set selected button clicked (popup dialog)
		else if (source == selectedButton)
		{
			colorChooser.setColor(paper.getSelectedTextColor());
			status("Color chooser dialog created for selected text");
			popupDialog.setVisible(false);
			flag = 3;
			canvasCall();
		}
		// Set cursor button clicked (popup dialog)
		else if (source == cursorButton)
		{
			colorChooser.setColor(paper.getCaretColor());
			status("Color chooser dialog created for cursor");
			popupDialog.setVisible(false);
			flag = 4;
			canvasCall();
		}
                // Set button clicked (popup dialog)
                else if (source == setButton)
		{
			popupDialog.setVisible(false);
			userColor = colorChooser.getColor();
			switch (flag)
			{
                                case 0: paper.setBackground(userColor); break;
                                case 1: paper.setForeground(userColor); break;
				case 2: paper.setSelectionColor(userColor); break;
				case 3: paper.setSelectedTextColor(userColor); break;
				case 4: paper.setCaretColor(userColor); break;
			} // End of flag switch
			status("New color value of " + userColor.getRed() + "," + userColor.getGreen() +
			       "," + userColor.getBlue() + " used");
			flag = -1;
			canvasCall();
                } // End of setButton click
		// Return button clicked (popup dialog)
		else if (source == returnButton)
		{
			status("Returned to available color list");
			popupDialog.setVisible(false);
			flag = -1;
			canvasCall();
		} // End of returnButton click
                // Command button clicked
                else if (source == commandButton)
                {
                        String command = statusField.getText();
                        try{
                                core.exec(command);
                                status("Command \'" + command + "\' executed");
                        }catch (Exception commandError) {
                                status("Command \'" + command + "\' failed to execute");
                        } // End of try-catch
                } // End of commandButton click

		}catch (Exception all) {
			handleError(all);
		} // End of try-catch

        } // End of actionPerformed override method

        /**
        * Method to track changes to specific GUI elements
        *@param the change event source
        *@return none
        */
        public void stateChanged(ChangeEvent e)
        {
                Object source = e.getSource();

                if (source == paper.getCaret())
                {
                        try{
                                // Update the current line number
                                lineField.setText("");
                                lineField.setText(String.valueOf(paper.getLineOfOffset(paper.getCaretPosition())+1));
                        }catch (Exception all) {
                                handleError(all);
                        }
                }
        } // End of stateChanged override method

	/**
	* Set up and display the GUI window and menus, including adding listeners, etc.
	*@param int screenHeight and screenWidth, representative of the current monitor's pixel size
	*@return none
	*/
	public void drawGUI(int screenWidth, int screenHeight)
	{
		// Set up the file chooser attributes
		fileBrowser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
		fileBrowser.setDragEnabled(false);
		fileBrowser.setFileHidingEnabled(false);
		fileBrowser.setAcceptAllFileFilterUsed(true);
		fileBrowser.setMultiSelectionEnabled(false);

		// Create all the menu items and assign them a mnemonic
		createItem = makeMenuItem("Create", "Create a new document", KeyEvent.VK_C);
		saveItem = makeMenuItem("Save", "Save the current document", KeyEvent.VK_S);
		loadItem = makeMenuItem("Load", "Load a stored document", KeyEvent.VK_L);
		hardcopyItem = makeMenuItem("Hard Copy", "Print the current document", KeyEvent.VK_H);
		quitItem = makeMenuItem("Quit", "Leave the program", KeyEvent.VK_Q);
		searchItem = makeMenuItem("Search", "Search / replace within the current document", KeyEvent.VK_S);
                clearallItem = makeMenuItem("Clear All", "Set the current document as blank", KeyEvent.VK_C);
                gotolineItem = makeMenuItem("Goto Line", "Move to a specific line within the document", KeyEvent.VK_G);
                removelineItem = makeMenuItem("Remove Line", "Remove the line the cursor is at", KeyEvent.VK_R);
		styleItem = makeMenuItem("Style", "Set up the font to display", KeyEvent.VK_S);
		canvasItem = makeMenuItem("Canvas", "Change the main window properties", KeyEvent.VK_C);
		directoryItem = makeMenuItem("Directory", "Set the default file browser directory", KeyEvent.VK_D);
		countItem = makeMenuItem("Count", "Display character / word / line count", KeyEvent.VK_C);
		timelapseItem = makeMenuItem("Time Lapse", "Display session / program length", KeyEvent.VK_T);
		virtualmachineItem = makeMenuItem("Virtual Machine", "Display memory information", KeyEvent.VK_V);
		helpItem = makeMenuItem("Help", "View the file Readme.txt", KeyEvent.VK_H);
		aboutItem = makeMenuItem("About", "Credits for the program", KeyEvent.VK_A);
		licenseItem = makeMenuItem("License", "View the GNU General Public License", KeyEvent.VK_L);
		overlayItem = makeMenuItem("Overlay", "Display current Overlay details", KeyEvent.VK_O);
		saveasItem = new JMenuItem("Save As", KeyEvent.VK_A); saveasItem.setDisplayedMnemonicIndex(5);
                        saveasItem.setFont(MENUITEM_FONT); saveasItem.setToolTipText("Save current document as a new filename");
		wordwrapItem = new JRadioButtonMenuItem("Word Wrap", true); wordwrapItem.setMnemonic(KeyEvent.VK_W);
			wordwrapItem.setFont(MENUITEM_FONT); wordwrapItem.setToolTipText("Enable / disable word wrap");
                        wordwrapItem.addActionListener(this);
                autocleanItem = new JRadioButtonMenuItem("Autoclean", true); autocleanItem.setMnemonic(KeyEvent.VK_A);
                        autocleanItem.setFont(MENUITEM_FONT); autocleanItem.setToolTipText("Enable / disable auto garbage collection");
                        autocleanItem.addActionListener(this);

                /* TEMPORARY */
                // (until physical printing is implemented)
		hardcopyItem.setEnabled(false);
                hardcopyItem.setToolTipText("Feature is currently incomplete");

		// Add any keyboard accelerators
		// Some are already defined by the system (like Copy) so alternates had to be used
		createItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, ActionEvent.CTRL_MASK)); // N for New
		saveItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, ActionEvent.CTRL_MASK));
		loadItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_L, ActionEvent.CTRL_MASK));
		quitItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Q, ActionEvent.CTRL_MASK));
		searchItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F, ActionEvent.CTRL_MASK)); // F for Find
                helpItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_H, ActionEvent.CTRL_MASK));
                gotolineItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_G, ActionEvent.CTRL_MASK));
                removelineItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_R, ActionEvent.CTRL_MASK));

		// Add the items to their menus and set a mnemonic for each menu
		systemMenu.setMnemonic(KeyEvent.VK_S); systemMenu.setFont(MENU_FONT);
		systemMenu.add(createItem); systemMenu.addSeparator();
		systemMenu.add(saveItem); systemMenu.addSeparator();
		systemMenu.add(saveasItem); systemMenu.addSeparator();
		systemMenu.add(loadItem); systemMenu.addSeparator();
		systemMenu.add(hardcopyItem); systemMenu.addSeparator();
		systemMenu.add(quitItem);
		functionMenu.setMnemonic(KeyEvent.VK_F); functionMenu.setFont(MENU_FONT);
		functionMenu.add(searchItem); functionMenu.addSeparator();
                functionMenu.add(clearallItem); functionMenu.addSeparator();
                functionMenu.add(gotolineItem); functionMenu.addSeparator();
                functionMenu.add(removelineItem);
		preferenceMenu.setMnemonic(KeyEvent.VK_P); preferenceMenu.setFont(MENU_FONT);
		preferenceMenu.add(styleItem); preferenceMenu.addSeparator();
		preferenceMenu.add(canvasItem); preferenceMenu.addSeparator();
                preferenceMenu.add(directoryItem); preferenceMenu.addSeparator();
                preferenceMenu.add(wordwrapItem); preferenceMenu.addSeparator();
                preferenceMenu.add(autocleanItem);
		reportMenu.setMnemonic(KeyEvent.VK_R); reportMenu.setFont(MENU_FONT);
		reportMenu.add(countItem); reportMenu.addSeparator();
		reportMenu.add(timelapseItem); reportMenu.addSeparator();
		reportMenu.add(virtualmachineItem);
                detailMenu.setMnemonic(KeyEvent.VK_D); detailMenu.setFont(MENU_FONT);
                detailMenu.add(helpItem); detailMenu.addSeparator();
                detailMenu.add(aboutItem); detailMenu.addSeparator();
                detailMenu.add(overlayItem); detailMenu.addSeparator();
                detailMenu.add(licenseItem);

		// Add the menus to the menu bar
		menuBar.setCursor(new Cursor(Cursor.HAND_CURSOR));
                menuBar.add(systemMenu);
                menuBar.add(functionMenu);
                menuBar.add(preferenceMenu);
                menuBar.add(reportMenu);
                menuBar.add(detailMenu);

		// Set up the text area properties
                paper.setLineWrap(true);
                paper.setWrapStyleWord(true);
		paper.setCaretColor(Color.RED);
		paper.setDragEnabled(true);
                paper.setBackground(new Color(245, 245, 245));
		paper.setForeground(new Color(0, 0, 0));
		paper.setSelectedTextColor(new Color(255, 255, 255)); // Text
		paper.setSelectionColor(new Color(0, 0, 0)); // Highlight
		paper.setFont(textAreaFont);
                paper.setBorder(new TitledBorder(new LineBorder(Color.BLACK, 0, true), ""));
                (paper.getCaret()).setBlinkRate(0);
                (paper.getCaret()).addChangeListener(this);
		roller.setWheelScrollingEnabled(true);

		// Set up the text field properties
                statusField.setFont(new Font("DialogInput", Font.PLAIN, 10));
                statusField.setToolTipText("Record of last action or command");
		statusField.setHorizontalAlignment(JTextField.LEFT);
                statusField.setEditable(true);
                statusField.setFocusable(true);
                statusField.setBackground(Color.WHITE);
                statusField.setForeground(Color.BLACK);
                statusField.setBorder(new LineBorder(Color.BLACK, 1, false));
		status("Scribe v" + VERSION + " initialized");

                // Set up the line number field
                lineField.setFont(statusField.getFont());
                lineField.setToolTipText("Line number the cursor is currently at");
                lineField.setHorizontalAlignment(JTextField.CENTER);
                lineField.setEditable(false);
                lineField.setFocusable(false);
                lineField.setBackground(statusField.getBackground());
                lineField.setForeground(statusField.getForeground());
                lineField.setBorder(statusField.getBorder());

                // Set the attributes based on the saved file
		loadPreferences();

		// Add the various elements to the content pane
		contentPane.add(roller, BorderLayout.CENTER);
                statusBar.setLayout(new BorderLayout());
                statusBar.add(commandButton, BorderLayout.WEST);
                statusBar.add(statusField, BorderLayout.CENTER);
                statusBar.add(lineField, BorderLayout.EAST);
                contentPane.add(statusBar, BorderLayout.SOUTH);

		// Complete main GUI window setup and display
                window.setBounds(0, 0, (screenWidth-(screenWidth/10)), (screenHeight-(screenHeight/10)));
                window.setLocationRelativeTo(null);
                window.setJMenuBar(menuBar);
		window.setContentPane(contentPane);
                core.gc();
		window.setVisible(true);
	} // End of drawGUI method

	/**
	* A method to reduce code redundancy by setting up a button
	* @param various attributes of the button
	* @return the completed button
	*/
	public JButton makeButton(String text, String toolTip, int mnemonic)
	{
		JButton created = new JButton(text);
		created.addActionListener(this);
		created.setCursor(BUTTON_CURSOR);
		created.setToolTipText(toolTip);
		if (mnemonic != 0)
			created.setMnemonic(mnemonic);
		return(created);
	} // End of makeButton method

	/**
	* A method to reduce code redundancy by setting up a menu item
	* @param various attributes of the menu item
	* @return the completed menu item
	*/
	public JMenuItem makeMenuItem(String text, String toolTip, int mnemonic)
	{
		JMenuItem created = new JMenuItem(text);
		created.addActionListener(this);
		created.setFont(MENUITEM_FONT);
		created.setCursor(MENUITEM_CURSOR);
		created.setToolTipText(toolTip);
		if (mnemonic != 0)
			created.setMnemonic(mnemonic);
		return(created);
	} // End of makeMenuItem method

	/**
	* Set the various elements to the stored configuration
	*@param none
	*@return none
	*/
	public void loadPreferences()
	{
		try{
			File configFile = new File("User.cfg");
			// Make sure the file exists and is not empty
			if ((configFile.exists()) && (configFile.length() > 0))
			{
				// Create the input stream
				BufferedReader load = new BufferedReader(new FileReader(configFile.getAbsolutePath()));
				// Initialize tokenizer to break on '|'
				StringTokenizer st = new StringTokenizer(load.readLine(), "|");

				// Set the font face, size, and style
				textAreaFont = new Font(st.nextToken(), Integer.parseInt(st.nextToken()), Integer.parseInt(st.nextToken()));
				paper.setFont(textAreaFont);
				// Set the background RGB
				paper.setBackground(new Color(Integer.parseInt(st.nextToken()),
                                                    Integer.parseInt(st.nextToken()), Integer.parseInt(st.nextToken())));
				// Set the foreground RGB
				paper.setForeground(new Color(Integer.parseInt(st.nextToken()),
                                                    Integer.parseInt(st.nextToken()), Integer.parseInt(st.nextToken())));
				// Set the highlight RGB
				paper.setSelectionColor(new Color(Integer.parseInt(st.nextToken()),
                                                        Integer.parseInt(st.nextToken()), Integer.parseInt(st.nextToken())));
				// Set the selected text RGB
				paper.setSelectedTextColor(new Color(Integer.parseInt(st.nextToken()),
                                                           Integer.parseInt(st.nextToken()), Integer.parseInt(st.nextToken())));
                                // Set the cursor RGB
				paper.setCaretColor(new Color(Integer.parseInt(st.nextToken()),
                                                    Integer.parseInt(st.nextToken()), Integer.parseInt(st.nextToken())));
				// Set word wrap to enabled / disabled
				if (st.nextToken().equalsIgnoreCase("ON"))
					wordwrapItem.setSelected(true);
				else
					wordwrapItem.setSelected(false);
				// Set auto clean to enabled / disabled
				if (st.nextToken().equalsIgnoreCase("ON"))
					autocleanItem.setSelected(true);
				else
					autocleanItem.setSelected(false);

                                // Apply settings based on radio buttons
                                paper.setLineWrap(wordwrapItem.isSelected());
                                paper.setWrapStyleWord(wordwrapItem.isSelected());

				// Set the default file browser file path
				filePath = st.nextToken();
				if (filePath.equalsIgnoreCase("default"))
					fileBrowser.setCurrentDirectory(null);
				else
					fileBrowser.setCurrentDirectory(new File(filePath));

				// Close the User.cfg stream and run the garbage collector
				load.close();
				core.gc();
			}
		}catch (Exception all) {
			handleError(all);
		}
	} // End of loadPreferences method

	/**
	* Creates a new untitled file
	*@param none
	*@return none
	*/
	public void createCall()
	{
		// Clear everything and create a new fresh document
		file = new File("");
		paper.setText(null);
		fileName = NEW_FILE;
		window.setTitle(fileName + TITLE);
		documentStart.setTime(System.currentTimeMillis());
		status("Created a new document (title: " + fileName + ")");
                core.gc();
	} // End of createCall method

	/**
	* Saves the file under the current filename
	*@param none
	*@return none
	*/
	public void saveCall()
	{
		try{
		if (fileName == NEW_FILE)
			saveasCall();
		else
		{
		// Save the file to the current filename
		if (file.exists())
		{
                        status("File " + file.getAbsoluteFile() + " re-saving...");
			paper.write(new PrintWriter(new BufferedWriter(new FileWriter(file.getAbsoluteFile()))));
                        core.gc();
                        statusField.setText(statusField.getText() + "done");
		}
		} // Not a new file
		}catch (Exception all) {
			handleError(all);
		}
	} // End of saveCall method

	/**
	* Save the file using a file browser
	*@param none
	*@return none
	*/
	public void saveasCall()
	{
		try{
		// Customize the file browser and display
		fileBrowser.setDialogType(JFileChooser.SAVE_DIALOG);
		fileBrowser.setApproveButtonMnemonic(KeyEvent.VK_S);
		fileBrowser.setApproveButtonToolTipText("Save as selected file");
		fileBrowser.setDialogTitle("File Browser - Save As");
		fileBrowser.setSelectedFile(file);
		result = fileBrowser.showDialog(window, "Save");

		// If a correct approved file has been selected, fill it to the text area
		if (result == JFileChooser.APPROVE_OPTION)
		{
                        status("File " + file.getAbsoluteFile() + " saving...");
			file = fileBrowser.getSelectedFile();
			paper.write(new PrintWriter(new BufferedWriter(new FileWriter(file.getAbsoluteFile()))));
			fileName = file.getName();
                        window.setTitle(fileName + TITLE);
                        core.gc();
                        statusField.setText(statusField.getText() + "done");
		} // File approved for saving
		}catch (Exception all) {
			handleError(all);
		}
	} // End of saveasCall

	/**
	* Loads a file using a file browser
	*@param none
	*@return none
	*/
	public void loadCall()
	{
		try{
		// Customize the file browser and display
		fileBrowser.setDialogType(JFileChooser.OPEN_DIALOG);
		fileBrowser.setApproveButtonMnemonic(KeyEvent.VK_L);
		fileBrowser.setApproveButtonToolTipText("Load the selected file");
		fileBrowser.setDialogTitle("File Browser - Load");
		fileBrowser.setSelectedFile(new File(""));
		result = fileBrowser.showDialog(window, "Load");

		// If a correct approved file has been selected, output the current text area
		if (result == JFileChooser.APPROVE_OPTION)
		{
			file = fileBrowser.getSelectedFile();
			// Prevent an OutOfMemory error
                        core.gc();
                        if (file.length() > core.freeMemory())
			{
				// If failed, display an error message and reset the file
				JOptionPane.showMessageDialog(window, (file.getName() + " is too large (" + (file.length()/1000/1000) +
							      "mb) and cannot be loaded."), "File Size Limit Error",
							      JOptionPane.ERROR_MESSAGE);
				status("File " + file.getAbsoluteFile() + " loading failed (exceeds file size limit)");
				file = new File("");
			}
			else
			{
				// Load the file if acceptable
                                status("File " + file.getAbsoluteFile() + " loading...");
				paper.read(new BufferedReader(new FileReader(file.getAbsoluteFile())), "Text");
				fileName = file.getName();
                                window.setTitle(fileName + TITLE);
				documentStart.setTime(System.currentTimeMillis());
                                // Set the line number width based on max lines
                                lineField.setColumns(String.valueOf(paper.getLineCount()).length());
                                if (lineField.getColumns() < 5)
                                        lineField.setColumns(5);
                                lineField.setText("1");
                                previousSearch = "";
                                statusField.setText(statusField.getText() + "done");
			}
		}
                core.gc();
		}catch (Exception all) {
			handleError(all);
		}
	} // End of loadCall method

	/**
	* Prints the current file
	*@param none
	*@return none
	*/
	public void hardcopyCall()
	{
	} // End of hardcopyCall method

	/**
	* Quits the program with various dialogs
	*@param none
	*@return none
	*/
	public void quitCall()
	{
		status("User is leaving...");
                choice = JOptionPane.showConfirmDialog(window, ("Are you sure you want to quit?"),
		                                       "Confirm Quit", JOptionPane.YES_NO_OPTION,
						       JOptionPane.PLAIN_MESSAGE, scribeIcon);
		if (choice == 0) // Yes
                {
                        statusField.setText(statusField.getText() + "yes (saving data to User.cfg)");
                        Color currentColor;
                        // Save user defaults
                        try{
                        PrintWriter configOut = new PrintWriter(new FileOutputStream("User.cfg", false), true);

			// Font info (face, size, style)
                        configOut.write(textAreaFont.getName() + "|" + textAreaFont.getStyle() + "|" +
                                        textAreaFont.getSize() + "|");
			// Background RGB
                        currentColor = paper.getBackground();
                        configOut.write(currentColor.getRed() + "|" + currentColor.getGreen() + "|" +
                                        currentColor.getBlue() + "|");
			// Foreground RGB
                        currentColor = paper.getForeground();
                        configOut.write(currentColor.getRed() + "|" + currentColor.getGreen() + "|" +
                                        currentColor.getBlue() + "|");
			// Highlight RGB
                        currentColor = paper.getSelectionColor();
                        configOut.write(currentColor.getRed() + "|" + currentColor.getGreen() + "|" +
                                        currentColor.getBlue() + "|");
			// Selected RGB
                        currentColor = paper.getSelectedTextColor();
                        configOut.write(currentColor.getRed() + "|" + currentColor.getGreen() + "|" +
                                        currentColor.getBlue() + "|");
			// Caret RGB
                        currentColor = paper.getCaretColor();
                        configOut.write(currentColor.getRed() + "|" + currentColor.getGreen() + "|" +
                                        currentColor.getBlue() + "|");
			// Word wrap on/off
                        if (wordwrapItem.isSelected())
                                configOut.write("ON|");
                        else
                                configOut.write("OFF|");
			// Autoclean on/off
                        if (autocleanItem.isSelected())
                                configOut.write("ON|");
                        else
                                configOut.write("OFF|");
			// File path
			if (filePath.equals(""))
				filePath = "default";
                        configOut.write(filePath);
                        configOut.close();
                        }catch (Exception all) {
                                handleError(all);
                        } // End of try-catch
			System.exit(0);
                }
		else // No
			statusField.setText(statusField.getText() + "no");
	} // End of quitCall method

	/**
	* Searches the text for a word, either look up or replace
	*@param none
	*@return none
	*/
	public void searchCall()
	{
		// Set dialog specific options and bounds
		popupDialog.setTitle("Search: " + fileName);
		popupDialog.setBounds(0, 0, 320, 120);
		popupDialog.setLocationRelativeTo(window);

		// Create dialog specific GUI items
		JLabel searchLabel = new JLabel("Search Field (case sensitive)");
		searchField = new JTextField(previousSearch, 25);
		searchField.setSelectionStart(0); searchField.setSelectionEnd(previousSearch.length());
		searchField.setToolTipText("The search key to be used");

                // Create the button inner panel
                JPanel buttonPanel = new JPanel();
                buttonPanel.setLayout(new GridLayout(1, 3, 5, 1));
                buttonPanel.add(searchButton);
                buttonPanel.add(replaceButton);
                buttonPanel.add(closeButton);

		// Add to the dialog and display
		popupPane.removeAll();
		popupPane.add(searchLabel);
		popupPane.add(searchField);
                popupPane.add(buttonPanel);
		popupDialog.setVisible(true);
		status("Search dialog created and awaiting command");
	} // End of searchCall method

	/**
	* Functionality for a Search button click
	*@param none
	*@return none
	*/
	public void searchSubCall()
	{
                String searchFor = searchField.getText();
                if (!searchFor.equals(""))
                {
		allText = paper.getText();
		previousSearch = searchFor;
		int foundIndex = allText.indexOf(searchFor, paper.getCaretPosition());

		if (foundIndex == -1) // Not found
		{
			status("Search dialog could not find \'" + searchFor + "\'");
			JOptionPane.showMessageDialog(popupDialog, "The search key \'" + searchFor + "\' was not found.", "\'" +
				 	 	      searchFor + "\' Not Found", JOptionPane.WARNING_MESSAGE);
		}
		else // Word found
		{
			// Select the found word
			paper.setSelectionStart(foundIndex);
			paper.setSelectionEnd(foundIndex+searchFor.length());
                        status("Search dialog found \'" + searchFor + "\' at location " + (paper.getCaretPosition() - searchFor.length()));
		}
                } // End of blank search key check
                else
                        JOptionPane.showMessageDialog(popupDialog, "Please enter a search key to use.",
                                                      "Empty Search Key", JOptionPane.WARNING_MESSAGE);
	} // End of searchSubCall method

	/**
	* Functionality for a Replace button click
	*@param none
	*@return none
	*/
	public void replaceSubCall()
	{
		String toReplace = paper.getSelectedText();
		// Replace any selected text
		if (toReplace != null)
		{
			String replaceWith = JOptionPane.showInputDialog(popupDialog,
					     ("Enter the word(s) to replace \'" + toReplace + "\'"),
					     "Replace With...", JOptionPane.QUESTION_MESSAGE);
			if (replaceWith != null) // Make sure Cancel button wasn't clicked
			{
				paper.replaceSelection(replaceWith);
				status("Replaced \'" + toReplace + "\' with \'" + replaceWith + "\' at location " +
				       paper.getSelectionStart());
			}
		}
		// Notify that no text is selected
		else
			JOptionPane.showMessageDialog(popupDialog, "Please select the text to replace.", "Nothing Selected",
						      JOptionPane.WARNING_MESSAGE);
	} // End of replaceSubCall method

        /**
        * Move to a specific line within the document bounds
        *@param none
        *@return none
        */
        public void gotoCall()
        {
                String choiceS = "none";
                try{
                        status("Goto Line dialog created");
                        do{
                        choiceS = JOptionPane.showInputDialog(window,
                                  "Enter line number to goto (1-" + paper.getLineCount() + ")",
                                  "Goto Line", JOptionPane.QUESTION_MESSAGE);
                        choice = 0;
                        // Loop through and make sure value is all digits
                        for (int i = 0; i < choiceS.length(); i++)
                        {
                                // If a non-digit is found, set flag and break
                                if (!Character.isDigit(choiceS.charAt(i)))
                                {
                                        choice = -1;
                                        break;
                                }
                        }
                        if (choice == 0)
                                choice = Integer.parseInt(choiceS);
                        }while ((choice < 1) || (choice > paper.getLineCount()));
                }catch (Exception ignore) { choice = -1; }
                if (choice != -1)
                {
                        try{
                                paper.setCaretPosition(paper.getLineStartOffset(choice-1));
                        }catch (Exception all) {
                                handleError(all);
                        }
                        status("Moved cursor position to line " + choice);
                }
                else
                        status("Goto Line dialog closed");
        } // End of gotoCall method

	/**
	* Changes the font attributes
	*@param none
	*@return none
	*/
	public void styleCall()
	{
		// Set dialog specific options and bounds
		GraphicsEnvironment current = GraphicsEnvironment.getLocalGraphicsEnvironment();
                popupDialog.setTitle("Style");
                popupDialog.setBounds(0, 0, 365, 205);
		popupDialog.setLocationRelativeTo(window);

		// Create and populate the font type list
		fontList = new JList(current.getAvailableFontFamilyNames());
		fontList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		fontList.setLayoutOrientation(JList.VERTICAL);
                fontList.setFixedCellWidth(200);
		fontList.setFixedCellHeight(18);
                fontList.setVisibleRowCount(7);
		fontList.setDragEnabled(false);
		fontList.setToolTipText("Available font list");

		// Create and populate the style combo box
		Vector styleType = new Vector();
		styleType.add("Plain");
		styleType.add("Bold");
		styleType.add("Italic");
		styleType.add("Bold-Italic");
		fontType = new JComboBox(styleType);
		fontType.setEditable(false);
		fontType.setToolTipText("Various applicable styles");

		// Create the font size text field
                sizeField = new JTextField(String.valueOf(textAreaFont.getSize()), 4);
                sizeField.setHorizontalAlignment(JTextField.CENTER);
		sizeField.setToolTipText("The point size of the font");

		// Calculate what style is used and relate to the combo box
		if (textAreaFont.getStyle() == Font.PLAIN)
			fontType.setSelectedItem("Plain");
		else if (textAreaFont.getStyle() == Font.BOLD)
			fontType.setSelectedItem("Bold");
		else if (textAreaFont.getStyle() == Font.ITALIC)
			fontType.setSelectedItem("Italic");
		else if (textAreaFont.getStyle() == (Font.BOLD|Font.ITALIC))
			fontType.setSelectedItem("Bold-Italic");

                // Create two panels used for border buffering
                JPanel sizePanel = new JPanel();
                sizePanel.setBorder(new TitledBorder(new EtchedBorder(EtchedBorder.RAISED), "Font Size"));
                sizePanel.add(sizeField);
                JPanel fontPanel = new JPanel();
                fontPanel.setBorder(new TitledBorder(new EtchedBorder(EtchedBorder.RAISED), "Font Type"));
                fontPanel.add(fontType);

                // Create the attribute internal panel
                JPanel attributePanel = new JPanel();
                attributePanel.setLayout(new GridLayout(2, 1, 1, 0));
                attributePanel.add(sizePanel);
                attributePanel.add(fontPanel);

                // Create the button internal panel
                JPanel buttonPanel = new JPanel();
                buttonPanel.setLayout(new GridLayout(1, 2, 5, 1));
                buttonPanel.add(selectButton);
                buttonPanel.add(closeButton);

		// Add to the dialog and display
		popupPane.removeAll();
                popupPane.add(new JScrollPane(fontList));
                popupPane.add(attributePanel);
                popupPane.add(buttonPanel);
		popupDialog.setVisible(true);

		fontList.setSelectedValue(((String)textAreaFont.getName()), true);
                selectButton.requestFocus();
		status("Style dialog created for font selection");
	} // End of styleCall method

	/**
	* Functionality for a Select button click
	*@param none
	*@return none
	*/
	public void selectSubCall()
	{
		// Get the result from the combo box
		String typeSelected = ((String)fontType.getSelectedItem());

		// Set the corresponding font constant
		if (typeSelected.equalsIgnoreCase("bold"))
			choice = Font.BOLD;
		else if (typeSelected.equalsIgnoreCase("italic"))
			choice = Font.ITALIC;
		else if (typeSelected.equalsIgnoreCase("bold-italic"))
			choice = (Font.BOLD|Font.ITALIC);
		else
			choice = Font.PLAIN;

		// Check to make sure the font size is valid
		String sizeResult = sizeField.getText();
		boolean errorFound = false;
		// Check each character inside the result for validity
		for (int i = 0; i < sizeResult.length(); i++)
		{
			char current = sizeResult.charAt(i);
			if (!Character.isDigit(current))
				errorFound = true;
		}

		if (errorFound)	// An invalid character was found
			JOptionPane.showMessageDialog(popupDialog, "Please enter a valid font size.",
						      "Invalid Font Size", JOptionPane.WARNING_MESSAGE);

		// No errors were found, so complete the process
		else
		{
		// Set the new font type and hide the popup dialog
		if (Integer.parseInt(sizeResult) > 1000)
			sizeResult = "1000";
		if (Integer.parseInt(sizeResult) == 0)
			sizeResult = "1";
		textAreaFont = new Font(((String)fontList.getSelectedValue()), choice, Integer.parseInt(sizeResult));
		paper.setFont(textAreaFont);
                status("Font set to " + (String)fontList.getSelectedValue() + " (" + typeSelected + ") at size " + sizeResult);
		}
	} // End of selectSubCall method

	/**
	* Changes the text area attributes
	*@param none
	*@return none
	*/
	public void canvasCall()
	{
		// Display the initial popup dialog
		if (flag == -1)
		{
			// Set dialog specific options and bounds
			popupDialog.setTitle("Canvas");
                        popupDialog.setBounds(0, 0, 150, 255);
			popupDialog.setLocationRelativeTo(window);

                        // Set the foreground based on the current color
                        backgroundButton.setForeground(paper.getBackground());
                        foregroundButton.setForeground(paper.getForeground());
                        highlightButton.setForeground(paper.getSelectionColor());
                        selectedButton.setForeground(paper.getSelectedTextColor());
                        cursorButton.setForeground(paper.getCaretColor());

                        // Create the formatted button internal panel
                        JPanel buttonPanel = new JPanel();
                        buttonPanel.setLayout(new GridLayout(5, 1, 0, 5));
                        buttonPanel.setBorder(new TitledBorder(new EtchedBorder(EtchedBorder.RAISED), "Categories"));
                        buttonPanel.add(backgroundButton);
                        buttonPanel.add(foregroundButton);
                        buttonPanel.add(highlightButton);
                        buttonPanel.add(selectedButton);
                        buttonPanel.add(cursorButton);

			// Add to the dialog and display
			popupPane.removeAll();
                        popupPane.add(buttonPanel);
			popupPane.add(closeButton);
			popupDialog.setVisible(true);
                        closeButton.requestFocus();
                        status("Canvas dialog created for color selection");
		}
		else // Display the color chooser dialog
		{
			// Set dialog specific options and bounds
			popupDialog.setTitle("Canvas Color Selector");
			popupDialog.setBounds(0, 0, 475, 420);
			popupDialog.setLocationRelativeTo(window);

                        // Create the internal button panel
                        JPanel buttonPanel = new JPanel();
                        buttonPanel.setLayout(new GridLayout(1, 2, 5, 0));
                        buttonPanel.add(setButton);
                        buttonPanel.add(returnButton);

			// Add to the dialog and display
			popupPane.removeAll();
			popupPane.add(colorChooser);
                        popupPane.add(buttonPanel);
			popupDialog.setVisible(true);
                        setButton.requestFocus();
		}
	} // End of canvasCall method

        /**
        * Change the default starting directory for all file browsers
        *@param none
        *@return none
        */
        public void directoryCall()
        {
		try{
			status("File browser start directory dialog created");
                        String result =  JOptionPane.showInputDialog(window, ("<HTML>Current directory: <I>" + filePath + "\nEnter new file browser start directory"),
                                                                     "Start Directory", JOptionPane.QUESTION_MESSAGE);
			// If the user pressed Cancel, this will throw a NullPointerException
			if (!result.equals(null))
			{
				if (result.equals(""))
					result = "default";
				filePath = result;
			}
		}catch (NullPointerException ignore) {
		}finally {
			status("File browser start directory set to " + filePath);
		}
        } // End of directoryCall method

	/**
	* Based on the checked state of the radio button, word wrap will be enabled or disabled
	*@param none
	*@return none
	*/
	public void wordwrapCall()
	{
		paper.setLineWrap(wordwrapItem.isSelected());
		paper.setWrapStyleWord(wordwrapItem.isSelected());
		if (wordwrapItem.isSelected()) // Has been enabled
			status("Word Wrap is enabled");
		else // Has been disabled
			status("Word Wrap is disabled");
	} // End of wordwrapCall method

        /**
        * Allow the user to activate or deactivate automatic garbage collection
        *@param none
        *@return none
        */
        public void autocleanCall()
        {
                if (autocleanItem.isSelected()) // Has been enabled
                {
                        status("Auto garbage collection is enabled");
                        core.gc();
                }
                else // Has been disabled
                        status("Auto garbage collection is disabled");
        } // End of autocleanCall method

	/**
	* Displays various count statistics based on radio button selection
	*@param none
	*@return none
	*/
	public void countCall()
	{
		// Variables used to count
		String fullText = paper.getText();
		long wordCount = 0;
		long charCount = fullText.length();
		char check;

		fullText = fullText.trim();
		// Loop through the whole string
		for (int i = 0; i < fullText.length(); i++)
		{
			check = fullText.charAt(i);
			// If a space is found with text following, increment the count
			if (Character.isWhitespace(check))
			{
				if (fullText.length() > i+1)
					if (!Character.isWhitespace(fullText.charAt(i+1)))
						wordCount++;
			} // Whitespace endif
		} // End of for loop
		if (!fullText.equals(""))
			wordCount++;

                status("Count dialog created");
                JOptionPane.showMessageDialog(window, ("Character: " + charCount + "\nWord: " + wordCount + "\nLine: " + paper.getLineCount()),
					      "Count: " + fileName, JOptionPane.INFORMATION_MESSAGE);
                status("Count dialog closed");
	} // End of countCall method

	/**
	* Calculates and displays the time lapsed between various points
	*@param none
	*@return none
	*/
	public void timelapseCall()
	{
                status("Time Lapse dialog created");
                Time current = new Time(System.currentTimeMillis());
                JOptionPane.showMessageDialog(window, ("<HTML>Document: <FONT FACE = \"DialogInput\">" +
                                             (new Time((current.getTime() - documentStart.getTime()) - 61200000L).toString()) +
                                                    "\n<HTML>Program: <FONT FACE = \"DialogInput\">" +
                                             (new Time((current.getTime() - programStart.getTime()) - 61200000L).toString())),
                                              ("Time Lapse to " + current.toString()), JOptionPane.INFORMATION_MESSAGE);
                status("Time Lapse dialog closed");
	} // End of timelapseCall method

	/**
	* Displays information on the Java Virtual Machine
	*@param none
	*@return none
	*/
	public void virtualmachineCall()
	{
                // Display the memory information, and give option to run gc
		status("Virtual Machine dialog created");
                int choice = JOptionPane.showConfirmDialog(window, ("<HTML>Free Memory: <FONT COLOR = red>" + core.freeMemory()/1000 + "<FONT COLOR = black>kb of <FONT COLOR = red>" +
                                                                    core.totalMemory()/1000 + "<FONT COLOR = black>kb" + "\nRun the garbage collector?"),
                                              "JVM Memory Report", JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);

                if (choice == 0) // Yes clicked
                {
                        core.gc();
                        status("Garbage collector executed (new free memory: " + core.freeMemory()/1000 + "kb)");
                }
                else // No clicked
                        status("Garbage collector was not executed");
	} // End of virtualmachineCall method

	/**
        * Brings up the Readme.txt dialog
	*@param none
        *@return none
	*/
	public void helpCall()
	{
		// Set dialog specific options and bounds
                popupDialog.setTitle("Help - Readme.txt");
                popupDialog.setBounds(0, 0, 700, 480);
		popupDialog.setLocationRelativeTo(window);

		// Create the main text area and properties
                JTextArea helpDisplay = new JTextArea("", 25, 59);
                helpDisplay.setEditable(false);
                helpDisplay.setBackground(Color.WHITE);
                helpDisplay.setForeground(Color.BLACK);
                helpDisplay.setLineWrap(true);
                helpDisplay.setWrapStyleWord(true);

		// Output the end result after a series of checks
		try{
                        File helpFile = new File("Readme.txt");
                        // Check if file exists before displaying
                        if (helpFile.exists())
			{
                                // Fill the text area with the file
                                helpDisplay.read(new BufferedReader(new FileReader(helpFile.getName())), "Text");
				popupPane.removeAll();
                                popupPane.add(new JScrollPane(helpDisplay));
				popupPane.add(closeButton);
				popupDialog.setVisible(true);
                                helpDisplay.requestFocus();
                                status("Readme.txt dialog created");
			}
			else // File not found
			{
                                JOptionPane.showMessageDialog(window, "The file Readme.txt could not be found in the running directory.\n",
                                                              "Readme Error", JOptionPane.ERROR_MESSAGE);
                                status("Readme.txt was not found in the running directory");
			}
                core.gc();
		}catch (Exception all) {
			handleError(all);
		} // End of try-catch
	} // End of helpCall method

	/**
	* Brings up the about dialog
	*@param none
	*@return none
	*/
	public void aboutCall()
	{
		status("About Scribe v" + VERSION + " dialog created");
                JOptionPane.showMessageDialog(window, "<HTML><UL>" +
                                                      "<LI> Carlo Guglielmin" +
                                                      "<LI> July 15, 2004" +
                                                      "<LI> GNU General Public License" +
                                                      "<LI> Dedicated to Kat" +
                                                      "</UL></FONT></HTML>",
                                             ("About Scribe v" + VERSION), JOptionPane.PLAIN_MESSAGE);
                status("About Scribe v" + VERSION + " dialog closed");
	} // End of aboutCall method

	/**
	* Brings up the GNU General Public License dialog
	*@param none
	*@return none
	*/
	public void licenseCall()
	{
		// Set dialog specific options and bounds
		popupDialog.setTitle("GNU General Public License");
                popupDialog.setBounds(0, 0, 600, 480);
		popupDialog.setLocationRelativeTo(window);

		// Create the main text area and properties
                JTextArea licenseDisplay = new JTextArea("", 25, 50);
		licenseDisplay.setEditable(false);
		licenseDisplay.setBackground(Color.WHITE);
		licenseDisplay.setForeground(Color.BLACK);
		licenseDisplay.setLineWrap(true);
		licenseDisplay.setToolTipText("Homepage at http://www.gnu.org/copyleft/gpl.html");
		licenseDisplay.setWrapStyleWord(true);

		// Output the end result after a series of checks
		try{
                        File licenseFile = new File("License.txt");
                        // Check if file exists before displaying
                        if (licenseFile.exists())
			{
                                // Fill the text area with the file
				licenseDisplay.read(new BufferedReader(new FileReader(licenseFile.getName())), "Text");
				licenseDisplay.append("\n[The GNU General Public License was copied directly from License.txt in the running directory.]");
				popupPane.removeAll();
				popupPane.add(new JScrollPane(licenseDisplay));
				popupPane.add(closeButton);
				popupDialog.setVisible(true);
                                licenseDisplay.requestFocus();
				status("GNU General Public License dialog created");
			}
			else // File not found
			{
                                JOptionPane.showMessageDialog(window, "The file License.txt could not be found in the running directory.\n" +
                                                              "Please fix the problem or download a new copy from\n" +
                                                              "http://www.gnu.org/copyleft/gpl.html as soon as possible.",
                                                              "License Error", JOptionPane.WARNING_MESSAGE);
                                status("License.txt was not found in the running directory");
			}
                core.gc();
		}catch (Exception all) {
			handleError(all);
		} // End of try-catch
	} // End of licenseCall method
} // End of Scribe class
