/****************************************
Chaplin Corp. 1999
Algorithm v1.0
--------------------------
Algorithm for growing cells in an environment.
--------------------------
This source code is not copyright and may
be used for anything as long as Chaplin Corp.
is mentioned in the final product as the creator
of the algorithm.

Feel free to modify or tweak this code as long
as Chaplin Corp. remains mentioned.
*****************************************/


#include "celltype.h"
#include <stdlib.h>		//used for rand()

/*******************************************************
This is the CELLTYPE constructor, it initializes
the cell and slips it into the cell queue
*******************************************************/
CELLTYPE::CELLTYPE (char c_energy, char c_race, 
					CELLTYPE *c_parent, int c_x, int c_y)
{
	//initialize the energy
	m_energy = c_energy;
	//initialize the race
	m_race = c_race;
	//the parent cell
	m_parent = c_parent;
	//the coordinates
	m_x = c_x; m_y = c_y;
	//if there is a parent
	if (c_parent)
	{
		//slip between the parent and the parent's nextcell
		m_nextcell = c_parent->m_nextcell;
		if (c_parent->m_nextcell)
			(c_parent->m_nextcell)->m_parent = this;
		c_parent->m_nextcell = this;
	}
	else
	{
		m_nextcell = 0;
	}

}


/*******************************************************
This is the CELLTYPE destructor, it removes the cell
from the queue and ties it back together so it remains
intact
*******************************************************/
CELLTYPE::~CELLTYPE()
{
	//if there is a parent
	if (m_parent)
	{
		//restore the link between the parent and the nextcell
		m_parent->m_nextcell = m_nextcell;
		if (m_nextcell)
			m_nextcell->m_parent = m_parent;
	}
	else if (m_nextcell)
	{
		m_nextcell->m_parent = 0;
	}

}



/*******************************************************
This is the CELLMAP constructor, it initializes everything,
allocates memory for the 2 grids, places all the food
and creates all the beginning cells
*******************************************************/
CELLMAP::CELLMAP(unsigned long w, unsigned long h, int nfood,
				 int foodgrown, int ncell, unsigned long cellvitality,
				 unsigned long cellspeed, int seed)
{
	//make sure cellvitality and cellspeed are above 0;
	if (!cellvitality) cellvitality++;
	if (!cellspeed) cellspeed++;
	m_cellvitality = cellvitality;
	m_cellspeed = cellspeed;
	//init the coordinates
	m_width = w; m_height = h;
	//init all counters
	m_foodNum = 0;
	m_cellNum = 0;
	m_deadFood = 0;
	m_deadCell = 0;
	//amount of food grown per cycle
	m_foodgrown = foodgrown;
	//init the random number generator
	srand(seed);
	//allocate grid memory
	m_cellgrid = new GRIDSTATE[w*h];
	//allocate bitmap memory
	m_colorgrid = new unsigned long[w*h];
	//zero-out everything
	for (unsigned int x = 0; x < w; x++)
	for (unsigned int y = 0; y < h; y++)
	{
		GRIDSTATE gs;
		gs.type = L_EMPTY;
		gs.data = 0;
		gs.energy = 0;
		SetGridType(x, y, gs);
	};
	//food cordinates
	int fx, fy;
	//if there is to be food, then create it
	if (nfood)
	{
		//for every food
		for (int i = 0; i < nfood; i++)
		{
			//create a random food
			do
			{
				fx = rand() % m_width;
				fy = rand() % m_height;
			}while (GetGridType(fx, fy));	//make sure its an empty position
			//update the amount of food
			m_foodNum++;
			//place the food on the grid
			GRIDSTATE gs;
			gs.type = L_FOOD;
			gs.data = 0;
			gs.energy = 1;
			SetGridType(fx, fy, gs);
		}
	}
	
	//create the original cell
	//m_cell is not placed in the window because we need it to
	//empty the cell queue in the CellMap destructor
	m_cell = new CELLTYPE (5, 1, 0, 0, 0);
	//create all the cells
	if (ncell)
	{
		//cells coordinates
		int cx, cy;
		for (int i = 0; i < ncell; i++)
		{
			//position a random cell
			do
			{
				cx = rand() % m_width;
				cy = rand() % m_height;
			}while (GetGridType(cx, cy));
			AdjustXY(cx, cy);
			//create the new cell at (x, y)
			CELLTYPE *c = new CELLTYPE(6, 1, m_cell, cx, cy);
			//add 1 to the amount of cells on the grid
			m_cellNum++;
			//place the cell on the grid
			GRIDSTATE gs;
			gs.type = L_CELL;
			gs.data = (unsigned long*)c;
			gs.energy = c->m_energy;
			SetGridType(cx, cy, gs);
		}
	}


}

/*******************************************************
The CELLMAP destructor, frees all allocated memory
*******************************************************/
CELLMAP::~CELLMAP()
{
	//free all memory allocated by the grids
	delete[] m_cellgrid;
	delete[] m_colorgrid;
	//delete all cells
	CELLTYPE *cc = m_cell->m_nextcell;
	CELLTYPE *cd = m_cell;
	do
	{
		//delete the cell
		delete cd;
		//this is the next cell
		cd = cc;
		//if its a valid cell, save its nextcell
		if (cc)
			cc = cc->m_nextcell;
	}
	while (cd);		//while its a valid cell
}

/*******************************************************
Returns the type of data located on the cell grid
*******************************************************/
unsigned char CELLMAP::GetGridType(int x, int y)
{
	//return the type at grid(x, y)
	return m_cellgrid[y*m_width+x].type;
}

/*******************************************************
Sets the right type of info on the cellgrid and colors
the color grid with the right colors
*******************************************************/
void CELLMAP::SetGridType(int x, int y, GRIDSTATE gstate)
{
	m_cellgrid[y*m_width+x] = gstate;
	switch (gstate.type)
	{
	case L_CELL: m_colorgrid[y*m_width+x] = 
					 colRGB(0, 0, gstate.energy*42);
		break;
	case L_FOOD: m_colorgrid[y*m_width+x] = 
					 colRGB(0, gstate.energy*42, 0);
		break;
	case L_EMPTY: m_colorgrid[y*m_width+x] = colRGB(0, 0, 0);
		break;
	}
}

/*******************************************************
This makes sure the x and y are within the map's boundaries
*******************************************************/
void CELLMAP::AdjustXY(int &x, int &y)
{
	if (x < 0)
	{
		x = m_width + x;
	}
	else if (x >= m_width)
	{
		x = x - m_width;
	}
	if (y < 0)
	{
		y = m_height + y;
	}
	else if (y >= m_height)
	{
		y = y - m_height;
	}
}


/*******************************************************
This it the main cell life cycle, it goes through every
cells once and moves, feeds or reproduces it.
*******************************************************/
void CELLMAP::CellCycle()
{
	CELLTYPE *currentcell = m_cell->m_nextcell;
	CELLTYPE *nextcell;
	//while were dealing with a good cell
	while (currentcell)
	{
		//save the next cell in case currentcell gets deleted
		nextcell = currentcell->m_nextcell;

		//if it needs to feed
		if (currentcell->m_energy < 6)
		{
			//makes a random decision
			int feel = rand() % 8;
			bool actionDone = false;
			int counter = 8;
			int x, y;
			//looks for food and eats it if found any
			while (!actionDone && counter)
			{
				switch (feel)
				{
				case 0: x = currentcell->m_x - 1;
						y = currentcell->m_y - 1;
						break;
				case 1: x = currentcell->m_x;
						y = currentcell->m_y - 1;
						break;
				case 2: x = currentcell->m_x + 1;
						y = currentcell->m_y - 1;
						break;
				case 3: x = currentcell->m_x - 1;
						y = currentcell->m_y;
						break;
				case 4: x = currentcell->m_x + 1;
						y = currentcell->m_y;
						break;
				case 5: x = currentcell->m_x - 1;
						y = currentcell->m_y + 1;
						break;
				case 6: x = currentcell->m_x;
						y = currentcell->m_y + 1;
						break;
				case 7: x = currentcell->m_x + 1;
						y = currentcell->m_y + 1;
						break;
				}
				//make sure we do this a maximum of 8 times
				counter--;
				if (!feel--) feel = 7;
				AdjustXY(x, y);
				//if we find food there
				if (GetGridType(x, y) == L_FOOD)
				{
					//get a hold of the food energy
					int energy = m_cellgrid[x+y*m_width].energy;

					//transfer energy from food to cell
					energy--;
					currentcell->m_energy++;
					//if food has no more energy then erase it
					if (!energy)
					{
						//one more dead food
						m_deadFood++;
						//one less food
						m_foodNum--;
						//empty the place where the plant used to be
						GRIDSTATE gs;
						gs.type = L_EMPTY;
						gs.data = 0;
						gs.energy = 0;
						SetGridType(x, y, gs);
					}
					else
					{
						//update the food energy
						GRIDSTATE gs;
						gs.type = L_FOOD;
						gs.data = 0;
						gs.energy = energy;
						SetGridType(x, y, gs);
					}
					actionDone = true;
				}
			}
			counter = 8;
			//a chance the cell remains idle
			if ((rand()%m_cellspeed)) actionDone = true;
			//if it ate, then it won't move
			//looks around to see if it can move
			while (!actionDone && counter)
			{

				switch (feel)
				{
				case 0: x = currentcell->m_x - 1;
						y = currentcell->m_y - 1;
						break;
				case 1: x = currentcell->m_x;
						y = currentcell->m_y - 1;
						break;
				case 2: x = currentcell->m_x + 1;
						y = currentcell->m_y - 1;
						break;
				case 3: x = currentcell->m_x - 1;
						y = currentcell->m_y;
						break;
				case 4: x = currentcell->m_x + 1;
						y = currentcell->m_y;
						break;
				case 5: x = currentcell->m_x - 1;
						y = currentcell->m_y + 1;
						break;
				case 6: x = currentcell->m_x;
						y = currentcell->m_y + 1;
						break;
				case 7: x = currentcell->m_x + 1;
						y = currentcell->m_y + 1;
						break;
				}
				counter--;
				//feel decides where it starts to look first
				if (!feel--) feel = 7;
				//adjust the coordinates so we dont have any negatives
				AdjustXY(x, y);
				//if the position is empty then the cell moves there
				if (GetGridType(x, y) == L_EMPTY)
				{
					//empty where the cell used to be
					GRIDSTATE gs;
					gs.type = L_EMPTY;
					gs.data = 0;
					gs.energy = 0;
					SetGridType(currentcell->m_x, currentcell->m_y, gs);
					//set the cell internal new position
					currentcell->m_x = x;
					currentcell->m_y = y;
					//if its strong and lucky, it wont lose any energy this cycle
					if ((rand()%m_cellvitality)) currentcell->m_energy++;
					//if it has no more energy kill it
					if (!--(currentcell->m_energy))
					{
						//delete the cell
						delete currentcell;
						//update info
						m_cellNum--;
						m_deadCell++;
						//empty the place where the cell used to be
						gs.type = L_EMPTY;
						gs.data = (unsigned long*)0;
						gs.energy = 0;
						SetGridType(x, y, gs);
					}
					else
					{
						//save the new cell position on the grid
						gs.type = L_CELL;
						gs.data = (unsigned long*)currentcell;
						gs.energy = currentcell->m_energy;
						SetGridType(x, y, gs);
					}
					actionDone = true;
				}
			}

		}
		else	//reproduce
		{
			//makes a random decision
			int feel = rand() % 8;
			bool actionDone = false;
			int counter = 8;
			int x, y;
			//looks around to find an empty spot
			while (!actionDone && counter)
			{
				switch (feel)
				{
				case 0: x = currentcell->m_x - 1;
						y = currentcell->m_y - 1;
						break;
				case 1: x = currentcell->m_x;
						y = currentcell->m_y - 1;
						break;
				case 2: x = currentcell->m_x + 1;
						y = currentcell->m_y - 1;
						break;
				case 3: x = currentcell->m_x - 1;
						y = currentcell->m_y;
						break;
				case 4: x = currentcell->m_x + 1;
						y = currentcell->m_y;
						break;
				case 5: x = currentcell->m_x - 1;
						y = currentcell->m_y + 1;
						break;
				case 6: x = currentcell->m_x;
						y = currentcell->m_y + 1;
						break;
				case 7: x = currentcell->m_x + 1;
						y = currentcell->m_y + 1;
						break;
				}
				counter--;
				if (!feel--) feel = 7;
				AdjustXY(x, y);
				//if its empty then reproduce there
				if (GetGridType(x, y) == L_EMPTY)
				{
					//create the new cell at the empty position
					CELLTYPE *temp = new CELLTYPE(1, 0, currentcell, x, y);
					//number of cells alive + 1
					m_cellNum++;
					//save the new cell position on the grid
					GRIDSTATE gs;
					gs.type = L_CELL;
					gs.data = (unsigned long*)temp;
					gs.energy = temp->m_energy;
					SetGridType(x, y, gs);
					//next cell's turn
					actionDone = true;
					//lose a bit of energy from reproducing
					currentcell->m_energy-=1;
				}
			}
		}
		currentcell = nextcell;
	}
}

/*******************************************************
This is the main food life cycle, it randomly
grows food on the map
*******************************************************/
void CELLMAP::FoodCycle()
{
	//chance2spread decides if the cell grow by alot
	//or just a little
	int chance2spread, x, y, type;
	//loop depending on how many foods are grown per cycle
	for (unsigned int i = 0; i < m_foodgrown; i++)
	{
		x = rand() % m_width;
		y = rand() % m_height;
		//if the location is empty
		type = GetGridType(x, y);
		if (!type)
		{
			//add a new food there
			GrowFood(x, y);
		}
		//if there's already a food there
		else if (type == L_FOOD)
		{
			chance2spread = rand() % 4;
			GrowFood(x, y);
			//grow the food in all directions
			if (!chance2spread)
			{
				GrowFood(x-1, y);
				GrowFood(x+1, y);
				GrowFood(x-1, y-1);
				GrowFood(x, y-1);
				GrowFood(x+1, y-1);
				GrowFood(x-1, y+1);
				GrowFood(x, y+1);
				GrowFood(x+1, y+1);
			}
		}
	}
}

/*******************************************************
This grows the food at the specified location
*******************************************************/
void CELLMAP::GrowFood(int x, int y)
{
	AdjustXY(x, y);
	if (GetGridType(x, y) != L_CELL)
	{
		//get the food energy
		int energy = m_cellgrid[x+y*m_width].energy;
		//raise it
		if (energy++ > 5) energy = 6;
		//save the new food energy on the grid
		GRIDSTATE gs;
		gs.type = L_FOOD;
		gs.data = 0;
		gs.energy = energy;
		SetGridType(x, y, gs);
	}
}
