#ifndef __UTILC
#define __UTILC

#include "globals.h"
#include "util.h"
#include "yytab.h"


FILE *listing;
extern int linenr;

/* Procedure printToken prints a token
 * and its lexeme to the listing file
*/

void printToken( TokenType token, const char* tokenString )
{
switch (token)
  { case IF:
    case THEN:
    case ELSE:
    case WHILE:
    case FOR:
    case TO:
    case STEP:
    case READ:
    case WRITE:
    case INTK:
    case REALK:
    case STP:
    case STB:
    case ENDB:
    case ENDP:
      fprintf(listing,
         "reserved word: %s\n",tokenString);
      break;
    case ASSIGN: fprintf(listing,"=\n"); break;
    case LT: fprintf(listing,"<\n"); break;
    case EQ: fprintf(listing,"==\n"); break;
    case GT: fprintf(listing,">\n"); break;
    case DIF: fprintf(listing,"<>\n"); break;
    case LPAR: fprintf(listing,"(\n"); break;
    case RPAR: fprintf(listing,")\n"); break;
    case SC: fprintf(listing,";\n"); break;
    case ADD: fprintf(listing,"+\n"); break;
    case SUB: fprintf(listing,"-\n"); break;
    case MUL: fprintf(listing,"*\n"); break;
    case DIV: fprintf(listing,"/\n"); break;
    case POW: fprintf(listing,"^\n"); break;
    case INT:
    case REAL:
      fprintf(listing,"NUM, val= %s\n",tokenString);
      break;
    case ID:
      fprintf(listing,"ID, name= %s\n",tokenString);
      break;
    case ERR:
      fprintf(listing,"ERROR: %s\n",tokenString);
      break;
    default: // should never happen
      fprintf(listing,"Unknown token: %d\n",token);
  }

}


/* Function newStmtNode creates a new statement
 * node for syntax tree construction
 */
TreeNode * newStmtNode(StmtKind kind)
{ TreeNode *t = new TreeNode;
  int i;
  if (t==NULL)
    ShowMessage("Out of memory error");
  else {
    for (i=0;i<MAXCHILDREN;i++) t->child[i] = NULL;
    t->sibling = NULL;
    t->nodekind = StmtK;
    t->kind.stmt = kind;
    t->lineno = linenr;
  }
  return t;
}

/* Function newExpNode creates a new expression 
 * node for syntax tree construction
 */
TreeNode * newExpNode(ExpKind kind)
{ TreeNode * t = new TreeNode;
  int i;
  if (t==NULL)
    ShowMessage("Out of memory error at line");
  else {
    for (i=0;i<MAXCHILDREN;i++) t->child[i] = NULL;
    t->sibling = NULL;
    t->nodekind = ExpK;
    t->kind.exp = kind;
    t->lineno = linenr;
    t->type = Void;
  }
  return t;
}

/* Function copyString allocates and makes a new
 * copy of an existing string
 */
char * copyString(char * s)
{ int n;
  char *tt;
  if (s==NULL) return NULL;
  n = strlen(s)+1;
  tt = (char*)malloc(n);
  if (tt==NULL)
    ShowMessage("Out of memory error at line");
  else strcpy(tt,s);
  return tt;
}

/* Variable indentno is used by printTree to
 * store current number of spaces to indent
 */
static indentno = -1;

/* macros to increase/decrease indentation */
#define INDENT indentno++
#define UNINDENT indentno--

/* printSpaces indents by printing spaces */
static void printSpaces(void)
{ int i;
  for (i=0;i<indentno;i++)
    fprintf(listing,"\t");
}


/* procedure printTree prints a syntax tree to the
 * listing file using indentation to indicate subtrees
 */
void printTree( TreeNode * tree )
{ int i;

  INDENT;

  while (tree != NULL) {
    printSpaces();
    if (tree->nodekind==StmtK)
    { switch (tree->kind.stmt) {
        case IfK:
          fprintf(listing,"If\n");
          break;
        case ThenK:
          fprintf(listing,"Then\n");
          break;
        case ElseK:
          fprintf(listing,"Else\n");
          break;
        case WhileK:
          fprintf(listing,"While\n");
          break;
        case ForK:
          fprintf(listing,"For\n");
          break;
        case ToK:
          fprintf(listing,"To\n");
          break;
        case StepK:
          fprintf(listing,"Step\n");
          break;
        case DoK:
          fprintf(listing,"Do\n");
          break;
        case AssignK:
          fprintf(listing,"Assign to: %s\n",tree->attr.name);
          break;
        case ReadK:
          fprintf(listing,"Read: %s\n",tree->attr.name);
          break;
        case WriteK:
          fprintf(listing,"Write\n");
          break;
        case DeclK:
          fprintf(listing,"Declare\n");
          break;
        default:
          fprintf(listing,"Unknown ExpNode kind\n");
          break;
      }
    }
    else if (tree->nodekind==ExpK)
    { switch (tree->kind.exp) {
        case OpK:
          fprintf(listing,"Op: ");
          printToken(tree->attr.op,"\0");
          break;
        case TestK:
          fprintf(listing,"Test: ");
          printToken(tree->attr.op,"\0");
          break;
        case ConstK:
          if (tree->type==INT_TYPE) 
             fprintf(listing,"Const: %d\n",(int)tree->attr.val);
          else
              fprintf(listing,"Const: %f\n",tree->attr.val);
          break;
        case IdK:
          if (tree->type==INT_TYPE)
             fprintf(listing,"Id: %s (int)\n",tree->attr.name);
          else if (tree->type==REAL_TYPE)
             fprintf(listing,"Id: %s (real)\n",tree->attr.name);
          else
             fprintf(listing,"Id: %s\n",tree->attr.name);
          break;
        default:
          fprintf(listing,"Unknown ExpNode kind\n");
          break;
      }
    }
    else fprintf(listing,"Unknown node kind\n");
    for (i=0;i<MAXCHILDREN;i++)
        if (tree->child[i]!=NULL)
           printTree(tree->child[i]);
    tree = tree->sibling;
  }
  UNINDENT;
}


void start(TreeNode *tree)
{
listing = fopen("arbore.txt", "wt");
if (listing==NULL)
   {
   ShowMessage("Eroare fisier");
   return;
   }
printTree(tree);
fclose(listing);
return;
}

//---------------------------------------------------------------------------

#endif

