/**************************************************************************
 *                                                                        *
 *  File:        hyper.h                                                  *
 *  Copyright:   (C) 2001, Florin Leon                                    *
 *  E-mail:      florinleon@yahoo.com                                     *
 *  Website:     http://www.angelfire.com/home/florinleon                 *
 *  Description: Hypercomplex 1.1 Library                                 *
 *               A header for hypercomplex numbers (x + yi + zj + wk)     *
 *               operations. It includes elementary functions (+,-,*,/),  *
 *               power, exponential, logarithm, trigonometric functions.  *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or modify  *
 *  it under the terms of the GNU General Public License as published by  *
 *  the Free Software Foundation; either version 1, or (at your option)   *
 *  any later version.                                                    *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful,       *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *  GNU General Public License for more details.                          *
 *                                                                        *
 *  You should have received a copy of the GNU General Public License     *
 *  along with this program; if not, write to the Free Software           *
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             *
 *                                                                        *
 **************************************************************************/

#ifndef _HYPER_H
#define _HYPER_H

#include <iostream.h>
#include <math.h>


class hypercomplex
{
    double r,i,j,k;

public:
    hypercomplex()
    {
        r=i=j=k=0;
    };

    hypercomplex(double x, double y=0, double z=0, double w=0)
    {
        r=x; i=y; j=z; k=w;
    };

    hypercomplex(const hypercomplex &h)
    {
        r=h.r; i=h.i; j=h.j; k=h.k;
    };

    ~hypercomplex() {};

    double GetReal() {return r;};
    void SetReal(double x) {r=x;};
    double GetImI() {return i;};
    void SetImI(double y) {i=y;};
    double GetImJ() {return j;};
    void SetImJ(double z) {j=z;};
    double GetImK() {return k;};
    void SetImK(double w) {k=w;};

    hypercomplex operator +(const hypercomplex &h);
    hypercomplex operator -(const hypercomplex &h);
    hypercomplex operator *(const hypercomplex &h);
    hypercomplex operator /(const hypercomplex &h);
    hypercomplex operator ^(hypercomplex h);
    hypercomplex& operator =(const hypercomplex &h);
    int operator ==(const hypercomplex &h);
    int operator !=(const hypercomplex &h);

    friend ostream& operator <<(ostream &out_data, const hypercomplex &h);
};

hypercomplex exp(hypercomplex h);
hypercomplex log(hypercomplex h);

hypercomplex sqrt(hypercomplex h);
hypercomplex conj(hypercomplex h);
double abs(hypercomplex h);

hypercomplex sin(hypercomplex h);
hypercomplex cos(hypercomplex h);
hypercomplex tan(hypercomplex h);
hypercomplex asin(hypercomplex h);
hypercomplex acos(hypercomplex h);
hypercomplex atan(hypercomplex h);

//---------------------------------------------------------------------------

hypercomplex hypercomplex::operator +(const hypercomplex &h)
{
    hypercomplex t;
    t.r = r+h.r;
    t.i = i+h.i;
    t.j = j+h.j;
    t.k = k+h.k;
    return t;
}


hypercomplex hypercomplex::operator -(const hypercomplex &h)
{
    hypercomplex t;
    t.r = r-h.r;
    t.i = i-h.i;
    t.j = j-h.j;
    t.k = k-h.k;
    return t;
}


hypercomplex hypercomplex::operator *(const hypercomplex &h)
{
    hypercomplex t;
    t.r = r*h.r - i*h.i - j*h.j + k*h.k;
    t.i = i*h.r + r*h.i - k*h.j - j*h.k;
    t.j = j*h.r - k*h.i + r*h.j - i*h.k;
    t.k = k*h.r + j*h.i + i*h.j + r*h.k;
    return t;
}


hypercomplex hypercomplex::operator /(const hypercomplex &h)
{
    double det;

    det = ((h.r-h.k)*(h.r-h.k) + (h.i+h.j)*(h.i+h.j)) *
        ((h.r+h.k)*(h.r+h.k) + (h.i-h.j)*(h.i-h.j));

    if (det==0)
    {
        cout << "Divide error" << endl;
        return *this;
    }

    hypercomplex invh,temp(r,i,j,k);
    double mod = h.r*h.r + h.i*h.i + h.j*h.j + h.k*h.k;

    invh.r = (h.r*mod - 2*h.k*(h.r*h.k-h.i*h.j))/det;
    invh.i = (-h.i*mod - 2*h.j*(h.r*h.k-h.i*h.j))/det;
    invh.j = (-h.j*mod - 2*h.i*(h.r*h.k-h.i*h.j))/det;
    invh.k = (h.k*mod - 2*h.r*(h.r*h.k-h.i*h.j))/det;

    return temp*invh;
}


hypercomplex hypercomplex::operator ^(hypercomplex h)
{
    hypercomplex t(r,i,j,k);
    return exp(h*log(t));
}


hypercomplex& hypercomplex::operator =(const hypercomplex &h)
{
    if (&h!=this)
        {
        r=h.r; i=h.i; j=h.j; k=h.k;
        }
    return *this;
}


int hypercomplex::operator ==(const hypercomplex &h)
{
    if (r==h.r && i==h.i && j==h.j && k==h.k) return 1;
    else return 0;
}


int hypercomplex::operator !=(const hypercomplex &h)
{
    if (r==h.r && i==h.i && j==h.j && k==h.k) return 0;
    else return 1;
}


ostream& operator <<(ostream &out_data, const hypercomplex &h)
{
    out_data << h.r;
    (h.i<0) ? out_data << " - " << -h.i : out_data << " + " << h.i;
    (h.j<0) ? out_data << "i - " << -h.j : out_data << "i + " << h.j;
    (h.k<0) ? out_data << "j - " << -h.k : out_data << "j + " << h.k;
    out_data << "k" << endl;
    return out_data;
}

//---------------------------------------------------------------------------

hypercomplex exp(hypercomplex h)
{
    hypercomplex t;
    double a_re,a_im, b_re,b_im, c_re,c_im, d_re,d_im;

    a_re = h.GetReal()-h.GetImK();
    a_im = h.GetImI()+h.GetImJ();
    b_re = h.GetReal()+h.GetImK();
    b_im = h.GetImI()-h.GetImJ();

    c_re = exp(a_re)*cos(a_im);
    c_im = exp(a_re)*sin(a_im);
    d_re = exp(b_re)*cos(b_im);
    d_im = exp(b_re)*sin(b_im);

    t.SetReal((c_re+d_re)/2.);
    t.SetImI((c_im+d_im)/2.);
    t.SetImJ((c_im-d_im)/2.);
    t.SetImK((d_re-c_re)/2.);

    return t;
}

//---------------------------------------------------------------------------

hypercomplex log(hypercomplex h)
{
    hypercomplex t;
    double a_re,a_im, b_re,b_im, c_re,c_im, d_re,d_im;

    a_re = h.GetReal()-h.GetImK();
    a_im = h.GetImI()+h.GetImJ();
    b_re = h.GetReal()+h.GetImK();
    b_im = h.GetImI()-h.GetImJ();

    c_re = 0.5*log(a_re*a_re+a_im*a_im);
    c_im = atan2(a_im,a_re);
    d_re = 0.5*log(b_re*b_re+b_im*b_im);
    d_im = atan2(b_im,b_re);

    t.SetReal((c_re+d_re)/2.);
    t.SetImI((c_im+d_im)/2.);
    t.SetImJ((c_im-d_im)/2.);
    t.SetImK((d_re-c_re)/2.);

    return t;
}

//---------------------------------------------------------------------------

hypercomplex sqrt(hypercomplex h)
{
    return h^0.5;
}


hypercomplex conj(hypercomplex h)
{
    hypercomplex t;
    double a_re,a_im, b_re,b_im, c_re,c_im, d_re,d_im;

    a_re = h.GetReal()-h.GetImK();
    a_im = h.GetImI()+h.GetImJ();
    b_re = h.GetReal()+h.GetImK();
    b_im = h.GetImI()-h.GetImJ();

    c_re = a_re;
    c_im = -a_im;
    d_re = b_re;
    d_im = -b_im;

    t.SetReal((c_re+d_re)/2.);
    t.SetImI((c_im+d_im)/2.);
    t.SetImJ((c_im-d_im)/2.);
    t.SetImK((d_re-c_re)/2.);

    return t;
}


double abs(hypercomplex h)
{
    return sqrt( h.GetReal()*h.GetReal() + h.GetImI()*h.GetImI()+
        h.GetImJ()*h.GetImJ()+ h.GetImK()*h.GetImK());
}

//---------------------------------------------------------------------------

hypercomplex sin(hypercomplex h)
{
    hypercomplex t;
    double a_re,a_im, b_re,b_im, c_re,c_im, d_re,d_im;

    a_re = h.GetReal()-h.GetImK();
    a_im = h.GetImI()+h.GetImJ();
    b_re = h.GetReal()+h.GetImK();
    b_im = h.GetImI()-h.GetImJ();

    c_re = 0.5*sin(a_re)*(exp(a_im)+exp(-a_im));
    c_im = 0.5*cos(a_re)*(exp(a_im)-exp(-a_im));
    d_re = 0.5*sin(b_re)*(exp(b_im)+exp(-b_im));
    d_im = 0.5*cos(b_re)*(exp(b_im)-exp(-b_im));

    t.SetReal((c_re+d_re)/2.);
    t.SetImI((c_im+d_im)/2.);
    t.SetImJ((c_im-d_im)/2.);
    t.SetImK((d_re-c_re)/2.);

    return t;
}


hypercomplex cos(hypercomplex h)
{
    hypercomplex t;
    double a_re,a_im, b_re,b_im, c_re,c_im, d_re,d_im;

    a_re = h.GetReal()-h.GetImK();
    a_im = h.GetImI()+h.GetImJ();
    b_re = h.GetReal()+h.GetImK();
    b_im = h.GetImI()-h.GetImJ();

    c_re = 0.5*cos(a_re)*(exp(a_im)+exp(-a_im));
    c_im = 0.5*sin(a_re)*(exp(-a_im)-exp(a_im));
    d_re = 0.5*cos(b_re)*(exp(b_im)+exp(-b_im));
    d_im = 0.5*sin(b_re)*(exp(-b_im)-exp(b_im));

    t.SetReal((c_re+d_re)/2.);
    t.SetImI((c_im+d_im)/2.);
    t.SetImJ((c_im-d_im)/2.);
    t.SetImK((d_re-c_re)/2.);

    return t;
}


hypercomplex tan(hypercomplex h)
{
    hypercomplex t;
    double a_re,a_im, b_re,b_im, c_re,c_im, d_re,d_im;

    a_re = h.GetReal()-h.GetImK();
    a_im = h.GetImI()+h.GetImJ();
    b_re = h.GetReal()+h.GetImK();
    b_im = h.GetImI()-h.GetImJ();

    c_re = 2*exp(-2*a_im)*sin(2*a_re)/(2*exp(-2*a_im)*cos(2*a_re)+1+exp(-4*a_im));
    c_im = (1-exp(-4*a_im))/(2*exp(-2*a_im)*cos(2*a_re)+1+exp(-4*a_im));
    d_re = 2*exp(-2*b_im)*sin(2*b_re)/(2*exp(-2*b_im)*cos(2*b_re)+1+exp(-4*b_im));
    d_im = (1-exp(-4*b_im))/(2*exp(-2*b_im)*cos(2*b_re)+1+exp(-4*b_im));

    t.SetReal((c_re+d_re)/2.);
    t.SetImI((c_im+d_im)/2.);
    t.SetImJ((c_im-d_im)/2.);
    t.SetImK((d_re-c_re)/2.);

    return t;
}


int csgn(double re, double im)
{
    if (re>0 || (re==0 && im>0)) return 1;
    if (re<0 || (re==0 && im<0)) return -1;
    return 0;
}


hypercomplex asin(hypercomplex h)
{
    hypercomplex t;
    double a_re,a_im, b_re,b_im, c_re,c_im, d_re,d_im;

    a_re = h.GetReal()-h.GetImK();
    a_im = h.GetImI()+h.GetImJ();
    b_re = h.GetReal()+h.GetImK();
    b_im = h.GetImI()-h.GetImJ();

    double s = sqrt(1 - 2*a_re*a_re + 2*a_im*a_im + pow(a_re,4) +
        2*a_re*a_re*a_im*a_im + pow(a_im,4));

    c_re = atan2(0.5*csgn(-2*a_re*a_im, a_re*a_re-a_im*a_im-1) *
        sqrt(2*s-2+2*a_re*a_re-2*a_im*a_im) + a_re ,
        0.5*sqrt(2*s+2-2*a_re*a_re+2*a_im*a_im)-a_im);
    c_im = -0.5 * log(s+a_re*a_re+a_im*a_im-sqrt(2*s+2-2*a_re*a_re+2*a_im*a_im) * a_im +
        csgn(-2*a_re*a_im, a_re*a_re-a_im*a_im-1) *
        sqrt(2*s-2+ 2*a_re*a_re - 2*a_im*a_im)*a_re);

    d_re = atan2(0.5*csgn(-2*b_re*b_im, b_re*b_re-b_im*b_im-1) *
        sqrt(2*s-2+2*b_re*b_re-2*b_im*b_im) + b_re ,
        0.5*sqrt(2*s+2-2*b_re*b_re+2*b_im*b_im)-b_im);
    d_im = -0.5 * log(s+b_re*b_re+b_im*b_im-sqrt(2*s+2-2*b_re*b_re+2*b_im*b_im) * b_im +
        csgn(-2*b_re*b_im, b_re*b_re-b_im*b_im-1) *
        sqrt(2*s-2+ 2*b_re*b_re - 2*b_im*b_im)*b_re);

    t.SetReal((c_re+d_re)/2.);
    t.SetImI((c_im+d_im)/2.);
    t.SetImJ((c_im-d_im)/2.);
    t.SetImK((d_re-c_re)/2.);

    return t;
}


hypercomplex acos(hypercomplex h)
{
    hypercomplex t;
    double a_re,a_im, b_re,b_im, c_re,c_im, d_re,d_im;

    a_re = h.GetReal()-h.GetImK();
    a_im = h.GetImI()+h.GetImJ();
    b_re = h.GetReal()+h.GetImK();
    b_im = h.GetImI()-h.GetImJ();

    double s = sqrt(1 - 2*a_re*a_re + 2*a_im*a_im + pow(a_re,4) +
        2*a_re*a_re*a_im*a_im + pow(a_im,4));

    c_re = atan2(0.5*csgn(2*a_re*a_im, -a_re*a_re+a_im*a_im+1) *
        sqrt(2*s+2-2*a_re*a_re+2*a_im*a_im) + a_im ,
        0.5*sqrt(2*s-2+2*a_re*a_re-2*a_im*a_im)+a_re);
    c_im = -0.5 * log(s+a_re*a_re+a_im*a_im+sqrt(2*s-2+2*a_re*a_re-2*a_im*a_im) * a_re -
        csgn(-2*a_re*a_im, a_re*a_re-a_im*a_im-1) *
        sqrt(2*s+2- 2*a_re*a_re + 2*a_im*a_im)*a_im);

    d_re = atan2(0.5*csgn(2*b_re*b_im, -b_re*b_re+b_im*b_im+1) *
        sqrt(2*s+2-2*b_re*b_re+2*b_im*b_im) + b_im ,
        0.5*sqrt(2*s-2+2*b_re*b_re-2*b_im*b_im)+b_re);
    d_im = -0.5 * log(s+b_re*b_re+b_im*b_im+sqrt(2*s-2+2*b_re*b_re-2*b_im*b_im) * b_re -
        csgn(-2*b_re*b_im, b_re*b_re-b_im*b_im-1) *
        sqrt(2*s+2- 2*b_re*b_re + 2*b_im*b_im)*b_im);

    t.SetReal((c_re+d_re)/2.);
    t.SetImI((c_im+d_im)/2.);
    t.SetImJ((c_im-d_im)/2.);
    t.SetImK((d_re-c_re)/2.);

    return t;
}


hypercomplex atan(hypercomplex h)
{
    hypercomplex t;
    double a_re,a_im, b_re,b_im, c_re,c_im, d_re,d_im;

    a_re = h.GetReal()-h.GetImK();
    a_im = h.GetImI()+h.GetImJ();
    b_re = h.GetReal()+h.GetImK();
    b_im = h.GetImI()-h.GetImJ();

    c_re = -0.5*atan2(-2*a_re/(1-2*a_im+a_re*a_re+a_im*a_im),
        (1-a_re*a_re-a_im*a_im)/(1-2*a_im+a_re*a_re+a_im*a_im));
    c_im = 0.25*(log(a_re*a_re+a_im*a_im+2*a_im+1) - log(a_re*a_re+a_im*a_im-2*a_im+1));

    d_re = -0.5*atan2(-2*b_re/(1-2*b_im+b_re*b_re+b_im*b_im),
        (1-b_re*b_re-b_im*b_im)/(1-2*b_im+b_re*b_re+b_im*b_im));
    d_im = 0.25*(log(b_re*b_re+b_im*b_im+2*b_im+1) - log(b_re*b_re+b_im*b_im-2*b_im+1));

    t.SetReal((c_re+d_re)/2.);
    t.SetImI((c_im+d_im)/2.);
    t.SetImJ((c_im-d_im)/2.);
    t.SetImK((d_re-c_re)/2.);

    return t;
}

//---------------------------------------------------------------------------

#endif

