package com.equitysoft.components;

import java.awt.*; import java.awt.event.*; import java.io.*; import java.net.*;
import java.awt.image.*; import java.util.*; import java.beans.*;

/**
* <p>IEButton is a 'hot' button of the type found in Internet browsers such as Netscape
* Navigator Version 4. When the cursor passes over the button it's border appears and the
* image 'lights-up'. The button can have a text label or a tooltip but not both. If no
* image is specified then the button is a text-only button. If '<code>setOpaque(true)</code>'
* is not invoked then the button is 'transparent' and a background image will show through.<p>
* A button can be made 'sticky' which means the button will stayed depressed when pushed, until
* it is pushed again. A sticky button can also be added to IEButton's 'group' which means
* only one button in the group can be depressed at one time. Whether a button is sticky or not
* it informs any registered <code>actionListener</code> if it is pushed. By using
* <code>setRepeat</code> a button will continuously generate <code>actionEvent</code>'s while
* the button is held down.
* <p>Any border thickness can be drawn and various label/image positions are supported.
* IEButton can be extended to change the way the border is drawn or how the image is grayed-out.
*
* @author Colin Mummery  e-mail: colin_mummery@my-deja.com
* @version Dec 17th, 2000
*
*/

public class IEButton extends Component implements Serializable, EnterExitListener {

	final static long serialVersionUID=5088819068576749880L;
/**
* The text position constant that designates that the button's text be used as a tooltip.
*/
	public final static int TOOLTIP=0;
/**
* The text position constant that places text as a label underneath the button's image or
* in the case of the tooltip position, below the button.
*/
	public final static int LABEL_SOUTH=1;
/**
* The text position constant that places text as a label to the right of the button's image.
*/
	public final static int LABEL_EAST=2;
/**
* The text position constant that places text as a label above the button's image or
* in the case of the tooltip position, above the button.
*/
	public final static int LABEL_NORTH=3;
/**
* The text position constant that places text as a label to the left of the button's image.
*/
	public final static int LABEL_WEST=4;

	protected final static int POSITIONS=5; //the number of label positions
	protected static Font MSG_FONT=new Font("Helvetica",Font.BOLD,12);

	private final static String EMPTY="";
	private final static int NORMAL=0;
	private final static int HLIGHT=1;
	private final static int DEPRESS=2;
	private final static int RELEASED=3;
	private final static int SIZE=36; //default dimension if no image
	private final static int DEFAULT_BORDER=8;
	private final static Font DEFAULT_FONT=new Font("Helvetica",Font.BOLD,12);
	protected static Frame frame;
	protected static IEButton currbutton;
	private transient IEBR runner;

	transient private Color back,fore,darker;
	transient private int action;
	transient private boolean loaded,exited,canrun,canrepeat;
	transient private int hgt,hless1,wid,wless1,imgxoff,imgyoff,fdesc,textxoff,textyoff;
	transient private Image img;
	transient private String imgfile;
	transient Image altimg,darkimg;
	transient private TBox txtbox; transient private RP rp;
	transient private IEButton handle; transient MA ma; transient static ResourceBundle rb;

	private Vector listeners,eelisteners; private boolean sliding;
	private String text; private int border_width,position,border;
	private boolean prefdimset,highlight,sticky,grayed_out;
	private boolean darkened,opaque,istraversable,inverted,repeat,isdown,ingroup;
	private long boxwait,interval; private int ttpos;
	private boolean outline;
	private Dimension prefdim;
	private Font currft;
	private Color background; //The background color of the button if opaque
	private String command; //set by setActionCommand

//--------------------------------------------------------------------------------------------
   protected static String getString(String key){
	if(rb==null){
		try{
			rb=ResourceBundle.getBundle("com.equitysoft.components.IEButtonResources");
		}catch(MissingResourceException mre){return "";}
	}
	return rb.getString(key);
   }
//--------------------------------------------------------------------------------------------
/**
* Constructs an instance of IEButton with no image , no text, with positioning of
* LABEL_SOUTH and a preferred size of 36 by 36.
* Intended for use by JavaBean development tools such as Sun's BDK but can
* be used directly to create a blank button.
*/
   public IEButton(){
	text=EMPTY; position=LABEL_SOUTH; classInit(); prefdim=new Dimension(SIZE,SIZE);
   }
//--------------------------------------------------------------------------------------------
   private void classInit(){
	handle=this;
	setBackground(Color.white); ttpos=LABEL_SOUTH;
	loaded=false; currft=DEFAULT_FONT; sticky=false; grayed_out=true; prefdimset=false;
	boxwait=800; border=DEFAULT_BORDER; opaque=false; outline=false;
	background=Color.lightGray; border_width=1;
	addListeners();
   }
//--------------------------------------------------------------------------------------------
   private void addListeners(){
	addMouseListener(ma=new MA()); addMouseMotionListener(new MMA());
	addFocusListener(new FA()); addKeyListener(new KA());
	addComponentListener(
		new ComponentAdapter(){
			public void componentResized(ComponentEvent ce){loaded=false;}
		}
	);
   }
//--------------------------------------------------------------------------------------------
/**
* Constructs an instance which is text only. This button is pretty much identical to
* the AWT.Button in appearance when highlighted.
*
* @param text A String which is the label of the button.
*/
   public IEButton(String text){this(null,text,LABEL_EAST);}
//--------------------------------------------------------------------------------------------
/**
* Constructs an instance of IEButton with the specified image but without any associated
* text. A text label or tooltip can be later added to the button by invoking <code>
* setText()</code>. The preferred size of the button will be the image dimension plus
* the default border inset (the default inset is 8 pixels).
*
* @param image The image to be displayed on the button.
*/
   public IEButton(Image image){this(image,null,TOOLTIP);}
//--------------------------------------------------------------------------------------------
/**
* Constructs an instance of IEButton with the given image and text and an implied text
* position of <code>LABEL_SOUTH</code>.
*
* @param image The Image to be displayed in the button above the text.
* @param text The String to be displayed below the image.
*
*/
   public IEButton(Image image,String text){this(image,text,LABEL_SOUTH);}
//--------------------------------------------------------------------------------------------
/**
* Constructs an instance of IEButton with the specified image and textual label or tooltip.
* The boolean <code>tooltip</code> specifies if the text is to be used as a tooltip or used
* as a label in the button.
*
* @param image The Image to be displayed in the button. If null no image is displayed and the
                  button is treated as a text-only button.
* @param text The String for the tooltip or label text of the button.
* 			If <code>null</code> then no text will be displayed.
* @param toolip A boolean value indicating if the button has a tooltip or text label.
* 			<code>true</code> if the button has a text tooltip, <code>false</code>
*			if the button has a text label.
*			The tooltip is a box of text that appears if the cursor remains over the
*			button for a preset period of time. The text label is text that appears
*			below the image. It is not possible to have both in the same button.
*/
   public IEButton(Image image,String text,int position){
	classInit(); img=image; this.position=position;
	if(text==null)this.text=EMPTY; else this.text=text; position=LABEL_SOUTH;
	createAlt(); calcDim();
   }
//---------------------------------------------------------------------------------------------
/**
* Invoked by another instance of IEButton with which the current instance is registered as a
* EnterExitEvent listener. This allows two related buttons to 'light-up' simultaneously if
* the cursor moves over either one (such behaviour is seen in a well known internet browser).
* Alternatively some other action such as the display of an explanatory text message might be
* shown if the cursor is over the button. 
*/
   public void buttonStateChanged(EnterExitEvent eee){
	if(eee.getState())setLitup(true); else setLitup(false);
   }
//---------------------------------------------------------------------------------------------
   private void createAlt(){if(img!=null){altimg=getAlternateImage(img);}}
//---------------------------------------------------------------------------------------------
/**
* This method fetches the 'grayed-out' image for the button. By overidding this method in an
* extended class the user can define their own way of filtering the image or provide an
* alternate image for when the mouse is not over the button.
*
* @param original The original image which is displayed when the cursor is over the button
* @return Image The image which is to be displayed when the cursor is not over the button.
* @see #setGrayOutInverted
*/
   public Image getAlternateImage(Image original){
	RGBImageFilter filter;

	if(inverted)filter=new IF(); else filter=new GOF();

	Image alt=createImage(new FilteredImageSource(original.getSource(),filter));
	prepareImage(alt,this);
	return alt;
   }
//---------------------------------------------------------------------------------------------
/**
* Changes the way the image is grayed-out.
* By default the image is 'grayed-out' by lightening colors. This method activates an alternate
* method which is to produce a negative of the original image. This look is best used with
* black and white images although color images can also respond well. Note that when the style
* of grayout is set for one button then all subsequently created button instances will
* automatically use this grayout style.
*
* @param inverted A boolean value to indicate how the image should be grayed out. 
* A <code>true</code> value indicates that an inversion filter should be used
* to produce the grayed-out image. 
* @see #isGrayOutInverted
*/
   public void setGrayOutInverted(boolean inverted){this.inverted=inverted; createAlt();}
/**
* Indicates how the image is currently grayed. The default is not to use inversion.
*
* @return boolean A value of <code>true</code> indicates an inversion filter is being used.
* @see #setGrayOutInverted
*/
   public boolean isGrayOutInverted(){return inverted;}
//---------------------------------------------------------------------------------------------
/**
* Decides if the image is to darkened or not when the button is pushed.
*
* @param darkened A boolean value to indicate if the image is darkened on pushing the button. 
* A <code>true</code> value indicates that the image should be darkened.
* @see #isDownDarkened
*/
   public void setDownDarkened(boolean darkened){
	this.darkened=darkened;
	if(darkened){
		darkimg=createImage(new FilteredImageSource(img.getSource(),new DF()));
		prepareImage(darkimg,this);
	}
   }
/**
* Indicates if the image is darkened when the button is pushed. The default is not to darken.
*
* @return boolean A value of <code>true</code> indicates that the image is darkened.
* @see #setDownDarkened
*/
   public boolean isDownDarkened(){return darkened;}

//---------------------------------------------------------------------------------------------
   private void readObject(ObjectInputStream in) throws IOException,ClassNotFoundException {
	handle=this; addListeners();
	in.defaultReadObject();
	int len=in.readInt(); if(len==0)return;
	byte[] buf=new byte[len]; in.read(buf);
	img=Toolkit.getDefaultToolkit().createImage(buf); prepareImage(img,this);
	createAlt();
   }
//---------------------------------------------------------------------------------------------
   private void writeObject(ObjectOutputStream out) throws IOException {
	out.defaultWriteObject();
	if(img==null)out.writeInt(0);
	else {
		out.writeInt((int)imgfile.length());
		FileInputStream fis=new FileInputStream(imgfile);
		byte[] buf=new byte[512]; int len;
		while((len=fis.read(buf))>-1)out.write(buf,0,len);
		fis.close(); out.flush();
	}
   }
//---------------------------------------------------------------------------------------------
/**
* Causes the specified EnterExitListener's <code>buttonStateChanged</code> method to 
* be invoked each time the cursor enters or exits the button.  The EnterExitListener
* object is added to a list of EnterExitListeners managed by 
* this button, it can be removed with <code>removeEnterExitListener.</code>
* 
* @param target The <code>EnterExitListener</code> to be added.
* @see #removeEnterExitListener
*/
   public void addEnterExitListener(Object target){
	if(eelisteners==null)eelisteners=new Vector(1); eelisteners.addElement(target);
   }
/** 
* Remove this EnterExitListener from the buttons list of EnterExitListeners.
* If not registered do nothing.
* 
* @param target The EnterExitListener to be removed.
* @see #addEnterExitListener
*/      
   public void removeEnterExitListener(Object target){eelisteners.removeElement(target);}
//---------------------------------------------------------------------------------------------
/** 
* This method is not normally invoked directly. The <code>buttonStateChanged</code> methods
* of all registered EnterExitListeners are invoked.
* 
* @param state <code>true</code> indicates the cursor has entered the button. <code>false</code>
* indicates that the cursor has exited the button.
* @see #addActionListener
*/      
   public void fireEnterExit(boolean state) {
	if(eelisteners==null)return;
	Vector targets; synchronized(this){targets=(Vector) eelisteners.clone();}
	EnterExitEvent eee=new EnterExitEvent(this,state);
	for (int i=0;i<targets.size();i++) {
		EnterExitListener target=(EnterExitListener)targets.elementAt(i);
		target.buttonStateChanged(eee);
	}
   }
//---------------------------------------------------------------------------------------------
/**
* This method invoked by JavaBean manipulation tools to set the image. it should
* not be invoked directly by the programmer. Instead set the image with the constructor.
*/
   public void setImageFile(String imgfile){
	this.imgfile=imgfile; if(imgfile==null || imgfile.equals(EMPTY)){img=null; return;}
	MediaTracker tracker=new MediaTracker(new Canvas());
	Image img=Toolkit.getDefaultToolkit().getImage(imgfile);
	tracker.addImage(img,0);
	try{tracker.waitForAll();}catch (InterruptedException ie){System.err.println("Error");}
	this.img = img;

	createAlt(); calcDim();
   }
/**
* This method is invoked by JavaBean tools. It should not be invoked directly by the programmer.
*
* @return An ImageInfo instance that contains JavaBean related data about the image.
*/
   public String getImageFile(){if(imgfile==null)return EMPTY; return imgfile;}
//---------------------------------------------------------------------------------------------
/**
* Changes the image at runtime. The new image should be the same size or smaller than the original image
* otherwise the placement of the image may be incorrect
* 
* @param img The <code>Image</code> which is to be the new button image
*/
   public void setImage(Image img){
	this.img=img; loaded=false; repaint();
   }
//---------------------------------------------------------------------------------------------
/**
* Sets the position of the tooltip. The values can be <code>LABEL_NORTH</code> or <code>LABEL_SOUTH</code>.
* The default is <code>LABEL_SOUTH</code>.
*
* @param position An <code>int</code> value to indicate the tooltip position
* @see #getTooltipPosition
*/
   public void setTooltipPosition(int position){
	if(position!=LABEL_NORTH && position!=LABEL_SOUTH)return; ttpos=position;
   }
/**
* Returns the current position of the tooltip if tooltips are being used for the button
* @see #setTooltipPosition
*/
   public int getTooltipPosition(){return ttpos;}
//---------------------------------------------------------------------------------------------
/**
* Sets the way in which the tooltip appears if it is used. If set to <code>true</code> the tooltip will
* slide into view rather than just appearing. This results in a slicker GUI depending on your
* point of view. The default is for sliding to be turned off (ie. <code>false</code>).
*
* @param enabled A boolean value to indicate if the tooltip is to slide
* @see #isSlidingTooltip
*/
   public void setSlidingTooltip(boolean sliding){this.sliding=sliding;}
/**
* Indicates if the tooltip is set to slide.
* @return <code>true</code> if the tooltip slides into view.
* @see #setSlidingTooltip
*/
   public boolean isSlidingTooltip(){return sliding;}
//---------------------------------------------------------------------------------------------
/**
* Invoked to enable or disable the button. If disabled the button's text label will be 
* grayed out as well as the image if present. Moving the cursor over the button has no effect.
*
* @param enabled A boolean value to indicate if the button is to be enabled
* @see #isEnabled
*/
   public void setEnabled(boolean enabled){super.setEnabled(enabled); action=NORMAL; repaint();}
/**
* Indicates if the button is enabled or not.
* @return <code>true</code> if the button is enabled.
* @see #setEnabled
*/
   public boolean isEnabled(){return super.isEnabled();}
//---------------------------------------------------------------------------------------------
   private void releaseButton(){isdown=false; action=NORMAL; repaint();}
//---------------------------------------------------------------------------------------------
   private void calcDim(){//calculates the prefdim in the absence of a setPreferredSize() call

	loaded=false;
	if(!prefdimset){ //have we explicitly set the size
		FontMetrics fm=getFontMetrics(currft); int width,height; int border2=border+border;
		if(img!=null){
			width=img.getWidth(null)+border2;
			height=img.getHeight(null)+border2;
			if(position==LABEL_SOUTH || position==LABEL_NORTH){
				height=height+fm.getHeight()+border;
				int altwidth=fm.stringWidth(text)+border2;
				if(altwidth>width)width=altwidth;
			}
			else if(position==LABEL_EAST || position==LABEL_WEST){
				width=fm.stringWidth(text)+border2+border+width;
				int althgt=fm.getHeight()+border2;
				if(althgt>height)height=althgt;
			}
		}
		else { //assume a text only button
			if(position!=TOOLTIP){
				width=fm.stringWidth(text)+border2+border2;
				height=fm.getHeight()+border2;
			}
			else {width=SIZE; height=SIZE;} //have no other way to work out the size
		}
		prefdim=new Dimension(width,height);
	}
	setSize(prefdim);
	Container p=getParent(); if(p!=null){p.invalidate(); p.validate();}
   }
//--------------------------------------------------------------------------------------------
   private void setFrame(){
	Container parent=getParent();
	while(parent!=null && !(parent instanceof Frame))parent=parent.getParent();
	if(parent!=null)frame=(Frame)parent;
   }
//--------------------------------------------------------------------------------------------
/**
* Overidden so that calls to <code>Component.setBounds</code> cause IEButton to recalculate positions
*/
   public void invalidate() {loaded=false; super.invalidate();}
//--------------------------------------------------------------------------------------------
/**
* Invoked by <code>Component.repaint()</code> - not usually invoked directly
*/
   public void update(Graphics G){paint(G);}
//---------------------------------------------------------------------------------------------
/**
* Invoked by <code>update</code> , not usually invoked directly
*/
   public void paint(Graphics G){
	if(!loaded){
		if(getParent()==null)return; if(position==TOOLTIP && frame==null)setFrame();
		fore=getForeground();
		back=(opaque)? background : getParent().getBackground(); darker=back.darker();
		Dimension d=getSize(); wid=d.width; hgt=d.height; wless1=wid-1; hless1=hgt-1;

		if(img!=null){
			imgxoff=(wid-img.getWidth(null))/2; if(imgxoff<0)imgxoff=0;
		}
		if(position!=TOOLTIP && !text.equals(EMPTY)){
			FontMetrics fm=getFontMetrics(currft); int swid=fm.stringWidth(text);
			if(position==LABEL_SOUTH || position==LABEL_NORTH){
				int imh=(img==null)? 0 : img.getHeight(null)+border;
				int off=(hgt-imh-fm.getHeight())/2;
				textxoff=(wid-swid)/2;
				if(position==LABEL_SOUTH){
					imgyoff=off; textyoff=hgt-off-fm.getDescent();}
				else {imgyoff=off+fm.getHeight()+border;
					textyoff=off+fm.getHeight()-fm.getDescent();
				}
			}
			else { //it's EAST or WEST
				if(img!=null){
					int ihgt=img.getHeight(null);
					if(ihgt>=hgt)imgyoff=0; else imgyoff=(hgt-ihgt)/2;
					imgxoff=(wid-img.getWidth(null)-border-swid)/2;
					if(position==LABEL_EAST)textxoff=
									imgxoff+img.getWidth(null)+border;
					else {textxoff=imgxoff; imgxoff=textxoff+swid+border;}
				}
				else textxoff=(wid-swid)/2;
				textyoff=(hgt-fm.getHeight())/2+fm.getHeight()-fm.getDescent();
			}
		}
		else if(img!=null){
			imgyoff=(hgt-img.getHeight(null))/2; if(imgyoff<0)imgyoff=0;
		}
		loaded=true;
	}
	if(opaque){G.setColor(back); G.fillRect(0,0,wid,hgt);}

	if(highlight && action==NORMAL && isEnabled()){action=HLIGHT; exited=false;}
	if(sticky && isdown)action=DEPRESS;
	switch(action){

		case DEPRESS: 
			if(img!=null)
				G.drawImage((darkened)? darkimg : img,imgxoff+1,imgyoff+1,null);
			if(position!=TOOLTIP){
				G.setColor(fore); G.setFont(currft);
				G.drawString(text,textxoff+1,textyoff+1);
			}
			paintBorder(G,wid,hgt,false,back,border_width,outline,fore); break;

		case RELEASED: //Needed for release by keystroke
			if(img!=null)G.drawImage(img,imgxoff,imgyoff,null); drawLabel(G);
			paintBorder(G,wid,hgt,true,back,border_width,outline,fore); break;

		case HLIGHT:
			if(!exited){//maybe the mouse has exited since we called repaint()
				if(img!=null)G.drawImage(img,imgxoff,imgyoff,null);
				drawLabel(G);
				paintBorder(G,wid,hgt,true,back,border_width,outline,fore);
				break;
			} //if it's exited then paint the default

		default:
			if(img!=null){
				if(grayed_out)G.drawImage(altimg,imgxoff,imgyoff,null);
				else if(!isEnabled())G.drawImage(altimg,imgxoff,imgyoff,null);
				else G.drawImage(img,imgxoff,imgyoff,null);
			}
			drawLabel(G);
	}
   }
//---------------------------------------------------------------------------------------------
/**
* A method that paints the button border.
* This method should be overidden in an extended class if the programmer wants to redefine
* the look of the border. The border inset should be set appropriately if the image
* and/or text label shouldn't overlap the border.
*
* @param G The GraphicsContext used to paint the border.
* @param width The width of the button in which the border can be painted.
* @param height The height of the button in which the border can be painted.
* @param up A boolean value that is <code>true</code> if the button is up.
* @param background The Color that should be lightened or darkened to achieve an up/down effect.
* @param width_of_border An int value indicating the pixel width of the border.
* @param draw_outline A boolean value indicating if the outline should be drawn.
* @param outline_color A Color instance to be used for drawing the outline.
*/
   public void paintBorder(Graphics G,int width,int height,boolean up,Color background,
	int width_of_border,boolean draw_outline,Color outline_color){

	if(border_width==0)return; int w=width-1, h=height-1, x=0, y=0;
	G.setColor(background);
	for(int i=0;i<width_of_border;i++){G.draw3DRect(x++,y++,w,h,up); w-=2; h-=2;}

	if(draw_outline){G.setColor(outline_color); G.drawRect(0,0,wless1,hless1);}
   }
//---------------------------------------------------------------------------------------------
   private void drawLabel(Graphics G){
	if(position!=TOOLTIP){
		G.setFont(currft);
		if(isEnabled()){G.setColor(fore); G.drawString(text,textxoff,textyoff);}
		else {
			G.setColor(Color.white); G.drawString(text,textxoff+1,textyoff);
			G.setColor(darker); G.drawString(text,textxoff,textyoff);
		}
	}
   }
//---------------------------------------------------------------------------------------------
/**
* Sets the position of the text relative to the image.
* The possible value are <code> TOOLTIP, LABEL_SOUTH, LABEL_EAST, LABEL_NORTH, LABEL_WEST
* </code>.
*
* @param position An <code>int</code> value indicating how the text associated with this button
* is to be used.
* @see #getTextPosition
*/
   public void setTextPosition(int position){
	if(position>=0 && position<POSITIONS){this.position=position; calcDim();}
   }
/**
* Returns a value indicating how the text associated with the button is currently being
* displayed.
*
* @return An int value which can be one of <code>TOOLTIP, LABEL_SOUTH, LABEL_EAST, LABEL_NORTH,
* LABEL_WEST</code>.
* @see #setTextPosition
* @see IEButton#IEButton(Image,String,int)
*/
   public int getTextPosition(){return position;}
//---------------------------------------------------------------------------------------------
/**
* Turns on or off the drawing of an outline around the button. The outline is one pixel wide and
* is the same color as the text color. The outline gives emphasis to the button. If the border
* width has a value of one when the outline is switched on, IEButton increases the border width
* to a value of two to give a better looking button.
*
* @param outline A boolean value of <code>true</code> indicates that an outline should be drawn
* around the button.
* @see #isOutlineDrawn
*/
   public void setOutlineDrawn(boolean outline){
	this.outline=outline; if(outline && border_width==1)border_width=2;
   }
/**
* Returns a boolean value indicating if an outline is to be drawn around the button.
*
* @return <code>true</code> if an outline is to be drawn around the button.
* @see #setOutlineDrawn
*/
   public boolean isOutlineDrawn(){return outline;}
//---------------------------------------------------------------------------------------------
/**
* Turns on or off button toggling. Toggling means that when the IEButton is pushed that button
* stays in the pushed position and any other IEButton already in the pushed position is
* released. The button can be said to be 'sticky'.
*
* @param sticky A boolean value of true indicates that toggling should be enabled
* @see #isSticky
*/
   public void setSticky(boolean sticky){this.sticky=sticky;}
/**
* Returns a boolean value indicating if toggling is enabled.
*
* @return <code>true</code> if toggling is enabled.
* @see #setSticky
*/
   public boolean isSticky(){return sticky;}
//---------------------------------------------------------------------------------------------
/**
* Designates a sticky button to be part of a single mutually exclusive group of IEButton
* instances that are all sticky. If set to <code>true</code> then only one button of the
* group of sticky buttons can be depressed at one time. If a button in the group is pushed
* only the pushed button's <code>ActionEvent</code> is fired. If the button is not sticky then
* it is automatically made sticky when added to the group.
* There can only be one unnamed group in an
* application. Programmers needing more than one group should use Sun's JFC classes.
*
* @param state A boolean value of<code>true</code> designates the button of part of the group.
* @see #isInGroup
*/
   public void setInGroup(boolean state){if(state)setSticky(true); ingroup=state;}
/**
* Returns a value to indicate if the button is part of a group or not
*
* @return boolean A value of <code>true</code> indicates the button is in the group.
*/
   public boolean isInGroup(){return ingroup;}
//---------------------------------------------------------------------------------------------
/**
* Enables or disables the graying out of the image on a button. Some programmers may prefer to
* have an image that is never grayed out even if the cursor is not over the button. The default
* state is that the image is grayed out when the cursor is not over the button.
*
* @param grayedout A boolean value of false indicates the image should not be grayed out when
* the cursor is not over the button. <code>setEnabled(false)</code> will still gray out the
* image though.
*
* @see #isImageGrayOutEnabled
*/
   public void setImageGrayOutEnabled(boolean grayed_out){this.grayed_out=grayed_out;}
/**
* Returns a boolean value indicating if graying out is enabled.
*
* @return <code>true</code> if the graying out of images occurs even if the cursor is not
* over the button.
*
* @see #setImageGrayOutEnabled
*/
   public boolean isImageGrayOutEnabled(){return grayed_out;}
//---------------------------------------------------------------------------------------------
/**
* Turns on or off permanent highlighting. If set to true the button will be always 
* highlighted , if set false (which is the default) the button will only be highlighted if
* the cursor is over it.
*
* @param on A boolean value to indicate if highlighting should be permanently on.
* @see #isLitup
*/
   public void setLitup(boolean on){
	highlight=on; action=NORMAL; repaint();
   }
/**
* Returns a boolean to indicate if permanent highlighting is on (a <code>true</code> value).
*
* @return <code>true</code> indicates that the button will always be highlighted.
* @see #setLitup
*/
   public boolean isLitup(){return highlight;}
//---------------------------------------------------------------------------------------------
/**
* Sets the font of the text.
*
* @param font The font in which the text is to appear.
* @see #getFont
*/
   public void setFont(Font font){
	super.setFont(font); currft=font; if(position!=TOOLTIP)calcDim();
   }
/**
* Returns the current font used to display the text associated with the button.
*
* @return The font being used to display the button's text.
* @see #setFont
*/
   public Font getFont(Font ft){return currft;}
//---------------------------------------------------------------------------------------------
/**
* Sets the button's border inset in pixels. This is the blankspace gap between the edge of
* the button and the  image. If the button has a text label then it is also the gap between
* the image and the text as well as the text and the buttons bottom edge. The default is 4.
*
* @param borderinset The new value of the border inset.
* @see#getBorderInset
*/
   public void setBorderInset(int borderinset){
	if(borderinset<0)border=0; else border=borderinset;
	if(prefdimset)loaded=false; else calcDim();
   }
/**
* Returns the current border inset.
*
* @return The number of pixels being used for the current inset.
* @see #setBorderInset
*/
   public int getBorderInset(){return border;}
//---------------------------------------------------------------------------------------------
/**
* Sets the width of the raised border around the button's edge
*
* @param border_width An int value indicating the border width to be used.
* @see #getBorderWidth
*/
   public void setBorderWidth(int border_width){
	if(border_width>=0)this.border_width=border_width;
  }
/**
* Returns the current border width being used to draw the button.
* @return An int value representing the current border width.
* @see #setBorderWidth
*/
   public int getBorderWidth(){return border_width;}
//---------------------------------------------------------------------------------------------
/**
* Sets the preferred size of the button. When the button is added to a Java container the
* container's layout manager will use this dimension to allocate space in the layout for the
* button. If the preferred size is never set or set to null then IEButton will calculate
* a preferred size based on the image size , the border inset and the presence (or absence)
* of a text label next to image. if the image is null
* then the button has a default preferred size of 36 by 36.
*
* @param dim The font in which the text is to appear.
* @see #getPreferredSize
*/
   public void setPreferredSize(Dimension dim){
	if(dim==prefdim)return; //so bean box doesn't set it every time, but does it always work?
	if(dim==null)prefdimset=false; else {prefdim=dim; prefdimset=true;} calcDim();
   }
/**
* Returns The current preferred size dimension.
*
* @return A Dimension instance representing the current preferred size.
* @see #setPreferredSize
*/
   public Dimension getPreferredSize(){
	if(prefdim==null)return new Dimension(SIZE,SIZE); else return prefdim;
   }
//---------------------------------------------------------------------------------------------
/**
* Returns the same value as <code>getPreferredSize()</code>
*
* @return The minimum dimension of the button. Used by layout managers to allocate space.
*/
   public Dimension getMinimumSize(){return getPreferredSize();}
//--------------------------------------------------------------------------------------------
/**
* Sets the text used for the button's text label or tooltip depending on the boolean value
* in the constructor. If the parameter is <code>null</code> then the text is set to the
* be an empty String.
*
* @param String The text of the label or tooltip.
* @see #getText
*/
   public void setText(String text){
	if(text==null)text=EMPTY; else this.text=text; loaded=false; repaint();
   }
/**
* Returns the buttons text string. Never returns <code>null</code>.
* 
* @return The text used for the button label or tooltip.
* @see #setText
*/
   public String getText(){return text;}
//---------------------------------------------------------------------------------------------
/**
* Sets the color with which the text is drawn. If not set explicitly the default is black.
*
* @param Color The color with which the text is drawn.
* @see #getTextColor
*/
   public void setTextColor(Color col){setForeground(col); fore=col;}
/**
* Returns the color with which the text is drawn.
* 
* @return The color with which the text is drawn.
* @see #setTextColor
*/
   public Color getTextColor(){return getForeground();}
//---------------------------------------------------------------------------------------------
/**
* Sets the button background to be opaque in which case the background is filled by a single
* color.
* If the color has not been set with a call to <code>setButtonBackgroundColor</code> then
* the color used will be background color of the button's container. By default IEButton
* instances are transparent and not opaque.
*
* @param opaque A boolean value where <code>true</code> means the background is to be filled.
* @see #isOpaque
* @see #setButtonBackground
*/
   public void setOpaque(boolean opaque){this.opaque=opaque;}
/**
* Indicates if the button is to be drawn with an opaque background. A value of <code>true</code>
* indicates that the background is opaque.
*
* @return boolean A value of <code>true</code> indicates that the button is opaque.
* @see #setOpaque
* @see #setButtonBackground
*/
   public boolean isOpaque(){return opaque;}
//---------------------------------------------------------------------------------------------
/**
* Sets the background color of the button. For this background color to be used then the button
* must be set to be opaque by invoking <code>setOpaque(true)</code>
*
* @param background The color with which the backgroound is drawn.
* @see #getButtonBackground
* @see #setOpaque
*/
   public void setButtonBackground(Color background){
	loaded=false; this.background=background;
   }
/**
* Returns the background color of the button. If never set this value is the background color
* of the button's parent container.
* .
* 
* @return The color which is used to fill the background of an opaque IEButton.
* @see #setButtonBackground
*/
   public Color getButtonBackground(){
	if(opaque)return background; else if(back!=null)return back; else return Color.lightGray;
   }
//---------------------------------------------------------------------------------------------
/**
* Sets the color used for the background of the tooltip.
* If not set explicitly the default is white. If the button doesn't have the tooltip enabled
* this method will do nothing.
*
* @param Color The color which will be used for the tooltip background.
* @see #getTooltipBackground
*/
   public void setTooltipBackground(Color col){
	if(col==null)setBackground(Color.white); else setBackground(col);
   }
/**
* Returns the color used for the tooltip background.
* 
* @return The color used for the tooltip background.
* @see #setTooltipBackground
*/
   public Color getTooltipBackground(){return getBackground();}
//---------------------------------------------------------------------------------------------
/**
* Sets a sticky button to be depressed or to be released from a depressed state. If another
* sticky button is currently depressed then that button is released. Setting a button to
* be depressed or released does <b>not</b> fire an <code>ActionEvent</code>.
* Only used if the method <code>isSticky()</code> returns a value of <code>true</code>.
*
* @param state A boolean value of <code>true</code> sets the button to be depressed.
* @see #isDepressed
*/
   public void setDepressed(boolean state){
	if(sticky){
		if(state && !isdown){ //depress it
			isdown=true; action=DEPRESS; repaint();
			if(ingroup && currbutton!=this){
				if(currbutton!=null)currbutton.releaseButton(); currbutton=this; 
			}
			
		}
		else if(isdown){releaseButton(); if(ingroup)currbutton=null;} //release it
	}
   }
/**
* Returns a boolean indicating if the button is depressed. Only used for 'sticky' buttons
* 
* @return A boolean value of <code>true</code> indicates the button is currently depressed.
* @see #setDepressed
*/
   public boolean isDepressed(){return isdown;}
//---------------------------------------------------------------------------------------------
/**
* This method returns the currently depressed button instance if any. Any button returned must
* necessarily be both depressed and designated to be sticky by first invoking the method
* <code>setSticky(true)</code>.
*
* @return The IEButton which is currently depressed. Returns <code>null</code> is no
* button is in the depressed state.
*/
   public static IEButton getDepressedButton(){return currbutton;}
//---------------------------------------------------------------------------------------------
/**
* Sets the delay (in MilliSeconds) before a tooltip appears after the cursor is moved
* over the button. The default is 800. The tooltip appears directly below the button and
* dissapears when the button is pushed or when the cursor is no longer over the button.
*
* @param long The delay is MilliSeconds before the tooltip appears.
* @see #getTooltipTimeDelay
*/
   public void setTooltipTimeDelay(long boxwait){this.boxwait=boxwait;}
/**
* Returns The current delay before a tooltip is shown as a <code>long</code> value.
* 
* @return A long value which is the current tooltip delay.
* @see #setTooltipTimeDelay
*/
   public long getTooltipTimeDelay(){return boxwait;}
//---------------------------------------------------------------------------------------------
  class MMA extends MouseMotionAdapter {
   public void mouseDragged(MouseEvent me) {
	if (!isEnabled())return; int x = me.getX(); int y = me.getY();
	if(!contains(me.getX(),me.getY())){
		canrepeat=false; if(!exited)ma.mouseExited(me);
	}
	else if(exited){fireEnterExit(true); exited=false; action=DEPRESS; repaint();}
   }
  }
//---------------------------------------------------------------------------------------------
/**
* Sets the <code>actionCommand</code> of the <code>ActionEvent</code> delivered when the
* button is pushed. If never set then the value is the text associated with the button. This
* mimics the behaviour of the <code>java.awt.Button</code> component. The mouse button must be
* released over the button for an event to be sent to registered listeners.
*
* @param command A String which is to be delivered with the <code>ActionEvent</code>.
* @see #getActionCommand
*/
   public void setActionCommand(String command){this.command=command;}
/**
* Returns the value for the <code>actionCommand</code> String. If never explicitly set the
* this method returns the value of the text associated with the button.
*
* @return The String representing the <code>actionCommand</code> fired by this button.
* @see #setActionCommand
*/
   public String getActionCommand(){if(command==null)return text; else return command;}
//---------------------------------------------------------------------------------------------
/**
* Causes the specified ActionListener's <code>actionPerformed</code> method to 
* be invoked each time the button is clicked.  The ActionListener
* object is added to a list of ActionListeners managed by 
* this button, it can be removed with <code>removeActionListener.</code>
* Note: ActionListeners will not be notified in any particular order.
* 
* @param target The <code>ActionListener</code> to be added.
* @see #removeActionListener
*/      
   public synchronized void addActionListener(ActionListener target){
	if(listeners==null)listeners=new Vector(1); listeners.addElement(target);
   }
/** 
* Remove this ActionListener from the buttons list of ActionListeners.
* If not registered do nothing.
* 
* @param target The ActionListener to be removed.
* @see #addActionListener
*/      
   public synchronized void removeActionListener(ActionListener target){
	listeners.removeElement(target);
   }
//---------------------------------------------------------------------------------------------
/**
* Indicates to the AWT if the focus can be transferred to the button and also taken away. The
* default return value is <code>false</code>. The method <code>setFocusTraversable</code>
* should be invoked to set the value to <code>true</code>.
*
* @see #setFocusTraversable
*/
   public boolean isFocusTraversable(){return istraversable;}
/**
* Sets the value which is returned by <code>isFocusTraversable()</code>. The default is
* <code>false</code>. If set to <code>true</code> then the focus can be transferred by the
* appropriate key sequence (the 'tab' key on Win95). Consecutive transfers moves the focus from
* component to component and when IEButton gets the focus the button becomes highlighted.
* When highlighted in this way a subsequent pressing of the 'enter' key will cause the buttons
* <code>ActionEvent</code> to be fired.
*
* @param state A boolean value to indicate if the focus is to be traversable or not across this
* button.
* @see #isFocusTraversable
*/
   public void setFocusTraversable(boolean state){istraversable=state;}
//---------------------------------------------------------------------------------------------
/**
* Causes the <code>ActionEvent</code> to be fired repeatedly if the button is held down.
* If this method is not invoked then the default state is that repeat firing won't be
* enabled.
*
* @param repeat A boolean value of <code>true</code> indicates that repeat firing occurs
* if the button is held down.
* @see #willRepeat
*/
   public void setRepeat(boolean repeat){this.repeat=repeat; interval=60;}
/**
* Indicates if the button will repeatedly fire the <code>ActionEvent</code> if the button
* is held down.
*
* @return A boolean value of <code>true</code> indicates the button will repeatedly fire the
* <code>ActionEvent</code>.
* @see #setRepeat
*/
   public boolean willRepeat(){return repeat;}
//---------------------------------------------------------------------------------------------
/**
* Sets the repeat interval of <code>ActionEvent</code> firings when the button is held down
* and <code>willRepeat()</code> returns a value of true.
*
* @param interval A long value which is the repeat interval in milliseconds. The default value
* is 60 milliseconds.
* @see #getRepeatInterval
*/
   public void setRepeatInterval(long interval){this.interval=interval;
   }
/**
* Gets the current value of the repeat interval.
*
* @return The repeat interval in milliseconds.
* @see #setRepeatInterval
*/
   public long getRepeatInterval(){return interval;}
//---------------------------------------------------------------------------------------------
/** 
* Invoking this method is equivalent to clicking the mouse over the button. This is not
* normally invoked directly. The <code>actionPeformed</code> methods of all registered
* ActionListeners are invoked.
* 
* @see #addActionListener
*/      
   public void fireAction() {
	if(listeners==null)return;
	Vector targets; synchronized(this){targets=(Vector) listeners.clone();}
	ActionEvent ae=new ActionEvent(this,0,(command==null)? text : command);
	for (int i=0;i<targets.size();i++) {
		ActionListener target=(ActionListener)targets.elementAt(i);
		target.actionPerformed(ae);
	}
   }
//---------------------------------------------------------------------------------------------
  private class FA extends FocusAdapter{
   public void focusGained(FocusEvent fe){
	if(istraversable){exited=false; action=HLIGHT; handle.repaint();}
   }
   public void focusLost(FocusEvent fe){
	if(istraversable){exited=true; action=NORMAL; handle.repaint();}
   }
  }
//---------------------------------------------------------------------------------------------
  private class KA extends KeyAdapter{
	boolean down;
   public void keyTyped(KeyEvent ke){}
   public void keyPressed(KeyEvent ke){
	if(istraversable && !down){
		if(ke.getKeyCode()==KeyEvent.VK_ENTER || ke.getKeyCode()==KeyEvent.VK_SPACE){
			down=true; action=DEPRESS; handle.repaint();
		}
	}
   }
   public void keyReleased(KeyEvent ke){
	if(istraversable){
		if(ke.getKeyCode()==KeyEvent.VK_ENTER || ke.getKeyCode()==KeyEvent.VK_SPACE){
			down=false; fireAction(); action=RELEASED; handle.repaint();
		}
	}
   }
  }
//---------------------------------------------------------------------------------------------
  class RP extends Thread{
	public void run(){
			if(interval<5)interval=5; 
			try{sleep(300);}catch(InterruptedException ie){}
		OUTER:while(canrepeat){
			try{sleep(interval);}catch(InterruptedException ie){break OUTER;}
			if(canrepeat)fireAction();
		}
		rp=null;
	}
  }//end of BRunner class
//---------------------------------------------------------------------------------------------
  class MA extends MouseAdapter {

   public void mouseEntered(MouseEvent me){
	if(!isEnabled())return; fireEnterExit(true); if(currbutton==handle)return;
	exited=false; if(!highlight){action=HLIGHT; repaint();}
	if(position==TOOLTIP && (!text.equals(EMPTY)) && runner==null){
								runner=new IEBR(); canrun=true; runner.start();
	}
   }
   public void mouseExited(MouseEvent me){
	if(!isEnabled())return; if(currbutton==handle)return;
	exited=true;
	canrun=false;
	if(txtbox!=null)txtbox.setVisible(false);
	action=NORMAL; repaint(); fireEnterExit(false);
   }
   public void mousePressed(MouseEvent me){
	if(!isEnabled())return;
	canrun=false;
	if(txtbox!=null)txtbox.setVisible(false);
	if(repeat && currbutton!=handle){
		if(rp==null){canrepeat=true; rp=new RP(); rp.start();}
	}
	if(sticky){
		if(!isdown){
			isdown=true; fireAction();
			if(ingroup){
				if(currbutton!=handle){
					if(currbutton!=null)currbutton.releaseButton(); currbutton=handle;
				}
			}
		}
		else { //releasing sticky
			isdown=false; canrepeat=false; if(ingroup)currbutton=null; return;
		}
	}
	action=DEPRESS; repaint();
	if(repeat){
		if(rp==null){canrepeat=true; rp=new RP(); rp.start();}
	}
   }
   public void mouseReleased(MouseEvent me){
	if(!isEnabled())return;
	if(!sticky){
		canrepeat=false;
		if(contains(me.getX(),me.getY())){
			action=HLIGHT; repaint(); fireAction();
		}
	}
	else if(ingroup){
		if(currbutton==null){action=HLIGHT; repaint(); fireAction();}
	}
	else if(!isdown){action=HLIGHT; repaint(); fireAction();}
   }
  }
//---------------------------------------------------------------------------------------------
  class IEBR extends Thread{
   public void run(){
	if(!canrun){runner=null; return;}
	try{Thread.sleep(boxwait);}catch(InterruptedException ie){runner=null; return;}
	if(!canrun || frame==null){runner=null; return;}

	int bottom=0; Point p; Dimension d; int h;
	txtbox=new TBox(); d=txtbox.getSize(); h=d.height;
	p=handle.getLocationOnScreen();
	if(!sliding){
		if(ttpos==LABEL_SOUTH)txtbox.setLocation(p.x,(p.y+handle.getSize().height+1));
		else txtbox.setLocation(p.x,p.y-d.height-1);
		txtbox.setVisible(true); runner=null; return;
	}
	bottom=p.y+handle.getSize().height+1;
	if(ttpos==LABEL_SOUTH)txtbox.setLocation(p.x,bottom);
	else txtbox.setLocation(p.x,p.y-1);
	txtbox.setSize(1,1); txtbox.setVisible(true); bottom+=h;

	FOR:for(int i=2;i<=h;i++){
		try{Thread.sleep(5);}catch(InterruptedException ie){break FOR;}

		if(!canrun){txtbox=null; break FOR;}
		if(ttpos==LABEL_SOUTH)txtbox.setBounds(p.x,bottom-i,d.width,i);
		else txtbox.setBounds(p.x,p.y-i,d.width,i);

	}
	runner=null;
   }
  }
//---------------------------------------------------------------------------------------------
  private class TBox extends Window {//orig
   TBox(){
	super(frame); setBackground(handle.getBackground());
	setLayout(new BorderLayout()); Label lab=new Label(text,Label.CENTER);
	lab.setFont(currft); lab.setForeground(fore); add(lab,BorderLayout.CENTER); pack(); setLayout(null);
   }
   public void paint(Graphics G){Dimension d=getSize();
	if(!canrun)return; G.drawRect(0,0,d.width-1,d.height-1);}
   public Insets getInsets(){return new Insets(1,1,1,1);}
  }
//---------------------------------------------------------------------------------------------
  private class GOF extends RGBImageFilter {
   public GOF(){canFilterIndexColorModel=true;}
   public int filterRGB(int x,int y,int rgb){
	int r=(rgb & (255 << 16)) >> 16;
	int g=(rgb & (255 <<8)) >> 8;
	int b=rgb & 255;
	r=(int)((r+g+b)/3);
	return (255 << 24) + (r << 16) + (r<<8) + r;
   }
  }//end of GOF
//---------------------------------------------------------------------------------------------
  class IF extends RGBImageFilter{
    public IF(){canFilterIndexColorModel = true;}
    public int filterRGB(int x, int y, int rgb){return (rgb ^ 0xffffff);}
  }
//---------------------------------------------------------------------------------------------
  class DF extends RGBImageFilter {
   public DF(){canFilterIndexColorModel = true;}
   public int filterRGB(int x, int y, int rgb) {return(rgb & 0xffafafaf);}
  }// end of DarkFilter
//---------------------------------------------------------------------------------------------
}//end of IEButton

