package com.equitysoft.hashstore;
import java.io.*; import java.text.*;

/**
 * This class is an extension of DiskHashTable that allows duplicate (ie. non-unique) keys.
 *
 * To retrieve values the programmer should first use <code>get</code> followed by repeated invocations of
 * <code>getNext</code> to retrieve all the duplicate entries for a given key.
 * <p>Caution should be exercised over
 * how many duplicates of a key are added. The duplicates form a chain and each new duplicate is added to the
 * end of the chain. This class maintains a "current entry" pointer to allow traversing and deletion of entries.
 * After hundreds of duplicates are added, performance may become a problem when adding or deleting
 * since the end of the chain has to be found. Retrieval should never be a problem.
 *<p>
 * None of the methods are synchronized. The class <code>SyncDuplicateDiskHashtable</code> provides a
 * synchronized version.<p>
 * This source code is subject to the terms and conditions of the General Public License
 *
 * @author  Colin Mummery - colin_mummery@my-deja.com - http://www.kagi.com/equitysoft
 */

public class DuplicateDiskHashtable extends DiskHashtable{

   public DuplicateDiskHashtable(File root) {
	super(root);
   }
//-------------------------------------------------------------------------
/**
 * Puts the specified <code>key</code> and <code>value</code> pair into the table even though the key value may
 * already exist in the table. The current key value is set to this entry.
 * @return Always returns <code>null</code>. The returned value is necessary for compatability with the super-class.
 * @see #getCurrentKey()
 */
   public Object put(Object key,Object value) throws IOException {
	super.put(key,value,false); return null;
   }
//--------------------------------------------------------------------------
/**
 * Gets the object associated with the the first instance of the key. A call to this method should be followed by
 * repeated calls to <code>getNext()</code> to retrieve subsequent values with the same key.
 * If the object causes a <code>ClassNotFoundException</code> when deserializing.
 * , then the exception is transformed into an <code>IOException</code> for convenience.
 * @return <code>null</code> if the key isn't found otherwise the associated object
 * @see #getNext()
 */
   public Object get(Object key) throws IOException{
	return super.get(key,false);
   }
//--------------------------------------------------------------------------
/**
 * This method can be repeatedly called after invoking <code>get</code> in order to retrieve all entries in the tabel
 * with the same key.
 * @return <code>null</code> if the key isn't found otherwise the value associated with the next key entry
 * @see #get(Object)
 */
   public Object getNext() throws IOException {
	return super.get(null,true);
   }
//--------------------------------------------------------------------------
/**
 * Remove all entries that match the given key. This method doesn't return a value as more than one entry may be deleted.
 * 
 * @return Always returns <code>null</code>. Returning something is necessary for compatability with the super-class.
 * @see #removeCurrent()
 */
   public Object remove(Object key) throws IOException { //Remove all keys that match the given key
	super.remove(key,false); return null;
   }
//--------------------------------------------------------------------------
/**
 * Returns the current key value or null if none is set. This method is useful if a program needs to check that a 
 * current value is set.
 */
   public Object getCurrentKey() throws IOException {
	if(current==null)return null; return loadObject(new File(keyroot,current));
   }
//--------------------------------------------------------------------------
/**
 * Returns the value associated with the current key value or null if no current key is set.
 */
   public Object getCurrentValue() throws IOException {
	if(current==null)return null; return loadObject(new File(dataroot,current));
   }
//--------------------------------------------------------------------------
/**
 * Removes the current entry in the table. The current entry is set by invoking <code>get</code> followed
 * by repeated calls to <code>getNext</code> to move to the entry to be deleted. After deletion the current entry
 * becomes the following entry with the same key value if there is one.
 * @return The value associated with the removed entry
 * @see #get(Object)
 * @see #getNext()
 * @see #remove(Object)
 */

   public Object removeCurrent() throws IOException { //We should leave the current value pointing at the next occurence
	if(current==null)return null;
	String loc=current; current=null;
	File keyfile=new File(keyroot,loc);
	if(!keyfile.exists())return null; //it's not there for some reason
	File datafile=new File(dataroot,loc); Object oldobj=loadObject(datafile);
	keyfile.delete(); datafile.delete();
//Now shift any subsequent files along by renaming
	while(true){
		loc=incName(loc,1);
		keyfile=new File(keyroot,loc);
		if(!keyfile.exists())return oldobj; //No more left to process
		if(current==null && loadObject(keyfile).equals(currkey))current=incName(loc,-1);
		keyfile.renameTo(new File(keyroot,incName(loc,-1)));
		datafile=new File(dataroot,loc);
		datafile.renameTo(new File(dataroot,incName(loc,-1)));
	}
   }
//--------------------------------------------------------------------------
}

