with DJGPP_Library; use DJGPP_Library;
with Dos_Interrupts; use Dos_Interrupts;
with Ada.Unchecked_Conversion;
with System.Bit_Ops; use System.Bit_Ops;
with System.Machine_Code; use System.Machine_Code;

package body Nmodem is

  function Short_to_Unsigned16 is new Ada.Unchecked_Conversion(Source=>Short_Integer,Target=>Unsigned_16);
  function Byte_To_Char is new Ada.Unchecked_Conversion(Source=>Unsigned_8,Target=>Character);
  function Char_To_Byte is new Ada.Unchecked_Conversion(Source=>Character,Target=>Unsigned_8);


--------------------------------------------------------------------
--  Procedure Serial_ISR                                          --
--                                                                --
--  Purpose:  To handle an interrupt generated by the null        --
--            modem port.                                         --
--                                                                --
--  Input: None                                                   --
--------------------------------------------------------------------
  Procedure Serial_ISR is
     begin
       serial_lock := 1;

       --Place character into next position in buffer
       ser_ch := Inportb(Short_To_Unsigned16(Short_Integer(open_port) +
                 Short_Integer(SER_RBF)));

       --Wrap buffer index around
       if ( (ser_end+1) > SERIAL_BUFFER_SIZE )
       then
       ser_end := 0;
       else
       ser_end := ser_end+1;
       end if;

       --Move character into buffer
       ser_buffer(ser_end) := ser_ch;

       char_ready := char_ready+1;
       
       Outportb(PIC_ICR, 16#20#);

       serial_lock := 0;
  end Serial_ISR;

----------------------------------------------------------------------
--  Function Ready_Serial                                           --
--                                                                  --
--  Purpose:  Return how many characters are in the serial_buffer   --
--                                                                  --
--  Input: None                                                     --
--  Output: Number of characters in the serial buffer               --
----------------------------------------------------------------------
  Function Ready_Serial return Integer is
    begin
    return char_ready;
  end Ready_Serial;


--------------------------------------------------------------------------
--  Function Serial_Read                                                --
--                                                                      --
--  Purpose:  To read a character from the serial_buffer and return it  --
--            to the caller.                                            --
--                                                                      --
--  Input: None                                                         --
--  Output: A Byte from the other machine                               --
--------------------------------------------------------------------------
  Function Serial_Read return Unsigned_8 is
    ch:Unsigned_8;
    begin
       while (serial_lock = 1) loop
         null;
       end loop;

       if ((ser_start+1) > SERIAL_BUFFER_SIZE) then
          ser_start := 0;
       else
          ser_start := ser_start+1;
       end if;

       ch := ser_buffer(ser_start);

       if (char_ready > 0) then
          char_ready := char_ready-1;
       end if;

    return ch;
  end Serial_Read;


---------------------------------------------------------------------
--  Procedure Serial_Write                                         --
--                                                                 --
--  Purpose:  To write a byte to the transmit buffer          --
--                                                                 --
--  Input:  The byte that we want to transfer                      --
---------------------------------------------------------------------
  Procedure Serial_Write(ch:Unsigned_8) is
        And_Result : Unsigned_8 := 16#00#; --Used for bitwise AND
        Op1 : Unsigned_8;
        Op2 : Unsigned_8;
         begin

         Op2 := 16#20#;
         Op1 := Inportb(Short_To_Unsigned16
             (Short_Integer(open_port) + Short_Integer(SER_LSR)));
         Bit_And(Op1'address,8,Op2'address,8,And_Result'address);
         --Wait for the transmit buffer to be empty
         while (And_Result /= 16#20#) loop
             Bit_And(Op1'address,8,Op2'address,8,And_Result'address);
         end loop;

         --Disable interrupts
         asm("cli");

         --Send the byte
         Outportb(Short_To_Unsigned16(Short_Integer(open_port) +
               Short_Integer(SER_THR)), ch);

         --Turn interrupts back on
         asm("sti");
  end Serial_Write;

---------------------------------------------------------------------
--  Procedure Open_Serial                                          --
--                                                                 --
--  Purpose: Set up the null modem so that it can run              --
--                                                                 --
--  Inputs: Port_Base, Baud, configuration                         --
---------------------------------------------------------------------
  Procedure Open_Serial(Port_Base:Unsigned_16; Baud:Unsigned_8;
                        Configuration:Unsigned_8)
  is
     Old_Int_Mask : Unsigned_8 := Inportb(PIC_IMR);
     And_Result : Unsigned_8;
     COM1_Mask : Unsigned_8 := 16#EF#;
  begin
     Open_Port := Port_Base;

     --Turn on Divisor Latch Registers and set BAUD rate
     Outportb(Short_To_Unsigned16(Short_Integer(open_port) +
              Short_Integer(SER_LCR)), SER_DIV_LATCH_ON);
     Outportb(Short_To_Unsigned16(Short_Integer(open_port) +
              Short_Integer(SER_DLL)), baud);
     Outportb(Short_To_Unsigned16(Short_Integer(open_port) +
              Short_Integer(SER_DLM)), 0);

     --Set configuration for port
     Outportb(Short_To_Unsigned16(Short_Integer(open_port) +
              Short_Integer(SER_LCR)), configuration);

     --Enable the interrupts
     Outportb(Short_To_Unsigned16(Short_Integer(open_port) +
              Short_Integer(SER_MCR)), SER_GP02);
     Outportb(Short_To_Unsigned16(Short_Integer(open_port) +
              Short_Integer(SER_IER)), 16#01#);

     --Hold off enabling PIC until we install ISR
       Set_Interrupt(INT_SER_PORT_0, Serial_ISR'address);

     --Enable the interrupt on PIC
     Bit_And(Old_Int_Mask'address,8,Com1_Mask'address,8,And_Result'address);
     Outportb(PIC_IMR,And_Result);

  end Open_Serial;

  
---------------------------------------------------------------------
--  Procedure Close_Serial                                         --
--                                                                 --
--  Purpose: Self Explanatory                                      --
--                                                                 --
--  Input:  The port we want to close                              --
---------------------------------------------------------------------

  Procedure Close_Serial(Port_Base:Unsigned_16)
  is begin
     null;
  end Close_Serial;
  
end Nmodem;

