//###################################################################################
//#																					#
//#			Table object class														#
//#				- Ales Daneu, 06/11/2002											#
//#																					#
//###################################################################################
#include "TableClass.h"

// Table limits
float TableLimits[2][2] = {{-178.45f, -88.9f, },
							{178.45f, 88.9f}};
// Pocket positions
// Right top
float Pocket1[2][2] = {{-83.25f, -178.45f},
						{-88.9f, -172.8f}};
int Pocket1_K = -1;
float Pocket1_N	= -261.7f;
// Right side
float Pocket2[2][2] = {{-88.9f, -4.0f},
						{-88.9f, 4.0f}};
// Right bottom
float Pocket3[2][2] = {{-88.9f, 172.8f},
						{-83.25f, 178.45f}};
int Pocket3_K = 1;
float Pocket3_N = 261.7f;
// Left bottom
float Pocket4[2][2] = {{88.9f, 172.8f},
						{83.25f, 178.45f}};
int Pocket4_K = -1;
float Pocket4_N = 261.7f;
// Left side
float Pocket5[2][2] = {{88.9f, -4.0f},
						{88.9f, 4.0f}};
// Left top
float Pocket6[2][2] = {{83.25f, -178.45f},
						{88.9f, -172.8f}};
int Pocket6_K = 1;
float Pocket6_N = -261.7f;

//-----------------------------------------------------------------------------
// Name: vDisplayXYZ()
// Desc: Display table on scene
//-----------------------------------------------------------------------------
void TableClass::vDisplayXYZ(void)
{
	Object3DClass::vDisplayXYZ(0.0f, 0.0f, 0.0f);
}

//-----------------------------------------------------------------------------
// Name: vBallCollisionDetection(*pBall,bGamePlay)
// Desc: Collision detection between ball and table
//-----------------------------------------------------------------------------
bool TableClass::vBallCollisionDetection(BallClass *pBall, const bool bGamePlay)
{
	D3DXVECTOR3		vecOldBallMovement;
	D3DXVECTOR3		vecNewBallMovement;
	D3DXVECTOR3		vecBallPosition = pBall->vecGetPosition();
	float			fOffsetX;
	float			fOffsetZ;
	float			fBallSpeed;
	float			fPocketLimit;
	bool			result;

	// Assume that there is no collision
	result = false;
	// No need to do anything of that if ball is not moving or it was already potted
	if ((!pBall->bGetIsBallMoving() && bGamePlay)|| pBall->bGetPotted())
		return result;
	// Bottom side of table was hit
	if ((vecBallPosition.z + BALL_RADIUS) > TableLimits[1][0])
	{
		// Check if ball has gone into (3) right bottom pocket
		//							   (4) left bottom pocket
		if ((vecBallPosition.x <= Pocket3[1][0]) ||
			(vecBallPosition.x >= Pocket4[1][0]))
		{
			pBall->vSetPotted(true);
			result = true;
		}
		else
		{
			vecOldBallMovement = pBall->vecGetBallMovement();
			// Reposition the ball to avoid double collision detection
			fOffsetZ = vecBallPosition.z + BALL_RADIUS - TableLimits[1][0];
			if (vecOldBallMovement.x > 0.0f)
				fOffsetX = (fOffsetZ / vecOldBallMovement.z) * vecOldBallMovement.x;
			else if (vecOldBallMovement.x < 0.0f)
				fOffsetX = -(fOffsetZ / vecOldBallMovement.z) * vecOldBallMovement.x;
			else
				fOffsetX = 0.0f;
			pBall->vUpdatePosition(&D3DXVECTOR3(fOffsetX, 0.0f, fOffsetZ));
			// Set new movement direction
			D3DXVec3Normalize(&vecNewBallMovement, &D3DXVECTOR3(vecOldBallMovement.x, 0.0f, -vecOldBallMovement.z));
			pBall->vSetBallMovement(&vecNewBallMovement);
			// Set new speed
			fBallSpeed = pBall->fGetBallSpeed();
			pBall->vSetBallSpeed(0.99f * fBallSpeed);
		}
	}
	// Top side of table was hit
	else if ((vecBallPosition.z - BALL_RADIUS) < TableLimits[0][0])
	{
		// Check if ball has gone into (1) right top pocket
		//							   (6) left top pocket
		if ((vecBallPosition.x <= Pocket1[0][0]) ||
			(vecBallPosition.x >= Pocket6[0][0]))
		{
			pBall->vSetPotted(true);
			result = true;
		}
		else
		{
			vecOldBallMovement = pBall->vecGetBallMovement();		
			// Reposition the ball to avoid double collision detection
			fOffsetZ = vecBallPosition.z - BALL_RADIUS - TableLimits[0][0];
			if (vecOldBallMovement.x > 0.0f)
				fOffsetX = (fOffsetZ / vecOldBallMovement.z) * vecOldBallMovement.x;
			else if (vecOldBallMovement.x < 0.0f)
				fOffsetX = -(fOffsetZ / vecOldBallMovement.z) * vecOldBallMovement.x;
			else
				fOffsetX = 0.0f;
			pBall->vUpdatePosition(&D3DXVECTOR3(fOffsetX, 0.0f, fOffsetZ));
			// Set new movement direction
			D3DXVec3Normalize(&vecNewBallMovement, &D3DXVECTOR3(vecOldBallMovement.x, 0.0f, -vecOldBallMovement.z));
			pBall->vSetBallMovement(&vecNewBallMovement);
			// Set new speed
			fBallSpeed = pBall->fGetBallSpeed();
			pBall->vSetBallSpeed(0.99f * fBallSpeed);
		}
	}
	else
	{
		// Left side of table was hit
		if ((vecBallPosition.x + BALL_RADIUS) > TableLimits[1][1])
		{
			// Check if ball has gone into (6) left top pocket
			//                             (5) left side pocket
			//							   (4) left bottom pocket
			if ((vecBallPosition.z <= Pocket6[1][1]) ||
				((vecBallPosition.z >= Pocket5[0][1]) && 
				(vecBallPosition.z <= Pocket5[1][1])) ||
				(vecBallPosition.z >= Pocket4[0][1]))
			{
				pBall->vSetPotted(true);
				result = true;
			}
			else
			{
				vecOldBallMovement = pBall->vecGetBallMovement();			
				// Reposition the ball to avoid double collision detection
				fOffsetX = vecBallPosition.x + BALL_RADIUS - TableLimits[1][1];
				if (vecOldBallMovement.z > 0.0f)
					fOffsetZ = -(fOffsetX / vecOldBallMovement.x) * vecOldBallMovement.z;
				else if (vecOldBallMovement.z < 0.0f)
					fOffsetZ = (fOffsetX / vecOldBallMovement.x) * vecOldBallMovement.z;
				else
					fOffsetZ = 0.0f;
				pBall->vUpdatePosition(&D3DXVECTOR3(fOffsetX, 0.0f, fOffsetZ));
				// Set new movement direction
				D3DXVec3Normalize(&vecNewBallMovement, &D3DXVECTOR3(-vecOldBallMovement.x, 0.0f, vecOldBallMovement.z));
				pBall->vSetBallMovement(&vecNewBallMovement);
				// Set new speed
				fBallSpeed = pBall->fGetBallSpeed();
				pBall->vSetBallSpeed(0.99f * fBallSpeed);
			}
		}
		// Right side of table was hit
		else if ((vecBallPosition.x - BALL_RADIUS) < TableLimits[0][1])
		{
			// Check if ball has gone into  (1) right top pocket
			//								(2) right side pocket
			//								(3) right bottom pocket
			if ((vecBallPosition.z <= Pocket1[1][1]) ||
				((vecBallPosition.z >= Pocket2[0][1]) && 
				(vecBallPosition.z <= Pocket2[1][1])) ||
				(vecBallPosition.z >= Pocket3[0][1]))
			{
				pBall->vSetPotted(true);
				result = true;
			}
			else
			{
				vecOldBallMovement = pBall->vecGetBallMovement();		
				// Reposition the ball to avoid double collision detection
				fOffsetX = vecBallPosition.x - BALL_RADIUS - TableLimits[0][1];
				if (vecOldBallMovement.z > 0.0f)
					fOffsetZ = -(fOffsetX / vecOldBallMovement.x) * vecOldBallMovement.z;
				else if (vecOldBallMovement.z < 0.0f)
					fOffsetZ = (fOffsetX / vecOldBallMovement.x) * vecOldBallMovement.z;
				else
					fOffsetZ = 0.0f;
				pBall->vUpdatePosition(&D3DXVECTOR3(fOffsetX, 0.0f, fOffsetZ));
				// Set new movement direction
				D3DXVec3Normalize(&vecNewBallMovement, &D3DXVECTOR3(-vecOldBallMovement.x, 0.0f, vecOldBallMovement.z));
				pBall->vSetBallMovement(&vecNewBallMovement);
				// Set new speed
				fBallSpeed = pBall->fGetBallSpeed();
				pBall->vSetBallSpeed(0.99f * fBallSpeed);
			}                                                                                                    
		}
		else
		{
			// Right side pockets
			if (vecBallPosition.x <= Pocket1[0][0])
			{
				// Top pocket
				if (vecBallPosition.z <= Pocket1[1][1])
				{
					// Check if ball will fall into pocket
					fPocketLimit = Pocket1_K * vecBallPosition.x + Pocket1_N;
					if (vecBallPosition.z <= fPocketLimit) 
					{
						pBall->vSetPotted(true);
						result = true;
					}
				}
				// Bottom pocket
				else if (vecBallPosition.z >= Pocket3[0][1])
				{
					// Check if ball will fall into pocket
					fPocketLimit = Pocket3_K * vecBallPosition.x + Pocket3_N;
					if (vecBallPosition.z >= fPocketLimit)
					{
						pBall->vSetPotted(true);
						result = true;
					}
				}
			}
			else if (vecBallPosition.x >= Pocket4[1][0])
			{
				// Top pocket
				if (vecBallPosition.z <= Pocket6[1][1])
				{
					// Check if ball will fall into pocket
					fPocketLimit = Pocket6_K * vecBallPosition.x + Pocket6_N;
					if (vecBallPosition.z <= fPocketLimit)
					{
						pBall->vSetPotted(true);
						result = true;
					}
				}
				// Bottom pocket
				else if (vecBallPosition.z >= Pocket4[0][1])
				{
					// Check if ball will fall into pocket
					fPocketLimit = Pocket4_K * vecBallPosition.x + Pocket4_N;
					if (vecBallPosition.z >= fPocketLimit)
					{
						pBall->vSetPotted(true);
						result = true;
					}
				}
			}
		}
	}
	return result;
}

//-----------------------------------------------------------------------------
// Name: vCueCollisionDetection(*pCue)
// Desc: Collision detection between cue and table
//-----------------------------------------------------------------------------
void TableClass::vCueCollisionDetection(CueClass *pCue)
{
	D3DXVECTOR3 vecPosition;
	D3DXVECTOR3	vecRotation;
	float		fDistanceX;
	float		fDistanceZ;
	float		fCueLengthOnTable;
	float		fCueToTableAngle;
	float		fLengthX;
	float		fLengthZ;
	float		fTemp;

	// Get cue position and rotation
	vecPosition = pCue->vecGetPosition();
	vecRotation = pCue->vecGetRotation();
	// Determine in which quadrant of table cue is and compute distance from cushion based on that
	if ((vecRotation.x >= 0.0f) && (vecRotation.x < 90.0f))
	{
		fDistanceX = TableLimits[1][1] - vecPosition.x;
		fDistanceZ = TableLimits[1][0] - vecPosition.z;
	}
	else if ((vecRotation.x >= 90.0f) && (vecRotation.x < 180.0f))
	{
		fDistanceX = TableLimits[1][1] - vecPosition.x;
		fDistanceZ = vecPosition.z - TableLimits[0][0];
	}
	else if ((vecRotation.x >= 180.0f) && (vecRotation.x < 270.0f))
	{
		fDistanceX = vecPosition.x - TableLimits[0][1];
		fDistanceZ = vecPosition.z - TableLimits[0][0];
	}
	else if ((vecRotation.x >= 270.0f) && (vecRotation.x <= 360.0f))
	{
		fDistanceX = vecPosition.x - TableLimits[0][1];
		fDistanceZ = TableLimits[1][0] - vecPosition.z;
	}
	// Cue is in starting position
	if ((vecRotation.x == 0.0f) || (vecRotation.x == 360.0f))
	{
		fCueLengthOnTable = fDistanceZ;
		fCueToTableAngle = 360.0f - (float)(atan((CUSHION_HEIGHT + CUE_RADIUS) / fCueLengthOnTable) * RADIAN_TO_DEGREES);
	}
	// Cue is rotated
	else
	{
		fLengthX = (float)fabs((float)sin(vecRotation.x / RADIAN_TO_DEGREES) * CUE_LENGTH);
		fLengthZ = (float)fabs((float)cos(vecRotation.x / RADIAN_TO_DEGREES) * CUE_LENGTH);
		// case when cue does not collide with cushion
		if ((fLengthX <= fDistanceX) && (fLengthZ <= fDistanceZ))
		{
			fCueToTableAngle = 360.0f;
		}
		// top/bottom
		else if (fLengthX < fDistanceX)
		{
			fTemp = (float)tan(vecRotation.x / RADIAN_TO_DEGREES) * fDistanceZ;
			fCueLengthOnTable = (float)sqrt(fTemp * fTemp + fDistanceZ * fDistanceZ);	
			fCueToTableAngle = 360.0f - (float)(atan((CUSHION_HEIGHT + CUE_RADIUS) / fCueLengthOnTable) * RADIAN_TO_DEGREES);
		}
		// left/right side
		else if (fLengthZ < fDistanceZ)
		{
			fTemp = fDistanceX / (float)tan(vecRotation.x / RADIAN_TO_DEGREES);
			fCueLengthOnTable = (float)sqrt(fTemp * fTemp + fDistanceX * fDistanceX);	
			fCueToTableAngle = 360.0f - (float)(atan((CUSHION_HEIGHT + CUE_RADIUS) / fCueLengthOnTable) * RADIAN_TO_DEGREES);
		}
		// pockets
		else
		{
			// try the same procedure as if cue collides with left/right side
			fTemp = fDistanceX / (float)tan(vecRotation.x / RADIAN_TO_DEGREES);
			// if the result is to high then use the same approach as for top/bottom
			if (fTemp > fDistanceZ)
				fTemp = (float)tan(vecRotation.x / RADIAN_TO_DEGREES) * fDistanceZ;
			fCueLengthOnTable = (float)sqrt(fTemp * fTemp + fDistanceX * fDistanceX);	
			fCueToTableAngle = 360.0f - (float)(atan((CUSHION_HEIGHT + CUE_RADIUS) / fCueLengthOnTable) * RADIAN_TO_DEGREES);
		}
	}
	// Set cue angle so the cue does not collide with table	(in case cue collided with table)
	if (((fCueToTableAngle < vecRotation.y) && (vecRotation.y > 270.0f)) ||
		(vecRotation.y < 90.0f))
		pCue->vSetRotation( &D3DXVECTOR3( vecRotation.x, fCueToTableAngle, vecRotation.z) );
}

//-----------------------------------------------------------------------------
// Name: vBaulkMovementCheck(*pBall)
// Desc: Checks if ball is within baulk limits (when cue ball is in hand)
//-----------------------------------------------------------------------------
void TableClass::vBaulkMovementCheck(BallClass *pBall)
{
	D3DXVECTOR3 vecPosition;
	float fRadius;
	float fDistanceZ;
	float fMultiplier;
	float fNewDistanceX;
	float fNewDistanceZ;

	vecPosition = pBall->vecGetPosition();
	fDistanceZ = vecPosition.z - BAULK_CENTER_Z;
	// If ball is over baulk just set it's Z component to maximum allowed
	if (fDistanceZ < 0.0f)
		pBall->vSetPosition( &D3DXVECTOR3(vecPosition.x, 0.0f, BAULK_CENTER_Z) );
	else
	{
		// Compute ball's radius (center in baulk center)
		fRadius = (float)sqrt((vecPosition.x * vecPosition.x) + fDistanceZ * fDistanceZ);
		// If radius is too big we have to set new position
		if (fRadius > BAULK_RADIUS)
		{
			// If we are left or right from baulk circle we have to change only position on Z axis
			if ((float)fabs(vecPosition.x) > BAULK_RADIUS)
				pBall->vSetPosition( &D3DXVECTOR3(vecPosition.x, 0.0f, BAULK_CENTER_Z) );
			else
			{
				//Compute position that complies to radius limitation
				fMultiplier = fDistanceZ / vecPosition.x;
				fNewDistanceX = (float)sqrt((BAULK_RADIUS * BAULK_RADIUS) / (1 + fMultiplier * fMultiplier));
				fNewDistanceZ = fNewDistanceX * (float)fabs(fMultiplier);	
				if (fMultiplier < 0.0f)
					fNewDistanceX = -fNewDistanceX;
				pBall->vSetPosition( &D3DXVECTOR3(fNewDistanceX, 0.0f, BAULK_CENTER_Z + fNewDistanceZ) );
			}
		}
	} 
}