//###################################################################################
//#																					#
//#			Snooker Client															#
//#				- Ales Daneu, 06/11/2002											#
//#																					#
//#			Portions of code are based on RPGQuest Client							#
//#				- Todd Barron, 12/03/2001											#
//#																					#
//###################################################################################
#include "Snooker.h"
#include "..\\Common\\PacketInfo.h"

//-----------------------------------------------------------------------------
// Name: WinMain()
// Desc: The application's entry point
//-----------------------------------------------------------------------------
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
	WNDCLASSEX	wndclass;
	HWND		hWnd;
	MSG			msg;
	DWORD		dwGovernor = timeGetTime();

	// Set up window attributes
	wndclass.cbSize			= sizeof(wndclass);
	wndclass.style			= CS_HREDRAW | CS_VREDRAW;
	wndclass.lpfnWndProc	= MsgProc;
	wndclass.cbClsExtra		= 0;
	wndclass.cbWndExtra		= 0;
	wndclass.hInstance		= hInstance;
	wndclass.hIcon			= LoadIcon( hInstance, MAKEINTRESOURCE(IDI_ICON) );
	wndclass.hCursor		= LoadCursor( NULL, IDC_ARROW );
	wndclass.hbrBackground	= (HBRUSH)(COLOR_WINDOW);
	wndclass.lpszMenuName	= MAKEINTRESOURCE(IDR_MENU);
	wndclass.lpszClassName	= szWndClass;	// Registered Class Name
	wndclass.hIconSm		= LoadIcon( hInstance, MAKEINTRESOURCE(IDI_ICON) );
		
	if( RegisterClassEx( &wndclass ) == 0 ) {
		exit(1);
	}
	
	// Create the main window
	hWnd = CreateWindow(	
		szWndClass,				// Class Name
		szProgramName,			// Name Displayed on Title Bar
		WS_OVERLAPPEDWINDOW,
		0,
		0,
		SCREEN_WIDTH,
		SCREEN_HEIGHT,
		NULL,
		NULL,
		hInstance,
		NULL );

	// Setup global window instance
	g_hInstance = hInstance;
	ShowWindow(hWnd, nCmdShow);
	UpdateWindow(hWnd);

	// Clear out player information	
	vInitializeObjects();
	// Init Direct graphics
	if( hrInit3D(hWnd) == E_FAIL ) {
		MessageBox( hWnd, "Direct3D Error", "Unable to initialize Direct 3D.", MB_ICONERROR );
		vCleanup();
		exit(1);
	}
	// Setup the 3D models and textures
	vInitGeometry();
	// Setup the view
	vSetupView();
	// Create scene lights
	vCreateLights();
	// Prepare game logic for new game
	vPrepareGame(hWnd);
	// Init sound system
	vInitSoundEngine(hWnd); 
	// Render
	vRenderScene();
	// Main game loop
	ZeroMemory( &msg, sizeof(msg) );
    while( msg.message != WM_QUIT ) {
        if( PeekMessage( &msg, NULL, 0U, 0U, PM_REMOVE ) ) {
            TranslateMessage( &msg );
            DispatchMessage( &msg );
        }
        else {
			// Dont update game more than 30 FPS
			if( timeGetTime() > dwGovernor+33 ) {
				dwGovernor = timeGetTime();
				// Update players and execute pending actions
				vUpdateObjects(hWnd);
				vUpdateNetwork(hWnd);
				vRenderScene();
			}
		}
    }
	// Cleanup the environment
	vCleanup();

	return(msg.wParam);
};

//-----------------------------------------------------------------------------
// Name: MsgProc()
// Desc: The application's message loop
//-----------------------------------------------------------------------------
LRESULT WINAPI MsgProc( HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
	int wmId, wmEvent;

	switch( msg )
    {
		case WM_COMMAND:
			wmId    = LOWORD(wParam); 
			wmEvent = HIWORD(wParam); 
			// Parse the menu selections:
			switch (wmId)
			{
				case IDM_HELP: // Help menu clicked
				   ShellExecute(hWnd, "open", "ADSnooker.chm", NULL, NULL, SW_SHOWNORMAL );
				   break;
				default:
				   return DefWindowProc(hWnd, msg, wParam, lParam);
			}
			break;
		case WM_SETCURSOR:
            // Turn off Windows cursor in fullscreen mode
            if( !g_bWindowed )
            {
                SetCursor( NULL );
                return TRUE;
            }
            break;
        case WM_DESTROY:
            PostQuitMessage( 0 );
			return 0;
		case WM_KEYDOWN:
            switch (wParam)
            {
                case VK_ESCAPE: // Quit
					PostMessage(hWnd, WM_QUIT, 0, 0);
                    return 0;
				case VK_F1:		// Bird view
					bBirdView = TRUE;
                    return 0;
				case VK_F2:		// Used to select yellow as ball on
					if (bChooseBallOn)
					{
						PlayerInfo[g_iCurrentPlayer].sBallOnValue = YELLOW_VALUE;
						bChooseBallOn = FALSE;
					}
					return 0;
				case VK_F3:		// Used to select green as ball on
					if (bChooseBallOn)
					{
						PlayerInfo[g_iCurrentPlayer].sBallOnValue = GREEN_VALUE;
						bChooseBallOn = FALSE;
					}
					return 0;
				case VK_F4:		// Used to select brown as ball on
					if (bChooseBallOn)
					{
						PlayerInfo[g_iCurrentPlayer].sBallOnValue = BROWN_VALUE;
						bChooseBallOn = FALSE;
					}
					return 0;
				case VK_F5:		// Used to select blue as ball on
					if (bChooseBallOn)
					{
						PlayerInfo[g_iCurrentPlayer].sBallOnValue = BLUE_VALUE;
						bChooseBallOn = FALSE;
					}
					return 0;
				case VK_F6:		// Used to select pink as ball on
					if (bChooseBallOn)
					{
						PlayerInfo[g_iCurrentPlayer].sBallOnValue = PINK_VALUE;
						bChooseBallOn = FALSE;
					}
					return 0;
				case VK_F7:		// Used to select black as ball on
					if (bChooseBallOn)
					{
						PlayerInfo[g_iCurrentPlayer].sBallOnValue = BLACK_VALUE;
						bChooseBallOn = FALSE;
					}
					return 0;
				case VK_N:		// Used to select non-multiplayer game
					if (bChooseGameType)
					{
						bNetworkGame = FALSE;
						bNextPlayer = TRUE;
						bChooseGameType = FALSE;
					}
					return 0;
				case VK_H:		// Used to select multiplayer game (host)
					if (bChooseGameType)
					{
						bNetworkGame = TRUE;
						bHostGame = TRUE;
					}
					return 0;
				case VK_J:		// Used to select multiplayer game (join)
					if (bChooseGameType)
					{
						bNetworkGame = TRUE;
						bHostGame = FALSE;
					}
					return 0;
				case VK_RETURN:	// Used to start a new game
					if (bGameOver)
						vPrepareGame(hWnd);
					return 0;
				case VK_UP:		// Used to set shot power
					bNextPlayer = FALSE;
					if (bIsPlayAllowed())
					{
						sShotPower++;
						if (sShotPower > MAX_POWER)
							sShotPower = MAX_POWER;
					}
					return 0;
				case VK_DOWN:	// Used to set shot power
					bNextPlayer = FALSE;
					if (bIsPlayAllowed())
					{
						sShotPower--;
						if (sShotPower < MIN_POWER)
							sShotPower = MIN_POWER;
					}
					return 0;
				case VK_SPACE:	// Do stroke
					bNextPlayer = FALSE;
					if (!bChooseBallOn && bIsPlayAllowed())
						bPerformShot = TRUE;
					return 0;
			}
			break;
		case WM_KEYUP:
            switch (wParam)
            {
                case VK_F1:		// Back to normal view
					bBirdView = FALSE;
                    return 0;
			}
			break;
		case WM_RBUTTONDOWN:	// Start moving cue ball
			bNextPlayer = FALSE;
			if (bBallInHand && bIsPlayAllowed())
			{
				bCueBallMove = TRUE;
				pointCursorStart.x = LOWORD(lParam);
				pointCursorStart.y = HIWORD(lParam);
				ClientToScreen( hWnd , &pointCursorStart );
			}
			return 0;
		case WM_RBUTTONUP:		// Stop moving cue ball
			if (bBallInHand && bIsPlayAllowed())
				bCueBallMove = FALSE;
			return 0;
		case WM_LBUTTONDOWN:	// Start moving cue
			bNextPlayer = FALSE;
			if (bIsPlayAllowed())
			{
				bCueMove = TRUE;
				pointCursorStart.x = LOWORD(lParam);
				pointCursorStart.y = HIWORD(lParam);
				ClientToScreen( hWnd , &pointCursorStart );
			}
			return 0;
		case WM_LBUTTONUP:		// Stop moving cue ball
			if (bIsPlayAllowed())
				bCueMove = FALSE;
			return 0;
		case WM_MOUSEWHEEL:		// Set shot power
			bNextPlayer = FALSE;
			if (bIsPlayAllowed())
			{
				sShotPower += (short)HIWORD(wParam)/WHEEL_DELTA;
				if (sShotPower < MIN_POWER)
					sShotPower = MIN_POWER;
				if (sShotPower > MAX_POWER)
					sShotPower = MAX_POWER;
			}
			return 0;
		case WM_MBUTTONDOWN:	// Do stroke
			bNextPlayer = FALSE;
			if (!bChooseBallOn && bIsPlayAllowed())
				bPerformShot = TRUE;
			return 0;
    }
    return DefWindowProc( hWnd, msg, wParam, lParam );
}

//-----------------------------------------------------------------------------
// Name: bIsPlayAllowed()
// Desc: Checks if the game can be continued
//-----------------------------------------------------------------------------
bool bIsPlayAllowed(void)
{
	return ((!bChooseGameType) && (!bConnecting) && (!bGameOver) && (g_iMyPlayerId == g_iCurrentPlayer) && bPacketsProcessed);
}

//-----------------------------------------------------------------------------
// Name: vCleanup()
// Desc: Cleans up Direct X Graphics & Other allocations
//-----------------------------------------------------------------------------
void vCleanup(void)
{
	int i;
	
	// Release Graphics
	for( i = 0 ; i < MAX_TEXTURES ; i++ ) {
		SAFE_RELEASE( g_pTexture[i] );
	}
	SAFE_RELEASE( pD3DXFont );
	SAFE_RELEASE( pd3dDevice );
	SAFE_RELEASE( pD3D );
	if( hFont ) {
		DeleteObject( hFont );
		hFont = NULL;
	}
	// Release Sounds
	SAFE_DELETE( g_GameSounds.gsBallHit );
	SAFE_DELETE( g_GameSounds.gsBallInPocket );
	// Release Direct Play Objects
	SAFE_RELEASE( g_pDeviceAddress );
    SAFE_RELEASE( g_pHostAddress );
	if( g_pDP ) {
		// Close our connection
        g_pDP->Close(0);
		// Release the DirectPlay object
        SAFE_RELEASE( g_pDP );
    }
	DeleteCriticalSection( &g_csModifyPlayer );
	// Uninitialize the COM library
	CoUninitialize();
	// Clear the Window class
	UnregisterClass( szWndClass, g_hInstance );
}

//-----------------------------------------------------------------------------
// Name: vPrepareGame(hWnd)
// Desc: Initializes game (rules, flow, etc.)
//-----------------------------------------------------------------------------
void vPrepareGame(HWND hWnd)
{
	int i;
	
	// Setup Communications System
	if( (hrInitializeDirectPlay(hWnd)) == -1 ) {
		MessageBox( hWnd, "DirectPlay Error", "Unable to initialize Direct Play.", MB_ICONERROR );
		vCleanup();
		exit(1);
	}
	// Initialize flags etc.
	bCueBallMove = FALSE;
	bCueMove = FALSE;
	bPerformShot = FALSE;
	sShotPower = MIN_POWER;
	bBirdView = FALSE;
	bShotDone = FALSE;
	bGameStart = TRUE;
	bGameOver = FALSE;
	bBallInHand = TRUE;
	bChooseBallOn = FALSE;
	bNextPlayer = FALSE;
	bChooseGameType = TRUE;
	bNetworkGame = FALSE;
	bHostGame = FALSE;
	bPacketsProcessed = TRUE;
	bScoreComputed = FALSE;
	bSendTurn = FALSE;
	bOtherPlayerLeft = FALSE;
	sprintf(szSystemMessage,"");
	sprintf(szTemporary,"");
	// Initialize Player info structure
	for (i = 0; i < MAX_PLAYERS; i++)
	{
		PlayerInfo[i].sBallOnValue = RED_VALUE;
		PlayerInfo[i].sScore = 0;
	}
	g_iCurrentPlayer = 0;
	// Set ball positions
	vSetOriginalBallPositions();
	// Set cue position & rotation
	cue.vSetPosition(&white.vecGetPosition());
	cue.vSetRotation(&ORIGINAL_ROT);
	// Cue collision detection
	vCueCollisionDetection();
	// Set camera rotation
	camera.vRotateCamera(&cue.vecGetRotation());
	// Initialize Stroke result structure
	vInitializeStrokeResult();
	// Compute remaining points on table
	vComputePointsOnTable();
}

//-----------------------------------------------------------------------------
// Name: vInitializeObjects()
// Desc: Initializes game object structures
//-----------------------------------------------------------------------------
void vInitializeObjects(void)
{
	int i;

	for( i = 0 ; i < MAX_PLAYERS ; i++ ) {
		memset(&PlayerInfo[i],0,sizeof(CLIENT_PLAYER_INFORMATION));
		// Initialize some members that don't need to be initialized over and over again
		PlayerInfo[i].bActive = 0;
		PlayerInfo[i].iPlayerID = i;
		sprintf(PlayerInfo[i].szPlayerName,"Player %d", i+1);
	}
	for( i = 0 ; i < MAX_TEXTURES ; i++ ) {
		g_pTexture[i] = NULL;
	}
	g_GameSounds.gsBallHit = NULL;
	g_GameSounds.gsBallInPocket = NULL;
}

//-----------------------------------------------------------------------------
// Name: vInitSoundEngine()
// Desc: Initializes sound system
//-----------------------------------------------------------------------------
void vInitSoundEngine(HWND hWnd)
{
	SoundEngine.hWnd = hWnd;
	if( !FAILED(SoundEngine.hrInitSoundSystem()) ) {
		// Ball hit sound
		g_GameSounds.gsBallHit = new GameSound;
		SoundEngine.hrLoadSound("Data\\Sound\\Ball_Hit.wav",g_GameSounds.gsBallHit);
		// Ball potted sound
		g_GameSounds.gsBallInPocket = new GameSound;
		SoundEngine.hrLoadSound("Data\\Sound\\Ball_Pocket.wav",g_GameSounds.gsBallInPocket);
	}
}

//-----------------------------------------------------------------------------
// Name: vUpdateObjects()
// Desc: Updates players position and settings, sends command packets
//-----------------------------------------------------------------------------
void vUpdateObjects(HWND hWnd)
{
	POINT			pointCursorPos;
	int				i;
	int				idx;
	int				idy;
	int				idz;
	short			sMaxPoints;
	short			sMaxPointCount;
	float			fMultiplyerX;
	float			fMultiplyerZ;
	bool			bAllRedsPotted = true;
	bool			bBallIn;
	void			*packet;
	PacketStroke	StrokeMsg;
	
	if ( !bShotDone ) {
		// Update cue ball position if it is being moved
		if ( bCueBallMove ) 
		{
			// Calculate how much the cursor moved
			GetCursorPos( &pointCursorPos );
			idx = pointCursorPos.x - pointCursorStart.x;
			idz = pointCursorStart.y - pointCursorPos.y;
			pointCursorStart = pointCursorPos; 
			// Calculate table/game screen size ratio
			fMultiplyerX = 177.8f / (float)g_Adapters[g_dwAdapter].d3ddmDesktop.Width;
			fMultiplyerZ = 356.9f / (float)g_Adapters[g_dwAdapter].d3ddmDesktop.Height;
			// Movement is dependant on cue rotation
			if ((cue.vecGetRotation().x >= 225.0f) && (cue.vecGetRotation().x < 315.0f))
				white.vUpdatePosition(&D3DXVECTOR3( -(float)idz * fMultiplyerZ,
													0.0f,												
													(float)idx * fMultiplyerX));
			else if ((cue.vecGetRotation().x >= 45.0f) && (cue.vecGetRotation().x < 135.0f))
				white.vUpdatePosition(&D3DXVECTOR3( (float)idz * fMultiplyerZ,
													0.0f,												
													-(float)idx * fMultiplyerX));
			else if ((cue.vecGetRotation().x >= 135.0f) && (cue.vecGetRotation().x < 225.0f))
				white.vUpdatePosition(&D3DXVECTOR3( -(float)idx * fMultiplyerX,
													0.0f,
													-(float)idz * fMultiplyerZ));
			else
				white.vUpdatePosition(&D3DXVECTOR3( (float)idx * fMultiplyerX,
													0.0f,
													(float)idz * fMultiplyerZ));
			// Check if cue ball is within limits
			table.vBaulkMovementCheck(&white);
			// Check cue ball - table collision
			table.vBallCollisionDetection(&white, false);
			// Check if cue ball collides with another ball
			white.vResetList();
			white.vAddToList(&black);
			white.vAddToList(&pink);
			white.vAddToList(&blue);
			white.vAddToList(&brown);
			white.vAddToList(&green);
			white.vAddToList(&yellow);
			for( i = 0 ; i < REDS_NUM ; i++ )
				white.vAddToList(&reds[i]);
			white.vBallsCollisionDetection();
			// Update cue position
			cue.vSetPosition( &white.vecGetPosition() );
			// Collision detection for cue
			vCueCollisionDetection();
			// Rotate camera
			camera.vRotateCamera(&cue.vecGetRotation());
		}
		// In case cue is being moved around
		else if ( bCueMove ) {
			// Get movement size (delta)
			GetCursorPos( &pointCursorPos );
			idx = pointCursorPos.x - pointCursorStart.x;
			idy = pointCursorStart.y - pointCursorPos.y;
			pointCursorStart = pointCursorPos; 
			// Update cue position
			cue.vRotatePosition(idx, idy);		
			// Cue collision detection
			vCueCollisionDetection();
			// Rotate camera
			camera.vRotateCamera(&cue.vecGetRotation());
		}
		// Do shot
		else if ( bPerformShot ) {
			// Perform shot
			cue.vPerformShot(&white,sShotPower);
			// If network game send stroke info to other players
			if (bNetworkGame)
			{
				// Create package with stroke info
				StrokeMsg.dwSize = sizeof(PacketStroke);
				StrokeMsg.dwType = PACKET_TYPE_STROKE;
				StrokeMsg.iPlayerID = g_dpnidLocalPlayer;
				StrokeMsg.fCuePosX = cue.vecGetPosition().x;
				StrokeMsg.fCuePosZ = cue.vecGetPosition().z;
				StrokeMsg.fCueRotX = cue.vecGetRotation().x;
				StrokeMsg.fCueRotY = cue.vecGetRotation().y;
				StrokeMsg.sPower = sShotPower;
				StrokeMsg.sBallOn = PlayerInfo[g_iCurrentPlayer].sBallOnValue;
				// Convert the packet to a void stream
				packet = (VOID*)&StrokeMsg;
				// Send the chat packet
				hrSendPeerMessage(-1,PACKET_TYPE_STROKE,packet);	
				bPacketsProcessed = FALSE;
			}
			// Set flags
			bBallInHand = FALSE;
			bPerformShot = FALSE;
			bShotDone = TRUE;
			bScoreComputed = FALSE;
		}
	}
	// Balls are moving (a stroke was made)
	else {
		EnterCriticalSection( &g_csModifyPlayer );
		//-----------------------------------------------------------------------------------
		// Ball movement and collision detection code
		//-----------------------------------------------------------------------------------
		// Prepare data for collision detection
		vPrepareCollisionDetectionData();
		// Move balls (also checks for collision between balls)
		if (black.vAdvance() >= 0)
			SoundEngine.hrPlaySound(g_GameSounds.gsBallHit);
		if (pink.vAdvance() >= 0)
			SoundEngine.hrPlaySound(g_GameSounds.gsBallHit);
		if (blue.vAdvance() >= 0)
			SoundEngine.hrPlaySound(g_GameSounds.gsBallHit);
		if (brown.vAdvance() >= 0)
			SoundEngine.hrPlaySound(g_GameSounds.gsBallHit);
		if (green.vAdvance() >= 0)
			SoundEngine.hrPlaySound(g_GameSounds.gsBallHit);
		if (yellow.vAdvance() >= 0)
			SoundEngine.hrPlaySound(g_GameSounds.gsBallHit);
		// Result only needed if cue ball did not hit any other ball yet
		if (!StrokeResult.bBallOnHitSet)
		{
			StrokeResult.sValueHit = white.vAdvance();
			if (StrokeResult.sValueHit >= 0)
				SoundEngine.hrPlaySound(g_GameSounds.gsBallHit);
			// True if ball on was hit
			StrokeResult.bBallOnHit = (StrokeResult.sValueHit == PlayerInfo[g_iCurrentPlayer].sBallOnValue);
			StrokeResult.bBallOnHitSet = (StrokeResult.sValueHit >= 0);
		}
		// just move
		else
		{
			if (white.vAdvance() >= 0)
				SoundEngine.hrPlaySound(g_GameSounds.gsBallHit);
		}
		for ( i = 0 ; i < REDS_NUM ; i++ ) {
			if (reds[i].vAdvance() >= 0)
				SoundEngine.hrPlaySound(g_GameSounds.gsBallHit);
		}

		//-----------------------------------------------------------------------------------
		// Game rules code
		//-----------------------------------------------------------------------------------
		// Ball - table collision detection && game rules
		if (!bAnyBallMoving() && !StrokeResult.bBallOnHit && StrokeResult.bBallOnHitSet)
		{
			// Red ball first hit when a color is ball on
			if (StrokeResult.sValueHit == RED_VALUE)
			{
	 			StrokeResult.sPenaltyPoints += MAXIMUM_PENALTY;
			}
			// Other ball than ball on hit
			else
			{
				if (PlayerInfo[g_iCurrentPlayer].sBallOnValue > StrokeResult.sValueHit)
					StrokeResult.sPenaltyPoints += (PlayerInfo[g_iCurrentPlayer].sBallOnValue > MINIMUM_PENALTY) ? PlayerInfo[g_iCurrentPlayer].sBallOnValue : MINIMUM_PENALTY;
				else
					StrokeResult.sPenaltyPoints += (StrokeResult.sValueHit > MINIMUM_PENALTY) ? StrokeResult.sValueHit : MINIMUM_PENALTY;
			}
			StrokeResult.bStrokeOK = false;
		}
		// Cue ball missed all object balls
		else if (!bAnyBallMoving() && !StrokeResult.bBallOnHitSet)
		{
			StrokeResult.sPenaltyPoints += (PlayerInfo[g_iCurrentPlayer].sBallOnValue > MINIMUM_PENALTY) ? PlayerInfo[g_iCurrentPlayer].sBallOnValue : MINIMUM_PENALTY;
			StrokeResult.bStrokeOK = false;
		}

		// Reds
		for ( i = 0 ; i < REDS_NUM ; i++ ) 
		{
			bBallIn = table.vBallCollisionDetection(&reds[i]);
			bAllRedsPotted = bAllRedsPotted && reds[i].bGetPotted();
			if (PlayerInfo[g_iCurrentPlayer].sBallOnValue != RED_VALUE)
			{
				// Red is not ball on but was potted
				if (bBallIn)
				{
					SoundEngine.hrPlaySound(g_GameSounds.gsBallInPocket);
					StrokeResult.sPenaltyPoints += (PlayerInfo[g_iCurrentPlayer].sBallOnValue > MINIMUM_PENALTY) ? PlayerInfo[g_iCurrentPlayer].sBallOnValue : MINIMUM_PENALTY;
					StrokeResult.bStrokeOK = false;
				}
			}
			else 
			{
				// Red is ball on and was potted
				if (bBallIn)
				{
					SoundEngine.hrPlaySound(g_GameSounds.gsBallInPocket);
					if (StrokeResult.sPoints == 0)
						StrokeResult.sPoints = RED_VALUE;
					else 
					{
						StrokeResult.sPenaltyPoints += MAXIMUM_PENALTY;
						StrokeResult.bStrokeOK = false;
					}
				}
			}
		}

		// Black
		bBallIn = table.vBallCollisionDetection(&black);
		if (PlayerInfo[g_iCurrentPlayer].sBallOnValue != BLACK_VALUE)
		{
			// Black is not ball on but was potted
			if (bBallIn)
			{
				SoundEngine.hrPlaySound(g_GameSounds.gsBallInPocket);
				StrokeResult.sPenaltyPoints += BLACK_VALUE;
				StrokeResult.bStrokeOK = false;
				black.vSetReposition(true);
			}
		}
		else 
		{
			// Black is ball on and was sucessfully potted
			if (bBallIn)
			{
				SoundEngine.hrPlaySound(g_GameSounds.gsBallInPocket);
				StrokeResult.sPoints = BLACK_VALUE;
				black.vSetReposition(!bAllRedsPotted);
			}
		}

		// Pink
		bBallIn = table.vBallCollisionDetection(&pink);
		if (PlayerInfo[g_iCurrentPlayer].sBallOnValue != PINK_VALUE)
		{
			// Pink is not ball on but was potted
			if (bBallIn)
			{
				SoundEngine.hrPlaySound(g_GameSounds.gsBallInPocket);
				StrokeResult.sPenaltyPoints += (PlayerInfo[g_iCurrentPlayer].sBallOnValue > PINK_VALUE) ? PlayerInfo[g_iCurrentPlayer].sBallOnValue : PINK_VALUE;
				StrokeResult.bStrokeOK = false;
				pink.vSetReposition(true);
			}
		}
		else 
		{
			// Pink is ball on and was sucessfully potted
			if (bBallIn)
			{
				SoundEngine.hrPlaySound(g_GameSounds.gsBallInPocket);
				StrokeResult.sPoints = PINK_VALUE;
				pink.vSetReposition(!bAllRedsPotted);
			}
		}

		// Blue
		bBallIn = table.vBallCollisionDetection(&blue);
		if (PlayerInfo[g_iCurrentPlayer].sBallOnValue != BLUE_VALUE)
		{
			// Blue is not ball on but was potted
			if (bBallIn)
			{
				SoundEngine.hrPlaySound(g_GameSounds.gsBallInPocket);
				StrokeResult.sPenaltyPoints += (PlayerInfo[g_iCurrentPlayer].sBallOnValue > BLUE_VALUE) ? PlayerInfo[g_iCurrentPlayer].sBallOnValue : BLUE_VALUE;
				StrokeResult.bStrokeOK = false;
				blue.vSetReposition(true);
			}
		}
		else 
		{
			// Blue is ball on and was sucessfully potted
			if (bBallIn)
			{
				SoundEngine.hrPlaySound(g_GameSounds.gsBallInPocket);
				StrokeResult.sPoints = BLUE_VALUE;
				blue.vSetReposition(!bAllRedsPotted);
			}
		}
		
		// Brown
		bBallIn = table.vBallCollisionDetection(&brown);
		if (PlayerInfo[g_iCurrentPlayer].sBallOnValue != BROWN_VALUE)
		{
			// Brown is not ball on but was potted
			if (bBallIn)
			{
				SoundEngine.hrPlaySound(g_GameSounds.gsBallInPocket);
				StrokeResult.sPenaltyPoints += (PlayerInfo[g_iCurrentPlayer].sBallOnValue > BROWN_VALUE) ? PlayerInfo[g_iCurrentPlayer].sBallOnValue : BROWN_VALUE;
				StrokeResult.bStrokeOK = false;
				brown.vSetReposition(true);
			}
		}
		else 
		{
			// Brown is ball on and was sucessfully potted
			if (bBallIn)
			{
				SoundEngine.hrPlaySound(g_GameSounds.gsBallInPocket);
				StrokeResult.sPoints = BROWN_VALUE;
				brown.vSetReposition(!bAllRedsPotted);
			}
		}

		// Green
		bBallIn = table.vBallCollisionDetection(&green);
		if (PlayerInfo[g_iCurrentPlayer].sBallOnValue != GREEN_VALUE)
		{
			// Green is not ball on but was potted
			if (bBallIn)
			{
				SoundEngine.hrPlaySound(g_GameSounds.gsBallInPocket);
				StrokeResult.sPenaltyPoints += (PlayerInfo[g_iCurrentPlayer].sBallOnValue > MINIMUM_PENALTY) ? PlayerInfo[g_iCurrentPlayer].sBallOnValue : MINIMUM_PENALTY;
				StrokeResult.bStrokeOK = false;
				green.vSetReposition(true);
			}
		}
		else 
		{
			// Green is ball on and was sucessfully potted
			if (bBallIn)
			{
				SoundEngine.hrPlaySound(g_GameSounds.gsBallInPocket);
				StrokeResult.sPoints = GREEN_VALUE;
				green.vSetReposition(!bAllRedsPotted);
			}
		}

		// Yellow
		bBallIn = table.vBallCollisionDetection(&yellow);
		if (PlayerInfo[g_iCurrentPlayer].sBallOnValue != YELLOW_VALUE)
		{
			// Yellow is not ball on but was potted
			if (bBallIn)
			{
				SoundEngine.hrPlaySound(g_GameSounds.gsBallInPocket);
				StrokeResult.sPenaltyPoints += (PlayerInfo[g_iCurrentPlayer].sBallOnValue > MINIMUM_PENALTY) ? PlayerInfo[g_iCurrentPlayer].sBallOnValue : MINIMUM_PENALTY;
				StrokeResult.bStrokeOK = false;
				yellow.vSetReposition(true);
			}
		}
		else 
		{
			// Yellow is ball on and was sucessfully potted
			if (bBallIn)
			{
				SoundEngine.hrPlaySound(g_GameSounds.gsBallInPocket);
				StrokeResult.sPoints = YELLOW_VALUE;
				yellow.vSetReposition(!bAllRedsPotted);
			}
		}

		// White
		bBallIn = table.vBallCollisionDetection(&white);
		if (bBallIn)
		{
			SoundEngine.hrPlaySound(g_GameSounds.gsBallInPocket);
			StrokeResult.sPenaltyPoints += (PlayerInfo[g_iCurrentPlayer].sBallOnValue > MINIMUM_PENALTY) ? PlayerInfo[g_iCurrentPlayer].sBallOnValue : MINIMUM_PENALTY;
			StrokeResult.bStrokeOK = false;
			bBallInHand = TRUE;
			white.vSetReposition(true);
		}

		//-----------------------------------------------------------------------------------
		// End of stroke code (compute points, prepare for next stroke)
		//-----------------------------------------------------------------------------------
		// All balls stopped moving, we are current player and this if clause was not entered yet
		if (!bAnyBallMoving() && (g_iMyPlayerId == g_iCurrentPlayer) && !bScoreComputed)
		{
			bShotDone = FALSE;
			sShotPower = MIN_POWER;
			// Reposition illegaly potted balls
			black.vReposition();
			pink.vReposition();
			blue.vReposition();
			brown.vReposition();
			green.vReposition();
			yellow.vReposition();
			white.vReposition();
			// Compute remaining points on table
			vComputePointsOnTable();	
			// Final check if stroke was OK
			if (StrokeResult.bStrokeOK && !bGameStart)
				StrokeResult.bStrokeOK = (StrokeResult.sPoints == PlayerInfo[g_iCurrentPlayer].sBallOnValue);
			// Aplly any negative and positive points
			PlayerInfo[g_iCurrentPlayer].sScore += StrokeResult.sPoints;
			if (!StrokeResult.bStrokeOK)
			{
				// add penalties to other players scores
				for (i = 0; i < MAX_PLAYERS; i++)
				{
					if (i != g_iCurrentPlayer)
						PlayerInfo[i].sScore += StrokeResult.sPenaltyPoints;
				}
				// Check if game over
				vCheckGameOver();
				if (!bGameOver)
				{
					// Choose next player
					g_iCurrentPlayer++;
					if (g_iCurrentPlayer >= MAX_PLAYERS)
						g_iCurrentPlayer = 0;
					// non network game specific code
					if (!bNetworkGame)
					{
						bNextPlayer = TRUE;			
						g_iMyPlayerId = g_iCurrentPlayer;
					}
					// Set ball on for next player
					if (!bAllRedsPotted)
						// reds are still on the table
						PlayerInfo[g_iCurrentPlayer].sBallOnValue = RED_VALUE;
					else
					{
						// as last stroke was unsucessfull there must be at least one color on table
						if (!yellow.bGetPotted())					
							PlayerInfo[g_iCurrentPlayer].sBallOnValue = YELLOW_VALUE;
						else if (!green.bGetPotted())
							PlayerInfo[g_iCurrentPlayer].sBallOnValue = GREEN_VALUE;
						else if (!brown.bGetPotted())
							PlayerInfo[g_iCurrentPlayer].sBallOnValue = BROWN_VALUE;
						else if (!blue.bGetPotted())
							PlayerInfo[g_iCurrentPlayer].sBallOnValue = BLUE_VALUE;
						else if (!pink.bGetPotted())
							PlayerInfo[g_iCurrentPlayer].sBallOnValue = PINK_VALUE;
						else
							PlayerInfo[g_iCurrentPlayer].sBallOnValue = BLACK_VALUE;
					}
					// Notify others who's turn it is
					if (bNetworkGame)
						bSendTurn = TRUE;
				}
			}
			else  // current player continues
			{
				// Set next ball on
				if (PlayerInfo[g_iCurrentPlayer].sBallOnValue == RED_VALUE)
					if (!bAllRedsPotted)
					{
						if (g_iMyPlayerId == g_iCurrentPlayer)
							bChooseBallOn = TRUE;
					}
					else
					{
						// as the last red was just potted there must be at least one color on table
						if (!yellow.bGetPotted())					
							PlayerInfo[g_iCurrentPlayer].sBallOnValue = YELLOW_VALUE;
						else if (!green.bGetPotted())
							PlayerInfo[g_iCurrentPlayer].sBallOnValue = GREEN_VALUE;
						else if (!brown.bGetPotted())
							PlayerInfo[g_iCurrentPlayer].sBallOnValue = BROWN_VALUE;
						else if (!blue.bGetPotted())
							PlayerInfo[g_iCurrentPlayer].sBallOnValue = BLUE_VALUE;
						else if (!pink.bGetPotted())
							PlayerInfo[g_iCurrentPlayer].sBallOnValue = PINK_VALUE;
						else
							PlayerInfo[g_iCurrentPlayer].sBallOnValue = BLACK_VALUE;
					}
				else
				{
					if (!bAllRedsPotted)
						PlayerInfo[g_iCurrentPlayer].sBallOnValue = RED_VALUE;
					else
					{
						// only colors (if any) can be still on table
						if (!yellow.bGetPotted())					
							PlayerInfo[g_iCurrentPlayer].sBallOnValue = YELLOW_VALUE;
						else if (!green.bGetPotted())
							PlayerInfo[g_iCurrentPlayer].sBallOnValue = GREEN_VALUE;
						else if (!brown.bGetPotted())
							PlayerInfo[g_iCurrentPlayer].sBallOnValue = BROWN_VALUE;
						else if (!blue.bGetPotted())
							PlayerInfo[g_iCurrentPlayer].sBallOnValue = BLUE_VALUE;
						else if (!pink.bGetPotted())
							PlayerInfo[g_iCurrentPlayer].sBallOnValue = PINK_VALUE;
						else if (!black.bGetPotted())
							PlayerInfo[g_iCurrentPlayer].sBallOnValue = BLACK_VALUE;
						else
						{
							// Check if it is a tie
							sMaxPoints = 0;
							sMaxPointCount = 0;
							for ( i = 0; i < MAX_PLAYERS; i++)
							{
								if (PlayerInfo[i].sScore == sMaxPoints)
									sMaxPointCount++;
								else if (PlayerInfo[i].sScore > sMaxPoints)
								{
									sMaxPoints = PlayerInfo[i].sScore;
									sMaxPointCount = 1;
								}
							}
							// If it is a tie respot the black and continue
							if (sMaxPointCount > 1)
							{
								black.vSetReposition(true);
								black.vReposition();
								PlayerInfo[g_iCurrentPlayer].sBallOnValue = BLACK_VALUE;
								vComputePointsOnTable();
							}
							else
								bGameOver = TRUE;
						}
					}
				}
			}
			// If network game send final ball positions to other players (to avoid errors due to computational errors)
			if (bNetworkGame)
			{
				// Send player scores
				for (i = 0; i < MAX_PLAYERS; i++)
					vSendScoreMsg(i);				
				// And now ball positions, first reds
				for (i = 0; i < REDS_NUM; i++)
					vSendBallPosMsg(i, &reds[i].vecGetPosition(), reds[i].bGetPotted());
				// black
				vSendBallPosMsg(REDS_NUM, &black.vecGetPosition(), black.bGetPotted());
				// pink
				vSendBallPosMsg(REDS_NUM + 1, &pink.vecGetPosition(), pink.bGetPotted());
				// blue
				vSendBallPosMsg(REDS_NUM + 2, &blue.vecGetPosition(), blue.bGetPotted());
				// brown
				vSendBallPosMsg(REDS_NUM + 3, &brown.vecGetPosition(), brown.bGetPotted());
				// green
				vSendBallPosMsg(REDS_NUM + 4, &green.vecGetPosition(), green.bGetPotted());
				// yellow 
				vSendBallPosMsg(REDS_NUM + 5, &yellow.vecGetPosition(), yellow.bGetPotted());
				// white
				vSendBallPosMsg(REDS_NUM + 6, &white.vecGetPosition(), white.bGetPotted());
			}
			if (bGameStart)
				bGameStart = FALSE;
			// Initalize Stroke data structure for next stroke
			vInitializeStrokeResult();	
			// Reset Cue position
			cue.vSetPosition(&white.vecGetPosition());
			// Cue collision detection
			vCueCollisionDetection();
			// Reset camera
			camera.vRotateCamera(&cue.vecGetRotation());
			bScoreComputed = TRUE;
		}
		// Ball stopped moving, we are not the current player and this if clause was not entered yet
		else if (!bAnyBallMoving() && (iBallPosCount == TOTAL_BALLS) && !bScoreComputed)
		{
			// Process ball position data
			for (i = 0; i < REDS_NUM; i++)
			{
				reds[i].vSetPosition(&arrBallPositions[i]);
				reds[i].vSetPotted(arrBallPotted[i]);
			}
			// black
			black.vSetPosition(&arrBallPositions[REDS_NUM]);
			black.vSetPotted(arrBallPotted[REDS_NUM]);
			black.vSetReposition(false);
			// pink
			pink.vSetPosition(&arrBallPositions[REDS_NUM + 1]);
			pink.vSetPotted(arrBallPotted[REDS_NUM + 1]);
			pink.vSetReposition(false);
			// blue
			blue.vSetPosition(&arrBallPositions[REDS_NUM + 2]);
			blue.vSetPotted(arrBallPotted[REDS_NUM + 2]);
			blue.vSetReposition(false);
			// brown
			brown.vSetPosition(&arrBallPositions[REDS_NUM + 3]);
			brown.vSetPotted(arrBallPotted[REDS_NUM + 3]);
			brown.vSetReposition(false);
			// green
			green.vSetPosition(&arrBallPositions[REDS_NUM + 4]);
			green.vSetPotted(arrBallPotted[REDS_NUM + 4]);
			green.vSetReposition(false);
			// yellow 
			yellow.vSetPosition(&arrBallPositions[REDS_NUM + 5]);
			yellow.vSetPotted(arrBallPotted[REDS_NUM + 5]);
			yellow.vSetReposition(false);
			// white
			white.vSetPosition(&arrBallPositions[REDS_NUM + 6]);
			white.vSetPotted(arrBallPotted[REDS_NUM + 6]);
			white.vSetReposition(false);
			iBallPosCount = 0;
			bShotDone = FALSE;
			sShotPower = MIN_POWER;
			if (bGameStart)
				bGameStart = FALSE;
			// Compute remaining points on table
			vComputePointsOnTable();
			// Initalize Stroke data structure for next stroke
			vInitializeStrokeResult();	
			// Reset Cue position
			cue.vSetPosition(&white.vecGetPosition());
			// Cue collision detection
			vCueCollisionDetection();
			// Reset camera
			camera.vRotateCamera(&cue.vecGetRotation());
			vSendOKMsg();
			bScoreComputed = TRUE;
		}
		LeaveCriticalSection( &g_csModifyPlayer );
	}
}

//-----------------------------------------------------------------------------
// Name: vCueCollisionDetection()
// Desc: Do cue collision detection and rotation check
//-----------------------------------------------------------------------------
void vCueCollisionDetection(void)
{
	int i;

	// Table - cue collision detection
	table.vCueCollisionDetection(&cue);
	// Cue - balls collision detection
	cue.vBallCollisionDetection(&black);
	cue.vBallCollisionDetection(&pink);
	cue.vBallCollisionDetection(&blue);
	cue.vBallCollisionDetection(&brown);
	cue.vBallCollisionDetection(&green);
	cue.vBallCollisionDetection(&yellow);
	for( i = 0 ; i < REDS_NUM ; i++ )
		cue.vBallCollisionDetection(&reds[i]);
	// Check maximum rotation
	cue.vCheckRotationUp();
}

//-----------------------------------------------------------------------------
// Name: vCheckGameOver()
// Desc: Determines if one player has enough points to win the game
//-----------------------------------------------------------------------------
void vCheckGameOver(void)
{
	int i;
	int j;

	bGameOver = FALSE;
	for ( i = 0; i < MAX_PLAYERS; i++)
	{
		bGameOver = TRUE;
		for ( j = 0; j < MAX_PLAYERS; j++)
		{				
			// Compare points between players
			if (i != j)
				bGameOver = bGameOver && ((PlayerInfo[i].sScore - PlayerInfo[j].sScore) > sPointsOnTable);
		}
		if (bGameOver)
		{
			g_iCurrentPlayer = i;
			break;
		}
	}
}

//-----------------------------------------------------------------------------
// Name: vComputePointsOnTable()
// Desc: Computes remaining points on table
//-----------------------------------------------------------------------------
void vComputePointsOnTable(void)
{
	int		i;
	short	sRedsNum;
	short	sPointsColor;

	sRedsNum = 0;
	for ( i = 0 ; i < REDS_NUM ; i++ ) 
	{
		if (!reds[i].bGetPotted())
			sRedsNum++;
	}
	sPointsColor = 0;
	if (!black.bGetPotted())
		sPointsColor += BLACK_VALUE;
	if (!pink.bGetPotted())
		sPointsColor += PINK_VALUE;
	if (!blue.bGetPotted())
		sPointsColor += BLUE_VALUE;
	if (!brown.bGetPotted())
		sPointsColor += BROWN_VALUE;
	if (!green.bGetPotted())
		sPointsColor += GREEN_VALUE;
	if (!yellow.bGetPotted())
		sPointsColor += YELLOW_VALUE;
	// Maximum possible points on table
	sPointsOnTable = sRedsNum * (RED_VALUE + BLACK_VALUE) + sPointsColor;
}

//-----------------------------------------------------------------------------
// Name: vInitializeStrokeResult()
// Desc: Initializes structure that saves result of current stroke
//-----------------------------------------------------------------------------
void vInitializeStrokeResult(void)
{
	StrokeResult.bBallOnHit = false;
	StrokeResult.bBallOnHitSet = false;
	StrokeResult.bStrokeOK = true;
	StrokeResult.sPenaltyPoints = 0;
	StrokeResult.sPoints = 0;
	StrokeResult.sValueHit = 0;
}

//-----------------------------------------------------------------------------
// Name: vPrepareCollisionDetectionData()
// Desc: Prepare data for collision detection
//-----------------------------------------------------------------------------
void vPrepareCollisionDetectionData(void)
{
	int		i;
	int		j;

	// Create sorted neighbour list for each ball
	// Black
	black.vResetList();
	black.vAddToList(&pink);
	black.vAddToList(&blue);
	black.vAddToList(&brown);
	black.vAddToList(&green);
	black.vAddToList(&yellow);
	black.vAddToList(&white);
	// Pink
	pink.vResetList();
	pink.vAddToList(&black);
	pink.vAddToList(&blue);
	pink.vAddToList(&brown);
	pink.vAddToList(&green);
	pink.vAddToList(&yellow);
	pink.vAddToList(&white);
	// Blue
	blue.vResetList();
	blue.vAddToList(&black);
	blue.vAddToList(&pink);
	blue.vAddToList(&brown);
	blue.vAddToList(&green);
	blue.vAddToList(&yellow);
	blue.vAddToList(&white);
	// Brown
	brown.vResetList();
	brown.vAddToList(&black);
	brown.vAddToList(&pink);
	brown.vAddToList(&blue);
	brown.vAddToList(&green);
	brown.vAddToList(&yellow);
	brown.vAddToList(&white);
	// Green
	green.vResetList();
	green.vAddToList(&black);
	green.vAddToList(&pink);
	green.vAddToList(&blue);
	green.vAddToList(&brown);
	green.vAddToList(&yellow);
	green.vAddToList(&white);
	// Yellow
	yellow.vResetList();
	yellow.vAddToList(&black);
	yellow.vAddToList(&pink);
	yellow.vAddToList(&blue);
	yellow.vAddToList(&brown);
	yellow.vAddToList(&green);
	yellow.vAddToList(&white);
	// White
	white.vResetList();
	white.vAddToList(&black);
	white.vAddToList(&pink);
	white.vAddToList(&blue);
	white.vAddToList(&brown);
	white.vAddToList(&green);
	white.vAddToList(&yellow);
	// Reds (also adds reds to each color's neighbour list)
	for (i = 0; i < REDS_NUM; i++)
	{
		black.vAddToList(&reds[i]);
		pink.vAddToList(&reds[i]);
		blue.vAddToList(&reds[i]);
		brown.vAddToList(&reds[i]);
		green.vAddToList(&reds[i]);
		yellow.vAddToList(&reds[i]);
		white.vAddToList(&reds[i]);	
		reds[i].vResetList();
		reds[i].vAddToList(&black);
		reds[i].vAddToList(&pink);
		reds[i].vAddToList(&blue);
		reds[i].vAddToList(&brown);
		reds[i].vAddToList(&green);
		reds[i].vAddToList(&yellow);
		reds[i].vAddToList(&white);
		for (j = 0; j < REDS_NUM; j++)
		{
			if (i != j)
				reds[i].vAddToList(&reds[j]);
		}
	}
}

//-----------------------------------------------------------------------------
// Name: bAnyBallMoving()
// Desc: Check if any ball is still moving
//-----------------------------------------------------------------------------
bool bAnyBallMoving(void)
{
	int		i;
	bool	bAnyRedMoving = false;

	// Is any red moving?
	for( i = 0 ; i < REDS_NUM ; i++ )
	{
		bAnyRedMoving = bAnyRedMoving || reds[i].bGetIsBallMoving();
		if (bAnyRedMoving)
			break;
	}
	// Is any red or color moving?
	return (bAnyRedMoving || black.bGetIsBallMoving() || pink.bGetIsBallMoving() ||
			blue.bGetIsBallMoving() || brown.bGetIsBallMoving() || green.bGetIsBallMoving() ||
			yellow.bGetIsBallMoving() || white.bGetIsBallMoving());
}

//-----------------------------------------------------------------------------
// Name: vSendBallPosMsg(i,*pVec)
// Desc: Send ball position packet
//-----------------------------------------------------------------------------
void vSendBallPosMsg(const int i, D3DXVECTOR3 *pVec, const bool bPotted)
{
	void			*packet;
	PacketBallPos	BallPosMsg;

	BallPosMsg.dwSize = sizeof(PacketBallPos);
	BallPosMsg.dwType = PACKET_TYPE_BALLPOS;
	BallPosMsg.iPlayerID = PlayerInfo[g_iCurrentPlayer].iPlayerID;
	BallPosMsg.iBallNo = i;
	BallPosMsg.fX = pVec->x;
	BallPosMsg.fZ = pVec->z;
	BallPosMsg.bPotted = bPotted;
	// Convert the packet to a void stream
	packet = (VOID*)&BallPosMsg;
	// Send the chat packet
	hrSendPeerMessage(-1,PACKET_TYPE_BALLPOS,packet);
}

//-----------------------------------------------------------------------------
// Name: vSendScoreMsg(i)
// Desc: Send score packet
//-----------------------------------------------------------------------------
void vSendScoreMsg(const int i)
{
	void			*packet;
	PacketScore		ScoreMsg;

	ScoreMsg.dwSize = sizeof(PacketScore);
	ScoreMsg.dwType = PACKET_TYPE_SCORE;
	ScoreMsg.iPlayerID = PlayerInfo[i].iPlayerID;
	ScoreMsg.sScore = PlayerInfo[i].sScore;
	// Convert the packet to a void stream
	packet = (VOID*)&ScoreMsg;
	// Send the chat packet
	hrSendPeerMessage(-1,PACKET_TYPE_SCORE,packet);
}

//-----------------------------------------------------------------------------
// Name: vSendOKMsg()
// Desc: Send ready message
//-----------------------------------------------------------------------------
void vSendOKMsg(void)
{
	void			*packet;
	PacketOK		OKMsg;

	OKMsg.dwSize = sizeof(PacketOK);
	OKMsg.dwType = PACKET_TYPE_OK;
	OKMsg.iPlayerID = PlayerInfo[g_iCurrentPlayer].iPlayerID;
	// Convert the packet to a void stream
	packet = (VOID*)&OKMsg;
	// Send the chat packet
	hrSendPeerMessage(-1,PACKET_TYPE_OK,packet);
}

//-----------------------------------------------------------------------------
// Name: vSendGameOverMsg()
// Desc: Send game over message
//-----------------------------------------------------------------------------
void vSendGameOverMsg(void)
{
	void			*packet;
	PacketGameOver	GameOverMsg;

	GameOverMsg.dwSize = sizeof(PacketGameOver);
	GameOverMsg.dwType = PACKET_TYPE_GAMEOVER;
	GameOverMsg.iPlayerID = PlayerInfo[g_iCurrentPlayer].iPlayerID;
	// Convert the packet to a void stream
	packet = (VOID*)&GameOverMsg;
	// Send the chat packet
	hrSendPeerMessage(-1,PACKET_TYPE_GAMEOVER,packet);
	// Now we can terminate session
	if (bHostGame)
		g_pDP->TerminateSession(NULL,0,0);
}

//-----------------------------------------------------------------------------
//
//
//
//							DIRECT 3D FUNCTIONS
//
//
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Name: hrInit3D()
// Desc: Initializes Direct 3D
//-----------------------------------------------------------------------------
HRESULT hrInit3D( HWND hWnd ) 
{
	D3DPRESENT_PARAMETERS	d3dpp; 
	HRESULT					hResult;
	RECT					rcWindowBounds;    // Saved window bounds for mode switches
	RECT					rcWindowClient;    // Saved client area size for mode switches
	D3DAdapterInfo			*pAdapterInfo = &g_Adapters[g_dwAdapter];
	D3DDeviceInfo			*pDeviceInfo  = &pAdapterInfo->devices[pAdapterInfo->dwCurrentDevice];
	D3DModeInfo				*pModeInfo    = &pDeviceInfo->modes[pDeviceInfo->dwCurrentMode];
	
	// Create the D3D object.
	if( NULL == ( pD3D = Direct3DCreate8( D3D_SDK_VERSION ) ) ) {
		MessageBox( hWnd, "Direct3D Error", "Unable to initialize Direct3D, please check version.", MB_ICONERROR );
		return(E_FAIL);
	}
	
	// Build a list of available devices
	hResult = hrBuildDeviceList(hWnd);
	if( hResult == VIDEODEVICEERROR_NOCOMPATIBLE ) {
		MessageBox( hWnd, "Adapter Error", "No compatible 3D adapters found.", MB_ICONERROR );
		return(E_FAIL);
	}
	else if( hResult == VIDEODEVICEERROR_NOWINDOWED ) {
		MessageBox( hWnd, "Adapter Error", "No available windowable 3D adapter found, please switch to 16 or 32 bit mode.", MB_ICONERROR );
		return(E_FAIL);
	}
	
	// Get the size of the window
	GetWindowRect( hWnd, &rcWindowBounds );
	GetClientRect( hWnd, &rcWindowClient );
	
	// Set to fullscreen
	if( !g_bWindowed ) 
		SetWindowLong( hWnd, GWL_STYLE, WS_POPUP|WS_SYSMENU|WS_VISIBLE );
	
	// Set temp pointers
	pAdapterInfo = &g_Adapters[g_dwAdapter];
	pDeviceInfo  = &pAdapterInfo->devices[pAdapterInfo->dwCurrentDevice];
	pModeInfo    = &pDeviceInfo->modes[pDeviceInfo->dwCurrentMode];

	// Set up the presentation parameters
	ZeroMemory( &d3dpp, sizeof(d3dpp) );
	d3dpp.Windowed					= pDeviceInfo->bWindowed;
	d3dpp.BackBufferCount			= 1;
	d3dpp.MultiSampleType			= pDeviceInfo->MultiSampleType;
	d3dpp.SwapEffect				= D3DSWAPEFFECT_FLIP;
	d3dpp.EnableAutoDepthStencil	= g_bUseDepthBuffer;
	d3dpp.AutoDepthStencilFormat	= pModeInfo->DepthStencilFormat;
	d3dpp.hDeviceWindow				= hWnd;
	
	// Run this code if in windowed mode
	if( g_bWindowed ) {
		d3dpp.BackBufferWidth  = (rcWindowClient.right - rcWindowClient.left);
		d3dpp.BackBufferHeight = (rcWindowClient.bottom - rcWindowClient.top);
		d3dpp.BackBufferFormat = pAdapterInfo->d3ddmDesktop.Format;
	}
	// Run this code if in fullscreen mode
	else {
		d3dpp.BackBufferWidth  = pModeInfo->Width;
		d3dpp.BackBufferHeight = pModeInfo->Height;
		d3dpp.BackBufferFormat = pModeInfo->Format;
	}
	
	// Create the device
	hResult = pD3D->CreateDevice( g_dwAdapter, pDeviceInfo->DeviceType, hWnd, pModeInfo->dwBehavior, &d3dpp, &pd3dDevice );
	if( !SUCCEEDED(hResult) ) {
		MessageBox( hWnd, "Device Error", "Failed to create display device!", MB_ICONERROR );
		return(E_FAIL);
	}
	
	// Turn on the zbuffer
	pd3dDevice->SetRenderState( D3DRS_ZENABLE, TRUE );
	// Turn on ambient lighting 
	pd3dDevice->SetRenderState( D3DRS_AMBIENT, 0x00101010 );
	
	return S_OK;
}

//-----------------------------------------------------------------------------
// Name: vUpdateCamera()
// Desc: Updates the camera position in the 3D scene
//-----------------------------------------------------------------------------
void vUpdateCamera(void)
{
	// Tell the device to use the camera view for the viewport
	pd3dDevice->SetTransform( D3DTS_VIEW, &camera.matGetCameraView(bBirdView || bShotDone, &white.vecGetPosition()) );
}

//-----------------------------------------------------------------------------
// Name: vRenderScene()
// Desc: Displays the 3d scene
//-----------------------------------------------------------------------------
void vRenderScene(void)
{	
	RECT				rTop = { 5, 5, SCREEN_WIDTH, 30 };
	RECT				rMid = { SCREEN_WIDTH/2 - 150, (SCREEN_HEIGHT/2-15), SCREEN_WIDTH/2 + 150, (SCREEN_HEIGHT/2+15) };
	RECT				rBottom = { 5, SCREEN_HEIGHT - 70, SCREEN_WIDTH, SCREEN_HEIGHT };
	char				szMessage[256];
	int					i;	
		
	// Clear the backbuffer and the zbuffer
    pd3dDevice->Clear( 0, NULL, D3DCLEAR_TARGET|D3DCLEAR_ZBUFFER, D3DCOLOR_XRGB(0,0,0), 1.0f, 0 );
	// Begin Rendering
	pd3dDevice->BeginScene();

	// Update the camera position
	vUpdateCamera();
	
	// Turn on texture filtering
	pd3dDevice->SetTextureStageState(0,D3DTSS_MINFILTER ,D3DTEXF_ANISOTROPIC);
	pd3dDevice->SetTextureStageState(0,D3DTSS_MAGFILTER ,D3DTEXF_ANISOTROPIC);

	// Display the table
	table.vDisplayXYZ();

	// Display balls - colors
	black.vDisplayXYZ();
	pink.vDisplayXYZ();
	blue.vDisplayXYZ();
	brown.vDisplayXYZ();	
	green.vDisplayXYZ();
	yellow.vDisplayXYZ();

	// Display cue ball
	white.vDisplayXYZ();

	// Display reds
	for ( i = 0 ; i < REDS_NUM ; i++ ) {
		reds[i].vDisplayXYZ();
	}

	// Display cue
	cue.vDisplayXYZ();
	
	// Display permanent game text
	sprintf(szMessage,"Shot power: %d", sShotPower);
	pD3DXFont->DrawText(szMessage, -1, &rBottom, DT_LEFT, D3DCOLOR_RGBA(255, 255, 255, 255));

	rBottom.left += 220;
	sprintf(szMessage,"Points on table: %d", sPointsOnTable);
	pD3DXFont->DrawText(szMessage, -1, &rBottom, DT_LEFT, D3DCOLOR_RGBA(255, 255, 255, 255));

	rBottom.left += 220;
	switch (PlayerInfo[g_iCurrentPlayer].sBallOnValue)
	{
		case RED_VALUE:
			sprintf(szMessage,"Ball on: Red");
			break;
		case YELLOW_VALUE:
			sprintf(szMessage,"Ball on: Yellow");
			break;
		case GREEN_VALUE:
			sprintf(szMessage,"Ball on: Green");
			break;
		case BROWN_VALUE:
			sprintf(szMessage,"Ball on: Brown");
			break;
		case BLUE_VALUE:
			sprintf(szMessage,"Ball on: Blue");
			break;
		case PINK_VALUE:
			sprintf(szMessage,"Ball on: Pink");
			break;
		case BLACK_VALUE:
			sprintf(szMessage,"Ball on: Black");
	}
	pD3DXFont->DrawText(szMessage, -1, &rBottom, DT_LEFT, D3DCOLOR_RGBA(255, 255, 255, 255));

	for (i = 0; i < MAX_PLAYERS; i++)
	{
		sprintf(szMessage,"%s's points: %d", PlayerInfo[i].szPlayerName, PlayerInfo[i].sScore);
		if (i == g_iCurrentPlayer)
			pD3DXFont->DrawText(szMessage, -1, &rTop, DT_LEFT, D3DCOLOR_RGBA(0, 255, 255, 255));
		else
			pD3DXFont->DrawText(szMessage, -1, &rTop, DT_LEFT, D3DCOLOR_RGBA(255, 255, 255, 255));
		rTop.left += 220;
	}

	// Display middle screen text if needed
	if (bGameOver)
	{
		if (bOtherPlayerLeft)
			sprintf(szMessage,"%s left the game!", szTemporary);
		else
			sprintf(szMessage,"%s won!", PlayerInfo[g_iCurrentPlayer].szPlayerName);
		pD3DXFont->DrawText(szMessage, -1, &rMid, DT_CENTER, D3DCOLOR_RGBA(0, 255, 255, 255));
		rMid.top += 30;
		rMid.bottom += 30;
		sprintf(szMessage,"Press ENTER to play again!", PlayerInfo[g_iCurrentPlayer].szPlayerName);
		pD3DXFont->DrawText(szMessage, -1, &rMid, DT_CENTER, D3DCOLOR_RGBA(0, 255, 255, 255));
	}
	else if (bChooseGameType)
	{
		rMid.top -= 60;
		rMid.bottom -= 60;
		sprintf(szMessage,"Choose game type:");
		pD3DXFont->DrawText(szMessage, -1, &rMid, DT_LEFT, D3DCOLOR_RGBA(0, 255, 255, 255));
		rMid.top += 30;
		rMid.bottom += 30;
		sprintf(szMessage,"N - Single computer");
		pD3DXFont->DrawText(szMessage, -1, &rMid, DT_LEFT, D3DCOLOR_RGBA(0, 255, 255, 255));
		rMid.top += 30;
		rMid.bottom += 30;
		sprintf(szMessage,"H - Network game (host)");
		pD3DXFont->DrawText(szMessage, -1, &rMid, DT_LEFT, D3DCOLOR_RGBA(0, 255, 255, 255));
		rMid.top += 30;
		rMid.bottom += 30;
		sprintf(szMessage,"J - Network game (join)");
		pD3DXFont->DrawText(szMessage, -1, &rMid, DT_LEFT, D3DCOLOR_RGBA(0, 255, 255, 255));
		rMid.top += 30;
		rMid.bottom += 30;
	}
	else if (bChooseBallOn)
	{
		sprintf(szMessage,"Choose next ball on!");
		pD3DXFont->DrawText(szMessage, -1, &rMid, DT_CENTER, D3DCOLOR_RGBA(0, 255, 255, 255));
	}
	else if (bNextPlayer)
	{
		sprintf(szMessage,"%s's turn!", PlayerInfo[g_iCurrentPlayer].szPlayerName);
		pD3DXFont->DrawText(szMessage, -1, &rMid, DT_CENTER, D3DCOLOR_RGBA(0, 255, 255, 255));
	}

	pD3DXFont->DrawText(szSystemMessage, -1, &rMid, DT_CENTER, D3DCOLOR_RGBA(255, 255, 255, 255));

	// Stop Rendering
	pd3dDevice->EndScene();
	// Output the scene
	if( (pd3dDevice->Present( NULL, NULL, NULL, NULL )) == D3DERR_DEVICELOST ) {
	}
}

//-----------------------------------------------------------------------------
// Name: vInitGeometry()
// Desc: Creates the geometry
//-----------------------------------------------------------------------------
void vInitGeometry(void)
{
	char	szFileName[256];
	int		i;

	// Create snooker table
	sprintf(szFileName,"Data\\Objects\\table.x");
	table.hLoad(szFileName,pd3dDevice);

	// Create snooker balls - colors
	sprintf(szFileName,"Data\\Objects\\black.x");
	black.hLoad(szFileName,pd3dDevice);
	black.vSetValue(BLACK_VALUE);
	
	sprintf(szFileName,"Data\\Objects\\pink.x");
	pink.hLoad(szFileName,pd3dDevice);
	pink.vSetValue(PINK_VALUE);
	
	sprintf(szFileName,"Data\\Objects\\blue.x");
	blue.hLoad(szFileName,pd3dDevice);
	blue.vSetValue(BLUE_VALUE);
	
	sprintf(szFileName,"Data\\Objects\\brown.x");
	brown.hLoad(szFileName,pd3dDevice);
	brown.vSetValue(BROWN_VALUE);
	
	sprintf(szFileName,"Data\\Objects\\green.x");
	green.hLoad(szFileName,pd3dDevice);
	green.vSetValue(GREEN_VALUE);
	
	sprintf(szFileName,"Data\\Objects\\yellow.x");
	yellow.hLoad(szFileName,pd3dDevice);
	yellow.vSetValue(YELLOW_VALUE);

	// Create cue ball
	sprintf(szFileName,"Data\\Objects\\white.x");
	white.hLoad(szFileName,pd3dDevice);
	white.vSetValue(WHITE_VALUE);

	// Create red balls
	sprintf(szFileName,"Data\\Objects\\red.x");
	for( i = 0 ; i < REDS_NUM ; i++ ) {
		reds[i].hLoad(szFileName,pd3dDevice);
		reds[i].vSetValue(RED_VALUE);
	}
	
	// Create cue
	sprintf(szFileName,"Data\\Objects\\cue.x");
	cue.hLoad(szFileName,pd3dDevice);
	cue.vSetPosition( &white.vecGetPosition() );

	// Create the common text font
	hFont = CreateFont(18, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, PROOF_QUALITY, 0, "courier new"); 
	D3DXCreateFont(pd3dDevice, hFont, &pD3DXFont); 
}

//-----------------------------------------------------------------------------
// Name: vSetOriginalBallPositions()
// Desc: Sets positions of snooker balls
//-----------------------------------------------------------------------------
void vSetOriginalBallPositions(void)
{
	int i;

	// Set position of colors
	black.vSetPosition(&BLACK_POS);
	black.vSetPotted(false);
	arrBallPositions[REDS_NUM] = BLACK_POS;
	pink.vSetPosition(&PINK_POS);
	pink.vSetPotted(false);
	arrBallPositions[REDS_NUM + 1] = PINK_POS;
	blue.vSetPosition(&BLUE_POS);
	blue.vSetPotted(false);
	arrBallPositions[REDS_NUM + 2] = BLUE_POS;
    brown.vSetPosition(&BROWN_POS);
	brown.vSetPotted(false);
	arrBallPositions[REDS_NUM + 3] = BROWN_POS;
	green.vSetPosition(&GREEN_POS);
	green.vSetPotted(false);
	arrBallPositions[REDS_NUM + 4] = GREEN_POS;
	yellow.vSetPosition(&YELLOW_POS);
	yellow.vSetPotted(false);
	arrBallPositions[REDS_NUM + 5] = YELLOW_POS;

	// Set position of cue ball
	white.vSetPosition(&WHITE_POS);
	white.vSetPotted(false);
	arrBallPositions[REDS_NUM + 6] = WHITE_POS;

	// Set position of reds
    reds[0].vSetPosition(&D3DXVECTOR3( 0.0f, 0.0f, -95.22f ));
	reds[0].vSetPotted(false);
	arrBallPositions[0] = D3DXVECTOR3( 0.0f, 0.0f, -95.22f );
	reds[1].vSetPosition(&D3DXVECTOR3( -3.0f, 0.0f, -101.22f ));
	reds[1].vSetPotted(false);
	arrBallPositions[1] = D3DXVECTOR3( -3.0f, 0.0f, -101.22f );
	reds[2].vSetPosition(&D3DXVECTOR3( 3.0f, 0.0f, -101.22f ));
	reds[2].vSetPotted(false);
	arrBallPositions[2] = D3DXVECTOR3( 3.0f, 0.0f, -101.22f );
	reds[3].vSetPosition(&D3DXVECTOR3( -6.0f, 0.0f, -107.22f ));
	reds[3].vSetPotted(false);
	arrBallPositions[3] = D3DXVECTOR3( -6.0f, 0.0f, -107.22f );
	reds[4].vSetPosition(&D3DXVECTOR3( 0.0f, 0.0f, -107.22f ));
	reds[4].vSetPotted(false);
	arrBallPositions[4] = D3DXVECTOR3( 0.0f, 0.0f, -107.22f );
	reds[5].vSetPosition(&D3DXVECTOR3( 6.0f, 0.0f, -107.22f ));
	reds[5].vSetPotted(false);
	arrBallPositions[5] = D3DXVECTOR3( 6.0f, 0.0f, -107.22f );
	reds[6].vSetPosition(&D3DXVECTOR3( -9.0f, 0.0f, -113.22f ));
	reds[6].vSetPotted(false);
	arrBallPositions[6] = D3DXVECTOR3( -9.0f, 0.0f, -113.22f );
	reds[7].vSetPosition(&D3DXVECTOR3( -3.0f, 0.0f, -113.22f ));
	reds[7].vSetPotted(false);
	arrBallPositions[7] = D3DXVECTOR3( -3.0f, 0.0f, -113.22f );
	reds[8].vSetPosition(&D3DXVECTOR3( 3.0f, 0.0f, -113.22f ));
	reds[8].vSetPotted(false);
	arrBallPositions[8] = D3DXVECTOR3( 3.0f, 0.0f, -113.22f );
	reds[9].vSetPosition(&D3DXVECTOR3( 9.0f, 0.0f, -113.22f ));
	reds[9].vSetPotted(false);
	arrBallPositions[9] = D3DXVECTOR3( 9.0f, 0.0f, -113.22f );
	reds[10].vSetPosition(&D3DXVECTOR3( -12.0f, 0.0f, -119.22f ));
	reds[10].vSetPotted(false);
	arrBallPositions[10] = D3DXVECTOR3( -12.0f, 0.0f, -119.22f );
	reds[11].vSetPosition(&D3DXVECTOR3( -6.0f, 0.0f, -119.22f ));
	reds[11].vSetPotted(false);
	arrBallPositions[11] = D3DXVECTOR3( -6.0f, 0.0f, -119.22f );
	reds[12].vSetPosition(&D3DXVECTOR3( 0.0f, 0.0f, -119.22f ));
	reds[12].vSetPotted(false);
	arrBallPositions[12] = D3DXVECTOR3( 0.0f, 0.0f, -119.22f );
	reds[13].vSetPosition(&D3DXVECTOR3( 6.0f, 0.0f, -119.22f ));
	reds[13].vSetPotted(false);
	arrBallPositions[13] = D3DXVECTOR3( 6.0f, 0.0f, -119.22f );
	reds[14].vSetPosition(&D3DXVECTOR3( 12.0f, 0.0f, -119.22f ));
	reds[14].vSetPotted(false);
	arrBallPositions[14] = D3DXVECTOR3( 12.0f, 0.0f, -119.22f );
	
	for (i = 0; i < TOTAL_BALLS; i++)
		arrBallPotted[i] = false;

	iBallPosCount = 0;
}

//-----------------------------------------------------------------------------
// Name: vCreateLights()
// Desc: Creates a point light and default material
//-----------------------------------------------------------------------------
void vCreateLights(void)
{
	D3DLIGHT8		d3dLight;
	
	// Clear out light structure
	ZeroMemory(&d3dLight, sizeof(D3DLIGHT8));
	d3dLight.Type = D3DLIGHT_POINT;
	
	d3dLight.Diffuse.r  = 1.0f;
	d3dLight.Diffuse.b  = 1.0f;
	d3dLight.Diffuse.g  = 1.0f;
	
	d3dLight.Position.x = 0.0f;
	d3dLight.Position.y = 100.0f;
	d3dLight.Position.z = 0.0f;
	d3dLight.Attenuation0 = 1.0f; 
	d3dLight.Attenuation1 = 0.0025f;
	d3dLight.Range      = 1000.0f;
	// Set the active light
    pd3dDevice->SetLight( 0, &d3dLight );
	// Enable the light
    pd3dDevice->LightEnable( 0, TRUE );

	d3dLight.Range        = 200.0f;
	d3dLight.Attenuation0 = 1.0f; 
	d3dLight.Attenuation1 = 0.01f;
	d3dLight.Diffuse.r  = 1.0f;
	d3dLight.Diffuse.g  = 1.0f;
	d3dLight.Diffuse.b  = 1.0f;
	d3dLight.Position.x = 100.0f;
	d3dLight.Position.y = 100.0f;
	d3dLight.Position.z = 200.0f;
	// Set the active light
    pd3dDevice->SetLight( 1, &d3dLight );
	// Enable the light
    pd3dDevice->LightEnable( 1, TRUE );

	d3dLight.Diffuse.r  = 1.0f;
	d3dLight.Diffuse.g  = 1.0f;
	d3dLight.Diffuse.b  = 1.0f;
	d3dLight.Position.x = 100.0f;
	d3dLight.Position.y = 100.0f;
	d3dLight.Position.z = -200.0f;
	// Set the active light
    pd3dDevice->SetLight( 2, &d3dLight );
	// Enable the light
    pd3dDevice->LightEnable( 2, TRUE );

	d3dLight.Diffuse.r  = 1.0f;
	d3dLight.Diffuse.g  = 1.0f;
	d3dLight.Diffuse.b  = 1.0f;
	d3dLight.Position.x = -100.0f;
	d3dLight.Position.y = 100.0f;
	d3dLight.Position.z = -200.0f;
	// Set the active light
    pd3dDevice->SetLight( 3, &d3dLight );
	// Enable the light
    pd3dDevice->LightEnable( 3, TRUE );

	d3dLight.Diffuse.r  = 1.0f;
	d3dLight.Diffuse.g  = 1.0f;
	d3dLight.Diffuse.b  = 1.0f;
	d3dLight.Position.x = -100.0f;
	d3dLight.Position.y = 100.0f;
	d3dLight.Position.z = 200.0f;
	// Set the active light
    pd3dDevice->SetLight( 4, &d3dLight );
	// Enable the light
    pd3dDevice->LightEnable( 4, TRUE );
	
	// Tell D3D to utilize lights
    pd3dDevice->SetRenderState( D3DRS_LIGHTING, TRUE );
}

//-----------------------------------------------------------------------------
// Name: vSetupView()
// Desc: Sets up the viewport and camera
//-----------------------------------------------------------------------------
void vSetupView(void)
{
	// Setup Projection 
	D3DXMatrixPerspectiveFovLH( &matProj, D3DX_PI/2, 1.0f, 1.0f, 500.0f );
	// Tell the device to use the above matrix for projection
	pd3dDevice->SetTransform( D3DTS_PROJECTION, &matProj );
}

//-----------------------------------------------------------------------------
// Name: hrBuildDeviceList()
// Desc: Builds a list of available 3D devices for the application
//-----------------------------------------------------------------------------
HRESULT hrBuildDeviceList(HWND hWnd)
{
	const DWORD			dwNumDeviceTypes = 2;
	const char			*szDeviceDescs[] = { "HAL", "REF" };
	const D3DDEVTYPE	DeviceTypes[] = { D3DDEVTYPE_HAL, D3DDEVTYPE_REF };
	D3DDISPLAYMODE		modes[100];
	D3DFORMAT			formats[20];
	BOOL				bFormatConfirmed[20];
	DWORD				dwBehavior[20];
	D3DFORMAT			fmtDepthStencil[20];
	D3DAdapterInfo		*pAdapter;
	BOOL				bHALExists = FALSE;
	BOOL				bHALIsWindowedCompatible = FALSE;
	BOOL				bHALIsDesktopCompatible = FALSE;
	BOOL				bHALIsSampleCompatible = FALSE;
	DWORD				dwNumFormats;
	DWORD				dwNumModes;
	DWORD				dwNumAdapterModes;
	unsigned int		iAdapter,iMode,iDevice;
	DWORD				a,d,f,m;
	D3DDISPLAYMODE		DisplayMode;
	BOOL				bFoundVidMode = FALSE;
	D3DDeviceInfo		*pDevice;
	
	for( iAdapter = 0; iAdapter < pD3D->GetAdapterCount(); iAdapter++ ) {
		// Fill in adapter info
		pAdapter  = &g_Adapters[g_dwNumAdapters];
		pD3D->GetAdapterDisplayMode( iAdapter, &pAdapter->d3ddmDesktop );
		pAdapter->dwNumDevices    = 0;
		pAdapter->dwCurrentDevice = 0;
		
		// Enumerate all display modes on this adapter
		dwNumFormats      = 0;
		dwNumModes        = 0;
		dwNumAdapterModes = pD3D->GetAdapterModeCount( iAdapter );
		
		// Add the adapter's current desktop format to the list of formats
		formats[dwNumFormats++] = pAdapter->d3ddmDesktop.Format;
		
		// Loop through modes
		for( iMode = 0; iMode < dwNumAdapterModes; iMode++ ) {
			// Get the display mode attributes
			pD3D->EnumAdapterModes( iAdapter, iMode, &DisplayMode );
			
			// Filter out low-resolution modes
			if( (DisplayMode.Width < SCREEN_WIDTH) || (DisplayMode.Height < SCREEN_HEIGHT) )
				continue;
			
			// Check if the mode already exists
			for( m = 0; m < dwNumModes; m++ ) {
				if( ( modes[m].Width  == DisplayMode.Width  ) &&
					( modes[m].Height == DisplayMode.Height ) &&
					( modes[m].Format == DisplayMode.Format ) )
					break;
			}
			
			// If we found a new mode, add it to the list of modes
			if( m == dwNumModes ) {
				modes[dwNumModes].Width       = DisplayMode.Width;
				modes[dwNumModes].Height      = DisplayMode.Height;
				modes[dwNumModes].Format      = DisplayMode.Format;
				modes[dwNumModes].RefreshRate = 0;
				dwNumModes++;
				// Check if the mode's format already exists
				for( f = 0; f < dwNumFormats; f++ ) {
					if( DisplayMode.Format == formats[f] )
						break;
				}
				// If the format is new, add it to the list
				if( f == dwNumFormats )
					formats[dwNumFormats++] = DisplayMode.Format;
			}
		}
		
		// Add devices to adapter
		for( iDevice = 0; iDevice < dwNumDeviceTypes; iDevice++ ) {
			// Fill in device info
			pDevice                 = &pAdapter->devices[pAdapter->dwNumDevices];
			pDevice->DeviceType     = DeviceTypes[iDevice];
			pD3D->GetDeviceCaps( iAdapter, DeviceTypes[iDevice], &pDevice->d3dCaps );
			strcpy(pDevice->szDesc,szDeviceDescs[iDevice]);
			pDevice->dwNumModes     = 0;
			pDevice->dwCurrentMode  = 0;
			pDevice->bCanDoWindowed = FALSE;
			pDevice->bWindowed		= FALSE;
			pDevice->MultiSampleType = D3DMULTISAMPLE_NONE;
			
			// Check each format to see if it is compatible with the application
			for( f = 0; f < dwNumFormats; f++ ) {
				bFormatConfirmed[f] = FALSE;
				fmtDepthStencil[f] = D3DFMT_UNKNOWN;
				
				// Skip formats that cannot be used as render targets on this device
				if( FAILED( pD3D->CheckDeviceType( iAdapter, pDevice->DeviceType, formats[f], formats[f], FALSE ) ) )
					continue;
				
				if( pDevice->DeviceType == D3DDEVTYPE_HAL ) {
					// This system has a HAL device
					bHALExists = TRUE;
					// Check if Windowed mode is possible
					if( pDevice->d3dCaps.Caps2 & D3DCAPS2_CANRENDERWINDOWED ) {
						// HAL can run in a window for some mode
						bHALIsWindowedCompatible = TRUE;
						
						if( f == 0 ) {
							// HAL can run in a window for the current desktop mode
							bHALIsDesktopCompatible = TRUE;
						}
					}
				}
				// Confirm the device/format for HW vertex processing
				if( pDevice->d3dCaps.DevCaps & D3DDEVCAPS_HWTRANSFORMANDLIGHT ) {
					if( pDevice->d3dCaps.DevCaps & D3DDEVCAPS_PUREDEVICE ) {
						dwBehavior[f] = D3DCREATE_HARDWARE_VERTEXPROCESSING | D3DCREATE_PUREDEVICE;
						
						if( SUCCEEDED( hrConfirmDevice( &pDevice->d3dCaps, dwBehavior[f], formats[f] ) ) )
							bFormatConfirmed[f] = TRUE;
					}
					
					if ( FALSE == bFormatConfirmed[f] ) {
						dwBehavior[f] = D3DCREATE_HARDWARE_VERTEXPROCESSING;
						
						if( SUCCEEDED( hrConfirmDevice( &pDevice->d3dCaps, dwBehavior[f], formats[f] ) ) )
							bFormatConfirmed[f] = TRUE;
					}
					
					if ( FALSE == bFormatConfirmed[f] ) {
						dwBehavior[f] = D3DCREATE_MIXED_VERTEXPROCESSING;
						
						if( SUCCEEDED( hrConfirmDevice( &pDevice->d3dCaps, dwBehavior[f], formats[f] ) ) )
							bFormatConfirmed[f] = TRUE;
					}
				}
				
				// Confirm the device/format for SW vertex processing
				if( FALSE == bFormatConfirmed[f] ) {
					dwBehavior[f] = D3DCREATE_SOFTWARE_VERTEXPROCESSING;
					
					if( SUCCEEDED( hrConfirmDevice( &pDevice->d3dCaps, dwBehavior[f], formats[f] ) ) )
						bFormatConfirmed[f] = TRUE;
				}
				
				// Find a suitable depth/stencil buffer format for this device/format
				if( bFormatConfirmed[f] && g_bUseDepthBuffer ) {
					if( !bFindDepthStencilFormat( 8, 0, iAdapter, pDevice->DeviceType,
						formats[f], &fmtDepthStencil[f] ) ) {
						bFormatConfirmed[f] = FALSE;
					}
				}
			}
			
			// Add modes that are confirmed to the list of available modes
			for( m = 0; m < dwNumModes; m++ ) {
				for( f = 0; f < dwNumFormats; f++ ) {
					if( modes[m].Format == formats[f] ) {
						if( bFormatConfirmed[f] == TRUE ) {
							// Add this mode to the device's list of valid modes
							pDevice->modes[pDevice->dwNumModes].Width      = modes[m].Width;
							pDevice->modes[pDevice->dwNumModes].Height     = modes[m].Height;
							pDevice->modes[pDevice->dwNumModes].Format     = modes[m].Format;
							pDevice->modes[pDevice->dwNumModes].dwBehavior = dwBehavior[f];
							pDevice->modes[pDevice->dwNumModes].DepthStencilFormat = fmtDepthStencil[f];
							pDevice->dwNumModes++;
							
							if( pDevice->DeviceType == D3DDEVTYPE_HAL )
								bHALIsSampleCompatible = TRUE;
						}
					}
				}
			}
			
			// Check if there was a device found that matched desired requirements
			for( m = 0; m < pDevice->dwNumModes; m++ ) {
				if( pDevice->modes[m].Width == SCREEN_WIDTH && pDevice->modes[m].Height == SCREEN_HEIGHT ) {
					pDevice->dwCurrentMode = m;
					if( g_bWindowed ) {
						if( pDevice->modes[m].Format == pAdapter->d3ddmDesktop.Format ) {
							bFoundVidMode = TRUE;
							break;
						}
					}
					else if( pDevice->modes[m].Format == D3DFMT_R5G6B5 ||
						pDevice->modes[m].Format == D3DFMT_X1R5G5B5 ||
						pDevice->modes[m].Format == D3DFMT_A1R5G5B5 ||
						pDevice->modes[m].Format == D3DFMT_X8R8G8B8 ||
						pDevice->modes[m].Format == D3DFMT_A8R8G8B8 ) {
						bFoundVidMode = TRUE;
						break;
					}
				}
			}
			
			// Check if device can work in a window
			if( bFormatConfirmed[0] && (pDevice->d3dCaps.Caps2 & D3DCAPS2_CANRENDERWINDOWED) ) {
				pDevice->bCanDoWindowed = TRUE;
				pDevice->bWindowed      = TRUE;
			}
			
			// If valid modes were found, keep this device
			if( pDevice->dwNumModes > 0 )
				pAdapter->dwNumDevices++;
		}
		// If valid devices were found, keep this adapter
		if( pAdapter->dwNumDevices > 0 )
			g_dwNumAdapters++;
	}
	
	// Return an error if no compatible devices were found
	if( !g_dwNumAdapters || !bFoundVidMode )
		return( VIDEODEVICEERROR_NOCOMPATIBLE );
	
	// Choose a default adapter
	for( a = 0; a < g_dwNumAdapters; a++ ) {
		for( d = 0; d < g_Adapters[a].dwNumDevices; d++ ) {
			if( (g_Adapters[a].devices[d].bWindowed && g_bWindowed) || (!g_bWindowed) ) {
				
				g_Adapters[a].dwCurrentDevice = d;
				g_dwAdapter = a;
				g_Adapters[a].devices[d].bWindowed = g_bWindowed;
				
				// Display a warning message
				if( g_Adapters[a].devices[d].DeviceType == D3DDEVTYPE_REF ) {
					if( !bHALExists )
						MessageBox( hWnd, "HAL Error", "Hardware Acceleration Not Found.", MB_ICONERROR );
					else if( !bHALIsSampleCompatible )
						MessageBox( hWnd, "HAL Error", "Your Hardware Acceleration Cannot Support This Application.", MB_ICONERROR );
					else if( !bHALIsWindowedCompatible && !g_bWindowed )
						MessageBox( hWnd, "HAL Error", "Your Hardware Acceleration Cannot Render To A Window.", MB_ICONERROR );
					else if( !bHALIsDesktopCompatible && !g_bWindowed )
						MessageBox( hWnd, "HAL Error", "Your Hardware Acceleration Cannot Render To A Window In The Current Desktop Setting.", MB_ICONERROR );
					else
						MessageBox( hWnd, "HAL Error", "Your Hardware Acceleration Cannot Support This Application In The Current Desktop Setting.", MB_ICONERROR );
				}
				
				return S_OK;
			}
		}
	}

    return( VIDEODEVICEERROR_NOWINDOWED );
}

//-----------------------------------------------------------------------------
// Name: hrConfirmDevice()
// Desc: Returns success if the device passed meets the apps requirements
//-----------------------------------------------------------------------------
HRESULT hrConfirmDevice( D3DCAPS8* pCaps, DWORD dwBehavior, D3DFORMAT Format )
{
	// Check for alpha support
	if( pCaps->TextureCaps & D3DPTEXTURECAPS_ALPHAPALETTE )
        return S_OK;
    if( pCaps->TextureCaps & D3DPTEXTURECAPS_ALPHA )
        return S_OK;

	return S_OK;

    return E_FAIL;
}

//-----------------------------------------------------------------------------
// Name: bFindDepthStencilFormat()
// Desc: Tries to find a valied stencil format for the passed requirements
//-----------------------------------------------------------------------------
BOOL bFindDepthStencilFormat( DWORD dwMinDepth, DWORD dwMinStencil, UINT iAdapter, D3DDEVTYPE DeviceType, D3DFORMAT TargetFormat, D3DFORMAT* pDepthStencilFormat )
{
    D3DFORMAT	d3dfFormat = D3DFMT_UNKNOWN;
	
	if( dwMinDepth <= 16 && dwMinStencil == 0 ) {
		d3dfFormat = D3DFMT_D16;
		if( (bValidateFormat(d3dfFormat, iAdapter, DeviceType, TargetFormat)) == TRUE ) {
			*pDepthStencilFormat = d3dfFormat;
			return(TRUE);
		}
	}
	if( dwMinDepth <= 15 && dwMinStencil <= 1 ) {
		d3dfFormat = D3DFMT_D15S1;
		if( (bValidateFormat(d3dfFormat, iAdapter, DeviceType, TargetFormat)) == TRUE ) {
			*pDepthStencilFormat = d3dfFormat;
			return(TRUE);
		}
	}
	if( dwMinDepth <= 24 && dwMinStencil == 0 ) {
		d3dfFormat = D3DFMT_D24X8;
		if( (bValidateFormat(d3dfFormat, iAdapter, DeviceType, TargetFormat)) == TRUE ) {
			*pDepthStencilFormat = d3dfFormat;
			return(TRUE);
		}
	}
	if( dwMinDepth <= 24 && dwMinStencil <= 8 ) {
		d3dfFormat = D3DFMT_D24S8;
		if( (bValidateFormat(d3dfFormat, iAdapter, DeviceType, TargetFormat)) == TRUE ) {
			*pDepthStencilFormat = d3dfFormat;
			return(TRUE);
		}
	}
	if( dwMinDepth <= 24 && dwMinStencil <= 4 ) {
		d3dfFormat = D3DFMT_D24X4S4;
		if( (bValidateFormat(d3dfFormat, iAdapter, DeviceType, TargetFormat)) == TRUE ) {
			*pDepthStencilFormat = d3dfFormat;
			return(TRUE);
		}
	}
	if( dwMinDepth <= 32 && dwMinStencil == 0 ) {
		d3dfFormat = D3DFMT_D32;
		if( (bValidateFormat(d3dfFormat, iAdapter, DeviceType, TargetFormat)) == TRUE ) {
			*pDepthStencilFormat = d3dfFormat;
			return(TRUE);
		}
	}

    return FALSE;
}

//-----------------------------------------------------------------------------
// Name: bValidateFormat()
// Desc: Checks if the passed format is compatible
//-----------------------------------------------------------------------------
BOOL bValidateFormat(D3DFORMAT d3dfFormatToTest, UINT iAdapter, D3DDEVTYPE DeviceType, D3DFORMAT TargetFormat)
{
	if( SUCCEEDED( pD3D->CheckDeviceFormat( iAdapter, DeviceType, TargetFormat, 
		D3DUSAGE_DEPTHSTENCIL, D3DRTYPE_SURFACE, d3dfFormatToTest ) ) ) {
        if( SUCCEEDED( pD3D->CheckDepthStencilMatch( iAdapter, DeviceType, 
			TargetFormat, TargetFormat, d3dfFormatToTest ) ) ) {
            return TRUE;
        }
    }
	return(FALSE);
}

//-----------------------------------------------------------------------------
//
//
//
//							DIRECT PLAY FUNCTIONS
//
//
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Name: hrInitializeDirectPlay()
// Desc: Initializes all Direct Play interfaces and objects
//-----------------------------------------------------------------------------
HRESULT hrInitializeDirectPlay( HWND hWindow ) 
{
	HRESULT	hReturn;
	
	// Initialize COM
	hReturn = CoInitialize( NULL );
	if( FAILED(hReturn) ) {
		MessageBox( hWindow, "Error Initializing COM", "DirectPlay Error", MB_ICONERROR );
		return hReturn;
	}

	// Initialize critical sections for multi-threading
	InitializeCriticalSection( &g_csModifyPlayer );
	
	// Create IDirectPlay8Peer Object
	if( FAILED( hReturn = CoCreateInstance( CLSID_DirectPlay8Peer, 
		NULL, 
		CLSCTX_INPROC_SERVER, 
		IID_IDirectPlay8Peer, 
		(LPVOID*) &g_pDP ) ) )
        MessageBox( hWindow, "Can't Create DPlayPeer", "DirectPlay Error", MB_ICONERROR );
	
	// Init IDirectPlay8Peer Message Handler
	if( FAILED( hReturn = g_pDP->Initialize( NULL, DirectPlayMessageHandler, 0 ) ) ) {
		MessageBox( hWindow, "Failed to Message Handler", "DirectPlay Error", MB_ICONERROR );
		return -1;
	}
	
	// Create a device address
	hReturn = CoCreateInstance( CLSID_DirectPlay8Address, NULL,CLSCTX_INPROC_SERVER, IID_IDirectPlay8Address, (LPVOID*) &g_pDeviceAddress );
	if( FAILED(hReturn) ) {
		MessageBox( hWindow, "Failed to Create Device", "CoCreateInstance()", MB_ICONERROR );
		return -1;
	}
	
	// Set our service provider to TCP/IP
	if( FAILED( hReturn = g_pDeviceAddress->SetSP( &CLSID_DP8SP_TCPIP ) ) ) {
		MessageBox( hWindow, "Failed to SetSP() for Device Address", "Invalid Param", MB_ICONERROR );
		return -1;
	}
	
	// Create a host address
	hReturn = CoCreateInstance( CLSID_DirectPlay8Address, NULL,CLSCTX_INPROC_SERVER, IID_IDirectPlay8Address, (LPVOID*) &g_pHostAddress );
	if( FAILED(hReturn) ) {
		MessageBox( hWindow, "Failed to Create Host Address()", "Invalid Param", MB_ICONERROR );
		return -1;
	}
	// Set the host address to TCP/IP
	if( FAILED( hReturn = g_pHostAddress->SetSP( &CLSID_DP8SP_TCPIP ) ) ) {
		MessageBox( hWindow, "Failed to SetSP() for Host Address", "Invalid Param", MB_ICONERROR );
		return -1;
	}

	return S_OK;
}

//-----------------------------------------------------------------------------
// Name: DirectPlayMessageHandler()
// Desc: Handles all Direct Play messages
//-----------------------------------------------------------------------------
HRESULT WINAPI DirectPlayMessageHandler( PVOID pvUserContext, DWORD dwMessageId, PVOID pMsgBuffer )
{
	HRESULT				hReturn = S_OK;
	PacketTurn			*TurnMsg;
	PacketTurn			TurnMsg_Send;
	PacketStroke		*StrokeMsg;
	PacketBallPos		*BallPosMsg;
	PacketScore			*ScoreMsg;
	PacketGameOver		*GameOverMsg;
	VOID				*packet;
	int					iPlayerID;
	
	switch( dwMessageId ) 
	{
		case DPN_MSGID_CREATE_PLAYER:
		{
			hrCreatePlayer(pvUserContext,pMsgBuffer);
			break;
		}
		
		case DPN_MSGID_DESTROY_PLAYER:
		{
			hrDestroyPlayer(pvUserContext,pMsgBuffer);
			break;
		}
		
		case DPN_MSGID_TERMINATE_SESSION:
		{
			PDPNMSG_TERMINATE_SESSION pTerminateSessionMsg;
			pTerminateSessionMsg = (PDPNMSG_TERMINATE_SESSION)pMsgBuffer;
			break;
		}
		
		case DPN_MSGID_RECEIVE:
		{
			PDPNMSG_RECEIVE pReceiveMsg;
			PacketGeneric	*PGen;

			pReceiveMsg = (PDPNMSG_RECEIVE)pMsgBuffer;
			
			// Cast it to a generic packet so we can check the type
			PGen = (PacketGeneric*)pReceiveMsg->pReceiveData;

			// Whose player turn it is message
			if ( PGen->dwType == PACKET_TYPE_TURN ) 
			{
				// Convert the packet to a chat packet
				TurnMsg = (PacketTurn*)pReceiveMsg->pReceiveData;
				// Process it
				EnterCriticalSection( &g_csModifyPlayer );
				iPlayerID = iGetPlayerID(TurnMsg->iPlayerID);
				g_iCurrentPlayer = iPlayerID;
				if (!TurnMsg->bChooseBallOn)
					PlayerInfo[g_iCurrentPlayer].sBallOnValue = TurnMsg->sBallOn;
				else if (g_iMyPlayerId == g_iCurrentPlayer)
					bChooseBallOn = TRUE;
				if (g_iMyPlayerId == g_iCurrentPlayer)
					bBallInHand = TurnMsg->bBallInHand;
				bNextPlayer = TRUE;
				LeaveCriticalSection( &g_csModifyPlayer );
			}
			// Stroke message
			else if ( PGen->dwType == PACKET_TYPE_STROKE )
			{
				// Convert the packet to a stroke packet
				StrokeMsg = (PacketStroke*)pReceiveMsg->pReceiveData;
				// Process it
				EnterCriticalSection( &g_csModifyPlayer );
				white.vSetPosition(&D3DXVECTOR3(StrokeMsg->fCuePosX, 0.0f, StrokeMsg->fCuePosZ));
				cue.vSetPosition(&white.vecGetPosition());
				cue.vSetRotation(&D3DXVECTOR3(StrokeMsg->fCueRotX, StrokeMsg->fCueRotY, 0.0f));
				sShotPower = StrokeMsg->sPower;
				PlayerInfo[g_iCurrentPlayer].sBallOnValue = StrokeMsg->sBallOn;
				cue.vPerformShot(&white, sShotPower);
				bBallInHand = FALSE;
				bPerformShot = FALSE;
				bShotDone = TRUE;
				bNextPlayer = FALSE;
				bScoreComputed = FALSE;
				LeaveCriticalSection( &g_csModifyPlayer );
			}
			else if ( PGen->dwType == PACKET_TYPE_BALLPOS )
			{
				// Convert the packet to a ballpos packet
				BallPosMsg = (PacketBallPos*)pReceiveMsg->pReceiveData;
				// Store balls positions
				EnterCriticalSection( &g_csModifyPlayer );
				arrBallPositions[BallPosMsg->iBallNo].x = BallPosMsg->fX;
				arrBallPositions[BallPosMsg->iBallNo].z = BallPosMsg->fZ;
				arrBallPotted[BallPosMsg->iBallNo] = BallPosMsg->bPotted;
				iBallPosCount++;
				LeaveCriticalSection( &g_csModifyPlayer );
			}
			else if ( PGen->dwType == PACKET_TYPE_SCORE )
			{
				// Convert the packet to a ballpos packet
				ScoreMsg = (PacketScore*)pReceiveMsg->pReceiveData;
				// Store balls positions
				EnterCriticalSection( &g_csModifyPlayer );
				iPlayerID = iGetPlayerID(ScoreMsg->iPlayerID);
				PlayerInfo[iPlayerID].sScore = ScoreMsg->sScore;
				LeaveCriticalSection( &g_csModifyPlayer );
			}
			else if ( PGen->dwType == PACKET_TYPE_OK )
			{
				// This packet is received when all ball positions ahve been successfully processed by other player
				bPacketsProcessed = TRUE;
				// Send turn message if flag is set
				if (bSendTurn)
				{
					TurnMsg_Send.dwSize = sizeof(PacketTurn);
					TurnMsg_Send.dwType = PACKET_TYPE_TURN;
					TurnMsg_Send.iPlayerID = PlayerInfo[g_iCurrentPlayer].iPlayerID;
					TurnMsg_Send.sBallOn = PlayerInfo[g_iCurrentPlayer].sBallOnValue;
					TurnMsg_Send.bChooseBallOn = bChooseBallOn;
					TurnMsg_Send.bBallInHand = bBallInHand;
					// Convert the packet to a void stream
					packet = (VOID*)&TurnMsg_Send;
					// Send the chat packet
					hrSendPeerMessage(-1,PACKET_TYPE_TURN,packet);
					bNextPlayer = TRUE;
					bSendTurn = FALSE;
					bBallInHand = FALSE;
					bChooseBallOn = FALSE;
				}
				if (bGameOver && !bOtherPlayerLeft)
					vSendGameOverMsg();
			}
			else if ( PGen->dwType == PACKET_TYPE_GAMEOVER )
			{	
				// Convert the packet to a chat packet
				GameOverMsg = (PacketGameOver*)pReceiveMsg->pReceiveData;
				// Process it
				EnterCriticalSection( &g_csModifyPlayer );
				iPlayerID = iGetPlayerID(GameOverMsg->iPlayerID);
				g_iCurrentPlayer = iPlayerID;
				bGameOver = TRUE;
				// Terminate session, so another player can also be the next host
				if (bHostGame)
					g_pDP->TerminateSession(NULL,0,0);
				LeaveCriticalSection( &g_csModifyPlayer );
			}
		}
		
		case DPN_MSGID_CONNECT_COMPLETE:
		{
			PDPNMSG_CONNECT_COMPLETE pConnectCompleteMsg;
			pConnectCompleteMsg = (PDPNMSG_CONNECT_COMPLETE)pMsgBuffer;
			// Enter this if clause only once (when connection is established)
			if (SUCCEEDED(pConnectCompleteMsg->hResultCode) && bConnecting)
			{
				sprintf(szSystemMessage,"");
				bConnecting = FALSE;
				bNextPlayer = TRUE;
			}
			break;
		}
	}
	
	return hReturn;
}

//-----------------------------------------------------------------------------
// Name: hrCreatePlayer()
// Desc: Function to create a player who joins the session
//-----------------------------------------------------------------------------
HRESULT	hrCreatePlayer( PVOID pvUserContext, PVOID pMsgBuffer )
{
	HRESULT					hReturn = S_OK;
    PDPNMSG_CREATE_PLAYER	pCreatePlayerMsg;
	char					strName[256];
	DWORD					dwSize = 0;
	DPN_PLAYER_INFO			*pdpPlayerInfo = NULL;
	int						i;
	void					*packet;
	PacketTurn				TurnMsg;
		
	// Get a Create Message pointer to the buffer
	pCreatePlayerMsg = (PDPNMSG_CREATE_PLAYER)pMsgBuffer;
	
    // Get the peer info and extract its name
    hReturn = g_pDP->GetPeerInfo( pCreatePlayerMsg->dpnidPlayer, pdpPlayerInfo, &dwSize, 0 );
    if( FAILED(hReturn) && hReturn != DPNERR_BUFFERTOOSMALL ) {
        hReturn = -1;
	}
	else {
		// Allocate memory for the player info
		pdpPlayerInfo = (DPN_PLAYER_INFO*) new BYTE[ dwSize ];
		
		ZeroMemory( pdpPlayerInfo, dwSize );
		pdpPlayerInfo->dwSize = sizeof(DPN_PLAYER_INFO);
		// Load the player info into the newly allocated data
		hReturn = g_pDP->GetPeerInfo( pCreatePlayerMsg->dpnidPlayer, pdpPlayerInfo, &dwSize, 0 );
		if( FAILED(hReturn) ) {
			hReturn = -1;
		}
		else {
			EnterCriticalSection( &g_csModifyPlayer );
			
			// Convert player name to ANSI
			DXUtil_ConvertWideStringToGeneric( strName, pdpPlayerInfo->pwszName );
			// Check if we are adding ourselves
			if( pdpPlayerInfo->dwPlayerFlags & DPNPLAYER_LOCAL ) {
				g_dpnidLocalPlayer = pCreatePlayerMsg->dpnidPlayer;
				g_iMyPlayerId = iAddPlayer(g_dpnidLocalPlayer,strName);
			}
			else {
				// Add the player to our game				
				i = iAddPlayer(pCreatePlayerMsg->dpnidPlayer,strName);
				// Send them a whose turn is it package (host always starts the game)
				if( bHostGame ) {
					TurnMsg.dwSize = sizeof(PacketTurn);
					TurnMsg.dwType = PACKET_TYPE_TURN;
					TurnMsg.iPlayerID = g_dpnidLocalPlayer;
					TurnMsg.sBallOn = RED_VALUE;
					TurnMsg.bChooseBallOn = FALSE;
					TurnMsg.bBallInHand = FALSE;
					// Convert the packet to a void stream
					packet = (VOID*)&TurnMsg;
					// Send the chat packet
					hReturn = hrSendPeerMessage(i,PACKET_TYPE_TURN,packet);
				}
			}

			SAFE_DELETE_ARRAY( pdpPlayerInfo );
			
			// Update the number of active players in a thread safe way
			InterlockedIncrement( &g_lNumberOfActivePlayers );

			if (g_lNumberOfActivePlayers == MAX_PLAYERS)
			{
				sprintf(szSystemMessage,"");
				bConnecting = FALSE;
				bNextPlayer = TRUE;
			}

			LeaveCriticalSection( &g_csModifyPlayer );
		}
	}
		
	return hReturn;
}

//-----------------------------------------------------------------------------
// Name: hrDestroyPlayer()
// Desc: Function to destroy a player who leaves the session
//-----------------------------------------------------------------------------
HRESULT	hrDestroyPlayer( PVOID pvUserContext, PVOID pMsgBuffer )
{
	PDPNMSG_DESTROY_PLAYER	pDestroyPlayerMsg;
	HRESULT					hReturn = S_OK;
	int						i;
		
	// Get a Destroy Message pointer to the buffer
	pDestroyPlayerMsg = (PDPNMSG_DESTROY_PLAYER)pMsgBuffer;
	
	// Update the number of active players in a thread safe way
	InterlockedDecrement( &g_lNumberOfActivePlayers );

	EnterCriticalSection( &g_csModifyPlayer );

	// Remove Player from list
	for( i = 0 ; i < MAX_PLAYERS ; i++ ) {
		if( PlayerInfo[i].bActive ) {
			if( PlayerInfo[i].iPlayerID == pDestroyPlayerMsg->dpnidPlayer ) {
				PlayerInfo[i].bActive = 0;
				// remember player that first left the session
				if (strcmp(szTemporary,"") == 0)
					sprintf(szTemporary,PlayerInfo[i].szPlayerName); 
				break;
			}
		}
	}
	// If we are host we end the game (not enough players left)
	if (bHostGame && !bGameOver)
		g_pDP->TerminateSession(NULL,0,0);
	// Set flags only when game is not over
	if (!bChooseGameType && !bGameOver)
	{
		bGameOver = TRUE;
		bOtherPlayerLeft = TRUE;
	}

	LeaveCriticalSection( &g_csModifyPlayer );
	
	return(hReturn);
}

//-----------------------------------------------------------------------------
// Name: hrHostGame()
// Desc: Function to host the game
//-----------------------------------------------------------------------------
HRESULT	hrHostGame( HWND hWindow)
{
	HRESULT					hReturn;
	char					szPeerName[256];
	char					szSessionName[256];
	WCHAR					wszPeerName[256];
	WCHAR					wszSessionName[256];
	DPN_APPLICATION_DESC	dnAppDesc;
	char					szIP[6];
	DWORD					dwLength = 256;
	DPN_PLAYER_INFO			dpPlayerInfo;
	DWORD					dwPort = 9000;
	
	// Read some settings from the config file
	FILE *fp = fopen("config.txt","r");
	fgets(szIP,16,fp);
	szIP[strlen(szIP)-1] = '\0';
	fgets(szPeerName,32,fp);
	szPeerName[strlen(szPeerName)-1] = '\0';
	fclose(fp);

	//
	// Setup our player information
	//
	DXUtil_ConvertGenericStringToWide( wszPeerName, szPeerName );
	ZeroMemory( &dpPlayerInfo, sizeof(DPN_PLAYER_INFO) );
	dpPlayerInfo.dwSize = sizeof(DPN_PLAYER_INFO);
	dpPlayerInfo.dwInfoFlags = DPNINFO_NAME;
	dpPlayerInfo.pwszName = wszPeerName;
	
	// Set us up to be non-asynchronous
	if( FAILED( hReturn = g_pDP->SetPeerInfo( &dpPlayerInfo, NULL, NULL, DPNSETPEERINFO_SYNC ) ) ) {
		MessageBox( hWindow, "Failed to SetPeerInfo()", "DirectPlay Error", MB_ICONERROR );
		return -1;
	}
	
	// Setup the application description
	sprintf(szSessionName,"%s's Game",szPeerName);
	DXUtil_ConvertGenericStringToWide( wszSessionName, szSessionName );
	
	ZeroMemory( &dnAppDesc, sizeof(DPN_APPLICATION_DESC) );
	dnAppDesc.dwSize			= sizeof(DPN_APPLICATION_DESC);
	dnAppDesc.guidApplication	= DP_ADSNOOKER;
	dnAppDesc.pwszSessionName	= wszSessionName;
	dnAppDesc.dwMaxPlayers		= MAX_PLAYERS;
	dnAppDesc.dwFlags			= 0;
	
	// Set the port number
	dwPort = 6000;
	
	// Add port number to address
	hReturn = g_pDeviceAddress->AddComponent(DPNA_KEY_PORT,&dwPort,sizeof(DWORD),DPNA_DATATYPE_DWORD);
	if( hReturn != S_OK ) {
		MessageBox( hWindow, "Failed to AddComponent()", "hrHostGame()", MB_ICONERROR );
		return -1;
	}
	
	// Host the game
	hReturn = g_pDP->Host(	&dnAppDesc,               
		&g_pDeviceAddress,        
		1,                        
		NULL, 
		NULL,               
		NULL,                     
		NULL );
	if( FAILED( hReturn ) ) {
		if( hReturn == DPNERR_INVALIDPARAM ) 
			MessageBox( hWindow, "Failed to Host()", "Invalid Param", MB_ICONERROR );
		else if( hReturn == DPNERR_INVALIDDEVICEADDRESS  ) 
			MessageBox( hWindow, "Failed to Host()", "Invalid Device Address", MB_ICONERROR );
		return -1;
	}
	
	return hReturn;
}

//-----------------------------------------------------------------------------
// Name: hrJoinGame()
// Desc: Function to join the game
//-----------------------------------------------------------------------------
HRESULT hrJoinGame( HWND hWnd )
{
	HRESULT					hReturn = S_OK;
	WCHAR					wszHostName[256];
	WCHAR					wszPeerName[256];
	char					szPeerName[256];
	char					szIP[256];
	DWORD					dwPort;
	DWORD					dwLength = 256;
	DPN_APPLICATION_DESC	dpnAppDesc;
	DPN_PLAYER_INFO			dpPlayerInfo;

	// Read some settings from the config file
	FILE *fp = fopen("config.txt","r");
	fgets(szIP,16,fp);
	szIP[strlen(szIP)-1] = '\0';
	fgets(szPeerName,32,fp);
	szPeerName[strlen(szPeerName)-1] = '\0';
	fclose(fp);

	// Set the peer info
	DXUtil_ConvertGenericStringToWide( wszPeerName, szPeerName );
	
	ZeroMemory( &dpPlayerInfo, sizeof(DPN_PLAYER_INFO) );
	dpPlayerInfo.dwSize = sizeof(DPN_PLAYER_INFO);
	dpPlayerInfo.dwInfoFlags = DPNINFO_NAME;
	dpPlayerInfo.pwszName = wszPeerName;
	
	// Make this a synchronous call
	if( FAILED( hReturn = g_pDP->SetPeerInfo( &dpPlayerInfo, NULL, NULL, DPNSETPEERINFO_SYNC ) ) ) {
		return -1;
	}
		
	// Prepare the application description
	ZeroMemory( &dpnAppDesc, sizeof( DPN_APPLICATION_DESC ) );
	dpnAppDesc.dwSize = sizeof( DPN_APPLICATION_DESC );
	dpnAppDesc.guidApplication = DP_ADSNOOKER;

	// Set the IP
	DXUtil_ConvertGenericStringToWide( wszHostName, szIP );
	// Set the port number
	dwPort = 6000;

	// Add host name to address
	hReturn = g_pHostAddress->AddComponent(DPNA_KEY_HOSTNAME,wszHostName,(wcslen(wszHostName)+1)*sizeof(WCHAR),DPNA_DATATYPE_STRING);
	if( hReturn != S_OK ) {
		MessageBox( hWnd, "Failed to AddComponent()", "hrJoinGame()", MB_ICONERROR );
		return -1;
	}
	// Add port number to address
	hReturn = g_pHostAddress->AddComponent(DPNA_KEY_PORT,&dwPort,sizeof(DWORD),DPNA_DATATYPE_DWORD);
	if( hReturn != S_OK ) {
		MessageBox( hWnd, "Failed to AddComponent()", "hrJoinGame()", MB_ICONERROR );
		return -1;
	}

	// Connect to the session
	hReturn = g_pDP->Connect(	&dpnAppDesc,
								g_pHostAddress,    
								g_pDeviceAddress,
								NULL,
								NULL,
								NULL,	
								0,
								NULL,
								NULL, 
								&g_hConnectAsyncOp,
								NULL); // flags
	
	if( hReturn != E_PENDING && FAILED(hReturn) ) {
		return -1;
	}

	return(hReturn);
}

//-----------------------------------------------------------------------------
// Name: hrSendPeerMessage()
// Desc: Used to send messages to other players via Direct Play
//-----------------------------------------------------------------------------
HRESULT	hrSendPeerMessage( int player, DWORD dwMessageType, PVOID pMsgBuffer )
{
	DPNHANDLE		hAsync;
    DWORD			dwLength;
	DPN_BUFFER_DESC bufferDesc;
	PacketGeneric	*PGen;

	// Cast to a generic packet to get the size
	PGen = (PacketGeneric*)pMsgBuffer;
	dwLength = PGen->dwSize;

	// Return if an empty packet
	if( dwLength == 0 ) 
		return(0);

	bufferDesc.dwBufferSize = dwLength;
	bufferDesc.pBufferData  = (BYTE*)pMsgBuffer;

	// Broadcast message
	if( player == -1 )
		g_pDP->SendTo( DPNID_ALL_PLAYERS_GROUP, &bufferDesc, 1, 0, NULL, &hAsync, DPNSEND_NOLOOPBACK | DPNSEND_GUARANTEED);
	// Send to a particular player
	else
		g_pDP->SendTo( PlayerInfo[player].iPlayerID, &bufferDesc, 1, 0, NULL, &hAsync, DPNSEND_GUARANTEED );

	return S_OK;
}

//-----------------------------------------------------------------------------
// Name: iAddPlayer()
// Desc: Adds a player to the rendering system
//-----------------------------------------------------------------------------
int iAddPlayer(DPNID dpid, char *szName)
{
	int i;

	for( i = 0 ; i < MAX_PLAYERS ; i++ ) {
		if( !PlayerInfo[i].bActive ) {
			break;
		}
	}	
	// No free slots
	if( i == MAX_PLAYERS )
		return(-1);

	PlayerInfo[i].bActive = 1;
	PlayerInfo[i].iPlayerID = dpid;
	PlayerInfo[i].sBallOnValue = RED_VALUE;
	PlayerInfo[i].sScore = 0;
	strcpy(PlayerInfo[i].szPlayerName,szName);
	return(i);
}

//-----------------------------------------------------------------------------
// Name: iGetPlayerID()
// Desc: Returns the player slot that matches the passed ID
//-----------------------------------------------------------------------------
int iGetPlayerID(unsigned int dpid)
{
	int i;

	// Remove Player from list
	for( i = 0 ; i < MAX_PLAYERS ; i++ ) {
		if( PlayerInfo[i].bActive ) {
			if( PlayerInfo[i].iPlayerID == dpid ) {
				return(i);
			}
		}
	}
	return(0);
}

//-----------------------------------------------------------------------------
// Name: vUpdateNetwork()
// Desc: Updates (or initializes) the network
//-----------------------------------------------------------------------------
void vUpdateNetwork( HWND hWnd )
{	
	if (bChooseGameType && bNetworkGame && bHostGame)
	{
		// Start host session
		hrHostGame(hWnd);
		bChooseGameType = FALSE;
		bConnecting = TRUE;
		sprintf(szSystemMessage, "Waiting for other player...");
	}
	else if (bChooseGameType && bNetworkGame && !bHostGame)
	{
		// Join a game
		hrJoinGame(hWnd);
		bChooseGameType = FALSE;
		bConnecting = TRUE;
		sprintf(szSystemMessage, "Connecting to host...");
	}
}