#include "Object3DClass.h"

// Constructor
Object3DClass::Object3DClass(void)
{
	pMesh = NULL;	
	pMeshMaterials = NULL;
	pD3DXMtrlBuffer = NULL;
	pd3dDevice = NULL;
}

// Destructor
Object3DClass::~Object3DClass(void)
{
	vCleanup();
}

//-----------------------------------------------------------------------------
// Name: Object3DClass::hLoad()
// Desc: Loads 3D data for the materials, geometry, and textures 
//-----------------------------------------------------------------------------
HRESULT Object3DClass::hLoad(char *szName,LPDIRECT3DDEVICE8 p3d)
{
	HRESULT			hResult = S_OK;
	int				i;
	D3DXMATERIAL*	d3dxMaterials;

	// Set the IDirect3DDevice8 pointer to the internal one
	pd3dDevice = p3d;
	// Initialize number of materials to zero
	dwNumMaterials = 0;
	// Load Login Geometry
	if( FAILED( D3DXLoadMeshFromX( 
		szName, 
		D3DXMESH_SYSTEMMEM, 
		pd3dDevice, 
		NULL, 
		&pD3DXMtrlBuffer, 
		&dwNumMaterials, 
		&pMesh ) ) ) {
		return(-1);
	}
	// Set temp pointer for easier to read code
	d3dxMaterials = (D3DXMATERIAL*)pD3DXMtrlBuffer->GetBufferPointer();
	// Create the materials and textures
	pMeshMaterials = new D3DMATERIAL8[dwNumMaterials];
	// Load in material information and textures
	for( i = 0; i < (signed)dwNumMaterials ; i++ ) {
		// Copy the material
		pMeshMaterials[i] = d3dxMaterials[i].MatD3D;
		// Set the ambient color for the material (D3DX does not do this)
		pMeshMaterials[i].Ambient = pMeshMaterials[i].Diffuse;
	}
	SAFE_RELEASE(pD3DXMtrlBuffer);

	return( hResult );
}

//-----------------------------------------------------------------------------
// Name: Object3DClass::vDisplayXYZ()
// Desc: Displays the 3D object at given cordinates and rotation 
//-----------------------------------------------------------------------------
void Object3DClass::vDisplayXYZ(float x, float y, float z, float rx, float ry, float rz)
{
	D3DXMATRIX	matWorld,matRotation,matTranslation,matScale;
	int			mat;
	
	// Set default translation
	D3DXMatrixIdentity( &matTranslation );
	// Rotate on X
	D3DXMatrixRotationX( &matRotation, rx/RADIAN_TO_DEGREES );
	D3DXMatrixMultiply(&matTranslation,&matRotation,&matTranslation);
	// Rotate on Y
	D3DXMatrixRotationY( &matRotation, ry/RADIAN_TO_DEGREES );
	D3DXMatrixMultiply(&matTranslation,&matRotation,&matTranslation);
	// Rotate on Z
	D3DXMatrixRotationZ( &matRotation, rz/RADIAN_TO_DEGREES );
	D3DXMatrixMultiply(&matTranslation,&matRotation,&matTranslation);
	// Move to X,Y,Z coordinates
	matTranslation._41 = x;
	matTranslation._42 = y;
	matTranslation._43 = z;
	// Set the matrix
	pd3dDevice->SetTransform( D3DTS_WORLD, &matTranslation );

	// Display each material
	for( mat = 0 ; mat < (signed)dwNumMaterials ; mat++ ) {
		pd3dDevice->SetMaterial( &pMeshMaterials[mat] );
		// Draw the mesh subset
		pMesh->DrawSubset( mat );
	}
}

//-----------------------------------------------------------------------------
// Name: Object3DClass::vDisplayXYZQuat()
// Desc: Displays the 3D object at given cordinates and rotation (quaternion)
//-----------------------------------------------------------------------------
void Object3DClass::vDisplayXYZQuat(float x, float y, float z, float rx, float ry, float rz)
{
	D3DXMATRIX		matPosition,matRotation,matTranslation;
	D3DXQUATERNION	quatRotation;
	int				mat;
	
	D3DXMatrixIdentity(&matPosition);
	// Step 1
	D3DXMatrixTranslation(&matTranslation,x,y,z);
	D3DXMatrixMultiply(&matPosition,&matTranslation,&matPosition);
	// Step 2
	D3DXQuaternionRotationYawPitchRoll(&quatRotation,rx/RADIAN_TO_DEGREES,ry/RADIAN_TO_DEGREES,rz/RADIAN_TO_DEGREES);
	// Step 3
	D3DXMatrixRotationQuaternion(&matRotation,&quatRotation);
	// Step 4
	D3DXMatrixMultiply(&matPosition,&matRotation,&matPosition);
	// Set the matrix
	pd3dDevice->SetTransform( D3DTS_WORLD, &matPosition );

	// Display each material
	for( mat = 0 ; mat < (signed)dwNumMaterials ; mat++ ) {
		pd3dDevice->SetMaterial( &pMeshMaterials[mat] );
		// Draw the mesh subset
		pMesh->DrawSubset( mat );
	}
}

//-----------------------------------------------------------------------------
// Name: Object3DClass::vCleanup()
// Desc: Releases all memory associated with the object
//-----------------------------------------------------------------------------
void Object3DClass::vCleanup(void)
{
	SAFE_RELEASE(pMesh);
	SAFE_DELETE_ARRAY(pMeshMaterials)
	SAFE_RELEASE(pD3DXMtrlBuffer);
}
