//###################################################################################
//#																					#
//#			Cue object class														#
//#				- Ales Daneu, 06/11/2002											#
//#																					#
//###################################################################################
#include "CueClass.h"

//-----------------------------------------------------------------------------
// Name: CueClass()
// Desc: Constructor
//-----------------------------------------------------------------------------
CueClass::CueClass()
{
	vecCueRotation = ORIGINAL_ROT;
}

//-----------------------------------------------------------------------------
// Name: vSetPosition(*pVec)
// Desc: Set position of cue and set visibility to true
//-----------------------------------------------------------------------------
void CueClass::vSetPosition(D3DXVECTOR3 *pVec)
{
	vecCuePosition = *pVec;
	bCueVisibility = true;
}

//-----------------------------------------------------------------------------
// Name: vecGetPosition()
// Desc: Get cue position
//-----------------------------------------------------------------------------
D3DXVECTOR3 CueClass::vecGetPosition(void)
{
	return vecCuePosition;
}

//-----------------------------------------------------------------------------
// Name: vAbsoluteDegrees()
// Desc: Change degrees to be in range 0 - 360
//-----------------------------------------------------------------------------
void CueClass::vAbsoluteDegrees(void)
{
	if (vecCueRotation.x < 0.0f)
		vecCueRotation.x += 360.0f;
	if (vecCueRotation.x > 360.0f)
		vecCueRotation.x -= 360.0f;

	if (vecCueRotation.y < 0.0f)
		vecCueRotation.y += 360.0f;
	if (vecCueRotation.y > 360.0f)
		vecCueRotation.y -= 360.0f;
}

//-----------------------------------------------------------------------------
// Name: vSetRotation(*pVec)
// Desc: Set cue rotation
//-----------------------------------------------------------------------------
void CueClass::vSetRotation(D3DXVECTOR3 *pVec)
{
	vecCueRotation = *pVec;
	// Set absolute values for degrees
	vAbsoluteDegrees();
}

//-----------------------------------------------------------------------------
// Name: vRotatePosition(idx,idy)
// Desc: Incremental rotation by given values
//-----------------------------------------------------------------------------
void CueClass::vRotatePosition(int idx, int idy)
{
	// Add rotation
	vecCueRotation += D3DXVECTOR3( -(float)idx/PIXEL_TO_DEGREES, -(float)idy/PIXEL_TO_DEGREES , 0.0f );
	// Set absolute values for degrees
	vAbsoluteDegrees();
}

//-----------------------------------------------------------------------------
// Name: vecGetRotation()
// Desc: Get current cue rotation
//-----------------------------------------------------------------------------
D3DXVECTOR3 CueClass::vecGetRotation(void)
{
	return vecCueRotation;
}

//-----------------------------------------------------------------------------
// Name: vDisplayXYZ()
// Desc: Display cue on scene (if visible)
//-----------------------------------------------------------------------------
void CueClass::vDisplayXYZ(void)
{
	if (bCueVisibility)
		Object3DClass::vDisplayXYZQuat(vecCuePosition.x, vecCuePosition.y, vecCuePosition.z,
										vecCueRotation.x, vecCueRotation.y, vecCueRotation.z);
}                                                                                                                                   

//-----------------------------------------------------------------------------
// Name: vPerformShot(*pCueBall,sShotPower)
// Desc: Perform shot (stroke)
//-----------------------------------------------------------------------------
void CueClass::vPerformShot( BallClass *pCueBall , const short sShotPower )
{
	float		fXm;
	float		fYm;
	float		fZm;
	D3DXVECTOR3 vecDirection;

	// Move cue forward along it's axis
	fXm = -(float)(CUE_DISTANCE * sin(vecCueRotation.x / RADIAN_TO_DEGREES));
	fYm = (float)(CUE_DISTANCE * sin(vecCueRotation.y / RADIAN_TO_DEGREES));
	fZm = -(float)(CUE_DISTANCE * cos(vecCueRotation.x / RADIAN_TO_DEGREES));
	vecCuePosition += D3DXVECTOR3( fXm , fYm , fZm );
	// Set cue ball movement
	vecDirection = -D3DXVECTOR3((float)sin(vecCueRotation.x / RADIAN_TO_DEGREES),
								0.0f,
								(float)cos(vecCueRotation.x / RADIAN_TO_DEGREES));	
	pCueBall->vSetBallMovement(&vecDirection);
	pCueBall->vSetBallSpeed(sShotPower);
	// Cue is invisible while balls are moving 
	bCueVisibility = false;
}

//-----------------------------------------------------------------------------
// Name: vCheckRotationUp()
// Desc: Check maximum angle between table and cue
//-----------------------------------------------------------------------------
void CueClass::vCheckRotationUp(void)
{
	// At most 30 degrees rotation up about X axis is allowed
	if (vecCueRotation.y < MAX_X_ROTATION_UP)
		vecCueRotation.y = MAX_X_ROTATION_UP;
}

//-----------------------------------------------------------------------------
// Name: vBallCollisionDetection(*pBall)
// Desc: Collision detection between cue and given ball
//-----------------------------------------------------------------------------
void CueClass::vBallCollisionDetection(BallClass *pBall)
{
	float	fDiffZ;
	float	fDiffX;
	float	fDiff;
	float	fAlfa;
	float	fGamma;
	float	fBeta;
	bool	bCollX = false;

	// Collision from side
	fDiff = D3DXVec3Length(&(pBall->vecGetPosition() - vecCuePosition));
	// Ball falls in cue range
	if (fDiff <= CUE_LENGTH)
	{
		// Calculate the angle of given ball, where cue ball is in the center of coordinate system.
		fDiffZ = pBall->vecGetPosition().z - vecCuePosition.z;
		fDiffX = pBall->vecGetPosition().x - vecCuePosition.x;
		if ((fDiffZ == 0.0f) && (fDiffX > 0.0f))
			fGamma = 90.0f;
		else if ((fDiffZ == 0.0f) && (fDiffX < 0.0f))
			fGamma = 270.0f;
		else if ((fDiffX == 0.0f) && (fDiffZ > 0.0f))
			fGamma = 360.0f;
		else if ((fDiffX == 0.0f) && (fDiffZ < 0.0f))
			fGamma = 180.0f;
		else
		{
			fGamma = (float)atan(fabs(fDiffZ) / fabs(fDiffX)) * RADIAN_TO_DEGREES;
			if ((fDiffZ > 0.0f) && (fDiffX > 0.0f))
				fGamma = 90.0f - fGamma;
			else if (fDiffZ > 0.0f)
				fGamma = 270.0f + fGamma;
			else if (fDiffX > 0.0f)
				fGamma = 90.0f + fGamma;
			else
				fGamma = 270.0f - fGamma;
		}		
		// Calculate minimum angle between cue and ball
		fAlfa = (float)asin((CUE_RADIUS + BALL_RADIUS) / fDiff) * RADIAN_TO_DEGREES;
		// Check if actual angle conforms to that
		fBeta = fGamma - vecCueRotation.x;
		bCollX = (fabs(fBeta) < fAlfa);
	}
	// Collision from top
	fDiff = D3DXVec3Length(&(pBall->vecGetPosition() - vecCuePosition));
	// Ball falls in cue range and it collides with cue from side
	if ((fDiff <= CUE_LENGTH) && bCollX)
	{
		// Set new cue rotation up to avoid ball
		fAlfa = 360.0f - (float)asin((CUE_RADIUS + BALL_RADIUS) / fDiff) * RADIAN_TO_DEGREES;
		if (vecCueRotation.y > fAlfa)
			vecCueRotation.y = fAlfa;
	}
}