//###################################################################################
//#																					#
//#			Ball object class														#
//#				- Ales Daneu, 06/11/2002											#
//#																					#
//###################################################################################
#include "BallClass.h"

//-----------------------------------------------------------------------------
// Name: vSetPosition(*pVec)
// Desc: Set position of the ball
//-----------------------------------------------------------------------------
void BallClass::vSetPosition(D3DXVECTOR3 *pVec)
{
	vecBallPosition = *pVec;
}

//-----------------------------------------------------------------------------
// Name: vecGetPosition()
// Desc: Return position of the ball
//-----------------------------------------------------------------------------
D3DXVECTOR3 BallClass::vecGetPosition(void)
{
	return vecBallPosition;
}

//-----------------------------------------------------------------------------
// Name: vUpdatePosition(*pVec)
// Desc: Update position of ball (pVec is subtracted from current position)
//-----------------------------------------------------------------------------
void BallClass::vUpdatePosition(D3DXVECTOR3 *pVec)
{
	vecBallPosition -= *pVec;
}

//-----------------------------------------------------------------------------
// Name: vPruneBallMovement()
// Desc: Avoid computational errors and it's consequences
//-----------------------------------------------------------------------------
void BallClass::vPruneBallMovement(void)
{
	bool bMoveX = true;
	bool bMoveY = true;
	bool bMoveZ = true;

	// If values are smaller than threshold set them to 0.0f
	if (fabs(vecBallMovement.x) < EPSILON)
	{
		vecBallMovement.x = 0.0f;
		bMoveX = false;
	}
	if (fabs(vecBallMovement.y) < EPSILON)
	{
		vecBallMovement.y = 0.0f;
		bMoveY = false;
	}
	if (fabs(vecBallMovement.z) < EPSILON)
	{
		vecBallMovement.z = 0.0f;
		bMoveZ = false;
	}
	// Check if ball is still moving
	bIsBallMoving = bMoveX || bMoveY || bMoveZ;	
}

//-----------------------------------------------------------------------------
// Name: vSetBallMovement(*pVec)
// Desc: Set movement for the ball
//-----------------------------------------------------------------------------
void BallClass::vSetBallMovement(D3DXVECTOR3 *pVec)
{
	// Set movement direction
	vecBallMovement = *pVec;
	// Get rid of computational errors
	vPruneBallMovement();
}

//-----------------------------------------------------------------------------
// Name: vecGetBallMovement()
// Desc: Get current movement of the ball
//-----------------------------------------------------------------------------
D3DXVECTOR3 BallClass::vecGetBallMovement(void)
{
	return vecBallMovement;
}

//-----------------------------------------------------------------------------
// Name: vSetBallSpeed(fBallSpeed)
// Desc: Mulitply current movement vector by given value
//-----------------------------------------------------------------------------
void BallClass::vSetBallSpeed(float fBallSpeed)
{
	// Set movement and store speed
	vecBallMovement *= fBallSpeed;
	fCurrentBallSpeed = fBallSpeed;
	// Prune small values
	vPruneBallMovement();
}

//-----------------------------------------------------------------------------
// Name: vSetValue(sValue)
// Desc: Set value (points) of the ball
//-----------------------------------------------------------------------------
void BallClass::vSetValue(const short sValue)
{
	sBallValue = sValue;
}

//-----------------------------------------------------------------------------
// Name: sGetValue()
// Desc: Get value (points) of the ball
//-----------------------------------------------------------------------------
short BallClass::sGetValue(void)
{
	return sBallValue;
}

//-----------------------------------------------------------------------------
// Name: fGetBallSpeed()
// Desc: Get current speed of the ball
//-----------------------------------------------------------------------------
float BallClass::fGetBallSpeed(void)
{
	return fCurrentBallSpeed;
}

//-----------------------------------------------------------------------------
// Name: bGetIsBallMoving()
// Desc: Check if ball is moving
//-----------------------------------------------------------------------------
bool BallClass::bGetIsBallMoving(void)
{
	return bIsBallMoving;
}

//-----------------------------------------------------------------------------
// Name: bGetCollisionDetectionDone()
// Desc: Check if collision detection was done in this step
//-----------------------------------------------------------------------------
bool BallClass::bGetCollisionDetectionDone(void)
{
	return bCollisionDetectionDone;
}

//-----------------------------------------------------------------------------
// Name: vSetPotted(bValue)
// Desc: Set flag to mark ball as potted/not potted
//-----------------------------------------------------------------------------
void BallClass::vSetPotted(const bool bValue)
{
	bPotted = bValue;
	// If potted then mark the ball as non moving
	if (bValue)
		bIsBallMoving = !bValue;
}

//-----------------------------------------------------------------------------
// Name: bGetPotted()
// Desc: Get value of potted flag
//-----------------------------------------------------------------------------
bool BallClass::bGetPotted(void)
{
	return bPotted;
}

//-----------------------------------------------------------------------------
// Name: vSetReposition(bValue)
// Desc: Set ball to be respotted for next stroke
//-----------------------------------------------------------------------------
void BallClass::vSetReposition(const bool bValue)
{
	bReposition = bValue;
}
	
//-----------------------------------------------------------------------------
// Name: vReposition()
// Desc: Respot the ball if it is marked so
//-----------------------------------------------------------------------------
void BallClass::vReposition(void)
{
	int			i;
	int			j;
	float		fMultiplier;
	bool		bCollided;
	D3DXVECTOR3	vecSavedPosition;

	if (bReposition)
	{
		// Ball needs to be respotted
		bReposition = false;
		bPotted = false;
		vecBallMovement = D3DXVECTOR3(0.0f, 0.0f, 0.0f);
		bIsBallMoving = false;
		fCurrentBallSpeed = 0.0f;
		// Try to respot the ball on it's original spot
		switch (sBallValue)
		{
			case BLACK_VALUE:
				vecBallPosition = BLACK_POS;
				break;
			case PINK_VALUE:
				vecBallPosition = PINK_POS;
				break;
			case BLUE_VALUE:
				vecBallPosition = BLUE_POS;
				break;
			case BROWN_VALUE:
				vecBallPosition = BROWN_POS;
				break;
			case GREEN_VALUE:
				vecBallPosition = GREEN_POS;
				break;
			case YELLOW_VALUE:
				vecBallPosition = YELLOW_POS;
				break;
			case WHITE_VALUE:
				vecBallPosition = WHITE_POS;
		}
		// Check if ball's original spot is not occupied
		bCollided = false;
		for (j = 0; j < TOTAL_BALLS - 1; j++)
		{
			if (vBallCollisionDetection(arrCloseNeighbours[j]))
			{
				bCollided = true;
				break;
			}
		}
		// Search for next available spot if original is occupied (only colors)
		if (bCollided && (sBallValue > RED_VALUE))
		{
			for (i = BLACK_VALUE; i > RED_VALUE; i--)
			{
				switch (i)
				{
					case BLACK_VALUE:
						vecBallPosition = BLACK_POS;
						break;
					case PINK_VALUE:
						vecBallPosition = PINK_POS;
						break;
					case BLUE_VALUE:
						vecBallPosition = BLUE_POS;
						break;
					case BROWN_VALUE:
						vecBallPosition = BROWN_POS;
						break;
					case GREEN_VALUE:
						vecBallPosition = GREEN_POS;
						break;
					case YELLOW_VALUE:
						vecBallPosition = YELLOW_POS;
				}
				// Check if this spot is free
				bCollided = false;
				for (j = 0; j < TOTAL_BALLS - 1; j++)
				{
					if (vBallCollisionDetection(arrCloseNeighbours[j]))
					{
						bCollided = true;
						break;
					}
				}
				if (!bCollided)
					break;
			}
		}
		// No free spots
		if (bCollided)
		{
			switch (sBallValue)
			{
				case BLACK_VALUE:
					vecBallPosition = BLACK_POS;
					fMultiplier = -1.0f;
					break;
				case PINK_VALUE:
					vecBallPosition = PINK_POS;
					fMultiplier = -1.0f;
					break;
				case BLUE_VALUE:
					vecBallPosition = BLUE_POS;
					fMultiplier = -1.0f;
					break;
				case BROWN_VALUE:
					vecBallPosition = BROWN_POS;
					fMultiplier = 1.0f;
					break;
				case GREEN_VALUE:
					vecBallPosition = GREEN_POS;
					fMultiplier = 1.0f;
					break;
				case YELLOW_VALUE:
					vecBallPosition = YELLOW_POS;
					fMultiplier = 1.0f;
					break;
				case WHITE_VALUE:
					vecBallPosition = WHITE_POS;
					fMultiplier = -1.0f;
			}
			// Move ball nearest to it's original position
			vecSavedPosition = vecBallPosition;
			while (bCollided)
			{
				bCollided = false;
				vecBallPosition = vecSavedPosition + fMultiplier * D3DXVECTOR3(0.0f, 0.0f, 0.01f);
				vecSavedPosition = vecBallPosition;
				for (j = 0; j < TOTAL_BALLS - 1; j++)
				{
					if (vBallCollisionDetection(arrCloseNeighbours[j]))
					{
						bCollided = true;
						break;
					}
				}
			}
		}
	}
}

//-----------------------------------------------------------------------------
// Name: vAddToList()
// Desc: Add a ball to sorted neighbours list (used for CD)
//-----------------------------------------------------------------------------
void BallClass::vAddToList(BallClass *pBall)
{
	int			i;
	int			j;
	float		fDistance;

	// In case list is empty
	if (arrCloseNeighbours[0] == NULL)
		arrCloseNeighbours[0] = pBall;
	else
	{
		// Calculate the distance (closest balls are on start of the list)
		fDistance = D3DXVec3Length(&(vecBallPosition - pBall->vecGetPosition()));
		for (i = 0; i < TOTAL_BALLS - 1; i++)
		{
			if (arrCloseNeighbours[i] == NULL)
				break;
			if (fDistance <= D3DXVec3Length(&(vecBallPosition - arrCloseNeighbours[i]->vecGetPosition())))
			{
				// position in list for given ball is found, now we need to move other members of the list
				for (j = TOTAL_BALLS - 2; j > i; j--)
					arrCloseNeighbours[j] = arrCloseNeighbours[j-1];
				break;
			}
		}
		// Store the ball on its position in list
		arrCloseNeighbours[i] = pBall;
	}
}

//-----------------------------------------------------------------------------
// Name: vResetList()
// Desc: Reset neighbours list (used for CD)
//-----------------------------------------------------------------------------
void BallClass::vResetList(void)
{
	int i;

	// Erase neighbours list
	for (i = 0; i < TOTAL_BALLS - 1; i++)
		arrCloseNeighbours[i] = NULL;
	// Mark that collision detection was not done yet
	bCollisionDetectionDone = false;
}

//-----------------------------------------------------------------------------
// Name: vDisplayXYZ()
// Desc: Used to display ball (only if it is not potted)
//-----------------------------------------------------------------------------
void BallClass::vDisplayXYZ(void)
{
	if (!bPotted)
		Object3DClass::vDisplayXYZ(vecBallPosition.x, vecBallPosition.y, vecBallPosition.z);
}

//-----------------------------------------------------------------------------
// Name: vAdvance()
// Desc: Move the ball on scene (also performs CD)
//-----------------------------------------------------------------------------
short BallClass::vAdvance(void)
{
	int			i;
	int			j;
	short		sResult;
	D3DXVECTOR3 vecMovementStep;
	D3DXVECTOR3 vecIncrementedPosition;

	// -1 is returned if no other ball was hit with this ball
	sResult = -1;
	// Only perform collision detection for balls that are not potted and are moving
	if (!bPotted && bIsBallMoving)
	{
		fCurrentBallSpeed *= 0.99f;
		vecBallMovement *= 0.99f;
		// Divide movement to 100 steps (to avoid features like ball moving straight through another ball etc.) 
		vecMovementStep = 0.01f * vecBallMovement;
		for (i = 1; i <= 100; i++)
		{
			vecBallPosition += vecMovementStep;
			// check if ball collides with any (unpotted) neighbour
			for (j = 0; j < TOTAL_BALLS - 1; j++)
			{
				if (!arrCloseNeighbours[j]->bGetCollisionDetectionDone() &&
					!arrCloseNeighbours[j]->bGetPotted())
					bCollisionDetectionDone = vBallCollisionDetection(arrCloseNeighbours[j]);
				if (bCollisionDetectionDone)
					break;
			}
			if (bCollisionDetectionDone)
				break;
		}
		// Get value of the ball that was hit (collided with)
		if ( bCollisionDetectionDone ) 
			sResult = arrCloseNeighbours[j]->sGetValue();
		// Prune movement
		vPruneBallMovement();
	}
	return sResult;
}

//-----------------------------------------------------------------------------
// Name: vBallsCollisionDetection()
// Desc: Do collision detection for this balls (without adavancing it in scene)
//-----------------------------------------------------------------------------
void BallClass::vBallsCollisionDetection(void)
{
	int i;

	for (i = 0; i < TOTAL_BALLS - 1; i++)
	{
		if (!arrCloseNeighbours[i]->bGetCollisionDetectionDone())
			bCollisionDetectionDone = vBallCollisionDetection(arrCloseNeighbours[i]);
		if (bCollisionDetectionDone)
			break;
	}
}

//-----------------------------------------------------------------------------
// Name: vFixOverlaps(*vecPos1,*vecPos2,*vecMovement)
// Desc: Moves balls appart when collision is detected to avoid overlaps
//-----------------------------------------------------------------------------
D3DXVECTOR3 BallClass::vFixOverlaps(D3DXVECTOR3 *vecPos1, D3DXVECTOR3 *vecPos2, D3DXVECTOR3 *vecMovement)
{
	float		fCurrentDifference;
	D3DXVECTOR3 vecMinimum;
	D3DXVECTOR3 vecMaximum;
	D3DXVECTOR3 vecCurrentPoint;

	// Search for new position along 100 units long path created by movement vector
	vecMinimum = *vecPos1;
	vecMaximum = *vecPos1 - 100 * (*vecMovement);
	vecCurrentPoint = vecMinimum;
	fCurrentDifference = D3DXVec3Length(&(*vecPos2 - vecCurrentPoint)) - 2 * BALL_RADIUS;
	// Loop until balls are not sufficiently spaced apart
	while ((float)fabs(fCurrentDifference) > 0.001f)
	{
		// Compute new candidate for final position
		if (fCurrentDifference < 0.0f)
		{
			if (vecMaximum != vecCurrentPoint)
				vecMinimum = vecCurrentPoint;
			vecCurrentPoint = vecCurrentPoint + (vecMaximum - vecMinimum) / 2;
		}
		else 
		{
			if (vecMinimum != vecCurrentPoint)
				vecMaximum = vecCurrentPoint;
			vecCurrentPoint = vecCurrentPoint - (vecMaximum - vecMinimum) / 2;
		}
		fCurrentDifference = D3DXVec3Length(&(*vecPos2 - vecCurrentPoint)) - 2 * BALL_RADIUS;
	}
	return vecCurrentPoint;
}

//-----------------------------------------------------------------------------
// Name: vBallCollisionDetection(*pBall)
// Desc: Perform collision detection with another ball
//-----------------------------------------------------------------------------
bool BallClass::vBallCollisionDetection(BallClass *pBall)
{
	bool		bResult;
	float		fThisBallSpeed;
	float		fOtherBallSpeed;
	float		fOverlapLength;
	float		fOffsetLength;
	float		fLengthThis;
	float		fLengthOther;
	float		fMultiplier;
	D3DXVECTOR3	vecCollisionDirection;
	D3DXVECTOR3	vecCollisionDirectionNorm;
	D3DXVECTOR3	vecThisBallMovementNorm;
	D3DXVECTOR3	vecOtherBallMovementNorm;
	D3DXVECTOR3	vecNewThisBallMovementNorm;
	D3DXVECTOR3	vecNewOtherBallMovementNorm;

	// Compute distance between the two balls and how much they overlap
	fOffsetLength = D3DXVec3Length(&(vecBallPosition - pBall->vecGetPosition()));
	fOverlapLength = fOffsetLength - 2 * BALL_RADIUS;
	// Initialy we assume that no collision has occured
	bResult = false;
	if (fOverlapLength < 0.0f)
	{
		// Balls overlap, therefore we have a collision situation
		bResult = true;
		fOverlapLength = (float)fabs(fOverlapLength);
		// Lengths of ball movement vectors
		fLengthThis = D3DXVec3Length(&vecBallMovement);
		fLengthOther = D3DXVec3Length(&pBall->vecGetBallMovement());
		if ((fLengthThis > 0.0f) && (fLengthOther > 0.0f))
		{
			// Both balls are moving
			D3DXVec3Normalize(&vecThisBallMovementNorm, &vecBallMovement);
			D3DXVec3Normalize(&vecOtherBallMovementNorm, &pBall->vecGetBallMovement());
			// Overlap is fixed by moving back the ball that has greater speed 
			if (fLengthThis > fLengthOther)
				vSetPosition(&vFixOverlaps(&vecBallPosition, &pBall->vecGetPosition(), &vecThisBallMovementNorm));
			else
				pBall->vSetPosition(&vFixOverlaps(&pBall->vecGetPosition(), &vecBallPosition, &vecOtherBallMovementNorm));
			// Now that the balls are properly positioned we can compute collision direction		
			vecCollisionDirection = vecBallPosition - pBall->vecGetPosition();
			D3DXVec3Normalize(&vecCollisionDirectionNorm, &vecCollisionDirection);
			// Different cases of collision
			// Balls moving under same angles (towards each other or in the same direction)
			if ((fabs(fabs(vecOtherBallMovementNorm.x) - fabs(vecThisBallMovementNorm.x)) < EPSILON) &&
				(fabs(fabs(vecOtherBallMovementNorm.z) - fabs(vecThisBallMovementNorm.z)) < EPSILON))
			{
				// Balls moving into directly opposite directions
				if (((vecThisBallMovementNorm.x * vecOtherBallMovementNorm.x) < 0.0f) || 
					((vecThisBallMovementNorm.z * vecOtherBallMovementNorm.z) < 0.0f))
				{
					// Balls collided on their front
					if (fabs(fabs(vecCollisionDirectionNorm.x) - fabs(vecThisBallMovementNorm.x)) < EPSILON)
					{
						vSetBallMovement(&vecOtherBallMovementNorm);
						pBall->vSetBallMovement(&vecThisBallMovementNorm);
					}
					// Balls collided on their side
					else
					{
						D3DXVec3Normalize(&vecNewThisBallMovementNorm, &(vecThisBallMovementNorm + vecCollisionDirectionNorm));
						vSetBallMovement(&vecNewThisBallMovementNorm);
						D3DXVec3Normalize(&vecNewOtherBallMovementNorm, &(vecOtherBallMovementNorm - vecCollisionDirectionNorm));
						pBall->vSetBallMovement(&vecNewOtherBallMovementNorm);	
					}
				}
				// Balls are moving into same direction
				else			
				{
					// Balls collided on their front
					if (fabs(fabs(vecCollisionDirectionNorm.x) - fabs(vecThisBallMovementNorm.x)) < EPSILON)
					{
						vSetBallMovement(&vecThisBallMovementNorm);
						pBall->vSetBallMovement(&vecOtherBallMovementNorm);
					}
					// Balls collided on their side
					else
					{
						D3DXVec3Normalize(&vecNewThisBallMovementNorm, &(vecThisBallMovementNorm + vecCollisionDirectionNorm));
						vSetBallMovement(&vecNewThisBallMovementNorm);
						D3DXVec3Normalize(&vecNewOtherBallMovementNorm, &(vecOtherBallMovementNorm - vecCollisionDirectionNorm));
						pBall->vSetBallMovement(&vecNewOtherBallMovementNorm);					
					}
				}		
			}
			// Balls moving under different angles, difference between movement vectors is more than 90 degrees                                                                                                                                       
			else if (((vecOtherBallMovementNorm.z * vecThisBallMovementNorm.z > 0.0f) &&
					(fabs(acos(vecOtherBallMovementNorm.x) - acos(vecThisBallMovementNorm.x)) > D3DX_PI/2)) ||
					((vecOtherBallMovementNorm.z * vecThisBallMovementNorm.z < 0.0f) &&
					(fabs(asin(vecOtherBallMovementNorm.z) - asin(vecThisBallMovementNorm.z)) > D3DX_PI/2)))
			{
				// Balls collide slightly from side
				if (fabs(fabs(vecCollisionDirectionNorm.x) - fabs(vecThisBallMovementNorm.x)) >= cos(D3DX_PI/4)) 
				{
					D3DXVec3Normalize(&vecNewThisBallMovementNorm, &(vecThisBallMovementNorm + vecCollisionDirectionNorm));
					vSetBallMovement(&vecNewThisBallMovementNorm);
				}
				// Balls collide directly (not from side)
				else
				{
					D3DXVec3Normalize(&vecNewThisBallMovementNorm, &(-vecThisBallMovementNorm + vecCollisionDirectionNorm));
					vSetBallMovement(&vecNewThisBallMovementNorm);
				}
				// Balls collide slightly from side
				if (fabs(fabs(vecCollisionDirectionNorm.x) - fabs(vecOtherBallMovementNorm.x)) >= cos(D3DX_PI/4))
				{
					D3DXVec3Normalize(&vecNewOtherBallMovementNorm, &(vecOtherBallMovementNorm - vecCollisionDirectionNorm));
					pBall->vSetBallMovement(&vecNewOtherBallMovementNorm);
				}
				// Balls collide directly (not from side)
				else
				{
					D3DXVec3Normalize(&vecNewOtherBallMovementNorm, &(-vecOtherBallMovementNorm - vecCollisionDirectionNorm));
					pBall->vSetBallMovement(&vecNewOtherBallMovementNorm);
				}
			}
			// Balls moving under different angles, difference between them is less than 90 degrees
			else
			{
				// Balls collide from side
				if (fabs(fabs(vecCollisionDirectionNorm.x) - fabs(vecThisBallMovementNorm.x)) > EPSILON) 
				{
					D3DXVec3Normalize(&vecNewThisBallMovementNorm, &(vecThisBallMovementNorm + vecCollisionDirectionNorm));
					vSetBallMovement(&vecNewThisBallMovementNorm);
				}
				// Balls collide directly (not from side)
				else
					vSetBallMovement(&vecCollisionDirectionNorm);
				// Balls collide from side
				if (fabs(fabs(vecCollisionDirectionNorm.x) - fabs(vecOtherBallMovementNorm.x)) > EPSILON)
				{
					D3DXVec3Normalize(&vecNewOtherBallMovementNorm, &(vecOtherBallMovementNorm - vecCollisionDirectionNorm));
					pBall->vSetBallMovement(&vecNewOtherBallMovementNorm);
				}
				// Balls collide directly (not from side)
				else
					pBall->vSetBallMovement(&(-vecCollisionDirectionNorm));
			}
			// Set new speeds for balls
			fThisBallSpeed = fGetBallSpeed();
			fOtherBallSpeed = pBall->fGetBallSpeed();
			vSetBallSpeed(0.98f * fOtherBallSpeed);
			pBall->vSetBallSpeed(0.98f * fThisBallSpeed);
		}
		// Only this ball is moving
		else if (fLengthThis > 0.0f)
		{
			// Fix ball overlaps
			D3DXVec3Normalize(&vecThisBallMovementNorm, &vecBallMovement);
			vSetPosition(&vFixOverlaps(&vecBallPosition, &pBall->vecGetPosition(), &vecThisBallMovementNorm));
			// Set new movement direction
			vecCollisionDirection = vecBallPosition - pBall->vecGetPosition();
			D3DXVec3Normalize(&vecCollisionDirectionNorm, &vecCollisionDirection);
			if (fabs(fabs(vecCollisionDirectionNorm.x) - fabs(vecThisBallMovementNorm.x)) >= cos(D3DX_PI/4))
			{
				D3DXVec3Normalize(&vecNewThisBallMovementNorm, &(vecThisBallMovementNorm + vecCollisionDirectionNorm));
				vSetBallMovement(&vecNewThisBallMovementNorm);
			}
			else
			{
				D3DXVec3Normalize(&vecNewThisBallMovementNorm, &(-vecThisBallMovementNorm + vecCollisionDirectionNorm));
				vSetBallMovement(&vecNewThisBallMovementNorm);
			}
			pBall->vSetBallMovement(&(-vecCollisionDirectionNorm));
			// Set new speeds for balls
			fThisBallSpeed = fGetBallSpeed();
			vSetBallSpeed(0.24f * fThisBallSpeed);
			pBall->vSetBallSpeed(0.74f * fThisBallSpeed);
		}
		// Only the other Ball is moving
		else if (fLengthOther > 0.0f)
		{
			// Fix overlaps
			D3DXVec3Normalize(&vecOtherBallMovementNorm, &pBall->vecGetBallMovement());
			pBall->vSetPosition(&vFixOverlaps(&pBall->vecGetPosition(), &vecBallPosition, &vecOtherBallMovementNorm));
			// Set new movement direction
			vecCollisionDirection = vecBallPosition - pBall->vecGetPosition();
			D3DXVec3Normalize(&vecCollisionDirectionNorm, &vecCollisionDirection);
			if (fabs(fabs(vecCollisionDirectionNorm.x) - fabs(vecOtherBallMovementNorm.x)) >= cos(D3DX_PI/4))
			{
				D3DXVec3Normalize(&vecNewOtherBallMovementNorm, &(vecOtherBallMovementNorm - vecCollisionDirectionNorm));
				pBall->vSetBallMovement(&vecNewOtherBallMovementNorm);
			}
			else
			{
				D3DXVec3Normalize(&vecNewOtherBallMovementNorm, &(-vecOtherBallMovementNorm - vecCollisionDirectionNorm));
				pBall->vSetBallMovement(&vecNewOtherBallMovementNorm);
			}
			vSetBallMovement(&(vecCollisionDirectionNorm));		
			// Set new speeds for balls
			fOtherBallSpeed = pBall->fGetBallSpeed();
			vSetBallSpeed(0.74f * fOtherBallSpeed);
			pBall->vSetBallSpeed(0.24f * fOtherBallSpeed);
		}
		// No movement but balls overlap
		else
		{
			vecCollisionDirection = vecBallPosition - pBall->vecGetPosition();
			fMultiplier = fOverlapLength / D3DXVec3Length(&vecCollisionDirection);
			vUpdatePosition(&D3DXVECTOR3(-fMultiplier * vecCollisionDirection.x,
										0.0f, -fMultiplier * vecCollisionDirection.z));
		}
	}
	return bResult;
}