import file_io

def pointer_eval(p):
    "Returns the value of an AGI view pointer."
    return ord(p[0]) + 256*ord(p[1])

def make_2byte(n):
    "Returns the two-byte big-endian encoding of a number."
    return chr(n % 256) + chr(n / 256)

class Vew2Vew:
    def __init__(self,infile,outfile):
        self.agiview = file_io.FileStreamer(infile)
        self.sciview = file_io.FileBuffer(outfile)
        self.viewheader = {}
        self.loopindex = []
        self.loopheaders = []
        self.Totalcels = 0

    def process(self):
        #Read the main header
        agi = self.agiview.get(2) #Throwaway - unknown meaning (See AGI documentation)
        self.viewheader['numloops'] = ord(self.agiview.get(1)) #Number of loops
        print "Number of loops:", self.viewheader['numloops']
        agi = self.agiview.get(2) #Description position (ls,ms)
        self.viewheader['DescPos'] = ord(agi[0]) + 256 * ord(agi[1])
        if self.viewheader['DescPos'] == 0:
                print "No description"
        else:
                print "Description location:", self.viewheader['DescPos']

        #Read each loop header
        for x in range(0,self.viewheader['numloops']):
                self.agiview.jump(3+2*(x+1))
                agi = self.agiview.get(2)
                self.loopindex = self.loopindex + [ord(agi[0]) + 256*ord(agi[1])]
        print self.loopindex

        #Get cel data about each loop
        for x in range(0,self.viewheader['numloops']):
                self.loopheaders = self.loopheaders + [{}]
                self.agiview.jump(self.loopindex[x])
                agi = self.agiview.get(1)
                self.loopheaders[x]['numcels'] = ord(agi)
                self.Totalcels = self.Totalcels + self.loopheaders[x]['numcels']
        print "Total number of cels:", self.Totalcels
        print self.loopheaders

        #Read individual cel headers, and collect addresses
        print "Cel offsets:"
        cels=[]
        for x in range(0,self.viewheader['numloops']):
                for y in range(1,self.loopheaders[x]['numcels']+1):
                        cels = cels + [{}]
                        cels[-1]['loopnum'] = x
                        cels[-1]['celnum'] = y
                        cels[-1]['celpos'] = self.loopindex[x]
                        self.agiview.jump(self.loopindex[x] + 2*y - 1)
                        agi = self.agiview.get(2)
                        cels[-1]['celpos'] = cels[-1]['celpos'] + ord(agi[0]) + 256 * ord(agi[1])
                        print cels[-1]['celpos'],
        ##print cels
        #Read more cel header information
        for x in range(0,self.Totalcels):
                self.agiview.jump(cels[x]['celpos'])
                cels[x]['x'] = ord(self.agiview.get(1)) # Width of the cel
                cels[x]['y'] = ord(self.agiview.get(1)) # Height of the cel
                agi = self.agiview.get(1) # Clear colour and mirroring
                cels[x]['clarity'] = ord(agi) & 15
                cels[x]['mirror'] = (ord(agi) & 240) / 16

        #Read and translate the actual pictures
        #### File header ####
        writeviewresource = "\x80\x00" # SCI file type marker
        writeviewresource += chr(self.viewheader['numloops'] % 256) + chr(self.viewheader['numloops'] / 256) # Number of loops in view
        # Mirror info (gather from cels)
        m = 0
        cel = 0
        for x in range(0,self.viewheader['numloops']):
                if ((cels[cel]['mirror'] & 8) == 8) and ((cels[cel]['mirror'] & x) != x):
                        m = m | (2**x)
                cel += self.loopheaders[x]['numcels']
        writeviewresource += chr(m % 256) + chr(m / 256)
        del m, cel
        writeviewresource += "\x00\x00\x00\x00" # Unknown bytes (possibly extra space for mirror flags)
        # Loop indices
        numcelspassed = 0
        for x in range(0,self.viewheader['numloops']):
                # Index is 8 + 2*numloops + x*4 + 2*[number of cels so far]
                index = 8 + x*4 + 2*(self.viewheader['numloops'] + numcelspassed)
                writeviewresource += chr(index % 256) + chr(index / 256)
                numcelspassed += self.loopheaders[x]['numcels']
        del numcelspassed
        self.sciview.put(writeviewresource)
        del writeviewresource

        writecellist = []
        for x in range(0,self.viewheader['numloops']):
                # [Number of cels in loop] then [unknown]
                writecellist += [chr(self.loopheaders[x]['numcels'] % 256) + chr(self.loopheaders[x]['numcels'] / 256) + "\x00\x00"]

        writeimagecels = ""
        newlocs = {}
        for x in range(0,self.Totalcels):
                xPos = 0
                YPos = 0
                FileIndex = cels[x]['celpos'] + 3
                if (cels[x]['celpos'] in newlocs.keys()):
                        writecellist[cels[x]['loopnum']] += newlocs[cels[x]['celpos']]
                else: # (not mirrored)
                        index = 8 + 6*self.viewheader['numloops'] + 2*self.Totalcels + len(writeimagecels)
                        codedindex = chr(index % 256) + chr(index / 256)
                        writecellist[cels[x]['loopnum']] += codedindex
                        # record codedindex for later use
                        newlocs[cels[x]['celpos']] = codedindex
                        del codedindex

                        writeimagecels += chr((cels[x]['x'] * 2) % 256) + chr((cels[x]['x'] * 2) / 256)
                        writeimagecels += chr(cels[x]['y'] % 256) + chr(cels[x]['y'] / 256)
                        writeimagecels += "\x00\x00"  #(x,y)-placement modifier. Unused in AGI.
                        writeimagecels += chr(cels[x]['clarity']) + "\x00" #Clear colour for this cel.
                        while YPos != cels[x]['y'] or xPos != 0:
                                self.agiview.jump(FileIndex)
                                agi = self.agiview.get(1)
                                FileIndex = FileIndex + 1
                                if ord(agi) == 0:
                                        Length = (2*cels[x]['x'] - xPos)
                                        while Length >= 16:
                                                writeimagecels += chr(cels[x]['clarity'] | 240)
                                                Length -= 15
                                        if Length > 0:
                                                writeimagecels += chr(cels[x]['clarity'] | 16*Length)
                                        writeimagecels += "\x00"
                                        xPos = 0
                                        YPos = YPos + 1
                                else:
                                        Length = (ord(agi) & 15) * 2
                                        Colour = (ord(agi) & 240) / 16
                                        xPos = xPos + Length
                                        while Length >= 16:
                                                writeimagecels += chr(Colour | 240)
                                                Length -= 15
                                        if Length > 0:
                                                writeimagecels += chr(Colour | 16*Length)

        for x in writecellist:
                self.sciview.put(x)
        del writecellist

        self.sciview.put(writeimagecels)
        del writeimagecels

        del self.agiview, self.sciview
        
if __name__ == "__main__":
    from sys import argv
    infile,outfile = "",""
    if len(argv) > 1:
        for x in argv[1:]:
            if x[0:2] == "-i":
                if len(x) != 2:
                    infile = x[2:]
            elif x[0:2] == "-o":
                if len(x) != 2:
                    outfile = x[2:]
        if infile == "":
            infile = raw_input("Input picture file name:")
        if outfile == "":
            outfile = raw_input("Output picture file name:")
        converter = Vew2Vew(infile,outfile)
        converter.process()
    else:
        print "VEW2VEW - AGI to SCI view resource converter\n"
        print "Usage:\nvew2vew [options]\n"
        print "Valid options:"
        print "-i[filename]\tSet input file to 'filename'. Prompt if no filename given on"
        print "\t\tcommand line. Note: no spaces between '-i' and filename argument."
        print "-o[filename]\tSet output file to 'filename'. Same behaviour as '-i' if no"
        print "\t\targument given."
        a = raw_input("Press ENTER to quit.")
        a = Vew2Vew(r"\john\lang\agi2sci\vew2vew\agiview.0",r"\john\lang\agi2sci\vew2vew\view.000")
        a.process()
