C
C     SELF AND MUTUAL IMPEDANCES
C     ******************************************************************
C     THIS PROGRAM COMPUTES:
C
C        I.  SELF IMPEDANCE OF ANY LENGTH DIPOLE
C        II. MUTUAL IMPEDANCE BETWEEN TWO IDENTICAL LINEAR DIPOLES
C
C     BASED ON THE INDUCED EMF METHOD AND THE IDEAL CURRENT DISTRIBUTION
C     OF EQUATION (4-56).
C
C        I.  SELF IMPEDANCE (INPUT IMPEDANCE)
C
C            1.  BASED ON CURRENT AT THE INPUT
C                  Zin = Rin + jXin
C                  Rin = INPUT RESISTANCE [EQU. (8-60a) & (8-61a)]
C                  Xin = INPUT REACTANCE [EQU. (8-60b) & (8-61b)]
C            2.  BASED ON CURRENT MAXIMUM
C
C                  Zinm = Rinm + jXinm
C                  Rinm = SELF RESISTANCE [EQU. (8-60a)]
C                  Xinm = SELF REACTANCE [EQU. (8-60b)]
C
C        II.  MUTUAL IMPEDANCE 
C
C            1.  BASED ON CURRENT AT THE INPUT
C                  Z21i = R21i + jX21i [EQU. (8-68)]
C                  R21i = INPUT MUTUAL RESISTANCE 
C                  Xin = INPUT MUTUAL REACTANCE 
C            2. BASED ON CURRENT MAXIMUM
C                  Z21m = R21m + jX21m [EQU. (8-70)]
C                  R21m = SELF MUTUAL RESISTANCE
C                  X21m = SELF MUTUAL REACTANCE
C
C     THE DIPOLES FOR MUTUAL IMPEDANCE COMPUTATIONS MUST BE IDENTICAL
C     WITH LENGTH OF ODD MULTIPLES OF HALF WAVELENGTH. 
C
C        OPTION I.  SELF IMPEDANCE (INPUT IMPEDANCE)       
C     
C          ** ARRAY INPUT PARAMETERS
C
C          1.  LENGTH OF DIPOLE (IN WAVELENGTHS)
C          2.  RADIUS OF DIPOLE (IN WAVELENGTHS)
C
C       OPTION II.  MUTUAL IMPEDANCE
C
C         CHOICE A:  SIDE-BY-SIDE (Fig. 8.19(a))
C
C           ** ARRAY INPUT PARAMETERS:
C
C           1.  LENGTH OF THE DIPOLES (IN WAVELENGTHS)
C           2.  HORIZONTAL DISPLACEMENT OF DIPOLES (IN WAVELENGTHS)
C
C         CHOICE B: COLLINEAR (Fig. 8.19(b))
C
C          ** ARRAY INPUT PARAMETERS:
C
C          1.  LENGTH OF THE DIPOLES (IN WAVELENGTHS)
C          2.  VERTICAL DISPLACEMENT OF DIPOLES (IN WAVELENGTHS)
C
C         CHOICE C. PARALLEL-IN-ECHELON (Fig. 8.19(c))
C
C          ** ARRAY INPUT PARAMETERS:
C
C          1.  LENGTH OF THE DIPOLES (IN WAVELENGTHS)
C          2.  VERTICAL DISPLACEMENT OF DIPOLES (IN WAVELENGTHS)
C          3.  HORIZONTAL DISPLACEMENT OF DIPOLES (IN WAVELENGTHS)
C
C     ** NOTE: ALL THE INPUT PARAMETERS ARE IN WAVELENGTHS.
C     ******************************************************************
C     Written by: Anastasis C. Polycarpou, Arizona State University
C
C     ******************************************************************
      INTEGER OPTION,DEVICE
      CHARACTER*1 CONFIG
      CHARACTER*12 FILNAM
      REAL PI,K,ETA,R21I,X21I,R21M,X21M,RIN,XIN,RINM,XINM,L,A,SINKL2,TOL
      REAL D,S,H
      COMMON /CONSTANTS/ PI,K,ETA

C     ********** CONSTANT DEFINITIONS **********
C
      PI=4.0*ATAN(1.0)
      K=2.*PI
      ETA=120.*PI
      TOL=1.E-6

C     ********** CHOICE OF OUTPUT **********
C
      WRITE(6,70)
      READ(5,140,ERR=999) DEVICE

      IF(DEVICE .EQ. 1) THEN
         DEVICE=6

      ELSE IF(DEVICE .EQ. 2) THEN
         WRITE(6,80)
         READ(5,*,ERR=999) FILNAM
         OPEN(UNIT=DEVICE,FILE=FILNAM,STATUS='UNKNOWN')

      ELSE 
         WRITE(6,90)
         STOP

      ENDIF


C     ********* READING OPTION # : 1 ---> SELF; 2 ---> MUTUAL *********
C
      WRITE(6,100)
      READ(5,140,ERR=999) OPTION

      IF(OPTION .EQ. 1) THEN

         CALL SELF(L,A,RINM,XINM) ! SELF IMPEDANCE CALCULATIONS

C     ********** OUTPUTING THE SELF IMPEDANCE **********
C
         WRITE(DEVICE,190)
         WRITE(DEVICE,210) L,A

         SINKL2=SIN(K*L/2.)

         IF(ABS(SINKL2).LT.TOL)THEN
            WRITE(DEVICE,120) RINM,XINM

         ELSE
            RIN=RINM/SINKL2**2
            XIN=XINM/SINKL2**2
            WRITE(DEVICE,110) RIN,XIN,RINM,XINM

         ENDIF


      ELSE IF(OPTION .EQ. 2) THEN

C     ********* READING CONFIGURATION OPTION *********
C     A ---> SBS; B ---> COLN; C ---> PARECH
C
         WRITE(6,130)
         READ(5,*,ERR=999) CONFIG

         IF (CONFIG .EQ. 'A' .OR. CONFIG .EQ. 'a') THEN
            CALL SBS(L,D,R21M,X21M) ! SIDE-BY SIDE

         ELSE IF(CONFIG .EQ. 'B' .OR. CONFIG .EQ. 'b') THEN
            CALL COLN(L,S,R21M,X21M) ! COLLINEAR

         ELSE IF(CONFIG .EQ. 'C' .OR. CONFIG .EQ. 'c') THEN
            CALL PARECH(L,D,H,R21M,X21M) ! PARALLEL-IN-ECHELON

         ELSE
            WRITE(6,170)
            STOP

         ENDIF

C     ********** OUTPUTING THE MUTUAL IMPEDANCE **********
C         
         WRITE(DEVICE,200)
         IF(CONFIG .EQ. 'A' .OR. CONFIG .EQ. 'a') THEN
            WRITE(DEVICE,250)
            WRITE(DEVICE,220) L,D

         ELSE IF(CONFIG .EQ. 'B' .OR. CONFIG .EQ. 'b') THEN
            WRITE(DEVICE,260)
            WRITE(DEVICE,230) L,S

         ELSE IF(CONFIG .EQ. 'C' .OR. CONFIG .EQ. 'c') THEN
            WRITE(DEVICE,270)
            WRITE(DEVICE,240) L,D,H

         ENDIF
            
         SINKL2=SIN(K*L/2.)

         IF(ABS(SINKL2).LT.TOL)THEN
            WRITE(DEVICE,160) R21M,X21M

         ELSE
            R21I=R21M/SINKL2**2
            X21I=X21M/SINKL2**2
            WRITE(DEVICE,150) R21I,X21I,R21M,X21M

         ENDIF

      ELSE

         WRITE(6,180)
         STOP

      ENDIF         

C     ********** FORMAT STATEMENTS **********

 70   FORMAT(3X,'OUTPUT DEVICE OPTION',/,6X
     $      ,'OPTION (1): SCREEN',/,6X,'OPTION (2): OUTPUT FILE',//,3X
     $      ,'OUTPUT DEVICE = ',$)

 80   FORMAT(3X,'INPUT THE DESIRED OUTPUT FILNAME (in single quotes) = '
     &      ,$)

 90   FORMAT(/,3X,'*** ERROR ***',/,3X
     $      ,'OUTPUTING DEVICE NUMBER SHOULD BE EITHER 1 OR 2')

 100  FORMAT(/,3X,'SELF OR MUTUAL IMPEDANCE OPTION',/,6X
     $      ,'OPTION (1): SELF IMPEDANCE',/,6X
     $      ,'OPTION (2): MUTUAL IMPEDANCE',//,3X,'OPTION NUMBER = ',$)

 110     FORMAT(/,3X,'RIN = ',F12.6,/,3X,'XIN = ',F12.6,/,
     $        3X,'RINM  = ',F12.6,/,3X,'XINM  = ',F12.6)

 120     FORMAT(/,3X,'RIN = INFINITY',/,3X,'XIN = INFINITY',/,
     $        3X,'RINM  = ',F12.6,/,3X,'XINM  = ',F12.6)

 130     FORMAT(/,3X,'DIPOLE CONFIGURATION OPTION',/
     &         ,6X,'OPTION (A): SIDE-BY-SIDE',/,6X
     &            ,'OPTION (B): COLINEAR',/,6X
     &            ,'OPTION (C): PARALLEL-IN-ECHELLON',//,3X
     &         ,'CONFIGURATION OPTION: A, B, or C (in single quotes) = '
     &         ,$)

 140     FORMAT(I1)

 150     FORMAT(/,3X,'R21I = ',F12.6,/,3X,'X21I = ',F12.6,/,
     $        3X,'R21M = ',F12.6,/,3X,'X21M = ',F12.6)

 160     FORMAT(/,3X,'R21I = INFINITY',/,3X,'X21I = INFINITY',/,
     $        3X,'R21M = ',F12.6,/,3X,'X21M = ',F12.6)

 170     FORMAT(/,3X,'***** ERROR *****',/,3X
     $         ,'THE CONFIGURATION OPTION IS INCORRECT',/)

 180     FORMAT(/,3X,'***** ERROR *****',/,3X
     $         ,'THE OPTION NUMBER IS INCORRECT',/)

 190     FORMAT(/,3X,'*********** PROGRAM OUTPUT ***********',/,3X,
     $         'SELF IMPEDANCE CALCULATION OF A DIPOLE',/)

 200     FORMAT(/,3X,'************* PROGRAM OUTPUT *************',/,3X,
     $         'MUTUAL IMPEDANCE CALCULATION OF TWO DIPOLES',/)

 210     FORMAT(3X,'LENGTH OF THE DIPOLE (IN WAVELENGTHS) = ',F12.6,/,3X
     $         ,'RADIUS OF THE DIPOLE (IN WAVELENGTHS) = ',E12.6)

 220     FORMAT(3X,'LENGTH OF THE TWO DIPOLES (IN WAVELENGTHS) = ',F12.6
     &         ,/,3X,'DISTANCE BETWEEN THE TWO DIPOLES',/,3X
     &         ,'(IN WAVELENGTHS) .......................... = ',F12.6)

 230     FORMAT(3X,'LENGTH OF THE TWO DIPOLES (IN WAVELENGTHS) = ',F12.6
     &         ,/,3X,'SEPARATION BETWEEN THE TWO DIPOLES',/,3X
     &         ,'(IN WAVELENGTHS) .......................... = ',F12.6)

 240     FORMAT(3X,'LENGTH OF THE TWO DIPOLES (IN WAVELENGTHS) = ',F12.6
     &         ,/,3X,'DISTANCE BETWEEN THE TWO DIPOLES',/,3X
     &         ,'(IN WAVELENGTHS) .......................... = ',F12.6,/
     &         ,3X,'RELATIVE HIGHT BETWEEN THE TWO DIPOLES',/,3X
     &         ,'(IN WAVELENGTHS) .......................... = ',F12.6) 

 250     FORMAT(3X,'SIDE-BY-SIDE CONFIGURATION',/)
 260     FORMAT(3X,'COLLINEAR CONFIGURATION',/)
 270     FORMAT(3X,'PARALLEL-IN-ECHELON CONFIGURATION',/)


         STOP  ! END OF THE MAIN PROGRAM  
C     

C     ********** ERROR CONDITIONS **********
C
 999     WRITE(6,1000)
 1000    FORMAT(/,3X,'***** ERROR *****',/,3X
     $         ,'INPUT DATA ARE NOT OF THE RIGHT FORMAT',/)

      END

C     ******************************************************************
C     CALCULATION OF SELF IMPEDANCE (BOTH RESISTANCE AND REACTANCE)
C     ******************************************************************
      SUBROUTINE SELF(L,A,RINM,XINM)
C     ******************************************************************

      REAL C,K,L,A,PI,ETA,RINM,XINM
      COMMON /CONSTANTS/ PI,K,ETA

      C=0.5772156649

C     *** INPUT SPECIFICATIONS ***

      WRITE(6,301)
      READ(5,*) L  ! LENFTH OF THE DIPOLE

      WRITE(6,302)
      READ(5,*) A  ! RADIUS OF THE DIPOLE

      RINM=(ETA/(2.*PI))*(C+LOG(K*L)-CI(K*L)+.5*SIN(K*L)*(SI(2.*K*L)-2.
     $     *SI(K*L))+.5*COS(K*L)*(C+LOG(K*L/2.)+CI(2.*K*L)-2.*CI(K*L)))
      XINM=(ETA/(4.*PI))*(2.*SI(K*L)+COS(K*L)*(2.*SI(K*L)-SI(2.*K*L))
     $     -SIN(K*L)*(2.*CI(K*L)-CI(2.*K*L)-CI((2.*K*A**2)/L)))

 301  FORMAT(/,3X,'LENGTH OF THE DIPOLE (IN WAVELENGTHS) = ',$)
 302  FORMAT(3X,'RADIUS OF THE DIPOLE (IN WAVELENGTHS) = ',$)

      RETURN
      END


C     ******************************************************************
C     CALCULATION OF MUTUAL IMPEDANCE FOR A SIDE-BY-SIDE CONFIGURATION
C     (BOTH RESISTANCE AND REACTANCE)
C     ******************************************************************
      SUBROUTINE SBS(L,D,R21M,X21M)
C     ******************************************************************

      REAL D,L,K,PI,ETA,U0,U1,U2,R21M,X21M,TOL
      PARAMETER(TOL=1.0E-5)
      COMMON /CONSTANTS/ PI,K,ETA

C     *** INPUT SPECIFICATIONS ***

      WRITE(6,303)
      READ(5,*) L  ! LENGTH OF THE DIPOLES

C     CHECKING POINT 
C     --------------
      IF(MOD(L,0.5) .LT. TOL .AND. MOD(NINT(L/0.5),2) .EQ. 1) THEN
         CONTINUE
      ELSE
         WRITE(6,11)
         STOP
      ENDIF

      WRITE(6,304)
      READ(5,*) D  ! HORIZONTAL DISPLACEMENT OF THE DIPOLES

      U0=K*D
      U1=K*(SQRT(D**2+L**2)+L)
      U2=K*(SQRT(D**2+L**2)-L)

      R21M=(ETA/(4.*PI))*(2.*CI(U0)-CI(U1)-CI(U2))
      X21M=-(ETA/(4.*PI))*(2.*SI(U0)-SI(U1)-SI(U2))

 11   FORMAT(3X,'*** ERROR: DIPOLE LENGTH MUST BE ODD MULTIPLES',/,3X
     &      ,'           OF HALF WAVELENGTH !!! ',/)
 303  FORMAT(/,3X,'LENGTH OF THE DIPOLES (IN WAVELENGTHS) = ',$)
 304  FORMAT(3X
     &      ,'HORIZONTAL DISPLACEMENT OF THE DIPOLES',/,3X
     &      ,'(IN WAVELENGTHS) ..................... = ',$)

      RETURN
      END

C     ******************************************************************
C     CALCULATION OF MUTUAL IMPEDANCE FOR A COLLINEAR CONFIGURATION
C     (BOTH RESISTANCE AND REACTANCE)
C     ******************************************************************
      SUBROUTINE COLN(L,S,R21M,X21M)
C     ******************************************************************

      REAL H,S,L,K,PI,ETA,U0,U1,U2,U3,R21M,X21M,TOL
      PARAMETER(TOL=1.0E-5)
      COMMON /CONSTANTS/ PI,K,ETA

C     *** INPUT SPECIFICATIONS ***

      WRITE(6,305)
      READ(5,*) L  !  LENGTH OF THE DIPOLES

C     CHECKING POINT 
C     --------------
      IF(MOD(L,0.5) .LT. TOL .AND. MOD(NINT(L/0.5),2) .EQ. 1) THEN
         CONTINUE
      ELSE
         WRITE(6,12)
         STOP
      ENDIF

      WRITE(6,306)
      READ(5,*) S  !  VERTICAL DISPLACEMENT OF THE DIPOLES

      H=L+S
      U0=K*H
      U1=2.*K*(H+L)
      U2=2.*K*(H-L)
      U3=(H**2-L**2)/H**2

      R21M=-(ETA/(8.*PI))*COS(U0)*(-2.*CI(2.*U0)+CI(U2)+CI(U1)
     $     -LOG(U3))+(ETA/(8.*PI))*SIN(U0)*(2.*SI(2.*U0)-SI(U2)
     $     -SI(U1)) 
      X21M=-(ETA/(8.*PI))*COS(U0)*(2.*SI(2.*U0)-SI(U2)-SI(U1))
     $     +(ETA/(8.*PI))*SIN(U0)*(2.*CI(2.*U0)-CI(U2)-CI(U1)
     $     -LOG(U3))

 12   FORMAT(3X,'*** ERROR: DIPOLE LENGTH MUST BE ODD MULTIPLES',/,3X
     &      ,'           OF HALF WAVELENGTH !!! ',/)
 305  FORMAT(/,3X,'LENGTH OF THE DIPOLES (IN WAVELENGTHS) = ',$)
 306  FORMAT(3X
     &      ,'VERTICAL DISPLACEMENT OF THE DIPOLES (IN WAVELENGTHS) = '
     &      ,$)

      RETURN
      END

C     ******************************************************************
C     CALCULATION OF MUTUAL IMPEDANCE FOR A PARALLEL-IN-ECHELON
C     CONFIGURATION (BOTH RESISTANCE AND REACTANCE)
C     ******************************************************************
      SUBROUTINE PARECH(L,D,H,R21M,X21M)
C     ******************************************************************

      REAL D,L,H,K,PI,ETA,R21M,X21M
      REAL W0,W1,W2,W3,W1P,W2P,W3P,TOL
      PARAMETER(TOL=1.0E-5)
      COMMON /CONSTANTS/ PI,K,ETA

C     *** INPUT SPECIFICATIONS ***

      WRITE(6,307)
      READ(5,*) L  !  LENGTH OF THE DIPOLES

C     CHECKING POINT 
C     --------------
      IF(MOD(L,0.5) .LT. TOL .AND. MOD(NINT(L/0.5),2) .EQ. 1) THEN
         CONTINUE
      ELSE
         WRITE(6,13)
         STOP
      ENDIF

      WRITE(6,308)
      READ(5,*) H  !  VERTICAL DISPLACEMENT OF DIPOLES 

      WRITE(6,309)
      READ(5,*) D  !  HORIZONTAL DISPLACEMENT OF DIPOLES

      W0=K*H
      W1=K*(SQRT(D**2+H**2)+H)
      W1P=K*(SQRT(D**2+H**2)-H)
      W2=K*(SQRT(D**2+(H-L)**2)+(H-L))
      W2P=K*(SQRT(D**2+(H-L)**2)-(H-L))
      W3=K*(SQRT(D**2+(H+L)**2)+(H+L))
      W3P=K*(SQRT(D**2+(H+L)**2)-(H+L))

      R21M=-(ETA/(8.*PI))*COS(W0)*(-2.*CI(W1)-2.*CI(W1P)+CI(W2)
     $     +CI(W2P)+CI(W3)+CI(W3P))
     $     +(ETA/(8.*PI))*SIN(W0)*(2.*SI(W1)-2.*SI(W1P)-SI(W2)
     $     +SI(W2P)-SI(W3)+SI(W3P))
      X21M=-(ETA/(8.*PI))*COS(W0)*(2.*SI(W1)+2.*SI(W1P)-SI(W2)
     $     -SI(W2P)-SI(W3)-SI(W3P))
     $     +(ETA/(8.*PI))*SIN(W0)*(2.*CI(W1)-2.*CI(W1P)-CI(W2)
     $     +CI(W2P)-CI(W3)+CI(W3P))

 13   FORMAT(3X,'*** ERROR: DIPOLE LENGTH MUST BE ODD MULTIPLES',/,3X
     &      ,'           OF HALF WAVELENGTH !!! ',/)
 307  FORMAT(/,3X,'LENGTH OF THE TWO DIPOLES (IN WAVELENGTHS) = ',$)
 308  FORMAT(3X
     &      ,'VERTICAL DISPLACEMENT OF THE DIPOLES (IN WAVELENGTHS) = '
     &      ,$)
 309  FORMAT(3X
     &      ,'HORIZONTAL DISPLACEMENT OF THE DIPOLES',/,3X
     &      ,'(IN WAVELENGTHS) ..................... = ',$)

      RETURN
      END


C     ******************************************************************
C     COSINE INTEGRAL
C     ******************************************************************
      REAL FUNCTION CI(U)
C     ******************************************************************

      INTEGER I,TI
      REAL C,SERIES,TEMP,RK,TRK,FACTORIAL,U

      IF(U.LE.5.0)THEN
         C=0.5772156649
         IMAX=500
         TOL=1.E-7
         SERIES=0.0

         DO I=1,IMAX
            RK=FLOAT(I)
            TRK=2.*RK
            TI=2*I
            FACTORIAL=FACT(TI)
            TEMP=SERIES
            SERIES=SERIES+(-1)**I*U**TI/(TRK*FACTORIAL)
            IF(ABS(TEMP-SERIES) .LT. TOL) GOTO 4
         ENDDO

 4       CI=C+LOG(U)+SERIES
      ELSE
         CI=F(U)*SIN(U)-G(U)*COS(U)
      ENDIF

      RETURN
      END

C     ******************************************************************
C     SINE INTEGRAL
C     ******************************************************************
      REAL FUNCTION SI(U)
C     ******************************************************************

      INTEGER I,TIP1
      REAL C,SERIES,TEMP,RK,TRKP1,FACTORIAL,U,PI2
      PI2=2.0*ATAN(1.0)

      IF(U.LE.5.0)THEN
         C=0.5772156649
         IMAX=500
         TOL=1.E-7
         SERIES=0.0

         DO I=0,IMAX
            RK=FLOAT(I)
            TRKP1=2.*RK+1.
            TIP1=2*I+1
            FACTORIAL=FACT(TIP1)
            TEMP=SERIES
            SERIES=SERIES+(-1)**I*U**TIP1/(TRKP1*FACTORIAL)
            IF(ABS(TEMP-SERIES) .LT. TOL) GOTO 5
         ENDDO
      
 5       SI=SERIES
      ELSE
         SI=PI2-F(U)*COS(U)-G(U)*SIN(U)
      ENDIF

      RETURN
      END

C     ******************************************************************
C     THIS FUNCTION IS USED BY CI() AND SI()
C     ******************************************************************
      REAL FUNCTION F(X)
C     ******************************************************************
      REAL X,NUMER,DENOM,A(4),B(4)
      
      DATA A/38.027264,265.187033,335.677320,38.102495/
      DATA B/40.021433,322.624911,570.236280,157.105423/

      NUMER=X**8+A(1)*X**6+A(2)*X**4+A(3)*X**2+A(4)
      DENOM=X**8+B(1)*X**6+B(2)*X**4+B(3)*X**2+B(4)
      F=(1./X)*NUMER/DENOM

      RETURN
      END

C     ******************************************************************
C     THIS FUNCTION IS USED BY CI() AND SI()
C     ******************************************************************
      REAL FUNCTION G(X)
C     ******************************************************************
      REAL X,NUMER,DENOM,A(4),B(4)

      DATA A/42.242855,302.757865,352.018498,21.821899/
      DATA B/48.196927,482.485984,1114.978885,449.690326/

      NUMER=X**8+A(1)*X**6+A(2)*X**4+A(3)*X**2+A(4)
      DENOM=X**8+B(1)*X**6+B(2)*X**4+B(3)*X**2+B(4)
      G=(1./X**2)*NUMER/DENOM

      RETURN
      END

C     ******************************************************************
C     THIS FUNCTION CALCULATES THE FANCTIONAL OF AN INTEGER NUMBER
C     ******************************************************************
      REAL FUNCTION FACT(IARG)
C     ******************************************************************
      
      FACT=1.0
          DO J=1,IARG
            FACT=FLOAT(J)*FACT
          ENDDO
      RETURN
      END



