C     *****************************************************************
C
C     THIS IS A MOMENT METHOD PROGRAM USING
C
C       I.   POCKLINGTON'S INTEGRAL EQUATION (8-24)
C       II.  HALLEN'S INTEGRAL EQUATION (8-27)
C
C     TO COMPUTE:
C
C       A.  CURRENT DISTRIBUTION
C       B.  INPUT IMPEDANCE 
C       C.  NORMALIZED AMPLITUDE RADIATION PATTERN
C
C     OF A LINEAR SYMMETRICALLY EXCITED DIPOLE.
C
C     THIS PROGRAM USES PULSE EXPANSION FOR THE ELECTRIC CURRENT MODE 
C     AND POINT-MATCHING FOR THE ELECTRIC FIELD AT THE CENTER OF EACH 
C     WIRE SEGMENT.
C
C     DELTA-GAP FEED MODEL IS USED IN BOTH FORMULATIONS.  IN ADDITION,
C     MAGNETIC-FRILL GENERATOR IS AVAILABLE IN THE POCKLINGTON'S
C     INTEGRAL EQUATION.
C
C       OPTION I.   POCKLINGTON'S INTEGRAL EQUATION
C       OPTION II.  HALLEN'S INTEGRAL EQUATION 
C
C                                                                     
C         ** INPUT DATA:                                                       
C
C         TL  = TOTAL DIPOLE LENGTH (IN WAVELENGTHS)                 
C         RA  = RADIUS OF THE WIRE (IN WAVELENGTHS)                        
C         NM  = TOTAL NUMBER OF SUBSECTIONS (MUST BE AN ODD INTEGER)       
C         IEX = OPTION TO USE EITHER MAGNETIC-FRILL GENERATOR OR DELTA GAP 
C         
C         IEX = 1 :  MAGNETIC-FRILL GENERATOR                            
C         IEX = 2 :  DELTA-GAP FEED                                           
C
C         ** NOTE:  IGNORE INPUT PARAMETER IEX WHEN CHOOSING OPTION II
C                   (i.e., HALLEN'S FORMULATION)
C     *****************************************************************
      INTEGER OPTION,DEVICE
      COMMON/SIZE/HL,RA,DZ,ZM,ZN,NMH
      CHARACTER*72 FILNAM

C     ********** CHOICE OF OUTPUT **********
C
      OPEN(UNIT=3,FILE='Curr-MoM.dat',STATUS='UNKNOWN')
      OPEN(UNIT=4,FILE='Patt-MoM.dat',STATUS='UNKNOWN')

      WRITE(6,3)
      READ(5,2,ERR=999) DEVICE

      IF(DEVICE .EQ. 1) THEN
         DEVICE=6

      ELSE IF(DEVICE .EQ. 2) THEN
         WRITE(6,4)
         READ(5,*,ERR=999) FILNAM
         OPEN(UNIT=DEVICE,FILE=FILNAM,STATUS='UNKNOWN')

      ELSE 
         WRITE(6,5)
         STOP

      ENDIF

C***  READING OPTION # : 1 ---> POCKLINGTON'S EQUATIONS
C                        2 ---> HALLEN'S EQUATION
C
      WRITE(6,60)
      READ(5,70,ERR=999) OPTION

      IF(OPTION .EQ. 1) THEN

         WRITE(6,10)
         READ(5,*,ERR=999) NM   ! # OF SUBDIVISIONS 

         IF(MOD(NM,2) .EQ. 0) THEN
            WRITE(6,9)
            STOP
         ENDIF

         WRITE(6,50) 
         READ(5,*,ERR=999) TL   ! TOTAL DIPOLE LENGTH
         WRITE(6,30)
         READ(5,*,ERR=999) RA   ! DIPOLE RADIUS
         WRITE(6,40)
         READ(5,*,ERR=999) IEX  ! EXCITATION
         WRITE(6,*)

         HL=TL/2.0              ! HALF DIPOLE LENGTH
         CALL PWRS(IEX,NM,DEVICE)      ! POCKLINGTON'S EQUATIONS

      ELSE IF(OPTION .EQ. 2) THEN ! HALLEN'S EQUATION

         WRITE(6,55)
         READ(5,*,ERR=999) NM   ! # OF SUBDIVISIONS 
         WRITE(6,50) 
         READ(5,*,ERR=999) TL   ! TOTAL DIPOLE LENGTH
         WRITE(6,30)
         READ(5,*,ERR=999) RA   ! DIPOLE RADIUS
         WRITE(6,*)

         CALL HALLEN(NM,TL,RA,DEVICE)

      ELSE 
         WRITE(6,*) '**** ERROR: WRONG OPTION NUMBER !!!!'
         STOP
      ENDIF

      IF(DEVICE .EQ. 6) THEN
         WRITE(DEVICE,1)
      ELSE IF(DEVICE .EQ. 2) THEN
         WRITE(6,1)
         WRITE(DEVICE,1)
      ENDIF


C     ********* FORMAT STATEMENTS **********
C     
 1    FORMAT(/,3X,'*** NOTE:',/,7X,
     $      'THE DIPOLE CURRENT DISTRIBUTION IS STORED IN Curr-MoM.dat'
     $      ,/,7X,
     $      'THE AMPLITUDE RADIATION PATTERN IS STORED IN Patt-MoM.dat'
     $      ,/,7X,
     $     '=========================================================')

 2    FORMAT(I1)
 3    FORMAT(3X,'OUTPUT DEVICE OPTION FOR THE OUTPUT PARAMETERS',/,6X
     $      ,'OPTION (1): SCREEN',/,6X,'OPTION (2): OUTPUT FILE',//,3X
     $      ,'OUTPUT DEVICE = ',$)

 4    FORMAT(3X,'INPUT THE DESIRED OUTPUT FILNAME (in single quotes) = '
     &      ,$)

 5    FORMAT(/,3X,'*** ERROR ***',/,3X
     $      ,'OUTPUTING DEVICE NUMBER SHOULD BE EITHER 1 OR 2')

 9    FORMAT(/,3X,'*** ERROR ***',/,3X
     $      ,'NUMBER OF SUBDIVISIONS SHOULD BE AN ODD NUMBER')

 10   FORMAT(3X,'NUMBER OF SUBDIVISIONS (ODD NUMBER) = ',$)
 30   FORMAT(3X,'RADIUS OF THE DIPOLE (WAVELENGTHS)= ',$)
 40   FORMAT(3X,'EXCITATION: 1 FOR MAGNETIC-FRILL; 2 FOR DELTA GAP = ',$
     $      )
 50   FORMAT(3X,'TOTAL DIPOLE LENGTH (WAVELENGTHS) = ',$)
 55   FORMAT(3X,'NUMBER OF SUBDIVISIONS (ODD OR EVEN NUMBER) = ',$)

 60   FORMAT(/,3X,'CHOICE OF POCKLINGTON S  OR HALLEN S EQN. ',/,6X
     $      ,'OPTION (1): POCKLINGTON S EQN.',/,6X
     $      ,'OPTION (2): HALLEN S EQN.',//,3X,'OPTION NUMBER = ',$)

 70   FORMAT(I1)

      STOP                      ! END OF THE MAIN PROGRAM  
C    
C     ********** ERROR CONDITIONS **********
C
 999  WRITE(6,1000)
 1000 FORMAT(/,3X,'***** ERROR *****',/,3X
     $      ,'INPUT DATA ARE NOT OF THE RIGHT FORMAT',/)

      
      END



C***********************************************************************
C     POCKLINGTON'S INTEGRAL EQUATIONS
C***********************************************************************
      SUBROUTINE PWRS(IEX,NM,DEVICE)

      INTEGER DEVICE
      PARAMETER(NMAX=200, NMT=2*NMAX-1)
      COMMON/SIZE/HL,RA,DZ,ZM,ZN,NMH
      COMMON/CONST/BETA,ETA,RAD,J
      COMPLEX ZMN(NMT),WA(NMT),CGA(NMAX),ZIN,J,CRT
      DIMENSION ETMM(181)
      EXTERNAL CGP

C     IF THE NUMBER OF UNKNOWNS IS GREATER THAN NMAX ==> SET IT TO NMAX
C     -----------------------------------------------------------------

      IF(NM.GT.NMAX)THEN
         WRITE(6,130) NMAX
         NM=NMAX
      END IF

C     SOME CONSTANTS
C     --------------

      PI=3.14159265
      RAD =PI/180.
      BETA=2.0*PI
      ETA =120.*PI
      NMH=0.5*(NM+1)
      J=CMPLX(0.,1.)
      DZ=2.*HL/NM

      WRITE(DEVICE,50) 2.0*HL,RA
      IF(IEX.EQ.1) WRITE(DEVICE,100)
      IF(IEX.EQ.2) WRITE(DEVICE,102)

      WRITE(DEVICE,54) NM

C     THE IMPEDANCE MATRIX HAS A TOEPLITZ PROPERTY, THEREFORE ONLY
C     NM ELEMENTS NEED TO BE COMPUTED, AND THE MATRIX IS FILLED IN A
C     FORM THAT CAN BE SOLVED BY A TOEPLITZ MATRIX SOLVING SUBROUTINE
C     ---------------------------------------------------------------

      ZM=HL-0.5*DZ
      B=0.5*DZ
      A=-0.5*DZ
      DO 4 I=1,NM
         ZN=HL-(I-0.5)*DZ
         CALL CSINT(CGP,A,B,79,CRT)
         ZMN(I)=CRT
         IF(I.EQ.1) GOTO 4
         ZMN(NM+I-1)=CRT
 4    CONTINUE

      RB=2.3*RA
      TLAB=2.*ALOG(2.3)
      DO 10 I=1,NM
         ZI=HL-(I-0.5)*DZ
         R1=BETA*SQRT(ZI*ZI+RA*RA)
         R2=BETA*SQRT(ZI*ZI+RB*RB)
         IF(IEX.EQ.1) THEN
            CGA(I)=-J*BETA**2/(ETA*TLAB)*(CEXP(-J*R1)/R1-CEXP(-J*R2)/R2)
         ELSE
            IF(IEX.NE.2) WRITE(6,999)
            IF(I.NE.NMH) THEN
               CGA(I)=0.
            ELSE
               CGA(I)=-J*BETA/(ETA*DZ)
            ENDIF
         ENDIF
 10   CONTINUE
      CALL TSLZ(ZMN,CGA,WA,NM)

C     OUTPUT THE CURRENT DISTRIBUTION ALONG OF THE DIPOLE
C     ---------------------------------------------------

      WRITE(3,104)
      DO 12 I=1,NMH
         XI=HL-(I-.5)*DZ
         YI=CABS(CGA(I))
C         WRITE(3,106)I, XI,CGA(I),YI
         WRITE(3,107)XI,YI,CGA(I),I
 12   CONTINUE

C     COMPUTATION OF THE INPUT IMPEDANCE
C     ----------------------------------

      ZIN=1./CGA(NMH)
      WRITE(DEVICE,108) ZIN

C     COMPUTATION OF AMPLITUDE RADIATION PATTERN OF THE ANTENNA
C     ---------------------------------------------------------

      CALL PATN(CGA,NM,ETMM)
      WRITE(4,110)
      DO 14 I=1,181
         XI=I-1.
         WRITE(4,112)  XI,ETMM(I)
 14   CONTINUE
      DO 15 I=182,361
         XI=I-1.
         WRITE(4,112)  XI,ETMM(362-I)
 15   CONTINUE

C     FORMAT STATEMENTS
C     -----------------

 50   FORMAT(15X,'WIRE ANTENNA PROBLEM'/,15X,'====================',//
     &      ,3X,'LENGTH = ',F6.4,' (WLS)',/,3X,'RADIUS OF THE WIRE =',
     &      F6.4,' (WLS) ')
 54   FORMAT(3X,'NUMBER OF SUBSECTIONS = ', I3)
 100  FORMAT(3X,'POCKLINGTON''S EQUATION AND MAGNETIC FRILL MODEL')
 102  FORMAT(3X,'POCKLINGTON''S EQUATION AND DELTA GAP MODEL')
 104  FORMAT('#',2X,'CURRENT DISTRIBUTION ALONG ONE HALF OF THE DIPOLE',
     &      /,'#',6X,'POSITION Z',3X,'MAGNITUDE',3X,'REAL PART',3X
     &      ,'IMAGINARY ',3X,3X,'SECTION (M)',/,'#')
 106  FORMAT(2X,I3,4X,F6.4,5X,F9.6,3X,F9.6,3X,F9.6)
 107  FORMAT(7X,F6.4,5X,F9.6,5X,F9.6,3X,F9.6,3X,4X,I3)
 108  FORMAT(/,3X,'INPUT IMPEDANCE:  Z = ',F12.1,3X,'j ',F12.1,'
     &      (OHMS)')
 110  FORMAT('#',2X,'RADIATION PATTERN VS OBSERVATION ANGLE THETA'
     &      ,/,'#',2X,'THETA (IN DEGREES)',2X,'MAGNITUDE (IN DB)')
 112  FORMAT(8X,F6.1,8X,F8.2)

 122  FORMAT(8X,'INCIDENT ANGLE=',F5.1,' DEGREES, POLARIZATION=',F5.1,
     &      ' DEGREEES'//10X,'CURRENT DISTRIBUTION ALONG THE DIPOLE'//
     &      8X,'POSITION Z',3X,'REAL PART',3X,'IM HALLEN S EQN.AGINARY ',3X
     &      ,'MAGNITUDE'/)
 124  FORMAT(10X,F6.4,5X,F9.6,3X,F9.6,3X,F9.6)

 128  FORMAT(12X,F6.1,8X,F10.2)
 130  FORMAT(2X,'RESETTING N1 TO LARGEST POSSIBLE FOR NMAX',/,
     &       2X,'N1 = NMAX ==> ',I4,/)
 999  FORMAT(5X,'******WARNING: NO SUCH OPTION. CHOOSE A VALID OPTION'/
     &      20X,'AND TRY AGAIN.')

      RETURN
      END

C     ******************************************************************
C     SUBROUTINE CGP
C     POCKLINGTON'S KERNEL
C     ******************************************************************
      COMPLEX FUNCTION CGP(Z)

      COMMON/SIZE/HL,RA,DZ,ZM,ZN,NMH
      COMMON/CONST/BETA,ETA,RAD,J
      COMPLEX J
      Z1=ZN-ZM +Z
      R=SQRT(RA*RA+Z1*Z1)
      CGP=CEXP(-J*BETA*R)*((1.+J*BETA*R)*(2.*R*R-3.*RA*RA)+
     &      (BETA*RA*R)**2)/(2.*BETA*R**5)
      RETURN
      END

C     ******************************************************************
C     SUBROUTINE PATN
C     COMPUTES THE RADIATION PATTERN
C     ******************************************************************
      SUBROUTINE PATN(CGA,NM,ETMM)

      COMMON/SIZE/HL,RA,DZ,ZM,ZN,NMH
      COMMON/CONST/BETA,ETA,RAD,J
      COMPLEX CGA(NM),J,CRT
      DIMENSION ETMM(181)
      DO 4 I=1,181
      THETA=(I-1.)*RAD
      CTH=COS(THETA)
      STH=SIN(THETA)
      IF(ABS(CTH).LE.1.E-3) THEN
         FT=1.
      ELSE
         FT=SIN(BETA*DZ*CTH*.5)/(BETA*DZ*CTH*.5)
      ENDIF
      CRT=0.
      DO 2 M=1,NM
         ZM=HL-(M-.5)*DZ
         CRT=CRT+CEXP(J*BETA*ZM*CTH)*FT*CGA(M)*DZ
 2    CONTINUE
      PTT=CABS(CRT)*STH*STH*ETA*0.5
      ETMM(I)=PTT
 4    CONTINUE
      AMAX=ETMM(1)
      DO 6 I=2,181
         IF(ETMM(I).GT.AMAX) AMAX=ETMM(I)
 6    CONTINUE
      DO 8 I=1,181
         PTT=ETMM(I)/AMAX
C         PTT=ETMM(I)
         IF(PTT.LE.1.E-5) PTT=1.E-5
         ETMM(I)=20.*ALOG10(PTT)
 8    CONTINUE
      RETURN
      END

C     ******************************************************************
C     SUBROUTINE TSLZ        NETLIB
C     INPUT:
C       (C)A(2*M - 1)        THE FIRST ROW OF THE T-MATRIX FOLLOWED BY
C                            ITS FIRST COLUMN BEGINNING WITH THE SECOND
C                            ELEMENT.  ON RETURN A IS UNALTERED.
C       (C)B(M)              THE RIGHT HAND SIDE VECTOR B.
C       (C)WA(2*M-2)         A WORK AREA VECTOR
C       (I)M                 ORDER OF MATRIX A.
C     OUTPUT:
C       (C)B(M)              THE SOLUTION VECTOR.
C     PURPOSE:
C       SOLVE A SYSTEM OF EQUATIONS DESCRIBED BY A TOEPLITZ MATRIX.
C       A * X = B
C     SUBROUTINES AND FUNCTIONS:
C       TOEPLITZ PACKAGE ... TSLZ1
C     ******************************************************************
      SUBROUTINE TSLZ(A,B,WA,M)

      INTEGER M
      COMPLEX A(2*M-1),B(M),WA(2*M-2)
      CALL TSLZ1(A,A(M+1),B,B,WA,WA(M-1),M)
      RETURN
      END

C     ******************************************************************
C     SUBROUTINE TSLZ1
C     ******************************************************************
      SUBROUTINE TSLZ1(A1,A2,B,X,C1,C2,M)

      INTEGER M
      COMPLEX A1(M),A2(M-1),B(M),X(M),C1(M-1),C2(M-1)
      INTEGER I1,I2,N,N1,N2
      COMPLEX R1,R2,R3,R5,R6
      R1=A1(1)
      X(1)=B(1)/R1
      IF(M.EQ.1) GOTO 80
      DO 70 N = 2, M
         N1 = N - 1
         N2 = N - 2
         R5 = A2(N1)
         R6 = A1(N)
         IF (N .EQ. 2) GO TO 20
            C1(N1) = R2
            DO 10 I1 = 1, N2
               I2 = N - I1
               R5 = R5 + A2(I1)*C1(I2)
               R6 = R6 + A1(I1+1)*C2(I1)
 10         CONTINUE
 20         CONTINUE
            R2 = -R5/R1
            R3 = -R6/R1
            R1 = R1 + R5*R3
            IF (N .EQ. 2) GO TO 40
            R6 = C2(1)
            C2(N1) = (0.0,0.0)
            DO 30 I1 = 2, N1
               R5 = C2(I1)
               C2(I1) = C1(I1)*R3 + R6
               C1(I1) = C1(I1) + R6*R2
               R6 = R5
 30         CONTINUE
 40         CONTINUE
            C2(1) = R3
            R5 = (0.0,0.0)
            DO 50 I1 = 1, N1
               I2 = N - I1
               R5 = R5 + A2(I1)*X(I2)
 50         CONTINUE
            R6 = (B(N) - R5)/R1
            DO 60 I1 = 1, N1
               X(I1) = X(I1) + C2(I1)*R6
 60         CONTINUE
            X(N) = R6
 70      CONTINUE
 80      CONTINUE
         RETURN
         END

C     ******************************************************************
C     SUBROUTINE CSINT
C     FAST ALGORITHM FORM OF THE SIMPSON'S INTEGRAL ROUTINE
C     ******************************************************************
      SUBROUTINE CSINT(CF,XL,XU,N,CRT)

      IMPLICIT COMPLEX (C)
      CRT=CF(XL)+CF(XU)
      HD=(XU-XL)/(N+1)
      DO 20 I=1,N
         XI=XL+I*HD
         IF(MOD(I,2).NE.0) THEN
            CRT=CRT+4.*CF(XI)
         ELSE
            CRT=CRT+2.*CF(XI)
         ENDIF
 20   CONTINUE
      CRT=CRT*HD*0.33333333
      RETURN
      END


C***********************************************************************
C     HALLEN'S INTEGRAL EQUATION
C***********************************************************************
      SUBROUTINE HALLEN(N1,L,RHO,DEVICE)

      INTEGER NMAX,MAXANG,DEVICE
      PARAMETER (NMAX=200,MAXANG=181)

      EXTERNAL KERNEL

      INTEGER I,J,N1,IPERM(NMAX)

      REAL Z,ETA,PI,L,DZ,DZ1,BK,UPPER,LOWER
      REAL THETA,POWER,PATTRN
      REAL RHO,RTOD,CUR,PHA,PMAX,BTAN2,PIVOT(NMAX),PWR(MAXANG)

      COMPLEX CJ,RES,ZIN
      COMPLEX ZMATRX(NMAX,NMAX),ELECUR(NMAX)

      CJ=CMPLX(0.E0,1.E0)
      PI=4.E0*ATAN(1.0E0)
      RTOD=180.E0/PI
      ETA=120.E0*PI
      BK=2.E0*PI

      IF(RHO.EQ.0.E0) WRITE(6,150)

      IF(N1.GT.NMAX)THEN
         WRITE(6,200) NMAX
         N1=NMAX
      END IF

C     ECHO INPUT DATA
C
      WRITE(DEVICE,51) L,RHO
      WRITE(DEVICE,52) N1


      DZ =L/FLOAT(2*(N1-1))
      DZ1=L/FLOAT(2*N1)
 
C     FILL THE MATRIX AND EXCITATION VECTOR OF THE SYSTEM: 
C      [ZMATRX] and [ELECUR]

      DO 10 I=1,N1
         Z=FLOAT(2*I-1)*DZ1/2.0
         ZMATRX(I,N1)=-COS(BK*Z)
         ELECUR(I)=-CJ*SIN(BK*Z)/(2.E0*ETA)
         DO 20 J=1,N1-1
            LOWER=FLOAT(J-1)*DZ1
            UPPER=FLOAT(J)*DZ1

C     PERFORM NUMERICAL INTEGRATION OF THE KERNEL FOR HALLEN'S
C     INTEGRAL EQUATION

            CALL SINTEG(KERNEL,UPPER,LOWER,10,RHO,Z,RES)
            ZMATRX(I,J)=RES
   20    CONTINUE
   10 CONTINUE

C   DECOMPOSE AND SOLVE THE SYSTEM FOR THE CURRENT DISTRIBUTION

      CALL LUDEC(ZMATRX,N1,NMAX,IPERM,PIVOT)
      CALL LUSOLV(ZMATRX,N1,NMAX,IPERM,ELECUR)

C   WRITE THE INPUT IMPEDANCE, AND CURRENT DISTRIBUTION

      ZIN=1.E0/ELECUR(1)
      IF(AIMAG(ZIN) .GE. 0.0) THEN
         WRITE(DEVICE,300) REAL(ZIN),'+',AIMAG(ZIN)
      ELSE
         WRITE(DEVICE,300) REAL(ZIN),'-',-AIMAG(ZIN)
      END IF

      WRITE(3,400)
      DO 40 I=1,N1-1
         CUR=CABS(ELECUR(I))
         PHA=RTOD*BTAN2(REAL(ELECUR(I)),AIMAG(ELECUR(I)))
         WRITE(3,500)I*DZ-DZ/2.,CUR,PHA,I
   40 CONTINUE

C   CALCULATE THE RADIATION PATTERN OF THE ANTENNA

      PMAX=-1.0E0
      DO 50 I =1,MAXANG
         THETA =FLOAT(I-1)/FLOAT(MAXANG-1)*PI
         PWR(I)=POWER(THETA,N1-1,DZ,L,PI,ELECUR)
         IF (PWR(I).GT.PMAX) PMAX=PWR(I)
50    CONTINUE

C   WRITE THE RADIATION PATTERN IN dB

      WRITE(4,600)
      DO 60 I=1,MAXANG
         THETA =FLOAT(I-1)/FLOAT(MAXANG-1)*180.0D0
         PATTRN=PWR(I)/PMAX
         IF(PATTRN.LE.1.0E-5)THEN
            PATTRN=-100.0E0
         ELSE
            PATTRN=20.0*ALOG10(PATTRN)
         END IF
         WRITE(4,700)THETA,PATTRN
60    CONTINUE
      DO 61 I=182,180+MAXANG
         THETA=FLOAT(I-1)/FLOAT(MAXANG-1)*180.0D0
         PATTRN=PWR(362-I)/PMAX
         IF(PATTRN.LE.1.0E-5)THEN
            PATTRN=-100.0E0
         ELSE
            PATTRN=20.0*ALOG10(PATTRN)
         END IF
         WRITE(4,700)THETA,PATTRN
 61   CONTINUE


C   FORMAT STATEMENTS

 51   FORMAT(15X,'WIRE ANTENNA PROBLEM'/,15X,'====================',//
     &      ,3X,'HALLEN S EQUATION',/,3X,'LENGTH = ',F6.4,' (WLS)',/,3X
     &      ,'RADIUS OF THE WIRE =',F6.4,' (WLS) ')
 52   FORMAT(3X,'NUMBER OF SUBSECTIONS = ', I3)
 150  FORMAT(3X,'WHEN RHO = 0.0, SINGULARITIES MUST BE REMOVED',/,
     &      3X,'FROM THE INTEGRAND, THAT IS BEYOND THE SCOPE',/,
     &      3X,'OF THIS WORK -- BE CAREFUL!!',/)
 200  FORMAT(3X,'RESETTING N1 TO LARGEST POSSIBLE FOR NMAX',/,
     &      3X,'N1 = NMAX ==> ',I4,/)
 300  FORMAT(/,3X,'INPUT IMPEDANCE:  Z = ',F12.1,3X,A,'j ',F12.1,' OHMS'
     &      )

 400  FORMAT('#',2X,'POSITION Z',5X,'CURRENT MAGNITUDE',5X
     &      ,'CURRENT PHASE',3X,'SECTION (M)',/,'#')

 500  FORMAT(1X,F12.6,7X,F12.6,7X,F12.6,7X,I4)

 600  FORMAT('#',2X,'RADIATION PATTERN VS OBSERVATION ANGLE THETA'
     &      ,/,'#',2X,'THETA (IN DEGREES)',2X,'MAGNITUDE (IN DB)')

 700  FORMAT(6X,2(2X,F11.2))

      RETURN
      END


      COMPLEX FUNCTION KERNEL(X,RHO,Z)

      IMPLICIT NONE

      REAL X,Z,RHO
  
C    KERNEL PROVIDES THE KERNEL OF HALLEN'S EQN FOR INTEGRATION 
C    SYMMETRY IS USED TO REDUCE THE SYSTEM OF EQUATIONS AND 
C    HENCE ALTERS THE KERNEL

      REAL PI,BK,R1,R2
      COMPLEX CJ

      CJ=CMPLX(0.E0,1.E0)
      PI=4.E0*ATAN(1.E0)
      BK=2.E0*PI
      R1=SQRT(RHO*RHO+(Z-X)*(Z-X))
      R2=SQRT(RHO*RHO+(Z+X)*(Z+X))
      KERNEL=CEXP(-CJ*BK*R1)/(4.E0*PI*R1)+CEXP(-CJ*BK*R2)/(4.E0*PI*R2)

      RETURN
      END

 
      SUBROUTINE SINTEG(FX,UL,LL,NO,RHO,Z,ANS)

      IMPLICIT NONE

      INTEGER NO
      REAL UL,LL,RHO,Z
      COMPLEX FX,ANS

C    PERFORM COMPLEX INTEGRATION USING SIXTEEN POINT 
C    GAUSSIAN QUADRATURE WITH INCREASING ACCURACY SET 
C    BY INTEGER NO

C    DESCRIPTION OF PARAMETERS
C    FX  -  COMPLEX FUNCTION OF A SINGLE REAL VARIABLE
C    UL  -  UPPER LIMIT OF THE INTEGRATION (REAL)
C    LL  -  LOWER LIMIT OF THE INTEGRATION (REAL)
C    NO  -  NUMBER OF DIVISIONS BETWEEN LL AND UL  (INTEGER)
C    RHO -  WIRE RADIUS IN WAVELENGTHS
C    Z   -  TESTING POINT ON THE AXIS OF THE WIRE
C    ANS -  RESULT OF INTEGRATION

      INTEGER I,J
      REAL X,ABSICA(16),WGHT(16),DEL,S
      COMPLEX SUM

      DATA ABSICA/-0.095012509837637E0,-0.281603550779259E0,
     1            -0.458016777657227E0,-0.617876244402644E0,
     2            -0.755404408355003E0,-0.865631202387832E0,
     3            -0.944575023073233E0,-0.989400934991650E0,
     4             0.095012509837637E0,0.281603550779259E0,
     5             0.458016777657227E0,0.617876244402644E0,
     6             0.755404408355003E0,0.865631202387832E0,
     7             0.944575023073233E0,0.989400934991650E0/
      DATA    WGHT/0.189450610455068E0,0.182603415044924E0,
     1             0.169156519395002E0,0.149595988816577E0,
     2             0.124628971255534E0,0.095158511682493E0,
     3             0.062253523938648E0,0.027152459411754E0,
     4             0.189450610455068E0,0.182603415044924E0,
     5             0.169156519395002E0,0.149595988816577E0,
     6             0.124628971255534E0,0.095158511682493E0,
     7             0.062253523938648E0,0.027152459411754E0/

      DEL=(UL-LL)/FLOAT(2*NO)
      SUM=(0.0E0,0.0E0)
      DO 1 J=1,NO
         S=LL+DFLOAT(2*J-1)*DEL
         DO 2 I=1,16
            X=S+ABSICA(I)*DEL
            SUM=SUM+WGHT(I)*FX(X,RHO,Z)
    2    CONTINUE
    1 CONTINUE

      ANS=SUM*DEL

      RETURN
      END


      REAL FUNCTION POWER(THETA,N1,WPULSE,L,PI,ELECUR)

      IMPLICIT NONE

      INTEGER N1
      REAL THETA,WPULSE,L,PI
      COMPLEX ELECUR(N1)

C    CALCULATES THE RADIATED POWER LEVEL AT ANGLE THETA RADIANS
C    TO THE DIPOLE AXIS.  SINCE THE PATTERN IS NORMALIZED TO THE 
C    MAXIMUM RADIATED POWER, COMMON CONSTANTS ARE REMOVED

      INTEGER I
      REAL STH,CTH,ARG,FT,ARGP
      COMPLEX CRT

      STH=SIN(THETA)
      CTH=COS(THETA)
      ARG=PI*WPULSE*CTH
      IF(ABS(ARG).LT.1.0E-3) THEN
         FT=1.0
      ELSE
         FT=SIN(ARG)/ARG
      END IF

      CRT=(0.0,0.0)
      DO 10 I=1,N1
         ARGP=PI*(-L+FLOAT(I-1)*2.0*WPULSE+WPULSE)*CTH
         CRT=CRT+CEXP((0.0,1.0)*ARGP)*FT*ELECUR(I)
         ARGP=PI*(-L+FLOAT(N1+I-1)*2.0*WPULSE+WPULSE)*CTH
         CRT=CRT+CEXP((0.0,1.0)*ARGP)*FT*ELECUR(I)
10    CONTINUE

      POWER=CABS(CRT)*STH*STH
 
      RETURN
      END

  
      SUBROUTINE LUSOLV(Z,N,NP,IPERM,V)

      IMPLICIT NONE

      INTEGER N,NP
      INTEGER IPERM(N)
      COMPLEX Z(NP,NP),V(N)

C    SOLVES LINEAR SYSTEM GIVEN THE LU DECOMPOSITION FROM LUDEC
C    FORCING VECTOR IS REPLACED WITH SOLUTION VECTOR UPON EXIT 

      INTEGER I,J,II
      COMPLEX TEMP

C    FORWARD SUBSTITUTION.
      DO 20 I=1,N
         TEMP=V(IPERM(I))
         V(IPERM(I))=V(I)
         DO 10 J=1,I-1
            TEMP=TEMP-Z(I,J)*V(J)
10       CONTINUE
         V(I)=TEMP
20    CONTINUE

C    BACKWARD SUBSTITUTION.
      DO 40 I=1,N
         II=N-I+1
         TEMP=V(II)
         DO 30 J=II+1,N
            TEMP=TEMP-Z(II,J)*V(J)
30       CONTINUE
         V(II)=TEMP/Z(II,II)
40    CONTINUE

      RETURN
      END

  
      SUBROUTINE LUDEC (Z,N,NP,IPERM,SCAL)

      INTEGER N,NP
      INTEGER IPERM(N)
      REAL    SCAL(N)
      COMPLEX Z(NP,NP)

C    REPLACES MATRIX BY ITS LU DECOMPOSITION

      INTEGER I,J,K,IMAX
      REAL    ZMAX,CAZ,TEST
      COMPLEX TEMP

C    GET SCALING INFO.
      DO 20 I=1,N
         ZMAX=0.E0
         DO 10 J=1,N
            CAZ=CABS(Z(I,J))
            IF (CAZ.GT.ZMAX) ZMAX=CAZ
10       CONTINUE
         SCAL(I)=1.E0/ZMAX
20    CONTINUE

C    CROUT's algorithm.
      DO 80 J=1,N
         DO 30 I=1,J-1
            DO 30 K=1,I-1
            Z(I,J)=Z(I,J)-Z(I,K)*Z(K,J)
30       CONTINUE
         ZMAX=0.E0
C    SEARCH FOR LARGEST PIVOT ELEMENT.
         DO 50 I=J,N
            DO 40 K=1,J-1
               Z(I,J)=Z(I,J)-Z(I,K)*Z(K,J)
40          CONTINUE
            TEST=SCAL(I)*CABS(Z(I,J))
            IF (TEST.GE.ZMAX) THEN
               IMAX=I
               ZMAX=TEST
            END IF
50       CONTINUE

C    INTERCHANGE THE ROWS.
         IF(J.NE.IMAX)THEN
            DO 60 K=1,N
               TEMP=Z(IMAX,K)
               Z(IMAX,K)=Z(J,K)
               Z(J,K)=TEMP
60          CONTINUE
            SCAL(IMAX)=SCAL(J)
         END IF
         IPERM(J)=IMAX
C    DIVIDE BY PIVOT ELEMENT.
         IF(J.NE.N)THEN
            DO 70 I=J+1,N
               Z(I,J)=Z(I,J)/Z(J,J)
70          CONTINUE
         END IF
80    CONTINUE

      RETURN
      END


      REAL FUNCTION BTAN2(Y,X)

      IMPLICIT NONE

      REAL Y,X

C    THIS FUNCTION IS COMPUTES THE ARCTANGENT GIVEN X,Y.  IT IS 
C    SIMILAR TO ATAN2 EXCEPT IT AVOIDS THE RUN TIME ERRORS ON 
C    SOME MACHINES FOR SMALL ARGUMENTS.

      REAL SMLT
      PARAMETER (SMLT=1.0E-7)

      IF(ABS(X) .LT. SMLT .AND. ABS(Y) .LT. SMLT)THEN
         BTAN2=0.0E0
      ELSE
         BTAN2=ATAN2(Y,X)
      END IF

      RETURN
      END




