C     MICROSTRIP ANTENNAS: RECTANGULAR-CIRCULAR
C*******************************************************************
C  THIS PROGRAM A FORTRAN PROGRAM THAT DESIGNS AND THEN COMPUTES THE
C  ANTENNA RADIATION CHARACTERISTICS OF:
C
C     I.   RECTANGULAR 
C     II.  CIRCULAR
C
C  MICROSTRIP PATCH ANTENNAS BASED ON THE CAVITY MODEL AND DOMINANT
C  MODE OPERATION FOR EACH.  THAT IS:
C
C     A.  TM(010) MODE FOR THE RECTANGULAR PATCH
C     B.  TM(011) MODE FOR THE CIRCULAR PATCH
C
C     ** INPUT PARAMETERS
C     1.  FREQ   = RESONANT FREQUENCY (in GHz)
C     2.  EPSR   = DIELECTRIC CONSTANT OF THE SUBSTRATE
C     3.  HEIGHT = HEIGHT OF THE SUBSTRATE (in cm)
C     4.  Y0     = POSITION OF THE RECESSED FEED POINT (in cm) 
C                  RELATIVE TO LEADING RADIATING EDGE OF RECTANGULAR
C                  PATCH.  NOT NECESSARY FOR CIRCULAR PATCH.
C
C     ** OUTPUT PARAMETERS
C     A.  RECTANGULAR PATCH:
C
C         1.  PHYSICAL WIDTH OF THE PATCH W (in cm)
C         2.  EFFECTIVE LENGTH OF PATCH Le (in cm)
C         3.  PHYSICAL LENGTH OF PATCH L (in cm)
C         4.  NORMALIZED E-PLANE AMPLITUDE PATTERN (in dB)
C         5.  NORMALIZED H-PLANE AMPLITUDE PATTERN (in dB) 
C         6.  E-PLANE HALF-POWER BEAMWIDTH (in degrees)
C         7.  H-PLANE HALF-POWER BEAMWIDTH (in degrees)  
C         8.  DIRECTIVITY (dimensionless and in dB)
C         9.  RESONANT INPUT RESISTANCE (in ohms)
C             a.  AT LEADING RADIATING EDGE (y = 0)
C             b.  AT RECESSED FEED POINT FROM LEADING RADIATING EDGE 
C                 (y = yo)
C
C     B.  CIRCULAR PATCH:
C
C         1.  PHYSICAL RADIUS OF THE PATCH a (in cm)
C         2.  EFFECTIVE RADIUS OF THE PATCH ae (in cm)
C         3.  NORMALIZED E-PLANE AMPLITUDE (in dB)
C         4.  NORMALIZED H-PLANE AMPLITUDE (in dB)
C         5.  E-PLANE HALF-POWER BEAMWIDTH (in degrees)
C         6.  H-PLANE HALF-POWER BEAMWIDTH (in degrees)
C         7.  DIRECTIVITY (dimensionless and in dB) 
C
C*******************************************************************
C     Written by: Anastasis C. Polycarpou, Arizona State University
C
C*******************************************************************

      INTEGER I,OPTION,DEVICE,NPNTS_TH,NPNTS_PHI,NPNTS_CIR
      PARAMETER(NPNTS_TH=181,NPNTS_PHI=361,NPNTS_CIR=91)
      REAL ETH_R(NPNTS_PHI),EPH_R(NPNTS_TH)
      REAL ETH_C(NPNTS_CIR),EPH_C(NPNTS_CIR)
      CHARACTER*72 FILNAM


C     ********** CHOICE OF OUTPUT **********
C
      OPEN(UNIT=3,FILE='Epl-Micr.dat',STATUS='UNKNOWN')
      OPEN(UNIT=4,FILE='Hpl-Micr.dat',STATUS='UNKNOWN')

      WRITE(6,20)
      READ(5,10,ERR=999) DEVICE

      IF(DEVICE .EQ. 1) THEN
         DEVICE=6

      ELSE IF(DEVICE .EQ. 2) THEN
         WRITE(6,30)
         READ(5,*,ERR=999) FILNAM
         OPEN(UNIT=DEVICE,FILE=FILNAM,STATUS='UNKNOWN')

      ELSE 
         WRITE(6,40)
         STOP

      ENDIF

C     ********* READING OPTION # : 1 ---> RECTANGULAR PATCH
C                                  2 ---> CIRCULAR PATCH
C
      WRITE(6,50)
      READ(5,10,ERR=999) OPTION

      IF(OPTION .EQ. 1) THEN    ! RECTANGULAR PATCH
         CALL RECTANGULAR(ETH_R,EPH_R,NPNTS_TH,NPNTS_PHI,DEVICE) 

C        **** OUTPUT THE E-PLANE AND H-PLANE NORMALIZED RADIATION PATTERNS
C
         WRITE(3,80)
         DO I=1,NPNTS_PHI
            WRITE(3,100) FLOAT(I-1)/FLOAT(NPNTS_PHI-1)*360.0
     $            ,ETH_R(I)
         ENDDO
         WRITE(4,95)
         DO I=(NPNTS_TH-1)/2+1,NPNTS_TH
            WRITE(4,100) FLOAT(I-1)/FLOAT(NPNTS_TH-1)*180.0-90.0
     $               ,EPH_R(I)    
         ENDDO
         DO I=1,(NPNTS_TH-1)/2+1
            WRITE(4,100) 270.0+FLOAT(I-1)/FLOAT(NPNTS_TH-1)*180.0
     $               ,EPH_R(I)    
         ENDDO

      ELSE IF(OPTION .EQ. 2) THEN
         CALL CIRCULAR(ETH_C,EPH_C,NPNTS_CIR,DEVICE) ! CIRCULAR PATCH

C        **** OUTPUT THE E-PLANE AND H-PLANE NORMALIZED RADIATION PATTERNS
C
         WRITE(3,80)
         DO I=1,NPNTS_CIR
            WRITE(3,100) FLOAT(I-1)/FLOAT(NPNTS_CIR-1)*90.0
     $            ,ETH_C(I)
         ENDDO
         DO I=NPNTS_CIR,1,-1
            WRITE(3,100) 360.0-FLOAT(I-1)/FLOAT(NPNTS_CIR-1)*90.0
     $            ,ETH_C(I)
         ENDDO

         WRITE(4,90)
         DO I=1,NPNTS_CIR
            WRITE(4,100) FLOAT(I-1)/FLOAT(NPNTS_CIR-1)*90.0
     $            ,EPH_C(I)
         ENDDO
         DO I=NPNTS_CIR,1,-1
            WRITE(4,100) 360.0-FLOAT(I-1)/FLOAT(NPNTS_CIR-1)*90.0
     $            ,EPH_C(I)
         ENDDO

      ELSE
         WRITE(6,60)
         STOP
         
      ENDIF

      IF(DEVICE .EQ. 6) THEN
         WRITE(DEVICE,5)
      ELSE IF(DEVICE .EQ. 2) THEN
         WRITE(6,5)
         WRITE(DEVICE,5)
      ENDIF


C     ********** FORMAT STATEMENTS **********

 5    FORMAT(/,3X,'*** NOTE:',/,7X,
     $      'THE E-PLANE AMPLITUDE PATTERN IS STORED IN Epl-Micr.dat'
     $      ,/,7X,
     $      'THE H-PLANE AMPLITUDE PATTERN IS STORED IN Hpl-Micr.dat',/
     &    ,7X,'======================================================='
     &      )

 10   FORMAT(I1)
 20   FORMAT(3X,'OUTPUT DEVICE OPTION FOR THE OUTPUT PARAMETERS',/,6X
     $      ,'OPTION (1): SCREEN',/,6X,'OPTION (2): OUTPUT FILE',//,3X
     $      ,'OUTPUT DEVICE = ',$)

 30   FORMAT(3X,'INPUT THE DESIRED OUTPUT FILNAME (in single quotes) = '
     &      ,$)

 40   FORMAT(/,3X,'*** ERROR ***',/,3X
     $      ,'OUTPUTING DEVICE NUMBER SHOULD BE EITHER 1 OR 2')
      
 50   FORMAT(/,3X,'PATCH GEOMETRY OPTION',/,6X
     $      ,'OPTION (1): RECTANGULAR PATCH',/,6X
     $      ,'OPTION (2): CIRCULAR PATCH',//,3X,'OPTION NUMBER = ',$)

 60   FORMAT(/,3X,'***** ERROR *****',/,3X
     $      ,'THE PATCH GEOMETRY OPTION IS INCORRECT',/)
      
 80   FORMAT('# E-PLANE RADIATION PATTERN',/,
     $       '# -------------------------',/,'#')

 90   FORMAT('# H-PLANE RADIATION PATTERN',/,
     $       '# -------------------------',/,'#')

 95   FORMAT('# H-PLANE RADIATION PATTERN',/,
     $      '# NOTE: THIS PATTERN IS ROTATED CCW BY 90 DEGREES',/,
     $      '# -----------------------------------------------',/,'#')

 100  FORMAT(F10.2,F16.4)

      STOP                      ! END OF THE MAIN PROGRAM  
C     

C     ********** ERROR CONDITIONS **********
C
 999  WRITE(6,1000)
 1000 FORMAT(/,3X,'***** ERROR *****',/,3X
     $      ,'INPUT DATA ARE NOT OF THE RIGHT FORMAT',/)

      END




C*******************************************************************
      SUBROUTINE RECTANGULAR(ETH,EPH,NPNTS_TH,NPNTS_PHI,DEVICE)
C*******************************************************************

      INTEGER I,NPNTS_PHI,NPNTS_TH,DEVICE
      REAL FREQ,EPSR,HEIGHT,YO,WIDTH,EREFF,DL,LAMBDA_O,LAMBDA
      REAL LEFF,L,KO,PI,PHI,PHIR,ARG,ETH(NPNTS_PHI),EPH(NPNTS_TH)
      REAL SINC,ETHVAL,ARG1,ARG2,SINC1,SINC2,EPHVAL,EHPBW,HHPBW
      REAL DR,DRDB,G1,G12,GIN,RIN0,RINY0,URECT,F1,F2
      REAL ETHMAX,EPHMAX
      EXTERNAL URECT,F1,F2


      PI=4.0*ATAN(1.0)

C**** INPUT PARAMETERS
C     ----------------
      WRITE(6,210) 
      READ(5,*) FREQ
      WRITE(6,220) 
      READ(5,*) EPSR
      WRITE(6,230) 
      READ(5,*) HEIGHT
      WRITE(6,240) 
      READ(5,*) YO
C     ----------------

C**** ECHO INPUT PARAMETERS
C     ---------------------
      WRITE(DEVICE,250)
      WRITE(DEVICE,255) FREQ
      WRITE(DEVICE,260) EPSR
      WRITE(DEVICE,265) HEIGHT
      WRITE(DEVICE,270) YO

C
C**** COMPUTE THE WIDTH OF THE PATCH
C
      WIDTH=30.0/(2.0*FREQ)*SQRT(2.0/(EPSR+1.0))

C**** COMPUTE EFFECTIVE DIELECTRIC CONSTANTS, EFFECTIVE LENGTH (IN CM)
C**** AND PHYSICAL LENGTH OF THE PATCH (IN CM)
C
      EREFF=(EPSR+1.0)/2.0+(EPSR-1)/(2.0*SQRT(1.0+12.0*HEIGHT/WIDTH))
      DL=0.412*HEIGHT*((EREFF+0.3)*(WIDTH/HEIGHT+0.264))/((EREFF-0.258)
     &      *(WIDTH/HEIGHT+0.8))
      LAMBDA_O=30.0/FREQ
      LAMBDA=30.0/(FREQ*SQRT(EREFF))
      LEFF=30.0/(2.0*FREQ*SQRT(EREFF))
      L=LEFF-2.0*DL

      WRITE(DEVICE,275)
      WRITE(DEVICE,280) WIDTH
      WRITE(DEVICE,285) LEFF
      WRITE(DEVICE,290) L

C**** COMPUTE THE NORMALIZED RADIATED FIELD

C--------- E-PLANE PATTERN: 0 < PHI < 90; 270 < PHI < 360

      KO=2.0*PI/LAMBDA_O
      ETHMAX=SIN(KO/2.0*HEIGHT)/(KO/2.0*HEIGHT)

      DO I=1,NPNTS_PHI
         PHI=FLOAT(I-1)/FLOAT(NPNTS_PHI-1)*360.0
         PHIR=PHI*PI/180.0

         IF(PHI .GT. 90.0 .AND. PHI .LT. 270) THEN
            ETH(I)=-60.00  
         ELSE
            ARG=KO/2.0*HEIGHT*COS(PHIR)

            IF(ARG .EQ. 0.0)THEN
               SINC=1.0
            ELSE
               SINC=SIN(ARG)/ARG
            ENDIF
            
            ETHVAL=SINC*COS(KO/2.0*LEFF*SIN(PHIR))/ETHMAX
            IF (ETHVAL .LT. 0.001) THEN
               ETH(I)=-60.0
            ELSE
               ETH(I)=20.0*ALOG10(ETHVAL)
            ENDIF
         ENDIF
      ENDDO

C--------- COMPUTE THE E-PLANE HPBW

      DO I=1,901                ! TOL=0.1 DEGREES
         PHI=FLOAT(I-1)/10.0
         PHIR=PHI*PI/180.0

         ARG=KO/2.0*HEIGHT*COS(PHIR)

         IF(ARG .EQ. 0.0)THEN
            SINC=1.0
         ELSE
            SINC=SIN(ARG)/ARG
         ENDIF
            
         ETHVAL=SINC*COS(KO/2.0*LEFF*SIN(PHIR))
         IF(ETHVAL .LT. 1.0/SQRT(2.0)) THEN
            EHPBW=PHI           ! ASSUMING MAXIMUM IS AT 0 DEG.
            GOTO 1
         ENDIF
      ENDDO
      WRITE(6,293)
 1    CONTINUE

      WRITE(DEVICE,291) EHPBW


C--------- H-PLANE PATTERN: 0 < TH < 180

      EPHMAX=SIN(KO/2.0*HEIGHT)/(KO/2.0*HEIGHT)

      DO I=1,NPNTS_TH
         TH=FLOAT(I-1)/FLOAT(NPNTS_TH-1)*180.0
         THR=TH*PI/180.0

         ARG1=KO/2.0*HEIGHT*SIN(THR)
         ARG2=KO/2.0*WIDTH*COS(THR)

         IF(ARG1 .EQ. 0.0)THEN
            SINC1=1.0
         ELSE
            SINC1=SIN(ARG1)/ARG1
         ENDIF
         IF(ARG2 .EQ. 0.0)THEN
            SINC2=1.0
         ELSE
            SINC2=SIN(ARG2)/ARG2
         ENDIF
            
         EPHVAL=SIN(THR)*SINC1*SINC2/EPHMAX
         IF (EPHVAL .LT. 0.001) THEN
            EPH(I)=-60.0
         ELSE
            EPH(I)=20.0*ALOG10(EPHVAL)
         ENDIF
      ENDDO


C--------- COMPUTE THE H-PLANE HPBW

      DO I=1,901                ! TOL=0.1 DEGREES
         TH=90.0+FLOAT(I-1)/10.0
         THR=TH*PI/180.0

         ARG1=KO/2.0*HEIGHT*SIN(THR)
         ARG2=KO/2.0*WIDTH*COS(THR)

         IF(ARG1 .EQ. 0.0)THEN
            SINC1=1.0
         ELSE
            SINC1=SIN(ARG1)/ARG1
         ENDIF
         IF(ARG2 .EQ. 0.0)THEN
            SINC2=1.0
         ELSE
            SINC2=SIN(ARG2)/ARG2
         ENDIF
            
         EPHVAL=SIN(THR)*SINC1*SINC2

         IF(EPHVAL .LT. 1.0/SQRT(2.0)) THEN
            HHPBW=TH-90.0       ! ASSUMING MAXIMUM IS AT 90 DEG.
            GOTO 2
         ENDIF
      ENDDO
      WRITE(6,294)
 2    CONTINUE

      WRITE(DEVICE,292) HHPBW

C-------- COMPUTE THE DIRECTIVITY
C
      CALL DIR_RECT(URECT,WIDTH,HEIGHT,LEFF,L,KO,0,180,0,180,DR,DRDB)

      WRITE(DEVICE,295) DR
      WRITE(DEVICE,296) DRDB

C-------- COMPUTE INPUT IMPEDANCE AT Y=0 AND Y=Yo
C
      CALL SINTEGR(F1,WIDTH,HEIGHT,LEFF,L,KO,0,180,G1)
      CALL SINTEGR(F2,WIDTH,HEIGHT,LEFF,L,KO,0,180,G12)

      GIN=2.0*(G1+G12)
      RIN0=1.0/GIN
      RINY0=RIN0*COS(PI*YO/L)**2

      WRITE(DEVICE,297) RIN0
      WRITE(DEVICE,298) RINY0


C     ********** FORMAT STATEMENTS **********

 210  FORMAT(/,3X,'INPUT THE RESONANT FREQUENCY (in GHz) = ',$)
 220  FORMAT(/,3X,'INPUT THE DIELECTRIC CONSTANT OF THE SUBSTRATE = ',$)
 230  FORMAT(/,3X,'INPUT THE HEIGHT OF THE SUBSTRATE (in cm) = ',$)
 240  FORMAT(/,3X,'INPUT THE POSITION OF THE RECESSED FEED POINT',/,
     $       9X,'RELATIVE TO THE LEADING RADIATING EDGE OF THE',/,
     $       9X,'RECTANGULAR PATCH (in cm) = ',$)
 250   FORMAT(/,3X,'INPUT PARAMETERS',/,3X,'================')
 255   FORMAT(/,3X,'RESONANT FREQUENCY (in GHz) = ',F16.4)
 260   FORMAT(3X,'DIELECTRIC CONSTANT OF THE SUBSTRATE = ',F16.4)
 265   FORMAT(3X,'HEIGHT OF THE SUBSTRATE (in cm) = ',F16.4)
 270   FORMAT(3X,'POSITION OF THE RECESSED FEED POINT (in cm) = '
     $       ,F16.4)
 275   FORMAT(/,3X,'OUTPUT PARAMETERS',/,3X,'=================')
 280   FORMAT(/,3X,'PHYSICAL WIDTH OF PATCH (in cm) = ',F16.4)
 285   FORMAT(3X,'EFFECTIVE LENGTH OF PATCH (in cm) = ',F16.4)
 290   FORMAT(3X,'PHYSICAL LENGTH OF PATCH (in cm) = ',F16.4)
 291   FORMAT(3X,'E-PLANE HPBW (in degrees) =', F16.4)
 292   FORMAT(3X,'H-PLANE HPBW (in degrees) =', F16.4)
 293   FORMAT(3X,'PROGRAM COULD NOT FIND E-PLANE HPBW')
 294   FORMAT(3X,'PROGRAM COULD NOT FIND H-PLANE HPBW')
 295   FORMAT(3X,'DIRECTIVITY OF RECTANGULAR PATCH (dimensionless) = '
     $       ,F16.4)
 296   FORMAT(3X,'DIRECTIVITY OF RECTANGULAR PATCH (dB) = ',F16.4)
 297   FORMAT(/,3X,'RESONANT INPUT RESISTANCE AT',/,3X,
     $       'LEADING RADIATING EDGE (y=0) = ',F16.4)
 298   FORMAT(/,3X,'RESONANT INPUT RESISTANCE AT RECESSED FEED ',/,3X,
     $       'POINT FROM LEADING RADIATING EDGE (y=y0) = ',F16.4)


      RETURN
      END




C*******************************************************************
      SUBROUTINE CIRCULAR(ETH_C,EPH_C,NPNTS_CIR,DEVICE)
C*******************************************************************

      INTEGER I,NPNTS_CIR,DEVICE
      REAL FREQ,EPSR,F,LAMBDA_O,HEIGHT,A,AE
      REAL KO,PI,TH,THR,ETH_C(NPNTS_CIR),EPH_C(NPNTS_CIR)
      REAL X,J0,J2,ETHMAX,EPHMAX,TOL,EHPBW,HHPBW
      PARAMETER(TOL=1.0E-5)
      REAL UCIRC
      EXTERNAL UCIRC

      PI=4.0*ATAN(1.0)

C**** INPUT PARAMETERS
C     ----------------
      WRITE(6,310) 
      READ(5,*) FREQ
      WRITE(6,320) 
      READ(5,*) EPSR
      WRITE(6,330) 
      READ(5,*) HEIGHT
C     ----------------

C**** ECHO INPUT PARAMETERS
C     ---------------------
      WRITE(DEVICE,335)
      WRITE(DEVICE,340) FREQ
      WRITE(DEVICE,345) EPSR
      WRITE(DEVICE,350) HEIGHT

C
C**** COMPUTE THE PHYSICAL RADIUS a (in cm) AND EFFECTIVE RADIUS ae
C     (in cm) OF PATCH
C
      LAMBDA_O=30.0/FREQ
      KO=2.0*PI/LAMBDA_O
      F=8.791/(FREQ*SQRT(EPSR))
      A=F/SQRT(1.0+2.0*HEIGHT/(PI*EPSR*F)*(ALOG(PI*F/(2.0*HEIGHT))
     $      +1.7726))
      AE=A*SQRT(1.0+2.0*HEIGHT/(PI*EPSR*A)*(ALOG(PI*A/(2.0*HEIGHT))
     $      +1.7726))


      WRITE(DEVICE,355)
      WRITE(DEVICE,360) A
      WRITE(DEVICE,365) AE

C**** COMPUTE THE NORMALIZED RADIATED FIELDS

C--------- E-PLANE AND H-PLANE PATTERNS: 0 < TH < 90

      DO I=1,NPNTS_CIR
         TH=FLOAT(I-1)/FLOAT(NPNTS_CIR-1)*90.0
         THR=TH*PI/180.
         X=KO*AE*SIN(THR)
         IF(X .EQ. 0.0)THEN
            J0= BESSJ0(X)
            J2=0.0
         ELSE
            J0= BESSJ0(X)
            J2= BESSJ(2,X)
         ENDIF

         ETH_C(I)=J0-J2         ! E-PLANE PATTERN
         EPH_C(I)=(J0+J2)*COS(THR) ! H-PLANE PATERN
      ENDDO

C     **** NORMALIZE THE PATTERNS AND TRANSLATE INTO dB
C
      ETHMAX=ABS(ETH_C(1))
      EPHMAX=ABS(EPH_C(1))
      DO I=2,NPNTS_CIR
         IF(ABS(ETH_C(I)) .GT. ETHMAX) ETHMAX=ABS(ETH_C(I))
         IF(ABS(EPH_C(I)) .GT. EPHMAX) EPHMAX=ABS(EPH_C(I))
      ENDDO

      DO I=1,NPNTS_CIR
         ETH_C(I)=ETH_C(I)/ETHMAX
         EPH_C(I)=EPH_C(I)/EPHMAX
         IF(ABS(ETH_C(I)-0.0) .LT. TOL) THEN 
            ETH_C(I)=-60.0      ! IN DB
         ELSE
            ETH_C(I)=20.*ALOG10(ETH_C(I))
         ENDIF
         IF(ABS(EPH_C(I)-0.0) .LT. TOL) THEN
            EPH_C(I)=-60.0      ! IN DB
         ELSE
            EPH_C(I)=20.*ALOG10(EPH_C(I))
         ENDIF
      ENDDO

C---- COMPUTE THE E-PLANE AND H-PLANE HPBWS
C

      DO I=1,NPNTS_CIR
         TH=FLOAT(I-1)/FLOAT(NPNTS_CIR-1)*90.0
         IF(ETH_C(I) .LT. -3.0) THEN ! 3 dB down
            EHPBW=TH
            GOTO 7
         ENDIF
      ENDDO
      WRITE(6,380)
 7    CONTINUE

      WRITE(DEVICE,370) EHPBW

      DO I=1,NPNTS_CIR
         TH=FLOAT(I-1)/FLOAT(NPNTS_CIR-1)*90.0
         IF(EPH_C(I) .LT. -3.0) THEN ! 3 dB down
            HHPBW=TH
            GOTO 8
         ENDIF
      ENDDO
      WRITE(6,385)
 8    CONTINUE

      WRITE(DEVICE,375) HHPBW

C-------- COMPUTE THE DIRECTIVITY
C
      CALL DIR_CIR(UCIRC,A,AE,KO,0,90,0,360,DR,DRDB)

      WRITE(DEVICE,390) DR
      WRITE(DEVICE,395) DRDB
         


C     ********** FORMAT STATEMENTS **********

 310  FORMAT(/,3X,'INPUT THE RESONANT FREQUENCY (in GHz) = ',$)
 320  FORMAT(/,3X,'INPUT THE DIELECTRIC CONSTANT OF THE SUBSTRATE = ',$)
 330  FORMAT(/,3X,'INPUT THE HEIGHT OF THE SUBSTRATE (in cm) = ',$)
 335  FORMAT(/,3X,'INPUT PARAMETERS',/,3X,'================')
 340  FORMAT(/,3X,'RESONANT FREQUENCY (in GHz) = ',F16.4)
 345  FORMAT(3X,'DIELECTRIC CONSTANT OF THE SUBSTRATE = ',F16.4)
 350  FORMAT(3X,'HEIGHT OF THE SUBSTRATE (in cm) = ',F16.4)
 355   FORMAT(/,3X,'OUTPUT PARAMETERS',/,3X,'=================')
 360   FORMAT(/,3X,'PHYSICAL RADIUS OF THE PATCH (in cm) = ',F16.4)
 365   FORMAT(3X,'EFFECTIVE RADIUS OF THE PATCH (in cm) = ',F16.4)
 370   FORMAT(3X,'E-PLANE HPBW (in degrees) =', F16.4)
 375   FORMAT(3X,'H-PLANE HPBW (in degrees) =', F16.4)
 380   FORMAT(3X,'PROGRAM COULD NOT FIND E-PLANE HPBW')
 385   FORMAT(3X,'PROGRAM COULD NOT FIND H-PLANE HPBW')
 390   FORMAT(3X,'DIRECTIVITY OF CIRCUGULAR PATCH (dimensionless) = '
     $       ,F16.4)
 395   FORMAT(3X,'DIRECTIVITY OF CIRCULAR PATCH (dB) = ',F16.4)



      RETURN
      END


C*******************************************************************
      FUNCTION BESSJ0(X)
C
C*** TAKEN FROM THE "NUMERICAL RECIPES IN FORTRAN" SECOND EDITION
C*******************************************************************
      REAL Y,P1,P2,P3,P4,P5,Q1,Q2,Q3,Q4,Q5,R1,R2,R3,R4,R5,R6,
     $    S1,S2,S3,S4,S5,S6
      DATA P1,P2,P3,P4,P5/1.E0,-.1098628627E-2,.2734510407E-4,
     $      -.2073370639E-5,.2093887211E-6/, Q1,Q2,Q3,Q4,Q5/-
     $      .1562499995E-1,.1430488765E-3,-.6911147651E-5,.7621095161E-6
     $      ,-.934945152E-7/
      DATA R1,R2,R3,R4,R5,R6/57568490574.E0,-13362590354.E0,651619640
     $      .7E0,-11214424.18E0,77392.33017E0,-184.9052456E0/,S1,S2,S3
     $      ,S4,S5,S6/57568490411.E0,1029532985.E0,9494680.718E0,59272
     $      .64853E0,267.8532712E0,1.E0/
      IF(ABS(X).LT.8.)THEN
        Y=X**2
        BESSJ0=(R1+Y*(R2+Y*(R3+Y*(R4+Y*(R5+Y*R6)))))
     $      /(S1+Y*(S2+Y*(S3+Y*(S4+Y*(S5+Y*S6)))))
      ELSE
        AX=ABS(X)
        Z=8./AX
        Y=Z**2
        XX=AX-.785398164
        BESSJ0=SQRT(.636619772/AX)*(COS(XX)*(P1+Y*(P2+Y*(P3+Y*(P4+Y
     $      *P5))))-Z*SIN(XX)*(Q1+Y*(Q2+Y*(Q3+Y*(Q4+Y*Q5)))))
      ENDIF
      RETURN
      END


C*******************************************************************
      FUNCTION BESSJ1(X)
C
C*** TAKEN FROM THE "NUMERICAL RECIPES IN FORTRAN" SECOND EDITION
C*******************************************************************
      REAL Y,P1,P2,P3,P4,P5,Q1,Q2,Q3,Q4,Q5,R1,R2,R3,R4,R5,R6,
     $      S1,S2,S3,S4,S5,S6
      DATA R1,R2,R3,R4,R5,R6/72362614232.E0,-7895059235.E0,242396853.1E0
     $      ,-2972611.439E0,15704.48260E0,-30.16036606E0/,S1,S2,S3,S4,S5
     $      ,S6/144725228442.E0,2300535178.E0,18583304.74E0,99447
     $      .43394E0,376.9991397E0,1.E0/
      DATA P1,P2,P3,P4,P5/1.E0,.183105E-2,-.3516396496E-4,.2457520174E-5
     $      ,-.240337019E-6/, Q1,Q2,Q3,Q4,Q5/.04687499995E0,-
     $      .2002690873E-3,.8449199096E-5,-.88228987E-6,.105787412E-6/

      IF(ABS(X).LT.8.)THEN
        Y=X**2
        BESSJ1=X*(R1+Y*(R2+Y*(R3+Y*(R4+Y*(R5+Y*R6)))))
     $      /(S1+Y*(S2+Y*(S3+Y*(S4+Y*(S5+Y*S6)))))
      ELSE
        AX=ABS(X)
        Z=8./AX
        Y=Z**2
        XX=AX-2.356194491
        BESSJ1=SQRT(.636619772/AX)*(COS(XX)*(P1+Y*(P2+Y*(P3+Y*(P4+Y
     $      *P5))))-Z*SIN(XX)*(Q1+Y*(Q2+Y*(Q3+Y*(Q4+Y*Q5)))))
     $      *SIGN(1.,X)
      ENDIF
      RETURN
      END



C*******************************************************************
      FUNCTION BESSJ(N,X)
C
C*** TAKEN FROM THE "NUMERICAL RECIPES IN FORTRAN" SECOND EDITION
C*******************************************************************
      PARAMETER (IACC=40,BIGNO=1.E10,BIGNI=1.E-10)
      IF(N.LT.2)PAUSE 'Bad argument N in BESSJ'
      TOX=2./X
      IF(X.GT.FLOAT(N))THEN
        BJM=BESSJ0(X)
        BJ=BESSJ1(X)
        DO 11 J=1,N-1
          BJP=J*TOX*BJ-BJM
          BJM=BJ
          BJ=BJP
11      CONTINUE
        BESSJ=BJ
      ELSE
        M=2*((N+INT(SQRT(FLOAT(IACC*N))))/2)
        BESSJ=0.
        JSUM=0
        SUM=0.
        BJP=0.
        BJ=1.
        DO 12 J=M,1,-1
          BJM=J*TOX*BJ-BJP
          BJP=BJ
          BJ=BJM
          IF(ABS(BJ).GT.BIGNO)THEN
            BJ=BJ*BIGNI
            BJP=BJP*BIGNI
            BESSJ=BESSJ*BIGNI
            SUM=SUM*BIGNI
          ENDIF
          IF(JSUM.NE.0)SUM=SUM+BJ
          JSUM=1-JSUM
          IF(J.EQ.N)BESSJ=BJP
12      CONTINUE
        SUM=2.*SUM-BJ
        BESSJ=BESSJ/SUM
      ENDIF
      RETURN
      END

C     ******************************************************************
      SUBROUTINE DIR_RECT(FNCT,WIDTH,HEIGHT,LEFF,L,KO,TL,TU,PL,PU,DR
     &      ,DRDB)
C     ******************************************************************
      INTEGER TL,TU,PL,PU,PLL,TLL
      REAL FNCT,WIDTH,HEIGHT,LEFF,L,KO,DR,DRDB

      PI=4.0*ATAN(1.0)
      THETA=PI/180.0
      PHI=PI/180.0
      PLL=PL+1

      PRAD=0.0
      UMAX=0.0
      DO 1 J=PLL,PU
         XJ=FLOAT(J)*PI/180.0
         TLL=TL+1
         DO 2 I=TLL,TU
            XI=FLOAT(I)*PI/180.0
            F=FNCT(XI,XJ,WIDTH,HEIGHT,LEFF,L,KO)
            IF (F.GT.UMAX) UMAX=F
            UA=THETA*PHI*F*SIN(XI)
            PRAD=PRAD+UA
 2       CONTINUE
 1    CONTINUE

      DR=4.0*PI*UMAX/PRAD
      DRDB=10.0*LOG10(DR)

      RETURN
      END

C     *****************************************************************
      REAL FUNCTION URECT(THETA,PHI,WIDTH,HEIGHT,LEFF,L,KO)
C     *****************************************************************
      REAL SINC1,SINC2,THETA,PHI,WIDTH,HEIGHT,LEFF,L,KO,TOL
      REAL ARG1,ARG2,ARG3
      PARAMETER(TOL=1.0E-6)

      ARG1=KO*HEIGHT/2.0*SIN(THETA)*COS(PHI)
      ARG2=KO*WIDTH/2.0*COS(THETA)
      ARG3=KO*LEFF/2.0*SIN(THETA)*SIN(PHI)

      IF(ABS(ARG1-0.0) .LT. TOL) THEN
         SINC1=1.0
      ELSE
         SINC1=SIN(ARG1)/ARG1
      ENDIF
      IF(ABS(ARG2-0.0) .LT. TOL) THEN
         SINC2=1.0
      ELSE
         SINC2=SIN(ARG2)/ARG2
      ENDIF

      URECT=SINC1**2*SINC2**2*SIN(THETA)**2*COS(ARG3)**2
      
      RETURN
      END

C     ******************************************************************
      SUBROUTINE SINTEGR(FNCT,WIDTH,HEIGHT,LEFF,L,KO,TL,TU,G)
C     ******************************************************************
      INTEGER TL,TU,TLL
      REAL FNCT,WIDTH,HEIGHT,LEFF,L,KO,RES,G

      PI=4.0*ATAN(1.0)
      THETA=PI/180.0

      RES=0.0
      TLL=TL+1
      DO 1 I=TLL,TU
         XI=FLOAT(I)*PI/180.0
         F=FNCT(XI,WIDTH,HEIGHT,LEFF,L,KO)
         RES=RES+THETA*F*SIN(XI)
 1    CONTINUE

      G=RES/(120.0*PI**2)

      RETURN
      END

C     *****************************************************************
      REAL FUNCTION F1(THETA,WIDTH,HEIGHT,LEFF,L,KO)
C     *****************************************************************
      REAL SINC,THETA,WIDTH,HEIGHT,LEFF,L,KO,TOL
      REAL ARG
      PARAMETER(TOL=1.0E-6)

      ARG=KO*WIDTH/2.0*COS(THETA)

      IF(ABS(ARG-0.0) .LT. TOL) THEN
         SINC=1.0
      ELSE
         SINC=SIN(ARG)/ARG
      ENDIF

      F1=(KO*WIDTH/2.0)**2*SINC**2*SIN(THETA)**2
      
      RETURN
      END

C     *****************************************************************
      REAL FUNCTION F2(THETA,WIDTH,HEIGHT,LEFF,L,KO)
C     *****************************************************************
      REAL SINC,THETA,WIDTH,HEIGHT,LEFF,L,KO,TOL
      REAL ARG
      PARAMETER(TOL=1.0E-6)

      ARG=KO*WIDTH/2.0*COS(THETA)

      IF(ABS(ARG-0.0) .LT. TOL) THEN
         SINC=1.0
      ELSE
         SINC=SIN(ARG)/ARG
      ENDIF

      F2=(KO*WIDTH/2.0)**2*SINC**2*SIN(THETA)**2
     $      *BESSJ0(KO*L*SIN(THETA))
      
      RETURN
      END



C     ******************************************************************
      SUBROUTINE DIR_CIR(FNCT,A,AE,KO,TL,TU,PL,PU,DR,DRDB)
C     ******************************************************************
      INTEGER TL,TU,PL,PU,PLL,TLL
      REAL FNCT,A,AE,KO,THETA,PHI,PI,PRAD,UMAX,DR,DRDB

      PI=4.0*ATAN(1.0)
      THETA=PI/180.0
      PHI=PI/180.0
      PLL=PL+1

      PRAD=0.0
      UMAX=0.0
      DO 1 J=PLL,PU
         XJ=FLOAT(J)*PI/180.0
         TLL=TL+1
         DO 2 I=TLL,TU
            XI=FLOAT(I)*PI/180.0
            F=FNCT(XI,XJ,A,AE,KO)
            IF (F.GT.UMAX) UMAX=F
            UA=THETA*PHI*F*SIN(XI)
            PRAD=PRAD+UA
 2       CONTINUE
 1    CONTINUE

      DR=4.0*PI*UMAX/PRAD
      DRDB=10.0*LOG10(DR)

      RETURN
      END

C     *****************************************************************
      REAL FUNCTION UCIRC(THETA,PHI,A,AE,KO)
C     *****************************************************************
      REAL THETA,PHI,A,AE,KO
      REAL X,J0,J2,J02P,J02


      X=KO*AE*SIN(THETA)

      IF(X .EQ. 0.0)THEN
         J0= BESSJ0(X)
         J2=0.0
      ELSE
         J0= BESSJ0(X)
         J2= BESSJ(2,X)
      ENDIF

      J02P=J0-J2
      J02=J0+J2

      UCIRC=(J02P*COS(PHI))**2+(J02*COS(THETA)*SIN(PHI))**2
      
      RETURN
      END

