C     ***************************************************************
C     THIS IS A FORTRAN PROGRAM THAT COMPUTES THE
C
C        I.   MAXIMUM DIRECTIVITY (DIMENSIONLESS AND IN dB)
C        II.  RADIATION RESISTANCE
C
C     FOR A SMALL (CONSTANT CURRENT) LOOP.  THE LOOP IS RADIADING 
C     INTO FREE SPACE.
C
C     THE DIRECTIVITY AND RADIATION RESISTANCE ARE CALCULATED USING 
C     THE TRAILING EDGE METHOD IN INCREMENTS OF 1 DEGREE IN THETA.
C
C        ** INPUT PARAMETERS
C        1.  A: LOOP RADIUS (IN WAVELENGTHS)
C
C        ** NOTE:
C        THE FAR-ZONE ELECTRIC FIELD COMPONENT E-PHI EXISTS FOR
C        0 < THETA < 180 AND 0 < PHI < 360.
C     ***************************************************************
      
C     Declaration of Variables
C     ------------------------
      REAL A,X,XI,PI,E,THETA,UMAX,PRAD,D,DDB,RR,TOL
      PARAMETER(TOL=1.0E-5)
      INTEGER DEVICE,IER
      CHARACTER*12 FILNAM

C     ********** CHOICE OF OUTPUT ***********************************
C
      WRITE(6,10)
      READ(5,20,ERR=999) DEVICE

      IF(DEVICE .EQ. 1) THEN
         DEVICE=6

      ELSE IF(DEVICE .EQ. 2) THEN
         WRITE(6,30)
         READ(5,*,ERR=999) FILNAM
         OPEN(UNIT=DEVICE,FILE=FILNAM,STATUS='UNKNOWN')

      ELSE 
         WRITE(6,40)
         STOP

      ENDIF

C     Definition of constants and initialization
C     ------------------------------------------

      PI=4.0*ATAN(1.0)
      E=120.0*PI
      THETA=PI/180.0
      UMAX=0.0
      PRAD=0.0

C     Input the radius of the loop
C     ----------------------------

      WRITE(6,50)
      READ(5,*,ERR=999) A

      WRITE(DEVICE,60)
      WRITE(DEVICE,70) A

C     -------------------------- Main Program --------------------------

      DO 1 I=1,180
         XI=FLOAT(I)*PI/180.0
         X=2.0*PI*A*SIN(XI)
         IF(ABS(X) .LT. TOL) THEN
            F=0.0
         ELSE
            CALL BESJ(X,1,F,0.0001,IER)
         ENDIF
         IF (IER .EQ. 2) WRITE (DEVICE,120)
         IF (IER .EQ. 3) WRITE (DEVICE,130)
         U=A**2*(2.0*PI)**2/8.0*E*F**2
         IF (U .GT. UMAX) UMAX=U
         UA=U*SIN(XI)*THETA*2.0*PI
         PRAD=PRAD+UA
 1    CONTINUE
      D=(4.0*PI*UMAX)/PRAD
      DDB=10.0*ALOG10(D)
      RR=2.0*PRAD

      WRITE(DEVICE,80)
      WRITE(DEVICE,90) D
      WRITE(DEVICE,100) DDB
      WRITE(DEVICE,110) RR

C     ********* FORMAT STATEMENTS *************************************

 10   FORMAT(3X,'OUTPUT DEVICE OPTION',/,6X
     $      ,'OPTION (1): SCREEN',/,6X,'OPTION (2): OUTPUT FILE',//,3X
     $      ,'OUTPUT DEVICE = ',$)

 20   FORMAT(I1)

 30   FORMAT(3X
     &      ,'INPUT THE DESIRED OUTPUT FILENAME (in single quotes) = ',$
     &      )
 40   FORMAT(/,3X,'*** ERROR ***',/,3X
     $      ,'OUTPUTING DEVICE NUMBER SHOULD BE EITHER 1 OR 2')
 50   FORMAT(/,3X,'RADIUS OF LOOP IN WAVELENGTHS = ',$)


 60   FORMAT(/,3X,'INPUT PARAMETERS',/,3X,'================')
 70   FORMAT(/,3X,'RADIUS OF LOOP IN WAVELENGTHS = ',F16.4,/)

 80   FORMAT(/,3X,'OUTPUT PARAMETERS',/,3X,'=================',/)
 90   FORMAT(3X,'DIRECTIVITY (DIMENSIONLESS) = ',F16.4)
 100  FORMAT(3X,'DIRECTIVITY (dB) = ',F16.4)
 110  FORMAT(3X,'RADIATION RESISTANCE (Ohms) = ',F16.4)
 120  FORMAT(3X
     &      ,'*** ERROR CODE: ARGUMENT OF BESSEL FUNCTION IS ',/,19X
     &      ,'NEGATIVE OR ZERO')
 130  FORMAT(3X,'REQUIRED ACCURACY NOT OBTAINED')

      STOP                      ! END OF THE MAIN PROGRAM 

C     ********** ERROR CONDITIONS *************************************
C
 999     WRITE(6,1000)
 1000    FORMAT(/,3X,'***** ERROR *****',/,3X
     $         ,'INPUT DATA ARE NOT OF THE RIGHT FORMAT',/)

      END


C     ****************************************************************
C     This subroutine computes the J Bessel function for a given 
C     argument and order.
C
C     Description of parameters: 
C               X - argument of the J Bessel function desired 
C               N - order of the J Bessel function desired 
C              BJ - resultant of the J Bessel function 
C               D - required accuracy
C             IER - resultant error code where  
C                   IER=0 no error
C                   IER=1 N is negative
C                   IER=2 X is negative or zero
C                   IER=3 required accuracy not obtained
C                   IER=4 range of N compared to X not correct (see remarks)
C     Remarks
C         N must be greater than or equal to zero, 
C         but it must be less than
C
C             20+10*X-X**2/3  for X less than or equal to 15
C             90+X/2          for X greater than 15
C                   
C     ****************************************************************
      
      SUBROUTINE BESJ(X,N,BJ,D,IER)

      BJ=.0
      IF(N) 10,20,20
 10   IER=1
      RETURN
 20   IF(X) 30,30,31
 30   IER=2    
      RETURN
 31   IF (X-15.) 32,32,34
 32   NTEST=20.+10.*X-X**2/3
      GO TO 36
 34   NTEST=90.+X/2.
 36   IF (N-NTEST) 40,38,38
 38   IER=4
      RETURN
 40   IER=0
      N1=N+1
      BPREV=.0
      IF (X-5.) 50,60,60
 50   MA=X+6.
      GO TO 70
 60   MA=1.4*X+60./X
 70   MB=N+IFIX(X)/4+2
      MZERO=MAX0(MA,MB)
      MMAX=NTEST
 100  DO 190 M=MZERO,MMAX,3
      FM1=1.0E-28
      FM=.0
      ALPHA=.0
      IF (M-(M/2)*2) 120,110,120
 110  JT=-1
      GO TO 130
 120  JT=1
 130  M2=M-2
      DO 160 K=1,M2
      MK=M-K
      BMK=2.*FLOAT(MK)*FM1/X-FM
      FM=FM1
      FM1=BMK
      IF (MK-N-1) 150,140,150
 140  BJ=BMK
 150  JT=-JT
      S=1+JT
 160  ALPHA=ALPHA+BMK*S
      BMK=2.*FM1/X-FM
      IF (N) 180,170,180
 170  BJ=BMK
 180  ALPHA=ALPHA+BMK
      BJ=BJ/ALPHA
      IF (ABS(BJ-BPREV)-ABS(D*BJ)) 200,200,190
 190  BPREV=BJ
      IER=3
 200  RETURN
      END

