C
C **************************************************************
C
      PROGRAM Log_Perd
C
C  REVISION 1.0
C
C  This program is the main driver for the log-periodic dipole
C  array analysis.  In this routine, the initial data are loaded,
C  the antenna is designed, and the output are controlled.
C  The analysis is run (subroutine da.for) only if it is necessary.
C  Otherwise, a design summary is output.
C
C ************************************************************** 
C
C  CREDITS
C
C     This program and all its subroutines were created by
C  Mr. Chris Bishop with the exception of the matrix inversion
C  routines, LUSOLV and LUDEC, and the sine and cosine integrals,
C  SI and CI.  The matrix inversion routines were created by
C  Dr. James T. Aberle, and the sine and cosine integrals were
C  created by Mr. Anastasis Polycarpou.  In both cases the routines
C  were made for the Telecommunications Research Center at Arizona
C  State University.
C
C ************************************************************** 
C
C     This FORTRAN program designs and analyzes a log periodic dipole
C     array based on the geometry of Figure 11.9(a), the data in
C     Figures 11.13 and 11.14, and the design equations of 
C     (11-28) - (11-34).
C
C     The user can tailor the design and analysis by appropriately 
C     choosing the following options and input parameters.
C
C       OPTION I:
C         1.  Specify Sigma and Tau.  This sets the directivity.
C         2.  Specify the directivity.  This sets Sigma and Tau for
C             an optimum directivity design.
C
C       OPTION II:
C         1.  Choose the boom spacing and boom diameter.  This sets the
C             characteristic impedance of the transmission line.
C         2.  Choose the desired input impedance and boom diameter.
C             This sets the boom spacing.
C
C       OPTION III:
C         1.  Round the calculated element diameters to the nearest
C             available wire diameters.
C         2.  Do not round the calculated element diameters.
C
C     After choosing the design options, proceed to the analysis
C     options.  The output depends on the analysis.
C
C       OPTION IV:
C         1.  No analysis.
C             Output:  Design summary -- list of physical parameters
C                         and geometry.
C
C         2.  Analyze at a single frequency.
C             Output:  Design summary.
C                      E- and H-plane and/or custom plane pattern at
C                         analysis frequency.
C                      Critical parameters at analysis frequency.
C
C         3.  Analyze over a frequency band.
C             Output:  Design summary.
C                      Critical parameters as a function of frequency
C                         including gain on boresight.
C
C         4.  Analyze at a single frequency and over a frequency band.
C             Output:  Design summary.
C                      E-, H-, or custom plane pattern at analysis
C                         frequency.
C                      Critical parameters at analysis frequency.  
C                      Critical parameters as a function of frequency
C                         including gain on boresight.
C
C
C     To complete the design and analysis of any option, the following
C     input and output parameters generally will be either specified or
C     calculated.  Not all parameters are required for all design and
C     analysis options.
C
C     ** INPUT PARAMETERS FOR ARRAY DESIGN
C     1. D0     = Desired directivity (dBi)
C     2. Flow   = Lower design frequency (MHz)
C     3. Fhigh  = Upper design frequency (MHz)
C     4. Rs     = Source resistance (Ohms)
C     5. ZCin   = Characteristic impedance of input line (ohms)
C     6. Rin    = Desired input impedance (real).  Typically this is
C                 equal to the characteristic impedance of the input
C                 transmission line (ZCin).
C     7. LLin   = Length of input transmission line from the source
C                 (external to the antenna) to the shortest element (m)
C     8. LD     = Desired length-to-diameter ratio of elements
C     9. LLout  = Length of termination transmission line from the
C                 longest element to the load (m)
C    10. Zout   = Termination impedance (ohms).  Typically this is
C                 equal to the characteristic impedance of the input
C                 transmission line (ZCin).
C    11. AFSC   = Analysis frequency for custom plane pattern (MHz)
C    12. AFSEH  = Analysis frequency for E- and H- plane patterns (MHz)
C    13. AFlow  = Lowest frequency for swept frequency data (MHz)
C    14. AFhigh = Highest frequency for swept frequency data (MHz)
C    15. Navail = # of available element diameters
C    16. Davail = Diameters of available wires or tubes (cm)
C    17. Title  = Name of design
C
C    ** OUTPUT PARAMETERS OF ARRAY DESIGN
C    1. N       = Number of antenna elements (integer)
C    2. L       = Lengths of antenna elements (m)
C    3. ZL      = Station (position) of each array element (m)
C    4. D       = Diameter of each array element (cm)
C    5. VSWR    = VSWR in input transmission line
C    6. ZinA    = Actual input impedance of design (ohms)
C    7. FTBR    = Front-to-back ratio of amplitude pattern (dB)
C    8. FSLL    = Front-to-maximum sidelobe level (dB)
C    9. Iin     = Current in input transmission line (A)
C   10. Iel     = Currents in array elements (A)
C   11. Iout    = Current in termination transmission line (A)
C   12. Vin     = Voltage in input transmission line (V)
C   13. Vel     = Voltages in array elements (V)
C   14. Vout    = Voltage in termination transmission line (V)
C   15. Epat    = E-plane pattern
C   16. Hpat    = H-plane pattern
C   17. Cpat    = Any desired custom plane pattern
C

C
C  The following variables are used in the analysis
C  TITLE      : Title of antenna design.
C  Fhigh      : Upper design frequency.
C  Flow       : Lower design frequency.
C  Tau        : Geometric ratio.
C  Sigma      : Spacing factor.
C  D0         : Desired Directivity.                      
C  LD         : Designed Length to Diameter ratio.
C  LLin       : Length of source transmission line.                                 
C  ZCin       : Characteristic impedance of source transmission line.
C  Rin        : Desired input impedance of array.
C  LLout      : Length of termination transmission line.
C  Zout       : Complex termination impedance.  Usually the imaginary
C               part is set to zero.
C  AFhigh     : Upper analysis frequency.
C  AFlow      : Lower analysis frequency.
C  AFpowr     : Number of frequency steps per decade
C  AFSEH      : Frequency at which to make E- and H- plane pattern plots.
C  AFSC       : Frequency at which to make custom pattern plot.
C  Phi        : Spherical coordinate.  0 deg = x axis.
C  Navail     : Number of available tape widths.
C  Davail(15) : Array of tape width sizes.           
C  OutFlag*8  : 1 : Summarize Design.
C             : 2 : E- plane Pattern.
C             : 3 : H- plane Pattern.
C             : 4 : Custom Plane Pattern.
C             : 5 : Data vs. Frequency.   
C             : 6 : Output to Screen.
C             : 7 : Output to Printer File.
C             : 8 : Number of current analysis
C                   eg. '3' means the H-plane pattern is being calculated.
C  Quant*2    : 1 : Quantize Boom Diameter.
C             : 2 : Quantize Element Diameter.
C  SB         : Center to center spacing of the tube (cm).
C  DB         : Diameter of transmission line tube (cm).
C  StopFlag   : TRUE = terminate execution.

C  AFstep     : Step size between analysis points, related to AFpowr.
C  Alpha      : Related to Tau and Sigma.  This is one-half of
C               the apex angle.
C  Bs         : Designed bandwidth.
C  Lmax       : Wavelength (in space) of lowest frequency.
C  Lnom       : Nominal length of boom.   
C  N          : Number of elements in the array (maximum = 30).
C  L(30)      : Array of dipole lengths.
C  ZL(30)     : Array of distances from a reference (not necessarily
C               the apex) to each element.
C  D(30)      : Array of dipole diameters.
C  Kavg       : Average length to diameter ratio.
C  Dbar       : Error of actual tape width to desired tape width.
C  Dsave      : Stores a tape width for further comparison.
C  ZA         : Approximate impedance of active region of array.
C  Sigmap     : Mean spacing factor.
C  dummy1     : Intermediate variable.
C  ZO         : Characteristic impedance of transmission line.
C  WATTS      : Real power accepted by the antenna.  User specified.
C               This variable is used to scale the current distributions
C  FMHZ       : Current analysis frequency.

C  c          : Speed of light in a vaccuum.
C  i, j       : Counters.
C  PI         : 3.14 etc.
C
C **************************************************************
C
C  These variables are used in the input routine   
      IMPLICIT NONE
      
      character*60 TITLE
      real         Fhigh, Flow, Tau, Sigma, D0, LD, Rs, LLin
      complex      ZCin
      real         Rin, LLout
      complex      Zout
      real         AFhigh, AFlow, AFpowr, AFSEH, AFSC, Phi 
      integer      Navail
      real         Davail(15)
      character*8  OutFlag
      character*2  Quant
      real         SB, DB      
      character*1  StopFlag
      
C  These parameters are used in the design routine
      integer      ms
      parameter   (ms = 30)
      real         AFstep, Alpha, Bs, Lmax, Lnom
      integer      N
      real         L(ms), ZL(ms+1), D(ms), Kavg, Dbar
      real         Dsave
      complex      ZA
      real         Sigmap, dummy1
      complex      ZO
      real         WATTS
      integer      i, j
      real         PI, C

C  These parameters are used in the analysis routine
      real         FMHz      
      
      PI = 2.0 * acos(0.0)
      C  = 299.7925
C
C
      CALL Input (TITLE, Fhigh, Flow, Tau, Sigma, D0, LD, Rs, LLin,
     1            ZCin, Rin, LLout, Zout, AFhigh, AFlow, AFpowr,
     2            AFSEH, AFSC, Phi, Navail, Davail, 
     3            OutFlag, Quant, SB, DB, StopFlag)
      
C     If the user decided to quit the program     
      IF (StopFlag.eq.'Y') GOTO 5000
      
      CALL OutputMessage

C     STEP 1.  Find Tau and Sigma if directivity was specified
      IF (Sigma.eq.0.0) CALL FindST (D0, Sigma, Tau)

C
C **************************************************************
C
C  DESIGN A LOG-PERIODIC DIPOLE ARRAY
C
C **************************************************************
C

C
C  Begin calculations
C
C
C  STEP 2.  Apex Angle
      Alpha  = atan( (1-Tau)/(4.0*Sigma) )
C  STEP 3.  Designed Bandwidth
      Bs     = (Fhigh/Flow)*(1.1+(7.7*(1.0-Tau)**2)/tan(Alpha))
C  Wavelength at lowest frequency
      Lmax   = C / Flow
C  STEP 4.  Nominal length of antenna boom 
      Lnom   = Lmax / 4.0 * (1.0 - 1.0/Bs)/tan(Alpha)
C  STEP 4.  Number of elements
      N      = 1 - log(Bs)/log(Tau)
C  Error trap if this array has too many elements
      if (N.gt.ms) then
        write (*,*) 'Program Terminated --'
        write (*,*) '  Array requires too mny elements'   
        write (*,*) '  N = ',N
        write (*,*) '  A maximum of ',ms,' elements is allowed.'
        write (*,*)
        write (*,*) '  To increase ms, change its value in each'
        write (*,*) '  subroutine.'
        STOP
      endif

C  Calculate element lengths, positions, and diameters
C  and convert to inches
      L (N)  = Lmax / 2.0
      ZL(N)  = L(N) / 2.0  / tan(Alpha)
      D (N)  = L(N) / LD   / 0.01    
      
C
      DO j = N-1,1,-1
        L (j) = L (j+1)*Tau
        ZL(j) = ZL(j+1)*Tau
        D (j) = D (j+1)*Tau
      END DO

C  Quantize elements to nearest tube diameter
C  and find true L/D ratio, Kavg
      if (Quant(2:2).eq.'Y') then
        Kavg = 0.0
        DO 60, j = 1, N, 1
          IF (NAVAIL.ne.0) THEN
            Dbar = 1e6
            DO 50, i = 1, NAVAIL, 1
              IF (abs(D                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                - 2.25)
C
        Sigmap = Sigma / sqrt(Tau)
        dummy1 = 1.0/(8.0*Sigmap*ZA/Rin)
      
C  This line replaces Figure 11.14  The source is Carrel [10]
        ZO = Rin*(dummy1 + sqrt(dummy1+1))
      
C
C  Note:  SB = DB * cosh( ZO/120 )
C         cosh(x) = 0.5 * (exp(x) + exp(-x))
        SB = DB * (0.5 * (exp(ZO/120.0) + exp(-ZO/120.0)))
      ENDIF
C
C  Make a design summary if desired
      if (OutFlag(1:1).eq.'Y') then
        write (*,*) 'writing file LP_DES.OUT    -- Design Summary'
        call Output2(N, ZL, L, D, ZO, TITLE, Fhigh, Flow, Tau, Sigma, 
     1               D0, LD, Rs, LLin, ZCin, Rin, LLout, Zout, SB, DB)
      endif

C  ANALYZE THE ANTENNA
C  Only execute the analysis routine if needed

C  Clear swept frequency output files
C  Only clear them if we need them
      if (OutFlag(5:5).eq.'Y') then
        write (*,*) 'Deleting file LP_SWEPT.DAT'
        OPEN  (1, FILE   = 'LP_swept.dat')
        CLOSE (1, STATUS = 'DELETE')
        write (*,*) 'Deleting file LP_GAIN.DAT'
        OPEN  (1, FILE   = 'LP_gain.dat')
        CLOSE (1, STATUS = 'DELETE')
        write (*,*) 'Deleting file LP_FTBR.DAT'
        OPEN  (1, FILE   = 'LP_ftbr.dat')
        CLOSE (1, STATUS = 'DELETE')
        write (*,*) 'Deleting file LP_Z_SRC.DAT'
        OPEN  (1, FILE   = 'LP_z_src.dat')
        CLOSE (1, STATUS = 'DELETE')
        write (*,*) 'Deleting file LP_Z_ANT.DAT'
        OPEN  (1, FILE   = 'LP_z_ant.dat')
        CLOSE (1, STATUS = 'DELETE')
        write (*,*) 'Deleting file LP_VSWR.DAT'
        OPEN  (1, FILE   = 'LP_vswr.dat')
        CLOSE (1, STATUS = 'DELETE')
      endif
C
C
C  Set the accepted input power
      WATTS = 1.0
      

C     Do custom plane analysis
      if (OutFlag(4:4).eq.'Y') then
        OutFlag(8:8) = '4'
        FMHZ = AFSC
        write (*,*) 'Custom plane analysis'
        write (*,*) 'FMHZ : ',FMHZ
        call Analyze (TITLE, N, ZL, LLin, LLout, L, D, DB, SB,
     1           Zout, ZCin, Rs, FMHZ, OutFlag, WATTS, Fhigh, Flow,
     2           Tau, Sigma, Phi)
      endif

C     Do E- plane analysis
      if (OutFlag(2:2).eq.'Y') then
        OutFlag(8:8) = '2'
        FMHZ = AFSEH
        write (*,*) 'E- plane analysis'
        write (*,*) 'FMHZ : ',FMHZ
        call Analyze (TITLE, N, ZL, LLin, LLout, L, D, DB, SB,
     1           Zout, ZCin, Rs, FMHZ, OutFlag, WATTS, Fhigh, Flow,
     2           Tau, Sigma, Phi)
      endif

C     Do H- plane analysis
      if (OutFlag(3:3).eq.'Y') then
        OutFlag(8:8) = '3'
        FMHZ = AFSEH
        write (*,*) 'H- plane analysis'
        write (*,*) 'FMHZ : ',FMHZ
        call Analyze (TITLE, N, ZL, LLin, LLout, L, D, DB, SB,
     1           Zout, ZCin, Rs, FMHZ, OutFlag, WATTS, Fhigh, Flow,
     2           Tau, Sigma, Phi)
      endif

C     Do the swept frequency analysis
      if (OutFlag(5:5).eq.'Y') then
        OutFlag(8:8) = '5'
C     Define step size for analysis
        AFstep = 2.0**(1.0/AFpowr)
        FMHZ = AFlow
C
C     Initialize the output file and the screen
        write (*,*) 'writing file LP_SWEPT.DAT  -- Composite file'
        write (*,*) 'writing file LP_GAIN.DAT   -- Gain vs. Frequency'
        write (*,*) 'writing file LP_FTBR.DAT   -- Front-to-Back Ratio'
        write (*,*) 'writing file LP_Z_SRC.DAT  -- Impedance at Source'
        write (*,*) 'writing file LP_Z_ANT.DAT  -- Impedance at Antenna'
        write (*,*) 'writing file LP_VSWR.DAT   -- VSWR in Source Line'
        OPEN (2,FILE='LP_swept.dat', ACCESS='Append')
        write (2,298) 'Freq','Peak Dir.','Gain on Boresight','F/B',
     1                'F/SLL','Re(ZinA)','Im(ZinA)','Mag(ZinA)',
     2                'Re(ZinAS)','Im(ZinAS)','Mag(ZinAS)','VSWR '
        write (2,299) 'MHz' ,'dBi', 'dBi', 'dB', 'dB', 'Ohms', 'Ohms',
     1                ' Ohms', 'Ohms', 'Ohms', 'Ohms'
        CLOSE (2)

        write (*,*)
        write (*,*) 'Swept frequency analysis'

  298   format (1X, A9,   8x, A10, 5x,a17,  7X, A3,  11X, A5,   6X,
     1          A10, 6X,A10, 6X,A10, 6x,a10, 6X,A10, 6x, a10, 11X, A5)
  299   format (1X, A9,   8X, A10,   5X, A10,   7X, A10,   6X,
     1          A10,   6X, A10,   6x, a10,  6x, a10, 6x, a10,  6x, a10)

C     Do analysis
  40    continue
        write (*,*) 'Frequency (MHz) : ',FMHZ
        call Analyze (TITLE, N, ZL, LLin, LLout, L, D, DB, SB,
     1           Zout, ZCin, Rs, FMHZ, OutFlag, WATTS, Fhigh, Flow,
     2           Tau, Sigma, Phi)
C
        FMHZ = FMHZ * AFstep
        if (FMHZ.le.AFhigh) then
          goto 40
        endif
      endif

C
 5000 CONTINUE
      write (*,*) 'All done!'
      end
C **************************************************************
C
      SUBROUTINE Analyze (TITLE, N, ZL, LLin, LLout, L, D, DB, SB,
     1               Zout, ZCin, Rs, FMHZ, OutFlag, WATTS, Fhigh,
     2               Flow, Tau, Sigma, Phi)
C
C  Input Variables:       
C  TITLE      : Title of antenna design.
C  N          : Number of elements in the array (maximum = 30).
C  ZL(30)     : Array of distances from a reference (not necessarily
C               the apex) to each element (m).
C  LLin       : Length of source transmission line (m).
C  LLout      : Length of termination transmission line (m).
C  L(30)      : Array of dipole lengths (m).
C  D(30)      : Array of dipole diameters (cm).
C  DB         : Diameter of transmission line tube (cm).
C  SB         : Center to center spacing of the tube (cm).
C  Zout       : Complex termination impedance.  Usually the imaginary
C               part is set to zero.
C  ZCin       : Characteristic impedance of source transmission line
C  Rs         : Source Resistance
C  FMHZ       : Current analysis frequency.
C  OutFlag*8  : 1 : Summarize Design.
C             : 2 : E- plane Pattern.
C             : 3 : H- plane Pattern.
C             : 4 : Custom Plane Pattern.
C             : 5 : Data vs. Frequency.   
C             : 6 : Output to Screen.
C             : 7 : Output to Printer File.
C             : 8 : Number of current analysis
C                   eg. '3' means the H-plane pattern is being calculated.
C  WATTS      : Real power accepted by the antenna.  User specified.
C               This variable is used to scale the current distributions
C  Fhigh      : Upper design frequency.
C  Flow       : Lower design frequency.
C  Tau        : Geometric ratio.
C  Sigma      : Spacing factor.
C  Theta      : Spherical coordinate.  0 deg = z axis.
C  Phi        : Spherical coordinate.  0 deg = x axis.
C
C  Output Variables:
C     Iin        : Current at feed point
C     Iout       : Current in termination impedance
C     Iel(N)     : Current in each dipole
C     Vin        : Voltage across feed point
C     Vout       : Voltage across termination impedance
C     Vel(N)     : Voltage across each dipole
C     ZinA       : Actual Input impedance, ohms
C     VSWR       : Input VSWR relative to ZCin ohms
C
C  Routine specific variables:
C     j               : Sqrt(-1).
C     PI              : 3.1415 etc.       
C     c               : Speed of light = 299.7925.
C     BETA            : 2 * pi / lambda.
C     ZO              : Characteristic impedance of antenna transmission
C                       line.
C     K               : Counter.
C     ZA(ms,ms)       : Antenna element open-circuit impedance matrix
C                       (YA**(-1)).
C     DZL(0:ms)       : Distances between elements.
C                       DZL(0) = Distance from feed to element 1.
C                       DZL(1) = Distance from element 1 to element 2.
C                       DZL(N) = Distance from element N to termination.
C     col(ms)         : A column vector used for inverting matrices.
C     indx(ms)        : A vector which keeps track of data in
C                       matrix inversions.
C     scal(ms)        : Output from LUDEC which yields scaling information
C     I(ms)           : Array containing the external excitations of the
C                       dipole array.  I(2..N) = 0 and I(1) = 1 for a
C                       single feed point.
C     YT(ms,ms)       : Transmission line short-circuit admittance
C                       matrix.
C     YA(ms,ms)       : Antenna element short-circuit admittance matrix.
C     YAYT(ms,ms)     : YA + YT.
C     ZAZT(ms,ms)     : (YA + YT)**(-1).
C     G               : Counter.
C     ZR              : Impedance of termination seen from longest
C                       dipole.
C     ZinAS           : Input impedance seen by the source
C     POWERIN         : Power accepted by antenna array with 1 Amp
C                       feed excitation, Watts.
C     SCALE           : Factor which multiplies currents to yield
C                       the desired input power.
C     RADIUS          : Distance from array center to point of gain
C                       calculation (units equivalent to wavelength).
C     NUMPOINT        : Number of points at which the pattern is
C                       calculated.
C     Theta           : Spherical coordinate.  0 deg = z axis.
C     EPAT(NUMPOINT)  : Array of calculated E-plane gain as a function
C                       of angle,
C     ANGLE(NUMPOINT) : Array containing the angles at which
C                       GAIN(NUMPOINT) was calculated.
C     GAIN_ISO        : Gain of antenna relative to isotropic.
C     HPAT(NUMPOINT)  : Array of calculated H-plane gain as a function
C                       of angle.
C     CPAT(NUMPOINT)  : Array of calculated custom plane gain as a 
C                       function of angle.
C     PeakSLL         : Peak side-lobe level.
C     MainPeak        : Maximum absolute gain (relative to 1 Amp input).
C     BackLobe        : Absolute gain of point opposite main peak.
C     FSLL            : Front to peak side lobe level (dB).
C     FTBR            : Front to back ratio (dB).
C     PD0             : Peak directivity in any angle.
C     DOA             : Actual directivity on boresight
C     ABS_GAM         : Absolute value of the reflection coefficient
C                     : used to calculate VSWR.  
C
C **************************************************************
C
      IMPLICIT none
C
C  Input variables
C 
      integer      ms
      parameter   (ms = 30)

      character*60 TITLE
      integer      N
      real         ZL(ms+1), LLin, LLout, L(ms), D(ms), DB, SB
      complex      Zout, ZCin
      real         Rs, FMHZ
      character*8  OutFlag
      real         WATTS, Fhigh, Flow, Tau, Sigma, Phi
      
C
C  Output variables
C
      real         VSWR
      complex      Iel(ms), Vel(ms), ZinA, Vin, Iin, Vout, Iout
C
C  Routine specific variables:
      complex    j
      real       PI, c, BETA
      complex    ZO
      integer    k       
      complex    ZA(ms, ms)
      real       DZL(0:ms)
      complex    col(ms)
      integer    indx(ms)
      real       scal(ms)
      complex    I(ms), YT(ms, ms), YA(ms, ms), YAYT(ms, ms)
      complex    ZAZT(ms, ms)
      integer    g
      complex    ZR, ZinAS
      real       POWERIN, SCALE, RADIUS        
      integer    NUMPOINT
      parameter (NUMPOINT = 360)
      real       Theta, EPAT(NUMPOINT), Angle(NUMPOINT), Gain_Iso
      real       HPAT(NUMPOINT), CPAT(NUMPOINT) 
      real       PeakSLL, MainPeak, BackLobe, FSLL, FTBR, PD0, DOA
      real       ABS_GAM
      real       big
      parameter (big = 1.0e+12)
C

C
      j  = CMPLX(0.0, 1.0)
      PI = 2.0 * ACOS(0.0)
      c  = 299.7925
C
C Note: BETA has units 2 PI / meters
      BETA = 2 * PI * FMHZ / c
C
C  Find ZO = 120*arccosh(SB/DB) ~ 120 ln(SB/DB + sqrt((SB/DB)**2 - 1))
      ZO = 120 * LOG( SB/DB + SQRT((SB/DB) ** 2 - 1) )
C
C  Convert the diameters to meters
      do 10, k = 1, N, 1
        D(k) = D(k) * 0.01
   10 continue
C
C  Defines the antenna element self- and mutual-impedance for an
C  N element dipole array.
      call FIND_ZA(N, BETA, D, ZL, L, ZA)
C
C  Restore diameters of dipoles
      do 20, k = 1, N, 1
        D(k) = D(k) / 0.01
   20 continue
C
C  Find the lengths of the transmission line segments between elements
C
      DZL(N) = LLout
      DO 1050 k = 1, N-1, 1
        DZL(k) = ZL(k+1) - ZL(k)
 1050 CONTINUE
      DZL(0) = LLin
C
C  I(k) 1 <= k <= N Defines the locations (and strengths) of the
C  source currents.  The subscript corresponds to the element number
C  across which the source is placed.  Note: The following sets all
C  feed currents to zero except at element 1.  This is applicable to
C  a Log Periodic Dipole Array (LPDA).
C
      DO 1040 k = 1, N, 1
        I(k) = CMPLX(0.0, 0.0)
 1040 CONTINUE
        I(1) = CMPLX(1.0, 0.0)
C
C
C  Calculate antenna voltages (Vel, Vout, Vin) and currents (Iel, Iout, Iin)
C
C  Find YT
C
      call FIND_YT(N, YT, DZL, BETA, ZO, Zout)
C
C  Find YA = (ZA) inverted
C  These inversion routines (LUDCMP and LUBKSB) are due to Numerical
C  Recipes in Fortran.
C  
      
      call LUDEC (ZA, N, ms, indx, scal) 
      DO 50 k = 1, N
        DO 60 g = 1, N
          col(g) = CMPLX(0.0, 0.0)
   60   CONTINUE
        col(k) = CMPLX(1.0, 0.0)
        call LUSOLV (ZA, N, ms, indx, col)
        DO 70 g = 1, N
          YA(g,k) = col(g)
   70   CONTINUE
   50 CONTINUE
      
C
C  Add admittances YA and YT
C
      DO 1205 g = 1, N, 1
        DO 1200 k = 1, N, 1
          YAYT(k,g) = YA(k,g) + YT(k,g)
 1200   CONTINUE
 1205 CONTINUE
C
C  Invert (YA + YT) = (YA + YT)^-1
C
      call LUDEC (YAYT, N, ms, indx, scal) 
      DO 150 k = 1, N
        DO 160 g = 1, N
          col(g) = CMPLX(0.0, 0.0)
  160   CONTINUE
        col(k) = CMPLX(1.0, 0.0)
        call LUSOLV (YAYT, N, ms, indx, col)
        DO 170 g = 1, N
          ZAZT(g,k) = col(g)
  170   CONTINUE
  150 CONTINUE
C
C  Find Vel = (YA + YT)^-1 * I
C
C  I = the excitation matrix, for a LPDA, I = [1 0 ... 0 0]
C
        DO 1310, g = 1, N, 1
          Vel(g) = CMPLX (0.0, 0.0)
          DO 1320, k = 1, N, 1
            Vel(g) = ZAZT(g, k) * I(k) + Vel(g)
1320      CONTINUE
1310    CONTINUE
C
C  Solve for the element currents by Iel = YA * Vel
C
        DO 1330, g = 1, N, 1
          Iel(g) = CMPLX (0.0, 0.0)
          DO 1340, k = 1, N, 1
            Iel(g) = YA(g, k) * Vel(k) + Iel(g)
1340      CONTINUE
1330    CONTINUE

C
C  Find termination current
C
      ZR = ZO * (Zout + j * ZO   * tan(BETA * LLout))
     1        / (ZO   + j * Zout * tan(BETA * LLout))
C
      if (abs(ZR).ne.0.0) then
        Iout = Vel(N) / ZR
      else
        Iout = CMPLX(big,0.0)
      endif
C
C  Find termination voltage and current
C  These equations can be found, for example, in Cheng.
C
      Vout  = Vel(N)* cos(BETA * LLout) - j*Iout  * ZO*sin(BETA * LLout)
      Iout  = Iout * cos(BETA * LLout) - j*Vel(N) / ZO*sin(BETA * LLout)
C
C  Calculate Input impedance at terminals of shortest element
C
      ZinA = Vel(1) / 1.0
C
C  Calculate VSWR referenced to ZCin Ohms
      ABS_GAM = CABS ((ZinA - ZCin) / (ZinA + ZCin))
      if (ABS_GAM.NE.1) then
        VSWR = ((1 + ABS_GAM) / (1 - ABS_GAM))
      else
        VSWR = 100000.0
      endif
C
C  Calculate input impedance measured at the source
C  Note Vin here is Vin' in the Log-Perd.doc file.
C
      Iin   = cos(BETA*LLin) + j*Vel(1) * sin(BETA*LLin) / ZCin
      Vin   = Vel(1) * cos(BETA*LLin) + j*ZCin * sin(BETA*LLin)
      ZinAS = Vin / Iin
      
C  Here we calculate Vin as defined by the Log-Perd.doc file
      Vin = (Rs + ZinAS)/ZinAS * Vin
C
C  Calculate the power accepted by the antenna array using an
C  excitation of 1 amp at the shortest element's terminals and
C  compare to desired input power.  Include source resistance.
C
      POWERIN = Iin * conjg(Iin) * (real(ZinAS) + Rs) / 2.0
C
C  Scale the currents so that the desired power (1 Watt) is attained
      SCALE = SQRT(WATTS / POWERIN)   
      
      DO 30 K = 1, N, 1
        Iel(K) = Iel(K) * SCALE
        Vel(K) = Vel(K) * SCALE
   30 CONTINUE
      Iin  = Iin  * SCALE
      Vin  = Vin  * SCALE
      Iout = Iout * SCALE
      Vout = Vout * SCALE
C
C  Prepare to find array pattern
C
      RADIUS = 100.0 / BETA * 2.0 * PI
C
C  Find gain of an isotropic source
      Gain_Iso = 10.0 * log10(WATTS / (4 * PI * RADIUS**2))
C
C  Make custom plane cut
C
C  Returns gain in dB relative to path of RADIUS meters
C  Must do this one before E- and H- because the others destroy the
C  value of Phi.
C
      IF (OutFlag(8:8).eq.'4') THEN
        DO 2010 K = 1, NUMPOINT, 1
          Theta = real(K) * 360.0 / NUMPOINT
          call PATTERN (BETA,Phi, Theta, L, ZL, N, Iel, RADIUS, CPAT(K))
          ANGLE(K) = Theta
 2010   CONTINUE
C
C  Find FTBR and FSLL
        call Find_SLL (NUMPOINT, CPAT, PeakSLL, MainPeak, BackLobe)
        FSLL     = CPAT(NUMPOINT/2) - PeakSLL
        FTBR     = CPAT(NUMPOINT/2) - BackLobe
        PD0      = MainPeak         - Gain_Iso
        DOA      = CPAT(NUMPOINT/2) - Gain_Iso
C
      ENDIF
C
C  Make E-plane cut
C  Returns gain in dB relative to path of RADIUS meters
C
      IF ((OutFlag(8:8).eq.'2').or.(OutFlag(8:8).eq.'5')) THEN
        Phi = 90.0
        DO 2011 K = 1, NUMPOINT, 1
          Theta = real(K) * 360.0 / NUMPOINT
          call PATTERN (BETA,Phi, Theta, L, ZL, N, Iel, RADIUS, EPAT(K))
          ANGLE(K) = Theta
 2011   CONTINUE
C
C  Find FTBR and FSLL
        call Find_SLL (NUMPOINT, EPAT, PeakSLL, MainPeak, BackLobe)
        FSLL     = EPAT(NUMPOINT/2) - PeakSLL
        FTBR     = EPAT(NUMPOINT/2) - BackLobe
        PD0      = MainPeak         - Gain_Iso
        DOA      = EPAT(NUMPOINT/2) - Gain_Iso
C
      ENDIF
C
C  Make H-plane cut
C  Returns gain in dB relative to path of RADIUS meters
C     
      IF (OutFlag(8:8).eq.'3') THEN
        Phi = 0.0
        DO 2012 K = 1, NUMPOINT, 1
          Theta = real(K) * 360.0 / NUMPOINT
          call PATTERN (BETA,Phi, Theta, L, ZL, N, Iel, RADIUS, HPAT(K))
          ANGLE(K) = Theta
 2012   CONTINUE
C
C  Find FTBR and FSLL
        call Find_SLL (NUMPOINT, HPAT, PeakSLL, MainPeak, BackLobe)
        FSLL     = HPAT(NUMPOINT/2) - PeakSLL
        FTBR     = HPAT(NUMPOINT/2) - BackLobe
        PD0      = MainPeak         - Gain_Iso
        DOA      = HPAT(NUMPOINT/2) - Gain_Iso
C
      ENDIF
C
C
C  Convert Vin, Iin, Vout, Iout to (magnitude (dB), phase (deg)) format
      call R2POL (Vin)
      call R2POL (Iin)
      call R2POL (Vout)
      call R2POL (Iout)
        if (real(Vin).NE.0.0) then
          Vin = CMPLX( 20 * log10( real(Vin) ), imag(Vin) )
        else
          Vin = CMPLX(-1000.0, imag(Vin) )
        endif
        if (real(Iin).NE.0.0) then
          Iin = CMPLX( 20 * log10( real(Iin) ), imag(Iin) )
        else
          Iin = CMPLX(-1000.0, imag(Iin) )
        endif
        if (real(Vout).NE.0.0) then
          Vout = CMPLX( 20 * log10( real(Vout) ), imag(Vout) )
        else
          Vout = CMPLX(-1000.0, imag(Vout) )
        endif
        if (real(Iout).NE.0.0) then
          Iout = CMPLX( 20 * log10( real(Iout) ), imag(Iout) )
        else
          Iout = CMPLX(-1000.0, imag(Iout) )
        endif   
        
        DO k = 1, N
          call R2POL (Vel(k))
          call R2POL (Iel(k))
          if (real(Vel(k)).NE.0.0) then
            Vel(k) = CMPLX( 20 * log10( real(Vel(k)) ), imag(Vel(k)) )
          else
            Vel(k) = CMPLX(-1000.0, imag(Vel(k)) )
          endif
          if (real(Iel(k)).NE.0.0) then
            Iel(k) = CMPLX( 20 * log10( real(Iel(k)) ), imag(Iel(k)) )
          else
            Iel(k) = CMPLX(-1000.0, imag(Iel(k)) )
          endif
        ENDDO
        
C
C     Write swept frequency data
      if (OutFlag(8:8).eq.'5') then
        OPEN (2,FILE='LP_swept.dat', ACCESS='Append')
        write (2,300) FMHZ, PD0, DOA, FTBR, FSLL, real(ZinA),
     1                 imag(ZinA), cabs(ZinA), real(ZinAS),
     2                 imag(ZinAS), cabs(ZinAS), VSWR
        CLOSE (2)
        OPEN (2,FILE='LP_gain.dat', ACCESS='Append')
        write (2,301) FMHZ, DOA
        CLOSE (2)
        OPEN (2,FILE='LP_ftbr.dat', ACCESS='Append')
        write (2,301) FMHZ, FTBR
        CLOSE (2)
        OPEN (2,FILE='LP_z_ant.dat', ACCESS='Append')
        write (2,301) FMHZ, cabs(ZinA)
        CLOSE (2)
        OPEN (2,FILE='LP_z_src.dat', ACCESS='Append')
        write (2,301) FMHZ, cabs(ZinAS)
        CLOSE (2)
        OPEN (2,FILE='LP_vswr.dat', ACCESS='Append')
        write (2,301) FMHZ, VSWR
        CLOSE (2)
      endif
C
  300 format (1X, F12.5, 4X, F12.5, 4X, F12.5, 4X, F12.5, 4X, F12.5, 4X,
     1        F12.5, 4X, F12.5, 4X, F12.5, 4X, F12.5, 4X, F12.5, 4X,
     2        F12.5, 4X, F12.5)
  301 format (1X, F12.5, 4X, F12.5)
C     
C     Write Single frequency data
      if (OutFlag(8:8).ne.'5') then
        if (OutFlag(8:8).eq.'2') write (*,*) 
     1     'writing file LP_EPL.OUT    -- Analysis in E-plane'
        if (OutFlag(8:8).eq.'3') write (*,*) 
     1     'writing file LP_HPL.OUT    -- Analysis in H-plane'
        if (OutFlag(8:8).eq.'4') write (*,*)
     1     'writing file LP_CPL.OUT    -- Analysis in custom plane'
        call OUTPUT1 (N, Rs, ZL, LLin, LLout, L, D, DB, SB, Zout, FMHZ,
     1            Vin, Iin, Vout, Iout, WATTS, ZCin, Iel, Vel, ZinA,
     2            ZinAS, ZO, VSWR, PD0, DOA, FTBR, FSLL, TITLE, OutFlag,
     3            Fhigh, Flow, Tau, Sigma, Phi)
        if (OutFlag(8:8).eq.'2') then
          write (*,*) 'Deleting file LP_EPAT.DAT'
          OPEN  (1, FILE   = 'LP_epat.dat')
          CLOSE (1, STATUS = 'DELETE')
          write (*,*) 'writing file LP_EPAT.DAT   -- E-plane gain'
          OPEN  (1, FILE   = 'LP_epat.dat')
          do K = 1, NUMPOINT
            Theta = real(K) * 360.0 / NUMPOINT
            write (1,400) Theta, EPAT(K) - Gain_Iso
          enddo
          CLOSE (1)
        endif
        if (OutFlag(8:8).eq.'3') then
          write (*,*) 'Deleting file LP_HPAT.DAT'
          OPEN  (1, FILE   = 'LP_hpat.dat')
          CLOSE (1, STATUS = 'DELETE')
          write (*,*) 'writing file LP_HPAT.DAT   -- H-plane gain'
          OPEN  (1, FILE   = 'LP_hpat.dat')
          do K = 1, NUMPOINT
            Theta = real(K) * 360.0 / NUMPOINT
            write (1,400) Theta, HPAT(K) - Gain_Iso
          enddo
          CLOSE (1)
        endif
        if (OutFlag(8:8).eq.'4') then
          write (*,*) 'Deleting file LP_CPAT.DAT'
          OPEN  (1, FILE   = 'LP_cpat.dat')
          CLOSE (1, STATUS = 'DELETE')
          write (*,*) 'writing file LP_CPAT.DAT   -- Custom plane gain'
          OPEN  (1, FILE   = 'LP_cpat.dat')
          do K = 1, NUMPOINT
            Theta = real(K) * 360.0 / NUMPOINT
            write (1,400) Theta, CPAT(K) - Gain_Iso
          enddo
          CLOSE (1)
        endif
      endif
  400 format (1x, f12.5, 8x, f12.5)
C
      return
      end
C **************************************************************
C
      SUBROUTINE Find_SLL(N, Gain, relmax, absmax, backlobe)
C
C  Input Variables:
C    N       : Number of elements in GAIN.
C    Gain(N) : Contains the pattern of the array.
C
C  Output Variables:
C    relmax   : Local maximum of input array (first side lobe level).
C    absmax   : Absolute maximum of input array.
C    backlobe : Value N/2 from the absolute maximum.
C
C  Routine Specific Variables:
C    peak     : Logical variable which flags the peak value.
C    k        : Counter
C
C **************************************************************
C
      IMPLICIT NONE
      integer N
      real    Gain(N), relmax, absmax, backlobe
      logical peak 
      integer k, Nover2
C
      Nover2 = N / 2
      backlobe = -10000.0
      absmax   = -10000.0
      relmax   = -10000.0
      do 10 k = 1, N
C  Notes: 1.  mod(k-1,N)+1 replaces k to account for a peak that
C             straddles indices 1 and N
C         2.  This code allows for two adjacent equal values
C         3.  N is added inside each mod() function because the
C             ForTran mod() formulation does not work for
C             negative numbers
C
C  Flow of following code:
C
C  IF peak THEN
C    IF absolute peak THEN
C    ELSE IF relative peak THEN
C
        peak = .false.
        if ( (Gain(mod(k-1+N,N)+1).GT.Gain(mod(k-2+N,N)+1))  .AND.
     1       (Gain(mod(k-1+N,N)+1).GE.Gain(mod(k    ,N)+1))  .AND.
     2       (Gain(mod(k-1+N,N)+1).GT.Gain(mod(k+1  ,N)+1)) ) then
           peak = .true.
        endif
        if ( (Gain(mod(k-1+N,N)+1).GT.Gain(mod(k-2+N,N)+1))  .AND.
     1       (Gain(mod(k-1+N,N)+1).GT.Gain(mod(k    ,N)+1)) ) then
           peak = .true.
        endif

        if (peak) then
          if (Gain(mod(k-1,N)+1).GE.absmax) then
            relmax   = absmax
            absmax   = Gain(k)
            backlobe = Gain(mod(k+Nover2-1, N) + 1)
          else if (Gain(mod(k-1,N)+1).GT.relmax) then
            relmax = Gain(mod(k-1,N)+1)
          endif
        endif
   10 continue
C
      return
      end

C **************************************************************
C
      SUBROUTINE FindST (D0, Sigma, Tau)

C     This routine finds Sigma and Tau given the desired directivity.
C
C     Input Variables:
C       D0    : Desired Directivity.
C
C     Output Variables:
C       Sigma : Spacing factor.
C       Tau   : Geometric ratio.
      
      real D0, Sigma, Tau
      
C     This table represents a linear interpolation of the points on the
C     optimum directivity line of Figure 11.13.
      if ((D0.ge. 7.0).and.(D0.lt. 7.5)) Tau=0.78 +(D0- 7.0)/0.5*(0.044)
      if ((D0.ge. 7.5).and.(D0.lt. 8.0)) Tau=0.824+(D0- 7.5)/0.5*(0.041)
      if ((D0.ge. 8.0).and.(D0.lt. 8.5)) Tau=0.865+(D0- 8.0)/0.5*(0.030)
      if ((D0.ge. 8.5).and.(D0.lt. 9.0)) Tau=0.895+(D0- 8.5)/0.5*(0.022)
      if ((D0.ge. 9.0).and.(D0.lt. 9.5)) Tau=0.917+(D0- 9.0)/0.5*(0.008)
      if ((D0.ge. 9.5).and.(D0.lt.10.0)) Tau=0.925+(D0- 9.5)/0.5*(0.017)
      if ((D0.ge.10.0).and.(D0.lt.10.5)) Tau=0.942+(D0-10.0)/0.5*(0.013)
      if ((D0.ge.10.5).and.(D0.le.11.0)) Tau=0.955+(D0-10.5)/0.5*(0.012)
      
      Sigma = 0.237838*Tau - 0.047484
      
      RETURN
      END

C **************************************************************
C
      SUBROUTINE FIND_YT (N, YT, DZL, BETA, ZO, Zout)
C
C  This routine calculates the transmission line short circuit
C  admittance matrix, YT.  An approximation is made in certain
C  cases when a division by zero error would normally result.
C
C  Input variables:
C    N    : Number of elements.
C    ms   : Maximum array size.
C    DZL  : Distances between elements = lengths of transmission lines.
C    BETA : 2*pi / wavelength.
C    ZO   : Characteristic impedance of transmission line.
C
C  Output variables:
C    YT   : Short circuit admittance matrix.
C
C  Routine specific variables:
C    I    : Counter.
C    K    : Counter.
C    SGN  : Sign of a cosine term.
C    T1   : Intermediate variable.
C    T2   : Intermediate variable.
C    YO   : 1 / ZO
C    j    : Square root of (-1)
C
C **************************************************************
C

C  Input variables
      IMPLICIT none
      integer    N, ms
      parameter (ms = 30)
      real       DZL(0:ms), BETA
      complex    ZO, Zout
      
C  Routine specific variables
      complex    YO, j     
      integer    I, K
      real       SGN
      complex    T1, T2
      real       tiny, big
      parameter (tiny = 1.0e-12)
      parameter (big  = 1.0e+12)
      
C  Output variables
      complex    YT(ms,ms)

C
      YO = 1 / ZO
      j  = CMPLX(0.0, 1.0)
C
C  initializes YT to 0,0
C
      DO 2020 I = 1, N, 1
        DO 2010 K = 1, N, 1
          YT(I,K) = CMPLX(0.0,0.0)
 2010   CONTINUE
 2020 CONTINUE
C
C

C  Calculates all elements except 1st and last columns
C
      IF (N.GT.2) THEN
        DO 2030 I = 2, N-1, 1
C  Error-trap those values which will yield a divide-by-zero
C  error.
C
C  Calculate matrix element (I,I)
C                = -jYO(COT (Beta*DZL(I-1)) + COT(Beta*DZL(I)) )
C
          IF (ABS( sin(BETA*DZL(I-1)) ).le.tiny) THEN
            T1 = CMPLX (big, 0.0)
          ELSE
            T1 = CMPLX (1.0/tan( BETA*DZL(I-1) ), 0.0)
          ENDIF
          IF (ABS(sin(BETA*DZL(I))).le.tiny) THEN
            T2 = CMPLX (big, 0.0)
          ELSE
            T2 = CMPLX (1.0/tan( BETA*DZL(I) ), 0.0)
          ENDIF
C
          YT(I,  I) = -j * (T1 + T2) * YO
C
C  Calculate matrix element (I-1,I) = -j * YO * CSC(Beta * DZL(I-1))
C
          IF (ABS( sin(BETA*DZL(I-1)) ).le.tiny) THEN
            SGN = SIGN( 1.0, cos(BETA*DZL(I-1)) )
            T1 = CMPLX (big * SGN, 0.0)
          ELSE
            T1 = CMPLX (1.0/sin( BETA*DZL(I-1) ), 0.0)
          ENDIF
          YT(I-1,I) = -j * YO * T1
C
C  Calculate matrix element (I+1,I) = -j * YO * CSC(Beta * DZL(I))
C
          IF (ABS(sin(BETA*DZL(I))).le.tiny) THEN
            SGN = SIGN( 1.0, cos(BETA*DZL(I)) )
            T2 = CMPLX (big * SGN, 0.0)
          ELSE
            T2 = CMPLX (1.0/sin( BETA*DZL(I) ), 0.0)
          ENDIF
          YT(I+1,I) = -j * YO * T2
C
C
 2030   CONTINUE
      ENDIF
C
C
C  Calculates remaining elements
C
C  Do element N
C  YT(N,N) = -j * YO * tan(Beta*DZL(N-1)) + ...equivalent termination
C
      if (abs( sin(BETA*DZL(N-1)) ).le.tiny) THEN
        T1 = CMPLX (big, 0.0)
      else
        T1 = CMPLX (1.0 / tan(BETA*DZL(N-1)), 0.0)
      endif
      if (cabs(YO*Zout*cos(BETA*DZL(N)) +j*sin(BETA*DZL(N))).eq.0.0)THEN
        T2 = CMPLX(big, 0.0)
      else
        T2 = YO * (      cos(BETA*DZL(N)) + j*YO*Zout*sin(BETA*DZL(N)))
     1          / (YO*Zout*cos(BETA*DZL(N)) + j*      sin(BETA*DZL(N)))
      endif
      if (cabs(T2).eq.0.0) T2 = CMPLX(tiny,0.0)
      YT(N,N) = -j * YO * T1 + T2
C
C  YT(N-1,N) = -j * YO * csc(Beta*DZL(N-1))
C
      IF (ABS( sin(BETA*DZL(N-1)) ).le.tiny) THEN
        SGN = SIGN( 1.0, cos(BETA*DZL(N-1)) )
        T1 = CMPLX (big * SGN, 0.0)
      ELSE
        T1 = CMPLX (1.0/sin( BETA*DZL(N-1) ), 0.0)
      ENDIF
      YT(N-1,N) = -j * YO * T1

C
C  Now do element 1
C
C  YT(1,1) = -j * YO * COT(Beta*DZL(1))
C
      IF (ABS( sin(BETA*DZL(1)) ).le.tiny) THEN
        T1 = CMPLX (big, 0.0)
      ELSE
        T1 = CMPLX (1.0 / tan(BETA*DZL(1)), 0.0)
      ENDIF
      YT(1,1) = -j * YO * T1
C
C  YT(2,1) = -j * YO * CSC(Beta*DZL(1))
C
      IF (ABS( sin(BETA*DZL(1)) ).le.tiny) THEN
        SGN = SIGN( 1.0, cos(BETA*DZL(1)) )
        T1 = CMPLX (big * SGN, 0.0)
      ELSE
        T1 = CMPLX (1.0/sin( BETA*DZL(1) ), 0.0)
      ENDIF
      YT(2,1) = -j * YO * T1
C
      return
      end

C **************************************************************
C
      SUBROUTINE FIND_ZA (N, BETA, D, ZL, L, ZA)
C
C   This subroutine finds the antenna self and mutual impedances 
C   between elements assuming the current distribution is sinusoidal.
C   Note that 'element' 1 contains the driving current source.
C
C  Input Variables:
C     N        : Number of elements. 
C     ms       : Maximum array size.
C     BETA     : 2 * pi / lambda.
C     D        : Diameter of wires (m) [sic].
C     ZL       : Distance from apex to element (m).
C     L        : Lengths of dipoles (m).
C
C  Output Variables:
C     ZA       : Matrix containing mutual impedances.
C
C  Routine specific variables:
C     PI       : 3.1415 etc.
C     ETA      : 120*PI
C     gamma    : Euler's number (0.5772156649).
C     LAMBDA   : Wavelength.
C     I        : Counter.
C     J        : Counter.
C     LENGTH   : An array which contains the lengths of the two
C                dipoles whose mutual impedance is being determined.
C     AELE     : Radius of current elements being considered.
C     ZL1      : Location of first element.
C     ZL2      : Location of second element.
C     BL       : BETA*LENGTH
C     BA2L     : BETA*(AELE**2) / LENGTH
C     Rr       : Real part of radiation impedance referred to
C                current maximum.
C     Ci       : Cosine integral function.
C     Si       : Sine integral function.
C     Xm       : Imaginary part of radiation impedance referred to 
C                current maximum.
C     DZL      : Distance between elements.
C     L1L2p    : Intermediate variables used by Harold King.
C     L1L2m    : Intermediate variables used by Harold King.
C     u0, u0p  : Intermediate variables used by Harold King.
C     v0, v0p  : Intermediate variables used by Harold King.
C     u1, v1   : Intermediate variables used by Harold King.
C     w1, y1   : Intermediate variables used by Harold King.
C     BLp      : Intermediate variables used by Harold King.
C     BLm      : Intermediate variables used by Harold King.
C     BD       : Intermediate variables used by Harold King.
C
C **************************************************************
C
C

C  Input variables
      IMPLICIT none
      integer    N, ms
      parameter (ms = 30)
      real       BETA, D(ms)
      real       ZL(ms)
      real       L(ms)
      
C  Routine specific variables      
      real    PI, ETA, gamma, LAMBDA
      integer I, J
      real    LENGTH(2), AELE, ZL1, ZL2, BL, BA2L, Rr, Ci, Si, Xm, DZL
      real    L1L2p, L1L2m, u0, u0p, v0, v0p, u1, v1, w1, y1
      real    BLp, BLm, BD

C  Output variables
      complex   ZA(ms, ms)

C
      PI     = 2.0 * ACOS(0.0)
      ETA    = 120*PI
      gamma  = 0.5772156649
      LAMBDA = 2 * PI / BETA
C
C  Initialize ZA matrix
      DO 6 I = 1, N, 1
        DO 5 J = 1, N, 1
          ZA(I,J) = CMPLX(0.0, 0.0)
    5   CONTINUE
    6 CONTINUE
C
      DO 1020 I = 1, N, 1
        DO 1010 J = I, N, 1
C  Define lengths of two wires under comparison
C  Note:  King's definition of L is half of Balanis'
          LENGTH(1) = L(I) / 2.0
          LENGTH(2) = L(J) / 2.0
          AELE      = D(I) / 2.0
          ZL1       = ZL(I)
          ZL2       = ZL(J)
C
C  This is Balanis' equation, so multiply length by 2
C  If this is a self-impedance, then
          IF (I.EQ.J) THEN
            BL = BETA*LENGTH(1)*2.0
            BA2L = BETA*AELE**2/(LENGTH(1)*2.0)
C  Note: missing scale factors which are added below
            Rr = gamma + log(BL) - Ci(BL)
     1        + 0.5*sin(BL)*(Si(2*BL) - 2*Si(BL))
     2        + 0.5*cos(BL)*(Ci(2*BL) - 2*Ci(BL) + gamma + log(BL/2.0))
            Xm =   -cos(BL)*(Si(2*BL) - 2*Si(BL))
     1             +sin(BL)*(Ci(2*BL) - 2*Ci(BL) + Ci(2*BA2L))+ 2*Si(BL)
            ZA(I,J) = ETA/2.0/PI*CMPLX(Rr,Xm/2.0)

C  Refer impedance to input terminals
            ZA(I,J) = ZA(I,J)/(sin(BL/2))**2
C
C  Note: The variables X1 and X2 have no meaning in the calculation of
C        a self impedance because the distance between dipoles is
C        moot with only dipole.
C
          ELSE
C  This is King's equation, so leave the lengths alone.
C  If this is a mutual impedance, then
            DZL = ZL2 - ZL1
            L1L2p = LENGTH(1)+LENGTH(2)
            L1L2m = LENGTH(1)-LENGTH(2)
            u0  = BETA*(sqrt(DZL**2 + L1L2p**2) - L1L2p)
            v0  = BETA*(sqrt(DZL**2 + L1L2p**2) + L1L2p)
            u0p = BETA*(sqrt(DZL**2 + L1L2m**2) - L1L2m)
            v0p = BETA*(sqrt(DZL**2 + L1L2m**2) + L1L2m)
            u1  = BETA*(sqrt(DZL**2 + LENGTH(1)**2) - LENGTH(1))
            v1  = BETA*(sqrt(DZL**2 + LENGTH(1)**2) + LENGTH(1))
            w1  = BETA*(sqrt(DZL**2 + LENGTH(2)**2) + LENGTH(2))
            y1  = BETA*(sqrt(DZL**2 + LENGTH(2)**2) - LENGTH(2))    
            BLp = BETA*L1L2p 
            BLm = BETA*L1L2m
            BD  = BETA*DZL
            Rr  = cos(BLp) *  (Ci(u0)  + Ci(v0)  - Ci(u1) - Ci(v1)
     1                       - Ci(w1)  - Ci(y1)  + 2*Ci(BD))
     2           +cos(BLm) *  (Ci(u0p) + Ci(v0p) - Ci(u1) - Ci(v1)
     3                       - Ci(w1)  - Ci(y1)  + 2*Ci(BD))
     4           +sin(BLp) * (-Si(u0)  + Si(v0)  + Si(u1) - Si(v1)
     5                       - Si(w1)  + Si(y1))
     6           +sin(BLm) * (-Si(u0p) + Si(v0p) + Si(u1) - Si(v1)
     7                       + Si(w1)  - Si(y1))

            Xm  = cos(BLp) * (-Si(u0)  - Si(v0)  + Si(u1) + Si(v1)
     1                       + Si(w1)  + Si(y1)  - 2*Si(BD))
     2           +cos(BLm) * (-Si(u0p) - Si(v0p) + Si(u1) + Si(v1)
     3                       + Si(w1)  + Si(y1)  - 2*Si(BD))
     4           +sin(BLp) * (-Ci(u0)  + Ci(v0)  + Ci(u1) - Ci(v1)
     5                       - Ci(w1)  + Ci(y1))
     6           +sin(BLm) * (-Ci(u0p) + Ci(v0p) + Ci(u1) - Ci(v1)
     7                       + Ci(w1)  - Ci(y1))
            
            ZA(I,J) = ETA/4/PI*CMPLX(Rr,Xm)
C
C  Refer impedance to input terminals
            ZA(I,J) = ZA(I,J) / sin(BETA*LENGTH(1))/sin(BETA*LENGTH(2))

C
C  Note:  The variable AELE has no meaning in the calculation of
C         a mutual impedance.  The size of the dipoles is not
C         considered in this model.
C
C  Take advantage of symmetry to halve computations
            ZA(J,I) = ZA(I,J)
          ENDIF
 1010   CONTINUE
 1020 CONTINUE
C
      return
      end
C
C **************************************************************
C
      SUBROUTINE Input(TITLE, Fhigh, Flow, Tau, Sigma, D0, LD, Rs, LLin,
     1                 ZCin, Rin, LLout, Zout, AFhigh, AFlow, AFpowr,
     2                 AFSEH, AFSC, Phi, Navail, Davail, 
     3                 OutFlag, Quant, SB, DB, StopFlag)


C  Output Variables 
C  TITLE      : Title of antenna design.
C  Fhigh      : Upper design frequency.
C  Flow       : Lower design frequency.
C  Tau        : Geometric ratio.
C  Sigma      : Spacing factor.
C  D0         : Desired Directivity.                      
C  LD         : Designed Length to Diameter ratio.
C  Rs         : Source Resistance
C  LLin       : Length of source transmission line.                                 
C  ZCin       : Characteristic impedance of source transmission line.
C  Rin        : Desired input impedance of array.
C  LLout      : Length of termination transmission line.
C  Zout       : Complex termination impedance.  Usually the imaginary
C               part is set to zero.
C  AFhigh     : Upper analysis frequency.
C  AFlow      : Lower analysis frequency.
C  AFpowr     : Number of frequency steps per octave
C  AFSEH      : Frequency at which to make E- and H- plane pattern plots.
C  AFSC       : Frequency at which to make custom pattern plot.
C  Phi        : Spherical coordinate.  0 deg = x axis.
C  Navail     : Number of available tape widths.
C  Davail(15) : Array of tape width sizes.           
C  OutFlag*8  : 1 : Summarize Design.
C             : 2 : E- plane Pattern.
C             : 3 : H- plane Pattern.
C             : 4 : Custom Plane Pattern.
C             : 5 : Data vs. Frequency.
C             : 6 : Unused, formerly:  Output to Screen.
C             : 7 : Unused, formerly:  Output to Printer File.
C             : 8 : Number of current analysis
C                   eg. '3' means the H-plane pattern is being calculated
C  Quant*2    : 1 : Unused, formerly:  Quantize Boom Diameter.
C             : 2 : Quantize Element Diameter.
C  SB         : Center to center spacing of the tube.
C  DB         : Diameter of transmission line tube.  
C  StopFlag   : TRUE = terminate execution of program.
C
C **************************************************************

      IMPLICIT NONE      
      
      character*60 TITLE
      real         Fhigh, Flow, Tau, Sigma, D0, Rs, LD, LLin
      complex      ZCin
      real         Rin, LLout
      complex      Zout
      real         AFhigh, AFlow, AFpowr, AFSEH, AFSC, Phi 
      integer      Navail
      real         Davail(15)
      character*8  OutFlag
      character*2  Quant
      real         SB, DB
      character*1  StopFlag

C  LOCAL VARIABLES
C    INPUT ROUTINE VARIABLES:
      integer      mt, st(20), sl(20)
      character*40 maintitles(20)
      character*48 subtitles(20, 15)
      character*48 mainlines(10),  sublines(20, 10)
      
      integer      i, inkey1, inkey2, dummyI
      logical      exist, FLAG
      real         Cr, Ci, dummyR
                         
C    Set the stop flag to default (don't stop)
      StopFlag = 'N'             
      
C    Define all screen titles

C     mt = number of lines in main screen
      mt = 14

C     Define main screen titles
      maintitles( 1)='Design Title :                          '
      maintitles( 2)='Upper Design Frequency                  '
      maintitles( 3)='Lower Design Frequency                  '
      maintitles( 4)='Tau, Sigma, and Directivity Choices...  '
      maintitles( 5)='Length to Diameter Ratio                '
      maintitles( 6)='Source Resistance                       '
      maintitles( 7)='Length of Source Transmission Line      '
      maintitles( 8)='Impedance of Source Transmission Line   '
      maintitles( 9)='Boom Spacing Choices...                 '
      maintitles(10)='Length of Termination Transmission Line '
      maintitles(11)='Termination Impedance                   '
      maintitles(12)='Tube Diameter Choices...                '
      maintitles(13)='Design Summary and Analysis Choices...  '
      maintitles(14)='Begin Design and Analysis               '
             
      mainlines  ( 1) ='Please enter a line number or enter 15 to save'
      mainlines  ( 2) =' and exit.'
      
C     Set number of subscreen titles (st) to 0 for all screens
C     Set number of subscreen lines (sl) to 1 for all screens      
      do i = 1,20
        st(i) = 0  
        sl(i) = 1
      enddo
C     Correct st where necessary
      st( 4) =  3
      st( 9) =  4
      st(12) = 18
      st(13) = 11
C     Correct sl where necessary
      sl( 4) =  6
      sl( 9) =  7
      sl(12) =  4
      sl(13) =  5
      
C     Define sublines
      sublines ( 1, 1)='Design Title :                                 '
      sublines ( 2, 1)='Upper Design Frequency :                       '
      sublines ( 3, 1)='Lower Design Frequency :                       '

      subtitles( 4, 1)='Exit                                           '
      subtitles( 4, 2)='Tau and Sigma :                                '
      subtitles( 4, 3)='Desired Directivity :                          '

      sublines ( 4, 1)='Enter either Tau and Sigma or the Desired      '
      sublines ( 4, 2)='Directivity.  Entering Desired Directivity     '
      sublines ( 4, 3)='will result in an optimum Tau-Sigma design.    '
      sublines ( 4, 4)=' '
      sublines ( 4, 5)='To change a parameter,                         '
      sublines ( 4, 6)='please enter its line number :                 '
      
      sublines ( 5, 1)='Length to Diameter Ratio :                     '
      sublines ( 6, 1)='Source Resistance :                            '
      sublines ( 7, 1)='Length of Source Transmission Line :           '
      sublines ( 8, 1)='Impedance of Source Transmission Line :        '
      
      subtitles( 9, 1)='Exit                                           '
      subtitles( 9, 2)='Boom Spacing :                                 '
      subtitles( 9, 3)='Enter Boom Diameter :                          '
      subtitles( 9, 4)='Desired Input Impedance :                      '
      
      sublines ( 9, 1)='Enter either Boom Spacing and Boom Diameter    '
      sublines ( 9, 2)='or Desired Input Impedance and Boom Diameter.  '
      sublines ( 9, 3)='The Boom Spacing should be measured from       '
      sublines ( 9, 4)='center to center.                              '
      sublines ( 9, 5)=' '
      sublines ( 9, 6)='To change a parameter,                         '
      sublines ( 9, 7)='please enter its line number :                 '
      
      sublines (10, 1)='Length of Termination Transmission Line :      '
      sublines (11, 1)='Termination Impedance :                        '
                      
      subtitles(12, 1)='Exit                                           '
      subtitles(12, 2)='Round Element Diameters to Nearest Size :      '
      subtitles(12, 3)='Number of Available Tube Diameters :           '
      subtitles(12, 4)='1st  Available Diameter :                      '
      subtitles(12, 5)='2nd  Available Diameter :                      '
      subtitles(12, 6)='3rd  Available Diameter :                      '
      subtitles(12, 7)='4th  Available Diameter :                      '
      subtitles(12, 8)='5th  Available Diameter :                      '
      subtitles(12, 9)='6th  Available Diameter :                      '
      subtitles(12,10)='7th  Available Diameter :                      '
      subtitles(12,11)='8th  Available Diameter :                      '
      subtitles(12,12)='9th  Available Diameter :                      '
      subtitles(12,13)='10th Available Diameter :                      '
      subtitles(12,14)='11th Available Diameter :                      '
      subtitles(12,15)='12th Available Diameter :                      '
      subtitles(12,16)='13th Available Diameter :                      '
      subtitles(12,17)='14th Available Diameter :                      '
      subtitles(12,18)='15th Available Diameter :                      '
                 
      sublines (12, 1)='At most 12 Available Diameters are allowed.    '
      sublines (12, 2)=' '
      sublines (12, 3)='To change a parameter,                         '
      sublines (12, 4)='please enter its line number :                 '
      
      subtitles(13, 1)='Exit                                           '
      subtitles(13, 2)='Summarize Design (y,n)                         '
      subtitles(13, 3)='Calculate E- and H- Plane Patterns (y,n)       '
      subtitles(13, 4)='  Analysis Frequency                           '
      subtitles(13, 5)='Calculate Custom Plane Pattern (y,n)           '
      subtitles(13, 6)='  Phi                                          '
      subtitles(13, 7)='  Analysis Frequency                           '
      subtitles(13, 8)='Calculate Gain Versus Frequency (y,n)          '
      subtitles(13, 9)='  Upper Analysis Frequency                     '
      subtitles(13,10)='  Lower Analysis Frequency                     '
      subtitles(13,11)='  Number of Frequency Steps per Octave         '
      
      sublines (13, 1)='Choices which are inset are disregarded if     '
      sublines (13, 2)='they are not required.                         '
      sublines (13, 3)=' '
      sublines (13, 4)='To change a parameter,                         '
      sublines (13, 5)='please enter its line number :                 '
                 
      sublines (14, 1)='Begin Design and Analysis                      '
      
C     Initialize variables to default values found in 'Log-Perd.INI'
C     if present.  Otherwise, create files with default settings.
C
C     Initialize default settings
      TITLE      = 'Log-Periodic Dipole Array Design'
      Fhigh      = 2000
      Flow       = 1000
      Tau        = 0.88
      Sigma      = 0.164
      D0         = 0.0
      LD         = 30.0
      Rs         = 0.0
      LLin       = 0.0
      ZCin       = CMPLX(50.0,0.0)
      Rin        = 50.0
      LLout      = 0.0
      Zout       = CMPLX(0.0, 0.0)
      AFhigh     = 2500
      AFlow      = 500
      AFpowr     = 20
      AFSEH      = 1500
      AFSC       = 1500
      Phi        = 45
      Navail     = 12
      Davail( 1) = 0.0625
      Davail( 2) = 0.125
      Davail( 3) = 0.1875
      Davail( 4) = 0.250
      Davail( 5) = 0.375
      Davail( 6) = 0.500
      Davail( 7) = 0.625
      Davail( 8) = 0.750
      Davail( 9) = 0.875
      Davail(10) = 1.000
      Davail(11) = 1.250
      Davail(12) = 1.500
      Davail(13) = 1.750
      Davail(14) = 2.000
      Davail(15) = 2.500
      OutFlag    = 'YNNNNYN0'
      Quant      = 'NN'
      SB         = 0.000
      DB         = 0.750
      
C     Read saved values
      INQUIRE(FILE='Log-Perd.INI',EXIST=exist)
      IF(exist) THEN

        OPEN(10,FILE='Log-Perd.INI',FORM='FORMATTED')
        read(10,6)  TITLE
        read(10,*)  Fhigh
        read(10,*)  Flow
        read(10,*)  Tau
        read(10,*)  Sigma
        read(10,*)  D0
        read(10,*)  LD
        read(10,*)  Rs
        read(10,*)  LLin
        read(10,*)  ZCin
        read(10,*)  Rin
        read(10,*)  LLout
        read(10,*)  Zout
        read(10,*)  AFhigh
        read(10,*)  AFlow
        read(10,*)  AFpowr
        read(10,*)  AFSEH
        read(10,*)  AFSC
        read(10,*)  Phi
        read(10,*)  Navail
        read(10,*)  Davail(1)
        read(10,*)  Davail(2)
        read(10,*)  Davail(3)
        read(10,*)  Davail(4)
        read(10,*)  Davail(5)
        read(10,*)  Davail(6)
        read(10,*)  Davail(7)
        read(10,*)  Davail(8)
        read(10,*)  Davail(9)
        read(10,*)  Davail(10)
        read(10,*)  Davail(11)
        read(10,*)  Davail(12)
        read(10,*)  Davail(13)
        read(10,*)  Davail(14)
        read(10,*)  Davail(15)
        read(10,7)  Outflag
        read(10,2)  Quant
        read(10,*)  SB
        read(10,*)  DB
        CLOSE(10)

      ELSE
        write(*,*) '*** Warning ***'
        write(*,*) 'File ''Log-Perd.INI'' not found'
        write(*,*) 'File will be created with default settings'
        pause 'Press enter to continue.'

        OPEN(10,FILE='Log-Perd.INI', FORM='FORMATTED')
        write(10,*)  Fhigh
        write(10,*)  Flow
        write(10,*)  Tau
        write(10,*)  Sigma
        write(10,*)  D0
        write(10,*)  LD
        write(10,*)  Rs
        write(10,*)  LLin
        write(10,*)  ZCin
        write(10,*)  Rin
        write(10,*)  LLout
        write(10,*)  Zout
        write(10,*)  AFhigh
        write(10,*)  AFlow
        write(10,*)  AFpowr
        write(10,*)  AFSEH
        write(10,*)  AFSC
        write(10,*)  Phi
        write(10,*)  Navail
        write(10,*)  Davail(1)
        write(10,*)  Davail(2)
        write(10,*)  Davail(3)
        write(10,*)  Davail(4)
        write(10,*)  Davail(5)
        write(10,*)  Davail(6)
        write(10,*)  Davail(7)
        write(10,*)  Davail(8)
        write(10,*)  Davail(9)
        write(10,*)  Davail(10)
        write(10,*)  Davail(11)
        write(10,*)  Davail(12)
        write(10,*)  Davail(13)
        write(10,*)  Davail(14)
        write(10,*)  Davail(15)
        write(10,7)  Outflag
        write(10,2)  Quant
        write(10,*)  SB
        write(10,*)  DB
        CLOSE(10)

      ENDIF
    1 format(a1)  
    2 format(a2)
    6 format(a60)
    7 format(a7)
    
C     Allow the user to change these parameters
C
  500 FLAG = .FALSE.

C     The number of screen titles on subscreen 12 depends on Navail
      st(12) = 3+Navail
      
  510 CONTINUE
C       Must initialize inkey in case there is an error reading them
        inkey1 = 0
        inkey2 = 0

        write (*,*)
        write (*,*)  'Please see Log-Perd.DOC for information about thes
     1e parameters'
        write (*,20)  1, maintitles( 1), TITLE
        write (*,31)  2, maintitles( 2), Fhigh, ' MHz '
        write (*,31)  3, maintitles( 3), Flow,  ' MHz '
        write (*,10)  4, maintitles( 4)

        if (Tau.ne.0.0) then
          write (*,35) '   Tau :                       ', Tau
          write (*,35) '   Sigma :                     ', Sigma
        else
          write (*,37) '   Directivity :               ', D0, ' dBi '
        endif

        write (*,30)  5, maintitles( 5), LD
        write (*,31)  6, maintitles( 6), Rs,   ' Ohms'
        write (*,31)  7, maintitles( 7), LLin, ' m   '
        write (*,40)  8, maintitles( 8), real(ZCin),' + j',imag(ZCin),
     1                   ' Ohms'
        write (*,10)  9, maintitles( 9)
        write (*,37) '   Boom Diameter :             ', DB, ' cm  '

        if (SB.ne.0.0) then
        write (*,37) '   Boom Spacing :              ', SB, ' cm  '
        else
        write (*,37) '   Desired Input Impedance :   ', Rin, ' Ohms'
        endif

        write (*,31) 10, maintitles(10), LLout, ' m   '
        write (*,40) 11, maintitles(11), real(Zout),' + j',imag(Zout),
     1                   ' Ohms'
        write (*,10) 12, maintitles(12)
        write (*,10) 13, maintitles(13)
        write (*,55) '   Design Summary :            ', OutFlag(1:1)
        write (*,55) '   E- and H-plane Patterns :   ', OutFlag(2:2)
        write (*,55) '   Custom Plane Pattern :      ', OutFlag(4:4)
        write (*,55) '   Swept Frequency Analysis :  ', OutFlag(5:5)
        write (*,10) 14, maintitles(14)
        
   10   format (1x, i3, 1x, a40)
   20   format (1x, i3, 1x, a15,      a60)
   30   format (1x, i3, 1x, a40,      f12.5)
   31   format (1x, i3, 1x, a40,      f12.5, a5)
   32   format (1x,     4x, a8,  32x, f12.5)
   35   format (1x,     1x, a31, 12x, f12.5)
   37   format (1x,     1x, a31, 12x, f12.5, a5)
   40   format (1x, i3, 1x, a40,      f12.5, a4, f12.5, a5)
   50   format (1x, i3, 1x, a40, 7x,  a1)
   55   format (1x,     1x, a31, 19x, a1)
   60   format (1x, i3, 1x, a40, 4x,  i2)
   
        write (*,71) mainlines(1),mainlines(2)
   71   format (1x,a46,a10)
        read (*,*,ERR=1050) inkey1
        if ((inkey1.ge.1).and.(inkey1.le.mt+1)) then
          FLAG = .TRUE.
        endif
      IF (FLAG.eqv..FALSE.) GOTO 510

  530 inkey2 = 0
C     If this is an item which brings up a subscreen then
      IF ((inkey1.eq.4).or.(inkey1.eq.9).or.(inkey1.eq.12)
     1    .or.(inkey1.eq.13)) THEN
          
        FLAG = .FALSE.      
  520   CONTINUE
C
C     Clear screen
        do i = 1,24
          write (*,*)
        enddo
      
        IF (inkey1.eq.4) THEN
          write (*,10) 1, subtitles( 4, 1)
          write (*,10) 2, subtitles( 4, 2)
          write (*,32) 'Tau   = ',Tau
          write (*,32) 'Sigma = ',Sigma
          write (*,31) 3, subtitles( 4, 3), D0, ' dBi '
        ELSEIF (inkey1.eq.9) THEN
          write (*,10) 1, subtitles( 9, 1)
          write (*,31) 2, subtitles( 9, 2), SB, ' cm  '
          write (*,31) 3, subtitles( 9, 3), DB, ' cm  '
          write (*,31) 4, subtitles( 9, 4), Rin, ' Ohms'
        ELSEIF (inkey1.eq.12) THEN
          write (*,10)  1, subtitles(12, 1)
          write (*,50)  2, subtitles(12, 2), Quant(2:2)
          write (*,60)  3, subtitles(12, 3), Navail
          do i = 1,Navail 
            write (*,31)  i+3, subtitles(12,i+3), Davail(i), ' cm  '
          enddo
        ELSEIF (inkey1.eq.13) THEN
          write (*,10)  1, subtitles(13, 1)
          write (*,50)  2, subtitles(13, 2), OutFlag(1:1)
          write (*,*)
          write (*,50)  3, subtitles(13, 3), OutFlag(2:2)
          write (*,31)  4, subtitles(13, 4), AFSEH,  ' MHz '
          write (*,*)
          write (*,50)  5, subtitles(13, 5), OutFlag(4:4)
          write (*,31)  6, subtitles(13, 6), Phi,    ' deg '
          write (*,31)  7, subtitles(13, 7), AFSC,   ' MHz '
          write (*,*)
          write (*,50)  8, subtitles(13, 8), OutFlag(5:5)
          write (*,31)  9, subtitles(13, 9), AFhigh, ' MHz '
          write (*,31) 10, subtitles(13,10), AFlow,  ' MHz '
          write (*,30) 11, subtitles(13,11), AFpowr
        ENDIF
        
        write (*,*)
        write (*,*)
        do i = 1,sl(inkey1)
          write (*,*) sublines (inkey1,i)
        enddo
        write (*,*)
        read (*,*,ERR=1050) inkey2
        if ((inkey2.ge.1).and.(inkey2.le.st(inkey1))) FLAG = .TRUE.
          
        IF (FLAG.eqv..FALSE.) GOTO 520
        
      ENDIF
      
      write (*,*)
      
C     Handle data which can be input from the main screen first,
C     then handle subscreen data
      IF (inkey2.eq.0) THEN
        write (*,*) maintitles(inkey1)
        write (*,*)
        if (inkey1.eq. 1) then
          write (*,81) 'Old Design Title was : ',TITLE
   81     format (1x, a23, a60)
          write (*,*) 'Enter New Title'
          read (*,70,ERR=1050) TITLE
        endif
        if (inkey1.eq. 2) then
          write (*,82) 'Old Upper Design Frequency was ',Fhigh,' MHz '
   82     format (1x, a31, f12.5, a5)
          write (*,*) 'Enter new Upper Design Frequency (MHz)'
          read (*,*,ERR=1050)  dummyR
          if (dummyR.le.0.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than or equal to 0.0'
            pause 'Press enter to continue.'
            goto 1075
          endif
          if (dummyR.lt.Flow) then
            write (*,*) 'Warning -- '
            write (*,*) 'Value is less than Lower Design Frequency'
            pause 'Press enter to continue.'
          endif
          Fhigh = dummyR
        endif
        if (inkey1.eq. 3) then
          write (*,83) 'Old Lower Design Frequency was ',Flow,' MHz '
   83     format (1x, a31, f12.5, a5)
          write (*,*) 'Enter new Lower Design Frequency (MHz)'
          read (*,*,ERR=1050) dummyR
          if (dummyR.le.0.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than or equal to 0.0'
            pause 'Press enter to continue.'
            goto 1075
          endif
          if (dummyR.gt.Fhigh) then
            write (*,*) 'Warning -- '
            write (*,*) 'Value is greater than Upper Design Frequency'
            pause 'Press enter to continue.'
          endif
          Flow = dummyR
        endif
        if (inkey1.eq. 5) then
          write (*,84) 'Old Length to Diameter Ratio was ',LD
   84     format (1x, a33, F12.5)
          write (*,*) 'Enter new Length to Diameter Ratio'
          read (*,*,ERR=1050)  dummyR
          if (dummyR.le.0.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than or equal to 0.0'
            pause 'Press enter to continue.'
            goto 1075
          endif
          if (dummyR.le.20.0) then
            write (*,*) 'Warning -- '
            write (*,*) 'With a ratio this small, the dipoles are'
            write (*,*) 'becoming fat which invalidates the'
            write (*,*) 'sinusoidal current distribution approximation.'
            pause 'Press enter to continue.'
          endif
          LD = dummyR
        endif
        if (inkey1.eq. 6) then
          write (*,85) 'Old Source Resistance was ',Rs, ' Ohms'
   85     format (1x, a26, f12.5, a5)
          write (*,*) 'Enter new Source Resistance (Ohms)'
          read (*,*,ERR=1050)  dummyR
          if (dummyR.lt.0.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than 0.0'
            pause 'Press enter to continue.'
            goto 1075
          endif
          Rs = dummyR
        endif
        if (inkey1.eq. 7) then
          write (*,86)'Old Source Transmission Line Length was ',LLin,
     1                 ' m   '
   86     format (1x, a40, f12.5, a5)
          write (*,*) 'Enter new Source Transmission Line Length (m)'
          read (*,*,ERR=1050)  dummyR
          if (dummyR.lt.0.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than 0.0'
            pause 'Press enter to continue.'
            goto 1075
          endif
          LLin = dummyR
        endif
        if (inkey1.eq. 8) then
          write (*,*)  'Old Source Transmission Line Characteristic'
          write (*,87) 'Impedance was ',real(ZCin),' + j ',imag(ZCin),
     1                 ' Ohms'
   87     format (1x, a14, f12.5, a5, f12.5, a5)
          write (*,*) 'Enter the new real part (Ohms) : '
          read (*,*,ERR=1050) Cr
          write (*,*) 'Enter the new imaginary part (Ohms) : '
          read (*,*,ERR=1050) Ci
          ZCin = CMPLX(Cr, Ci)
        endif
        if (inkey1.eq.10) then
          write (*,88) 'Old Termination Transmission Line Length was '
     1                 ,LLout, ' m   '
   88     format (1x, a45, f12.5, a5)
          write(*,*)'Enter new Termination Transmission Line Length (m)'
          read (*,*,ERR=1050) dummyR
          if (dummyR.lt.0.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than 0.0'
            pause 'Press enter to continue.'
            goto 1075
          endif
          LLout = dummyR
        endif
        if (inkey1.eq.11) then
          write (*,89)  'Old Termination Impedance was ',
     1                  real(Zout),' + j ',imag(Zout), ' Ohms'
   89     format (1x, a30, f12.5, a5, f12.5, a5)
          write (*,*) 'Enter the new real part (Ohms) : '
          read (*,*,ERR=1050) Cr
          write (*,*) 'Enter the new imaginary part (Ohms) : '
          read (*,*,ERR=1050) Ci
          Zout = CMPLX(Cr, Ci)
        endif
        if (inkey1.eq.14) goto 1100
        if (inkey1.eq.15) then
          StopFlag = 'Y'
          goto 1000
        endif
      ELSEIF (inkey1.eq.4) THEN
        if (inkey2.eq. 1) goto 500
        if (inkey2.eq. 2) then
          write (*,*) 'Desired directivity will be set to zero.'
          write (*,*) 'Enter Tau :   '
          read  (*,*,ERR=1050) dummyR
          if (dummyR.le.0.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than 0.0'
            pause 'Press enter to continue.'
            goto 1075
          elseif (dummyR.ge.1.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is greater than or equal to 1.0'
            pause 'Press enter to continue.'
            goto 1075
          endif
          Tau = dummyR
          write (*,*) 'Enter Sigma : '
          read  (*,*,ERR=1050) dummyR
          if (dummyR.lt.0.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than 0.0'
            pause 'Press enter to continue.'
            goto 1075
          endif
          Sigma = dummyR
          D0 = 0.0
        endif
        if (inkey2.eq. 3) then
          write (*,*) 'Tau and Sigma will be set to zero.'
          write (*,*) 'Enter Desired Directivity (dBi) : '
          read  (*,*,ERR=1050) dummyR
          if (dummyR.lt.7.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than 7.0'
            write (*,*)
            write (*,*) 'For low gain designs, enter Tau and Sigma'
            write (*,*) 'by hand.'
            write (*,*)
            pause 'Press enter to continue.'
            goto 1075
          elseif (dummyR.gt.11.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is greater than 11.0'
            write (*,*)
            write (*,*) 'For high gain designs, enter Tau and Sigma'
            write (*,*) 'by hand.'
            write (*,*)
            pause 'Press enter to continue.'
            goto 1075
          endif
          D0 = dummyR
          Tau = 0.0
          Sigma = 0.0
        endif
      ELSEIF (inkey1.eq.9) THEN
        if (inkey2.eq. 1) goto 500
        if (inkey2.eq. 2) then
          write (*,*) 'Enter Boom Spacing (cm) : '
          read (*,*,ERR=1050) dummyR
          if (dummyR.lt.0.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than 0.0'
            pause 'Press enter to continue.'
            goto 1075
          endif
          if (dummyR.le.DB) then
            write (*,*) 'Warning -- '
            write (*,*) 'Spacing is less than or equal to the diameter'
            pause 'Press enter to continue.'
          endif
          SB = dummyR
          Rin = 0.0
        endif
        if (inkey2.eq. 3) then
          write (*,*) 'Enter Boom Diamter (cm) : '
          read (*,*,ERR=1050) dummyR
          if (dummyR.lt.0.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than 0.0'
            pause 'Press enter to continue.'
            goto 1075
          endif
          if ((dummyR.gt.SB).and.(SB.gt.0.0)) then
            write(*,*)'Warning -- '
            write(*,*)'Diameter is greater than or equal to the spacing'
            pause 'Press enter to continue.'
          endif
          DB = dummyR
        endif
        if (inkey2.eq. 4) then
          write (*,*) 'Enter Desired Input Impedance (Ohms) : '
          read (*,*,ERR=1050) dummyR
          if (dummyR.lt.0.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than 0.0'
            pause 'Press enter to continue.'
            goto 1075
          endif
          Rin = dummyR
          SB = 0.0
        endif
      ELSEIF (inkey1.eq.12) THEN
        if (inkey2.eq. 1) goto 500
        if (inkey2.eq. 2) then
          if (Quant(2:2).eq.'Y') then
            Quant(2:2) = 'N'
          else
            Quant(2:2) = 'Y'
          endif
        endif
        if (inkey2.eq. 3) then
          write (*,*) 'Enter Number of Available Tube/Wire Diameters : '
          read (*,*,ERR=1050) dummyI
          if (dummyI.le.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than or equal to 0'
            pause 'Press enter to continue.'
            goto 1075
          elseif (dummyI.gt.12) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is greater than 12'
            pause 'Press enter to continue.'
            goto 1075
          endif
          Navail = dummyI
        endif
        if ((inkey2.ge. 4).and.(inkey2.le.15)) then
          write (*,*) 'Enter Tube Size (cm) : '
          read (*,*,ERR=1050) dummyR
          if (dummyR.lt.0.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than 0.0'
            pause 'Press enter to continue.'
            goto 1075
          endif
          Davail(inkey2-3) = dummyR
        endif
      ELSEIF (inkey1.eq.13) THEN
        if (inkey2.eq. 1) goto 500
        if (inkey2.eq. 2) then
          if (OutFlag(1:1).eq.'Y') then
            OutFlag(1:1) = 'N'
          else
            OutFlag(1:1) = 'Y'
          endif
        endif
        if (inkey2.eq. 3) then
          if (OutFlag(2:2).eq.'Y') then
            OutFlag(2:2) = 'N'
            OutFlag(3:3) = 'N'
          else
            OutFlag(2:2) = 'Y'
            OutFlag(3:3) = 'Y'
          endif
        endif
        if (inkey2.eq. 4) then
          write (*,*) 'Enter Analysis Frequency (MHz) : '
          read (*,*,ERR=1050) dummyR
          if (dummyR.le.0.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than or equal to 0.0'
            pause 'Press enter to continue.'
            goto 1075
          endif
          AFSEH = dummyR
        endif
        if (inkey2.eq. 5) then
          if (OutFlag(4:4).eq.'Y') then
            OutFlag(4:4) = 'N'
          else
            OutFlag(4:4) = 'Y'
          endif
        endif
        if (inkey2.eq. 6) then
          write (*,*) 'Enter Phi (deg) : '
          read (*,*,ERR=1050) Phi
        endif
        if (inkey2.eq. 7) then
          write (*,*) 'Enter Analysis Frequency (MHz) : '
          read (*,*,ERR=1050) dummyR
          if (dummyR.lt.0.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than 0.0'
            pause 'Press enter to continue.'
            goto 1075
          endif
          AFSC = dummyR
        endif
        if (inkey2.eq. 8) then
          if (OutFlag(5:5).eq.'Y') then
            OutFlag(5:5) = 'N'
          else
            OutFlag(5:5) = 'Y'
          endif
        endif
        if (inkey2.eq. 9) then
          write (*,*) 'Enter Upper Analysis Frequency (MHz) : '
          read (*,*,ERR=1050) dummyR
          if (dummyR.le.0.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than or equal to 0.0'
            pause 'Press enter to continue.'
            goto 1075
          endif
          if (dummyR.lt.AFlow) then
            write (*,*) 'Warning -- '
            write (*,*) 'Value is less than Lower Analysis Frequency'
            pause 'Press enter to continue.'
          endif
          AFhigh = dummyR
        endif
        if (inkey2.eq.10) then
          write (*,*) 'Enter Lower Analysis Frequency (MHz) : '
          read (*,*,ERR=1050) dummyR
          if (dummyR.le.0.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than or equal to 0.0'
            pause 'Press enter to continue.'
            goto 1075
          endif
          if (dummyR.gt.AFhigh) then
            write (*,*) 'Warning -- '
            write (*,*) 'Value is greater than Upper Analysis Frequency'
            pause 'Press enter to continue.'
          endif
          AFlow = dummyR
        endif
        if (inkey2.eq.11) then
          write (*,*) 'Enter Frequency Step'
          write (*,*) 'Number of steps per octave : '
          read (*,*,ERR=1050) dummyR
          if (dummyR.le.0.0) then
            write (*,*) 'Error -- '
            write (*,*) 'Value is less than or equal to 0.0'
            pause 'Press enter to continue.'
            goto 1075
          endif
          AFpowr = dummyR
        endif
        
      ENDIF
   70 format (a60)

C     If we are on a subscreen, stay on that screen
 1025 IF (inkey2.ne.0) THEN
        GOTO 530
      ELSE
        GOTO 500
      ENDIF
        
C     Handle errors in reading data
 1050 write (*,*) 'An error occurred reading the input.'
      write (*,*) 'Please try again.'
      pause 'Press enter to continue.'
C     Rig it so that if an error occurs when entering the line
C     number on a subscreen, we return to that subscreen
 1075 if (inkey1.eq. 4) inkey2 = 1
      if (inkey1.eq. 9) inkey2 = 1
      if (inkey1.eq.12) inkey2 = 1      
      if (inkey1.eq.13) inkey2 = 1
      GOTO 1025
      
 1100 CONTINUE
C     Test frequencies to be sure they are okay
C     Only execute if we are designing (not saving and exiting)
      if (Fhigh.le.Flow) then
        write (*,*)
     1     'Upper design frequency is less than lower design frequency.'
        pause 'Press enter to continue'
        goto 500
      endif
      if ((OutFlag(5:5).eq.'Y').and.(AFhigh.le.AFlow)) then
        write (*,*)
     1 'Upper analysis frequency is less than lower analysis frequency.'
        pause 'Press enter to continue'
        inkey1 = 13
        goto 530
      endif

C     Save entered data
C     Execute whether we are exiting or continuing with design
 1000 write (*,*) 'Saving data to file Log-Perd.INI'

      OPEN(10,FILE='Log-Perd.INI', FORM='FORMATTED')
      write(10,6)  TITLE
      write(10,*)  Fhigh
      write(10,*)  Flow
      write(10,*)  Tau
      write(10,*)  Sigma
      write(10,*)  D0
      write(10,*)  LD
      write(10,*)  Rs
      write(10,*)  LLin
      write(10,*)  ZCin
      write(10,*)  Rin
      write(10,*)  LLout
      write(10,*)  Zout
      write(10,*)  AFhigh
      write(10,*)  AFlow
      write(10,*)  AFpowr
      write(10,*)  AFSEH
      write(10,*)  AFSC
      write(10,*)  Phi
      write(10,*)  Navail
      write(10,*)  Davail(1)
      write(10,*)  Davail(2)
      write(10,*)  Davail(3)
      write(10,*)  Davail(4)
      write(10,*)  Davail(5)
      write(10,*)  Davail(6)
      write(10,*)  Davail(7)
      write(10,*)  Davail(8)
      write(10,*)  Davail(9)
      write(10,*)  Davail(10)
      write(10,*)  Davail(11)
      write(10,*)  Davail(12)
      write(10,*)  Davail(13)
      write(10,*)  Davail(14)
      write(10,*)  Davail(15)
      write(10,7)  Outflag
      write(10,2)  Quant
      write(10,*)  SB
      write(10,*)  DB
      CLOSE(10)
      
      RETURN
      END

C **************************************************************
     
      SUBROUTINE LUSOLV(Z,N,NP,IPERM,V)

C **************************************************************

      IMPLICIT NONE

      INTEGER N,NP
      INTEGER IPERM(N)
      COMPLEX Z(NP,NP),V(N)

C    SOLVES LINEAR SYSTEM GIVEN THE LU DECOMPOSITION FROM LUDEC
C    FORCING VECTOR IS REPLACED WITH SOLUTION VECTOR UPON EXIT 

      INTEGER I,J,II
      COMPLEX TEMP

C    FORWARD SUBSTITUTION.
      DO 20 I=1,N
         TEMP=V(IPERM(I))
         V(IPERM(I))=V(I)
         DO 10 J=1,I-1
            TEMP=TEMP-Z(I,J)*V(J)
10       CONTINUE
         V(I)=TEMP
20    CONTINUE

C    BACKWARD SUBSTITUTION.
      DO 40 I=1,N
         II=N-I+1
         TEMP=V(II)
         DO 30 J=II+1,N
            TEMP=TEMP-Z(II,J)*V(J)
30       CONTINUE
         V(II)=TEMP/Z(II,II)
40    CONTINUE

      RETURN
      END

C **************************************************************
  
      SUBROUTINE LUDEC (Z,N,NP,IPERM,SCAL)

C **************************************************************

      INTEGER N,NP
      INTEGER IPERM(N)
      REAL    SCAL(NP)
      COMPLEX Z(NP,NP)

C    REPLACES MATRIX BY ITS LU DECOMPOSITION

      INTEGER I,J,K,IMAX
      REAL    ZMAX,CAZ,TEST
      COMPLEX TEMP

C    GET SCALING INFO.
      DO 20 I=1,N
         ZMAX=0.E0
         DO 10 J=1,N
            CAZ=CABS(Z(I,J))
            IF (CAZ.GT.ZMAX) ZMAX=CAZ
10       CONTINUE
         SCAL(I)=1.E0/ZMAX
20    CONTINUE

C    CROUT's algorithm.
      DO 80 J=1,N
         DO 30 I=1,J-1
            DO 30 K=1,I-1
            Z(I,J)=Z(I,J)-Z(I,K)*Z(K,J)
30       CONTINUE
         ZMAX=0.E0
C    SEARCH FOR LARGEST PIVOT ELEMENT.
         DO 50 I=J,N
            DO 40 K=1,J-1
               Z(I,J)=Z(I,J)-Z(I,K)*Z(K,J)
40          CONTINUE
            TEST=SCAL(I)*CABS(Z(I,J))
            IF (TEST.GE.ZMAX) THEN
               IMAX=I
               ZMAX=TEST
            END IF
50       CONTINUE

C    INTERCHANGE THE ROWS.
         IF(J.NE.IMAX)THEN
            DO 60 K=1,N
               TEMP=Z(IMAX,K)
               Z(IMAX,K)=Z(J,K)
               Z(J,K)=TEMP
60          CONTINUE
            SCAL(IMAX)=SCAL(J)
         END IF
         IPERM(J)=IMAX
C    DIVIDE BY PIVOT ELEMENT.
         IF(J.NE.N)THEN
            DO 70 I=J+1,N
               Z(I,J)=Z(I,J)/Z(J,J)
70          CONTINUE
         END IF
80    CONTINUE

      RETURN
      END


C **************************************************************
C
      SUBROUTINE OUTPUT1 (N, Rs, ZL, LLin, LLout, L, D, DB, SB, Zout,
     1             FMHz, Vin, Iin, Vout, Iout, WATTS, ZCin, IA, VA,
     2             ZinA, ZinAS, ZO, VSWR, PDO, DOA, FTBR, FSLL, TITLE, 
     3             OutFlag, Fhigh, Flow, Tau, Sigma, Phi)

C  Input variables:
C     N          : Number of elements
C     Rs         : Source resistance
C     ZL(1..ms)  : Distance from apex to each dipole
C     LLin       : Distance from apex to source
C     LLout      : Distance from apex to termination
C     L(ms)      : Length of dipoles
C     D(ms)      : Diameter of dipoles (cm)
C     DB         : Diameter of one wire in twin lead
C     SB         : Spacing of twin lead
C     Zout       : Terminating impedance
C     FMHZ       : Frequency in megahertz
C     Vin        : Voltage across source
C     Iin        : Current in source
C     Vout       : Voltage across termination
C     Iout       : Current in termination
C     WATTS      : Real power accepted by antenna
C     ZCin       : Characteristic impedance of source transmission line
C     IA(1..ms)  : Current in each dipole
C     VA(1..ms)  : Voltage across each element
C     ZinA       : Input impedance   
C     ZinAS      : Input impedance seen by the source
C     ZO         : Characteristic impedance of antenna transmission
C                       line.
C     VSWR       : Input VSWR relative to ZCin ohms
C     PDO        : Maximum gain of array
C     DOA        : Actual directivity
C     FTBR       : Front to back ratio of array
C     FSLL       : Front to sidelobe level of array
C     TITLE      : Name of array
C     OutFlag*8  : 1 : Summarize Design.
C                : 2 : E- plane Pattern.
C                : 3 : H- plane Pattern.
C                : 4 : Custom Plane Pattern.
C                : 5 : Data vs. Frequency.   
C                : 6 : Output to Screen.
C                : 7 : Output to Printer File.
C                : 8 : Number of current analysis
C                  eg. '3' means the H-plane pattern is being calculated.
C     Fhigh      : Upper design frequency
C     Flow       : Lower design frequency
C     Tau        : Ratio of adjacent element lengths ( < 1 )
C     Sigma      : Related to the apex angle
C     Phi        : Angle of custom plane pattern

C  Routine Specific Variables
C     I          : Counter
C     Z2APEX     : Distance from LLin or LLout to apex
C     dummyR     : Real dummy variable
C     dummyR2    : Real dummy variable
C     R1         : Real dummy variable
C     I1         : Real dummy variable
C     R2         : Real dummy variable
C     I2         : Real dummy variable
C
C **************************************************************
C
      implicit NONE
      integer      ms
      parameter   (ms = 30)

      integer      N
      real         Rs, ZL(ms+1), LLin, LLout, L(ms), D(ms), DB, SB
      complex      Zout
      real         FMHZ
      complex      Vin, Iin, Vout, Iout
      real         WATTS
      complex      ZCin, IA(ms), VA(ms), ZinA, ZinAS, ZO
      real         VSWR, PDO, DOA, FTBR, FSLL
      character*60 TITLE      
      character*8  OutFlag
      real         Fhigh, Flow, Tau, Sigma, Phi
      real         R1, R2, I1, I2, Z2APEX
      integer      I
      

      if (OutFlag(8:8).eq.'2') OPEN (1, FILE='LP_Epl.out')
      if (OutFlag(8:8).eq.'3') OPEN (1, FILE='LP_Hpl.out')
      if (OutFlag(8:8).eq.'4') OPEN (1, FILE='LP_Cpl.out')
      
      write (1,*)   'DIPOLE ARRAY DESIGN AND ANALYSIS'
      write (1,*)   TITLE(1:60)

      if (OutFlag(8:8).eq.'2') write (1,*)  'E-plane Pattern Data'
      if (OutFlag(8:8).eq.'3') write (1,*)  'H-plane Pattern Data'
      if (OutFlag(8:8).eq.'4') write (1,*)  'Custom Plane Pattern Data'
      if (OutFlag(8:8).eq.'4') write (1,10) 'Plane Angle = ', Phi,
     1                                      ' degrees'
   10 format (a14, f12.5, a8)

      write (1,*)
      write (1,206)   'Ele.','Z','L','D','  Volts',' ','Current',' '
      write (1,206)   ' ','(m)','(m)','(cm)','(dBV)','(deg)',
     1                '(dBAmp)','(deg)'
C
      R1 = real(Vin)
      I1 = imag(Vin)
      R2 = real(Iin)
      I2 = imag(Iin)
      Z2APEX = ZL(1) - LLin
      write (1,200) 'Source',Z2APEX,'*******','*******',
     1              R1, I1, R2, I2
      do 20 I = 1, N, 1
        R1 = real( VA(I) )
        I1 = imag( VA(I) )
        R2 = real( IA(I) )
        I2 = imag( IA(I) )
        write (1,201) I, ZL(I), L(I), D(I), real(VA(I)), imag(VA(I)),
     1                real(IA(I)), imag(IA(I))
   20 continue
      R1 = real(Vout)
      I1 = imag(Vout)
      R2 = real(Iout)
      I2 = imag(Iout)
      Z2APEX = ZL(N) + LLout
      write (1,200) 'Term. ',Z2APEX,'******* ','******* ',
     1              R1, I1, R2, I2
C
      write (1,210) 'Currents and voltages relative to ',
     1               WATTS,' Watt of input power.'
      write (1,*)
      write (1,*)   'Other Design Parameters'
      write (1,203) '  Upper Design Frequency (MHz)           : ',Fhigh
      write (1,203) '  Lower Design Frequency (MHz)           : ',Flow
      write (1,203) '  Tau                                    : ',Tau
      write (1,203) '  Sigma                                  : ',Sigma
      write (1,203) '  Source Impedance (Ohms)                : ',Rs
      write (1,*)   '  Characteristic Impedance of Source '
      write (1,213) '    Transmission Line                    : ',
     1                   real(ZCin),' + j ',imag(ZCin)
      write (1,213) '  Terminating Impedance (Ohms)           : ',
     1                   real(Zout),' + j ',imag(Zout)
      write (1,*)   '  Characteristic Impedance of Termination '
      write (1,213) '    Transmission Line                    : ',
     1                   real(ZO),' + j ',imag(ZO)
      write (1,*)
      write (1,*)   'Transmission Line Of Antenna'
      write (1,203) '  Boom Diameter (cm)                     : ',DB
      write (1,203) '  Boom Spacing (cm)                      : ',SB
      write (1,213) '  Characteristic Impedance (Ohms)        : ',
     1               real(ZO),' + j ',imag(ZO)
      write (1,*  )
      write (1,204) 'Array Parameters Analyzed At ',FMHZ,' MHz'
      write (1,203) '  Array Peak Directivity (dBi)           : ',PDO
      write (1,203) '  Array Directivity on Boresight (dBi)   : ',DOA
      write (1,203) '  Array Front-to-Back ratio (dB)         : ',FTBR
C     If FSLL > 8000, there is no sidelobe
      if (FSLL.gt.8000.0) then
       write (1,*)   '  Array Front-to-Sidelobe Level (dB)     : ******'
      else
       write (1,203) '  Array Front-to-Sidelobe Level (dB)     : ',FSLL
      endif
      write (1,*)   '  Input Impedance at shortest element '
      write (1,213) '      (Complex - Ohms)                   : ',
     1                     real(ZinA),' + j ',imag(ZinA)
      write (1,*)   '  Input Impedance at shortest element '
      write (1,203) '      (Magnitude - Ohms)                 : ',
     1                     cabs(ZinA)
      write (1,*)   '  Input Impedance seen by the source '
      write (1,213) '      (Complex - Ohms)                   : ',
     1                     real(ZinAS),' + j ',imag(ZinAS)
      write (1,*)   '  Input Impedance seen by the source '
      write (1,203) '      (Magnitude - Ohms)                 : ',
     1                     cabs(ZinAS)
      write (1,*)   '  VSWR caused by mismatch between the source'
      write (1,*)   '    transmission line and the antenna and relative'
      write (1,209) '    to ',real(ZCin),' + j ',imag(ZCin),
     1                   ' Ohms: ',VSWR
      CLOSE (1)
      
  200 format (1X, A6, 1X, F7.4, 1X, A7,   1X, A7,   1X, F7.2, 1X,
     1        F7.2, 3X, F7.2, 1X, F7.2)
  201 format (1X, I6, 1X, F7.4, 1X, F7.4, 1X, F7.5, 1X, F7.2, 1X,
     1        F7.2, 3X, F7.2, 1X, F7.2)
  203 format (1X, A43, F12.5)
  204 format (1X, A29, F12.5, A4)
  206 format (1X, A6, 1X, A7,   1X, A7,   1X, A7,   1X, A7,   1X,
     1        A7,   3X, A7,   1X, A7)
  209 format (1X, A7, F12.5, A5, F12.5, A12, F12.5)
  210 format (1X, A34, F4.1, A21)
  213 format (1X, A43, F12.5, A5, F12.5)
C
      return
      end
C **************************************************************
C
      SUBROUTINE OUTPUT2 (N, ZL, L, D, ZO, 
     1                 TITLE, Fhigh, Flow, Tau, Sigma, D0, LD, Rs, LLin,
     2                 ZCin, Rin, LLout, Zout, SB, DB)
C
       
C  This output routine produces a design summary.
C
C  Calculated values:
C  N          : Number of elements.
C  ZL(1..ms)  : Distance from apex to each dipole.
C  L(ms)      : Length of dipoles.
C  D(ms)      : Diameter of dipoles (cm).
C  ZO         : Characteristic impedance of array transmission line.
C
C  Design Inputs 
C  TITLE      : Title of antenna design.
C  Fhigh      : Upper design frequency.
C  Flow       : Lower design frequency.
C  Tau        : Geometric ratio.
C  Sigma      : Spacing factor.
C  D0         : Desired Directivity.                      
C  LD         : Designed Length to Diameter ratio.
C  Rs         : Source resistance
C  LLin       : Length of source transmission line.                                 
C  ZCin       : Characteristic impedance of source transmission line.
C  Rin        : Desired input impedance of array.
C  LLout      : Length of termination transmission line.
C  Zout       : Complex termination impedance.  Usually the imaginary
C               part is set to zero.
C  SB         : Center to center spacing of the tube.
C  DB         : Diameter of transmission line tube.  
C
C  Routine Specific Variables
C  I          : Counter
C  Z2APEX     : Distance from LLin or LLout to apex
C  Alpha      : 1/2 apex angle.
C
C **************************************************************
C
      implicit NONE 
      integer    ms
      parameter (ms = 30)
      
      integer      N
      real         ZL(ms+1), L(ms), D(ms)
      complex      ZO

      character*60 TITLE
      real         Fhigh, Flow, Tau, Sigma, D0, LD, Rs, LLin
      complex      ZCin
      real         Rin, LLout
      complex      Zout
      real         SB, DB

      integer      I
      real         Z2APEX, Alpha

      Alpha = atan((1-Tau)/4.0/Sigma)*180/3.141592654
      
      OPEN (1, FILE='LP_des.out')
      write (1,*)   'DIPOLE ARRAY DESIGN'
      write (1,*)   TITLE(1:60)
      write (1,*)
      write (1,106)   'Ele.','Z','L','D'
      write (1,106)   ' ','(m)','(m)','(cm)'
C
      Z2APEX = LLout + ZL(1)
      write (1,100) 'Term. ',Z2APEX,'*******','*******'
      do 10 I = 1, N, 1
        write (1,101) I, ZL(I), L(I), D(I)
   10 continue
      Z2APEX = ZL(N) - LLin
      write (1,100) 'Source',Z2APEX,'******* ','******* '
C
      write (1,*)
      write (1,*)   'Design Parameters'
      write (1,102) '  Upper Design Frequency (MHz)    : ',Fhigh
      write (1,102) '  Lower Design Frequency (MHz)    : ',Flow
      write (1,102) '  Tau                             : ',Tau
      write (1,102) '  Sigma                           : ',Sigma
      write (1,102) '  Alpha (deg)                     : ',Alpha
      if (D0.ne.0.0) then
        write (1,102) '  Desired Directivity             : ',D0
      else
        write (1,  *) '  Desired Directivity             : ******'
      endif
      write (1,*)
      write (1,*)   'Source and Source Transmission Line '
      write (1,102) '  Source Resistance (Ohms)        : ',Rs
      write (1,102) '  Transmission Line Length (m)    : ',LLin
      write (1,103) '  Characteristic Impedance (Ohms) : ',Real(ZCin),
     1              ' + j ',Imag(ZCin)
      write (1,*)      
      write (1,*)   'Antenna and Antenna Transmission Line'
      write (1,102) '  Length-to-Diameter Ratio        : ',LD
      write (1,102) '  Boom Diameter (cm)              : ',DB
      write (1,102) '  Boom Spacing (cm)               : ',SB
      write (1,103) '  Characteristic impedance (Ohms) : ',Real(ZO),
     1              ' + j ',Imag(ZO)
      write (1,102) '  Desired Input Impedance (Ohms)  : ',Rin
      write (1,*)
      write (1,*)   'Termination and Termination Transmission Line'
      write (1,103) '  Termination impedance    (Ohms) : ',Real(Zout),
     1              ' + j ',Imag(Zout)
      write (1,102) '  Transmission Line Length (m)    : ',LLout
      write (1,103) '  Characteristic impedance (Ohms) : ',Real(ZO),
     1              ' + j ',Imag(ZO)
      
      
C
  100 format (1X, A6, 1X, F7.4, 1X, A7,   1X, A7)
  101 format (1X, I6, 1X, F7.4, 1X, F7.4, 1X, F7.5)  
  102 format (1X, A36, F12.5)
  103 format (1X, A36, F12.5, A5, F12.5)
  106 format (1X, A6, 1X, A7,   1X, A7,   1X, A7)
C
      CLOSE (1)
      return
      end
C **************************************************************
C
      SUBROUTINE OutputMessage
C     Writes output file names to the screen

      write (*,*) '                                                    '
      write (*,*) '           Output files for Log-Perd.FOR            '
      write (*,*) '                                                    '
      write (*,*) 'Some or all files may be used                       '
      write (*,*) 'depending on menu choices.                          '
      write (*,*) '                                                    '
      write (*,*) ' 1.  LP_DES  .OUT -- Design Summary                 '
      write (*,*) ' 2.  LP_EPL  .OUT -- Analysis in E-plane            '
      write (*,*) ' 3.  LP_EPAT .DAT -- E-plane Pattern Data           '
      write (*,*) ' 4.  LP_HPL  .OUT -- Analysis in H-plane            '
      write (*,*) ' 5.  LP_HPAT .DAT -- H-plane Pattern Data           '
      write (*,*) ' 6.  LP_CPL  .OUT -- Analysis in Custom Plane       '
      write (*,*) ' 7.  LP_CPAT .DAT -- Custom Plane Pattern Data      '
      write (*,*) ' 8.  LP_SWEPT.DAT -- Composite Swept Frequency Data '
      write (*,*) ' 9.  LP_GAIN .DAT -- Gain vs. Frequency             '
      write (*,*) '10.  LP_FTBR .DAT -- Front-to-Back Ratio vs. Freq   '
      write (*,*) '11.  LP_SRC  .DAT -- Impedance at Source vs. Freq   '
      write (*,*) '12.  LP_Z_ANT.DAT -- Impedance at Antenna vs. Freq  '
      write (*,*) '13.  LP_VSWR .DAT -- VSWR vs. Frequency             '
      write (*,*) '                                                    '
      write (*,*) '                                                    '
      write (*,*) '                                                    '
      pause 'Press enter to continue.'
      RETURN
      END

C **************************************************************
C
      SUBROUTINE PATTERN(BETA,Phi,Theta,L,ZL,N,Iel,RADIUS,GAIN)
C
C  Input variables:
C    BETA      : Wavenumber.
C    Phi       : Direction in which the gain
C                is being determined.
C    Theta     : Direction in which the gain
C                is being determined.
C    L(1..ms)  : Array containing the lengths of the dipoles.
C    ZL(1..ms) : Array containing locations of dipoles.
C    N         : Number of dipoles.
C    Iel       : Array containing the currents on each dipole.
C    RADIUS    : Distance from array center to point of gain
C                calculation (units equivalent to wavelength).
C
C  Routine Specific Variables:
C    DipPat    : The pattern of the dipole
C    DipAmp    : The magnitude of the dipole pattern
C    IsoPat    : Pattern of an isotropic radiator
C    PAT       : The total (sum of all dipoles) pattern
C    KB        : The direction cosines of the point at which the gain
C                is calculated
C    PI        : 3.1415 etc.
C    ETA       : 120 * PI ohms
C    COSTH1    : cos (theta1).  Theta1 = Theta of the dipole defined
C    SINTH1    : sin (theta1)            in Balanis
C    PHASE     : The phase shift of the far-field point relative to
C                the coordinate origin using the far-field approximation
C    TINY      : A small value used for testing for a small result
C
C  Output variables:
C    GAIN      : Gain of array in the specified (Phi, Theta) direction
C
C **************************************************************
C

C
      implicit NONE
      integer N, I, ms
      parameter (ms = 30)
      real TINY
      parameter (TINY = 1.0E-12)
C
      complex IsoPAT, DipPat, Iel(ms), PAT, j
      real ZL(ms), RADIUS
      real Phi, Theta, L(ms)
      real KB(3), BETA, GAIN, PI, KL2
      real ETA, cosTH1, sinTH1, PHASE
C
      PI  = 2.0 * acos(0.0)
      j   = CMPLX(0.0,1.0)
      ETA = 120.0 * PI

C   Compute the direction cosines.
C   KB(1) = X DIRECTION
C   KB(2) = Y DIRECTION
C   KB(3) = Z DIRECTION
C
      KB(1) = sin(Theta * PI / 180.0) * cos(Phi * PI / 180.0)
      KB(2) = sin(Theta * PI / 180.0) * sin(Phi * PI / 180.0)
      KB(3) = cos(Theta * PI / 180.0)
C
C
C  Compute the gain
C   Compare the following to equation 4-62a.
C
      PAT = CMPLX (0.0, 0.0)

      DO 7 I = 1, N, 1
        PHASE   = mod(BETA * (RADIUS - ZL(I)*KB(3)),2*pi)
        IsoPAT  = j*ETA * cexp(-j*PHASE)/(2.0*PI*RADIUS)

        KL2     = BETA * L(I) / 2.0
        cosTH1  = KB(2)
        sinTH1  = sqrt(1.0 - cosTH1**2)
        if (abs(sinTH1).gt.TINY) then
          DipPat = IsoPAT*(cos(KL2*cosTH1) - cos(KL2))/sinTH1
        else
          DipPat = CMPLX(0.0,0.0)
        endif
C
C  Calculate pattern by adding the fields for each element
        PAT = PAT + Iel(I) * DipPat
    7 CONTINUE
C
C  This is equation 4-64, almost.  Also check equation 2-12a.
      if (cabs(PAT).gt.0.0) then
        GAIN = 10 * log10( cabs(PAT)**2 / 2.0 / ETA)
      else
        GAIN = -1000.0
      endif
C
      return
      end
C **************************************************************
C
      SUBROUTINE R2POL (Iel)
C
C  Converts a complex variable in rectangular form to a complex
C  variable whose real part is a magnitude and whose imaginary
C  part is its angle in degrees.
C
C  Input variables:
C    Iel : A single antenna current or voltage.
C    Note: The actual variable may also be Vel
C
C  Output variables:
C    Iel : A single antenna current or voltage.  The original Iel
C         is destroyed.
C
C  Routine specific variables:
C    MAGNITUDE : Magnitude of current (linear).
C    ANGLE     : Angle of phase in degrees.
C    PI        : 3.1415 etc.
C
C **************************************************************
C
      IMPLICIT none
      complex Iel
      real    MAGNITUDE, ANGLE, PI
      
      PI = 2.0 * ACOS(0.0)

C
      MAGNITUDE = abs(Iel)
      
      if (real(Iel).ne.0.0) then
        ANGLE = ATAN2( IMAG(Iel), REAL(Iel) ) * 180.0 / PI
      elseif (imag(Iel).ge.0.0) then
        ANGLE =  90.0
      else
        ANGLE = -90.0
      endif
      Iel = CMPLX(MAGNITUDE, ANGLE)
C
      return
      end
CC     ******************************************************************
C     COSINE INTEGRAL
C     ******************************************************************
      REAL FUNCTION CI(U)
C     ******************************************************************

      INTEGER I,TI
      REAL C,SERIES,TEMP,RK,TRK,FACTORIAL,U

      IF(U.LE.5.0)THEN
	 C=0.5772156649
	 IMAX=500
	 TOL=1.E-7
	 SERIES=0.0

	 DO I=1,IMAX
	    RK=FLOAT(I)
	    TRK=2.*RK
	    TI=2*I
	    FACTORIAL=FACT(TI)
	    TEMP=SERIES
	    SERIES=SERIES+(-1)**I*U**TI/(TRK*FACTORIAL)
	    IF(ABS(TEMP-SERIES) .LT. TOL) GOTO 4
	 ENDDO 

   4     CI=C+LOG(U)+SERIES
      ELSE
	 CI=F(U)*SIN(U)-G(U)*COS(U)
      ENDIF

      RETURN
      END

C     ******************************************************************
C     SINE INTEGRAL
C     ******************************************************************
      REAL FUNCTION SI(U)
C     ******************************************************************

      INTEGER I,TIP1
      REAL C,SERIES,TEMP,RK,TRKP1,FACTORIAL,U,PI2
      PI2=2.0*ATAN(1.0)

      IF(U.LE.5.0)THEN
	 C=0.5772156649
	 IMAX=500
	 TOL=1.E-7
	 SERIES=0.0

	 DO I=0,IMAX
	    RK=FLOAT(I)
	    TRKP1=2.*RK+1.
	    TIP1=2*I+1
	    FACTORIAL=FACT(TIP1)
	    TEMP=SERIES
	    SERIES=SERIES+(-1)**I*U**TIP1/(TRKP1*FACTORIAL)
	    IF(ABS(TEMP-SERIES) .LT. TOL) GOTO 5
	 ENDDO
      
 5       SI=SERIES
      ELSE
	 SI=PI2-F(U)*COS(U)-G(U)*SIN(U)
      ENDIF

      RETURN
      END

C     ******************************************************************
C     THIS FUNCTION IS USED BY CI() AND SI()
C     ******************************************************************
      REAL FUNCTION F(X)
C     ******************************************************************
      REAL X,NUMER,DENOM,A(4),B(4)
      
      DATA A/38.027264,265.187033,335.677320,38.102495/
      DATA B/40.021433,322.624911,570.236280,157.105423/

      NUMER=X**8+A(1)*X**6+A(2)*X**4+A(3)*X**2+A(4)
      DENOM=X**8+B(1)*X**6+B(2)*X**4+B(3)*X**2+B(4)
      F=(1./X)*NUMER/DENOM

      RETURN
      END

C     ******************************************************************
C     THIS FUNCTION IS USED BY CI() AND SI()
C     ******************************************************************
      REAL FUNCTION G(X)
C     ******************************************************************
      REAL X,NUMER,DENOM,A(4),B(4)

      DATA A/42.242855,302.757865,352.018498,21.821899/
      DATA B/48.196927,482.485984,1114.978885,449.690326/

      NUMER=X**8+A(1)*X**6+A(2)*X**4+A(3)*X**2+A(4)
      DENOM=X**8+B(1)*X**6+B(2)*X**4+B(3)*X**2+B(4)
      G=(1./X**2)*NUMER/DENOM

      RETURN
      END

C     ******************************************************************
C     THIS FUNCTION CALCULATES THE FACTORIAL OF AN INTEGER NUMBER
C     ******************************************************************
      REAL FUNCTION FACT(IARG)
C     ******************************************************************
      
      FACT=1.0
	  DO J=1,IARG
	    FACT=FLOAT(J)*FACT
	  ENDDO
      RETURN
      END




