C     ******************************************************************
C     THIS PROGRAM DESIGNS AN OPTIMUM GAIN PYRAMIDAL HORN BASED ON THE
C     FORMULATION OF SECTION 13.4.3, EQUATIONS (13-55) - (13-58b).
C
C     THE PROGRAM COMPUTES THE:
C
C     I.   HORN DIMENSIONS A1, B1, RHOE, RHOH, PE, PH (IN CM)
C     II.  HORN FLARE ANGLES PSIE (E-PLANE) AND PSIH (H-PLANE) 
C          (IN DEGREES)
C
C       ** INPUT PARAMETERS:
C
C       1.  DESIRED GAIN Go (IN dB)
C       2.  FREQUENCY (IN GHz)
C       3.  FEED RECTANGULAR WAVEGUIDE DIMENSIONS: A, B (IN CM)
C
C       ** NOTE: REFER TO FIGURE 13.18 FOR THE GEOMETRY
C
C     THIS PROGRAM USES A BISECTION METHOD ROUTINE TO FIND THE ROOT OF
C     THE DESIGN EQUATION (NUMERICAL RECIPES in FORTRAN, SECOND EDITION,
C     PAGE 347)
C     ******************************************************************
C     Written by: Anastasis C. Polycarpou, Arizona State University
C
C     ******************************************************************
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  ) THEN
         WRITE(6,150)
         READ(5,*,ERR=999) FILNAM
         OPEN(UNIT=DEVICE,FILE=FILNAM,STATUS='UNKNOWN')

      ELSE 
         WRITE(6,160)
         STOP

      ENDIF

C     *** INPUT SPECIFICATIONS ***
C
      WRITE(6,50)
      READ(5,*,ERR=999) GODB

      WRITE(6,60)
      READ(5,*,ERR=999) FREQ

      WRITE(6,70)
      READ(5,*,ERR=999) A

      WRITE(6,80)
      READ(5,*,ERR=999) B

C     *** CONVERT THE GAIN FROM dB TO DIMENSIONLESS ***
C
      GO=10.0**(GODB/10.0)
      LAMBDA=30.0/FREQ

C     *** ROOT ESTIMATION OF THE DESIGN EQUATION (14-56) ***
C
      GUESS=GO/(2.*PI*SQRT(2.*PI))  !  INITIAL TRIAL VALUE BASED ON
                                    !  EQUATION (14-57)
      LOWBND=GUESS-FRACTION*GUESS
      UPBND=GUESS+FRACTION*GUESS
      CHI=RTBIS(DESEQU,LOWBND,UPBND,TOL,JMAX)

C     *** COMPUTE RHOE AND RHOH (IN CM) ***
C     * THE CALCULATIONS ARE BASED ON EQUATIONS (14-56a) AND (14-56b) *
C
      RHOE=CHI*LAMBDA
      RHOH=(GO**2/(8.*PI**3*CHI))*LAMBDA

C     *** COMPUTE A1 AND B1 (IN CM) ***
C     * THE CALCULATIONS ARE BASED ON EQUATIONS (14-58a) AND (14-58b) *
C
      A1=(GO/(2.*PI)*SQRT(3./(2.*PI*CHI)))*LAMBDA
      B1=SQRT(2.*CHI)*LAMBDA

C     *** COMPUTE PE AND PH (IN CM) ***
C     * THE CALCULATIONS ARE BASED ON EQUATIONS (14-49a) AND (14-49b) *
C
      PE=(B1-B)*SQRT((RHOE/B1)**2-0.25)
      PH=(A1-A)*SQRT((RHOH/A1)**2-0.25)

C     *** COMPUTE HORN FLARE ANGLES PSIE AND PSIH (IN DEGREES) ***
C
      PSIE=RTOD*ASIN(B1/(2.*RHOE))
      PSIH=RTOD*ASIN(A1/(2.*RHOH))

C     *** ECHO THE INPUT PARAMETERS ***
C
      WRITE(DEVICE,110)
      WRITE(DEVICE,120) GODB,FREQ,A,B

C     *** OUTPUT THE DESIGNED PARAMETERS ***
C
      WRITE(DEVICE,90)
      WRITE(DEVICE,100) A1,B1,RHOE,RHOH,PE,PH,PSIE,PSIH

C     *** FORMAT STATEMENTS ***
C
 50   FORMAT(3X,'THE DESIRED GAIN OF THE HORN IN DB = ',$)
 60   FORMAT(3X,'THE FREQUENCY OF OPERATION IN GHz = ',$)
 70   FORMAT(3X,'THE HORN DIMENSION A IN CM = ',$)
 80   FORMAT(3X,'THE HORN DIMENSION B IN CM = ',$)

 90   FORMAT(/,3X,'DESIGNED PARAMETERS FOR THE OPTIMUM GAIN HORN',/,3X
     $      ,'---------------------------------------------',/)

 100  FORMAT(3X,'A1   = ',F12.6,1X,'CM',5X,'B1   = ',F12.6,1X,'CM',/,
     $       3X,'RHOE = ',F12.6,1X,'CM',5X,'RHOH = ',F12.6,1X,'CM',/,
     $       3X,'Pe   = ',F12.6,1X,'CM',5X,'Ph   = ',F12.6,1X,'CM',/,
     $       3X,'PSIE = ',F12.6,1X,'DEG',4X,'PSIH = ',F12.6,1X,'DEG')

 110  FORMAT(/,3X,'*************** PROGRAM OUTPUT ***************',/,3X,
     $      'INPUT PARAMETERS ',/,3X,'----------------',/)

 120  FORMAT(3X,'DESIRED GAIN OF THE HORN   = ',F12.6,1X,'dB',/,3X
     $      ,'FREQUENCY OF OPERATION     = ',F12.6,1X,'GHz',/,3X
     $      ,'FEED WAVEGUIDE DIMENSION A = ',F12.6,1X,'CM',/,3X
     $      ,'FEED WAVEGUIDE DIMENSION B = ',F12.6,1X,'CM')

 130  FORMAT(3X,'OUTPUT DEVICE OPTION',/,6X
     $      ,'OPTION (1): SCREEN',/,6X,'OPTION (2): OUTPUT FILE',//,3X
     $      ,'OUTPUT DEVICE = ',$)

 140  FORMAT(I1)

 150  FORMAT(3X,'INPUT THE DESIRED OUTPUT FILNAME (in single quotes) = '
     &      ,$)

 160  FORMAT(/,3X,'*** ERROR ***',/,3X
     $      ,'OUTPUTING DEVICE NUMBER SHOULD BE EITHER 1 OR 2')

      STOP  ! END OF THE MAIN PROGRAM  


C     ********** ERROR CONDITIONS **********
C
 999     WRITE(6,1000)
 1000    FORMAT(/,3X,'***** ERROR *****',/,3X
     $         ,'INPUT DATA ARE NOT OF THE RIGHT FORMAT',/)

      END

C     ******************************************************************
C     DESIGN EQUATION FOR AN OPTIMUM GAIN PYRAMIDAL HORN
C     ******************************************************************
      FUNCTION DESEQU(CHI)
C     ******************************************************************

      REAL CHI,GO,A,B,F1,F2,F3,PI,LAMBDA
      COMMON /PARAMETERS/ PI,GO,A,B,LAMBDA

      F1=(SQRT(2.*CHI)-B/LAMBDA)**2*(2.*CHI-1.)
      F2=((GO/(2.*PI))*SQRT(3./(2.*PI))*(1./SQRT(CHI))-A/LAMBDA)**2
      F3=(GO**2)/((6.*PI**3)*CHI)-1.
      DESEQU=F1-F2*F3

      RETURN
      END

C     ******************************************************************
C     ROOT FINDER BASED ON BISECTION
C     ******************************************************************
      FUNCTION RTBIS(FUNC,X1,X2,XACC,JMAX)
C     ******************************************************************

      FMID=FUNC(X2)
      F=FUNC(X1)
      IF(F*FMID.GE.0.)THEN
         WRITE(6,12) 
         WRITE(6,13)
         STOP
      ENDIF
      IF(F.LT.0.)THEN
         RTBIS=X1
         DX=X2-X1
      ELSE
         RTBIS=X2
         DX=X1-X2
      ENDIF
      DO 11 J=1,JMAX
         DX=DX*.5
         XMID=RTBIS+DX
         FMID=FUNC(XMID)
         IF(FMID.LE.0.)RTBIS=XMID
         IF(ABS(DX).LT.XACC .OR. FMID.EQ.0.) RETURN
 11   CONTINUE
      WRITE(6,14)
 12   FORMAT(/,3X,'***** ERROR *****',/,3X
     $      ,'THERE IS NO ROOT WITHIN THE GIVEN LIMITS',/)
 13   FORMAT(3X,'OPTIONS:',2X,'(1)  USE ANOTHER INITIAL GUESS VALUE',/
     $      ,13X,'(2)  INCREASE THE PARAMETER CALLED FRACTION',/,13X
     $      ,'(3)  BOTH THE ABOVE',/)
 14   FORMAT(/,3X,'***** ERROR *****',/,3X
     $      ,'THE BISECTION ROUTINE DID NOT CONVERGE',/,3X
     $      ,'INCREASE THE PARAMETER JMAX',/)
      END






