C
C     PYRAMIDAL HORN: ANALYSIS
C     ******************************************************************
C     THIS PROGRAM COMPUTES FOR A PYRAMIDAL HORN THE:
C
C     I.    FAR-FIELD E- AND H-PLANE AMPLITUDE PATTERNS BASED ON THE
C           THEORY OF SECTION 13.4 EQUATIONS (13-46) - (13-48c)
C     II.   DIRECTIVITY (IN dB) BASED ON EQUATION (13-52)
C     III.  DIRECTIVITY (IN dB) OF THE CORRESPONDING E-PLANE SECTORAL
C           HORN BASED ON EQUATION (13-19)
C     IV.   DIRECTIVITY (IN dB) OF THE CORRESPONDING H-PLANE SECTORAL 
C           HORN BASED ON EQUATION (13-41)
C
C        ** INPUT PARAMETERS:
C
C       1.  RHO1 (IN WAVELENGTHS)
C       2.  RHO2 (IN WAVELENGTHS)
C       3.  WAVEGUIDE DIMENSIONS a & b (IN WAVELENGTHS)
C       4.  HORN APERTURE DIMENSIONS a1 & b1 (IN WAVELENGTHS)
C
C        ** NOTE: REFER TO FIGURE 13.18 FOR THE GEOMETRY.
C                 THE E- AND H-PLANE AMPLITUDE PATTERNS ARE STORED IN 
C                 TWO DATA FILES NAMELY Epl-Horn.dat AND Hpl-Horn.dat,
c                 RESPECTIVELY.
C     ******************************************************************
C     Written by: Panayiotis A. Tirkas, Arizona State University
C
C     ******************************************************************
      INTEGER DEVICE
      COMPLEX J,ETHETAC,EPHIC,CAPFP,CAPFPP,CAPF
      COMPLEX EXP1,EXP2,EXP3,AINT1,AINT2
      REAL X1,X2,X3,Y1,Y2,Y3,KY,Dp,DpdB,De,DedB,Dh,DhdB
      REAL ETHETA(721),EPHI(721)
      REAL KXP,KXPP,T1P,T2P,T1PP,T2PP
      CHARACTER*12 FILNAM
      EXTERNAL FRNELS

C     CONSTANT DEFINITIONS
C     
      PI=4.0*ATAN(1.0)
      DTOR=PI/180.
      RTOD=180.0/PI
      J=CMPLX(0.0,1.0)
      ETHETAMAX=1.0E-10
      EPHIMAX=1.0E-10

C
      OPEN(UNIT=7,FILE='Epl-Horn.dat',STATUS='UNKNOWN')
      OPEN(UNIT=8,FILE='Hpl-Horn.dat',STATUS='UNKNOWN')
C
C     ********** CHOICE OF OUTPUT **********
C
      WRITE(DEVICE,70)
      READ(5,100,ERR=999) DEVICE

      IF(DEVICE .EQ. 1) THEN
         DEVICE=6
      ELSE IF(DEVICE .EQ. 2) THEN
         WRITE(6,80)
         READ(5,*,ERR=999) FILNAM
         OPEN(UNIT=DEVICE,FILE=FILNAM,STATUS='UNKNOWN')
      ELSE 
         WRITE(6,90)
         STOP
      ENDIF

C     *** READ THE INPUT DATA ***
C
      WRITE(6,105)
      WRITE(6,110)
      READ(5,*) RHO1
      WRITE(6,120)
      READ(5,*) RHO2

      WRITE(6,130)
      WRITE(6,140)
      READ(5,*) A
      WRITE(6,150)
      READ(5,*) B

      WRITE(6,160)
      WRITE(6,170)
      READ(5,*) A1
      WRITE(6,180)
      READ(5,*) B1

C
      KXP=PI/A1
      T1P=SQRT(1.0/(2.0*PI*PI*RHO2))*(-PI*A1-RHO2*KXP)
      T2P=SQRT(1.0/(2.0*PI*PI*RHO2))*(+PI*A1-RHO2*KXP)
C
      KXPP=-PI/A1
      T1PP=SQRT(1.0/(2.0*PI*PI*RHO2))*(-PI*A1-RHO2*KXPP)
      T2PP=SQRT(1.0/(2.0*PI*PI*RHO2))*(+PI*A1-RHO2*KXPP)
      IF(T1P.GE.0)THEN
         CALL FRNELS(X1,Y1,SQRT(PI/2.0)*T1P)
      ELSE
         CALL FRNELS(X1,Y1,-SQRT(PI/2.0)*T1P)
         X1=-X1
         Y1=-Y1
      ENDIF
      IF(T2P.GE.0)THEN
         CALL FRNELS(X2,Y2,SQRT(PI/2.0)*T2P)
      ELSE
         CALL FRNELS(X2,Y2,-SQRT(PI/2.0)*T2P)
         X2=-X2
         Y2=-Y2
      ENDIF
      CAPFP=X2 - X1 -J*(Y2-Y1)
C
      IF(T1PP.GE.0)THEN
         CALL FRNELS(X1,Y1,SQRT(PI/2.0)*T1PP)
      ELSE
         CALL FRNELS(X1,Y1,-SQRT(PI/2.0)*T1PP)
         X1=-X1
         Y1=-Y1
      ENDIF
      IF(T2PP.GE.0)THEN
         CALL FRNELS(X2,Y2,SQRT(PI/2.0)*T2PP)
      ELSE
         CALL FRNELS(X2,Y2,-SQRT(PI/2.0)*T2PP)
         X2=-X2
         Y2=-Y2
      ENDIF
C
      CAPFPP=(X2-X1)-J*(Y2-Y1)
      EXP1=CEXP(J*KXP**2*RHO2/(2.0*2.0*PI))
      EXP2=CEXP(J*KXPP**2*RHO2/(2.0*2.0*PI))
      AINT1=0.5*SQRT(RHO2/2.0)*(EXP1*CAPFP+EXP2*CAPFPP)
C
      DO 10 I=1,721
         THETA=FLOAT(I-1)*DTOR/2.0
         KY=2.0*PI*SIN(THETA)
         T1=SQRT(1.0/(2.0*PI*PI*RHO1))*(-PI*B1-KY*RHO1)
         T2=SQRT(1.0/(2.0*PI*PI*RHO1))*(+PI*B1-KY*RHO1)
         IF(T1.GE.0)THEN
            CALL FRNELS(X1,Y1,SQRT(PI/2.0)*T1)
         ELSE
            CALL FRNELS(X1,Y1,-SQRT(PI/2.0)*T1)
            X1=-X1
            Y1=-Y1
         ENDIF
         IF(T2.GE.0)THEN
            CALL FRNELS(X2,Y2,SQRT(PI/2.0)*T2)
         ELSE
            CALL FRNELS(X2,Y2,-SQRT(PI/2.0)*T2)
            X2=-X2
            Y2=-Y2
         ENDIF
         CAPF=(X2-X1)-J*(Y2-Y1)
         EXP3=CEXP(J*KY**2*RHO1/(2.0*2.0*PI))
         AINT2=SQRT(RHO1/2.0)*EXP3*CAPF
         ARG3=(1.0+COS(THETA))
C
         ETHETAC=J/2.0*ARG3*AINT1*AINT2
         ETHETA(I)=CABS(ETHETAC)
 10   CONTINUE
      WRITE(7,190)
      DO 20 I=1,721
         IF(ETHETA(I).GT.ETHETAMAX)ETHETAMAX=ETHETA(I)
 20   CONTINUE
      DO 30 I=1,721
         THETA=FLOAT(I-1)/2.0
         IF(ETHETA(I).GT.1.0E-6)THEN
            ETHETADB=20.0*ALOG10(ETHETA(I)/ETHETAMAX)
         ELSE
            ETHETADB=-80.0
         ENDIF
         WRITE(7,200)THETA,ETHETADB
 30   CONTINUE

      DO 40 I=1,721
         THETA=FLOAT(I-1)*DTOR/2.0
         KXP=2.0*PI*SIN(THETA)+PI/A1
         T1P=SQRT(1.0/(2.0*PI*PI*RHO2))*(-PI*A1-RHO2*KXP)
         T2P=SQRT(1.0/(2.0*PI*PI*RHO2))*(+PI*A1-RHO2*KXP)
C
         KXPP=2.0*PI*SIN(THETA)-PI/A1
         T1PP=SQRT(1.0/(2.0*PI*PI*RHO2))*(-PI*A1-RHO2*KXPP)
         T2PP=SQRT(1.0/(2.0*PI*PI*RHO2))*(+PI*A1-RHO2*KXPP)
         IF(T1P.GE.0)THEN
            CALL FRNELS(X1,Y1,SQRT(PI/2.0)*T1P)
         ELSE
            CALL FRNELS(X1,Y1,-SQRT(PI/2.0)*T1P)
            X1=-X1
            Y1=-Y1
         ENDIF
         IF(T2P.GE.0)THEN
            CALL FRNELS(X2,Y2,SQRT(PI/2.0)*T2P)
         ELSE
            CALL FRNELS(X2,Y2,-SQRT(PI/2.0)*T2P)
            X2=-X2
            Y2=-Y2
         ENDIF
         CAPFP=X2 - X1 - J*(Y2-Y1)
C     
         IF(T1PP.GE.0)THEN
            CALL FRNELS(X1,Y1,SQRT(PI/2.0)*T1PP)
         ELSE
            CALL FRNELS(X1,Y1,-SQRT(PI/2.0)*T1PP)
            X1=-X1
            Y1=-Y1
         ENDIF
         IF(T2PP.GE.0)THEN
            CALL FRNELS(X2,Y2,SQRT(PI/2.0)*T2PP)
         ELSE
            CALL FRNELS(X2,Y2,-SQRT(PI/2.0)*T2PP)
            X2=-X2
            Y2=-Y2
         ENDIF
         CAPFPP=X2 - X1 - J*(Y2-Y1)
         EXP1=CEXP(J*KXP**2*RHO2/(2.0*2.0*PI))
         EXP2=CEXP(J*KXPP**2*RHO2/(2.0*2.0*PI))
         AINT1=0.5*SQRT(RHO2/2.0)*(EXP1*CAPFP+EXP2*CAPFPP)
C
         KY=0.0
         T1=SQRT(1.0/(2.0*PI*PI*RHO1))*(-PI*B1-KY*RHO1)
         T2=SQRT(1.0/(2.0*PI*PI*RHO1))*(+PI*B1-KY*RHO1)
         IF(T1.GE.0)THEN
            CALL FRNELS(X1,Y1,SQRT(PI/2.0)*T1)
         ELSE
            CALL FRNELS(X1,Y1,-SQRT(PI/2.0)*T1)
            X1=-X1
            Y1=-Y1
         ENDIF
         IF(T2.GE.0)THEN
            CALL FRNELS(X2,Y2,SQRT(PI/2.0)*T2)
         ELSE
            CALL FRNELS(X2,Y2,-SQRT(PI/2.0)*T2)
            X2=-X2
            Y2=-Y2
         ENDIF
         CAPF=(X2-X1)-J*(Y2-Y1)
         EXP3=CEXP(J*KY**2*RHO1/(2.0*2.0*PI))
         AINT2=SQRT(RHO1/2.0)*EXP3*CAPF
         ARG3=(1.0+COS(THETA))
         EPHIC=J/2.0*ARG3*AINT1*AINT2
         EPHI(I)=CABS(EPHIC)
 40   CONTINUE
      WRITE(8,195)
      DO 50 I=1,721
         IF(EPHI(I).GT.EPHIMAX)EPHIMAX=EPHI(I)
 50   CONTINUE
      DO 60 I=1,721
         THETA=FLOAT(I-1)/2.0
         IF(EPHI(I).GT.1.0E-6)THEN
            EPHIDB=20.0*ALOG10(EPHI(I)/EPHIMAX)
         ELSE
            EPHIDB=-80.0
         ENDIF
         WRITE(8,200) THETA,EPHIDB
 60   CONTINUE
C
      U=1.0/SQRT(2.0)*(SQRT(RHO2)/A1+A1/SQRT(RHO2))
      V=1.0/SQRT(2.0)*(SQRT(RHO2)/A1-A1/SQRT(RHO2))
      W=B1/SQRT(2.0*RHO1)
C
      CALL FRNELS(X1,Y1,SQRT(PI/2.0)*U)

      IF(V.GE.0)THEN
         CALL FRNELS(X2,Y2,SQRT(PI/2.0)*V)
      ELSE
         CALL FRNELS(X2,Y2,-SQRT(PI/2.0)*V)
         X2=-X2
         Y2=-Y2
      ENDIF
      
      CALL FRNELS(X3,Y3,SQRT(PI/2.0)*W)

C     PROGRAM OUTPUT
C
      WRITE(DEVICE,270)

C     DIRECTIVITY FOR THE PYRAMIDAL HORN
C
      Dp = 8.0*PI*RHO1*RHO2/(A1*B1)*((X1-X2)**2+(Y1-Y2)**2)*
     &      (X3**2+Y3**2)
      DpdB=10*ALOG10(Dp)

      WRITE(DEVICE,205) 
      WRITE(DEVICE,210) DpdB
      WRITE(DEVICE,220) Dp

C     DIRECTIVITY FOR THE E-PLANE SECTORAL HORN
C
      De = 64.0*A*RHO1/(PI*B1)*(X3**2+Y3**2)
      DedB=10*ALOG10(De)

      WRITE(DEVICE,225) 
      WRITE(DEVICE,210) DedB
      WRITE(DEVICE,220) De

C     DIRECTIVITY FOR THE H-PLANE SECTORAL HORN
C
      Dh = 4.0*PI*B*RHO2/A1*((X1-X2)**2+(Y1-Y2)**2)
      DhdB=10*ALOG10(Dh)

      WRITE(DEVICE,245) 
      WRITE(DEVICE,210) DhdB
      WRITE(DEVICE,220) Dh


      IF(DEVICE .EQ. 6) THEN
         WRITE(DEVICE,5)
      ELSE IF(DEVICE .EQ. 2) THEN
         WRITE(6,5)
         WRITE(DEVICE,5)
      ENDIF


C     ********** FORMAT STATEMENTS **********

 5    FORMAT(/,3X,'*** NOTE:',/,7X,
     $      'THE E-PLANE AMPLITUDE PATTERN IS STORED IN Epl-Horn.dat'
     $      ,/,7X,
     $      'THE H-PLANE AMPLITUDE PATTERN IS STORED IN Hpl-Horn.dat',/
     &      ,7X,'======================================================'
     &      )

 70   FORMAT(3X,'OUTPUT DEVICE OPTION',/,6X
     $      ,'OPTION (1): SCREEN',/,6X,'OPTION (2): OUTPUT FILE',//,3X
     $      ,'OUTPUT DEVICE = ',$)

 80   FORMAT(3X,'INPUT THE DESIRED OUTPUT FILNAME (in single quotes) = '
     &      ,$)

 90   FORMAT(/,3X,'*** ERROR ***',/,3X
     $      ,'OUTPUTING DEVICE NUMBER SHOULD BE EITHER 1 OR 2')

 100  FORMAT(I1)
 105  FORMAT(/,3X,'E-PLANE AND H-PLANE HORN SPECIFICATIONS',/,3X
     &      ,'---------------------------------------',/)
 110  FORMAT(3X,'RHO1 (in wavelengths) = ',$)
 120  FORMAT(3X,'RHO2 (in wavelengths) = ',$)
 130  FORMAT(/,3X,'WAVEGUIDE DIMENSIONS',/,3X,'--------------------',/)
 140  FORMAT(3X,'A (in wavelengths) = ',$)
 150  FORMAT(3X,'B (in wavelengths) = ',$)
 160  FORMAT(/,3X,'HORN APERTURE DIMENSIONS',/,3X
     &      ,'------------------------',/)
 170  FORMAT(3X,'A1 (in wavelengths) = ',$)
 180  FORMAT(3X,'B1 (in wavelengths) = ',$)
 190  FORMAT('#',9X,'THETA         E-THETA',/,'#')
 195  FORMAT('#',9X,'THETA          E-PHI',/,'#')
 200  FORMAT(2(F16.6))
 205  FORMAT(3X,'PYRAMYDAL HORN',/,3X,'--------------')
 210  FORMAT(3X,'DIRECTIVITY = ',F16.3,2X,'dB')
 220  FORMAT(3X,'DIRECTIVITY = ',F16.3,2X,'dimensionless')
 225  FORMAT(/,3X,'E-PLANE SECTORAL HORN',/,3X,'---------------------')
 230  FORMAT(3X,'DIRECTIVITY = ',F16.3,2X,'dB')
 240  FORMAT(3X,'DIRECTIVITY = ',F16.3,2X,'dimensionless')
 245  FORMAT(/,3X,'H-PLANE SECTORAL HORN',/,3X,'---------------------')
 250  FORMAT(3X,'DIRECTIVITY = ',F16.3,2X,'dB')
 260  FORMAT(3X,'DIRECTIVITY = ',F16.3,2X,'dimensionless')
 270  FORMAT(/,3X
     &      ,'******************************************************',/
     &      ,3X,'PROGRAM OUTPUT',/,3X
     &      ,'******************************************************',/)


      STOP                      ! END OF THE MAIN PROGRAM
C     

C     ********** ERROR CONDITIONS **********
C
 999     WRITE(6,1000)
 1000    FORMAT(/,3X,'***** ERROR *****',/,3X
     $         ,'INPUT DATA ARE NOT OF THE RIGHT FORMAT',/)

      END

C
C

C**********************************************************
C
C     SUBROUTINE FRNELS      (AFTER BOERSMA)
C
C     INPUT:
C       (R)XS                ARGUMENT
C
C     OUTPUT:
C       (R)C,S               REAL AND IMAG PARTS OF INTEGRAL
C
C     PURPOSE:
C       CALCULATE FRESNEL INTEGRAL.  (SEE BOERSMA)
C
C     USAGE:
C       FRNELS(C,S,XS)
C
C**********************************************************
      SUBROUTINE FRNELS(C,S,XS)
      DIMENSION A(12),B(12),CC(12),D(12)
      DATA A/1.595769140,-0.000001702,-6.808568854,-0.000576361,6.920691
     *902,-0.016898657,-3.050485660,-0.075752419,0.850663781,-0.0256390
     *41,-0.150230960,0.034404779/
      DATA B/-0.000000033,4.255387524,-0.000092810,-7.780020400,-0.00952
     *0895,5.075161298,-0.138341947,-1.363729124,-0.403349276,0.7022220
     *16,-0.216195929,0.019547031/
      DATA CC/0.,-0.024933975,0.000003936,0.005770956,0.000689892,-0.009
     *497136,0.011948809,-0.006748873,0.000246420,0.002102967,-0.001217
     *930,0.000233939/
      DATA D/0.199471140,0.000000023,-0.009351341,0.000023006,0.00485146
     *6,0.001903218,-0.017122914,0.029064067,-0.027928955,0.016497308,-
     *.005598515,0.000838386/
      IF (XS.LE.0.0) GOTO 414
      X=XS
      X=X*X
      FR=0.0
      FI=0.0
      K=13
      IF(X-4.0) 10,40,40
   10 Y=X/4.0
   20 K=K-1
      FR=(FR+A(K))*Y
      FI=(FI+B(K))*Y
      IF(K-2) 30,30,20
   30 FR=FR+A(1)
      FI=FI+B(1)
      C=(FR*COS(X)+FI*SIN(X))*SQRT(Y)
      S=(FR*SIN(X)-FI*COS(X))*SQRT(Y)
      RETURN
   40 Y=4.0/X
   50 K=K-1
      FR=(FR+CC(K))*Y
      FI=(FI+D(K))*Y
      IF(K-2) 60,60,50
   60 FR=FR+CC(1)
      FI=FI+D(1)
      C=0.5+(FR*COS(X)+FI*SIN(X))*SQRT(Y)
      S=0.5+(FR*SIN(X)-FI*COS(X))*SQRT(Y)
      RETURN
  414 C=-0.0
      S=-0.0
      RETURN
      END
C
