C     *****************************************************************
C     THIS PROGRAM COMPUTES THE RADIATION CHARACTERISTICS:
C
C     I.   LINEAR ARRAYS(UNIFORM & BROADSIDE NONUNIFORM)
C     II.  PLANAR ARRAY (BROADSIDE UNIFORM)
C
C     THE UNIFORM AND BROADSIDE NONUNIFORM LINEAR ARRAYS HAVE N ELEMENTS
C     PLACED EQUIDISTANTLY ALONG THE Z-AXIS.
C
C     BROADSIDE PLANAR UNIFORM ARRAY HAS M x N ELEMENTS PLACED
C     EQUIDISTANTLY ALONG THE X AND Y AXES  
C     
C     OPTION I.  LINEAR ARRAYS
C
C       OPTION A.  UNIFORM
C         ** CHOICES: ARRAY TYPE
C
C         1. BROADSIDE (MAXIMUM ALONG THETA = 0 DEGREES)
C         2. ORDINARY END-FIRE
C            (a). MAXIMUM ALONG THETA = 0 DEGREES
C            (b). MAXIMUM ALONG THETA = 180 DEGREES
C         3. HANSEN WOODYARD END-FIRE
C            (a). MAXIMUM ALONG THETA = 0 DEGREES
C            (b). MAXIMUM ALONG THETA = 180 DEGREES
C         4. SCANNING (MAXIMUM ALONG THETA = THETA_MAX)
C
C         ** INPUT PARAMETERS:
C
C            1. NUMBER OF ELEMENTS
C            2. SPACING BETWEEN THE ELEMENTS (IN WAVELENGTHS)
C            3. DIRECTION OF ARRAY MAXIMUM (THETA_MAX IN DEGREES)
C
C         ** PROGRAM OUTPUT:
C
C            1. NORMALIZED ARRAY FACTOR
C            2. DIRECTIVITY (DIMENSIONLESS & IN dB) USING NUMERICAL 
C               INTEGRATION OF THE ARRAY FACTOR
C            3. HALF-POWER BEAMWIDTH (IN DEGREES) USING AN ITERATIVE
C               METHOD (FOR ALL MAXIMA IN THE PATTERN)
C
C       OPTION B.  NONUNIFORM BROADSIDE
C         ** CHOICES: ARRAY TYPE
C
C         1. BINOMIAL
C         2. DOLPH-TSCHEBYSCHEFF
C
C         ** BINOMIAL ARRAY INPUT PARAMETERS:
C
C         1. NUMBER OF ELEMENTS
C         2. SPACING BETWEEN THE ELEMENTS (IN WAVELENGTHS)
C
C         ** DOLPH-TSCHEBYSCHEFF INPUT PARAMETERS:
C
C         1. NUMBER OF ELEMENTS
C         2. SPACING BETWEEN THE ELEMENTS (IN WAVELENGTHS)
C         3. SIDE LOBE LEVEL (IN POSITIVE dB; i.e., 30 dB)
C
C         ** PROGRAM OUTPUT:
C
C         1. NORMALIZED EXCITATION COEFFICIENTS (An)
C         2. NORMALIZED ARRAY FACTOR
C         3. DIRECTIVITY (IN dB) USING NUMERICAL INTEGRATION OF THE 
C            ARRAY FACTOR
C         4. HALF-POWER BEAMWIDTH (IN DEGREES) USING AN ITERATIVE 
C            METHOD (FOR ALL MAXIMA THAT OCCUR IN THE PATTERN)
C
C     OPTION II.  PLANAR ARRAY (BROADSIDE UNIFORM)
C
C         ** ARRAY INPUT PARAMETERS:
C
C         1. NUMBER OF ARRAY ELEMENTS IN X-DIRECTION
C         2. SPACING BETWEEN THE ELEMENTS IN X-DIRECTION (IN
C            WAVELENGTHS)
C         3. NUMBER OF ARRAY ELEMENTS IN Y-DIRECTION
C         4. SPACING BETWEEN THE ELEMENTS IN Y-DIRECTION (IN
C            WAVELENGTHS)
C         5. MAXIMUM BEAM DIRECTION - ANGLE THETA (IN DEGREES)
C         6. MAXIMUM BEAM DIRECTION - ANGLE PHI (IN DEGREES)
C         7. THE PHI ANGLE (IN DEGREES) AT WHICH THE 2-D ANTENNA
C            PATTERN NEEDS TO BE EVALUATED (PHIEVAL IN DEG.)
C
C      NOTE: ONLY THE ELEVATION ANTENNA PATTERN IS EVALUATED.  THIS
C      ----- PATTERN RANGES FROM THETA=0 DEG. TO THETA=180 DEG. WHEREAS 
C            PHI REMAINS CONSTANT AT PHIEVAL.  IF THE PATTERN NEEDS TO
C            BE EVALUATED IN THE BACKSIDE REGION OF THE 2-D ARRAY,
C            THEN THE PROGRAM NEEDS TO BE RE-RUN FOR A NEW PHIEVAL
C            WHICH MUST BE EQUAL TO THE PREVIOUS VALUE PLUS 180 DEG.
C
C         ** PROGRAM OUTPUT:
C
C         1. NORMALIZED ARRAY FACTOR EVALUATED AT A GIVEN ANGLE 
C            (PHIEVAL)
C         2. UNIFORM PROGRESSIVE PHASE SHIFT IN X AND Y DIRECTIONS 
C            (IN DEGREES)
C         3. DIRECTIVITY (IN dB) USING NUMERICAL INTEGRATION OF THE 
C            ARRAY FACTOR
C         4. HALF-POWER BEAMWIDTH (IN DEGREES) FOR ALL MAXIMA THAT 
C            OCCUR IN THE ELEVATION PLANE OF THE 2-D ANTENNA PATTERN
C
C
C     ALL THE INPUT PARAMETERS ARE IN TERMS OF THE WAVELENGTH.
C     *****************************************************************
C     Written by: Anastasis C. Polycarpou, Arizona State University
C
C     *****************************************************************

      INTEGER DEVICE,ARRTYPE,ARRNAME,GEOMETRY,PHIEVAL,PHIMAX2,THMAX2
      CHARACTER*12 FILNAM
      REAL PI,K,TOL,ARRFAC(1801),AF2(1801),HPBW(25),THMAXBW(25),COEF(50)
      REAL ARRFAC2(181,361)
      COMMON /CONSTANTS/ PI,K,TOL

C     ********** CONSTANT DEFINITIONS **********
C
      PI=4.0*ATAN(1.0)
      K=2.*PI
      TOL=1.0E-6

C     ********** ARRAY FACTOR DATA **********
C     *** ALWAYS STORED IN AN OUTPUT FILE NAMED "ArrFac.dat" ***
C
      OPEN(UNIT=1,FILE='ArrFac.dat',STATUS='UNKNOWN') 

      
C     ********** CHOICE OF OUTPUT **********
C
      WRITE(6,10)
      READ(5,40,ERR=999) DEVICE

      IF(DEVICE .EQ. 1) THEN
         DEVICE=6

      ELSE IF(DEVICE .EQ. 2) THEN
         WRITE(6,20)
         READ(5,*,ERR=999) FILNAM
         OPEN(UNIT=DEVICE,FILE=FILNAM,STATUS='UNKNOWN')

      ELSE 
         WRITE(6,30)
         STOP

      ENDIF

C     *** CHOICE OF LINEAR OR PLANAR ARRAY
C
      WRITE(6,35)
      READ(5,40,ERR=999) GEOMETRY 

      IF(GEOMETRY .EQ. 1) THEN

C     *** CHOICE OF ARRAY TYPE (UNIFORM OR NONUNIFORM) ***
C
         WRITE(6,50)
         READ(5,40,ERR=999) ARRTYPE

         IF(ARRTYPE .EQ. 1) THEN
            WRITE(6,70)
            READ(5,40,ERR=999) ARRNAME


C        *** READ THE INPUT DATA ***
C
            WRITE(6,80)
            READ(5,*) NELEM

            WRITE(6,90)
            READ(5,*) D


            IF(ARRNAME .EQ. 1) THEN ! BROADSIDE ARRAY

C        *** ECHO INPUT INFORMATION ***
C
               WRITE(DEVICE,170)
               WRITE(DEVICE,180)
               WRITE(DEVICE,190)NELEM,D

C        *** ARRAY FACTOR CALCULATION ***
C
               DO I=1,1801
                  THETA=FLOAT(I-1)/10.*PI/180.
                  CALL BROADSIDE(THETA,NELEM,D,AF)
                  ARRFAC(I)=AF
               ENDDO

            ELSE IF(ARRNAME .EQ. 2) THEN ! ORDINARY END-FIRE ARRAY

C        *** INPUT ANGLE OF MAXIMUM (THETA = 0 OR 180 DEGREES)
C
               WRITE(6,100)
               READ(5,*) THMAX            


C        *** ECHO INPUT INFORMATION ***
C
               WRITE(DEVICE,170)
               WRITE(DEVICE,230)
               WRITE(DEVICE,240)NELEM,D,THMAX

C        *** ARRAY FACTOR CALCULATION ***
C
               DO I=1,1801
                  THETA=FLOAT(I-1)/10.*PI/180.
                  CALL ORDENDFIRE(THETA,NELEM,D,THMAX,AF)
                  ARRFAC(I)=AF
               ENDDO

            ELSE IF(ARRNAME .EQ. 3) THEN ! HANSEN-WOODYARD END-FIRE ARRAY

C        *** INPUT ANGLE OF MAXIMUM (THETA = 0 OR 180 DEGREES)
C
               WRITE(6,100)
               READ(5,*) THMAX            

C        *** ECHO INPUT INFORMATION ***
C
               WRITE(DEVICE,170)
               WRITE(DEVICE,250)
               WRITE(DEVICE,240)NELEM,D,THMAX

C        *** ARRAY FACTOR CALCULATION ***
C
               DO I=1,1801
                  THETA=FLOAT(I-1)/10.*PI/180.
                  CALL HWENDFIRE(THETA,NELEM,D,THMAX,AF)
                  ARRFAC(I)=AF
               ENDDO


            ELSE IF(ARRNAME .EQ. 4) THEN ! SCANNING ARRAY

C        *** INPUT ANGLE OF MAXIMUM (ANY ANGLE IN DEGREES)
C
               WRITE(6,110)
               READ(5,*) THMAX            

C        *** ECHO INPUT INFORMATION ***
C
               WRITE(DEVICE,170)
               WRITE(DEVICE,260)
               WRITE(DEVICE,240)NELEM,D,THMAX

C        *** ARRAY FACTOR CALCULATION ***
C
               DO I=1,1801
                  THETA=FLOAT(I-1)/10.*PI/180.
                  CALL SCANNING(THETA,NELEM,D,THMAX,AF)
                  ARRFAC(I)=AF
               ENDDO


            ENDIF



         ELSE IF(ARRTYPE .EQ. 2) THEN
            WRITE(6,140)
            READ(5,40,ERR=999) ARRNAME

C        *** READ THE INPUT DATA ***
C
            WRITE(6,80)
            READ(5,*) NELEM

            WRITE(6,90)
            READ(5,*) D


            IF(ARRNAME .EQ. 1) THEN ! BINOMIAL ARRAY

C        *** ECHO INPUT INFORMATION ***
C
               WRITE(DEVICE,170)
               WRITE(DEVICE,270)
               WRITE(DEVICE,190)NELEM,D

C        *** ARRAY FACTOR CALCULATION ***
C
               DO I=1,1801
                  THETA=FLOAT(I-1)/10.*PI/180.
                  CALL BINOMIAL(THETA,NELEM,D,AF,COEF,NCOEF)
                  ARRFAC(I)=AF
               ENDDO


            ELSE IF(ARRNAME .EQ. 2) THEN ! DOLPH-TSCHEBYSCHEFF ARRAY

               WRITE(6,150)
               READ(5,*) RDB    ! SIDE LOBE LEVEL IN dB

C        *** ECHO INPUT INFORMATION ***
C
               WRITE(DEVICE,170)
               WRITE(DEVICE,280)
               WRITE(DEVICE,290)NELEM,D,RDB

C        *** ARRAY FACTOR CALCULATION ***
C
               DO I=1,1801
                  THETA=FLOAT(I-1)/10.*PI/180.
                  CALL TSCEBY(THETA,NELEM,D,RDB,AF,COEF,NCOEF)
                  ARRFAC(I)=AF
               ENDDO

            ENDIF


         ELSE
            WRITE(6,60)
            STOP
         ENDIF


C     *** NORMALIZATION OF THE ARRAY FACTOR ***
C
         AFMAX=ABS(ARRFAC(1))
         DO I=2,1801
            IF(ABS(ARRFAC(I)) .GT. AFMAX) AFMAX = ABS(ARRFAC(I))
         ENDDO

         DO I=1,1801
            ARRFAC(I)=ARRFAC(I)/AFMAX
         ENDDO



C     *** OUTPUT THE ARRAY PATTERN DATA FROM 0 TO 360 DEGREES ***
C
         WRITE(1,11)
         DO I=1,1801,2
            IF (ABS(ARRFAC(I)) .LT. 1.0E-6) THEN
               WRITE(1,2) FLOAT(I-1)/10.,-120.
            ELSE
               WRITE(1,2) FLOAT(I-1)/10.,20.*ALOG10(ABS(ARRFAC(I)))
            ENDIF
         ENDDO
         DO I=1800,1,-2
            IF (ABS(ARRFAC(I)) .LT. 1.0E-6) THEN
               WRITE(1,2) 360.0-FLOAT(I-1)/10.,-120.
            ELSE
               WRITE(1,2) 360.0-FLOAT(I-1)/10.,20.*ALOG10(ABS(ARRFAC(I))
     &               )
            ENDIF
         ENDDO


C     *** CALCULATION OF THE DIRECTIVITY AND HPBW ***
C     
C     *** CALCULATION OF U = AF^2 ***

         DO I=1,1801
            AF2(I)=ARRFAC(I)**2
         ENDDO

         CALL DIRECTIVITY(DIRECT,AF2,0,180)
         CALL HPBWCALC(HPBW,AF2,IMAX,THMAXBW)


C     *** OUTPUT THE DIRECTIVITY AND HPBW FOR EACH MAXIMUM ***
C
         WRITE(DEVICE,210)
         WRITE(DEVICE,120) DIRECT
         WRITE(DEVICE,125) 10.0**(DIRECT/10.0)
         WRITE(DEVICE,200) IMAX
         DO I=1,IMAX
            WRITE(DEVICE,130) I,HPBW(I),THMAXBW(I)
         ENDDO

         IF(ARRTYPE .EQ. 2) THEN
C     *** PRINT OUT THE EXCITATION COEFFICIENTS ***
C
            WRITE(DEVICE,220)
            DO I=1,NCOEF
               WRITE(DEVICE,300) I,COEF(I)
            ENDDO
            WRITE(DEVICE,160)
            DO I=1,NCOEF
               WRITE(DEVICE,300) I,COEF(I)/COEF(NCOEF)
            ENDDO
         ENDIF


      ELSE IF(GEOMETRY .EQ. 2) THEN


C        *** READ THE INPUT DATA ***
C
         WRITE(6,320)
         READ(5,*) MX
         WRITE(6,340)
         READ(5,*) NY
         
         WRITE(6,330)
         READ(5,*) DX
         WRITE(6,350)
         READ(5,*) DY

         WRITE(6,360)
         READ(5,45,ERR=999) THMAX2
         WRITE(6,370)
         READ(5,45,ERR=999) PHIMAX2
         WRITE(6,375)
         READ(5,45,ERR=999) PHIEVAL


C        *** ECHO INPUT INFORMATION ***
C
         WRITE(DEVICE,380)
         WRITE(DEVICE,390)
         WRITE(DEVICE,400) MX,DX
         WRITE(DEVICE,410) NY,DY
         WRITE(DEVICE,420) THMAX2,PHIMAX2
         WRITE(DEVICE,430) PHIEVAL

C        *** PROGRESSIVE PHASE SHIFT CALCULATION ***
C
         DTOR=PI/180.
         BETAX=-K*DX*SIN(DTOR*THMAX2)*COS(DTOR*PHIMAX2)
         BETAY=-K*DY*SIN(DTOR*THMAX2)*SIN(DTOR*PHIMAX2)

C        *** ARRAY FACTOR CALCULATION ***
C
         DO I=1,181
            DO J=1,361
               THETA=FLOAT(I-1)*PI/180.
               PHI=FLOAT(J-1)*PI/180.
               CALL PLANAR(THETA,PHI,MX,NY,DX,DY,BETAX,BETAY,AF)
               ARRFAC2(I,J)=AF
            ENDDO
         ENDDO


         PHI=FLOAT(PHIMAX2)*PI/180. ! THE AF IS EVALUATED AT THE MAX BEAM 
         DO I=1,1801            ! ORIENTATION AND AT MORE POINTS
            THETA=FLOAT(I-1)/10.*PI/180.
            CALL PLANAR(THETA,PHI,MX,NY,DX,DY,BETAX,BETAY,AF)
            AF2(I)=AF**2        ! USED BY HPBWCALC SUBROUTINE
         ENDDO


C        *** OUTPUT THE ARRAY FACTOR AS IS EVALUATED AT PHI=PHIEVAL
C            AS WELL AS THE BACK SIDE (e.g. PHI=PHIEVAL+180) 
C
         WRITE(1,12)
         DO I=1,181
            IF (ABS(ARRFAC2(I,PHIEVAL+1)) .LT. 1.0E-6) THEN
               WRITE(1,2) FLOAT(I-1),-120.
            ELSE
               WRITE(1,2) FLOAT(I-1),20.*ALOG10(ABS(ARRFAC2(I,PHIEVAL+1)
     $               ))
            ENDIF
         ENDDO

         IF(PHIEVAL+180 .LE. 360) THEN
            PHIEVAL=PHIEVAL+180
         ELSE
            PHIEVAL=(PHIEVAL+180)-360
         ENDIF

         DO I=180,1,-1
            IF (ABS(ARRFAC2(I,PHIEVAL+1)) .LT. 1.0E-6) THEN
               WRITE(1,2) 360.0-FLOAT(I-1),-120.
            ELSE
               WRITE(1,2) 360.0-FLOAT(I-1),20.*ALOG10(ABS(ARRFAC2(I
     &               ,PHIEVAL+1)))
            ENDIF
         ENDDO


C        *** CALCULATION OF THE DIRECTIVITY OF THE 2-D PATTERN ***
C     
         CALL DIRECT_PLANAR(DIRECT,ARRFAC2,0,180,0,360)


C        *** CALCULATION OF THE HPBW IN THE ELEVATION PLANE ***
C     
         CALL HPBWCALC(HPBW,AF2,IMAX,THMAXBW)

         WRITE(DEVICE,210)

C        *** OUTPUT PROGRESSIVE PHASE SHIFT BETAX AND BETAY ***
C
         WRITE(DEVICE,128) BETAX*180.0/PI
         WRITE(DEVICE,129) BETAY*180.0/PI

C        *** OUTPUT THE DIRECTIVITY OF THE 2-D PATTERN ***
C            (Based only on the fields above the xy-plane)

         WRITE(DEVICE,126) DIRECT+10.0*ALOG10(0.5),(10.0**(DIRECT/10.0))
     &         /2.0

C        *** OUTPUT THE DIRECTIVITY OF THE 2-D PATTERN ***
C            (Based on the fields above and below the xy-plane)

         WRITE(DEVICE,127) DIRECT,10.0**(DIRECT/10.0)
         
C        *** OUTPUT THE HPBW FOR EACH MAXIMUM (ELEVATION PLANE) ***
C
         WRITE(DEVICE,440) IMAX
         DO I=1,IMAX
            WRITE(DEVICE,130) I,HPBW(I),THMAXBW(I)
         ENDDO


      ELSE
         WRITE(DEVICE,310)
      ENDIF

      IF(DEVICE .EQ. 6) THEN
         WRITE(DEVICE,5)
      ELSE IF(DEVICE .EQ. 2) THEN
         WRITE(6,5)
         WRITE(DEVICE,5)
      ENDIF


C     ********** FORMAT STATEMENTS **********
C
 2    FORMAT(f12.3,f16.5)

 5    FORMAT(/,3X,'*** NOTE:',/,7X,
     $      'THE NORMALIZED ARRAY FACTOR (in dB) IS STORED IN',/,7X
     &      ,'AN OUTPUT FILE CALLED  ............   ArrFac.dat',/,7X
     &      ,'================================================')

 10   FORMAT(3X,'DEVICE OPTION FOR OUTPUTING PROGRAM PARAMETERS',/,6X
     $      ,'OPTION (1): SCREEN',/,6X,'OPTION (2): OUTPUT FILE',//,3X
     $      ,'OUTPUT DEVICE = ',$)

 11   FORMAT('# LINEAR ARRAY',/,'#')
 12   FORMAT('# PLANAR ARRAY',/,'#')

 20   FORMAT(3X,'INPUT THE DESIRED OUTPUT FILNAME (single quotes) = ',$)


 30   FORMAT(/,3X,'*** ERROR ***',/,3X
     $      ,'OUTPUTING DEVICE NUMBER SHOULD BE EITHER 1 OR 2')

 35   FORMAT(/,3X,'LINEAR OR PLANAR ARRAY',/,6X
     $      ,'OPTION (1): LINEAR ARRAY',/,6X
     $      ,'OPTION (2): PLANAR ARRAY',//,3X,'OPTION NUMBER = ',$)

 40   FORMAT(I1)
 45   FORMAT(I3)

 50   FORMAT(/,3X,'UNIFORM OR NONUNIFORM ARRAY',/,6X
     $      ,'OPTION (1): UNIFORM ARRAY',/,6X
     $      ,'OPTION (2): NONUNIFORM ARRAY',//,3X,'OPTION NUMBER = ',$)

 60   FORMAT(/,3X,'*** ERROR ***',/,3X
     $      ,'THE ARRAY TYPE OPTION NUMBER SHOULD BE EITHER 1 OR 2')


 70   FORMAT(/,3X,'ARRAY NAMES',/,6X
     $      ,'OPTION (1): BROADSIDE (MAXIMUM ALONG THETA = 0 DEG.)',/,6X
     $      ,'OPTION (2): ORDINARY END-FIRE',/,6X
     $      ,'OPTION (3): HANSEN WOODYARD END-FIRE',/,6X
     $      ,'OPTION (4): SCANNING',//,3X,'OPTION NUMBER = ',$)


 80   FORMAT(/,3X,'NUMBER OF ELEMENTS = ',$)
 90   FORMAT(/,3X,'SPACING d BETWEEN THE ELEMENTS (IN WAVELENGTHS) = ',$
     $      )
 100  FORMAT(/,3X
     $      ,'ANGLE WHERE MAXIMUM OCCURS (THETA = 0 OR 180 DEG.) = ',$)
 110  FORMAT(/,3X
     $      ,'ANGLE WHERE MAXIMUM OCCURS (ANY ANGLE IN DEGREES) = ',$)
 120  FORMAT(3X,'DIRECTIVITY = ',F16.3,2X,'dB')
 125  FORMAT(3X,'DIRECTIVITY = ',F16.3,2X,'dimensionless',/)
 126  FORMAT(3X
     &      ,'DIRECTIVITY BASED ONLY ON THE FIELDS ABOVE THE XY-PLANE',
     &      /,3X,'D = ',F16.3,2X,'dB',/,3X,'D = ',F16.3,2X
     &      ,'dimensionless',/)
 127  FORMAT(3X
     & ,'DIRECTIVITY BASED ON THE FIELDS ABOVE AND BELOW THE XY-PLANE',
     &      /,3X,'D = ',F16.3,2X,'dB',/,3X,'D = ',F16.3,2X
     &      ,'dimensionless')

 128  FORMAT(3X,'PROGRESSIVE PHASE SHIFT IN X-DIRECTION',F16.3
     &      ,' DEGREES')
 129  FORMAT(3X,'PROGRESSIVE PHASE SHIFT IN Y-DIRECTION',F16.3
     &      ,' DEGREES',/)

 130  FORMAT(3X,'HPBW FOR MAXIMUM #',1X,I2,2X,'= ',F6.1,2X,'DEGREES'
     $      ,3X,'---> THMAX =  ',F6.1,2X,'DEGREES')
 140  FORMAT(/,3X,'ARRAY NAMES',/,6X
     $      ,'OPTION (1): BINOMIAL',/,6X
     $      ,'OPTION (2): DOLPH-TSCHEBYSHEFF',//,3X,'OPTION NUMBER = ',$
     $      )
 150  FORMAT(/,3X,'SIDE LOBE LEVEL (IN dB) = ',$)
 160  FORMAT(/,3X,'NORMALIZED EXCITATION COEFFICIENTS',/)
 170  FORMAT(/,3X
     &      ,'******************************************************',/
     &      ,3X,'PROGRAM OUTPUT',/,3X
     &      ,'******************************************************',/)
 180  FORMAT(3X,'INPUT SPECIFICATIONS',/,3X
     $      ,'--------------------',/,3X,'BROADSIDE ARRAY')
 190  FORMAT(/,3X,'NUMBER OF ARRAY ELEMENTS = ',I2,/,3X
     $      ,'SPACING BETWEEN THE ELEMENTS (IN WAVELENGTHS) = ',F8.3)
 200  FORMAT(3X,'NUMBER OF MAXIMA BETWEEN 0 AND 180 DEGREES = ',I2)
 210  FORMAT(/,3X,'OUTPUT CHARACTERISTICS OF THE ARRAY',/,3X
     $      ,'-----------------------------------')
 220  FORMAT(/,3X,'EXCITATION COEFFICIENTS FOR THE ARRAY DESIGN',/)
 230  FORMAT(3X,'INPUT SPECIFICATIONS',/,3X
     $      ,'--------------------',/,3X,'ORDINARY END-FIRE ARRAY')
 240  FORMAT(/,3X,'NUMBER OF ARRAY ELEMENTS = ',I2,/,3X
     $      ,'SPACING BETWEEN THE ELEMENTS (IN WAVELENGTHS) = ',F8.3
     $      ,/,3X,'MAXIMUM NEEDS TO OCCUR AT',F8.3)
 250  FORMAT(3X,'INPUT SPECIFICATIONS',/,3X
     $      ,'--------------------',/,3X
     $      ,'HANSEN WOODYARD END-FIRE ARRAY')
 260  FORMAT(3X,'INPUT SPECIFICATIONS',/,3X
     $      ,'--------------------',/,3X,'SCANNING ARRAY')
 270  FORMAT(3                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                PTION NUMBER SHOULD BE EITHER 1 OR 2')

 320  FORMAT(/,3X,'NUMBER OF ELEMENTS IN THE X-DIRECTION = ',$)
 330  FORMAT(/,3X,'SPACING dx BETWEEN THE ELEMENTS (IN WAVELENGTHS) = '
     $      ,$)

 340  FORMAT(/,3X,'NUMBER OF ELEMENTS IN THE Y-DIRECTION = ',$)
 350  FORMAT(/,3X,'SPACING dy BETWEEN THE ELEMENTS (IN WAVELENGTHS) = '
     $      ,$)

 360  FORMAT(/,3X,'MAXIMUM BEAM DIRECTION - ANGLE THETA',1X
     $      ,'(IN DEGREES - INTEGER #) = '
     $      ,$)
 370  FORMAT(/,3X,'MAXIMUM BEAM DIRECTION - ANGLE PHI',1X
     $      ,'(IN DEGREES - INTEGER #) = ',$)

 375  FORMAT(/,3X
     $      ,'THE PATTERN IS EVALUATED AT AN ANGLE',1X
     $      ,'PHI (IN DEGREES - INTEGER #) = ',$)

 380  FORMAT(/,3X
     $      ,'******************************************************',/
     $      ,3X,'PROGRAM OUTPUT',/,3X
     $      ,'******************************************************',/)
 390  FORMAT(3X,'INPUT SPECIFICATIONS',/,3X
     $      ,'--------------------',/,3X,'PLANAR ARRAY')

 400  FORMAT(/,3X,'NUMBER OF ARRAY ELEMENTS IN X-DIRECTION =
     $      ',I2,/,3X,'SPACING BETWEEN THE ELEMENTS IN X-DIRECTION', 
     $      ' (IN WAVELENGTHS) = ',F8.3)

 410  FORMAT(/,3X,'NUMBER OF ARRAY ELEMENTS IN Y-DIRECTION =
     $      ',I2,/,3X,'SPACING BETWEEN THE ELEMENTS IN Y-DIRECTION', 
     $      ' (IN WAVELENGTHS) = ',F8.3)

 420  FORMAT(/,3X,'MAXIMUM BEAM DIRECTION - THETA (IN DEGREES) = ',I3,
     $      /,3X,'MAXIMUM BEAM DIRECTION - PHI (IN DEGREES) = ',I3)

 430  FORMAT(/,3X
     $      ,'THE 2-D ANTENNA PATTERN IS EVALUATED AT AN ANGLE',1X
     $      ,'PHI (IN DEGREES) = ',I3)

 440  FORMAT(/,3X,'ELEVATION PLANE: NUMBER OF MAXIMA BETWEEN 0 AND 180'
     &      ,1X,'DEGREES = ',I2)

         STOP  ! END OF THE MAIN PROGRAM  
C     

C     ********** ERROR CONDITIONS **********
C
 999     WRITE(6,1000)
 1000    FORMAT(/,3X,'***** ERROR *****',/,3X
     $         ,'INPUT DATA ARE NOT OF THE RIGHT FORMAT',/)

      END



C     ******************************************************************
      SUBROUTINE BROADSIDE(THETA,NELEM,D,AF)
C     ******************************************************************

      INTEGER NELEM
      REAL PI,TOL,K,D,AF,BETA,PSI,FLNE
      COMMON /CONSTANTS/ PI,K,TOL


      BETA = 0.0  ! PROGRESSIVE PHASE SHIFT BETWEEN THE ELEMENTS
      PSI=K*D*COS(THETA)+BETA  ! TOTAL PHASE

C     *** CALCULATION OF THE ARRAY FACTOR ***
C
      FLNE=FLOAT(NELEM)
      IF(ABS(SIN(PSI/2.)) .LT. TOL) THEN
         AF=1.0
      ELSE
         AF=SIN(FLNE*PSI/2.)/(FLNE*SIN(PSI/2.))
      ENDIF

      RETURN
      END



C     ******************************************************************
      SUBROUTINE ORDENDFIRE(THETA,NELEM,D,THMAX,AF)
C     ******************************************************************

      INTEGER NELEM
      REAL PI,K,TOL,D,AF,BETA,PSI,FLNE
      COMMON /CONSTANTS/ PI,K,TOL

      IF(ABS(THMAX) .LT. TOL) THEN
         BETA = -K*D            ! PROGRESSIVE PHASE SHIFT BETWEEN THE
                                ! ELEMENTS
      ELSE IF(ABS(THMAX-180.0) .LT. TOL) THEN
         BETA = K*D             ! PROGRESSIVE PHASE SHIFT BETWEEN THE
                                ! ELEMENTS
      ENDIF
      PSI=K*D*COS(THETA)+BETA   ! TOTAL PHASE

C     *** CALCULATION OF THE ARRAY FACTOR ***
C
      FLNE=FLOAT(NELEM)
      IF(ABS(SIN(PSI/2.)) .LT. TOL) THEN
         AF=1.0
      ELSE
         AF=SIN(FLNE*PSI/2.)/(FLNE*SIN(PSI/2.))
      ENDIF

      RETURN
      END


C     ******************************************************************
      SUBROUTINE HWENDFIRE(THETA,NELEM,D,THMAX,AF)
C     ******************************************************************

      INTEGER NELEM
      REAL PI,K,TOL,D,AF,BETA,PSI,FLNE
      COMMON /CONSTANTS/ PI,K,TOL

      IF(ABS(THMAX) .LT. TOL) THEN
         BETA = -(K*D+PI/FLOAT(NELEM)) ! PROGRESSIVE PHASE SHIFT BETWEEN
                                       ! THE ELEMENTS
      ELSE IF(ABS(THMAX-180.) .LT. TOL) THEN
         BETA = K*D+PI/FLOAT(NELEM)    ! PROGRESSIVE PHASE SHIFT BETWEEN
                                       ! THE ELEMENTS
      ENDIF
         PSI=K*D*COS(THETA)+BETA ! TOTAL PHASE

C     *** CALCULATION OF THE ARRAY FACTOR ***
C
      FLNE=FLOAT(NELEM)
      IF(ABS(SIN(PSI/2.)) .LT. TOL) THEN
         AF=1.0
      ELSE
         AF=SIN(FLNE*PSI/2.)/(FLNE*SIN(PSI/2.))
      ENDIF

      RETURN
      END



C     ******************************************************************
      SUBROUTINE SCANNING(THETA,NELEM,D,THMAX,AF)
C     ******************************************************************

      INTEGER NELEM
      REAL PI,K,TOL,D,AF,BETA,PSI,FLNE
      COMMON /CONSTANTS/ PI,K,TOL

      BETA = -K*D*COS(THMAX*PI/180.) ! PROGRESSIVE PHASE SHIFT BETWEEN
                                     ! THE ELEMENTS
      PSI=K*D*COS(THETA)+BETA        ! TOTAL PHASE

C     *** CALCULATION OF THE ARRAY FACTOR ***
C
      FLNE=FLOAT(NELEM)
      IF(ABS(SIN(PSI/2.)) .LT. TOL) THEN
         AF=1.0
      ELSE
         AF=SIN(FLNE*PSI/2.)/(FLNE*SIN(PSI/2.))
      ENDIF

      RETURN
      END



C     ******************************************************************
      SUBROUTINE DIRECTIVITY(DR,AF2,TL,TU)
C     ******************************************************************
C
      INTEGER TL,TU
      REAL AF2(1801)
      PI=4.0*ATAN(1.0)

      PRAD=0.0
      UMAX=0.0
C
      DTHETA=(PI/180.0)/10.

      DO I=TL,TU*10
         XI=FLOAT(I)/10.*PI/180.0
         
         IF(AF2(I+1) .GT. UMAX) UMAX=AF2(I+1)
         UA=DTHETA*AF2(I+1)*SIN(XI)
         PRAD=PRAD+UA
      ENDDO
      PRAD=2.*PI*PRAD

      DR=4.0*PI*UMAX/PRAD
      DR=10.*ALOG10(DR)

      RETURN
      END




C     ******************************************************************
      SUBROUTINE HPBWCALC(HPBW,AF2,IMAX,THMAX)
C     ******************************************************************
C
      INTEGER IMAX
      REAL AF2(1801),X(2),Y(2),ROOT(50),HPBW(25),THMAX(25)
      PI=4.0*ATAN(1.0)
      TOL=0.1         ! One tenth of the maximum
                      ! Everything above 0.9 is considered a MAX

      IMAX=0
      K=0
      DO I=1,1801

         IF(ABS(AF2(I)-1.) .LT. TOL .AND. (K+2)/2 .EQ. IMAX+1) THEN
            IMAX=IMAX+1
            THMAX(IMAX)=FLOAT(I-1)/10.
         ENDIF


         IF(I .GT. 1 .AND. ABS(AF2(I)-1.) .LT. TOL .AND. AF2(I) .GT.
     &         AF2(I-1) .AND. K .NE. 0)THEN
            THMAX(IMAX)=FLOAT(I-1)/10.
         ENDIF

         IF(I .GT. 1)THEN
            Y(1)=AF2(I)-0.5
            Y(2)=AF2(I-1)-0.5
            X(1)=(FLOAT(I-1))/10.
            X(2)=(FLOAT(I-2))/10.
            SIGN=Y(1)*Y(2)
            IF(SIGN .LT. 0.0) THEN
               K=K+1
               ROOT(K)=X(2)-Y(2)*(X(2)-X(1))/(Y(2)-Y(1))
               IF(K .GE. 2 .AND. Y(2) .GT. Y(1)) THEN
                  HPBW(IMAX)=ROOT(K)-ROOT(K-1)
               ELSE IF(K .EQ. 1 .AND. Y(2) .GT. Y(1))THEN
                  HPBW(IMAX)=2.*ROOT(K)
               ENDIF
            ENDIF
         ENDIF
      ENDDO

      IF(THMAX(IMAX) .GT. ROOT(K)) THEN
         HPBW(IMAX)=2.*(180.-ROOT(K))
      ENDIF


      RETURN
      END



C     ******************************************************************
      SUBROUTINE BINOMIAL(THETA,NELEM,D,AF,COEF,NCOEF)
C     ******************************************************************

      INTEGER NELEM,NCOEF
      REAL PI,TOL,K,D,AF,FLNE,THETA,COEF(50),U
      LOGICAL EVEN,ODD
      COMMON /CONSTANTS/ PI,K,TOL

C     *** EXCITATION COEFFICIENTS ***
C
      IF(2*(NELEM/2) .EQ. NELEM) THEN  ! EVEN # OF ELEMENTS
         NCOEF=NELEM/2
         EVEN=.TRUE.
         ODD=.FALSE.
      ELSE                             ! ODD # OF ELEMENTS
         NCOEF=(NELEM+1)/2
         EVEN=.FALSE.
         ODD=.TRUE.
      ENDIF

      FLNE=FLOAT(NELEM)

      DO I=1,NCOEF
         COEF(I)=1.0
         DO J=1,NCOEF-I
            COEF(I)=COEF(I)*(FLNE-FLOAT(J))/FLOAT(J)
         ENDDO
      ENDDO
      IF(ODD) COEF(1)=COEF(1)/2.0

C     *** CALCULATION OF THE ARRAY FACTOR ***
C
      U=PI*D*COS(THETA)
      IF(EVEN) THEN
         AF=0.0
         DO I=1,NCOEF
            AF=AF+COEF(I)*COS((2*I-1)*U)
         ENDDO
      ELSE IF(ODD) THEN
         AF=0.0
         DO I=1,NCOEF
            AF=AF+COEF(I)*COS(2*(I-1)*U)
         ENDDO
      ENDIF

      RETURN
      END



C     ******************************************************************
      SUBROUTINE TSCEBY(THETA,NELEM,D,RDB,AF,COEF,NCOEF)
C     ******************************************************************

      INTEGER NELEM,NCOEF,M,EN
      REAL PI,TOL,K,D,AF,RDB,THETA,COEF(50),U,RO,ZO
      LOGICAL EVEN,ODD
      COMMON /CONSTANTS/ PI,K,TOL

C     *** CALCULATE THE VOLTAGE RATIO ***
C
      RO=10.**(RDB/20.)

C     *** CALCULATE Z0 ***
C
      P=FLOAT(NELEM-1)
      ZO=0.5*((RO+SQRT(RO**2-1.))**(1./P)+(RO-SQRT(RO**2-1.))**(1./P))

C     *** EXCITATION COEFFICIENTS ***
C
      IF(2*(NELEM/2) .EQ. NELEM) THEN  ! EVEN # OF ELEMENTS
         NCOEF=NELEM/2
         M=NCOEF
         EVEN=.TRUE.
         ODD=.FALSE.
         DO I=1,M
            COEF(I)=0.0
            DO J=I,M
               COEF(I)=COEF(I)+(-1)**(M-J)*ZO**(2*J-1)*FACT(J+M-2)*(2
     &               *M-1)/(FACT(J-I)*FACT(J+I-1)*FACT(M-J))
            ENDDO
         ENDDO
      ELSE                             ! ODD # OF ELEMENTS
         NCOEF=(NELEM+1)/2
         M=NCOEF-1
         EVEN=.FALSE.
         ODD=.TRUE.
         DO I=1,M+1
            COEF(I)=0.0
            DO J=I,M+1
               IF(I .EQ. 1) THEN
                  EN=2
               ELSE
                  EN=1
               ENDIF
               COEF(I)=COEF(I)+(-1)**(M-J+1)*ZO**(2*(J-1))*FACT(J+M-2)*2
     &               *M/(EN*FACT(J-I)*FACT(J+I-2)*FACT(M-J+1))
            ENDDO
         ENDDO
      ENDIF

      
C     *** CALCULATION OF THE ARRAY FACTOR ***
C
      U=PI*D*COS(THETA)
      IF(EVEN) THEN
         AF=0.0
         DO I=1,NCOEF
            AF=AF+COEF(I)*COS((2*I-1)*U)
         ENDDO
      ELSE IF(ODD) THEN
         AF=0.0
         DO I=1,NCOEF
            AF=AF+COEF(I)*COS(2*(I-1)*U)
         ENDDO
      ENDIF

      RETURN
      END


C     ******************************************************************
      REAL FUNCTION FACT(IARG)
C     ******************************************************************
      
      FACT=1.0
          DO J=1,IARG
            FACT=FLOAT(J)*FACT
          ENDDO
      RETURN
      END



C     ******************************************************************
      SUBROUTINE PLANAR(THETA,PHI,MX,NY,DX,DY,BETAX,BETAY,AF)
C     ******************************************************************

      INTEGER MX,NY
      REAL PI,TOL,K,DX,DY,BETAX,BETAY,PSIX,PSIY,FLMX,FLNY
      REAL THETA,PHI,AF,AFX,AFY
      COMMON /CONSTANTS/ PI,K,TOL

      PSIX=K*DX*SIN(THETA)*COS(PHI)+BETAX ! TOTAL PHASE IN X-DIRECTION
      PSIY=K*DY*SIN(THETA)*SIN(PHI)+BETAY ! TOTAL PHASE IN Y-DIRECTION

C     *** CALCULATION OF THE ARRAY FACTOR ***
C
      FLMX=FLOAT(MX)
      FLNY=FLOAT(NY)
 
      IF(ABS(SIN(PSIX/2.)) .LT. TOL) THEN
         AFX=1.0
      ELSE
         AFX=SIN(FLMX*PSIX/2.)/(FLMX*SIN(PSIX/2.))
      ENDIF
 
      IF(ABS(SIN(PSIY/2.)) .LT. TOL) THEN
         AFY=1.0
      ELSE
         AFY=SIN(FLNY*PSIY/2.)/(FLNY*SIN(PSIY/2.))
      ENDIF

      AF=AFX*AFY

      RETURN
      END




C     ******************************************************************
      SUBROUTINE DIRECT_PLANAR(DR,ARRFAC2,TL,TU,PL,PU)
C     ******************************************************************
C
      INTEGER TL,TU,PL,PU
      REAL ARRFAC2(181,361)
      PI=4.0*ATAN(1.0)

      PRAD=0.0
      UMAX=0.0
C
      DTHETA=(PI/180.0)
      DPHI=(PI/180.0)

      DO J=PL,PU
         DO I=TL,TU
            XI=FLOAT(I)*PI/180.0
            XJ=FLOAT(J)*PI/180.0

            IF(ARRFAC2(I+1,J+1)**2 .GT. UMAX) UMAX=ARRFAC2(I+1,J+1)**2
            UA=DTHETA*DPHI*ARRFAC2(I+1,J+1)**2*SIN(XI)
            PRAD=PRAD+UA
         ENDDO
      ENDDO

      DR=4.0*PI*UMAX/PRAD
      DR=10.*ALOG10(DR)

      RETURN
      END

