/*  XMMS plugin for dymanic noise filtering
 *  Kills the background tape hiss from old recordings.
 *  Copyright (C) 2001 by Rowland
 *  Structure cribbed from echo_plugin by Carl van Schaik,
 *  but the functional code is all mine.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Dynamic hiss filter plugin for XMMS.
   This version uses a rather simpleminded filter aglorithm,
   which does not have a sharp cutoff.  In a future version I may
   try something more sophisticated.  This basically works,
   but there is noticable "pumping" of the residual hiss.
*/

#include "config.h"

#include <xmms/plugin.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include "libxmms/configfile.h"
#include "filter.h"

static void init(void);
static void cleanup(void);
static int mod_samples(gpointer * d, gint length, AFormat afmt, gint srate, gint nch);

#define MAX_SRATE 50000
#define MAX_CHANNELS 2
#define BYTES_PS 2


gint filter_threshold= 0;

EffectPlugin filter_ep = /* xmms/plugin.h */
{
  NULL,
  NULL,
  "Filter Plugin " VERSION,
  init,
  cleanup,
  filter_about,
  filter_configure, /* gui.c */
  mod_samples,
  NULL /* query_format */
};



#define MAXPREVS 20
#define BANDS 8  /* The more bands, the better, up to a point.  
		    The quality of the filter algorithm imposes a limit */
int oneside= 0;
gint32 threshold[BANDS];
int filtcount[BANDS-1] = {10, 8, 6, 4, 3, 2, 1}; /* number of voxels back for filter */

/* one per channel per band... */
typedef struct {
#define SQUELCHMULT 11000
  gint32 prevs[MAXPREVS];
  /*   ...Previous voxels, averaged in to filter apart bands.
       One between each pair of adjacent bands */
  unsigned long squelchgain; 
  /* ...*SQUELCHMULT allows less obstrusive squelch switching */
  unsigned char squelching;
} FilterABand;

/* one per channel... */
typedef struct {
  FilterABand B[BANDS];
} FilterAChannel;

FilterAChannel Channel[2];


EffectPlugin *get_eplugin_info(void)
{
	return &filter_ep;
}

static void init(void)
{
  int j;
  ConfigFile *cfg;
  
  if (sizeof(short) != sizeof(gint16))
    abort();
  cfg = xmms_cfg_open_default_file();
  xmms_cfg_read_int(cfg, "filter_plugin", "threshold", &filter_threshold);
  xmms_cfg_free(cfg);
  memset(&Channel[0], 0, sizeof(Channel[0]));
  memset(&Channel[1], 0, sizeof(Channel[1]));
  filter_set_threshold(filter_threshold);
}

static void cleanup(void)
{
}


/* here from gui.c */
void filter_set_threshold(int athreshold) {
  int band;

  /* take the exponent, more or less.. */
  gint32 ethreshold= 1;  ethreshold <<= (athreshold/2);
  if (athreshold& 1)  ethreshold= (ethreshold*3)/2;
  printf("\nfilter_set_threshold(%i)\n", ethreshold);
  threshold[0]= 0;
  threshold[1]= ethreshold / 3;
  threshold[2]= ethreshold / 2;
  for (band= 3; band< BANDS; band++)
    threshold[band]= ethreshold;
}

/* here from oss_write_audio in audio.c */
static int mod_samples(gpointer * d, gint length, 
		       AFormat afmt, gint srate, gint nch)
{
  gint i;
  gint16 *data = (gint16 *) * d;
  static gint old_srate, old_nch;

  /* miminum squelchable voxels in a row before we squelch... */
#define PRECISION 256
#define HEADROOM 128

  /* don't support these formats... */  
  if (!(afmt == FMT_S16_NE || /* xmms/plugin.h */
	(afmt == FMT_S16_LE && G_BYTE_ORDER == G_LITTLE_ENDIAN) ||
	(afmt == FMT_S16_BE && G_BYTE_ORDER == G_BIG_ENDIAN)))
    return length;
  
  /* do the work... */ 
  for (i = 0; i < length / BYTES_PS; i+= 2)  {
    int channel, band;
    for (channel= 0; channel< 2; channel++) {
      gint32 in, data_, inf[BANDS];
      /* ...input voxels are 16 bits, but
	 we'll use 32 to avoid rounding errors */
      in = data[i+channel]* PRECISION;   /* for better precision */

      data_= 0;
      if (channel && oneside) { /* do only one channel.  good for testing. */
	/* do nothing */
      }  else {
	register int j;
	
	/* filter each frequency band... */
	for (band=0; band< BANDS; band++) {
	  gint32 out_; /* output voxel for this band */
	  FilterABand* pB= &Channel[channel].B[band];
	  register gint32 inf_;
	  gint32 threshold_= threshold[band]* PRECISION;
	  gint32* pprevs= pB->prevs;
	  unsigned long squelchgain= pB->squelchgain;
	  unsigned long safesquelch;
	  if (band< (BANDS-1)) {
	    register gint32 in_;
	    register gint32 * pprev_;  int j0;
	    inf_= inf[band];
	    in_ = band? inf_ : in; /* previous filter's output */
	    /* ...voxel input to the filter */
	    inf_= in_; /* the lowpass part */
	    j0= MAXPREVS-filtcount[band];  pprev_= &pprevs[j0];
	    for (j= j0; j<MAXPREVS; j++)
	      { inf_+= *pprev_;  pprev_++; }
	    inf_ /= filtcount[band]+1;
	    inf[band]= inf_;
	    inf[band+1]= in_ - inf_;  /* the highpass part */
	  }

          inf_ = inf[band];
	  out_= inf_;
	  if (!pB->squelching) {
	    if (squelchgain< SQUELCHMULT)
	      squelchgain+= SQUELCHMULT/256; /* come out of squelch quickly
					       so we don't blunt attacks */
	    if ((inf_< threshold_) && (inf_> -threshold_)) {
	      /* squelchable voxel */
	      pB->squelching= 1;
	      if (squelchgain) squelchgain--;
	    }
	  }
	  if (pB->squelching) {
	    if (squelchgain) squelchgain--;
	    if ((inf_> threshold_) || (inf_< -threshold_)) {
	      pB->squelching= 0;  /* kick out of squelch mode */
	    }	    
	  }

	  safesquelch= squelchgain*HEADROOM;  safesquelch/= SQUELCHMULT;
	  out_ *= safesquelch;  out_/= HEADROOM;
	    
	  pB->squelchgain= squelchgain;
	  data_ += out_;  /* reassemble the bands here, after filtering */

	  if (band< (BANDS-1)) {
	    /* shift the prevs for next time... */
	    register gint32 * pprev_, * pprev_1;  int j0;
	    j0= MAXPREVS-filtcount[band];
	    pprev_= &pprevs[j0]; pprev_1= pprev_+1;
	    for (j= j0; j< MAXPREVS-1; j++)
	      { *pprev_= *pprev_1;  pprev_++; pprev_1++; }
	    pprevs[MAXPREVS-1]= band? inf[band-1] : in;
	  }
	}
	
	data[i+ channel] = data_ / PRECISION;  /* reverse *PRECISION above */
      
      }
    }
    }
  
  return length;
}
