/*  XMMS plugin for dymanic noise filtering
 *  Kills the background tape hiss from old recordings.
 *  Copyright (C) 2001 by Rowland
 *  Structure cribbed from echo_plugin by Carl van Schaik,
 *  but the functional code is all mine.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Dynamic hiss filter plugin for XMMS.
   This version uses a rather simpleminded filter aglorithm,
   which does not have a sharp cutoff.  In a future version I may
   try something more sophisticated.  This basically works,
   but there is noticable "pumping" of the residual hiss.
*/

#include "config.h"

#include <xmms/plugin.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <math.h>
#include "libxmms/configfile.h"
#include "filter.h"

static void init(void);
static void cleanup(void);
static int mod_samples(gpointer * d, gint length, AFormat afmt, gint srate, gint nch);

#define MAX_SRATE 50000
#define MAX_CHANNELS 3  /* left, right, and difference */
#define BYTES_PS 2


gint filter_threshold= 0;

gint filter_lp_mode= 0;  /* nonzero to filter average and diffrence bands, better for LPs */
gint filter_softsquelch= 0;

EffectPlugin filter_ep = /* xmms/plugin.h */
{
  NULL,
  NULL,
  "Filter Plugin " PVERSION " for XMMS " VERSION,
  init,
  cleanup,
  filter_about,
  filter_configure, /* gui.c */
  mod_samples,
  NULL /* query_format */
};


#define MAXBANDS 8  /* The more bands, the better, up to a point.  
		    The quality of the filter algorithm imposes a limit */
#define MAXMAXPREVS 30 /* the most voxels any filter will ever look back */
int oneside= 0;
gint32 Threshold8[MAXBANDS];
int Filtcount8[8-1] = {10, 8, 6, 4, 3, 2, 1}; /* number of voxels back for filter */

/* update these whenever we change to a different number of bands... */
int Maxprevs= 20;
int Bands= 8;
int* Filtcount= Filtcount8; /* number of voxels back for filter */
gint32* Threshold= Threshold8;



/* one per channel per band... */
typedef struct {
#define SQUELCHMULT 11000
  gint32 prevs[MAXMAXPREVS];
  /*   ...Previous voxels, averaged in to filter apart bands.
       One between each pair of adjacent bands */
  unsigned long squelchgain;
  unsigned long squelchgaintarget; /* smoother squelch cutoff */
  /* ...*SQUELCHMULT allows less obtrusive squelch switching */
  unsigned long absinf; /* used in softsquelch */
  unsigned char squelching;
} FilterABand;

/* one per channel... */
typedef struct {
  FilterABand B[MAXBANDS];
} FilterAChannel;

FilterAChannel Channel[MAX_CHANNELS];


EffectPlugin *get_eplugin_info(void)  {
  return &filter_ep;
}

static void init(void)
{
  int j;
  ConfigFile *cfg;
  
  if (sizeof(short) != sizeof(gint16))
    abort();
  cfg = xmms_cfg_open_default_file();
  xmms_cfg_read_int(cfg, "filter_plugin", "threshold", &filter_threshold);
  xmms_cfg_free(cfg);
  memset(&Channel[0], 0, sizeof(Channel[0]));
  memset(&Channel[1], 0, sizeof(Channel[1]));
  filter_set_threshold(filter_threshold);
}

static void cleanup(void)  {
}


/* here from gui.c */
void filter_set_threshold(int athreshold) {
  int band;

  /* take the exponent, more or less.. */
  gint32 ethreshold= 0;

  if (athreshold) {
    ethreshold= 100;
    while (athreshold--) {
      ethreshold*= 8;  ethreshold/= 5;
    }
    ethreshold/= 70;
  }
  printf("\nfilter_set_threshold(%i) softsquelch=%i\n",
	 ethreshold, filter_softsquelch);
  Threshold8[0]= 0;
  Threshold8[1]= ethreshold / 3;
  Threshold8[2]= ethreshold / 2;
  for (band= 3; band< Bands; band++)
    Threshold8[band]= ethreshold;
}



static int mod_sample(gint32 voxel, int channel);

/* here from oss_write_audio in audio.c */
static int mod_samples(gpointer * d, gint length, /* typically 4096 */
		       AFormat afmt, gint srate, gint nch)
{
  gint i;
  gint16 *data = (gint16 *) * d;
  static gint old_srate, old_nch;
#ifdef LP_MODE
  int lp_mode= filter_lp_mode;
#endif

  /* don't support these formats... */  
  if (!(afmt == FMT_S16_NE || /* xmms/plugin.h */
	(afmt == FMT_S16_LE && G_BYTE_ORDER == G_LITTLE_ENDIAN) ||
	(afmt == FMT_S16_BE && G_BYTE_ORDER == G_BIG_ENDIAN)))
    return length;
  
  /* do the work... */ 
  for (i = 0; i < length / BYTES_PS; i+= 2)  {
    int channel;
  #ifdef LP_MODE
    if (lp_mode) {
      register gint16 data0= data[i+0];
      register gint16 data1= data[i+1];
      gint32 datamono= data0; /* the mono component */
      gint32 datadiff;
      datadiff = mod_sample(data1- data0, 2); /* the difference component */
      data0= datamono- datadiff/2;
      data[i+0] = data0;
      data[i+1]= data0+ datadiff;
    }
  #endif 
    for (channel= 0; channel< 2; channel++) {
      if (channel && oneside) { /* do only one channel.  good for testing. */
	/* do nothing */
      }  else data[i+ channel]= mod_sample(data[i+channel], channel);
    }
  }
 
  return length;
}


static int mod_sample(gint32 voxel, int channel) {
  /* returns modified voxel */
  
#define PRECISION 256
#define HEADROOM 128
  gint32 in= voxel, data_, band, inf[Bands];
  /* ...input voxels are 16 bits, but
     we'll use 32 to avoid rounding errors */
  in*= PRECISION;   /* for better precision */
  data_= 0;
    
  /* filter each frequency band... */
  for (band=0; band< Bands; band++)  {
    gint32 out_; /* output voxel for this band */
    FilterABand* pB= &Channel[channel].B[band];
    register gint32 inf_;
    gint32* pprevs= pB->prevs;

    if (band< (Bands-1)) {
      register gint32 in_;
      register int j;
      register gint32 * pprev_;  int j0;
      inf_= inf[band];
      in_ = band? inf_ : in; /* previous filter's output */
      /* ...voxel input to the filter */
      inf_= in_; /* the lowpass part */
      j0= Maxprevs-Filtcount[band];  pprev_= &pprevs[j0];
      for (j= j0; j<Maxprevs; j++)
	{ inf_+= *pprev_;  pprev_++; }
      inf_ /= Filtcount[band]+1;
      inf[band]= inf_;
      inf[band+1]= in_ - inf_;  /* the highpass part */
    }

    { /* now we've got the band split out.  let's process it... */    
      gint softsquelch= filter_softsquelch;
      unsigned long squelchgain= pB->squelchgain;
      unsigned long squelchgaintarget= pB->squelchgaintarget;
      unsigned long safesquelch;
      gint32 threshold_= Threshold[band]* PRECISION;
    #ifdef SOFTSQUELCH 
      if (softsquelch) threshold_*= 2;
    #endif
    #ifdef LP_MODE
      if (channel== 2)  threshold_*= 2;  /* it's the difference channel */
    #endif
      
      inf_ = inf[band];
      out_= inf_;
      if (threshold_) {
	long absinf= abs(inf_);
	
	if (!pB->squelching) {
	  if (squelchgain< SQUELCHMULT) /* coming out of squelch */
	    squelchgain+= SQUELCHMULT/256;
	  if (squelchgain> SQUELCHMULT) /* coming out of squelch */
	    squelchgain= SQUELCHMULT;
	  /* ...come out of squelch quickly so we don't blunt attacks */
	  
	  if (absinf< threshold_) {
	    /* squelchable voxel. start squelching. */
	    pB->squelching= 1;
          #ifdef SOFTSQUELCH        
	    squelchgaintarget= 0; /* so will get computed below */
          #endif
	  }
	}
	if (pB->squelching) {
        #ifdef SOFTSQUELCH 
	  if (softsquelch) {
	    if (absinf> pB->absinf) {
	      /* better guess for squelch target.. */
	      long maybesquelchgaintarget=
		(long)(SQUELCHMULT*absinf)/threshold_;
	      maybesquelchgaintarget= maybesquelchgaintarget*absinf/threshold_;
	      if (maybesquelchgaintarget> SQUELCHMULT)
		maybesquelchgaintarget= SQUELCHMULT;
	      pB->squelchgaintarget= squelchgaintarget= maybesquelchgaintarget;
	      pB->absinf= absinf;
	    }
	    
	    if (squelchgain> squelchgaintarget) squelchgain--;
	  } else
        #endif
	    if (squelchgain> 0) squelchgain--;
	  if (absinf> threshold_) {
	    pB->squelching= 0;  /* kick out of squelch mode */
	  }	    
	}
	
	safesquelch= squelchgain*HEADROOM;  safesquelch/= SQUELCHMULT;
	out_ *= safesquelch;  out_/= HEADROOM;
	pB->squelchgain= squelchgain;
      }
    }
    data_ += out_;  /* reassemble the bands here, after filtering */
    
    if (band< (Bands-1)) {
      /* shift the prevs for next time... */
      register int j;
      register gint32 * pprev_, * pprev_1;  int j0;
      j0= Maxprevs-Filtcount[band];
      pprev_= &pprevs[j0]; pprev_1= pprev_+1;
      for (j= j0; j< Maxprevs-1; j++)
	{ *pprev_= *pprev_1;  pprev_++; pprev_1++; }
      pprevs[Maxprevs-1]= band? inf[band-1] : in;
    }
  }

  data_ /= PRECISION;
  /* ...reverse *PRECISION above */
  return data_;
}
